/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif
#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#ifndef PATH_MAX
#define PATH_MAX 4096
#endif

#include <glib.h>
#include <glib-object.h>
#include <gdk/gdkkeysyms.h>
#include <gtk/gtk.h>

#if !GTK_CHECK_VERSION(2, 6, 0)
#define GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID (-2)
#endif

#include <libxfcegui4/libxfcegui4.h>

#define EXO_API_SUBJECT_TO_CHANGE
#include <exo/exo.h>

#include <xfmedia/xfmedia-playlist.h>
#include "playlist-files.h"
#include "jumptofilewin.h"

#define PLAYLIST_CACHE_FILE  "xfmedia/playlist.m3u"

enum {
    PLAYLIST_COL_NUM = 0,
    PLAYLIST_COL_TITLE,
    PLAYLIST_COL_LENGTH_S,
    PLAYLIST_COL_FILENAME,
    PLAYLIST_COL_LENGTH,
    PLAYLIST_COL_METADATA_LOADED,
    PLAYLIST_COL_LAST_MTIME,
    PLAYLIST_N_COLS
};

enum {
    PLIST_SIG_CLEARED = 0,
    PLIST_SIG_OPENED,
    PLIST_SIG_SAVED,
    PLIST_SIG_SCROLLED,
    PLIST_SIG_ENTRY_ADDED,
    PLIST_SIG_ENTRY_CHANGED,
    PLIST_SIG_ENTRY_REMOVED,
    PLIST_SIG_ENTRY_ACTIVATED,
    PLIST_SIG_SHUFFLE_TOGGLED,
    PLIST_SIG_REPEAT_TOGGLED,
    PLIST_SIG_ADD_CLICKED,
    PLIST_N_SIGS
};

/* NOTE: keep in sync with mediamarks.c.  though i don't remember why. */
enum {
    DROP_TGT_REORDER = 0,
    DROP_TGT_URILIST,
    DROP_TGT_STRING
};

static const GtkTargetEntry tv_targets[] = {
    { "PLAYLIST_ITEM", GTK_TARGET_SAME_WIDGET, DROP_TGT_REORDER },
    { "text/uri-list", 0, DROP_TGT_URILIST },
    { "STRING", 0, DROP_TGT_STRING },
};

static void xfmedia_playlist_class_init(XfmediaPlaylistClass *klass);
static void xfmedia_playlist_init(XfmediaPlaylist *plist);
static void xfmedia_playlist_finalize(GObject *object);

static gboolean xfmedia_playlist_save_as_cb(GtkWidget *w, gpointer user_data);
static gint xfmedia_playlist_entry_ref_get_filtered_index(XfmediaPlaylistEntryRef *ref);

struct _XfmediaPlaylistEntryRef
{
    XfmediaPlaylist *plist;
    GtkTreeRowReference *rref;
};

struct _XfmediaPlaylistPriv
{
    GtkWidget *treeview;
    GtkListStore *file_list;
    GtkTreeModelFilter *filtered_file_list;
    GtkTreeSelection *tree_sel;
    GQueue *playlist_queue;
    
    GtkTooltips *ttips;
    
    GtkWidget *jtf_box;
    
    GtkWidget *popup_menu;
    GtkWidget *shuffle_btn;
    GtkWidget *repeat_btn;
    
    GtkWidget *add_menu;
    GtkWidget *external_menu;
    
    guint8 *shuffle_bitmap;
    guint shuffle_bitmap_len;
    
    gchar *allowed_extensions;
    gchar *chooser_last_dir;
    
    guint autosave_id;
    gboolean autosave_is_dirty;
    gboolean usersave_is_dirty;
    
    XfmediaPlaylistEntryRef *bold_ref;
    XfmediaPlaylistEntryRef *italic_ref;
};

static GtkVBoxClass *parent_class = NULL;
static guint __signals[PLIST_N_SIGS] = { 0 };

/* VOID:STRING,GtkToggleButton* */
static void
playlist_marshal_VOID__TBUTTON(GClosure *closure,
        GValue *return_value, guint n_param_values, const GValue *param_values,
        gpointer invocation_hint, gpointer marshal_data)
{
    typedef void (*GMarshalFunc_VOID__TBUTTON)(gpointer data1,
            GtkToggleButton *button, gpointer data2);
    
    register GMarshalFunc_VOID__TBUTTON callback;
    register GCClosure *cc = (GCClosure*)closure;
    register gpointer data1, data2;
    
    g_return_if_fail (n_param_values == 2);
    
    if(G_CCLOSURE_SWAP_DATA(closure)) {
        data1 = closure->data;
        data2 = g_value_peek_pointer(param_values + 0);
    } else {
        data1 = g_value_peek_pointer(param_values + 0);
        data2 = closure->data;
    }
    
    callback = (GMarshalFunc_VOID__TBUTTON)(marshal_data ? marshal_data : cc->callback);
    
    callback(data1, g_value_peek_pointer(param_values + 1), data2);
}

GType
xfmedia_playlist_get_type()
{
    static GType plist_type = 0;
    
    if(!plist_type) {
        static const GTypeInfo plist_info = {
            sizeof(XfmediaPlaylistClass),
            NULL,
            NULL,
            (GClassInitFunc)xfmedia_playlist_class_init,
            NULL,
            NULL,
            sizeof(XfmediaPlaylist),
            0,
            (GInstanceInitFunc)xfmedia_playlist_init,
        };
        
        plist_type = g_type_register_static(GTK_TYPE_VBOX, "XfmediaPlaylist",
                &plist_info, 0);
    }
    
    return plist_type;
}

static void
xfmedia_playlist_class_init(XfmediaPlaylistClass *klass)
{
    GObjectClass *gobject_class;
    
    gobject_class = (GObjectClass *)klass;
    
    parent_class = g_type_class_peek_parent(klass);
    
    gobject_class->finalize = xfmedia_playlist_finalize;
    
    __signals[PLIST_SIG_CLEARED] = g_signal_new("playlist-cleared",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, playlist_cleared), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_OPENED] = g_signal_new("playlist-opened",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, playlist_opened), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_SAVED] = g_signal_new("playlist-saved",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, playlist_saved), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_SCROLLED] = g_signal_new("playlist-scrolled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, playlist_scrolled), NULL,
            NULL, g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_ENTRY_ADDED] = g_signal_new("entry-added",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, entry_added), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_ENTRY_CHANGED] = g_signal_new("entry-changed",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, entry_changed), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_ENTRY_REMOVED] = g_signal_new("entry-removed",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, entry_removed), NULL, NULL,
            g_cclosure_marshal_VOID__STRING, G_TYPE_NONE, 1, G_TYPE_STRING);
    
    __signals[PLIST_SIG_ENTRY_ACTIVATED] = g_signal_new("entry-activated",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, entry_activated), NULL, NULL,
            g_cclosure_marshal_VOID__INT, G_TYPE_NONE, 1, G_TYPE_INT);
    
    __signals[PLIST_SIG_SHUFFLE_TOGGLED] = g_signal_new("shuffle-toggled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, shuffle_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_REPEAT_TOGGLED] = g_signal_new("repeat-toggled",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, repeat_toggled), NULL, NULL,
            g_cclosure_marshal_VOID__VOID, G_TYPE_NONE, 0);
    
    __signals[PLIST_SIG_ADD_CLICKED] = g_signal_new("add-button-clicked",
            XFMEDIA_TYPE_PLAYLIST, G_SIGNAL_RUN_LAST,
            G_STRUCT_OFFSET(XfmediaPlaylistClass, add_button_clicked), NULL,
            NULL, playlist_marshal_VOID__TBUTTON, G_TYPE_NONE, 1,
            GTK_TYPE_TOGGLE_BUTTON);
}

static void
xfmedia_playlist_init(XfmediaPlaylist *plist)
{
    plist->priv = g_new0(XfmediaPlaylistPriv, 1);
}

static void
xfmedia_playlist_finalize(GObject *object)
{
    XfmediaPlaylist *plist;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(object));
    plist = XFMEDIA_PLAYLIST(object);
    
    if(plist->priv->autosave_is_dirty)
        xfmedia_playlist_save_default_playlist(plist);
    
    g_free(plist->priv->allowed_extensions);
    g_free(plist->priv->shuffle_bitmap);
    
    if(plist->priv->chooser_last_dir)
        g_free(plist->priv->chooser_last_dir);
    
    g_object_unref(G_OBJECT(plist->priv->ttips));
    
    g_free(plist->priv);
    plist->priv = NULL;
    
    G_OBJECT_CLASS(parent_class)->finalize(object);
} 

static gboolean
xfmedia_playlist_autosave_timeout(gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    
    plist->priv->autosave_id = 0;
    
    if(!plist->priv->autosave_is_dirty)
        return FALSE;
    
    xfmedia_playlist_save_default_playlist(plist);
    
    return FALSE;
}

static void
xfmedia_playlist_schedule_autosave(XfmediaPlaylist *plist)
{
    if(plist->priv->autosave_id)
        return;
    
    plist->priv->autosave_is_dirty = TRUE;
    
    plist->priv->autosave_id = g_timeout_add(7500,
                                             xfmedia_playlist_autosave_timeout,
                                             plist);
}

static void
xfmedia_playlist_renumber_file_list(GtkListStore *ls, gint from_index,
        gint to_index)
{
    GtkTreeIter itr;
    gint i, nrows;
    
    DBG("renumbering from %d to %d", from_index, to_index);
    
    if(from_index < 0)
        from_index = 0;
    i = from_index + 1;
    nrows = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(ls), NULL);
    if(to_index >= nrows || to_index < 0)
        to_index = nrows - 1;
    
    if(from_index == 0 && gtk_tree_model_get_iter_first(GTK_TREE_MODEL(ls), &itr)) {
        gtk_list_store_set(ls, &itr, PLAYLIST_COL_NUM, 1, -1);
        i++;
    } else if(from_index == 0 || (from_index > 0
            && !gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(ls), &itr,
                                             NULL, from_index-1)))
    {
        return;
    }
    
    for(; gtk_tree_model_iter_next(GTK_TREE_MODEL(ls), &itr); i++) {
        gtk_list_store_set(ls, &itr, PLAYLIST_COL_NUM, i, -1);
        if(i == to_index + 1)
            break;
    }
}

static void
xfmedia_playlist_cell_render_title(GtkTreeViewColumn *tree_column,
        GtkCellRenderer *cell, GtkTreeModel *tree_model, GtkTreeIter *iter,
        gpointer data)
{
    XfmediaPlaylist *plist = data;
    GtkTreePath *cur_path = NULL;
    gint *indices, cur_index = -1;
    
    cur_path = gtk_tree_model_get_path(tree_model, iter);
    if(!cur_path)
        return;
    indices = gtk_tree_path_get_indices(cur_path);
    if(indices)
        cur_index = indices[0];
    gtk_tree_path_free(cur_path);
    if(!indices)
        return;
    
    if(plist->priv->bold_ref && xfmedia_playlist_entry_ref_get_filtered_index(plist->priv->bold_ref) == cur_index)
        g_object_set(G_OBJECT(cell), "weight", PANGO_WEIGHT_BOLD, NULL);
    else
        g_object_set(G_OBJECT(cell), "weight", PANGO_WEIGHT_NORMAL, NULL);
    
    if(plist->priv->italic_ref && xfmedia_playlist_entry_ref_get_filtered_index(plist->priv->italic_ref) == cur_index)
        g_object_set(G_OBJECT(cell), "style", PANGO_STYLE_ITALIC, "style-set", TRUE, NULL);
    else
        g_object_set(G_OBJECT(cell), "style-set", FALSE, NULL);
}

static void
xfmedia_playlist_treeview_scroll_cb(GtkAdjustment *adj, gpointer user_data)
{
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_SCROLLED], 0);
}

static void
xfmedia_playlist_shuffle_toggle_cb(GtkToggleButton *b, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    gboolean set;
    
    set = gtk_toggle_button_get_active(b);
    if(set)
        xfmedia_playlist_shuffle_unset_all(plist);
    
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_SHUFFLE_TOGGLED], 0);
}

static void
xfmedia_playlist_repeat_toggle_cb(GtkToggleButton *b, gpointer user_data)
{
    g_signal_emit(G_OBJECT(user_data), __signals[PLIST_SIG_REPEAT_TOGGLED], 0);
}

static void
xfmedia_playlist_add_cb(GtkToggleButton *tb, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    
    DBG("emitting add-clicked signal");
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ADD_CLICKED], 0, tb);
}

static void
xfmedia_playlist_remove_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GList *selected, *l, *rrefs = NULL;
    guint low_index = -1, idx;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected) {
        for(l = selected; l; l = l->next) {
            GtkTreePath *path;
            gint *indices;
            
            path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                    (GtkTreePath *)l->data);
            
            if(path) {
                indices = gtk_tree_path_get_indices(path);
                idx = indices[0];
                if(idx < low_index)
                    low_index = idx;
                
                rrefs = g_list_prepend(rrefs,
                    gtk_tree_row_reference_new(GTK_TREE_MODEL(plist->priv->file_list), path));
                gtk_tree_path_free(path);
            }
            gtk_tree_path_free((GtkTreePath *)l->data);
        }
        g_list_free(selected);
        
        for(l = rrefs; l; l = l->next) {
            GtkTreeRowReference *rref = l->data;
            GtkTreePath *path = gtk_tree_row_reference_get_path(rref);
            gint *indices, path_index;
            
            indices = gtk_tree_path_get_indices(path);
            path_index = indices[0];
            
            xfmedia_playlist_remove_entry(plist, path_index);
            gtk_tree_path_free(path);
            gtk_tree_row_reference_free(rref);
        }
        g_list_free(rrefs);
    }
    
    DBG("renumbering from %d", low_index);
    if(low_index != -1)
        xfmedia_playlist_renumber_file_list(plist->priv->file_list, low_index, -1);        
}

static void
xfmedia_playlist_go_up_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkTreePath *child_path;
    GtkTreeIter me, above_me;
    gint idx, path_index;
    GList *selected;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected && g_list_length(selected) == 1) {
        child_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                (GtkTreePath *)selected->data);
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                &me, child_path))
        {
            if(gtk_tree_path_prev((GtkTreePath *)selected->data)) {
                if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                        &above_me, (GtkTreePath *)selected->data))
                {
                    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                            &above_me, PLAYLIST_COL_NUM, &idx, -1);
                    gtk_list_store_set(plist->priv->file_list, &above_me,
                            PLAYLIST_COL_NUM, idx+1, -1);
                    gtk_list_store_set(plist->priv->file_list, &me,
                            PLAYLIST_COL_NUM, idx, -1);
                    gtk_list_store_swap(plist->priv->file_list, &me, &above_me);
                    path_index = xfmedia_tree_path_to_index((GtkTreePath *)selected->data);
                    xfmedia_playlist_shuffle_swap_entries(plist, path_index, path_index-1);
                    xfmedia_playlist_schedule_autosave(plist);
                }
            }
        }
        if(child_path)
            gtk_tree_path_free(child_path);
    }
    
    if(selected) {
        g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(selected);
    }
}

static void
xfmedia_playlist_go_down_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkTreePath *child_path;
    GtkTreeIter me, below_me;
    gint idx, path_index;
    GList *selected;
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(selected && g_list_length(selected) == 1) {
        child_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                (GtkTreePath *)selected->data);
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                &me, child_path))
        {
            gtk_tree_path_next((GtkTreePath *)selected->data);
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &below_me, (GtkTreePath *)selected->data))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &below_me, PLAYLIST_COL_NUM, &idx, -1);
                gtk_list_store_set(plist->priv->file_list, &below_me,
                        PLAYLIST_COL_NUM, idx-1, -1);
                gtk_list_store_set(plist->priv->file_list, &me,
                        PLAYLIST_COL_NUM, idx, -1);
                gtk_list_store_swap(plist->priv->file_list, &me, &below_me);
                path_index = xfmedia_tree_path_to_index((GtkTreePath *)selected->data);
                xfmedia_playlist_shuffle_swap_entries(plist, path_index, path_index+1);
                xfmedia_playlist_schedule_autosave(plist);
            }
        }
        if(child_path)
            gtk_tree_path_free(child_path);
    }
    
    if(selected) {
        g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(selected);
    }
}

static gint
playlist_dirty_confirm(GtkWindow *parent)
{
    const gchar *primary = _("Save changes to this playlist before closing?");
    const gchar *secondary = _("You've changed this playlist since it was last saved.");
    
    return xfce_message_dialog(parent, "Xfmedia", GTK_STOCK_DIALOG_WARNING,
            primary, secondary,
            XFCE_CUSTOM_STOCK_BUTTON, _("Do_n't save"),
                GTK_STOCK_NO, GTK_RESPONSE_NO,
            GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
            GTK_STOCK_SAVE, GTK_RESPONSE_YES, NULL);
}

static void
xfmedia_playlist_new_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkWidget *toplevel;
    gint resp;
    
    if(plist->priv->usersave_is_dirty
       && xfmedia_playlist_get_n_entries(plist) > 0)
    {
        toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
        resp = playlist_dirty_confirm(GTK_WINDOW(toplevel));
        switch(resp) {
            case GTK_RESPONSE_CANCEL:
                return;
            case GTK_RESPONSE_YES:
                if(xfmedia_playlist_save_as_cb(GTK_WIDGET(plist), plist))
                    plist->priv->usersave_is_dirty = FALSE;
                else
                    return;
                break;
            case GTK_RESPONSE_NO:
                plist->priv->usersave_is_dirty = FALSE;
                break;
        }
    }
    
    xfmedia_playlist_clear(plist);
    xfmedia_playlist_shuffle_init_entries(plist, 0);
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_CLEARED], 0);
}

static gboolean
xfmedia_playlist_save_as_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkWidget *topwin, *chooser;
    GtkFileFilter *filters[4];
    gchar *filename;
    gboolean ret = FALSE;
    
    if(!xfmedia_playlist_get_n_entries(plist))
        return TRUE;
    
    topwin = gtk_widget_get_toplevel(w);
    
    chooser = gtk_file_chooser_dialog_new(_("Save Playlist"), GTK_WINDOW(topwin),
            GTK_FILE_CHOOSER_ACTION_SAVE, GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL, GTK_STOCK_SAVE, GTK_RESPONSE_ACCEPT, NULL);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(chooser), FALSE);
    if(plist->priv->chooser_last_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    
    filters[0] = gtk_file_filter_new();
    gtk_file_filter_set_name(filters[0], _("Autodetect type by extension"));
    gtk_file_filter_add_pattern(filters[0], "*");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filters[0]);
    filters[1] = gtk_file_filter_new();
    gtk_file_filter_set_name(filters[1], _("Flat file"));
    gtk_file_filter_add_pattern(filters[1], "*");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filters[1]);
    filters[2] = gtk_file_filter_new();
    gtk_file_filter_set_name(filters[2], "M3U");
    gtk_file_filter_add_pattern(filters[2], "*.m3u");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filters[2]);
    filters[3] = gtk_file_filter_new();
    gtk_file_filter_set_name(filters[3], "PLS");
    gtk_file_filter_add_pattern(filters[3], "*.pls");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filters[3]);
    
    gtk_widget_show(chooser);
    
    if(gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
        if(filename) {
            /* meh, no need for GtkFileChooser, and I left this out of the API
            * for no good reason */
            GtkFileFilter *the_filter = gtk_file_chooser_get_filter(GTK_FILE_CHOOSER(chooser));
            XfmediaPlaylistFormat format = XFMEDIA_PLAYLIST_FORMAT_FLAT_FILE;
            
            if(the_filter == filters[0])
                format = XFMEDIA_PLAYLIST_FORMAT_UNKNOWN;
            else if(the_filter == filters[1])
                format = XFMEDIA_PLAYLIST_FORMAT_FLAT_FILE;
            else if(the_filter == filters[2])
                format = XFMEDIA_PLAYLIST_FORMAT_M3U;
            else if(the_filter == filters[3])
                format = XFMEDIA_PLAYLIST_FORMAT_PLS;
            
            ret = xfmedia_playlists_save(plist, filename, format);
            if(ret)
                g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_SAVED], 0, filename);
            g_free(filename);
        }
        if(plist->priv->chooser_last_dir)
            g_free(plist->priv->chooser_last_dir);
        plist->priv->chooser_last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(chooser));

    }
    gtk_widget_destroy(chooser);
    
    return ret;
}

static void
xfmedia_playlist_open_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkWidget *topwin, *chooser;
    GtkFileFilter *filter;
    gchar *filename;
    gint resp;
    
    topwin = gtk_widget_get_toplevel(w);
    
    if(plist->priv->usersave_is_dirty
       && xfmedia_playlist_get_n_entries(plist) > 0)
    {
        resp = playlist_dirty_confirm(GTK_WINDOW(w));
        switch(resp) {
            case GTK_RESPONSE_CANCEL:
                return;
            case GTK_RESPONSE_YES:
                if(xfmedia_playlist_save_as_cb(w, plist))
                    plist->priv->usersave_is_dirty = FALSE;
                else
                    return;
                break;
            case GTK_RESPONSE_NO:
                plist->priv->usersave_is_dirty = FALSE;
                break;
        }
    }
    
    chooser = gtk_file_chooser_dialog_new(_("Open Playlist"), GTK_WINDOW(topwin),
            GTK_FILE_CHOOSER_ACTION_OPEN, GTK_STOCK_CANCEL,
            GTK_RESPONSE_CANCEL, GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT, NULL);
    gtk_file_chooser_set_select_multiple(GTK_FILE_CHOOSER(chooser), FALSE);
    if(plist->priv->chooser_last_dir)
        gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER(chooser),
                plist->priv->chooser_last_dir);
    gtk_dialog_set_default_response(GTK_DIALOG(chooser), GTK_RESPONSE_ACCEPT);
    
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("All Files"));
    gtk_file_filter_add_pattern(filter, "*");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    filter = gtk_file_filter_new();
    gtk_file_filter_set_name(filter, _("Playlists"));
    gtk_file_filter_add_pattern(filter, "*.m3u");
    gtk_file_filter_add_pattern(filter, "*.pls");
    gtk_file_filter_add_pattern(filter, "*.asx");
    gtk_file_chooser_add_filter(GTK_FILE_CHOOSER(chooser), filter);
    
    gtk_widget_show(chooser);
    
    if(gtk_dialog_run(GTK_DIALOG(chooser)) == GTK_RESPONSE_ACCEPT) {
        gtk_widget_hide(chooser);
        filename = gtk_file_chooser_get_filename(GTK_FILE_CHOOSER(chooser));
        if(filename) {
            xfmedia_playlist_clear(plist);
            if(xfmedia_playlists_load(plist, filename)) {
                xfmedia_playlist_schedule_autosave(plist);
                g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_OPENED], 0, filename);
            }
            g_free(filename);
        }
        if(plist->priv->chooser_last_dir)
            g_free(plist->priv->chooser_last_dir);
        plist->priv->chooser_last_dir = gtk_file_chooser_get_current_folder(GTK_FILE_CHOOSER(chooser));
    }
    gtk_widget_destroy(chooser);
    xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}

static gboolean
xfmedia_playlist_treeview_keypress_cb(GtkWidget *w, GdkEventKey *evt,
        gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    
    if((evt->keyval == GDK_Return || evt->keyval == GDK_KP_Enter) ) {
        GList *selected;
        
        selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
        if(selected) {
            GtkTreePath *child_path;
            gint *indices;
            
            child_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                    (GtkTreePath *)selected->data);
            
            if(child_path) {
                indices = gtk_tree_path_get_indices(child_path);
                g_signal_emit(G_OBJECT(plist),
                        __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
                gtk_tree_path_free(child_path);
                
                if(plist->priv->jtf_box)
                    gtk_widget_destroy(plist->priv->jtf_box);
            }
            
            g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
            g_list_free(selected);
        }
        
        return TRUE;
    } else if(evt->keyval == GDK_Delete || evt->keyval == GDK_KP_Delete) {
        xfmedia_playlist_remove_cb(w, plist);
        return TRUE;
    }
    
    return FALSE;
}

static gboolean
xfmedia_playlist_treeview_buttonpress_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GList *selected;
    
    if(evt->button == 1 && evt->type == GDK_2BUTTON_PRESS) {
        selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
        if(selected) {
            GtkTreePath *child_path;
            gint *indices;
            
            child_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                    (GtkTreePath *)selected->data);
            
            if(child_path) {
                indices = gtk_tree_path_get_indices(child_path);
                g_signal_emit(G_OBJECT(plist),
                        __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
                gtk_tree_path_free(child_path);
                
                if(plist->priv->jtf_box)
                    gtk_widget_destroy(plist->priv->jtf_box);
            }
            
            /* free selection stuff */
            g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
            g_list_free(selected);
        }
    } else if(evt->button == 3) {
        GtkTreePath *mouse_over = NULL;
        
        if(gtk_tree_view_get_path_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                evt->x, evt->y, &mouse_over, NULL, NULL, NULL))
        {
            gtk_tree_selection_unselect_all(plist->priv->tree_sel);
            gtk_tree_selection_select_path(plist->priv->tree_sel, mouse_over);
            gtk_tree_path_free(mouse_over);
            
            gtk_menu_popup(GTK_MENU(plist->priv->popup_menu), NULL, NULL, NULL,
                    NULL, evt->button, evt->time);
            return TRUE;  /* eat event so DnD doesn't kick in */
        }
    }
    
    return FALSE;
}

static gboolean
xfmedia_playlist_treeview_drag_drop_cb(GtkWidget *widget,
        GdkDragContext *drag_context, gint x, gint y, guint time,
        gpointer user_data)
{
    GdkAtom target = GDK_NONE;
    GtkTreeViewDropPosition drop_pos = GTK_TREE_VIEW_DROP_BEFORE;
    GtkTreePath *path = NULL;
    static GtkTargetList *tlist = NULL;
    
    /* don't let the default handler activate */
    g_signal_stop_emission_by_name(G_OBJECT(widget), "drag-drop");
    
    if(!tlist) {
        /* this isn't thread-safe, but i don't care at this point */
        tlist = gtk_target_list_new(tv_targets, 3);
    }
    
    target = gtk_drag_dest_find_target(widget, drag_context, tlist);
    if(target == GDK_NONE)
        return FALSE;
    
    DBG("got target");
    
    if(!gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(widget), x, y, &path, &drop_pos)) {
        GtkTreeModel *model = gtk_tree_view_get_model(GTK_TREE_VIEW(widget));
        gint nitems = gtk_tree_model_iter_n_children(model, NULL);
        
        if(path)
            gtk_tree_path_free(path);
        
        if(nitems > 0) {
            drop_pos = GTK_TREE_VIEW_DROP_AFTER;
            path = gtk_tree_path_new_from_indices(nitems - 1, -1);
        } else {
            drop_pos = GTK_TREE_VIEW_DROP_BEFORE;
            path = gtk_tree_path_new_from_indices(0, -1);
        }
    }
    
    gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(widget), path, drop_pos);
    
    if(path)
        gtk_tree_path_free(path);
    
    gtk_drag_get_data(widget, drag_context, target, time);
    gtk_tree_view_set_drag_dest_row(GTK_TREE_VIEW(widget), NULL,
            GTK_TREE_VIEW_DROP_BEFORE);
    
    return TRUE;
}

static void
xfmedia_playlist_treeview_drag_get_cb(GtkWidget *widget,
        GdkDragContext *drag_context, GtkSelectionData *data, guint info,
        guint time, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GdkAtom playlist_item = gdk_atom_intern("PLAYLIST_ITEM", FALSE);
    
    TRACE("entering");
    
    if(data->target == playlist_item) {
        GtkTreeSelection *sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(widget));
        GList *selected;
        GtkTreeModel *model = NULL;
        GtkTreeIter *itr;
        
        selected = gtk_tree_selection_get_selected_rows(sel, &model);
        if(!selected) {
            DBG("failed to get selected item");
            return;
        }
        
        itr  = g_new0(GtkTreeIter, 1);
        gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->filtered_file_list),
                itr, (GtkTreePath *)selected->data);
        
        gtk_selection_data_set(data, playlist_item, 8,
                (gpointer)&itr, sizeof(itr));
        
        g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
        g_list_free(selected);
    }
}

static guint
recurse_add_dirs(XfmediaPlaylist *plist, const gchar *dirname, gint add_index)
{
    GDir *dir;
    const gchar *file;
    gchar fullpath[PATH_MAX], *p;
    GList *file_list = NULL, *dir_list = NULL, *l;
    guint nadded = 0;
    
    dir = g_dir_open(dirname, 0, NULL);
    if(!dir)
        return 0;
    
    while((file = g_dir_read_name(dir))) {
        gboolean is_dir;
        
        g_snprintf(fullpath, PATH_MAX, "%s%s%s", dirname, G_DIR_SEPARATOR_S, file);
        is_dir = g_file_test(fullpath, G_FILE_TEST_IS_DIR);
        
        if(!is_dir && plist->priv->allowed_extensions) {
            p = g_strrstr(file, ".");
            if(!p || !*(p+1))
                continue;
            if(!strstr(plist->priv->allowed_extensions, p))
                continue;
        }
        
        if(is_dir) {
            dir_list = g_list_insert_sorted(dir_list, g_strdup(fullpath),
                    (GCompareFunc)g_ascii_strcasecmp);
        } else {
            file_list = g_list_insert_sorted(file_list, g_strdup(fullpath),
                    (GCompareFunc)g_ascii_strcasecmp);
        }
    }
    g_dir_close(dir);
    
    for(l = dir_list; l; l = l->next) {
        gint nadded_local;
        nadded_local = recurse_add_dirs(plist, (const gchar *)l->data, add_index);
        if(add_index != -1)
            add_index += nadded_local;
        nadded += nadded_local;
        g_free(l->data);
    }
    if(dir_list)
        g_list_free(dir_list);
    
    gdk_threads_enter();
    for(l = file_list; l; l = l->next) {
        gchar *title_utf8 = xfmedia_filename_to_name(l->data);
        if(add_index == -1) {
            xfmedia_playlist_append_entry(plist, title_utf8, -1,
                    (const gchar *)l->data, FALSE);
        } else {
            xfmedia_playlist_insert_entry(plist, add_index++, title_utf8, -1,
                    (const gchar *)l->data, FALSE);
        }
        nadded++;
        g_free(title_utf8);
        g_free(l->data);
    }
    gdk_threads_leave();
    
    if(file_list)
        g_list_free(file_list);
    
    return nadded;
}

struct RecurseData {
    XfmediaPlaylist *plist;
    gchar *basedir;
    gint add_index;
};

static gpointer
recurse_add_dirs_th(gpointer data)
{
    struct RecurseData *recurse_data = data;
    guint nadded;
    
    nadded = recurse_add_dirs(recurse_data->plist, recurse_data->basedir,
            recurse_data->add_index);
    
    if(nadded > 0) {
        gdk_threads_enter();
        //xfmedia_playlist_shuffle_add_entries(recurse_data->mwin->plist,
        //        -1, nadded);
        xfmedia_playlist_treeview_scroll_cb(NULL, recurse_data->plist);
        gdk_threads_leave();
    }
    
    g_free(recurse_data->basedir);
    g_free(recurse_data);
    
    return NULL;
}


static void
xfmedia_playlist_treeview_drag_rcv_cb(GtkWidget *widget,
        GdkDragContext *drag_context, gint x, gint y,  GtkSelectionData *data,
        guint info, guint time, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GdkAtom _PLAYLIST_ITEM = gdk_atom_intern("PLAYLIST_ITEM", FALSE);
    GdkAtom _URI_LIST = gdk_atom_intern("text/uri-list", FALSE);
    GdkAtom _STRING = gdk_atom_intern("STRING", FALSE);
    GtkTreePath *dest_path = NULL;
    GtkTreeViewDropPosition drop_pos;
    GtkTreeIter *src_itr = NULL, dest_itr;
    gboolean drag_success = FALSE;
    
    DBG("got DnD target: %s", gdk_atom_name(data->target));
    
    /* default handlers suck */
    g_signal_stop_emission_by_name(G_OBJECT(widget), "drag-data-received");
    
    if(data->target == _PLAYLIST_ITEM) {
        if(data->data && (src_itr = *(GtkTreeIter **)data->data)
                && gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                                                     x, y, &dest_path, &drop_pos))
        {
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->filtered_file_list),
                                       &dest_itr, dest_path))
            {
                GtkTreeIter child_src_itr, child_dest_itr;
                gint src_index = 0, dest_index = 0;
                
                gtk_tree_model_filter_convert_iter_to_child_iter(plist->priv->filtered_file_list,
                        &child_src_itr, src_itr);
                gtk_tree_model_filter_convert_iter_to_child_iter(plist->priv->filtered_file_list,
                        &child_dest_itr, &dest_itr);
                
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &child_src_itr, PLAYLIST_COL_NUM, &src_index, -1);
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &child_dest_itr, PLAYLIST_COL_NUM, &dest_index, -1);
                
				if(src_index != dest_index) {
					src_index--;
					dest_index--;
					
					DBG("src_index=%d, dest_index=%d", src_index, dest_index);
					
					if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
							|| drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
					{
						DBG("putting src before dest");
						/* gtk oddity: if dest is the first item in the list, then
						 * gtk_list_store_move_before(ls, src, dest) will move src
						 * to the _end_ of the list.  somehow it thinks the item
						 * before the zeroth item is the last item. this sucks. */
						if(dest_index == 0) {
							GtkTreeModel *model = GTK_TREE_MODEL(plist->priv->file_list);
							/* first convert our iters into row refs */
							GtkTreePath *srcpath = gtk_tree_model_get_path(model,
                                    &child_src_itr);
							GtkTreePath *destpath = gtk_tree_model_get_path(model,
                                    &child_dest_itr);
							GtkTreeRowReference *srcref =
                                    gtk_tree_row_reference_new(model, srcpath);
							GtkTreeRowReference *destref =
                                    gtk_tree_row_reference_new(model, destpath);
							gtk_tree_path_free(srcpath);
							gtk_tree_path_free(destpath);
							
							/* then move the src _after_ the dest */
							gtk_list_store_move_after(plist->priv->file_list,
									src_itr, &dest_itr);
							
							/* now get new iters from the row refs */
							srcpath = gtk_tree_row_reference_get_path(srcref);
							destpath = gtk_tree_row_reference_get_path(destref);
							gtk_tree_model_get_iter(model, &child_src_itr, srcpath);
							gtk_tree_model_get_iter(model, &child_dest_itr, destpath);
							gtk_tree_path_free(srcpath);
							gtk_tree_path_free(destpath);
							
							/* swap the two items, so now dest is before src */
							gtk_list_store_swap(plist->priv->file_list, &child_src_itr,
									&child_dest_itr);
							
							/* ditch the row refs */
							gtk_tree_row_reference_free(srcref);
							gtk_tree_row_reference_free(destref);
							
							/* man, that was stupid */
						} else {
							gtk_list_store_move_before(plist->priv->file_list,
									&child_src_itr, &child_dest_itr);
						}
					} else {
						DBG("putting src after dest");
						gtk_list_store_move_after(plist->priv->file_list,
                                &child_src_itr, &child_dest_itr);
					}
					
					if(src_index < dest_index)
						xfmedia_playlist_renumber_file_list(plist->priv->file_list,
								src_index, dest_index);
					else
						xfmedia_playlist_renumber_file_list(plist->priv->file_list,
								dest_index, src_index);
					xfmedia_playlist_shuffle_swap_entries(plist, src_index, dest_index);
					
					xfmedia_playlist_schedule_autosave(plist);
					drag_success = TRUE;
				}
            } else {
                DBG("one of the itrs is invalid");
            }
            
            gtk_tree_path_free(dest_path);
        } else {
            DBG("gtk_tree_view_get_dest_row_at_pos() failed");
        }
        
        if(src_itr)
            g_free(src_itr);
    } else if(data->target == _URI_LIST) {
        GList *uris, *l;
        gint from_index = -1, idx, num_added = 0;
        
        DBG("we have an uri list: '%s'", (gchar *)data->data);
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            GtkTreePath *child_dest_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                    dest_path);
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &dest_itr, child_dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            if(child_dest_path)
                gtk_tree_path_free(child_dest_path);
            gtk_tree_path_free(dest_path);
        }
        
        idx = from_index;
        uris = xfmedia_uri_list_extract_uris((const gchar *)data->data);
        for(l = uris; l; l = l->next) {
            gchar *filename = l->data, *title_utf8, *tmp;
            
            if(!strstr(filename, ":/")) {
                tmp = g_strdup_printf("file://%s", filename);
                g_free(filename);
                filename = tmp;
            }
            
            if(!g_ascii_strncasecmp(filename, "file://", 7)) {
                tmp = g_filename_from_uri(filename, NULL, NULL);
                if(!tmp)
                    memmove(filename, filename+7, strlen(filename+7)+1);
                else {
                    g_free(filename);
                    filename = tmp;
                }
            }
            
            DBG("adding '%s'", filename);
            
            if(g_file_test(filename, G_FILE_TEST_IS_DIR)) {
                struct RecurseData *recurse_data = g_new0(struct RecurseData, 1);
                recurse_data->plist = plist;
                recurse_data->basedir = g_strdup(filename);
                recurse_data->add_index = from_index;
                g_thread_create(recurse_add_dirs_th, recurse_data, FALSE, NULL);
            } else {                
                title_utf8 = xfmedia_filename_to_name(filename);
                if(from_index == -1)
                    xfmedia_playlist_append_entry(plist, title_utf8, -1, filename, FALSE);
                else {
                    xfmedia_playlist_insert_entry(plist, idx, title_utf8, -1, filename, FALSE);
                    if(idx != -1)
                        idx++;
                }
                g_free(title_utf8);
            
                num_added++;
            }
            
            g_free(filename);
        }
        if(uris) {
            if(from_index == -1) {
                from_index = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
                from_index -= num_added - 1;
            }
            xfmedia_playlist_renumber_file_list(plist->priv->file_list,
                    from_index, -1);
            g_list_free(uris);
            
            xfmedia_playlist_treeview_scroll_cb(NULL, plist);
            //xfmedia_playlist_shuffle_add_entries(plist, from_index, num_added);
        }
        
        drag_success = TRUE;
    } else if(data->target == _STRING) {
        gchar *filename = g_strdup((gchar *)data->data);
        gint from_index = -1;
        
        if(gtk_tree_view_get_dest_row_at_pos(GTK_TREE_VIEW(plist->priv->treeview),
                    x, y, &dest_path, &drop_pos))
        {
            GtkTreePath *child_dest_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
                    dest_path);
            if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->file_list),
                    &dest_itr, child_dest_path))
            {
                gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list),
                        &dest_itr, PLAYLIST_COL_NUM, &from_index, -1);
                if(drop_pos == GTK_TREE_VIEW_DROP_BEFORE
                        || drop_pos == GTK_TREE_VIEW_DROP_INTO_OR_BEFORE)
                {
                    from_index--;
                }
            }
            if(child_dest_path)
                gtk_tree_path_free(child_dest_path);
            gtk_tree_path_free(dest_path);
        }
        
        if(filename) {
            gchar *title_utf8, *tmp;
            
            if(!strstr(filename, ":/")) {
                tmp = g_strdup_printf("file://%s", filename);
                g_free(filename);
                filename = tmp;
            }
            
            if(!g_ascii_strncasecmp(filename, "file://", 7)) {
                tmp = g_filename_from_uri(filename, NULL, NULL);
                if(!tmp)
                    memmove(filename, filename+7, strlen(filename+7)+1);
                else {
                    g_free(filename);
                    filename = tmp;
                }
            }
            
            DBG("adding '%s'", filename);
            
            title_utf8 = xfmedia_filename_to_name(filename);
            xfmedia_playlist_insert_entry(plist, from_index, title_utf8, -1,
                    filename, FALSE);
            g_free(title_utf8);
            
            xfmedia_playlist_renumber_file_list(plist->priv->file_list,
                    from_index, -1);
            xfmedia_playlist_treeview_scroll_cb(NULL, plist);
            //xfmedia_playlist_shuffle_add_entries(plist, from_index, 1);
            drag_success = TRUE;
        }
        
        g_free(filename);
    }
    
    gtk_drag_finish(drag_context, drag_success, FALSE, time);
}

static gboolean
xfmedia_playlist_external_menu_cb(GtkWidget *w, GdkEventButton *evt,
        gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    
    if(evt->button == 3 && plist->priv->external_menu) {
        gtk_menu_popup(GTK_MENU(plist->priv->external_menu), NULL, NULL, NULL,
                NULL, evt->button, evt->time);
        return TRUE;
    }
    
    return FALSE;
}

static void
xfmedia_playlist_jtf_activate_cb(GtkTreePath *activated_path,
        gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    gint *indices;
    
    indices = gtk_tree_path_get_indices(activated_path);
    if(indices)
        g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_ACTIVATED], 0, indices[0]);
}

static gboolean
playlist_filter_visible_func(GtkTreeModel *model, GtkTreeIter *itr,
        gpointer data)
{
    XfmediaPlaylist *plist = data;
    
    if(!plist->priv->jtf_box)
        return TRUE;
    
    return xfmedia_jump_to_file_filter_visible_func(model, itr,
            g_object_get_data(G_OBJECT(plist->priv->jtf_box), "xfmedia-jtfdata"));
}




/******
 * public api
 ******/

GtkWidget *
xfmedia_playlist_new()
{
    XfmediaPlaylist *plist;
    GtkWidget *topvbox, *sw, *hbox, *treeview, *btn, *img, *evtbox, *spacer,
        *menu, *mi;
    GdkPixbuf *pix;
    GtkListStore *ls;
    GtkTreeModelFilter *fm;
    GtkTreeViewColumn *col;
    GtkCellRenderer *render;
    GtkTooltips *ttips;
    gint iw, ih;
    
    plist = g_object_new(XFMEDIA_TYPE_PLAYLIST, NULL);
    topvbox = GTK_WIDGET(plist);
    gtk_box_set_spacing(GTK_BOX(topvbox), BORDER);
    
    ttips = plist->priv->ttips = gtk_tooltips_new();
    g_object_ref(G_OBJECT(ttips));
    gtk_object_sink(GTK_OBJECT(ttips));
    
    sw = gtk_scrolled_window_new(NULL, NULL);
    gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw), GTK_POLICY_NEVER,
            GTK_POLICY_ALWAYS);
    gtk_scrolled_window_set_shadow_type(GTK_SCROLLED_WINDOW(sw),
            GTK_SHADOW_ETCHED_IN);
    gtk_widget_show(sw);
    gtk_box_pack_start(GTK_BOX(topvbox), sw, TRUE, TRUE, 0);
    
    plist->priv->file_list = ls = gtk_list_store_new(PLAYLIST_N_COLS,
            G_TYPE_INT, G_TYPE_STRING, G_TYPE_STRING, G_TYPE_STRING,
            G_TYPE_INT, G_TYPE_BOOLEAN, G_TYPE_INT);
    
    plist->priv->filtered_file_list = fm
            = GTK_TREE_MODEL_FILTER(gtk_tree_model_filter_new(GTK_TREE_MODEL(ls), NULL));
    gtk_tree_model_filter_set_visible_func(fm, playlist_filter_visible_func,
            plist, NULL);
    
    plist->priv->treeview = treeview
            = gtk_tree_view_new_with_model(GTK_TREE_MODEL(plist->priv->filtered_file_list));
    gtk_tree_view_set_headers_visible(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_headers_clickable(GTK_TREE_VIEW(treeview), FALSE);
    gtk_tree_view_set_reorderable(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), TRUE);
    gtk_tree_view_set_enable_search(GTK_TREE_VIEW(treeview), FALSE);
    gtk_container_set_focus_child(GTK_CONTAINER(plist), treeview);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("#"), render, "text",
            PLAYLIST_COL_NUM, NULL);
    gtk_tree_view_column_set_resizable(col, FALSE);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render = exo_cell_renderer_ellipsized_text_new();
	if(gtk_major_version == 2 && gtk_minor_version >= 6)
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END, NULL);
	else {
		g_object_set(G_OBJECT(render), "ellipsize", EXO_PANGO_ELLIPSIZE_END,
				"ellipsize-set", TRUE, NULL);
	}
    col = gtk_tree_view_column_new_with_attributes(_("Title"), render, "text",
            PLAYLIST_COL_TITLE, NULL);
    g_object_set(G_OBJECT(col), "resizable", TRUE, "expand", TRUE, NULL);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    render = gtk_cell_renderer_text_new();
    col = gtk_tree_view_column_new_with_attributes(_("Length"), render, "text",
            PLAYLIST_COL_LENGTH_S, NULL);
    gtk_tree_view_column_set_resizable(col, FALSE);
    gtk_tree_view_column_set_cell_data_func(col, render,
            xfmedia_playlist_cell_render_title, plist, NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), col);
    
    plist->priv->tree_sel = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(plist->priv->tree_sel, GTK_SELECTION_MULTIPLE);
    gtk_widget_show(treeview);
    gtk_container_add(GTK_CONTAINER(sw), treeview);
    gtk_widget_add_events(treeview, GDK_BUTTON_PRESS|GDK_KEY_PRESS);
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
            G_CALLBACK(xfmedia_playlist_treeview_buttonpress_cb), plist);
    g_signal_connect(G_OBJECT(treeview), "key-press-event",
            G_CALLBACK(xfmedia_playlist_treeview_keypress_cb), plist);
    g_signal_connect(G_OBJECT(gtk_tree_view_get_vadjustment(GTK_TREE_VIEW(treeview))),
            "value-changed", G_CALLBACK(xfmedia_playlist_treeview_scroll_cb), plist);
    gtk_tree_view_enable_model_drag_source(GTK_TREE_VIEW(treeview),
            GDK_BUTTON1_MASK, tv_targets, 3, GDK_ACTION_MOVE);
    gtk_drag_source_set(treeview, GDK_BUTTON1_MASK, tv_targets, 3, GDK_ACTION_MOVE);
    gtk_tree_view_enable_model_drag_dest(GTK_TREE_VIEW(treeview), tv_targets,
            3, GDK_ACTION_COPY);
    g_signal_connect(G_OBJECT(treeview), "drag-data-received",
            G_CALLBACK(xfmedia_playlist_treeview_drag_rcv_cb), plist);
    g_signal_connect(G_OBJECT(treeview), "drag-data-get",
            G_CALLBACK(xfmedia_playlist_treeview_drag_get_cb), plist);
    g_signal_connect(G_OBJECT(treeview), "drag-drop",
            G_CALLBACK(xfmedia_playlist_treeview_drag_drop_cb), plist);
    
    evtbox = gtk_event_box_new();
    gtk_event_box_set_visible_window(GTK_EVENT_BOX(evtbox), FALSE);
    gtk_widget_show(evtbox);
    gtk_box_pack_start(GTK_BOX(topvbox), evtbox, FALSE, FALSE, 0);
    gtk_widget_add_events(evtbox, GDK_BUTTON_PRESS);
    g_signal_connect(G_OBJECT(evtbox), "button-press-event",
            G_CALLBACK(xfmedia_playlist_external_menu_cb), plist);
    
    hbox = gtk_hbox_new(FALSE, BORDER-4);
    gtk_widget_show(hbox);
    gtk_container_add(GTK_CONTAINER(evtbox), hbox);
    
    img = gtk_image_new_from_stock(GTK_STOCK_ADD, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_toggle_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_add_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Add entry to playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_remove_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Remove entry from playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_go_up_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Move selected playlist entry up"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_go_down_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Move selected playlist entry down"), NULL);
    
    spacer = gtk_alignment_new(0, 0, 0, 0);
    gtk_widget_show(spacer);
    gtk_box_pack_start(GTK_BOX(hbox), spacer, TRUE, TRUE, 0);
    
    gtk_icon_size_lookup(GTK_ICON_SIZE_SMALL_TOOLBAR, &iw, &ih);
    
    pix = xfce_themed_icon_load("media-playlist-shuffle", iw);
    if(!pix)
        pix = xfce_themed_icon_load("media-shuffle", iw);
    if(!pix)
        pix = xfce_themed_icon_load("stock_media-shuffle", iw);
    if(!pix)
        pix = xfce_themed_icon_load("xfmedia-shuffle", iw);
    if(pix) {
        img = gtk_image_new_from_pixbuf(pix);
        gtk_widget_show(img);
        g_object_unref(G_OBJECT(pix));
        plist->priv->shuffle_btn = btn = gtk_toggle_button_new();
        gtk_container_add(GTK_CONTAINER(btn), img);
    } else
        plist->priv->shuffle_btn = btn = gtk_toggle_button_new_with_label("S");
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Randomize play order"), NULL);
    
    pix = xfce_themed_icon_load("media-playlist-repeat", iw);
    if(!pix)
        pix = xfce_themed_icon_load("repeat", iw);
    if(!pix)
        pix = xfce_themed_icon_load("stock_repeat", iw);
    if(!pix)
        pix = xfce_themed_icon_load("xfmedia-repeat", iw);
    if(pix) {
        img = gtk_image_new_from_pixbuf(pix);
        gtk_widget_show(img);
        g_object_unref(G_OBJECT(pix));
        plist->priv->repeat_btn = btn = gtk_toggle_button_new();
        gtk_container_add(GTK_CONTAINER(btn), img);
    } else
        plist->priv->repeat_btn = btn = gtk_toggle_button_new_with_label("R");
    gtk_widget_show(btn);
    gtk_box_pack_start(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "toggled",
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Repeat after playlist finishes"), NULL);
    
    spacer = gtk_alignment_new(0, 0, 0, 0);
    //gtk_widget_set_size_request(spacer, 25, -1);
    gtk_widget_show(spacer);
    gtk_box_pack_start(GTK_BOX(hbox), spacer, TRUE, TRUE, 0);
    
    img = gtk_image_new_from_stock(GTK_STOCK_SAVE_AS, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_save_as_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Save current playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_OPEN, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_open_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Open a saved playlist"), NULL);
    
    img = gtk_image_new_from_stock(GTK_STOCK_NEW, GTK_ICON_SIZE_SMALL_TOOLBAR);
    gtk_widget_show(img);
    btn = gtk_button_new();
    gtk_container_add(GTK_CONTAINER(btn), img);
    gtk_widget_show(btn);
    gtk_box_pack_end(GTK_BOX(hbox), btn, FALSE, FALSE, 0);
    g_signal_connect(G_OBJECT(btn), "clicked",
            G_CALLBACK(xfmedia_playlist_new_cb), plist);
    gtk_tooltips_set_tip(ttips, btn, _("Start a new playlist"), NULL);
    
    plist->priv->popup_menu = menu = gtk_menu_new();
    gtk_widget_show(menu);
    gtk_icon_size_lookup(GTK_ICON_SIZE_MENU, &iw, &ih);
    
    img = gtk_image_new_from_stock(GTK_STOCK_REMOVE, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("_Remove Entry"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_remove_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_UP, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("Move _Up"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_go_up_cb), plist);
    
    img = gtk_image_new_from_stock(GTK_STOCK_GO_DOWN, GTK_ICON_SIZE_MENU);
    gtk_widget_show(img);
    mi = gtk_image_menu_item_new_with_mnemonic(_("Move _Down"));
    gtk_widget_show(mi);
    gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(mi), img);
    gtk_menu_shell_append(GTK_MENU_SHELL(menu), mi);
    g_signal_connect(G_OBJECT(mi), "activate",
            G_CALLBACK(xfmedia_playlist_go_down_cb), plist);
    
    return GTK_WIDGET(plist);
}

gboolean
xfmedia_playlist_load_default_playlist(XfmediaPlaylist *plist)
{
    gchar *file;
    gboolean ret = FALSE;
    
    /* use xfce_resource_lookup() so distros could provide a
     * default playlist if they wanted to */
    file = xfce_resource_lookup(XFCE_RESOURCE_CACHE, PLAYLIST_CACHE_FILE);
    if(file) {
        ret = xfmedia_playlists_load(plist, file);
        g_free(file);
    }
    
    return ret;
}

gboolean
xfmedia_playlist_save_default_playlist(XfmediaPlaylist *plist)
{
    gchar *file;
    gboolean ret = FALSE;
    
    file = xfce_resource_save_location(XFCE_RESOURCE_CACHE, PLAYLIST_CACHE_FILE,
                                       TRUE);
    if(file) {
        ret = xfmedia_playlists_save(plist, file, XFMEDIA_PLAYLIST_FORMAT_M3U);
        g_free(file);
    }
    
    return ret;
}

static void
xfmedia_playlist_insert_entry_internal(XfmediaPlaylist *plist, GtkTreeIter *itr,
        gint idx, const gchar *title, gint length, const gchar *filename,
        gboolean metadata_loaded)
{
    gchar length_str[32], *filename_new = NULL;
    gint last_mtime = 0;
    
    /* munge the filename for .iso files */
    if((g_str_has_suffix(filename, ".iso")
        || g_str_has_suffix(filename, ".ISO"))
       && !g_str_has_prefix(filename, "dvd:"))
    {
        filename_new = g_strdup_printf("dvd:/%s", filename);
    }
    
    if(length >= 3600) {
        g_snprintf(length_str, 32, "%02d:%02d:%02d", length/3600,
                   (length%3600)/60, (length%3600)%60);
    } else if(length >= 0)
        g_snprintf(length_str, 32, "%02d:%02d", length/60, length%60);
    
    if(metadata_loaded) {
        GTimeVal now;
        g_get_current_time(&now);
        last_mtime = (gint)now.tv_sec;
    }
    
    gtk_list_store_set(plist->priv->file_list, itr,
            PLAYLIST_COL_NUM, idx + 1,
            PLAYLIST_COL_TITLE, title,
            PLAYLIST_COL_LENGTH, length,
            PLAYLIST_COL_FILENAME, filename_new ? filename_new : filename,
            PLAYLIST_COL_LENGTH_S, (length >= 0) ? length_str : "??:??",
            PLAYLIST_COL_METADATA_LOADED, metadata_loaded,
            PLAYLIST_COL_LAST_MTIME, last_mtime,
            -1);
    
    xfmedia_playlist_shuffle_add_entries(plist, idx, 1);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_ADDED], 0, idx);
    
    g_free(filename_new);
    
    xfmedia_playlist_schedule_autosave(plist);
}

guint
xfmedia_playlist_append_entry(XfmediaPlaylist *plist, const gchar *title,
        gint length, const gchar *filename, gboolean metadata_loaded)
{
    GtkTreeIter itr;
    gint idx;
    
    idx = gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
    
    gtk_list_store_append(plist->priv->file_list, &itr);
    
    xfmedia_playlist_insert_entry_internal(plist, &itr, idx, title, length,
            filename, metadata_loaded);
    
    return idx;
}

guint
xfmedia_playlist_insert_entry(XfmediaPlaylist *plist, guint idx,
        const gchar *title, gint length, const gchar *filename,
        gboolean metadata_loaded)
{
    GtkTreeIter itr;
    
    gtk_list_store_insert(plist->priv->file_list, &itr, idx);
    
    xfmedia_playlist_insert_entry_internal(plist, &itr, idx, title, length,
            filename, metadata_loaded);
    
    return idx;
}

void
xfmedia_playlist_modify_entry(XfmediaPlaylist *plist, guint idx,
        const gchar *title, gint length, const gchar *filename)
{
    GtkTreeIter itr;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, idx))
    {
        return;
    }
    
    if(title) {
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_TITLE, title, -1);
    }
    
    if(length >= 0) {
        gchar *length_str;
        
        if(length >= 3600) {
            length_str = g_strdup_printf("%02d:%02d:%02d", length/3600,
                                         (length%3600)/60, (length%3600)%60);
        } else
            length_str = g_strdup_printf("%02d:%02d", length/60, length%60);
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_LENGTH_S, length_str, -1);
        g_free(length_str);
    }
    
    if(filename) {
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_FILENAME, filename, -1);
    }
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_CHANGED], 0, idx);
    
    xfmedia_playlist_schedule_autosave(plist);
}

gboolean
xfmedia_playlist_remove_entry(XfmediaPlaylist *plist, guint idx)
{
    gchar *filename = NULL;
    GtkTreeIter itr;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, idx))
    {
        return FALSE;
    }
    
    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
            PLAYLIST_COL_FILENAME, &filename, -1);
    
    gtk_list_store_remove(plist->priv->file_list, &itr);
    xfmedia_playlist_shuffle_remove_entries(plist, idx, 1);
    
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_ENTRY_REMOVED], 0,
            filename ? filename : "");
    
    if(filename)
        g_free(filename);
    
    xfmedia_playlist_schedule_autosave(plist);
    
    return TRUE;
}

void
xfmedia_playlist_clear(XfmediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_list_store_clear(plist->priv->file_list);
    g_signal_emit(G_OBJECT(plist), __signals[PLIST_SIG_CLEARED], 0);
    
    xfmedia_playlist_schedule_autosave(plist);
}


gboolean
xfmedia_playlist_get(XfmediaPlaylist *plist, guint idx, gchar **title,
        gint *length, gchar **filename)
{
    GtkTreeIter itr;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist) && (title || length || filename), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, idx))
    {
        return FALSE;
    }
    
    if(title)
        *title = NULL;
    if(length)
        *length = -1;
    if(filename)
        *filename = NULL;
    
    if(title && length && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(title && length) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_LENGTH, length, -1);
    } else if(title && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(length && filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_LENGTH, length,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else if(title) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_TITLE, title, -1);
    } else if(length) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_LENGTH, length, -1);
    } else if(filename) {
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_FILENAME, filename, -1);
    } else
        return FALSE;
    
    return TRUE;
}

guint
xfmedia_playlist_get_n_entries(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), 0);
    
    return gtk_tree_model_iter_n_children(GTK_TREE_MODEL(plist->priv->file_list), NULL);
}

void
xfmedia_playlist_set_dirty(XfmediaPlaylist *plist,
                           gboolean is_dirty)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    plist->priv->usersave_is_dirty = is_dirty;
    plist->priv->autosave_is_dirty = is_dirty;
    
    if(!is_dirty && plist->priv->autosave_id) {
        g_source_remove(plist->priv->autosave_id);
        plist->priv->autosave_id = 0;
    }
}

gboolean
xfmedia_playlist_get_dirty(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    return plist->priv->usersave_is_dirty;
}

gint
xfmedia_playlist_get_selected(XfmediaPlaylist *plist)
{
    gint *indices, idx = -1;
    GList *selected;
    GtkTreePath *child_path;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), -1);
    
    selected = gtk_tree_selection_get_selected_rows(plist->priv->tree_sel, NULL);
    if(!selected)
        return -1;
    
    child_path = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list,
            (GtkTreePath *)selected->data);
    
    if(child_path) {
        indices = gtk_tree_path_get_indices(child_path);
        idx = indices[0];
        gtk_tree_path_free(child_path);
    }
    
    g_list_foreach(selected, (GFunc)gtk_tree_path_free, NULL);
    g_list_free(selected);
    
    return idx;
}

void
xfmedia_playlist_set_bold_entry(XfmediaPlaylist *plist,
        XfmediaPlaylistEntryRef *ref)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && xfmedia_playlist_entry_ref_valid(ref));
    
    if(plist->priv->bold_ref)
        xfmedia_playlist_entry_ref_destroy(plist->priv->bold_ref);
    
    plist->priv->bold_ref = xfmedia_playlist_entry_ref_copy(ref);
    
    /* hack alert: work around gtk 2.6(?) oddity where the redraw doesn't
     * happen unless you scroll */
    gtk_widget_queue_draw(plist->priv->treeview);
}
void
xfmedia_playlist_set_italic_entry(XfmediaPlaylist *plist,
        XfmediaPlaylistEntryRef *ref)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && xfmedia_playlist_entry_ref_valid(ref));
    
    if(plist->priv->italic_ref)
        xfmedia_playlist_entry_ref_destroy(plist->priv->italic_ref);
    
    plist->priv->italic_ref = xfmedia_playlist_entry_ref_copy(ref);
}

void
xfmedia_playlist_set_metadata_loaded(XfmediaPlaylist *plist, guint idx,
        gboolean metadata_loaded)
{
    GtkTreeIter itr;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, idx))
    {
        return;
    }
    
    if(metadata_loaded) {
        GTimeVal now;
        g_get_current_time(&now);
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_METADATA_LOADED, TRUE,
                PLAYLIST_COL_LAST_MTIME, (gint)now.tv_sec,
                -1);
    } else {
        gtk_list_store_set(plist->priv->file_list, &itr,
                PLAYLIST_COL_METADATA_LOADED, FALSE, -1);
    }
}

gboolean
xfmedia_playlist_get_metadata_loaded(XfmediaPlaylist *plist, guint idx)
{
    GtkTreeIter itr;
    gboolean metadata_loaded = FALSE;
    struct stat st;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    if(!gtk_tree_model_iter_nth_child(GTK_TREE_MODEL(plist->priv->file_list),
            &itr, NULL, idx))
    {
        return FALSE;
    }
    
    gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
            PLAYLIST_COL_METADATA_LOADED, &metadata_loaded, -1);
    
    if(metadata_loaded) {
        gchar *filename = NULL;
        gint last_mtime;
        
        gtk_tree_model_get(GTK_TREE_MODEL(plist->priv->file_list), &itr,
                PLAYLIST_COL_FILENAME, &filename,
                PLAYLIST_COL_LAST_MTIME, &last_mtime, -1);
        if(!stat(filename, &st)) {
            if((gint)st.st_mtime > last_mtime)
                metadata_loaded = FALSE;
        }
        
        g_free(filename);
    }
    
    return metadata_loaded;
}    

void
xfmedia_playlist_context_menu_prepend_item(XfmediaPlaylist *plist,
        GtkMenuItem *menu_item)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_menu_shell_prepend(GTK_MENU_SHELL(plist->priv->popup_menu),
            GTK_WIDGET(menu_item));
}

void
xfmedia_playlist_context_menu_append_item(XfmediaPlaylist *plist,
        GtkMenuItem *menu_item)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    gtk_menu_shell_append(GTK_MENU_SHELL(plist->priv->popup_menu),
            GTK_WIDGET(menu_item));
}

void
xfmedia_playlist_set_shuffle_state(XfmediaPlaylist *plist, gboolean state)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    g_signal_handlers_block_by_func(G_OBJECT(plist->priv->shuffle_btn),
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plist->priv->shuffle_btn), state);
    g_signal_handlers_unblock_by_func(G_OBJECT(plist->priv->shuffle_btn),
            G_CALLBACK(xfmedia_playlist_shuffle_toggle_cb), plist);
    xfmedia_playlist_shuffle_unset_all(plist);
}
    
gboolean
xfmedia_playlist_get_shuffle_state(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(plist->priv->shuffle_btn));
}

void
xfmedia_playlist_set_repeat_state(XfmediaPlaylist *plist, gboolean state)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    g_signal_handlers_block_by_func(G_OBJECT(plist->priv->repeat_btn),
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
    gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(plist->priv->repeat_btn), state);
    g_signal_handlers_unblock_by_func(G_OBJECT(plist->priv->repeat_btn),
            G_CALLBACK(xfmedia_playlist_repeat_toggle_cb), plist);
}

gboolean
xfmedia_playlist_get_repeat_state(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    return gtk_toggle_button_get_active(GTK_TOGGLE_BUTTON(plist->priv->repeat_btn));
}

void
xfmedia_playlist_scroll_to_index(XfmediaPlaylist *plist, gint idx,
        gdouble align)
{
    GtkTreePath *path, *child_path;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    child_path = gtk_tree_path_new_from_indices(idx, -1);
    path = gtk_tree_model_filter_convert_child_path_to_path(plist->priv->filtered_file_list, child_path);
    if(path) {
        gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW(plist->priv->treeview),
                path, NULL, TRUE, align, 0.0);
        gtk_tree_path_free(path);
    }
    gtk_tree_path_free(child_path);
}

gboolean
xfmedia_playlist_get_visible_range(XfmediaPlaylist *plist, guint *start,
        guint *end)
{
    GtkTreeView *treeview;
    GdkRectangle rect;
    gint cell_x = 0, cell_y = 0;
    gint wx = 0, wy = 0;
    GtkTreeIter itr;
    GtkTreePath *path_start = NULL, *path_end = NULL;
    gboolean path_start_good = FALSE, path_end_good = FALSE, ret = FALSE;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist) && start && end, FALSE);
    
    if(!GTK_WIDGET_REALIZED(plist->priv->treeview))
        return FALSE;
    treeview = GTK_TREE_VIEW(plist->priv->treeview);
    
    /* get visible rectangle */
    gtk_tree_view_get_visible_rect(treeview, &rect);
    
    /* get a tree path and iter for the bottommost visible item */
    gtk_tree_view_tree_to_widget_coords(treeview, rect.x,
            rect.y+rect.height-1, &wx, &wy);
    if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_end, NULL,
            &cell_x, &cell_y))
    {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->filtered_file_list),
                &itr, path_end))
        {
            path_end_good = TRUE;
        }
    }
    
    /* get a tree path and iter for the topmost visible item */
    gtk_tree_view_tree_to_widget_coords(treeview, rect.x, rect.y, &wx, &wy);
    if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_start, NULL,
            &cell_x, &cell_y))
    {
        if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->filtered_file_list),
                &itr, path_start))
        {
            path_start_good = TRUE;
        }
    }
    
    if(path_start_good && !path_end_good) {
        GtkTreePath *path_1 = gtk_tree_path_new_first();
        GdkRectangle bg_rect = { 0, 0, 0, 0 };
        
        gtk_tree_view_get_background_area(GTK_TREE_VIEW(plist->priv->treeview),
                path_1, NULL, &bg_rect);
        gtk_tree_path_free(path_1);
        
        if(bg_rect.height > 0) {
            /* move back a bit and try to see where the last item is */
            gint y_coord = rect.y + rect.height - 1 - bg_rect.height;
            while(y_coord > rect.y) {
                gtk_tree_view_tree_to_widget_coords(treeview, rect.x,
                        y_coord, &wx, &wy);
                if(gtk_tree_view_get_path_at_pos(treeview, wx, wy, &path_end,
                        NULL, &cell_x, &cell_y))
                {
                    if(gtk_tree_model_get_iter(GTK_TREE_MODEL(plist->priv->filtered_file_list),
                            &itr, path_end))
                    {
                        path_end_good = TRUE;
                        break;
                    }
                }
                y_coord -= bg_rect.height;
            }
        }
    }
    
    if(path_start_good && path_end_good) {
        GtkTreePath *child_path_start, *child_path_end;
        
        child_path_start = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list, path_start);
        child_path_end = gtk_tree_model_filter_convert_path_to_child_path(plist->priv->filtered_file_list, path_end);
        
        if(child_path_start && child_path_end) {
            gint *i_start, *i_end;
            
            i_start = gtk_tree_path_get_indices(child_path_start);
            i_end = gtk_tree_path_get_indices(child_path_end);
        
            *start = i_start[0];
            *end = i_end[0];
        
            ret = TRUE;
        }
        
        if(child_path_start)
            gtk_tree_path_free(child_path_start);
        if(child_path_end)
            gtk_tree_path_free(child_path_end);
    }
    
    if(path_start)
        gtk_tree_path_free(path_start);
    if(path_end)
        gtk_tree_path_free(path_end);
    
    return ret;
}

static gint
plist_sort_string(GtkTreeModel *model, GtkTreeIter *a, GtkTreeIter *b,
        gpointer user_data)
{
    gint ret, type = GPOINTER_TO_UINT(user_data);
    gchar *str_a = NULL, *str_b = NULL;
    
    switch(type) {
        case XFMEDIA_PLAYLIST_SORT_TITLE:
            gtk_tree_model_get(model, a, PLAYLIST_COL_TITLE, &str_a, -1);
            gtk_tree_model_get(model, b, PLAYLIST_COL_TITLE, &str_b, -1);
            break;
        case XFMEDIA_PLAYLIST_SORT_FILENAME:
            gtk_tree_model_get(model, a, PLAYLIST_COL_FILENAME, &str_a, -1);
            gtk_tree_model_get(model, b, PLAYLIST_COL_FILENAME, &str_b, -1);
            break;
        default:
            g_return_val_if_reached(0);
    }
    
    if(!str_a && !str_b)
        ret = 0;
    else if(!str_a) {
        ret = -1;
        g_free(str_b);
    } else if(!str_b) {
        ret = 1;
        g_free(str_a);
    } else {
        ret = g_utf8_collate(str_a, str_b);
        g_free(str_a);
        g_free(str_b);
    }
    
    return ret;
}

static void
plist_sort_random(XfmediaPlaylist *plist)
{
    gint total, *new_order, i, tmp;
    gchar *order_map;
    
    total = xfmedia_playlist_get_n_entries(plist);
    if(total < 2)
        return;
    new_order = g_malloc(sizeof(gint)*total);
    order_map = g_malloc0(sizeof(gchar)*total);
    
    for(i = 0; i < total; i++) {
        do {
#ifdef HAVE_SRANDOM
            tmp = random() % total;
#else
            tmp = rand() % total;
#endif
        } while(order_map[tmp]);
        
        order_map[tmp] = 0xff;
        new_order[i] = tmp;
    }
    
    gtk_list_store_reorder(plist->priv->file_list, new_order);
    
    g_free(new_order);
    g_free(order_map);
}

void
xfmedia_playlist_sort(XfmediaPlaylist *plist, XfmediaPlaylistSortType type)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    switch(type) {
        case XFMEDIA_PLAYLIST_SORT_TITLE:
            gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_TITLE, plist_sort_string,
                    GUINT_TO_POINTER(type), NULL);
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_TITLE, GTK_SORT_ASCENDING);
            /* then ditch the sort column so we don't continually sort */
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, GTK_SORT_ASCENDING);
            break;
        
        case XFMEDIA_PLAYLIST_SORT_FILENAME:
            gtk_tree_sortable_set_sort_func(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_FILENAME, plist_sort_string,
                    GUINT_TO_POINTER(type), NULL);
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    PLAYLIST_COL_FILENAME, GTK_SORT_ASCENDING);
            /* then ditch the sort column so we don't continually sort */
            gtk_tree_sortable_set_sort_column_id(GTK_TREE_SORTABLE(plist->priv->file_list),
                    GTK_TREE_SORTABLE_UNSORTED_SORT_COLUMN_ID, GTK_SORT_ASCENDING);
            break;
        
        case XFMEDIA_PLAYLIST_SORT_RANDOM:
            plist_sort_random(plist);
            break;
        
        default:
            g_return_if_reached();
    }
    
    xfmedia_playlist_renumber_file_list(plist->priv->file_list, 0, -1);
    xfmedia_playlist_treeview_scroll_cb(NULL, plist);
}


void
xfmedia_playlist_set_external_menu(XfmediaPlaylist *plist,
        GtkMenu *menu)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist) && GTK_IS_MENU(menu));
    
    plist->priv->external_menu = GTK_WIDGET(menu);
}

void
xfmedia_playlist_set_allowed_extensions(XfmediaPlaylist *plist,
        const gchar *extensions)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    g_free(plist->priv->allowed_extensions);
    plist->priv->allowed_extensions = g_strdup(extensions);
}

G_CONST_RETURN gchar *
xfmedia_playlist_get_allowed_extensions(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), NULL);
    
    return plist->priv->allowed_extensions;
}

void
xfmedia_playlist_trigger_new_playlist(XfmediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_new_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_open_playlist(XfmediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_open_cb(GTK_WIDGET(plist), plist);
}

void
xfmedia_playlist_trigger_save_playlist(XfmediaPlaylist *plist)
{
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    xfmedia_playlist_save_as_cb(GTK_WIDGET(plist), plist);
}

static gboolean
playlist_jtf_box_dead_idled(XfmediaPlaylist *plist)
{
    GtkWidget *toplevel;
    GdkCursor *cursor;
    
    gdk_threads_enter();
    
    if(plist->priv->bold_ref && xfmedia_playlist_entry_ref_valid(plist->priv->bold_ref)) {
        gint idx = xfmedia_playlist_entry_ref_get_index(plist->priv->bold_ref);
        
        if(idx >= 0)
            xfmedia_playlist_scroll_to_index(plist, idx, 0.5);
    }
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    cursor = gdk_cursor_new(GDK_LEFT_PTR);
    gdk_window_set_cursor(toplevel->window, cursor);
    gdk_cursor_unref(cursor);
    
    gdk_threads_leave();
    
    return FALSE;
}

static void
jtf_box_destroy_cb(GtkWidget *w, gpointer user_data)
{
    XfmediaPlaylist *plist = user_data;
    GtkWidget *toplevel;
    GdkCursor *cursor;
    
    plist->priv->jtf_box = NULL;
    
    toplevel = gtk_widget_get_toplevel(GTK_WIDGET(plist));
    
    cursor = gdk_cursor_new(GDK_WATCH);
    gdk_window_set_cursor(toplevel->window, cursor);
    gdk_cursor_unref(cursor);
    
    /* FIXME: risky? */
    while(gtk_events_pending())
        gtk_main_iteration();
    
    gtk_tree_model_filter_refilter(plist->priv->filtered_file_list);
    
    g_idle_add((GSourceFunc)playlist_jtf_box_dead_idled, plist);
}

void
xfmedia_playlist_trigger_jump_to_file(XfmediaPlaylist *plist)
{
    GList *children;
    
    g_return_if_fail(XFMEDIA_IS_PLAYLIST(plist));
    
    if(plist->priv->jtf_box)
        return;
    
    plist->priv->jtf_box =
            xfmedia_jump_to_file_get_widget(GTK_TREE_VIEW(plist->priv->treeview),
            GTK_TREE_MODEL(plist->priv->filtered_file_list), PLAYLIST_COL_TITLE,
            xfmedia_playlist_jtf_activate_cb, plist);
    gtk_widget_show(plist->priv->jtf_box);
    gtk_box_pack_start(GTK_BOX(plist), plist->priv->jtf_box, FALSE, FALSE, 0);
    gtk_box_reorder_child(GTK_BOX(plist), plist->priv->jtf_box, 0);
    g_signal_connect(G_OBJECT(plist->priv->jtf_box), "destroy",
            G_CALLBACK(jtf_box_destroy_cb), plist);
    
    /* ick */
    children = gtk_container_get_children(GTK_CONTAINER(plist->priv->jtf_box));
    if(children) {
        gtk_widget_grab_focus(GTK_WIDGET(children->data));
        g_list_free(children);
    }
}

gboolean
xfmedia_playlist_jump_to_file_box_has_focus(XfmediaPlaylist *plist)
{
    GList *children;
    gboolean ret = FALSE;
    
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    
    if(!plist->priv->jtf_box)
        return FALSE;
    
    children = gtk_container_get_children(GTK_CONTAINER(plist->priv->jtf_box));
    if(!children)
        return FALSE;
    
    if(GTK_WIDGET_HAS_FOCUS(GTK_WIDGET(children->data)))
        ret = TRUE;
    
    g_list_free(children);
    
    return ret;
}

gboolean
xfmedia_playlist_has_focus(XfmediaPlaylist *plist)
{
    g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist), FALSE);
    return GTK_WIDGET_HAS_FOCUS(plist->priv->treeview);
}

/*
 * playlist shuffle-play management
 */

gboolean
xfmedia_playlist_shuffle_check(XfmediaPlaylist *plist, guint idx)
{
    g_return_val_if_fail(plist->priv->shuffle_bitmap
            && idx < plist->priv->shuffle_bitmap_len, FALSE);
    
    return (plist->priv->shuffle_bitmap[idx] == 0x0);
}

gboolean
xfmedia_playlist_shuffle_check_set(XfmediaPlaylist *plist, guint idx)
{
    g_return_val_if_fail(plist->priv->shuffle_bitmap
            && idx < plist->priv->shuffle_bitmap_len, FALSE);
    
    if(plist->priv->shuffle_bitmap[idx])
        return FALSE;
    
    return (plist->priv->shuffle_bitmap[idx] = 0x1);  /* yes, a single '=' is correct */
}

void
xfmedia_playlist_shuffle_unset(XfmediaPlaylist *plist, guint idx)
{
    g_return_if_fail(idx < plist->priv->shuffle_bitmap_len);
    
    plist->priv->shuffle_bitmap[idx] = 0x0;
}
    
void
xfmedia_playlist_shuffle_unset_all(XfmediaPlaylist *plist)
{
    if(!plist->priv->shuffle_bitmap)
        return;
    
    memset(plist->priv->shuffle_bitmap, 0x0, plist->priv->shuffle_bitmap_len);
}

void
xfmedia_playlist_shuffle_add_entries(XfmediaPlaylist *plist, gint start,
        guint len)
{
    g_return_if_fail(len > 0);
    
    plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap,
            plist->priv->shuffle_bitmap_len + len);
    
    if(start == -1 || start >= plist->priv->shuffle_bitmap_len)
        memset(plist->priv->shuffle_bitmap+plist->priv->shuffle_bitmap_len, 0x0, len);
    else {
        memmove(plist->priv->shuffle_bitmap+start+len, plist->priv->shuffle_bitmap+start, len);
        memset(plist->priv->shuffle_bitmap+start, 0x0, len);
    }
        
    plist->priv->shuffle_bitmap_len += len;
}

void
xfmedia_playlist_shuffle_remove_entries(XfmediaPlaylist *plist, guint start,
        gint len)
{
    g_return_if_fail(len != 0 && start < plist->priv->shuffle_bitmap_len);
    
    if(len < 0)
        len = plist->priv->shuffle_bitmap_len - start;
    
    if(len == plist->priv->shuffle_bitmap_len) {
        g_free(plist->priv->shuffle_bitmap);
        plist->priv->shuffle_bitmap = NULL;
    } else {
        memmove(plist->priv->shuffle_bitmap+start, plist->priv->shuffle_bitmap+start+len, len);
        plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap,
                plist->priv->shuffle_bitmap_len - len);
    }
    
    plist->priv->shuffle_bitmap_len -= len;
}

void
xfmedia_playlist_shuffle_swap_entries(XfmediaPlaylist *plist, guint a, guint b)
{
    guint8 tmp;
    
    g_return_if_fail(a < plist->priv->shuffle_bitmap_len
            && b < plist->priv->shuffle_bitmap_len && a != b);
    
    tmp = plist->priv->shuffle_bitmap[a];
    plist->priv->shuffle_bitmap[a] = plist->priv->shuffle_bitmap[b];
    plist->priv->shuffle_bitmap[b] = tmp;
}

void
xfmedia_playlist_shuffle_init_entries(XfmediaPlaylist *plist, guint len)
{
    if(len == 0) {
        if(plist->priv->shuffle_bitmap) {
            g_free(plist->priv->shuffle_bitmap);
            plist->priv->shuffle_bitmap = NULL;
            plist->priv->shuffle_bitmap_len = 0;
        }
    } else {
        plist->priv->shuffle_bitmap = g_realloc(plist->priv->shuffle_bitmap, len);
        memset(plist->priv->shuffle_bitmap, 0x0, len);
        plist->priv->shuffle_bitmap_len = len;
    }
}


/* row references */

XfmediaPlaylistEntryRef *
xfmedia_playlist_entry_ref_new_for_index(XfmediaPlaylist *plist, guint idx)
{
	XfmediaPlaylistEntryRef *ref;
	GtkTreePath *path;
	
	g_return_val_if_fail(XFMEDIA_IS_PLAYLIST(plist)
			&& idx < xfmedia_playlist_get_n_entries(plist), NULL);
	
	ref = g_new0(XfmediaPlaylistEntryRef, 1);
	ref->plist = plist;
	
	path = gtk_tree_path_new_from_indices(idx, -1);
	ref->rref = gtk_tree_row_reference_new(GTK_TREE_MODEL(plist->priv->file_list),
			path);
	gtk_tree_path_free(path);
	
	return ref;
}

XfmediaPlaylistEntryRef *
xfmedia_playlist_entry_ref_copy(XfmediaPlaylistEntryRef *ref)
{
    XfmediaPlaylistEntryRef *new_ref;
    
    g_return_val_if_fail(ref, NULL);
    
    new_ref = g_new(XfmediaPlaylistEntryRef, 1);
    new_ref->plist = ref->plist;
    new_ref->rref = gtk_tree_row_reference_copy(ref->rref);
    
    return new_ref;
}

gboolean
xfmedia_playlist_entry_ref_valid(XfmediaPlaylistEntryRef *ref)
{
    g_return_val_if_fail(ref, FALSE);
    
    return gtk_tree_row_reference_valid(ref->rref);
}

gint
xfmedia_playlist_entry_ref_get_index(XfmediaPlaylistEntryRef *ref)
{
	GtkTreePath *path;
	gint *indices, idx;
	
	g_return_val_if_fail(ref, -1);
	
	if(!gtk_tree_row_reference_valid(ref->rref))
		return -1;
	
	path = gtk_tree_row_reference_get_path(ref->rref);
	indices = gtk_tree_path_get_indices(path);
	idx = indices[0];
	gtk_tree_path_free(path);
	
	return idx;
}

static gint
xfmedia_playlist_entry_ref_get_filtered_index(XfmediaPlaylistEntryRef *ref)
{
    GtkTreePath *child_path, *path;
	gint *indices, idx = -1;
	
	g_return_val_if_fail(ref, -1);
	
	if(!gtk_tree_row_reference_valid(ref->rref))
		return -1;
	
	child_path = gtk_tree_row_reference_get_path(ref->rref);
    path = gtk_tree_model_filter_convert_child_path_to_path(ref->plist->priv->filtered_file_list, child_path);
    if(path) {
        indices = gtk_tree_path_get_indices(path);
        idx = indices[0];
        gtk_tree_path_free(path);
    }
    gtk_tree_path_free(child_path);
	
	return idx;
}

void
xfmedia_playlist_entry_ref_destroy(XfmediaPlaylistEntryRef *ref)
{
	g_return_if_fail(ref);
	
	gtk_tree_row_reference_free(ref->rref);
	g_free(ref);
}
