/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "remote.h"

#include <stdio.h>

#ifdef HAVE_STRING_H
#include <string.h>
#endif

#ifdef HAVE_STDLIB_H
#include <stdlib.h>
#endif

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif

#ifdef HAVE_SYS_TYPES_H
#include <sys/types.h>
#endif

#ifdef HAVE_SYS_STAT_H
#include <sys/stat.h>
#endif

#ifdef HAVE_DBUS

#ifndef DBUS_API_SUBJECT_TO_CHANGE  /* exo 0.2 sets this, but 0.3 doesn't */
#define DBUS_API_SUBJECT_TO_CHANGE
#endif
#include <dbus/dbus.h>
#include <dbus/dbus-glib.h>
#include <dbus/dbus-glib-lowlevel.h>

#ifdef DBUS_USE_OLD_API
#define dbus_bus_request_name dbus_bus_acquire_service
#define dbus_bus_start_service_by_name dbus_bus_activate_service
#define dbus_bus_get_unique_name dbus_bus_get_base_service
#define dbus_bus_name_has_owner dbus_bus_service_exists
#define DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT DBUS_SERVICE_FLAG_PROHIBIT_REPLACEMENT
#define DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER DBUS_SERVICE_REPLY_PRIMARY_OWNER
#endif

#ifndef DBUS_NAME_FLAG_DO_NOT_QUEUE
#define DBUS_NAME_FLAG_DO_NOT_QUEUE DBUS_NAME_FLAG_PROHIBIT_REPLACEMENT
#endif

#endif  /* HAVE_DBUS */

#include <libxfce4util/libxfce4util.h>

#include "remote.h"
#include <xfmedia/xfmedia-playlist.h>
#include "mainwin.h"
#include "main.h"
#include "xfmedia-internal.h"
#include "playlist-files.h"
#include "xfmedia-common.h"

#ifdef HAVE_DBUS

extern DBusConnection *__xfmedia_dbus_get_bus_connection();


static void xfmedia_dbus_unregister(DBusConnection *connection, void *user_data);
static DBusHandlerResult xfmedia_dbus_message(DBusConnection *connection, DBusMessage *message, void *user_data);

static const struct DBusObjectPathVTable xfmedia_dbus_vtable =
{
    xfmedia_dbus_unregister,
    xfmedia_dbus_message,
    NULL,
};

static void
xfmedia_dbus_unregister(DBusConnection *connection, void *user_data)
{
    XfmediaMainwin *mwin = user_data;
    
    if(mwin->session_id == -1)
        return;
    
    g_warning("Recieved disconnection from D-BUS.  Perhaps the session bus daemon has quit?");
}

static DBusHandlerResult
xfmedia_dbus_message(DBusConnection *connection, DBusMessage *message,
        void *user_data)
{
    XfmediaMainwin *mwin = user_data;
    DBusHandlerResult ret = DBUS_HANDLER_RESULT_NOT_YET_HANDLED;
    DBusMessage *msg_ret = NULL;
    DBusError derr;
    
    DBG("You've got DBusMessage!");
    
    if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_IS_RUNNING))
    {
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_CLEAR_PLAYLIST))
    {
        xfmedia_playlist_clear(mwin->plist);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_LOAD_PLAYLIST))
    {
        gchar *filename = NULL;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            gchar *lcfn = g_filename_from_utf8(filename, -1, NULL, NULL, NULL);
            
            if(lcfn) {
                if(xfmedia_playlists_load(mwin->plist, lcfn))
                    msg_ret = dbus_message_new_method_return(message);
                g_free(lcfn);
            }
        } else
            dbus_error_free(&derr);

#ifdef DBUS_USE_OLD_API
        if(filename)
            dbus_free(filename);
#endif
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_LOAD_PLAYLIST);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_SAVE_PLAYLIST))
    {
        gchar *filename = NULL;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            gchar *lcfn = g_filename_from_utf8(filename, -1, NULL, NULL, NULL);
            
            if(lcfn) {
                if(xfmedia_playlists_save(mwin->plist, lcfn, XFMEDIA_PLAYLIST_FORMAT_UNKNOWN))
                    msg_ret = dbus_message_new_method_return(message);
                g_free(lcfn);
            }
        } else
            dbus_error_free(&derr);

#ifdef DBUS_USE_OLD_API
        if(filename)
            dbus_free(filename);
#endif
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_LOAD_PLAYLIST);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_GET_PLAYLIST_ENTRY))
    {
        guint idx;

        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_UINT32, &idx,
                    DBUS_TYPE_INVALID))
        {
            gchar *title = NULL, *filename = NULL;
            gint length = -1;
            
            if(xfmedia_playlist_get(mwin->plist, idx, &title, &length, &filename)) {
                gchar *utf8_fn = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
                
                if(utf8_fn) {
                    if(!title)
                        title = g_strdup(_("(no title)"));
                    msg_ret = dbus_message_new_method_return(message);
                    dbus_message_append_args(msg_ret,
#ifdef DBUS_USE_OLD_API
                            DBUS_TYPE_STRING, title,
                            DBUS_TYPE_INT32, length,
                            DBUS_TYPE_STRING, utf8_fn,
#else
                            DBUS_TYPE_STRING, &title,
                            DBUS_TYPE_INT32, &length,
                            DBUS_TYPE_STRING, &utf8_fn,
#endif
                            DBUS_TYPE_INVALID);
                    
                    g_free(utf8_fn);
                }
            }
            g_free(title);
            g_free(filename);
        } else
            dbus_error_free(&derr);
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_GET_PLAYLIST_ENTRY);

        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);

        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_GET_PLAYLIST))
    {
        gchar **strv = NULL;
        gchar *cur = NULL;
        guint nentries, i;
        
        nentries = xfmedia_playlist_get_n_entries(mwin->plist);
        strv = g_new(gchar *, nentries + 1);
        for(i = 0; i < nentries; i++) {
            if(xfmedia_playlist_get(mwin->plist, i, &cur, NULL, NULL))
                strv[i] = cur;
        }
        strv[i] = NULL;
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_message_append_args(msg_ret,
#ifdef DBUS_USE_OLD_API
                DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, strv, nentries,
#else
                DBUS_TYPE_ARRAY, DBUS_TYPE_STRING, &strv, nentries,
#endif
                DBUS_TYPE_INVALID);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);

        g_strfreev(strv);

        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_ADD_FILE))
    {
        gchar *filename = NULL;
        gint idx = -1;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                    DBUS_TYPE_STRING, &filename,
                    DBUS_TYPE_INT32, &idx,
                    DBUS_TYPE_INVALID)
                && filename && *filename)
        {
            gchar *lcfn = g_filename_from_utf8(filename, -1, NULL, NULL, NULL);
            
            if(lcfn) {
                gchar *name_utf8 = xfmedia_filename_to_name(filename);
                if(idx < 0)
                    idx = xfmedia_playlist_append_entry(mwin->plist, name_utf8, -1, lcfn, FALSE);
                else
                    idx = xfmedia_playlist_insert_entry(mwin->plist, idx, name_utf8, -1, lcfn, FALSE);
                g_free(name_utf8);
            
                msg_ret = dbus_message_new_method_return(message);
                dbus_message_append_args(msg_ret,
#ifdef DBUS_USE_OLD_API
                        DBUS_TYPE_INT32, idx,
#else
                        DBUS_TYPE_INT32, &idx,
#endif
                        DBUS_TYPE_INVALID);

                g_free(lcfn);
            }
        } 
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_ADD_FILE);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
#ifdef DBUS_USE_OLD_API
        if(filename)
            dbus_free(filename);
#endif
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_REMOVE_FILE))
    {
        gint idx = 0;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                DBUS_TYPE_UINT32, &idx,
                DBUS_TYPE_INVALID))
        {
            if(xfmedia_playlist_remove_entry(mwin->plist, idx))
                msg_ret = dbus_message_new_method_return(message);
        } else
            dbus_error_free(&derr);
        
        if(!msg_ret)
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_REMOVE_FILE);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PLAY))
    {
        gint idx = -1;
        gboolean done = FALSE;
        
        dbus_error_init(&derr);
        if(dbus_message_get_args(message, &derr,
                DBUS_TYPE_INT32, &idx,
                DBUS_TYPE_INVALID))
        {
            if(idx == -1) {
                if(xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY
                            && xfmedia_xine_get_param(mwin->xfx, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)
                {
                    xfmedia_xine_set_param(mwin->xfx, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
                    done = TRUE;
                } else if(mwin->cur_playing)
                    idx = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing);
                else
                    idx = xfmedia_playlist_get_selected(mwin->plist);
            }
            
            if(done || xfmedia_mainwin_play_file_at_index(mwin, idx))
                msg_ret = dbus_message_new_method_return(message);
            else
                msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR, XFMEDIA_REMOTE_PLAY);
            
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        } else {
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR, XFMEDIA_REMOTE_PLAY);
            dbus_connection_send(connection, msg_ret, NULL);
            dbus_message_unref(msg_ret);
        }
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_TOGGLE_PLAY))
    {
        xfmedia_mainwin_toggle_playpause(mwin);
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PAUSE))
    {
        /* yeah, xfmedia_mainwin_toggle_playpause() is crappy.  it's used to
         * start playback from the stopped state as well, which we don't really
         * want.  so let's fudge it. */
        if(xfmedia_xine_get_status(mwin->xfx) == XINE_STATUS_PLAY) {
            xfmedia_mainwin_toggle_playpause(mwin);
            msg_ret = dbus_message_new_method_return(message);
        } else
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR, XFMEDIA_REMOTE_PAUSE);
        
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_STOP))
    {
        xfmedia_mainwin_stop(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_PREV))
    {
        xfmedia_mainwin_prev(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_NEXT))
    {
        xfmedia_mainwin_next(mwin);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_QUIT))
    {
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        xfmedia_quit(mwin, XFMEDIA_QUIT_MODE_EXT);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_NOW_PLAYING))
    {
        gint status, now_playing;
        gchar *title = NULL, *filename = NULL;
        gint length = -1;

        status = xfmedia_xine_get_status(mwin->xfx);
        if(status == XINE_STATUS_PLAY
           && (now_playing = xfmedia_playlist_entry_ref_get_index(mwin->cur_playing)) != -1
           && xfmedia_playlist_get(mwin->plist, now_playing, &title, &length, &filename)) 
        {
            gchar *utf8_fn = g_filename_to_utf8(filename, -1, NULL, NULL, NULL);
            
            if(utf8_fn) {
                if(!title)
                    title = g_strdup(_("(no title)"));

                msg_ret = dbus_message_new_method_return(message);
                dbus_message_append_args(msg_ret,
#ifdef DBUS_USE_OLD_API
                        DBUS_TYPE_INT32, now_playing,
                        DBUS_TYPE_STRING, title,
                        DBUS_TYPE_INT32, length,
                        DBUS_TYPE_STRING, utf8_fn,
#else
                        DBUS_TYPE_INT32, &now_playing,
                        DBUS_TYPE_STRING, &title,
                        DBUS_TYPE_INT32, &length,
                        DBUS_TYPE_STRING, &utf8_fn,
#endif
                        DBUS_TYPE_INVALID);
                
                g_free(utf8_fn);
            }

            g_free(filename);
            g_free(title);
        }
        
        if(!msg_ret) {
            msg_ret = dbus_message_new_error(message, XFMEDIA_DBUS_ERROR,
                    XFMEDIA_REMOTE_NOW_PLAYING);
        }
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
         
        ret = DBUS_HANDLER_RESULT_HANDLED;
    } else if(dbus_message_is_method_call(message, XFMEDIA_DBUS_INTERFACE,
            XFMEDIA_REMOTE_TRIGGER_JTF))
    {
        xfmedia_mainwin_show(mwin);
        gtk_widget_grab_focus(mwin->window);
        xfmedia_playlist_trigger_jump_to_file(mwin->plist);
        
        msg_ret = dbus_message_new_method_return(message);
        dbus_connection_send(connection, msg_ret, NULL);
        dbus_message_unref(msg_ret);
        
        ret = DBUS_HANDLER_RESULT_HANDLED;
    }
    
    return ret;
}

static void
xfmedia_remote_write_location_file(gint session_id)
{
    const gchar *env_vars[] = {
        "DBUS_SESSION_BUS_ADDRESS",
        "DBUS_SESSION_BUS_PID",
        NULL,
    };
    FILE *fp;
    gchar *filename;
    gint i;
    const gchar *value;
    
    g_return_if_fail(session_id >= 0);
    
    filename = g_strdup_printf("%s/xfmedia-remote-env.%d.%d", g_get_tmp_dir(),
                               (gint)getuid(), session_id);
    if(g_file_test(filename, G_FILE_TEST_EXISTS)) {
        if(unlink(filename)) {
            g_warning("Unable to remove '%s'", filename);
            g_free(filename);
            return;
        }
    }
    
    fp = fopen(filename, "w");
    if(!fp) {
        g_warning("Unable to create '%s'", filename);
        g_free(filename);
        return;
    }
    
    for(i = 0; env_vars[i]; ++i) {
        value = g_getenv(env_vars[i]);
        if(value)
            fprintf(fp, "%s=%s\n", env_vars[i], value);
    }
    
    fclose(fp);
    
    chmod(filename, 0600);
    g_free(filename);
}

gint
xfmedia_remote_init(XfmediaMainwin *mwin)
{
    DBusConnection *dbus_conn;
    GError *err = NULL;
    DBusError derr;
    gint i, ret;
    gchar name[64];
    
    g_return_val_if_fail(mwin, -1);
    g_return_val_if_fail(mwin->session_id == -1, mwin->session_id);
    
    dbus_g_thread_init();
    
    dbus_conn = __xfmedia_dbus_get_bus_connection();
    if(G_UNLIKELY(!dbus_conn)) {
        if(err) {
            g_warning("Failed to connect to D-BUS session instance (%d): %s." \
                    "  Remote control interface will not be available.",
                    err->code, err->message);
            g_error_free(err);
        }
        return -1;
    }
    
    for(i = 0; i < MAX_INSTANCES; i++) {
        g_snprintf(name, 64, XFMEDIA_DBUS_SERVICE_FMT, i);
        dbus_error_init(&derr);
        ret = dbus_bus_request_name(dbus_conn, name,
                                    DBUS_NAME_FLAG_DO_NOT_QUEUE, &derr);
        if(ret < 0) {
            g_warning("Unable to acquire D-BUS service '%s': %s." \
                "  Remote control interface will not be available.",
                name, derr.message);
            dbus_error_free(&derr);
            return -1;
        } else if(ret == DBUS_REQUEST_NAME_REPLY_PRIMARY_OWNER) {
            DBG("got ownership with id %d", i);
            break;
        } else {
            DBG("looks like someone else owns id %d", i);
        }
    }
    
    if(i >= MAX_INSTANCES) {
        g_warning("Unable to find free slot for remote control.");
        return -1;
    }
    
    g_snprintf(name, 64, XFMEDIA_DBUS_PATH_FMT, i);
    if(dbus_connection_register_object_path(dbus_conn, name,
            &xfmedia_dbus_vtable, mwin))
    {
        DBG("Successfully registered remote with session id %d.", i);
        xfmedia_remote_write_location_file(i);
        return i;
    }
    
    g_warning("Unable to register with D-BUS daemon." \
            "  Remote control interface will not be available.");
    
    return -1;
}

void
xfmedia_remote_cleanup(XfmediaMainwin *mwin)
{
    DBusConnection *dbus_conn;
    
    g_return_if_fail(mwin);
    
    dbus_conn = __xfmedia_dbus_get_bus_connection();
    
    if(G_UNLIKELY(!dbus_conn || !dbus_connection_get_is_connected(dbus_conn)))
        return;
    
    if(G_LIKELY(mwin->session_id >= 0)) {
        gchar path_name[PATH_MAX];
        
        g_snprintf(path_name, PATH_MAX, "%s/xfmedia-remote-env.%d.%d",
                   g_get_tmp_dir(), (gint)getuid(), mwin->session_id);
        unlink(path_name);
        
        g_snprintf(path_name, PATH_MAX, XFMEDIA_DBUS_PATH_FMT, mwin->session_id);
        mwin->session_id = -1;
        dbus_connection_unregister_object_path(dbus_conn, path_name);
    }
}

#else /* !defined(HAVE_DBUS) */

gint
xfmedia_remote_init(XfmediaMainwin *mwin)
{
    g_warning("Not compiled with D-BUS support.  Remote control interface is not available");
    
    return FALSE;
}

void
xfmedia_remote_cleanup(XfmediaMainwin *mwin)
{
    /* noop */
}

#endif /* !defined(HAVE_DBUS) */
