/*
 *  xfmedia - simple gtk2 media player based on xine
 *
 *  Copyright (c) 2004-2005 Brian Tarricone, <bjt23@cornell.edu>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; version 2 of the License ONLY.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <libxfce4util/libxfce4util.h>

#include <xfmedia/xfmedia-video-window.h>

struct _XfmediaVideoWindowPriv
{
    GtkWidget *video_widget;
    gboolean is_fullscreen;
    gint aspect_width;
    gint aspect_height;
    
    GdkCursor *blank_cursor;
    GdkCursor *normal_cursor;
    GdkCursor *hand_cursor;
    
    GdkRectangle non_fs_geom;
    guint cursor_timeout;
    gboolean cursor_over_spubutton;
    
    gboolean fixed_size;
    gint fixed_width;
    gint fixed_height;
};

enum
{
    PROP_0 = 0,
    PROP_VIDEO_WIDGET,
    PROP_FULLSCREEN,
    PROP_ASPECT,
};

static void xfmedia_video_window_class_init(XfmediaVideoWindowClass *klass);
static void xfmedia_video_window_init(XfmediaVideoWindow *video_window);
static void xfmedia_video_window_set_property(GObject *obj,
                                              guint property_id,
                                              const GValue *value,
                                              GParamSpec *pspec);
static void xfmedia_video_window_get_property(GObject *obj,
                                              guint property_id,
                                              GValue *value,
                                              GParamSpec *pspec);
static void xfmedia_video_window_finalize(GObject *obj);

static void xfmedia_video_window_realize(GtkWidget *widget);


G_DEFINE_TYPE(XfmediaVideoWindow, xfmedia_video_window, GTK_TYPE_WINDOW)


static void
xfmedia_video_window_ensure_cursors(XfmediaVideoWindow *video_window)
{
    if(G_UNLIKELY(!video_window->priv->blank_cursor)) {
        const gchar blank_bitmap_data[8*8] = { 0x0, };
        GdkColor black = { 0, 0, 0, 0 };
        GdkPixmap *bm = gdk_bitmap_create_from_data(NULL, blank_bitmap_data,
                                                    8, 8);
        video_window->priv->blank_cursor = gdk_cursor_new_from_pixmap(bm,
                                                                      bm,
                                                                      &black,
                                                                      &black,
                                                                      8, 8);
        g_object_unref(G_OBJECT(bm));
        video_window->priv->normal_cursor = gdk_cursor_new(GDK_LEFT_PTR);
        video_window->priv->hand_cursor = gdk_cursor_new(GDK_HAND2);
    }
}

static gboolean
xfmedia_video_window_disable_cursor(XfmediaVideoWindow *video_window)
{
    xfmedia_video_window_ensure_cursors(video_window);
    
    gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                          video_window->priv->blank_cursor);
    
    return TRUE;
}

static void
xfmedia_xine_spu_button_enter_cb(gpointer xfx,
                                 XfmediaVideoWindow *video_window)
{
    xfmedia_video_window_ensure_cursors(video_window);
    
    gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                          video_window->priv->hand_cursor);
    video_window->priv->cursor_over_spubutton = TRUE;
}

static void
xfmedia_xine_spu_button_leave_cb(gpointer xfx,
                                 XfmediaVideoWindow *video_window)
{
    xfmedia_video_window_ensure_cursors(video_window);
    
    gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                          video_window->priv->normal_cursor);
    video_window->priv->cursor_over_spubutton = FALSE;
}

static gboolean
xfmedia_video_window_motion_notify_cb(XfmediaVideoWindow *video_window,
                                      GdkEventMotion *evt)
{
    xfmedia_video_window_ensure_cursors(video_window);
    if(video_window->priv->cursor_over_spubutton) {
        gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                              video_window->priv->hand_cursor);
    } else {
        gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                              video_window->priv->normal_cursor);
    }
    
    return FALSE;
}

static void
xfmedia_video_window_fullscreen(XfmediaVideoWindow *video_window)
{
    gtk_window_get_position(GTK_WINDOW(video_window),
                            &video_window->priv->non_fs_geom.x,
                            &video_window->priv->non_fs_geom.y);
    gtk_window_get_size(GTK_WINDOW(video_window),
                        &video_window->priv->non_fs_geom.width,
                        &video_window->priv->non_fs_geom.height);
    
    gtk_window_fullscreen(GTK_WINDOW(video_window));
    
    g_signal_connect(G_OBJECT(video_window), "motion-notify-event",
                     G_CALLBACK(xfmedia_video_window_motion_notify_cb),
                     video_window);
    video_window->priv->cursor_timeout = g_timeout_add(4000,
            (GSourceFunc)xfmedia_video_window_disable_cursor,
            video_window);
}

static void
xfmedia_video_window_unfullscreen(XfmediaVideoWindow *video_window)
{
    g_source_remove(video_window->priv->cursor_timeout);
    video_window->priv->cursor_timeout = 0;
    g_signal_handlers_disconnect_by_func(G_OBJECT(video_window),
                                         G_CALLBACK(xfmedia_video_window_motion_notify_cb),
                                         video_window);
    
    gtk_window_unfullscreen(GTK_WINDOW(video_window));
    
    gtk_window_resize(GTK_WINDOW(video_window),
                      video_window->priv->non_fs_geom.width,
                      video_window->priv->non_fs_geom.height);
    gtk_window_move(GTK_WINDOW(video_window),
                    video_window->priv->non_fs_geom.x,
                    video_window->priv->non_fs_geom.y);
    
    if(video_window->priv->aspect_width != 0
       && video_window->priv->aspect_height != 0)
    {
        xfmedia_video_window_set_aspect(video_window,
                                        video_window->priv->aspect_width,
                                        video_window->priv->aspect_height);
    }
    
    video_window->priv->cursor_over_spubutton = FALSE;
    xfmedia_video_window_ensure_cursors(video_window);
    gdk_window_set_cursor(GTK_WIDGET(video_window)->window,
                          video_window->priv->normal_cursor);
}




static void
xfmedia_video_window_class_init(XfmediaVideoWindowClass *klass)
{
    GObjectClass *gobject_class = (GObjectClass *)klass;
    GtkWidgetClass *widget_class = (GtkWidgetClass *)klass;
    
    gobject_class->set_property = xfmedia_video_window_set_property;
    gobject_class->get_property = xfmedia_video_window_get_property;
    gobject_class->finalize = xfmedia_video_window_finalize;
    
    widget_class->realize = xfmedia_video_window_realize;
    
    g_object_class_install_property(gobject_class, PROP_VIDEO_WIDGET,
                                    g_param_spec_object("video-widget",
                                                        "Video Widget",
                                                        "The video window's video widget",
                                                        GTK_TYPE_WIDGET,
                                                        G_PARAM_READWRITE));
    
    g_object_class_install_property(gobject_class, PROP_FULLSCREEN,
                                    g_param_spec_boolean("fullscreen",
                                                         "Fullscreen",
                                                         "If the video window is fullscreen or not",
                                                         FALSE,
                                                         G_PARAM_READWRITE));
    
    g_object_class_install_property(gobject_class, PROP_ASPECT,
                                    g_param_spec_double("aspect",
                                                        "Aspect Ratio",
                                                        "The aspect ratio of the window",
                                                        0.0, G_MAXDOUBLE, 0.0,
                                                        G_PARAM_READWRITE));
}

static void
xfmedia_video_window_init(XfmediaVideoWindow *video_window)
{
    video_window->priv = g_new0(XfmediaVideoWindowPriv, 1);
    
    gtk_window_set_title(GTK_WINDOW(video_window), _("Xfmedia Video"));
    gtk_window_set_role(GTK_WINDOW(video_window), "xfmedia-video-window");
    gtk_container_set_border_width(GTK_CONTAINER(video_window), 0);
    gtk_widget_add_events(GTK_WIDGET(video_window), GDK_POINTER_MOTION_MASK);
}

static void
xfmedia_video_window_set_property(GObject *obj,
                                  guint property_id,
                                  const GValue *value,
                                  GParamSpec *pspec)
{
    XfmediaVideoWindow *video_window = XFMEDIA_VIDEO_WINDOW(obj);
    
    switch(property_id) {
        case PROP_VIDEO_WIDGET:
            xfmedia_video_window_set_video_widget(video_window,
                                                  g_value_get_object(value));
            break;
        
        case PROP_FULLSCREEN:
            xfmedia_video_window_set_fullscreen(video_window,
                                                g_value_get_boolean(value));
            break;
    }
}

static void
xfmedia_video_window_get_property(GObject *obj,
                                  guint property_id,
                                  GValue *value,
                                  GParamSpec *pspec)
{
    XfmediaVideoWindow *video_window = XFMEDIA_VIDEO_WINDOW(obj);
    
    switch(property_id) {
        case PROP_VIDEO_WIDGET:
            g_value_set_object(value, video_window->priv->video_widget);
            break;
        
        case PROP_FULLSCREEN:
            g_value_set_boolean(value, video_window->priv->is_fullscreen);
            break;
    }
}

static void
xfmedia_video_window_finalize(GObject *obj)
{
    XfmediaVideoWindow *video_window = XFMEDIA_VIDEO_WINDOW(obj);
    
    if(video_window->priv->cursor_timeout)
        g_source_remove(video_window->priv->cursor_timeout);
    
    if(G_LIKELY(video_window->priv->blank_cursor)) {
        gdk_cursor_unref(video_window->priv->blank_cursor);
        gdk_cursor_unref(video_window->priv->normal_cursor);
        gdk_cursor_unref(video_window->priv->hand_cursor);
    }
    
    g_free(video_window->priv);
    
    G_OBJECT_CLASS(xfmedia_video_window_parent_class)->finalize(obj);
}

static void
xfmedia_video_window_realize(GtkWidget *widget)
{
    XfmediaVideoWindow *video_window = XFMEDIA_VIDEO_WINDOW(widget);
    
    GTK_WIDGET_CLASS(xfmedia_video_window_parent_class)->realize(widget);
    
    if(video_window->priv->is_fullscreen)
        xfmedia_video_window_fullscreen(video_window);
}



GtkWidget *
xfmedia_video_window_new(GtkWidget *video_widget)
{
    return g_object_new(XFMEDIA_TYPE_VIDEO_WINDOW,
                        "video-widget", video_widget,
                        NULL);
}

void
xfmedia_video_window_set_video_widget(XfmediaVideoWindow *video_window,
                                      GtkWidget *video_widget)
{
    g_return_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window));
    
    if(video_widget == video_window->priv->video_widget)
        return;
    
    if(video_window->priv->video_widget) {
        g_signal_handlers_disconnect_by_func(G_OBJECT(video_window->priv->video_widget),
                                             G_CALLBACK(xfmedia_xine_spu_button_enter_cb),
                                             video_window);
        g_signal_handlers_disconnect_by_func(G_OBJECT(video_window->priv->video_widget),
                                             G_CALLBACK(xfmedia_xine_spu_button_leave_cb),
                                             video_window);
        gtk_container_remove(GTK_CONTAINER(video_window),
                             video_window->priv->video_widget);
    }
    
    gtk_container_add(GTK_CONTAINER(video_window), video_widget);
    video_window->priv->video_widget = video_widget;
    
    g_signal_connect(G_OBJECT(video_widget), "spu-button-enter",
                     G_CALLBACK(xfmedia_xine_spu_button_enter_cb), video_window);
    g_signal_connect(G_OBJECT(video_widget), "spu-button-leave",
                     G_CALLBACK(xfmedia_xine_spu_button_leave_cb), video_window);
}

GtkWidget *
xfmedia_video_window_get_video_widget(XfmediaVideoWindow *video_window)
{
    g_return_val_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window), NULL);
    return video_window->priv->video_widget;
}

void
xfmedia_video_window_set_fullscreen(XfmediaVideoWindow *video_window,
                                    gboolean fullscreen)
{
    
    g_return_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window));
    
    if(fullscreen == video_window->priv->is_fullscreen)
        return;
    
    video_window->priv->is_fullscreen = fullscreen;
    
    if(!GTK_WIDGET_REALIZED(GTK_WIDGET(video_window)))
        return;
    
    if(fullscreen)
        xfmedia_video_window_fullscreen(video_window);
    else
        xfmedia_video_window_unfullscreen(video_window);
}

gboolean
xfmedia_video_window_get_fullscreen(XfmediaVideoWindow *video_window)
{
    g_return_val_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window), FALSE);
    return video_window->priv->is_fullscreen;
}

void
xfmedia_video_window_set_aspect(XfmediaVideoWindow *video_window,
                                gint width,
                                gint height)
{
    GdkGeometry geom;
    
    g_return_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window)
                     && width >= 0 && height >= 0);
    
    if(width == video_window->priv->aspect_width
       && height == video_window->priv->aspect_height)
    {
        return;
    }
    
    video_window->priv->aspect_width = width;
    video_window->priv->aspect_height = height;
    
    if(video_window->priv->is_fullscreen || video_window->priv->fixed_size)
        return;
    
    if(!GTK_WIDGET_REALIZED(video_window->priv->video_widget))
        gtk_widget_realize(video_window->priv->video_widget);
    
    if(width == 0 || height == 0) {
        gtk_window_set_geometry_hints(GTK_WINDOW(video_window),
                                      video_window->priv->video_widget,
                                      NULL, 0);
    } else {
        /*geom.min_width = geom.max_width = width;
        geom.min_height = geom.max_height = height;*/
        geom.base_width = width;
        geom.base_height = height;
        geom.min_aspect = geom.max_aspect = ((gdouble)width) / height;
        gtk_window_set_geometry_hints(GTK_WINDOW(video_window),
                                      video_window->priv->video_widget,
                                      &geom,
                                      GDK_HINT_MIN_SIZE
                                      | GDK_HINT_MAX_SIZE
                                      | GDK_HINT_BASE_SIZE
                                      | GDK_HINT_ASPECT);
    }
}

void
xfmedia_video_window_get_aspect(XfmediaVideoWindow *video_window,
                                gint *width,
                                gint *height)
{
    g_return_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window) && width && height);
    *width = video_window->priv->aspect_width;
    *height = video_window->priv->aspect_height;
}

void
xfmedia_video_window_set_fixed_size(XfmediaVideoWindow *video_window,
                                    gint width,
                                    gint height)
{
    g_return_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window));
    
    if(width < 0 || height < 0) {
        video_window->priv->fixed_size = FALSE;
        xfmedia_video_window_set_aspect(video_window,
                                        video_window->priv->aspect_width,
                                        video_window->priv->aspect_height);
    } else {
        video_window->priv->fixed_size = TRUE;
        video_window->priv->fixed_width = width;
        video_window->priv->fixed_height = height;
        gtk_window_set_geometry_hints(GTK_WINDOW(video_window),
                                      video_window->priv->video_widget,
                                      NULL, 0);
        gtk_widget_set_size_request(GTK_WIDGET(video_window), width, height);
        gtk_window_resize(GTK_WINDOW(video_window), width, height);
    }
}

gboolean
xfmedia_video_window_has_fixed_size(XfmediaVideoWindow *video_window)
{
    g_return_val_if_fail(XFMEDIA_IS_VIDEO_WINDOW(video_window), FALSE);
    return video_window->priv->fixed_size;
}
