/*
 * Copyright (C) 2000-2007 the xine project
 *
 * This file is part of xine, a free video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA
 *
 * 
 * Xine plugin for Mozilla/Firefox 
 *      written by Claudio Ciccani <klan@users.sf.net>
 *      
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/time.h>
#include <fcntl.h>
#include <unistd.h>

#include <pthread.h>

#ifdef _POSIX_PRIORITY_SCHEDULING
# include <sched.h>
#endif

#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/keysym.h>

#include <xine.h>
#include <xine/xineutils.h>

#include "npapi.h"
#include "npruntime.h"

#include "playlist.h"


#define VERSION_CODE(M, m, s)  (((M) << 16) + ((m) << 8) + (s))
#define XINE_VERSION_CODE      VERSION_CODE(XINE_MAJOR_VERSION,\
                                            XINE_MINOR_VERSION,\
                                            XINE_SUB_VERSION)

#ifdef LOG
# define log(x, ...) \
 do {\
      fprintf (stderr, "xine-plugin (%d): " x "\n", getpid(), ##__VA_ARGS__);\
      fflush  (stderr);\
 } while (0)
#else
# define log(x, ...)
#endif

static int instance_num = 0;

typedef struct {
  NPP                 instance;

  xine_t             *xine;
  xine_vo_driver_t   *vo_driver;
  xine_ao_driver_t   *ao_driver;
  xine_stream_t      *stream;
  xine_event_queue_t *event_queue;
  xine_osd_t         *osd;

  Display            *display;
  int                 screen;
  Window              parent;
  Window              window;
  int                 x, y;
  int                 w, h;

  int                 loop;
  int                 start; /* start time */
  int                 autostart;
  
  char                demux[32];
  
  char                base[1024];
  
  playlist_entry_t   *list;
  playlist_entry_t   *track;
  int                 playlist_type;

  pthread_mutex_t     mutex;
  pthread_t           thread;
  int                 playing;
  
#ifdef ENABLE_SCRIPTING
  NPObject           *object;
#endif
} xine_plugin_t;

#define OSD_WIDTH      384
#define OSD_HEIGHT      80
#define OSD_TIMEOUT      5 /* in seconds */

#define OSD_TEXT_PLAY  ">"
#define OSD_TEXT_STOP  "}"
#define OSD_TEXT_PAUSE "<"
#define OSD_TEXT_QUIT  "{"

/*****************************************************************************/

/* 
 * Parse a time string in the format HH:MM:SS
 * and return the time value in seconds.
 */
static int strtime (const char *str)
{
  char *p = (char *) str;
  int   t = 0;
  int   i;
  
  for (i = 0; i < 3; i++) {
    t *= 60;
    t += atoi (p);
    p  = strchr (p, ':');
    if (!p)
      break;
    p++;
  }
  
  return t;
}

/*****************************************************************************/

static void dest_size_cb (void *data,
                          int video_width, int video_height,
                          double video_pixel_aspect,
                          int *dest_width, int *dest_height,
                          double *dest_pixel_aspect)
{
  xine_plugin_t *this = data;

  *dest_width = this->w;
  *dest_height = this->h;
  *dest_pixel_aspect = video_pixel_aspect ? : 1.0;
}

static void frame_output_cb (void *data,
                             int video_width, int video_height,
                             double video_pixel_aspect,
                             int *dest_x, int *dest_y,
                             int *dest_width, int *dest_height,
                             double *dest_pixel_aspect, int *win_x, int *win_y)
{
  xine_plugin_t *this = data;

  *dest_x = 0;
  *dest_y = 0;
  *win_x = this->x;
  *win_y = this->y;
  *dest_width = this->w;
  *dest_height = this->h;
  *dest_pixel_aspect = video_pixel_aspect ? : 1.0;
}

#ifdef XINE_VISUAL_TYPE_X11_2
static void lock_display_cb (void *data)
{
  xine_plugin_t *this = data;
  
  pthread_mutex_lock (&this->mutex);
  XLockDisplay (this->display);
}

static void unlock_display_cb (void *data)
{
  xine_plugin_t *this = data;
  
  XUnlockDisplay (this->display);
  pthread_mutex_unlock (&this->mutex);
}
#endif

/*****************************************************************************/

static void osd_show_text (xine_plugin_t *this, const char *text) {
  char *s = (char *) text;
  int   y = 0;

  if (!this->osd)
    return;

  pthread_mutex_lock (&this->mutex);

  xine_osd_clear (this->osd);

  while (s && *s) {
    char *n = strchr (s, '\n');
    if (n) {
      char buf[n-s+1];
      int w, h;
      strncpy (buf, s, n-s);
      buf[n-s] = '\0';
      xine_osd_draw_text (this->osd, 0, y, buf, XINE_OSD_TEXT1);
      xine_osd_get_text_size (this->osd, buf, &w, &h);
      y += h;
    } else {
      xine_osd_draw_text (this->osd, 0, y, s, XINE_OSD_TEXT1);
    }
    s = n;
    if (s) s++;
  }

  if (xine_osd_get_capabilities (this->osd) & XINE_OSD_CAP_UNSCALED)
    xine_osd_show_unscaled (this->osd, 0);
  else
    xine_osd_show (this->osd, 0);
  
  xine_osd_hide (this->osd,
                 xine_get_current_vpts (this->stream) +
                 OSD_TIMEOUT * 90000);

  pthread_mutex_unlock (&this->mutex);
}

static const char *stream_error (xine_stream_t *stream) {
  int err = xine_get_error (stream);
  
  switch (err) {
    case XINE_ERROR_NO_INPUT_PLUGIN:
      return "no input plugin to handle stream";
    case XINE_ERROR_NO_DEMUX_PLUGIN:
      return "no demuxer plugin to handle stream";
    case XINE_ERROR_DEMUX_FAILED:
      return "demuxer plugin failed, stream probably corrupted";
    case XINE_ERROR_MALFORMED_MRL:
      return "corrupted mrl";
    case XINE_ERROR_INPUT_FAILED:
      return "input plugin failed";
    default:
      break;
  }
  
  return "unknown error";
} 

static void *player_thread (void *data) {
  xine_plugin_t *this = data;
  
  pthread_setcancelstate (PTHREAD_CANCEL_DISABLE, NULL);

  while (this->track && this->playing) {
    playlist_entry_t *track = this->track;
    const char       *error;
    char              buf[4096];
    int               len = 0;
    int               ret;
    
    if (!strstr (track->mrl, "://") && access (track->mrl, F_OK))
      len = snprintf (buf, sizeof(buf), "%s", this->base);
    len += snprintf (buf+len, sizeof(buf)-len, "%s", track->mrl);
    if (*this->demux)
      snprintf (buf+len, sizeof(buf)-len, "#demux:%s", this->demux);
    
    log ("opening \"%s\"...", buf);
    NPN_Status (this->instance, "xine-plugin: opening stream...");
    
#if XINE_VERSION_CODE >= VERSION_CODE(1,1,4)
    pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
    ret = xine_open (this->stream, buf);
    pthread_setcancelstate (PTHREAD_CANCEL_DISABLE, NULL);
#else
    ret = xine_open (this->stream, buf);
#endif
    if (ret) {
      log ("... stream opened ...");
      NPN_Status (this->instance, "xine-plugin: ... stream opened ...");
    
      if (this->autostart) {
        if (xine_play (this->stream, 0, this->start ? : track->start)) {
          log ("... playback started.");
          NPN_Status (this->instance, "xine-plugin: ... playback started.");
          break;
        }
      }
      else {
        log ("... playback will start on user's request.");
        NPN_Status (this->instance, "xine-plugin: ... press ENTER to start playback.");
        break;
      }
    }
    
    error = stream_error (this->stream);
    log ("%s!", error);
    snprintf (buf, sizeof(buf), "xine-plugin: %s!", error); 
    NPN_Status (this->instance, buf);
    
    pthread_mutex_lock (&this->mutex);
    this->track = track->next;
    playlist_remove (&this->list, track);
    pthread_mutex_unlock (&this->mutex);
  }

  while (this->playing) {
    XEvent evt;
    int    got = 0;
    int    key;

    pthread_mutex_lock (&this->mutex);
    if (this->window) {
      XLockDisplay (this->display);
      got = XPending (this->display);
      if (got)
        XNextEvent (this->display, &evt);
      XUnlockDisplay (this->display);
    }
    pthread_mutex_unlock (&this->mutex);

    if (!got) {
      pthread_mutex_lock (&this->mutex);
      if (this->window) {
        Window tmp;
        /* We don't receive ConfigureNotify events,
         * so we have to query the window position periodically.
         */
        XLockDisplay (this->display);
        XTranslateCoordinates (this->display, this->window,
                               DefaultRootWindow (this->display),
                               0, 0, &this->x, &this->y, &tmp);
        XUnlockDisplay (this->display);
      }
      pthread_mutex_unlock (&this->mutex);
      
      if (this->playing) {
        pthread_setcancelstate (PTHREAD_CANCEL_ENABLE, NULL);
        xine_usec_sleep (10000);
        pthread_setcancelstate (PTHREAD_CANCEL_DISABLE, NULL);
      }
      continue;
    }

    if (evt.xany.window != this->window)
      continue;

    switch (evt.type) {
      case KeyPress:
        if (!this->track)
          break;
        key = XLookupKeysym (&evt.xkey, 0);
        switch (key) {
          case XK_Escape:
            osd_show_text (this, OSD_TEXT_QUIT);
            xine_stop (this->stream);
            xine_close (this->stream);
            break;
          case XK_Return:
            if (xine_get_status (this->stream) == XINE_STATUS_PLAY) {
              xine_stop (this->stream);
              osd_show_text (this, OSD_TEXT_STOP);
            } else {
              if (xine_play (this->stream, 0, this->start ? : this->track->start))
                osd_show_text (this, OSD_TEXT_PLAY);
            }
            break;
          case XK_Pause:
          case XK_space:
            if (xine_get_param (this->stream,
                                XINE_PARAM_SPEED) == XINE_SPEED_PAUSE) {
              xine_set_param (this->stream,
                              XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
              osd_show_text (this, OSD_TEXT_PLAY);
            } else {
              xine_set_param (this->stream,
                              XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
              osd_show_text (this, OSD_TEXT_PAUSE);
            }
            break;
          case XK_Right:
            if (xine_get_stream_info (this->stream,
                                      XINE_STREAM_INFO_SEEKABLE)) {
              int pos = 0;
              if (xine_get_pos_length (this->stream, NULL, &pos, NULL))
                xine_play (this->stream, 0, pos + 30000);
            }
            break;
          case XK_Left:
            if (xine_get_stream_info (this->stream,
                                      XINE_STREAM_INFO_SEEKABLE)) {
              int pos = 0;
              if (xine_get_pos_length (this->stream, NULL, &pos, NULL))
                xine_play (this->stream, 0, pos - 30000);
            }
            break;
          case XK_F1:
          case XK_F2:
          case XK_F3:
          case XK_F4:
          case XK_F5:
          case XK_F6:
          case XK_F7:
          case XK_F8:
          case XK_F9:
            if (xine_get_stream_info (this->stream,
                                      XINE_STREAM_INFO_SEEKABLE))
              xine_play (this->stream, (key - XK_F1 + 1) * 65535 / 10, 0);
            break;
          case XK_Up:
            xine_set_param (this->stream, XINE_PARAM_AUDIO_VOLUME,
                            xine_get_param (this->stream,
                                            XINE_PARAM_AUDIO_VOLUME) + 10);
            break;
          case XK_Down:
            xine_set_param (this->stream, XINE_PARAM_AUDIO_VOLUME,
                            xine_get_param (this->stream,
                                            XINE_PARAM_AUDIO_VOLUME) - 10);
            break;
          case XK_Tab:{
              char buf[22];
              int  pos = 0;

              xine_get_pos_length (this->stream, NULL, &pos, NULL);
              pos /= 1000;
              snprintf (buf, sizeof(buf), "%d:%02d:%02d",
                        pos / 3600, (pos / 60) % 60, pos % 60);
              osd_show_text (this, buf);
            }
            break;
          case XK_I:
          case XK_i:{
              char buf[512];
              int  vw, vh, vr, vb;
              int  ar, ac, ab;
              int  bit, len = 0;

              vw = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_VIDEO_WIDTH);
              vh = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_VIDEO_HEIGHT);
              vr = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_FRAME_DURATION);
              vb = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_VIDEO_BITRATE);
              ar = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_AUDIO_SAMPLERATE);
              ac = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_AUDIO_CHANNELS);
              ab = xine_get_stream_info (this->stream,
                                         XINE_STREAM_INFO_AUDIO_BITRATE);

              bit = xine_get_stream_info (this->stream,
                                          XINE_STREAM_INFO_BITRATE);
              xine_get_pos_length (this->stream, NULL, NULL, &len);
              len /= 1000;

              snprintf (buf, sizeof(buf),
                        "Media: %d:%02d:%02d (%dKb/s)\n"
                        "Video: %dx%d %.1f(r) (%dKb/s)\n"
                        "Audio: %dKHz %d(c) (%dKb/s)",
                        len / 3600, (len / 60) % 60, len % 60, bit / 1000,
                        vw, vh, vr ? 90000.f/vr : 0, vb / 1000, 
                        ar / 1000, ac, ab / 1000);
              osd_show_text (this, buf);
            } 
            break;
          default:
            break;
        }
        break;
        
      case ButtonPress:{
          xine_input_data_t input;
          x11_rectangle_t   rect;
          
          rect.x = evt.xbutton.x;
          rect.y = evt.xbutton.y;
          rect.w = 0;
          rect.h = 0;
          xine_port_send_gui_data (this->vo_driver, 
                                   XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, 
                                   (void *) &rect);
        
          input.event.type = XINE_EVENT_INPUT_MOUSE_BUTTON;
          input.event.stream = this->stream;
          input.event.data = &input;
          input.event.data_length = sizeof(xine_input_data_t);
          input.button = evt.xbutton.button;
          input.x = rect.x;
          input.y = rect.y;
          gettimeofday (&input.event.tv, NULL);
        
          xine_event_send (this->stream, &input.event);
        }
        break;
      
      case MotionNotify:{
          xine_input_data_t input;
          x11_rectangle_t   rect;
          
          rect.x = evt.xmotion.x;
          rect.y = evt.xmotion.y;
          rect.w = 0;
          rect.h = 0;
          xine_port_send_gui_data (this->vo_driver, 
                                   XINE_GUI_SEND_TRANSLATE_GUI_TO_VIDEO, 
                                   (void *) &rect);
        
          input.event.type = XINE_EVENT_INPUT_MOUSE_MOVE;
          input.event.stream = this->stream;
          input.event.data = &input;
          input.event.data_length = sizeof(xine_input_data_t);
          input.x = rect.x;
          input.y = rect.y;
          gettimeofday (&input.event.tv, NULL);
        
          xine_event_send (this->stream, &input.event);
        }
        break;    

      case Expose:
        if (evt.xexpose.count == 0)
          xine_gui_send_vo_data (this->stream,
                                 XINE_GUI_SEND_EXPOSE_EVENT, &evt);
        break;
        
      case ConfigureNotify:
        this->w = evt.xconfigure.width;
        this->h = evt.xconfigure.height;
        break;

      default:
        break;
    }
  }

  return (void *) 0;
}

static void player_stop (xine_plugin_t *this) {
  if (this->playing) {
    log ("stopping player...");
    this->playing = 0;
    pthread_mutex_lock (&this->mutex);
    pthread_cancel (this->thread);
    pthread_mutex_unlock (&this->mutex);
    pthread_join (this->thread, NULL);
    log ("...player stopped.");
  }
}

static int player_start (xine_plugin_t *this) {
  log ("starting player...");
  this->playing = 1;
  if (pthread_create (&this->thread, NULL, player_thread, this)) {
    log ("pthread_create() failed!");
    this->playing = 0;
  }
#ifdef _POSIX_PRIORITY_SCHEDULING
  else {
    /* Make the thread start now. */
    sched_yield ();
  }
#endif
  return this->playing;
}

/*****************************************************************************/

static xine_t *xine_create (void) {
  xine_t *xine;
  char    path[1024];

  xine = xine_new ();
  if (!xine) {
    log ("xine_new() failed!");
    return NULL;
  }

  snprintf (path, sizeof(path), "%s", getenv ("XINERC") ? : "");
  if (!*path) {
    snprintf (path, sizeof(path), "%s/.xine", xine_get_homedir());
    mkdir (path, 0755);
    snprintf (path, sizeof(path), "%s/.xine/config", xine_get_homedir());
  }

  xine_config_load (xine, path);

  xine_init (xine);

  return xine;
}

static void event_listner_cb (void *data, const xine_event_t *event) {
  xine_plugin_t *this = data;

  switch (event->type) {
    case XINE_EVENT_UI_PLAYBACK_FINISHED:
      if (this->playing && this->track) {
        playlist_entry_t *next = this->track->next;
        
        log ("playback finished.");
        
        if (next) {
          player_stop (this);
          this->track = next;
          player_start (this);
        }
        else if (--this->loop > 0) {
          if (this->list == this->track) {
            xine_play (this->stream, 0, this->start ? : this->track->start);
            xine_usec_sleep (100);
          } 
          else {
            player_stop (this);
            this->track = this->list;
            player_start (this);
          }
        }
        else {
          NPN_Status (this->instance, "xine-plugin: playback finished.");
        }
      }
      break;

    case XINE_EVENT_MRL_REFERENCE:
      if (event->data) {
        xine_mrl_reference_data_t *ref = event->data;

        log ("got reference mrl \"%s\".", ref->mrl);
        
        /* FIXME: must remove the mrl that generated the reference. */
        pthread_mutex_lock (&this->mutex);
        playlist_insert (&this->list, ref->mrl, 0);
        pthread_mutex_unlock (&this->mutex);
      }
      break;

    case XINE_EVENT_PROGRESS:
      if (event->data) {
        xine_progress_data_t *data = event->data;
        char buf[256];

        snprintf (buf, sizeof(buf), "%s %d%%",
                  data->description, data->percent);
        osd_show_text (this, buf);
      }
      break;

    default:
      break;
  }
}

static NPError stream_create (xine_plugin_t *this) {

  if (!this->vo_driver) {
    if (this->window) {
      const char   *driver = NULL;
      x11_visual_t  visual;
      
      /* Workaround for xv video driver returning 
       * success from probe either if xv port is busy.
       */
      if (instance_num > 1)
        driver = "xshm"; 

      visual.display = this->display;
      visual.screen = this->screen;
      visual.d = this->window;

      visual.user_data = this;
      visual.dest_size_cb = dest_size_cb;
      visual.frame_output_cb = frame_output_cb;
      
#ifdef XINE_VISUAL_TYPE_X11_2
      visual.lock_display = lock_display_cb;
      visual.unlock_display = unlock_display_cb;

      this->vo_driver = xine_open_video_driver (this->xine, driver,
                                                XINE_VISUAL_TYPE_X11_2, &visual);
#else
      this->vo_driver = xine_open_video_driver (this->xine, driver,
                                                XINE_VISUAL_TYPE_X11, &visual);
#endif
    } else {
      this->vo_driver = xine_open_video_driver (this->xine, "none",
                                                XINE_VISUAL_TYPE_NONE, NULL);
    }

    if (!this->vo_driver) {
      log ("xine_open_video_driver() failed!");
      NPN_Status (this->instance, "xine-plugin: error opening video driver.");
      return NPERR_MODULE_LOAD_FAILED_ERROR;
    }
  }

  if (!this->ao_driver) {
    this->ao_driver = xine_open_audio_driver (this->xine, NULL, NULL);
    if (!this->ao_driver) {
      log ("xine_open_audio_driver() failed!");
      NPN_Status (this->instance, "xine-plugin: error opening audio driver.");
      this->ao_driver = xine_open_audio_driver (this->xine, "none", NULL);
    }
  }

  if (!this->stream) {
    this->stream = xine_stream_new (this->xine,
                                    this->ao_driver, this->vo_driver);
    if (!this->stream) {
      log ("xine_stream_new() failed!");
      return NPERR_OUT_OF_MEMORY_ERROR;
    }
    xine_gui_send_vo_data (this->stream, 
                           XINE_GUI_SEND_DRAWABLE_CHANGED, (void *)this->window);
    xine_gui_send_vo_data (this->stream,
                           XINE_GUI_SEND_VIDEOWIN_VISIBLE, (void *) 1);
    /* Load xine logo. */
    if (xine_open (this->stream, LOGO_PATH))
      xine_play (this->stream, 0, 0);
  }

  if (!this->event_queue) {
    this->event_queue = xine_event_new_queue (this->stream);
    if (!this->event_queue) {
      log ("xine_event_new_queue() failed!");
      return NPERR_OUT_OF_MEMORY_ERROR;
    }

    xine_event_create_listener_thread (this->event_queue,
                                       event_listner_cb, this);
  }

  if (!this->osd) {
    this->osd = xine_osd_new (this->stream, 0, 0, OSD_WIDTH, OSD_HEIGHT);
    if (!this->osd) {
      log ("xine_osd_new() failed!");
      return NPERR_OUT_OF_MEMORY_ERROR;
    }
    xine_osd_set_font (this->osd, "cetus", 16);
    xine_osd_set_text_palette (this->osd,
                               XINE_TEXTPALETTE_WHITE_BLACK_TRANSPARENT,
                               XINE_OSD_TEXT1);
    xine_osd_set_position (this->osd, 3, 3);
  }

  return NPERR_NO_ERROR;
}

/*****************************************************************************/

#ifdef ENABLE_SCRIPTING

#include "methods.h"

typedef struct {
  NPObject       base;
  
  xine_plugin_t *plugin;
  
  NPIdentifier   methods[NUM_METHODS];
  int            methods_count;
} NPPObject;

static NPObject *NPPObject_Allocate (NPP instance, NPClass *class) {
  NPPObject *object;
  
  log ("NPPObject_Allocate( instance=%p, class=%p )", instance, class);
  
  if (!instance || !instance->pdata)
    return NULL;
  
  object = NPN_MemAlloc (sizeof(NPPObject));
  if (!object)
    return NULL;
    
  object->base._class = class;
  object->base.referenceCount = 1;
  object->plugin = instance->pdata;
  object->methods_count = 0;
  
  return &object->base;
}

static int NPPObject_GetMethodID (NPObject *npobj, NPIdentifier name) {
  NPPObject *object = (NPPObject *) npobj;
  int        i;
  
  for (i = 0; i < object->methods_count; i++) {
    if (object->methods[i] == name)
      return i;
  }
  
  for (; i < NUM_METHODS; i++) {
    object->methods[i] = NPN_GetStringIdentifier (methodName[i]);
    object->methods_count++;
    if (object->methods[i] == name)
      return i;
  }
  
  return METHOD_ID_None;
}

static bool NPPObject_HasMethod (NPObject *npobj, NPIdentifier name) {  
  log ("NPPObject_HasMethod( npobj=%p, name=%p )", npobj, name);
  
  if (NPN_IdentifierIsString (name) &&
      NPPObject_GetMethodID (npobj, name) != METHOD_ID_None)
    return true;
  
  return false;
}

static bool NPPObject_Invoke (NPObject *npobj, NPIdentifier name,
                              const NPVariant *args, uint32_t argCount,
                              NPVariant *result) {
  xine_plugin_t *this;
  int            method;
  int            ret = 0;
  
  log ("NPPObject_Invoke( npobj=%p, name=%p, args=%p, argCount=%u, result=%p )",
       npobj, name, args, argCount, result);
       
  this = ((NPPObject *)npobj)->plugin;
  method = NPPObject_GetMethodID (npobj, name);
  
  switch (method) {
    case METHOD_ID_play:
    case METHOD_ID_Play:
    case METHOD_ID_DoPlay:
      log ("Plugin::Play()");
      if (this->track) {
        if (this->playing) {
          ret = 1;
          if (xine_get_status (this->stream) != XINE_STATUS_PLAY)
            ret = xine_play (this->stream, 0, this->start ? : this->track->start);
          xine_set_param (this->stream, XINE_PARAM_SPEED, XINE_SPEED_NORMAL);
        } else {
          ret = player_start (this);
        }
      }
      break;
    case METHOD_ID_pause:
    case METHOD_ID_Pause:
    case METHOD_ID_DoPause:
      log ("Plugin::Pause()");
      if (this->playing) {
        xine_set_param (this->stream, XINE_PARAM_SPEED, XINE_SPEED_PAUSE);
        ret = 1;
      }
      break;
    case METHOD_ID_stop:
    case METHOD_ID_Stop:
    case METHOD_ID_DoStop:
      /* FIXME: for Quicktime Stop() means Pause(). */
      log ("Plugin::Stop()");
      if (this->playing) {
        xine_stop (this->stream);
        ret = 1;
      }
      break;
    case METHOD_ID_CanPlay:
      log ("Plugin::CanPlay()");
      if (this->playing) {
        ret = (xine_get_status (this->stream) != XINE_STATUS_PLAY ||
               xine_get_param (this->stream, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE);
      } else {
        ret = 1;
      }
      break;
    case METHOD_ID_CanPause:
      log ("Plugin::CanPause()");
      if (this->playing) {
        ret = (xine_get_status (this->stream) == XINE_STATUS_PLAY &&
               xine_get_param (this->stream, XINE_PARAM_SPEED) != XINE_SPEED_PAUSE);
      }
      break;
    case METHOD_ID_CanStop:
      log ("Plugin::CanStop()");
      if (this->playing)
        ret = (xine_get_status (this->stream) == XINE_STATUS_PLAY);
      break;
    case METHOD_ID_GetLength:
    case METHOD_ID_GetDuration:
      log ("Plugin::GetLength()");
      if (this->playing)
        xine_get_pos_length (this->stream, NULL, NULL, &ret);
      break;
    case METHOD_ID_GetPosition:
    case METHOD_ID_GetCurrentPosition:
    case METHOD_ID_GetTime:
      log ("Plugin::GetPosition()");
      if (this->playing) {
        xine_get_pos_length (this->stream, NULL, &ret, NULL);
        if (method == METHOD_ID_GetCurrentPosition)
          ret /= 1000; /* WMP returns the position in seconds. */
      }
      break;
    case METHOD_ID_SetPosition:
    case METHOD_ID_SetCurrentPosition:
    case METHOD_ID_SetTime:
      if (argCount >= 1 && NPVARIANT_IS_INT32 (args[0])) {
        int pos = NPVARIANT_TO_INT32 (args[0]);
        if (method == METHOD_ID_SetCurrentPosition)
          pos *= 1000; /* WMP sets the position in seconds. */
        log ("Plugin::SetPosition(%d)", pos);
        this->start = pos;
        if (this->playing) {
          if (xine_get_status (this->stream) == XINE_STATUS_PLAY &&
              xine_get_stream_info (this->stream, XINE_STREAM_INFO_SEEKABLE))
            ret = xine_play (this->stream, 0, pos);
        } else {
          ret = 1;
        }
      }
      break;
    case METHOD_ID_Rewind:
      log ("Plugin::Rewind()");
      if (this->playing && this->track) {
          if (xine_get_status (this->stream) == XINE_STATUS_PLAY &&
              xine_get_stream_info (this->stream, XINE_STREAM_INFO_SEEKABLE))
            ret = xine_play (this->stream, 0, this->start ? : this->track->start);
      }
      break; 
    case METHOD_ID_GetStartTime:
      log ("Plugin::GetStartTime()");
      ret = this->start;
      break;
    case METHOD_ID_SetStartTime:
      if (argCount >= 1 && NPVARIANT_IS_INT32 (args[0])) {
        log ("Plugin::SetStartTime(%d)", NPVARIANT_TO_INT32 (args[0]));
        this->start = NPVARIANT_TO_INT32 (args[0]);
      }
      break;
    case METHOD_ID_GetTimeScale:
      log ("Plugin::GetTimeScale()");
      ret = 1000;
      break;       
    case METHOD_ID_GetPlayState:
      log ("Plugin::GetPlayState()");
      if (this->playing) {
        if (xine_get_status (this->stream) == XINE_STATUS_PLAY) {
          if (xine_get_param (this->stream, XINE_PARAM_SPEED) == XINE_SPEED_PAUSE)
            ret = 4; /* Paused */
          else
            ret = 3; /* Playing */
        }
        /* Contacting, Buffering and Seeking are not supported */
      }
      break;
    case METHOD_ID_GetSource:
    case METHOD_ID_GetURL:
    case METHOD_ID_GetFileName:
      log ("Plugin::GetSource()");
      if (this->track) {
        int   len = strlen (this->track->mrl);
        char *mrl = NPN_MemAlloc (len+1);
        memcpy (mrl, this->track->mrl, len+1);
        STRINGN_TO_NPVARIANT (mrl, len, *result);
        /* Must return now */
        return true;
      }
      break;
    case METHOD_ID_SetSource:
    case METHOD_ID_SetURL:
    case METHOD_ID_SetFileName:
      if (argCount >= 1 && NPVARIANT_IS_STRING (args[0])) {
        const char *mrl = NPVARIANT_TO_STRING(args[0]).utf8characters;
        log ("Plugin::SetSource(%s)", mrl);
        player_stop (this);
        playlist_free (&this->list);
        this->track = playlist_insert (&this->list, mrl, 0);
        ret = player_start (this);
      }
      break;
    case METHOD_ID_GetAutoStart:
    case METHOD_ID_GetAutoPlay:
      log ("Plugin::GetAutoStart()");
      ret = this->autostart;
      break;
    case METHOD_ID_SetAutoStart:
    case METHOD_ID_SetAutoPlay:
      if (argCount >= 1) {
        log ("Plugin::SetAutoStart(%d)", NPVARIANT_TO_BOOLEAN (args[0]));
        this->autostart = NPVARIANT_TO_BOOLEAN (args[0]);
        ret = 1;
      }
      break;
    case METHOD_ID_GetLoop:
    case METHOD_ID_GetIsLooping:
      log ("Plugin::GetLoop()");
      ret = (this->loop > 1);
      break;
    case METHOD_ID_SetLoop:
    case METHOD_ID_SetIsLooping:
      if (argCount >= 1) {
        log ("Plugin::SetLoop(%d)", NPVARIANT_TO_BOOLEAN (args[0]));
        if (NPVARIANT_TO_BOOLEAN (args[0]))
          this->loop = 0x7fffffff;
        else
          this->loop = 1;
        ret = 1;
      }
      break;
    case METHOD_ID_GetNumLoop:
    case METHOD_ID_GetPlayCount:
      log ("Plugin::GetNumLoop()");
      ret = this->loop;
      break;
    case METHOD_ID_SetNumLoop:
    case METHOD_ID_SetPlayCount:
      if (argCount >= 1 && NPVARIANT_IS_INT32 (args[0])) {
        log ("Plugin::SetNumLoop(%d)", NPVARIANT_TO_INT32 (args[0]));
        this->loop = NPVARIANT_TO_INT32 (args[0]);
        ret = 1;
      }
      break;
    case METHOD_ID_GetClipWidth:
      log ("Plugin::GetClipWidth()");
      if (this->playing)
        ret = xine_get_stream_info (this->stream, XINE_STREAM_INFO_VIDEO_WIDTH);
      break;
    case METHOD_ID_GetClipHeight:
      log ("Plugin::GetClipHeight()");
      if (this->playing)
        ret = xine_get_stream_info (this->stream, XINE_STREAM_INFO_VIDEO_HEIGHT);
      break;
    case METHOD_ID_HasNextEntry:
      log ("Plugin::HasNextEntry()");
      ret = (this->track && this->track->next);
      break;
    case METHOD_ID_HasPrevEntry:
      log ("Plugin::HasPrevEntry()");
      ret = (this->track && this->track->prev != this->list);
      break;
    case METHOD_ID_DoNextEntry:
      log ("Plugin::DoNextEntry()");
      if (this->track && this->track->next) {
        player_stop (this);
        this->track = this->track->next;
        ret = player_start (this);
      }
      break;
    case METHOD_ID_DoPrevEntry:
      log ("Plugin::DoPrevEntry()");
      if (this->track && this->track->prev != this->list) {
        player_stop (this);
        this->track = this->track->prev;
        ret = player_start (this);
      }
      break;
    case METHOD_ID_GetNumEntries:
      log ("Plugin::GetNumEntries()");
      if (this->list)
        ret = this->list->prev->id + 1;
      break;
    case METHOD_ID_GetCurrentEntry:
      log ("Plugin::GetCurrentEntry()");
      if (this->track)
        ret = this->track->id;
      break;
    case METHOD_ID_SetCurrentEntry:
      /* This is a Xine's extension: select a playlist entry by its ID. */
      if (argCount >= 1 && NPVARIANT_IS_INT32 (args[0])) {
        playlist_entry_t *entry;      
        log ("Plugin::SetCurrentEntry(%d)", NPVARIANT_TO_INT32 (args[0]));
        for (entry = this->list; entry; entry = entry->next) {
          if (entry->id == NPVARIANT_TO_INT32 (args[0]))
            break;
        }
        if (entry) {
          ret = 1;
          if (entry != this->track) {
            player_stop (this);
            this->track = entry;
            ret = player_start (this);
          }
        }
      }
      break;
    default:
      log ("called unsupported method");
      return false;
  }
  
  INT32_TO_NPVARIANT (ret, *result);
  
  return true;
}

static bool NPPObject_InvokeDefault (NPObject *npobj, 
                                     const NPVariant *args, uint32_t argCount, 
                                     NPVariant *result) {
  log ("NPPObject_Invoke( npobj=%p, args=%p, argCount=%u )", npobj, args, argCount);
 
  return false;
}

/*
 * The following properties are only used by WMP.
 */

static bool NPPObject_HasProperty (NPObject *npobj, NPIdentifier name) {
  log ("NPPObject_HasProperty( npobj=%p, name=%p )", npobj, name);
  
  if (name == NPN_GetStringIdentifier ("controls")  ||
      name == NPN_GetStringIdentifier ("URL")       ||
      name == NPN_GetStringIdentifier ("SRC")       ||
      name == NPN_GetStringIdentifier ("Filename")  ||
      name == NPN_GetStringIdentifier ("autoStart") ||
      name == NPN_GetStringIdentifier ("playCount") ||
      name == NPN_GetStringIdentifier ("currentPosition"))
    return true;
  
  return false;
}

static bool NPPObject_GetProperty (NPObject *npobj, NPIdentifier name, NPVariant *result) {
  xine_plugin_t *this;
  
  log ("NPPObject_GetProperty( npobj=%p, name=%p )", npobj, name);
  
  this = ((NPPObject *)npobj)->plugin;
  
  if (name == NPN_GetStringIdentifier ("controls")) {
    NPN_RetainObject (npobj);
    OBJECT_TO_NPVARIANT (npobj, *result);
    return true;
  }
  else if (name == NPN_GetStringIdentifier ("URL") ||
           name == NPN_GetStringIdentifier ("SRC") ||
           name == NPN_GetStringIdentifier ("Filename")) {
    if (this->track) {
      int   len = strlen (this->track->mrl);
      char *mrl = NPN_MemAlloc (len+1);
      memcpy (mrl, this->track->mrl, len+1);
      STRINGN_TO_NPVARIANT (mrl, len, *result);
      return true;
    }
  }
  else if (name == NPN_GetStringIdentifier ("autoStart")) {
    BOOLEAN_TO_NPVARIANT (this->autostart, *result);
    return true;
  }
  else if (name == NPN_GetStringIdentifier ("playCount")) {
    INT32_TO_NPVARIANT (this->loop, *result);
    return true;
  }
  else if (name == NPN_GetStringIdentifier ("currentPosition")) {
    int pos = 0;
    if (this->playing)
      xine_get_pos_length (this->stream, NULL, &pos, NULL);
    else
      pos = this->start;
    pos /= 1000;
    INT32_TO_NPVARIANT (pos, *result);
    return true;
  }
 
  return false;
}

static bool NPPObject_SetProperty (NPObject *npobj, NPIdentifier name, const NPVariant *value) {
  xine_plugin_t *this;
  
  log ("NPPObject_SetProperty( npobj=%p, name=%p, value=%p )", npobj, name, value);
  
  this = ((NPPObject *)npobj)->plugin;
  
  if (name == NPN_GetStringIdentifier ("URL") ||
      name == NPN_GetStringIdentifier ("SRC") ||
      name == NPN_GetStringIdentifier ("Filename")) {
    if (NPVARIANT_IS_STRING (*value)) {
      const char *mrl = NPVARIANT_TO_STRING(*value).utf8characters;
      log ("Plugin.SRC = '%s'", mrl);
      player_stop (this);
      playlist_free (&this->list);
      this->track = playlist_insert (&this->list, mrl, 0);
      player_start (this);
      return true;
    }
  }
  else if (name == NPN_GetStringIdentifier ("autoStart")) {
    log ("Plugin.autoStart = %d", NPVARIANT_TO_BOOLEAN (*value));
    this->autostart = NPVARIANT_TO_BOOLEAN (*value);
    return true;
  }
  else if (name == NPN_GetStringIdentifier ("playCount")) {
    if (NPVARIANT_IS_INT32 (*value)) {
      log ("Plugin.playCount = %d", NPVARIANT_TO_INT32 (*value));
      this->loop = NPVARIANT_TO_INT32 (*value);
      return true;
    }
  }
  else if (name == NPN_GetStringIdentifier ("currentPosition")) {
    if (NPVARIANT_IS_INT32 (*value)) {
      log ("Plugin.currentPosition = %d", NPVARIANT_TO_INT32 (*value));
      /* Probably we should seek here */
      this->start = NPVARIANT_TO_INT32 (*value) * 1000;
      return true;
    }
  }
  
  return false;
}

static bool NPPObject_RemoveProperty (NPObject *npobj, NPIdentifier name) {
  log ("NPPObject_RemoveProperty( npobj=%p, name=%p )", npobj, name);
  
  return false;
}

static NPClass pluginClass = {
  structVersion:  NP_CLASS_STRUCT_VERSION,
  allocate:       NPPObject_Allocate,
  hasMethod:      NPPObject_HasMethod,
  invoke:         NPPObject_Invoke,
  invokeDefault:  NPPObject_InvokeDefault,
  hasProperty:    NPPObject_HasProperty,
  getProperty:    NPPObject_GetProperty,
  setProperty:    NPPObject_SetProperty,
  removeProperty: NPPObject_RemoveProperty,
};
#endif /* ENABLE_SCRIPTING */

/*****************************************************************************/

#define BUILTIN_MIMETYPES  "audio/x-scpls: pls: Winamp playlist;" \
                           "application/smil: smi, smil: SMIL playlist;" \
                           "application/xspf+xml: xspf: XSPF playlist;" \
                           "application/x-xine-plugin: : Xine plugin"

/* Called only once after installation. */
char *NPP_GetMIMEDescription (void) {
  static char *dsc = NULL;
  xine_t      *xine;
  const char  *tmp;
  int          len;

  log ("NPP_GetMIMEDescription()");

  if (dsc)
    return dsc;

  xine = xine_create ();
  if (!xine)
    return NULL;
  
  tmp = xine_get_mime_types (xine);
  len = strlen (tmp) + strlen (BUILTIN_MIMETYPES) + 1;
  dsc = malloc (len);
  if (dsc) {
    strcpy (dsc, tmp);
    strcat (dsc, BUILTIN_MIMETYPES);
  }
  
  xine_exit (xine);

  return dsc;
}

NPError NPP_GetValue (NPP instance, NPPVariable variable, void *value) {
  xine_plugin_t *this;
  
  log ("NPP_GetValue( instance=%p, variable=%d )", instance, variable);

  switch (variable) {
    case NPPVpluginNameString:
      *((char **) value) = "Xine Plugin";
      break;
    case NPPVpluginDescriptionString:
      *((char **) value) = 
                  "Xine Plugin version " VERSION ", "          
                  "(c) <a href=http://www.xinehq.de>The Xine Project</a>.<br>"
                  "Windows Media Player / RealPlayer / QuickTime compatible.";
      break;
#ifdef ENABLE_SCRIPTING 
    case NPPVpluginScriptableNPObject:
      if (!instance || !instance->pdata)
        return NPERR_INVALID_INSTANCE_ERROR;
      
      this = instance->pdata;
      if (!this->object) {
        this->object = NPN_CreateObject (instance, &pluginClass);
        if (!this->object)
          return NPERR_OUT_OF_MEMORY_ERROR;
      }
      
      *((NPObject **) value) = NPN_RetainObject (this->object);
      break;
#endif /* ENABLE_SCRIPTING */ 
    default:
      return NPERR_GENERIC_ERROR;
  }

  return NPERR_NO_ERROR;
}

NPError NPP_SetValue (NPP instance, NPNVariable variable, void *value) {
  log ("NPP_GetValue( instance=%p, variable=%d )", instance, variable);
  
  return NPERR_GENERIC_ERROR;
}

NPError NPP_Initialize (void) {
  log ("NPP_Initialize()");
  
  if (!xine_check_version (XINE_MAJOR_VERSION, XINE_MINOR_VERSION, XINE_SUB_VERSION)) {
    fprintf (stderr, "xine-plugin: incompatible xine-lib version!\n");
    return NPERR_INCOMPATIBLE_VERSION_ERROR;
  }

  /* XInitThreads is buggy when called after other Xlib functions! */
#ifndef XINE_VISUAL_TYPE_X11_2
  if (!XInitThreads ()) {
    log ("XInitThreads() failed!");
    return NPERR_GENERIC_ERROR;
  }
#endif

  return NPERR_NO_ERROR;
}

NPError NPP_New (NPMIMEType mimetype, NPP instance, uint16 mode,
                 int16 argc, char *argn[], char *argv[], NPSavedData *saved) {
  xine_plugin_t       *this;
  pthread_mutexattr_t  attr;
  char                *mrl = NULL;
  int                  loop = 1;
  int                  start = 0;
  int                  autostart = 1;
  int                  geturl = 0;
  char                *demux;
  int                  i;

  log ("NPP_New( mimetype=%s, instance=%p, mode=%d, saved=%p )",
       mimetype, instance, mode, saved);

  if (!instance)
    return NPERR_INVALID_INSTANCE_ERROR;

  for (i = 0; i < argc; i++) {
    log ("param(%s) = \"%s\"", argn[i], argv[i]);
    
    if (argn[i] == NULL)
      continue;

    if (!strcmp (argn[i], "PARAM")) {
      /* Begins list of plugin parameters. */
      if (!mrl)
        geturl = 1;
    }
    else if (!strcasecmp (argn[i], "controls")) {
      /* Ignore Real Player control panels. */
      if (strcasecmp (argv[i], "ImageWindow"))
        return NPERR_INVALID_PARAM;
    }
    else if (!strcasecmp (argn[i], "autostart") ||
             !strcasecmp (argn[i], "autoplay")) {
      if (!strcmp (argv[i], "0") ||
          !strcasecmp (argv[i], "false"))
        autostart = 0;
    }
    else if (!strcasecmp (argn[i], "loop")) {
      if (!strcasecmp (argv[i], "true"))
        loop = 0x7fffffff;
      else if (isdigit (*argv[i]))
        loop = atoi (argv[i]);
    } 
    else if (!strcasecmp (argn[i], "repeat")  ||
             !strcasecmp (argn[i], "numloop") ||
             !strcasecmp (argn[i], "playcount")) {
      loop = atoi (argv[i]);
    }
    else if (!strcasecmp (argn[i], "starttime")) {
      start = strtime (argv[i]) * 1000;
    } 
    else if (!strcasecmp (argn[i], "currentposition")) {
      start = atoi (argv[i]) * 1000;
    }
    else if (!strcasecmp (argn[i], "src")) {
      if (!mrl && *argv[i])
        mrl = argv[i];
    }
    else if (!strcasecmp (argn[i], "url")   ||
             !strcasecmp (argn[i], "href")  ||
             !strcasecmp (argn[i], "qtsrc") ||
             !strcasecmp (argn[i], "filename")) {
      /* Override. */
      if (*argv[i]) {
        mrl = argv[i];
        geturl = 1;
      }
    }
  }

  this = NPN_MemAlloc (sizeof(xine_plugin_t));
  if (!this)
    return NPERR_OUT_OF_MEMORY_ERROR;

  memset (this, 0, sizeof(xine_plugin_t));

  this->instance = instance;
  this->loop = loop;
  this->start = start;
  this->autostart = autostart;

  this->xine = xine_create ();
  if (!this->xine) {
    NPN_MemFree (this);
    return NPERR_GENERIC_ERROR;
  }

  this->display = XOpenDisplay (getenv ("DISPLAY"));
  if (!this->display) {
    log ("XOpenDisplay() failed!");
    xine_exit (this->xine);
    NPN_MemFree (this);
    return NPERR_GENERIC_ERROR;
  }

  XLockDisplay (this->display);
  this->screen = DefaultScreen (this->display);
  XUnlockDisplay (this->display);

  demux = xine_get_demux_for_mime_type (this->xine, mimetype);
  if (demux && *demux)
    snprintf (this->demux, sizeof(this->demux), "%s", demux);

  if (mrl)
    this->track = playlist_insert (&this->list, mrl, this->start);

  pthread_mutexattr_init (&attr);
  pthread_mutexattr_settype (&attr, PTHREAD_MUTEX_RECURSIVE);
  pthread_mutex_init (&this->mutex, &attr);
  pthread_mutexattr_destroy (&attr);   

  instance->pdata = this;
  
  instance_num++;
  
  if (geturl && mrl) {
    if (!strstr (mrl, "://")             ||
        !strncasecmp (mrl, "file://", 7) ||
        !strncasecmp (mrl, "http://", 7))
      NPN_GetURL (instance, mrl, NULL);
  }

  return NPERR_NO_ERROR;
}

NPError NPP_SetWindow (NPP instance, NPWindow *window) {
  xine_plugin_t *this;

  log ("NPP_SetWindow( instance=%p, window=%p )", instance, window);
  
  if (!instance || !instance->pdata)
    return NPERR_INVALID_INSTANCE_ERROR;

  this = instance->pdata;

  if (!window || !window->window) {
    if (this->stream) {
      xine_set_param (this->stream, XINE_PARAM_IGNORE_VIDEO, 1);
    }
    else if (this->window) {
      log ("destroying window.");
      pthread_mutex_lock (&this->mutex);
      
      XLockDisplay (this->display);
      XUnmapWindow (this->display, this->window);
      XDestroyWindow (this->display, this->window);
      XUnlockDisplay (this->display);
      this->window = 0;
      this->parent = 0;
      
      pthread_mutex_unlock (&this->mutex);
    }
  } 
  else {
    if (this->window == 0 || this->parent != (Window) window->window) {
      unsigned int blackpix;
      Window       tmp;

      pthread_mutex_lock (&this->mutex);

      XLockDisplay (this->display);

      if (this->window) {
        log ("window changed.");
        XUnmapWindow (this->display, this->window);
        XDestroyWindow (this->display, this->window);
      }

      blackpix = BlackPixel (this->display, this->screen);

      this->parent = (Window) window->window;
      this->window = XCreateSimpleWindow (this->display, this->parent,
                                          0, 0, window->width, window->height, 
                                          0, blackpix, blackpix);
      XSelectInput (this->display, this->window,
                    ExposureMask | KeyPressMask |
                    ButtonPressMask | PointerMotionMask |
                    StructureNotifyMask | PropertyChangeMask );
      XRaiseWindow (this->display, this->window);
      XMapWindow (this->display, this->window);
      
      XTranslateCoordinates (this->display, this->window,
                             DefaultRootWindow (this->display),
                             0, 0, &this->x, &this->y, &tmp);
      this->w = window->width;
      this->h = window->height;

      XUnlockDisplay (this->display);

      pthread_mutex_unlock (&this->mutex);

      if (this->stream) {
        xine_gui_send_vo_data (this->stream,
                               XINE_GUI_SEND_DRAWABLE_CHANGED, 
                               (void *) this->window);
      }
    }
    else if (this->w != window->width || this->h != window->height) {
      log ("window resized: %dx%d -> %dx%d.",
            this->w, this->h, (int) window->width, (int) window->height);
            
      pthread_mutex_lock (&this->mutex);
      
      XLockDisplay (this->display);
      XResizeWindow (this->display, this->window, 
                     window->width, window->height);
      XUnlockDisplay (this->display);
      
      this->w = window->width;
      this->h = window->height;
      
      pthread_mutex_unlock (&this->mutex);
    }
    else {
      /* Expose */
      if (this->stream && !this->playing)
        xine_gui_send_vo_data (this->stream, 
                               XINE_GUI_SEND_EXPOSE_EVENT, NULL );
    }
    
    if (this->stream)
      xine_set_param (this->stream, XINE_PARAM_IGNORE_VIDEO, 0);
  }

  if (!this->stream) {
    int ret;
    
    ret = stream_create (this);
    if (ret)
      return ret;
      
    /* Start playback now since these protocols
     * are not supported by mozilla & therefore
     * NPP_NewStream() won't be called.
     */
    if (this->track && strstr (this->track->mrl, "://") &&
       (!strncasecmp (this->track->mrl, "mms", 3) ||
        !strncasecmp (this->track->mrl, "pnm", 3) ||
        !strncasecmp (this->track->mrl, "rtsp", 4)))
      player_start (this);
  }

  return NPERR_NO_ERROR;
}

NPError NPP_NewStream (NPP instance, NPMIMEType mimetype,
                       NPStream *stream, NPBool seekable, uint16 *stype) {
  xine_plugin_t *this;
  char          *demux;
  char          *tmp;

  log ("NPP_NewStream( instance=%p, mimetype=%s, stream=%p, seekable=%d )",
       instance, mimetype, stream, seekable);

  if (!instance || !instance->pdata)
    return NPERR_INVALID_INSTANCE_ERROR;

  this = instance->pdata;
  
  if (this->playing) {
    *stype = NP_NORMAL;
    return NPERR_NO_ERROR;
  }

  if (!this->stream) {
    int ret = stream_create (this);
    if (ret)
      return ret;
  }

  /* Check for playlist. */
  this->playlist_type = playlist_type (mimetype, stream->url);
  if (this->playlist_type) {
    log ("playlist detected, requesting a local copy.");
    NPN_Status (instance, "xine-plugin: playlist detected, requesting a local copy.");
    *stype = NP_ASFILEONLY;
    return NPERR_NO_ERROR;
  }

  demux = xine_get_demux_for_mime_type (this->xine, mimetype);
  if (demux && *demux)
    snprintf (this->demux, sizeof(this->demux), "%s", demux);
    
  snprintf (this->base, sizeof(this->base), "%s", stream->url);
  tmp = strrchr (this->base, '/');
  if (tmp)
    *(tmp+1) = '\0';
  
  playlist_free (&this->list);
  this->track = playlist_insert (&this->list, stream->url, 0);

  player_start (this);

  *stype = NP_NORMAL;

  return NPERR_NO_ERROR;
}

int32 NPP_WriteReady (NPP instance, NPStream *stream) {
  log ("NPP_WriteReady( instance=%p, stream=%p )", instance, stream);

  return 0x0fffffff;
}

int32 NPP_Write (NPP instance, NPStream *stream,
                 int32 offset, int32 len, void *buffer) {
  log ("NPP_Write( instance=%p, stream=%p, offset=%d, len=%d, buffer=%p )",
       instance, stream, (int) offset, (int) len, buffer);

  return -1;
}

void NPP_StreamAsFile (NPP instance, NPStream *stream, const char *file) {
  xine_plugin_t *this;
  char          *tmp;

  log ("NPP_StreamAsFile( instance=%p, stream=%p, file=%s )",
       instance, stream, file);

  if (!instance || !instance->pdata)
    return;

  this = instance->pdata;
  
  snprintf (this->base, sizeof(this->base), "%s", stream->url);
  tmp = strrchr (this->base, '/');
  if (tmp)
    *(tmp+1) = '\0';

  playlist_free (&this->list);  
  if (!playlist_load (this->playlist_type, file, &this->list)) {
    log ("no mrl found in \"%s\".", file);
    NPN_Status (instance, "xine-plugin: no mrl found in playlist.");
    return;
  }
  this->track = this->list;

  player_start (this);
}

NPError NPP_DestroyStream (NPP instance, NPStream *stream, NPReason reason) {
  log ("NPP_DestroyStream( instance=%p, stream=%p, reason=%d )",
       instance, stream, reason);

  if (!instance || !instance->pdata)
    return NPERR_INVALID_INSTANCE_ERROR;

  return NPERR_NO_ERROR;
}

void NPP_Print (NPP instance, NPPrint *printinfo) {
  log ("NPP_Print( instance=%p, printinfo=%p )", instance, printinfo);
}

void NPP_URLNotify (NPP instance, const char* url,
                    NPReason reason, void* notifyData) {
  log ("NPP_URLNotify( instance=%p, url=%s, reason=%d, notifyData=%p )",
       instance, url, reason, notifyData);
}

NPError NPP_Destroy (NPP instance, NPSavedData **save) {
  xine_plugin_t *this;

  log ("NPP_Destroy( instance=%p, save=%p )", instance, save);

  if (!instance || !instance->pdata)
    return NPERR_INVALID_INSTANCE_ERROR;

  this = instance->pdata;
 
#ifdef ENABLE_SCRIPTING 
  if (this->object)
    NPN_ReleaseObject (this->object);
#endif

  player_stop (this);

  if (this->osd)
    xine_osd_free (this->osd);

  if (this->event_queue)
    xine_event_dispose_queue (this->event_queue);

  if (this->stream)
    xine_dispose (this->stream);

  if (this->vo_driver)
    xine_close_video_driver (this->xine, this->vo_driver);

  if (this->ao_driver)
    xine_close_audio_driver (this->xine, this->ao_driver);

  if (this->xine)
    xine_exit (this->xine);

  if (this->display) {
    if (this->window) {
      XLockDisplay (this->display);
      XUnmapWindow (this->display, this->window);
      XDestroyWindow (this->display, this->window);
      XUnlockDisplay (this->display);
    }
    XCloseDisplay (this->display);
  }

  playlist_free (&this->list);

  pthread_mutex_destroy (&this->mutex);

  NPN_MemFree (this);

  instance->pdata = NULL;
  
  instance_num--;

  return NPERR_NO_ERROR;
}

void NPP_Shutdown (void) {
  log ("NPP_Shutdown()");
}
