/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * utils.c - assorted utilities
 */

#include <unistd.h>
#include <stdlib.h>
#include <stdio.h>
#include <time.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/ipc.h>
#include <sys/msg.h>
#include <sys/shm.h>
#include <string.h>
#include <gtk/gtk.h>
#include "types.h"
#include "log.h"
#include "utils.h"
#include "preferences.h"
#include "support.h"
#include "history.h"

#if WANT_HAMLIB
#	include <hamlib/rig.h>
# include "hamlib-utils.h"
#endif

gint statusbartimer;
extern GtkWidget *mainwindow;
extern GtkWidget *b4dialog;
extern preferencestype preferences;
extern statetype state;
extern remotetype remote;
extern gchar **bandsplit;
extern gchar **modesplit;
extern gchar **qso;
extern dxcc_data *dxcc;
extern gchar *xlogdir;
extern gint remotetimer;
extern gint clocktimer, savetimer;
extern glong msgid;
extern GHashTable *prefixes;
extern void *shareCall;
extern GList *logwindowlist;
extern gchar *searchpx;

#if WANT_HAMLIB
extern GList *riglist;
#endif


/* calculate location string from latitude and longitude information,
 * returned value has to be freed */
gchar *
setlocation (gdouble lat, gint NS, gdouble lon, gint EW)
{
	gchar *location, *ns, *ew;

	if (NS == 1)
		ns = g_strdup (_("N"));
	else
		ns = g_strdup (_("S"));
	if (EW == 1)
		ew = g_strdup (_("E"));
	else
		ew = g_strdup (_("W"));
	location = g_strdup_printf ("%3.2f%s%3.2f%s", lat, ns, lon, ew);
	g_free (ns);
	g_free (ew);

	return (location);
}

/* update the menu items of an optionmenu */
void
updateoptionmenu (GtkWidget * optionmenu, gchar ** split)
{
	GtkWidget *menu, *menuitem;
	gint index = 0;

	menu = gtk_menu_new ();
	for (;;)
		{
			if (split[index] == NULL)
	break;
			menuitem = gtk_menu_item_new_with_label (split[index]);
			gtk_widget_show (menuitem);
			gtk_menu_shell_append (GTK_MENU_SHELL (menu), menuitem);
			index++;
		}
	gtk_option_menu_set_menu (GTK_OPTION_MENU (optionmenu), menu);
}

/* removing leading and trailing whitespaces from an array of strings */
void
deletespaces (gchar ** split)
{
	gint index = 0;

	for (;;)
		{
			if (split[index] == NULL)
	break;
			g_strstrip (split[index]);
			index++;
		}
}

/* clear statusbar */
static gint
statusbar_timeout (gpointer data)
{
	GtkWidget *statusbar;

	statusbar = lookup_widget (mainwindow, "statusbar");
	gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, _("Ready."));
	gtk_timeout_remove (statusbartimer);
	state.statustimer = FALSE;
	return FALSE;
}

/* change the statusbar */
void
update_statusbar (gchar * string)
{
	GtkWidget *statusbar;

	statusbar = lookup_widget (mainwindow, "statusbar");
	gtk_statusbar_pop (GTK_STATUSBAR (statusbar), 1);
	gtk_statusbar_push (GTK_STATUSBAR (statusbar), 1, string);
	if (state.statustimer)
		gtk_timeout_remove (statusbartimer);
	statusbartimer = gtk_timeout_add (30000, statusbar_timeout, NULL);
	state.statustimer = TRUE;
}

/* Needed by g_hash_table_foreach_remove, to free every key in the hash table */
gboolean
remove_prefixes (gchar * key, gpointer value, gpointer user_data)
{
	g_free (key);
	return (TRUE);
}

/* free mem in state struct */
static void
freestate (void)
{
	g_free (state.mylocation);
	g_free (state.rigrst);
	g_free (state.searchstr);
}

/* free mem in prefs struct */
static void
freeprefs (void)
{
	g_free (preferences.savedir);
	g_free (preferences.b4columns);
}

/* free mem in remote struct */
static void
freeremote (void)
{
	g_free (remote.program);
}

/* cleanup of variables, used at exit */
void
save_windowsize_and_cleanup (void)
{
	gint i;
	GtkWidget *bandoptionmenu, *modeoptionmenu, *view_toolbar, *worked;

	/* free the dxcc array */
	for (i = 0; i < state.countries; i++)
		{
			g_free (dxcc[i].countryname);
			g_free (dxcc[i].continent);
			g_free (dxcc[i].px);
			g_free (dxcc[i].exceptions);
		}
	g_free (dxcc);

	/* free the qso array */
	for (i = 0; i < QSO_FIELDS; i++)
		g_free (qso[i]);
	g_free (qso);

	g_list_free (logwindowlist);

	gtk_window_get_size (GTK_WINDOW(mainwindow), 
		&preferences.width, &preferences.height);
	gtk_window_get_position (GTK_WINDOW(mainwindow), 
		&preferences.x, &preferences.y);
	bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
	modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
	preferences.modeoptionmenu = 
		gtk_option_menu_get_history (GTK_OPTION_MENU (modeoptionmenu));
	preferences.bandoptionmenu = 
		gtk_option_menu_get_history (GTK_OPTION_MENU (bandoptionmenu));
	view_toolbar = lookup_widget (mainwindow, "view_toolbar");
	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(view_toolbar)))
		preferences.viewtoolbar = 1;
	else
		preferences.viewtoolbar = 0;
	worked = lookup_widget (mainwindow, "worked");
	if (gtk_check_menu_item_get_active (GTK_CHECK_MENU_ITEM(worked)))
		preferences.viewb4 = 1;
	else
		preferences.viewb4 = 0;
	gtk_window_get_size (GTK_WINDOW(b4dialog), 
		&preferences.b4width, &preferences.b4height);
	gtk_window_get_position (GTK_WINDOW(b4dialog), 
		&preferences.b4x, &preferences.b4y);
	savepreferences ();
	savehistory ();

#if WANT_HAMLIB
	if (preferences.hamlib > 0)
		stop_hamlib ();
	if (riglist)
		g_list_free (riglist);
#endif

	if (prefixes)
		{
			g_hash_table_foreach_remove (prefixes, (GHRFunc) remove_prefixes, NULL);
			g_hash_table_destroy (prefixes);
		}
	g_strfreev (bandsplit);
	g_strfreev (modesplit);
	g_free (xlogdir);
	if (remotetimer != -1)
		g_source_remove (remotetimer);
	if (savetimer != -1)
		g_source_remove (savetimer);
	if (clocktimer != -1)
		g_source_remove (clocktimer);

	/* remove message queue */
	if (msgid != -1)
		msgctl (msgid, IPC_RMID, 0);

	/* detach shared mem and destroy it */
	if (state.shmid != -1)
		{
			shmdt (shareCall);
			shmctl (state.shmid, IPC_RMID, NULL);
		}

	g_free(searchpx);

	freestate ();
	freeprefs ();
	freeremote ();
}

/* find dot in frequency and return correct band */
gchar *
finddot (gchar * str)
{
	gchar *qsoband, *end, *j, *found = NULL;

	qsoband = g_strdup (str);
	end = qsoband + strlen (qsoband);
	for (j = qsoband; j < end; ++j)
		{
			switch (*j)
			{
				case '.':
				*j = '\0';
				break;
			}
		}
	if (g_ascii_strcasecmp (qsoband, "1") == 0)
		found = "1.8";
	else if (g_ascii_strcasecmp (qsoband, "3") == 0)
		found = "3.5";
	else if (g_ascii_strcasecmp (qsoband, "7") == 0)
		found = "7";
	else if (g_ascii_strcasecmp (qsoband, "29") == 0)
		found = "28";
	else found = qsoband;

	if (atoi (qsoband) > 30 )
	{
		if (atoi (qsoband) > 49 && atoi (qsoband) < 55)
			found = "50";
		else if (atoi (qsoband) > 143 && atoi (qsoband) < 149)
			found = "144";
		else if (atoi (qsoband) > 221 && atoi (qsoband) < 226)
			found = "222";
		else if (atoi (qsoband) > 429 && atoi (qsoband) < 440)
			found = "432";
		else if (atoi (qsoband) > 901 && atoi (qsoband) < 929)
			found = "902";
		else if (atoi (qsoband) > 1239 && atoi (qsoband) < 1301)
			found = "1296";
		else 
			found = qsoband;
	}
	return (g_strdup(found));
}

/* find dot in frequency and return string before the dot */
gchar *
findint (gchar * freq)
{
	gchar *end, *j;

	end = freq + strlen (freq);
	for (j = freq; j < end; ++j)
		{
			switch (*j)
	{
	case '.':
		*j = '\0';
		break;
	}
		}
	return (freq);
}

/* try to find the correct band for dupe checking */
gchar *
findband (gchar * arg)
{
	long long fr;
	gchar *str;

	str = g_new0 (gchar, 100);

	fr = strtoll(arg, (char **)NULL, 10);

	if (fr > 1325)		/* no dot */
		{
			sprintf (str, "%lld", fr);
			if (strlen (str) > 9)
	str[strlen (str) - 9] = '\0';
			else if (strlen (str) > 6)
	str[strlen (str) - 6] = '\0';
			else if (strlen (str) > 3)
	str[strlen (str) - 3] = '\0';
			fr = strtoll(str, (char **)NULL, 10);
		}

	switch (fr)
		{
		case 29:
			fr = 28;
			break;
		case 50 ... 54:
			fr = 50;
			break;
		case 144 ... 148:
			fr = 144;
			break;
		case 222 ... 225:
			fr = 222;
			break;
		case 420 ... 450:
			fr = 420;
			break;
		case 902 ... 928:
			fr = 902;
			break;
		case 1240 ... 1325:
			fr = 1250;
			break;
		default:
			break;
		}
	sprintf (str, "%lld", fr);
	return (str);
}

/* get the current date, returned value has to be freed */
gchar *
getdate (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar datenow[20], *date;
	GError *error;

	time (&current);
	timestruct = localtime (&current);
	strftime (datenow, 20, "%d %b %Y", timestruct);

	if (!g_utf8_validate (datenow, -1, NULL ))
	{
		date = g_locale_to_utf8 (datenow, -1, NULL, NULL, &error);
		if (!date) 
		{
				g_warning (_("Unable to convert '%s' to UTF-8: %s"), datenow, 
					error->message);
				g_error_free (error);
			}
 	}
	else date = g_strdup (datenow);
	
	return (date);
}

/* get the current time, returned value has to be freed */
gchar *
gettime (void)
{
	time_t current;
	struct tm *timestruct = NULL;
	gchar stimenow[20];

	time (&current);
	timestruct = localtime (&current);
	strftime (stimenow, 20, "%H%M", timestruct);
	return (g_strdup (stimenow));
}

/* look up mode in a list of modes */
gchar *
lookup_mode (gchar * selectedmode)
{
	gint modeindex = 0;

	for (;;)
		{
			if (g_ascii_strcasecmp (selectedmode, modesplit[modeindex]) == 0)
	{
		break;
	}
			modeindex++;
		}
	return (g_strdup (modesplit[modeindex]));
}

/* look up band in a list of bands */
gchar *
lookup_band (gchar * selectedband)
{
	gint bandindex = 0;

	for (;;)
		{
			if (g_ascii_strcasecmp (selectedband, bandsplit[bandindex]) == 0)
	{
		break;
	}
			bandindex++;
		}
	return (g_strdup (bandsplit[bandindex]));
}

/* clock which updates a label in the statusbar */
gint
updateclock (void)
{
	GtkWidget *clocklabel;
	GString *timestr = g_string_new ("");
	gchar *nowdate, *nowtime;

	nowdate = getdate ();
	nowtime = gettime ();
	clocklabel = lookup_widget (mainwindow, "clocklabel");
	g_string_printf (timestr, "%s", nowdate);
	timestr = g_string_append_c (timestr, ' ');
	timestr = g_string_append (timestr, nowtime);
	timestr = g_string_append (timestr, " GMT");
	gtk_label_set_text (GTK_LABEL (clocklabel), timestr->str);
	g_string_free (timestr, TRUE);
	g_free (nowdate);
	g_free (nowtime);
	return 1;
}

/* Put color in the right format for XParseColor, so it can be processed by gdk_color_parse */
gchar *
color_parse (gchar * value)
{
	gchar **valuesplit = NULL;
	gchar *rgbsyntax;

	valuesplit = g_strsplit (value, ",", 3);
	rgbsyntax =
		g_strconcat ("rgb:", valuesplit[0], "/", valuesplit[1], "/",
		 valuesplit[2], NULL);
	g_strfreev (valuesplit);
	return (rgbsyntax);
}



gint
autosave (void)
{
	gint i;
	logtype *logwindow;
	gchar *logfile, *temp;
	gboolean message = FALSE;

	for (i = 0; i < g_list_length (logwindowlist); i++)
		{
			logwindow = g_list_nth_data (logwindowlist, i);
			if (logwindow->logchanged)
	{
		logfile =
			g_strconcat (preferences.savedir, G_DIR_SEPARATOR_S,
			 logwindow->logname, ".xlog", NULL);
		savelog (logwindow, logfile, TYPE_FLOG, 1, logwindow->qsos);
		logwindow->logchanged = FALSE;
		gtk_label_set_text (GTK_LABEL (logwindow->label),
						logwindow->logname);
		g_free (logfile);
		message = TRUE;
	}
		}

	if (message)
		{
			temp = g_strdup_printf (_("Log(s) autosaved"));
			update_statusbar (temp);
			g_free (temp);
		}
	return 1;
}

/*
 * check visibility of fields in the loglist and show/hide the corresponding
 * widgets in the QSO frame
 */
void
set_qsoframe (gpointer arg)
{
	gint i, j;
	GtkWidget *endhbox, *powerhbox, *namehbox, *locatorhbox, *qthhbox,
		*unknown1hbox, *unknown2hbox, *powerbutton, *powerlabel, *qslhbox,
		*locatorframe, *remarksvbox;
	logtype *logw = (logtype *) arg;

	for (i = 0; i < QSO_FIELDS; i++)
		{
			for (j = 0; j < logw->columns; j++)
	{
		if (i == logw->logfields[j])
			break;
	}
			if (i == GMTEND)
	{
		endhbox = lookup_widget (mainwindow, "endhbox");
		if (j == logw->columns)
			gtk_widget_hide (endhbox);
		else
			gtk_widget_show (endhbox);
	}
			else if (i == POWER)
	{
		powerhbox = lookup_widget (mainwindow, "powerhbox");
		powerbutton = lookup_widget (mainwindow, "powerbutton");
		powerlabel = lookup_widget (mainwindow, "powerlabel");
		if (j == logw->columns)
			gtk_widget_hide (powerhbox);
		else
			{
				gtk_widget_show (powerhbox);
				if (preferences.hamlib > 0)
		{
			gtk_widget_show (powerbutton);
			gtk_widget_hide (powerlabel);
		}
				else
		{
			gtk_widget_hide (powerbutton);
			gtk_widget_show (powerlabel);
		}
			}
	}
			else if (i == QSLOUT)
	{
		qslhbox = lookup_widget (mainwindow, "qslhbox");
		if (j == logw->columns)
			gtk_widget_hide (qslhbox);
		else
			gtk_widget_show (qslhbox);
	}
			else if (i == NAME)
	{
		namehbox = lookup_widget (mainwindow, "namehbox");
		if (j == logw->columns)
			gtk_widget_hide (namehbox);
		else
			gtk_widget_show (namehbox);
	}
			else if (i == QTH)
	{
		qthhbox = lookup_widget (mainwindow, "qthhbox");
		if (j == logw->columns)
			gtk_widget_hide (qthhbox);
		else
			gtk_widget_show (qthhbox);
	}
			else if (i == LOCATOR)
	{
		locatorhbox = lookup_widget (mainwindow, "locatorhbox");
		locatorframe = lookup_widget (mainwindow, "locatorframe");
		if (j == logw->columns)
			{
				gtk_widget_hide (locatorhbox);
				gtk_widget_hide (locatorframe);
			}
		else
			{
				gtk_widget_show (locatorhbox);
				gtk_widget_show (locatorframe);
			}
	}
			else if (i == U1)
	{
		unknown1hbox = lookup_widget (mainwindow, "unknown1hbox");
		if (j == logw->columns)
			gtk_widget_hide (unknown1hbox);
		else
			gtk_widget_show (unknown1hbox);
	}
			else if (i == U2)
	{
		unknown2hbox = lookup_widget (mainwindow, "unknown2hbox");
		if (j == logw->columns)
			gtk_widget_hide (unknown2hbox);
		else
			gtk_widget_show (unknown2hbox);
	}
			else if (i == REMARKS)
	{
		remarksvbox = lookup_widget (mainwindow, "remarksvbox");
		if (j == logw->columns)
			gtk_widget_hide (remarksvbox);
		else
			gtk_widget_show (remarksvbox);
	}
		}
}

gboolean
fileexist (gchar * path)
{
	FILE *fp = NULL;

	fp = fopen (path, "r");

	if (fp)
		{
			fclose (fp);
			return (TRUE);
		}
	else
		return (FALSE);
}

static gchar *
strshift (gchar * str, size_t shift)
{
  gchar *start, *stop;

  start = str;
  stop = str + strlen (str);
  memmove (start + shift, start, stop - start + 1);

  return (str + shift);
}


gchar *
my_strreplace (gchar * str, const gchar * from, const gchar * to)
{
  gint tolen, fromlen, shift;
  gchar *start, *stop, *p;

  tolen = strlen (to);
  fromlen = strlen (from);
  shift = tolen - fromlen;
  start = str;
  stop = str + strlen (str);

  while (NULL != (p = strstr (start, from)))
    {
      start = strshift (p + fromlen, shift);
      stop = stop + shift;
      memmove (p, to, tolen);
    }

  return str;
}
