/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001 - 2004 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* gui_searchdialog.c - creation and destruction of the search dialog
 *
 * we create a dialog where a string can be entered. If OK is clicked this
 * string is checked against the callsigns in all logs. If there is a partial
 * match, the QSO is selected. If there is no match, a warinig is displayed.
 */
 
#include <gtk/gtk.h>
#include <string.h>

#include "gui_searchdialog.h"
#include "support.h"
#include "types.h"
#include "log.h"

extern statetype state;
extern GList *searchhistory;
extern GList *logwindowlist;
extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;

#define SEARCHHISTORY 10

static void
on_searchokbutton_clicked (GtkButton * button, gpointer user_data)
{
	GtkWidget *dialog_vbox, *dialog_action_area, *searchresultokbutton, 
		*searchdialog, *searchresultdialog, *resultlabel, 
		*searchcheckbutton, *searchcombo;
	GdkPixbuf *searchresultdialog_icon_pixbuf;
	gchar *searchstr, *upsearchstr, *callsign, *current, *nr, *labeltext;
	gint i;
	gboolean valid, result = FALSE, searchopen;
	GList *node;
	logtype *logwindow;
	GtkTreeModel *model;
	GtkTreeIter iter;
	GtkTreeSelection *selection;
	GtkTreePath *path;

	searchdialog = gtk_widget_get_toplevel (GTK_WIDGET (button));
	searchcombo = lookup_widget (searchdialog, "searchcombo");
	searchcheckbutton = lookup_widget (searchdialog, "searchcheckbutton");

	searchopen = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(searchcheckbutton));
	searchstr =	gtk_editable_get_chars 
		(GTK_EDITABLE (GTK_BIN(searchcombo)->child), 0, -1);

	node = g_list_find_custom (searchhistory, searchstr, (GCompareFunc) strcmp);
	if (!node)
	{
		searchhistory =	g_list_prepend (searchhistory, g_strdup(searchstr));
		gtk_combo_box_append_text (GTK_COMBO_BOX (searchcombo), searchstr);
	}
	if (g_list_length (searchhistory) > SEARCHHISTORY)
		searchhistory = g_list_remove (searchhistory, g_list_last (searchhistory)->data);

	upsearchstr = g_ascii_strup (searchstr, -1);

	for (i = 0; i < g_list_length (logwindowlist); i++)
	{
		logwindow = g_list_nth_data (logwindowlist, i);
		model = gtk_tree_view_get_model (GTK_TREE_VIEW(logwindow->treeview));
		valid = gtk_tree_model_get_iter_first (model, &iter);
		while (valid)
		{
			gtk_tree_model_get (model, &iter, CALL, &callsign, -1);
			if (g_strrstr (callsign, upsearchstr))
			{
				/* construct an id for this search */
				current = g_strdup_printf ("%d", i);
				gtk_tree_model_get (model, &iter, NR, &nr, -1);
				current =	g_strconcat (current, "/", nr, NULL);
				if (!g_strrstr (state.searchstr, current))
				{
					gtk_notebook_set_current_page	(GTK_NOTEBOOK (mainnotebook), i);
					selection =	gtk_tree_view_get_selection(GTK_TREE_VIEW(logwindow->treeview));
					gtk_tree_selection_select_iter(selection, &iter);
					path = gtk_tree_model_get_path (model, &iter);
					gtk_tree_view_scroll_to_cell(GTK_TREE_VIEW (logwindow->treeview), path, NULL, TRUE, 0.5, 0.0);
					gtk_tree_path_free (path);
					/* add id to the array */
					state.searchstr =	g_strconcat (state.searchstr, current, ",", NULL);
					result = TRUE;
					break;
				}
			}
			valid = gtk_tree_model_iter_next (model, &iter);
		}
		if (result)
			break;
	}

	if (!result)
	{
		state.searchstr = g_strdup ("");

		searchresultdialog = gtk_dialog_new ();
		gtk_widget_set_size_request (searchresultdialog, 350, 150); 
		gtk_window_set_title (GTK_WINDOW (searchresultdialog), _("xlog - searchresult"));
		gtk_window_set_position (GTK_WINDOW (searchresultdialog), GTK_WIN_POS_MOUSE);
		searchresultdialog_icon_pixbuf = create_pixbuf ("xlog.png");
		if (searchresultdialog_icon_pixbuf)
			{
				gtk_window_set_icon (GTK_WINDOW (searchresultdialog), searchresultdialog_icon_pixbuf);
				g_object_unref (searchresultdialog_icon_pixbuf);
			}
	
		dialog_vbox = GTK_DIALOG (searchresultdialog)->vbox;
		gtk_widget_show (dialog_vbox);
	
		labeltext =	g_strdup_printf (_("\'%s\' was not found"), searchstr);
		resultlabel = gtk_label_new (labeltext);
		g_free (labeltext);
		gtk_widget_show (resultlabel);
		gtk_box_pack_start (GTK_BOX (dialog_vbox), resultlabel, TRUE, TRUE, 0);
		gtk_label_set_justify (GTK_LABEL (resultlabel), GTK_JUSTIFY_CENTER);
		gtk_label_set_line_wrap (GTK_LABEL (resultlabel), TRUE);
		gtk_misc_set_padding (GTK_MISC (resultlabel), 30, 20);

		dialog_action_area = GTK_DIALOG (searchresultdialog)->action_area;
		gtk_widget_show (dialog_action_area);
		gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_SPREAD);

		searchresultokbutton = gtk_button_new_from_stock ("gtk-ok");
		gtk_widget_show (searchresultokbutton);
		gtk_dialog_add_action_widget (GTK_DIALOG (searchresultdialog), searchresultokbutton, GTK_RESPONSE_OK);
		GTK_WIDGET_SET_FLAGS (searchresultokbutton, GTK_CAN_DEFAULT);

		gtk_dialog_run (GTK_DIALOG(searchresultdialog));
		gtk_editable_delete_text (GTK_EDITABLE (GTK_BIN(searchcombo)->child), 0, -1);
		gtk_widget_destroy (searchresultdialog);
	}

	if (!searchopen)
	{
		state.searchstr = g_strdup ("");
		gtk_widget_set_sensitive (mainwindow, 1);
		gtk_widget_destroy (searchdialog);
	}

	g_free (searchstr);
	g_free (upsearchstr);
}


/* search cancel */
static void
on_searchcancelbutton_clicked (GtkButton * button, gpointer user_data)
{
	state.searchstr = g_strdup ("");
	gtk_widget_set_sensitive (mainwindow, 1);
	gtk_widget_destroy (gtk_widget_get_toplevel (GTK_WIDGET (button)));
}

/* search changed, reset saved state */
static void
on_searchcombo_changed (GtkEditable * editable, gpointer user_data)
{
	state.searchstr = g_strdup ("");
}

/* toggle 'keep this dialog open' */
static void
on_searchcheckbutton_toggled (GtkToggleButton * togglebutton,
						gpointer user_data)
{
	gboolean searchopen;

	searchopen = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON(togglebutton));
	if (searchopen)
		gtk_widget_set_sensitive (mainwindow, 1);
	else
		gtk_widget_set_sensitive (mainwindow, 0);
}

static gboolean
on_searchdialog_delete_event (GtkWidget * widget, GdkEvent * event,
						gpointer user_data)
{
	gtk_widget_set_sensitive (mainwindow, 1);
	return FALSE;
}

void
on_menu_search_activate (GtkMenuItem * menuitem, gpointer user_data)
{
	GtkWidget *searchdialog, *dialog_vbox, *vbox, *searchlabel, *searchcombo,
		*searchcheckbutton, *dialog_action_area, *searchcancelbutton, 
		*searchokbutton;
	GdkPixbuf *searchdialog_icon_pixbuf;
	gint i, num;
	gchar *s;

	searchdialog = gtk_dialog_new ();
	gtk_window_set_title (GTK_WINDOW (searchdialog), _("xlog - search"));
	gtk_window_set_position (GTK_WINDOW (searchdialog), GTK_WIN_POS_MOUSE);
	searchdialog_icon_pixbuf = create_pixbuf ("xlog.png");
	if (searchdialog_icon_pixbuf)
		{
			gtk_window_set_icon (GTK_WINDOW (searchdialog), searchdialog_icon_pixbuf);
			g_object_unref (searchdialog_icon_pixbuf);
		}

	dialog_vbox = GTK_DIALOG (searchdialog)->vbox;
	gtk_widget_show (dialog_vbox);

	vbox = gtk_vbox_new (FALSE, 10);
	gtk_widget_show (vbox);
	gtk_box_pack_start (GTK_BOX (dialog_vbox), vbox, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (vbox), 20);

	searchlabel = gtk_label_new (_("Select or enter a callsign (or part of a callsign) to search for in the log"));
	gtk_widget_show (searchlabel);
	gtk_box_pack_start (GTK_BOX (vbox), searchlabel, FALSE, FALSE, 0);
	gtk_label_set_justify (GTK_LABEL (searchlabel), GTK_JUSTIFY_CENTER);
	gtk_label_set_line_wrap (GTK_LABEL (searchlabel), TRUE);

	searchcombo = gtk_combo_box_entry_new_text ();
	gtk_widget_show (searchcombo);
	gtk_box_pack_start (GTK_BOX (vbox), searchcombo, FALSE, FALSE, 0);

	searchcheckbutton = gtk_check_button_new_with_mnemonic (_("Keep this dialog open"));
	gtk_widget_show (searchcheckbutton);
	gtk_box_pack_start (GTK_BOX (vbox), searchcheckbutton, FALSE, FALSE, 0);

	dialog_action_area = GTK_DIALOG (searchdialog)->action_area;
	gtk_widget_show (dialog_action_area);
	gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area), GTK_BUTTONBOX_SPREAD);

	searchcancelbutton = gtk_button_new_from_stock ("gtk-cancel");
	gtk_widget_show (searchcancelbutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (searchdialog), searchcancelbutton, GTK_RESPONSE_CANCEL);
	GTK_WIDGET_SET_FLAGS (searchcancelbutton, GTK_CAN_DEFAULT);

	searchokbutton = gtk_button_new_from_stock ("gtk-ok");
	gtk_widget_show (searchokbutton);
	gtk_dialog_add_action_widget (GTK_DIALOG (searchdialog), searchokbutton, GTK_RESPONSE_OK);
	GTK_WIDGET_SET_FLAGS (searchokbutton, GTK_CAN_DEFAULT);

	g_signal_connect ((gpointer) searchdialog, "delete_event",
										G_CALLBACK (on_searchdialog_delete_event),
										NULL);
	g_signal_connect ((gpointer) searchcombo, "changed",
										G_CALLBACK (on_searchcombo_changed),
										NULL);
	g_signal_connect ((gpointer) searchcheckbutton, "toggled",
										G_CALLBACK (on_searchcheckbutton_toggled),
										NULL);
	g_signal_connect ((gpointer) searchcancelbutton, "clicked",
										G_CALLBACK (on_searchcancelbutton_clicked),
										NULL);
	g_signal_connect ((gpointer) searchokbutton, "clicked",
										G_CALLBACK (on_searchokbutton_clicked),
										NULL);

	/* Store pointers to all widgets, for use by lookup_widget(). */
	GLADE_HOOKUP_OBJECT_NO_REF (searchdialog, searchdialog, "searchdialog");
	GLADE_HOOKUP_OBJECT (searchdialog, searchcombo, "searchcombo");
	GLADE_HOOKUP_OBJECT (searchdialog, searchcheckbutton, "searchcheckbutton");

	if (searchhistory)
	{
		num = g_list_length (searchhistory);
		for (i = 0; i < num; i++)
		{
			s = g_list_nth_data (searchhistory, i);
			gtk_combo_box_append_text (GTK_COMBO_BOX (searchcombo), s);
		}
	}
	gtk_widget_set_sensitive (mainwindow, 0);
	gtk_widget_show (searchdialog);
}
