/*
 * xlog - GTK+ logging program for amateur radio operators
 * Copyright (C) 2001-2005 Joop Stakenborg <pg4i@amsat.org>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU Library General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * remote.c - remote data support
 */

/*
 * Messages send to xlog should consist of fields, separated by the '\1' character.
 * Fields are started with a descriptor, followed by a colon (':'). Possible fields are:
 *
 ****************************************************************************
 * program:<name of program which sends the QSO information>
 * version:<version string for the message queue, must be '1' for now>
 * date:<date of QSO, preferable 'dd mmm yyyy'>
 * time:<start time of QSO, preferably in GMT >
 * endtime:<end time of QSO, preferably GMT (mandatory in some countries)>
 * call:<callsign of your contact (will be converted to uppercase)>
 * mhz:<frequency in MHz>
 * mode:<any one of CW,SSB,RTTY,PSK31,etc (will be converted to uppercase)>
 * tx:<report (RST) which you have send>
 * rx:<report (RST) which you have received>
 * name:<name of the operator you have contacted>
 * qth:<town of the operator you have contacted>
 * notes:<additional notes>
 * power:<power you have used (mandatory in some countries)>
 * locator:<QRA locator, as used in VHF QSO's>
 * free1: <information to put in freefield1>
 * free2: <information to put in freefield2>
 ****************************************************************************
 *
 * See sendtoxlog.c, included with the sources for examples.
 */

#include <stdlib.h>
#include <sys/types.h>
#include <errno.h>
#include <gtk/gtk.h>
#include <string.h>

#if defined(HAVE_SYS_IPC_H)
#include <sys/ipc.h>
#include <sys/msg.h>
#endif

#include "remote.h"
#include "types.h"
#include "log.h"
#include "utils.h"
#include "support.h"

#if WANT_HAMLIB
# include "hamlib-utils.h"
#endif

msgtype msgbuf;
remotetype remote;
extern statetype state;
extern preferencestype preferences;
extern glong msgid;
extern GtkWidget *mainwindow;
extern GtkWidget *mainnotebook;
extern gchar **qso;
extern gchar **bandsplit;
extern gchar **modesplit;
extern GList *logwindowlist;

/* find colon in a string and return the right part */
static gchar *
getargument (gchar * remotestring)
{
	gchar **split, *found = NULL;

	split = g_strsplit (remotestring, ":", 2);
	if (split[1])
		found = g_strdup (split[1]);
	g_strfreev (split);
	return (g_strdup (found));
}


static void
addtolog_or_qsoframe (gint type, gchar * entry)
{
	GtkWidget *bandoptionmenu, *modeoptionmenu,
		*bandentry, *modeentry, *endhbox, *namehbox, *qthhbox,
		*locatorhbox, *powerhbox, *unknown1hbox, *unknown2hbox,
		*dateentry, *gmtentry, *callentry, *endentry, *rstentry,
		*myrstentry, *powerentry, *nameentry, *qthentry,
		*locatorentry, *unknownentry1, *unknownentry2, *remtv,
		*remarksvbox;
	GtkTextBuffer *b;
	gchar *temp, **remoteinfo, *argument = NULL, *remarks;
	gint i, j = 0, bandindex, modeindex, err = 0;
	logtype *logw;
	GtkTreeIter iter;
	GtkTreeModel *model;
	GtkTreePath *path;
#if WANT_HAMLIB
	GString *digits = g_string_new ("");
#endif

	if (type == 88 && entry && (strlen (entry) > 0))
	{
		logw = g_list_nth_data (logwindowlist, gtk_notebook_get_current_page (GTK_NOTEBOOK (mainnotebook)));

		for (i = 0; i < QSO_FIELDS; i++) qso[i] = g_strdup ("");
		remarks = g_strdup ("");

		bandentry = lookup_widget (mainwindow, "bandentry");
		bandoptionmenu = lookup_widget (mainwindow, "bandoptionmenu");
		modeentry = lookup_widget (mainwindow, "modeentry");
		modeoptionmenu = lookup_widget (mainwindow, "modeoptionmenu");
		endhbox = lookup_widget (mainwindow, "endhbox");
		namehbox = lookup_widget (mainwindow, "namehbox");
		qthhbox = lookup_widget (mainwindow, "qthhbox");
		locatorhbox = lookup_widget (mainwindow, "locatorhbox");
		powerhbox = lookup_widget (mainwindow, "powerhbox");
		unknown1hbox = lookup_widget (mainwindow, "unknown1hbox");
		unknown2hbox = lookup_widget (mainwindow, "unknown2hbox");

		remoteinfo = g_strsplit (entry, "\1", 0);
		for (;;)
		{
			if (remoteinfo[j] == NULL)
			break;

			if (g_ascii_strncasecmp (remoteinfo[j], "version:", 8) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
					remote.version = atoi (argument);
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "program:", 8) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
					remote.program = g_strdup (argument);
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "date:", 5) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
						qso[DATE] = g_strndup (argument, 15);
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "time:", 5) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
					qso[GMT] = g_strndup (argument, 8);
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "call:", 5) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					g_ascii_strup (argument, -1);
					qso[CALL] = g_strndup (argument, 15);
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "mhz:", 4) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (g_ascii_strcasecmp (argument, "HAMLIB") == 0)
#if WANT_HAMLIB
					{
						get_frequency ();
						digits = convert_frequency ();
						qso[BAND] = g_strdup (digits->str);
						g_string_free (digits, TRUE);
					}
					else
						qso[BAND] = g_strndup (argument, 15);
#else
						err = 1;
					else
						qso[BAND] = g_strndup (argument, 15);
#endif
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "mode:", 5) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (g_ascii_strcasecmp (argument, "HAMLIB") == 0)
#if WANT_HAMLIB
					{
						get_mode ();
						qso[MODE] = rigmode (state.rigmode);
					}
					else
					{
						g_ascii_strup (argument, -1);
						qso[MODE] = g_strndup (argument, 8);
					}
#else
						err = 1;
					else
					{
						g_ascii_strup (argument, -1);
						qso[MODE] = g_strndup (argument, 8);
					}
#endif
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "tx:", 3) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (g_ascii_strcasecmp (argument, "HAMLIB") == 0)
#if WANT_HAMLIB
					{
						get_smeter ();
						qso[RST] = g_strdup (state.rigrst);
					}
					else
						qso[RST] = g_strndup (argument, 15);
#else
						err = 1;
					else
						qso[RST] = g_strndup (argument, 15);
#endif
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "rx:", 3) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
					qso[MYRST] = g_strndup (argument, 15);
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "endtime:", 8) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (GTK_WIDGET_VISIBLE (endhbox))
						qso[GMTEND] = g_strndup (argument, 8);
				} /* don't bother adding endtime to remarks field */
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "name:", 5) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (GTK_WIDGET_VISIBLE (namehbox))
						qso[NAME] = g_strndup (argument, 30);
					else
					{
						if (g_ascii_strcasecmp (remarks, "") == 0)
							remarks = g_strdup (argument);
						else
							remarks = g_strconcat (remarks, ", ", argument, NULL);
					}
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "qth:", 4) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (GTK_WIDGET_VISIBLE (qthhbox))
						qso[QTH] = g_strndup (argument, 30);
					else
					{
						if (g_ascii_strcasecmp (remarks, "") == 0)
							remarks = g_strdup (argument);
						else
							remarks = g_strconcat (remarks, ", ", argument, NULL);
					}
				}
			}	

			else if (g_ascii_strncasecmp (remoteinfo[j], "notes:", 6) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (g_ascii_strcasecmp (remarks, "") == 0)
						remarks = g_strdup (argument);
					else
					remarks = g_strconcat (remarks, ", ", argument, NULL);
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "power:", 6) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (g_ascii_strcasecmp (argument, "HAMLIB") == 0)
#if WANT_HAMLIB
					{
						get_powerlevel ();
						if (GTK_WIDGET_VISIBLE (powerhbox))
							qso[POWER] = g_strdup_printf ("%d", state.rigpower * 256 / 100000);
					}
					else
					{
						if (GTK_WIDGET_VISIBLE (powerhbox))
							qso[POWER] = g_strndup (argument, 8);
					} /* don't bother adding power to the remarks field */
#else
						err = 1;
					else
					{
						if (GTK_WIDGET_VISIBLE (powerhbox))
							qso[POWER] = g_strndup (argument, 8);
					}
#endif
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "locator:", 8) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (GTK_WIDGET_VISIBLE (locatorhbox))
						qso[LOCATOR] = g_strndup (argument, 30);
					else
					{
						if (g_ascii_strcasecmp (remarks, "") == 0)
							remarks = g_strdup (argument);
						else
							remarks = g_strconcat (remarks, ", ", argument, NULL);
					}
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "free1:", 6) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{	
	 				if (GTK_WIDGET_VISIBLE (unknown1hbox))
						qso[U1] = g_strndup (argument, 30);
					else
					{
						if (g_ascii_strcasecmp (remarks, "") == 0)
							remarks = g_strdup (argument);
						else
							remarks = g_strconcat (remarks, ", ", argument, NULL);
					}
				}
			}

			else if (g_ascii_strncasecmp (remoteinfo[j], "free2:", 6) == 0)
			{
				if ((argument = getargument (remoteinfo[j])))
				{
					if (GTK_WIDGET_VISIBLE (unknown2hbox))
						qso[U2] = g_strndup (argument, 30);
					else
					{
						if (g_ascii_strcasecmp (remarks, "") == 0)
							remarks = g_strdup (argument);
						else
							remarks = g_strconcat (remarks, ", ", argument, NULL);
					}
				}
			}
		j++;
		}
		g_strfreev (remoteinfo);
		g_free (argument);

		if (g_ascii_strcasecmp (remarks, "") != 0)
			qso[REMARKS] = g_strndup (remarks, 80);
		g_free (remarks);

			/* try to fill in empty fields */
		if (g_ascii_strcasecmp (qso[DATE], "") == 0)
			qso[DATE] = getdate ();
		if (g_ascii_strcasecmp (qso[GMT], "") == 0)
			qso[GMT] = gettime ();
		if (g_ascii_strcasecmp (qso[BAND], "") == 0)
		{
			if (preferences.bandseditbox == 0)
			{
				bandindex = gtk_combo_box_get_active (GTK_COMBO_BOX(bandoptionmenu));
				qso[BAND] = lookup_band (bandindex);
			}
			else
				qso[BAND] =	gtk_editable_get_chars (GTK_EDITABLE (bandentry), 0, -1);
		}
		if (g_ascii_strcasecmp (qso[MODE], "") == 0)
		{
			if (preferences.modeseditbox == 0)
			{
				modeindex = gtk_combo_box_get_active (GTK_COMBO_BOX(modeoptionmenu));
				qso[MODE] = lookup_mode (modeindex);
			}
			else
				qso[MODE] =	gtk_editable_get_chars (GTK_EDITABLE (modeentry), 0, -1);
		}
	
		if (preferences.remoteadding == 1 && logw)
		{
			qso[NR] = g_strdup_printf ("%d", ++logw->qsos);
	
			/* add the QSO */
			model = gtk_tree_view_get_model (GTK_TREE_VIEW(logw->treeview));
			gtk_tree_store_prepend (GTK_TREE_STORE (model), &iter, NULL);
			gtk_tree_store_set (GTK_TREE_STORE (model), &iter,
				NR,	qso[NR], DATE, qso[DATE], GMT, qso[GMT], GMTEND, qso[GMTEND],
				CALL, qso[CALL], BAND, qso[BAND],	MODE, qso[MODE], RST, qso[RST],
				MYRST, qso[MYRST], QSLOUT, qso[QSLOUT], QSLIN, qso[QSLIN],
				POWER, qso[POWER], NAME, qso[NAME], QTH, qso[QTH], LOCATOR,
				qso[LOCATOR], U1, qso[U1], U2, qso[U2], REMARKS, qso[REMARKS], -1);
	
			/* scroll there */
			path = gtk_tree_path_new_from_string ("0");
			gtk_tree_view_scroll_to_cell (GTK_TREE_VIEW(logw->treeview),	path, NULL, TRUE, 1.0, 0.0);
			gtk_tree_path_free (path);
			
      /* save with every log change */
			if (preferences.saving == 2)
			{
				savelog (logw, logw->filename, TYPE_FLOG, 1, logw->qsos);
				logw->logchanged = FALSE;
				temp = g_strdup_printf (_("QSO %s added to %s log, log saved"), qso[NR], logw->logname);
			}
			else
			{ /* autosave */
				temp = g_strdup_printf (_("Remote data received from %s (#%d), QSO %s added"), remote.program, ++remote.nr, qso[NR]);
				logw->logchanged = TRUE;
				gtk_label_set_text (GTK_LABEL (logw->label),
				g_strconcat (logw->logname, " *", NULL));
			}
			update_statusbar (temp);
		}
		else
		{
			if (g_ascii_strcasecmp (qso[DATE], "") == 0)
				qso[DATE] = getdate ();
			if (g_ascii_strcasecmp (qso[GMT], "") == 0)
				qso[GMT] = gettime ();
			dateentry = lookup_widget (mainwindow, "dateentry");
			if (strlen(qso[DATE]) > 0)
				gtk_entry_set_text (GTK_ENTRY (dateentry), qso[DATE]);
			gmtentry = lookup_widget (mainwindow, "gmtentry");
			if (strlen(qso[GMT]) > 0)
				gtk_entry_set_text (GTK_ENTRY (gmtentry), qso[GMT]);
			if (GTK_WIDGET_VISIBLE (endhbox))
			{
				endentry = lookup_widget (mainwindow, "endentry");
				if (strlen(qso[GMTEND]) > 0)
					gtk_entry_set_text (GTK_ENTRY (endentry), qso[GMTEND]);
			}
			callentry = lookup_widget (mainwindow, "callentry");
			if (strlen(qso[CALL]) > 0)
				gtk_entry_set_text (GTK_ENTRY (callentry), qso[CALL]);
			if (preferences.modeseditbox == 1)
				if (strlen(qso[MODE]) > 0)
					gtk_entry_set_text (GTK_ENTRY (modeentry), qso[MODE]);
			if (preferences.bandseditbox == 1)
				if (strlen(qso[BAND]) > 0)
					gtk_entry_set_text (GTK_ENTRY (bandentry), qso[BAND]);
			rstentry = lookup_widget (mainwindow, "rstentry");
			if (strlen(qso[RST]) > 0)
				gtk_entry_set_text (GTK_ENTRY (rstentry), qso[RST]);
			myrstentry = lookup_widget (mainwindow, "myrstentry");
			if (strlen(qso[MYRST]) > 0)
				gtk_entry_set_text (GTK_ENTRY (myrstentry), qso[MYRST]);
			if (GTK_WIDGET_VISIBLE (powerhbox))
			{
				powerentry = lookup_widget (mainwindow, "powerentry");
				if (strlen(qso[POWER]) > 0)
					gtk_entry_set_text (GTK_ENTRY (powerentry), qso[POWER]);
			}
			if (GTK_WIDGET_VISIBLE (namehbox))
			{
				nameentry = lookup_widget (mainwindow, "nameentry");
				if (strlen(qso[NAME]) > 0)
					gtk_entry_set_text (GTK_ENTRY (nameentry), qso[NAME]);
			}
			if (GTK_WIDGET_VISIBLE (qthhbox))
			{
				qthentry = lookup_widget (mainwindow, "qthentry");
				if (strlen(qso[QTH]) > 0)
					gtk_entry_set_text (GTK_ENTRY (qthentry), qso[QTH]);
			}
			if (GTK_WIDGET_VISIBLE (locatorhbox))
			{
				locatorentry = lookup_widget (mainwindow, "locatorentry");
				if (strlen(qso[LOCATOR]) > 0)
					gtk_entry_set_text (GTK_ENTRY (locatorentry), qso[LOCATOR]);
			}
			if (GTK_WIDGET_VISIBLE (unknown1hbox))
			{
				unknownentry1 = lookup_widget (mainwindow, "unknownentry1");
				if (strlen(qso[U1]) > 0)
					gtk_entry_set_text (GTK_ENTRY (unknownentry1), qso[U1]);
			}
			if (GTK_WIDGET_VISIBLE (unknown2hbox))
			{
				unknownentry2 = lookup_widget (mainwindow, "unknownentry2");
				if (strlen(qso[U2]) > 0)
					gtk_entry_set_text (GTK_ENTRY (unknownentry2), qso[U2]);
			}
			remarksvbox = lookup_widget (mainwindow, "remarksvbox");
			if (GTK_WIDGET_VISIBLE (remarksvbox))
			{
				remtv = lookup_widget (mainwindow, "remtv");
				b = gtk_text_view_get_buffer (GTK_TEXT_VIEW (remtv));
				if (strlen(qso[REMARKS]) > 0)
					gtk_text_buffer_set_text (b, qso[REMARKS], -1);
			}
			temp = g_strdup_printf (_("Remote data received from %s (#%d)"), remote.program, ++remote.nr);
		}
		if (err == 1)
			update_statusbar (_("Warning: No hamlib support for remote data"));
		else
			update_statusbar (temp);
		g_free (temp);

		remote.program = g_strdup ("unknown");
		remote.version = 0;
	}
}

gint
remote_entry (void)
{
	ssize_t status = -1;
#if defined(HAVE_SYS_IPC_H)
	glong msgtyp = 0;

	status = msgrcv (msgid, (void *) &msgbuf, 1024, msgtyp,
			 MSG_NOERROR | IPC_NOWAIT);
#endif
	if (status != -1)
		addtolog_or_qsoframe (msgbuf.mtype, msgbuf.mtext);
	return 1;
}
