/*\
|*| PLatform dependent code for accessing CDROM units.
|*| This one's for Solaris.  I hope it works.
\*/
#ifndef CDROMSOLARIS_H
#define CDROMSOLARIS_H
/*\
|*| Solaris specific code
\*/

#define GET_LBA(entry) ((entry.cdte_format == CDROM_LBA) ? \
				entry.cdte_addr.lba : \
					((entry.cdte_addr.msf.minute * 60 + \
					entry.cdte_addr.msf.second) * 75 + \
					entry.cdte_addr.msf.frame))

/*\ Translate speed to ioctl constant.  Solaris only. \*/
static int
trans_speed(int speed)
{
	if (speed <=  1) return CDROM_NORMAL_SPEED;
	if (speed <=  2) return CDROM_DOUBLE_SPEED;
	if (speed <=  4) return CDROM_QUAD_SPEED;
	if (speed <= 12) return CDROM_TWELVE_SPEED;
	return CDROM_MAXIMUM_SPEED;
}

/*\
|*| Open cdrom device
|*|  Return -1 on error.
\*/
static int
cdrom_open(const char *device, int &flags)
{
	int fd;
	int flg = *flags;

	fd = open(device, O_RDONLY);
	if (fd < 0) return -1;
	if (!(flg & FLAG_FAIL_BLK) &&
	    (ioctl(fd, CDROMSBLKMODE, CDROM_BLK_2352) < 0)) {
		fprintf(stderr, "xmms-cdread: Failed to set cdrom audio block mode: %s\n",
				strerror(errno));
		flg |= FLAG_FAIL_BLK;
	}
	if (!(flg & FLAG_FAIL_SPD) &&
	    (ioctl(fd, CDROMSDRVSPEED, trans_speed(cd_cfg.cdrom_speed)) < 0)) {
		fprintf(stderr, "xmms-cdread: Failed to set cdrom speed: %s\n",
				strerror(errno));
		if (errno == ENOTTY) {
			close(fd);
			return -1;
		}
		flg |= FLAG_FAIL_SPD;
	}
	*flags = flg;
	return fd;
}

/*\
|*| Close cdrom device
|*|  Return -1 on error.
\*/
static int
cdrom_close(int cdfd)
{
	ioctl(cdfd, CDROMSBLKMODE, CDROM_BLK_512);
	return close(cdfd);
}

/*\ Read the toc into:
|*|  cd->first_trk .. cd->last_trk
|*|  cd->lba[..], cd->data[..]
|*|
|*| Return -1 on error.
\*/
static int
cdrom_read_toc(struct cd_struct *cd, int cdfd)
{
	int i;
	struct cdrom_tochdr toc_hdr;
	struct cdrom_tocentry entry;

	if (ioctl(cdfd, CDROMREADTOCHDR, &toc_hdr) < 0) {
		if (errno != EIO)
			fprintf(stderr, "xmms-cdread: Failed to read cd table of contents: %s\n",
					strerror(errno));
		cd->first_trk = 1;
		cd->last_trk = 0;
		return -1;
	}
	cd->first_trk = toc_hdr.cdth_trk0;
	cd->last_trk = toc_hdr.cdth_trk1;
	i = cd->last_trk + 1;
	entry.cdte_track = CDROM_LEADOUT;
	entry.cdte_format = CDROM_MSF;
	ioctl(cdfd, CDROMREADTOCENTRY, &entry);
	cd->lba[i] = GET_LBA(entry);
	cd->data[i] = entry.cdte_ctrl & 4;
	while (--i >= cd->first_trk) {
		entry.cdte_track = i;
		entry.cdte_format = CDROM_MSF;
		ioctl(cdfd, CDROMREADTOCENTRY, &entry);
		cd->lba[i] = GET_LBA(entry);
		cd->data[i] = entry.cdte_ctrl & 4;
	}
	return 0;
}

/*\ Read btw frames of audio data into buf,
|*|  from device cdfd, at position lba
|*|  Return number of successfully read frames, -1 on error.
\*/
static int
cdrom_read_audio(int cdfd, int lba, char *buf, int btw)
{
	int rtr = 3;
	do {
		struct cdrom_cdda cdda;
		cdda.cdda_data = (caddr_t)buf;
		cdda.cdda_length = btw;
		cdda.cdda_addr = lba;
		cdda.cdda_subcode = CDROM_DA_NO_SUBCODE;
		if (ioctl(cdfd, CDROMCDDA, &cdda) >= 0)
			return cdda.cdda_length;
	} while (--rtr >= 0);
	return -1;
}

/*\ Play audio from lba address from, to lba address to
|*|  return -1 on failure
\*/
static int
cdrom_play_lba(int cdfd, int from, int to)
{
	struct cdrom_msf cmsf;
	cmsf.cdmsf_min0 = from / (75 * 60);
	cmsf.cdmsf_sec0 = (from / 75) % 60;
	cmsf.cdmsf_frame0 = from % 75;
	cmsf.cdmsf_min1 = to / (75 * 60);
	cmsf.cdmsf_sec1 = (to / 75) % 60;
	cmsf.cdmsf_frame1 = to % 75;
	return ioctl(cdfd, CDROMPLAYMSF, &cmsf);
}

/*\ Stop audio playback
|*|  return -1 on failure
\*/ 
static int
cdrom_stop(int cdfd)
{
	/*\ CDROMPAUSE: Rude hack, because stop takes so long.
	|*|             It looks like it actually stops the disc spinning..
	\*/
	if (cdfd >= 0) return ioctl(cdfd, CDROMPAUSE, 0);
	return 0;
}

/*\ Pause/resume audio playback
|*|  return -1 on failure
\*/
static int
cdrom_pause(int cdfd, short p)
{
	return ioctl(cdfd, p ? CDROMPAUSE : CDROMRESUME);
}

/*\ Get currently playing relative time
|*|  return -1 on failure
|*|  (Closes cdfd if not playing)
\*/
static gint32
cdrom_get_time(struct cd_struct *cd)
{
	gint32 f;
	struct cdrom_subchnl cdsc;

	if (cd->cdfd < 0)
		return -1;
	cdsc.cdsc_format = CDROM_MSF;
	if (ioctl(cd->cdfd, CDROMSUBCHNL, &cdsc) < 0)
		return -2;
	if (cdsc.cdsc_audiostatus == CDROM_AUDIO_ERROR)
		return -2;
	if (cdsc.cdsc_audiostatus == CDROM_AUDIO_COMPLETED)
		return -1;
	f = ((cdsc.cdsc_absaddr.msf.minute * 60 +
		cdsc.cdsc_absaddr.msf.second) * 75 +
		cdsc.cdsc_absaddr.msf.frame);
	if (f > (end_lba - 20))
		return -1;
	f -= cd->lba[cur_trk];
	if (f < 0) f = 0;
	return (f * 40) / 3;
}

#endif /*\ CDROMSOLARIS_H \*/
