/*  XMMS - Cross-platform multimedia player
 *  Copyright (C) 1998-2000  Peter Alm, Mikael Alm, Olle Hallnas, Thomas Nilsson and 4Front Technologies
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "jack.h"
#include <errno.h>

/* the software volume control was inspired from the alsa output plugin
 * found in xmms source tree Output/alsa/audio.c (version 1.2.10+cvs20050809)
 *
 *  XMMS - ALSA output plugin
 *  Copyright (C) 2001-2003 Matthieu Sozeau <mattam@altern.org>
 *  Copyright (C) 1998-2003  Peter Alm, Mikael Alm, Olle Hallnas,
 *                           Thomas Nilsson and 4Front Technologies
 *  Copyright (C) 1999-2005  Haavard Kvaalen
 *  Copyright (C) 2005       Takashi Iwai
 */

#define STEREO_ADJUST(type, type2, endian)					\
do {										\
	type *ptr = data;							\
	for (i = 0; i < length; i += 4)						\
	{									\
		*ptr = type2##_TO_##endian(type2##_FROM_## endian(*ptr) *	\
					   jackxmms_cfg.vol.left / 100);		\
		ptr++;								\
		*ptr = type2##_TO_##endian(type2##_FROM_##endian(*ptr) *	\
					   jackxmms_cfg.vol.right / 100);		\
		ptr++;								\
	}									\
} while (0)

#define MONO_ADJUST(type, type2, endian)					\
do {										\
	type *ptr = data;							\
	for (i = 0; i < length; i += 2)						\
	{									\
		*ptr = type2##_TO_##endian(type2##_FROM_## endian(*ptr) *	\
					   vol / 100);				\
		ptr++;								\
	}									\
} while (0)

#define VOLUME_ADJUST(type, type2, endian)		\
do {							\
	if (channels == 2)				\
		STEREO_ADJUST(type, type2, endian);	\
	else						\
		MONO_ADJUST(type, type2, endian);	\
} while (0)

#define STEREO_ADJUST8(type)				\
do {							\
	type *ptr = data;				\
	for (i = 0; i < length; i += 2)			\
	{						\
		*ptr = *ptr * jackxmms_cfg.vol.left / 100;	\
		ptr++;					\
		*ptr = *ptr * jackxmms_cfg.vol.right / 100;	\
		ptr++;					\
	}						\
} while (0)

#define MONO_ADJUST8(type)			\
do {						\
	type *ptr = data;			\
	for (i = 0; i < length; i++)		\
	{					\
		*ptr = *ptr * vol / 100;	\
		ptr++;				\
	}					\
} while (0)

#define VOLUME_ADJUST8(type)			\
do {						\
	if (channels == 2)			\
		STEREO_ADJUST8(type);		\
	else					\
		MONO_ADJUST8(type);		\
} while (0)


//static 
void volume_adjust(void* data, int length, int fmt, int channels)
{
	int i, vol;

	if ((jackxmms_cfg.vol.left == 100 && jackxmms_cfg.vol.right == 100) ||
	    (channels == 1 &&
	     (jackxmms_cfg.vol.left == 100 || jackxmms_cfg.vol.right == 100)))
		return;

	vol = MAX(jackxmms_cfg.vol.left, jackxmms_cfg.vol.right);

	switch (fmt)
	{
		case AFMT_S16_LE:
			VOLUME_ADJUST(gint16, GINT16, LE);
			break;
		case AFMT_U16_LE:
			VOLUME_ADJUST(guint16, GUINT16, LE);
			break;
		case AFMT_S16_BE:
			VOLUME_ADJUST(gint16, GINT16, BE);
			break;
		case AFMT_U16_BE:
			VOLUME_ADJUST(guint16, GUINT16, BE);
			break;
		case AFMT_S8:
			VOLUME_ADJUST8(gint8);
			break;
		case AFMT_U8:
			VOLUME_ADJUST8(guint8);
			break;
		default:
			g_warning("volue_adjust(): unhandled format: %d", fmt);
			break;
	}
}

/* end of volume adjustements routine */

static char* get_mixer_device(void)
{
	char *name;
	
	if (jackxmms_cfg.use_alt_mixer_device && jackxmms_cfg.alt_mixer_device)
		name = g_strdup(jackxmms_cfg.alt_mixer_device);
	else if (jackxmms_cfg.mixer_device > 0)
		name = g_strdup_printf("%s%d", DEV_MIXER, jackxmms_cfg.mixer_device);
	else
		name = g_strdup(DEV_MIXER);

	return name;
}

void jackxmms_get_volume(int *l, int *r)
{
	int fd, v, cmd, devs;
	gchar *devname;

	if (jackxmms_cfg.soft_volume)
	{
		*l = jackxmms_cfg.vol.left;
		*r = jackxmms_cfg.vol.right;
		return;
	}

	devname = get_mixer_device();
	fd = open(devname, O_RDONLY);
	g_free(devname);

	/*
	 * We dont show any errors if this fails, as this is called
	 * rather often
	 */
	if (fd != -1)
	{
		ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs);
		if ((devs & SOUND_MASK_PCM) && (jackxmms_cfg.use_master==0))
			cmd = SOUND_MIXER_READ_PCM;
		else if ((devs & SOUND_MASK_VOLUME) && (jackxmms_cfg.use_master==1))
			cmd = SOUND_MIXER_READ_VOLUME;
		else
		{
			close(fd);
			return;
		}
		ioctl(fd, cmd, &v);
		*r = (v & 0xFF00) >> 8;
		*l = (v & 0x00FF);
		close(fd);
	}
}

void jackxmms_set_volume(int l, int r)
{
	int fd, v, cmd, devs;
	gchar *devname;

	if (jackxmms_cfg.soft_volume)
	{
		jackxmms_cfg.vol.left = l;
		jackxmms_cfg.vol.right = r;
		return;
	}

	devname = get_mixer_device();
	fd = open(devname, O_RDONLY);

	if (fd != -1)
	{
		ioctl(fd, SOUND_MIXER_READ_DEVMASK, &devs);
		if ((devs & SOUND_MASK_PCM) && (jackxmms_cfg.use_master==0))
			cmd = SOUND_MIXER_WRITE_PCM;
		else if ((devs & SOUND_MASK_VOLUME) && (jackxmms_cfg.use_master==1))
			cmd = SOUND_MIXER_WRITE_VOLUME;
		else
		{
			close(fd);
			return;
		}
		v = (r << 8) | l;
		ioctl(fd, cmd, &v);
		close(fd);
	}
	else
		g_warning("jackxmms_set_volume(): Failed to open mixer device (%s): %s",
			  devname, strerror(errno));
	g_free(devname);
}
