/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>

#include <xmms/configfile.h>

#include "singit_debug.h"
#include "singit_macros.h"
#include "singit_config_private.h"
#include "singit_config_gen.h"

#include "singit_dialogs.h"
#include "singit_plugin_scanner.h"
#include "singit_displayer_plugin.h"

#include "singit_main.h"

#include "dlg_singit_config.h"

#include "singit_wgt_colorbutton.h"
#include "singit_tools.h"
#include "dlg_multiline.h"

static GtkWidget *singit_config_win = NULL;
static GtkWidget
	*config_vbox,
		*config_notebook,
		*config_displayer_vbox, *config_displayer_label,
			*config_filefound_frame, *config_filefound_table,
				*config_filefound_basepath_label,
				*config_filefound_basepath_entry,
				*config_filefound_basepath_button,
				*config_filefound_hsep1,
				*config_filefound_construct_filename_label,
				*config_filefound_construct_filename_entry,
				*config_filefound_construct_filename_button,
				*config_filefound_extension_label,
				*config_filefound_extension_entry,
				*config_filefound_extension_button,
			*config_font_frame,
				*config_font_hbox,
				*config_font_entry,
				*config_font_browse,
				*config_fontsel,
			*config_color_frame,
				*config_color_vbox,

				*config_color_background_hbox_widget,
				*config_color_background_label,
				*config_color_background_button,

				*config_color_normal_text_hbox_widget,
				*config_color_normal_text_label,
				*config_color_normal_text_button,

				*config_color_normal_bar_hbox_widget,
				*config_color_normal_bar_label,
				*config_color_normal_bar_button,

				*config_color_progress_text_hbox_widget,
				*config_color_progress_text_label,
				*config_color_progress_text_button,

				*config_color_progress_bar_hbox_widget,
				*config_color_progress_bar_label,
				*config_color_progress_bar_button,

		*config_displayer_toggle_vbox, *config_displayer_toggle_label,
			*config_toggle_yz_check_button,
			*config_show_empty_check_button,
			*config_auto_resize_check_button,
			*config_hide_if_not_found_check_button,
			*config_show_ball_check_button,
			*config_guess_sync_lyrics_check_button,
			*config_find_optimal_font_check_button,
			*config_use_xmms_usleep_check_button,
			*config_show_time_indicator_check_button,
#		ifdef CODEDEBUG
			*config_debug_frame, *config_debug_vbox,
				*config_debug_enable_check_button,
				*config_debug_exclusive_check_button,
				*config_debug_level_hbox,
					*config_debug_level_label,
					*config_debug_level_spinbutton,
#		endif

		*config_dialogs_hbox, *config_dialogs_label,
			*config_dialogs_vbox,
				*config_about_button,
				*config_about_hseparator,
#			ifdef HAVE_TAG_MANAGEMENT_SUPPORT
				*config_tag_manager_button,
				*config_tag_manager_hseparator,
#			endif
				*config_editor_start_button,
				*config_reaction_hbox,
					*config_reaction_label,
					*config_reaction_spinbutton,
				*config_framerate_limit_hseparator,
				*config_framerate_limit_hbox,
					*config_framerate_limit_label,
					*config_framerate_limit_spinbutton,
		*config_dis_plugin_vbox, *config_dis_plugin_label,
				*config_dis_plugin_clist,
				*config_dis_plugin_scrolled_window,
			*config_dis_plugin_hbox,
				*config_dis_plugin_configure_button,
				*config_dis_plugin_about_button,
				*config_dis_plugin_use_check_button,
		*bottom_button_hbox,
			*bottom_button_ok,
			*bottom_button_cancel,
			*bottom_button_apply;

static GtkObject *config_reaction_spinbutton_adj = NULL,
#ifdef CODEDEBUG
	*config_debug_level_spinbutton_adj = NULL,
#endif
	*config_framerate_limit_spinbutton_adj = NULL;

// static GtkWidget *font_select_dialog = NULL;
/*
static GtkTooltips
		*plugin_clist_tooltips = NULL;
*/
typedef struct {

	guchar *color_array;
	guint exporter_selection;

	gboolean updating_displayer_list;

} ConfigStatus;

ConfigStatus config_status;
extern SingitStatus singit_status;
extern SingitConfigGen *singit_config;

static void config_font_browse_ok(GtkWidget * w, gpointer data)
{
        gtk_entry_set_text(GTK_ENTRY(config_font_entry),
		gtk_font_selection_dialog_get_font_name(GTK_FONT_SELECTION_DIALOG(config_fontsel)));
        gtk_widget_destroy(config_fontsel);
}

static void config_font_browse_cb(GtkWidget * w, gpointer data)
{
	gchar *temp;

        if (!config_fontsel)
        {
		temp = g_strconcat(_("Select lyric font"), ":", NULL);
                config_fontsel = gtk_font_selection_dialog_new(temp);
		g_free(temp);
		gtk_window_set_modal(GTK_WINDOW(config_fontsel), TRUE);
                gtk_font_selection_dialog_set_font_name(GTK_FONT_SELECTION_DIALOG(config_fontsel),
			GET_SCD->lyricFontName);
                gtk_signal_connect(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(config_fontsel)->ok_button), "clicked",
			GTK_SIGNAL_FUNC(config_font_browse_ok), NULL);
                gtk_signal_connect_object(GTK_OBJECT(GTK_FONT_SELECTION_DIALOG(config_fontsel)->cancel_button),
			"clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy), GTK_OBJECT(config_fontsel));
                gtk_signal_connect(GTK_OBJECT(config_fontsel), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed),
			&config_fontsel);
                gtk_widget_show(config_fontsel);
        }
}

// gtk_signal_connect(GTK_OBJECT(config_color_text_button), "color_changed", GTK_SIGNAL_FUNC(color_button_color_changed), NULL);
static void color_button_color_changed (ColorButton *gcb)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_config.c [color_button_color_changed]\n"));
#endif
}

static void start_singit_editor(GtkWidget *button, gpointer config)
{
	singit_config_save_editor();
	singit_editor_show();
}

static void multi_line_dialog_new_text_event(MultiLineDialog *mldlg, gpointer user_data)
{
	gchar *text = NULL, *new_text;
	SingitConfigData *scd = GET_SCD;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_config.c [multi_line_dialog_new_text_event]\n"));
#endif

	text = multi_line_dialog_get_text(mldlg);
	new_text = tools_replace_string(text, "/,", ",");
	g_free(text);
	text = tools_replace_string(new_text, ",", "/,");
	g_free(new_text);
	if (text[strlen(text)-1] != '/') {
		new_text = text;
		text = g_strconcat(new_text, "/", 0);
		g_free(new_text);
	}

	switch ((gint) user_data) {
	case 0:
		if (scd->basePath) { g_free(scd->basePath); }
		scd->basePath = text;
		gtk_entry_set_text(GTK_ENTRY(config_filefound_basepath_entry),
			scd->basePath);
		break;
	case 1:
		if (scd->basePath) { g_free(scd->constructFilename); }
		scd->constructFilename = text;
		gtk_entry_set_text(GTK_ENTRY(config_filefound_construct_filename_entry),
			scd->constructFilename);
		break;
	case 2:
		if (scd->basePath) { g_free(scd->lyricExtension); }
		scd->lyricExtension = text;
		gtk_entry_set_text(GTK_ENTRY(config_filefound_extension_entry),
			scd->lyricExtension);
		break;
	default:
		if (text) { g_free(text); }
	}
}

static void config_open_multiline_editor_ok(GtkWidget *button, gpointer user_data)
{
	GtkWidget * dialog;

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("dlg_singit_config.c [config_open_multiline_editor_ok]\n"));
#endif

	dialog = multi_line_dialog_new("Editor");
	gtk_window_set_modal(GTK_WINDOW(dialog), TRUE);
	gtk_signal_connect(GTK_OBJECT(dialog),
		"new_text", GTK_SIGNAL_FUNC(multi_line_dialog_new_text_event), user_data);
	gtk_signal_connect(GTK_OBJECT(dialog),
		"button_pressed", GTK_SIGNAL_FUNC(multi_line_dialog_hide), user_data);

	switch ((gint) user_data) {
	case 0: multi_line_dialog_set_text(MULTI_LINE_DIALOG(dialog),
			g_strstrip(gtk_entry_get_text(GTK_ENTRY(config_filefound_basepath_entry)))
		);
		break;
	case 1: multi_line_dialog_set_text(MULTI_LINE_DIALOG(dialog),
			gtk_entry_get_text(GTK_ENTRY(config_filefound_construct_filename_entry))
		);
		break;
	case 2: multi_line_dialog_set_text(MULTI_LINE_DIALOG(dialog),
			g_strstrip(gtk_entry_get_text(GTK_ENTRY(config_filefound_extension_entry)))
		);
		break;
	}

	multi_line_dialog_show(MULTI_LINE_DIALOG(dialog), GTK_WINDOW(singit_config_win));
}

static void uwt_mode_changed_button_clicked_event (GtkButton *button, gpointer user_data)
{
}

/*********************************************
	displayer plugin funtions
*********************************************/

static void config_dis_plugins_plugins(void)
{
	GList *glist;
	gchar *description[2] = { 0 }, *tooltip, *enabled_str;
	DisplayerPlugin *dp;
	gint i = 0, column;

	glist = get_dis_plugin_list();

	gtk_clist_clear(GTK_CLIST(config_dis_plugin_clist));

	while (glist)
	{
		dp = (DisplayerPlugin *) glist->data;
		if (is_dis_plugin_enabled(i)) {
			enabled_str =  _("enabled");
			description[1] = g_strconcat(dp->description, "   (",
				enabled_str, ")", NULL);
			tooltip = g_strdup_printf("%s   [%s] (%s)", dp->description,
				g_basename(dp->filename), enabled_str);
		}
		else {
			description[1] = g_strdup(dp->description);
			tooltip = g_strdup_printf("%s  [%s]", dp->description,
				g_basename(dp->filename));
		}

		if (i < 10) {
			description[0] = g_strdup_printf("[%i]", (i + 1) % 10);
		}
		column = gtk_clist_append(GTK_CLIST(config_dis_plugin_clist), description);
/*		gtk_tooltips_set_tip(plugin_clist_tooltips,
			gtk_clist_get_column_widget(GTK_CLIST(config_dis_plugin_clist), column),
			tooltip, NULL); */
		g_free(description[0]);
		g_free(description[1]);
		g_free(tooltip);
		glist = glist->next;
		i++;
	}

	gtk_widget_set_sensitive(config_dis_plugin_use_check_button, FALSE);
	gtk_widget_set_sensitive(config_dis_plugin_configure_button, FALSE);
	gtk_widget_set_sensitive(config_dis_plugin_about_button, FALSE);
}

void config_dis_plugins_rescan(void)
{
	gint sel;
	GtkAdjustment *adj;
	gfloat pos;

	if (!singit_config_win) { return; }

	if(GTK_CLIST(config_dis_plugin_clist)->selection)
		sel = (gint) GTK_CLIST(config_dis_plugin_clist)->selection->data;
	else
		sel = -1;
	adj = gtk_clist_get_vadjustment(GTK_CLIST(config_dis_plugin_clist));
	pos = adj->value;
	config_dis_plugins_plugins();
	gtk_adjustment_set_value(adj, pos);
	gtk_clist_set_vadjustment(GTK_CLIST(config_dis_plugin_clist), adj);
	if(sel != -1)
		gtk_clist_select_row(GTK_CLIST(config_dis_plugin_clist), sel, 0);
}

static void config_dis_plugins_use_cb(GtkWidget * w, gpointer data)
{
	gint sel;
	SingitConfigData *scd;

	g_return_if_fail(singit_config != NULL);

	scd = GET_SCD;

	if (!(GTK_CLIST(config_dis_plugin_clist)->selection) ||
		config_status.updating_displayer_list) { return; }

	sel = (gint) GTK_CLIST(config_dis_plugin_clist)->selection->data;

	if (set_dis_plugin_status(sel, GTK_TOGGLE_BUTTON(w)->active) == FALSE)
		{ goto rescan; }
	if (scd &&
		scd->song &&
		singit_song_text_found(scd->song) &&
		GTK_TOGGLE_BUTTON(w)->active)
	{
		dis_plugin_show(sel, scd->song);
	}
	g_free(scd->enabled_dplugins);
	scd->enabled_dplugins = dis_plugin_stringify_enabled_list();

rescan:
	config_dis_plugins_rescan();
}

static void config_dis_plugins_configure(GtkWidget * w, gpointer data)
{
	gint sel = (gint) GTK_CLIST(config_dis_plugin_clist)->selection->data;

	dis_plugin_configure(sel);
}

static void config_dis_plugins_about(GtkWidget * w, gpointer data)
{
	gint sel = (gint) GTK_CLIST(config_dis_plugin_clist)->selection->data;

	dis_plugin_about(sel);
}

static void config_dis_plugins_clist_clicked(GtkWidget * w, gint row, gint column, GdkEvent *event)
{
	DisplayerPlugin *dp;
	gint index;
	GList *dplist;

	if (GTK_CLIST(config_dis_plugin_clist)->selection)
	{
		dplist = get_dis_plugin_list();
		index = (gint) GTK_CLIST(config_dis_plugin_clist)->selection->data;
		dp = g_list_nth(dplist, index)->data;

		gtk_widget_set_sensitive(config_dis_plugin_use_check_button, TRUE);
		config_status.updating_displayer_list = TRUE;
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(config_dis_plugin_use_check_button), is_dis_plugin_enabled(index));
		config_status.updating_displayer_list = FALSE;

		gtk_widget_set_sensitive(config_dis_plugin_configure_button, (dp && dp->configure));
		gtk_widget_set_sensitive(config_dis_plugin_about_button, (dp && dp->about));

		if (event && event->type == GDK_2BUTTON_PRESS)
			gtk_signal_emit_by_name(GTK_OBJECT(config_dis_plugin_configure_button), "clicked");
	}
	else
	{
		gtk_widget_set_sensitive(config_dis_plugin_configure_button, FALSE);
		gtk_widget_set_sensitive(config_dis_plugin_about_button, FALSE);
	}
}

// * -------------------------------------------------------------------------------- *


static void singit_config_cleanup()
{
	SingitConfigData *scd = GET_SCD;

#	ifdef CODEDEBUG
	DEBUG(2, ("dlg_singit_config.c [singit_config_cleanup]\n"));
#endif

	if (singit_config_win && singit_config_win->window && scd) {
		gdk_window_get_position(singit_config_win->window,
			&(scd->configWinPosX), &(scd->configWinPosY));
		singit_config_save_positions();
	}

	g_free(config_status.color_array);

	singit_config_gen_detach(singit_config);
}

static void singit_config_apply_clicked(GtkButton *button, gpointer user_data)
{
	gchar tmp[13] = { 0 };
#define GET_RGB_AS_STRING(a,b) { sprintf(tmp, "rgb:%.2x/%.2x/%.2x", \
		(gushort) ((guchar *) (COLOR_BUTTON(a)->color))[0], \
		(gushort) ((guchar *) (COLOR_BUTTON(a)->color))[1], \
		(gushort) ((guchar *) (COLOR_BUTTON(a)->color))[2]); \
		b = g_strdup(tmp); }

	gchar *temp;
	SingitConfigData *scd = GET_SCD;

#ifdef CODEDEBUG
	DEBUG(2, ("dlg_singit_config.c [singit_config_apply_clicked] : "));
#endif

	if (!(singit_config_win &&
		singit_config_win->window &&
		singit_config_gen_attach(singit_config)))
	{
#ifdef CODEDEBUG
		DEBUG(2, ("failed\n"));
#endif
		return;
	}
#ifdef CODEDEBUG
	DEBUG(2, ("Ok\n"));
#endif

	pthread_mutex_lock(&singit_status.config_rw_mutex);

	g_free(scd->basePath);
	g_free(scd->constructFilename);
	g_free(scd->lyricExtension);
	g_free(scd->lyricFontName);

	g_free(scd->backgroundColor);
	g_free(scd->normalTextColor);
	g_free(scd->normalBarColor);
	g_free(scd->progressTextColor);
	g_free(scd->progressBarColor);

/*	g_free(scd->htmlHeader);
	g_free(scd->htmlTail);
	g_free(scd->lyricLine);*/
	g_free(scd->enabled_dplugins);

	/* set new values */

	scd->basePath = temp = gtk_editable_get_chars(GTK_EDITABLE(config_filefound_basepath_entry), 0, -1);
	if (temp) {
		scd->basePath = g_strdup(g_strstrip(temp));
		g_free(temp);
		if ((scd->basePath[strlen(scd->basePath)-1]) != '/') {
			temp = scd->basePath;
			scd->basePath = g_strconcat(temp, "/", NULL);
			g_free(temp);
		}
	}
	scd->constructFilename = gtk_editable_get_chars(GTK_EDITABLE(config_filefound_construct_filename_entry), 0, -1);
	scd->lyricExtension = gtk_editable_get_chars(GTK_EDITABLE(config_filefound_extension_entry), 0, -1);
	scd->lyricFontName = gtk_editable_get_chars(GTK_EDITABLE(config_font_entry), 0, -1);
	scd->toggleYZ = GTK_TOGGLE_BUTTON(config_toggle_yz_check_button)->active;
	scd->showEmpty = GTK_TOGGLE_BUTTON(config_show_empty_check_button)->active;
	scd->autoResize = GTK_TOGGLE_BUTTON(config_auto_resize_check_button)->active;
	scd->hideIfNotFound = GTK_TOGGLE_BUTTON(config_hide_if_not_found_check_button)->active;
	scd->showJumpingBall = GTK_TOGGLE_BUTTON(config_show_ball_check_button)->active;
	scd->guessSyncLyrics = GTK_TOGGLE_BUTTON(config_guess_sync_lyrics_check_button)->active;
#ifdef CODEDEBUG
	scd->debugEnable = GTK_TOGGLE_BUTTON(config_debug_enable_check_button)->active;
	scd->debugLevelExcl = GTK_TOGGLE_BUTTON(config_debug_exclusive_check_button)->active;
	scd->debugLevel =
		gtk_spin_button_get_value_as_int(GTK_SPIN_BUTTON(config_debug_level_spinbutton));
#endif
	scd->findOptimalFont = GTK_TOGGLE_BUTTON(config_find_optimal_font_check_button)->active;
	scd->useXMMSuSleep = GTK_TOGGLE_BUTTON(config_use_xmms_usleep_check_button)->active;
	scd->showTimeIndicator = GTK_TOGGLE_BUTTON(config_show_time_indicator_check_button)->active;
	scd->enabled_dplugins = dis_plugin_stringify_enabled_list();

	GET_RGB_AS_STRING(config_color_background_button, scd->backgroundColor);
	GET_RGB_AS_STRING(config_color_normal_text_button, scd->normalTextColor);
	GET_RGB_AS_STRING(config_color_normal_bar_button, scd->normalBarColor);
	GET_RGB_AS_STRING(config_color_progress_text_button, scd->progressTextColor);
	GET_RGB_AS_STRING(config_color_progress_bar_button, scd->progressBarColor);

	scd->reactionTime = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON(config_reaction_spinbutton));

	scd->framerateLimit = gtk_spin_button_get_value_as_int
		(GTK_SPIN_BUTTON(config_framerate_limit_spinbutton));

	pthread_mutex_unlock(&singit_status.config_rw_mutex);

	singit_config_gen_emit_update(SINGIT_CONFIG_GEN(singit_config));

	singit_config_gen_save(SINGIT_CONFIG_GEN(singit_config));

	singit_config_gen_detach(SINGIT_CONFIG_GEN(singit_config));
}

static void singit_config_ok_clicked(GtkButton *button, gpointer user_data)
{
	/* apply and save config */
	singit_config_apply_clicked(NULL, NULL);
	singit_config_cleanup();

	/* close and destroy window */
	gtk_widget_destroy(singit_config_win);

	singit_main_finish(FALSE);
}

static void singit_config_cancel_clicked(GtkButton *button, gpointer user_data)
{
	singit_config_cleanup();

	/* close and destroy window */
	gtk_widget_destroy(singit_config_win);

	singit_main_finish(FALSE);
}

static gint dlg_singit_config_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	singit_config_cleanup();

	/* close and destroy window */
	gtk_widget_destroy(singit_config_win);

	singit_main_finish(FALSE);

	return (FALSE);
}

static void add_new_widget(GtkWidget *widget, gchar *name, gboolean show, GSList **group)
{
	if (group != NULL)
		(*group) = gtk_radio_button_group(GTK_RADIO_BUTTON(widget));
//	gtk_widget_set_name(widget, name);
//	gtk_widget_ref(widget);
//	gtk_object_set_data_full(GTK_OBJECT(singit_config_win), name,
//		widget, (GtkDestroyNotify) gtk_widget_unref);
	if (show)
		gtk_widget_show(widget);
}

static void add_color_button(gchar *trans_label, gchar *color_str, gint col_arr_offset,
	GtkWidget **col_hbox, GtkWidget **col_label, GtkWidget **col_button)
{
	GdkColor color;

	gdk_color_parse(color_str, &color);
	config_status.color_array[col_arr_offset] = color.red;
	config_status.color_array[col_arr_offset + 1] = color.green;
	config_status.color_array[col_arr_offset + 2] = color.blue;

	*col_hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(config_color_vbox), *col_hbox);
	*col_label = gtk_label_new(trans_label);
	gtk_box_pack_start(GTK_BOX(*col_hbox), *col_label, TRUE, TRUE, 0);
	*col_button = color_button_new(trans_label, 60, 20,
		&config_status.color_array[col_arr_offset], 3);
	gtk_box_pack_start(GTK_BOX(*col_hbox), *col_button, FALSE, TRUE, 0);
	gtk_widget_show(*col_hbox);
	gtk_widget_show(*col_label);
	gtk_widget_show(*col_button);
}

static void singit_config_displayer_page(SingitConfigData *scd)
{
	config_displayer_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(config_displayer_vbox), 2);
	gtk_widget_show (config_displayer_vbox);
	config_displayer_label = gtk_label_new(_("Lyrics displayer"));
	gtk_widget_show(config_displayer_label);
	gtk_notebook_append_page(GTK_NOTEBOOK(config_notebook), config_displayer_vbox, config_displayer_label);

	config_filefound_frame = gtk_frame_new(_("Lyric file constructor"));
	gtk_box_pack_start(GTK_BOX(config_displayer_vbox), config_filefound_frame, FALSE, FALSE, 0);
	gtk_widget_show(config_filefound_frame);

	config_filefound_table = gtk_table_new (3, 4, FALSE);
	gtk_table_set_row_spacings (GTK_TABLE (config_filefound_table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (config_filefound_table), 3);
	gtk_container_set_border_width(GTK_CONTAINER(config_filefound_table), 2);
	gtk_container_add(GTK_CONTAINER(config_filefound_frame), config_filefound_table);
	gtk_widget_show(config_filefound_table);

/* lyric basepath */
	config_filefound_basepath_label = gtk_label_new(_("Lyrics dir"));
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_basepath_label, 0, 1, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_widget_show(config_filefound_basepath_label);

	config_filefound_basepath_entry = gtk_entry_new();
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_basepath_entry, 1, 2, 0, 1,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_entry_set_text(GTK_ENTRY(config_filefound_basepath_entry),
		scd->basePath);
	gtk_widget_show(config_filefound_basepath_entry);

	config_filefound_basepath_button = gtk_button_new_with_label ("...");
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_basepath_button, 2, 3, 0, 1,
		GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_signal_connect(GTK_OBJECT(config_filefound_basepath_button), "clicked",
		GTK_SIGNAL_FUNC(config_open_multiline_editor_ok), (gpointer) 0);
	gtk_widget_show (config_filefound_basepath_button);

	config_filefound_hsep1 = gtk_hseparator_new();
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_hsep1, 0, 3, 1, 2,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_widget_show(config_filefound_hsep1);

	/* lyric filename constructor */
	config_filefound_construct_filename_label = gtk_label_new(_("%1 = filename"));
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_construct_filename_label, 0, 1, 2, 3,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
//	gtk_widget_show(config_filefound_construct_filename_label);

        config_filefound_construct_filename_entry = gtk_entry_new();
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_construct_filename_entry, 1, 2, 2, 3,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_entry_set_text(GTK_ENTRY(config_filefound_construct_filename_entry),
		scd->constructFilename);
//	gtk_widget_show(config_filefound_construct_filename_entry);

	config_filefound_construct_filename_button = gtk_button_new_with_label ("...");
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_construct_filename_button, 2, 3, 2, 3,
		GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_signal_connect(GTK_OBJECT(config_filefound_construct_filename_button), "clicked",
		GTK_SIGNAL_FUNC(config_open_multiline_editor_ok), (gpointer) 1);
//	gtk_widget_show (config_filefound_construct_filename_button);

/* lyric extension */
	config_filefound_extension_label = gtk_label_new(_("Lyric extension"));
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_extension_label, 0, 1, 3, 4,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_widget_show(config_filefound_extension_label);

        config_filefound_extension_entry = gtk_entry_new();
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_extension_entry, 1, 2, 3, 4,
		GTK_EXPAND | GTK_SHRINK | GTK_FILL, GTK_EXPAND | GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_entry_set_text(GTK_ENTRY(config_filefound_extension_entry),
		scd->lyricExtension);
	gtk_widget_show(config_filefound_extension_entry);

	config_filefound_extension_button = gtk_button_new_with_label ("...");
	gtk_table_attach (GTK_TABLE (config_filefound_table), config_filefound_extension_button, 2, 3, 3, 4,
		GTK_SHRINK | GTK_FILL, GTK_SHRINK | GTK_FILL, 0, 0);
	gtk_signal_connect(GTK_OBJECT(config_filefound_extension_button), "clicked",
		GTK_SIGNAL_FUNC(config_open_multiline_editor_ok), (gpointer) 2);
	gtk_widget_show (config_filefound_extension_button);

	/* font selection box (from XMMS) */
        config_font_frame = gtk_frame_new(_("Lyrics font"));
        gtk_box_pack_start(GTK_BOX(config_displayer_vbox), config_font_frame, FALSE, FALSE, 2);
	gtk_widget_show(config_font_frame);

        config_font_hbox = gtk_hbox_new(FALSE, 5);
        gtk_container_border_width(GTK_CONTAINER(config_font_hbox), 5);
        gtk_container_add(GTK_CONTAINER(config_font_frame), config_font_hbox);
	gtk_widget_show(config_font_hbox);

        config_font_entry = gtk_entry_new();
        gtk_box_pack_start(GTK_BOX(config_font_hbox), config_font_entry, TRUE, TRUE, 0);
	gtk_entry_set_text(GTK_ENTRY(config_font_entry),
		scd->lyricFontName);
	gtk_widget_show(config_font_entry);

        config_font_browse = gtk_button_new_with_label(_("Browse"));
        gtk_signal_connect(GTK_OBJECT(config_font_browse), "clicked", GTK_SIGNAL_FUNC(config_font_browse_cb), NULL);
        gtk_widget_set_usize(config_font_browse, 85, 17);
        gtk_box_pack_start(GTK_BOX(config_font_hbox), config_font_browse, FALSE, TRUE, 0);
	gtk_widget_show(config_font_browse);

/* color selection frame */
	config_color_frame = gtk_frame_new(_("Colors"));
	gtk_box_pack_start(GTK_BOX(config_displayer_vbox), config_color_frame, FALSE, FALSE, 0);
	gtk_widget_show(config_color_frame);

	config_color_vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(config_color_frame), config_color_vbox);
	gtk_container_set_border_width(GTK_CONTAINER(config_color_vbox), 5);
	gtk_widget_show(config_color_vbox);

	add_color_button(_("Background Color"), scd->backgroundColor, 0,
		&config_color_background_hbox_widget,
		&config_color_background_label,
		&config_color_background_button);

	add_color_button(_("Normal Text Color"), scd->normalTextColor, 3,
		&config_color_normal_text_hbox_widget,
		&config_color_normal_text_label,
		&config_color_normal_text_button);

	add_color_button(_("Normal Bar Color"), scd->normalBarColor, 6,
		&config_color_normal_bar_hbox_widget,
		&config_color_normal_bar_label,
		&config_color_normal_bar_button);

	add_color_button(_("Progress Text Color"), scd->progressTextColor, 9,
		&config_color_progress_text_hbox_widget,
		&config_color_progress_text_label,
		&config_color_progress_text_button);

	add_color_button(_("Progress Bar Color"), scd->progressBarColor, 12,
		&config_color_progress_bar_hbox_widget,
		&config_color_progress_bar_label,
		&config_color_progress_bar_button);
}

static void singit_config_options_page(SingitConfigData *scd)
{
#ifdef CODEDEBUG
	gchar *cat_str = NULL;
#endif

	config_displayer_toggle_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_set_border_width(GTK_CONTAINER(config_displayer_toggle_vbox), 2);
	gtk_widget_show (config_displayer_toggle_vbox);
	config_displayer_toggle_label = gtk_label_new(_("Options"));
	gtk_widget_show(config_displayer_toggle_label);
	gtk_notebook_append_page(GTK_NOTEBOOK(config_notebook),
		config_displayer_toggle_vbox, config_displayer_toggle_label);

	config_toggle_yz_check_button = gtk_check_button_new_with_label(_("Toggle y/z-key"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_toggle_yz_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_toggle_yz_check_button),
		scd->toggleYZ);
	gtk_widget_show(config_toggle_yz_check_button);

	config_show_empty_check_button = gtk_check_button_new_with_label(_("Show empty lines"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_show_empty_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_show_empty_check_button),
		scd->showEmpty);
	gtk_widget_show(config_show_empty_check_button);

	config_auto_resize_check_button = gtk_check_button_new_with_label(_("Automagically resize lyric displayer"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_auto_resize_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_auto_resize_check_button),
		scd->autoResize);
	gtk_widget_show(config_auto_resize_check_button);

	config_hide_if_not_found_check_button =
		gtk_check_button_new_with_label
			(_("Hide displayer if no lyrics were found"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox),
		config_hide_if_not_found_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(config_hide_if_not_found_check_button),
		scd->hideIfNotFound);
	gtk_widget_show(config_hide_if_not_found_check_button);

	config_show_ball_check_button =
		gtk_check_button_new_with_label
			(_("Show the jumping ball"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox),
		config_show_ball_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(config_show_ball_check_button),
		scd->showJumpingBall);
	gtk_widget_show(config_show_ball_check_button);

	config_guess_sync_lyrics_check_button = gtk_check_button_new_with_label(_("Guess if we have synced lyrics and set ball"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_guess_sync_lyrics_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_guess_sync_lyrics_check_button),
		scd->guessSyncLyrics);
	gtk_widget_show(config_guess_sync_lyrics_check_button);

	config_find_optimal_font_check_button = gtk_check_button_new_with_label(_("Find the optimal font size (slow fullscreen start)"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_find_optimal_font_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_find_optimal_font_check_button),
		scd->findOptimalFont);
	gtk_widget_show(config_find_optimal_font_check_button);

	config_use_xmms_usleep_check_button = gtk_check_button_new_with_label(_("Use XMMS usleep to save CPU time"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_use_xmms_usleep_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_use_xmms_usleep_check_button),
		scd->useXMMSuSleep);
	gtk_widget_show(config_use_xmms_usleep_check_button);

	config_show_time_indicator_check_button = gtk_check_button_new_with_label(_("Show time indicators on empty lines"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_show_time_indicator_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_show_time_indicator_check_button),
		scd->showTimeIndicator);
	gtk_widget_show(config_show_time_indicator_check_button);

#ifdef CODEDEBUG
/* Debug frame */
	config_debug_frame = gtk_frame_new(_("Debug options"));
	gtk_box_pack_start(GTK_BOX(config_displayer_toggle_vbox), config_debug_frame, TRUE, FALSE, 10);
	gtk_widget_show(config_debug_frame);

	config_debug_vbox = gtk_vbox_new(FALSE, 5);
	gtk_container_add(GTK_CONTAINER(config_debug_frame), config_debug_vbox);
	gtk_container_set_border_width(GTK_CONTAINER(config_debug_vbox), 3);
	gtk_widget_show(config_debug_vbox);

	config_debug_enable_check_button = gtk_check_button_new_with_label(_("Enable console debug output"));
	gtk_box_pack_start(GTK_BOX(config_debug_vbox),
		config_debug_enable_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_debug_enable_check_button),
		scd->debugEnable);
	gtk_widget_show(config_debug_enable_check_button);

	config_debug_exclusive_check_button = gtk_check_button_new_with_label(_("Use exclusive debug level"));
	gtk_box_pack_start(GTK_BOX(config_debug_vbox),
		config_debug_exclusive_check_button, FALSE, FALSE, 0);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_debug_exclusive_check_button),
		scd->debugLevelExcl);
	gtk_widget_show(config_debug_exclusive_check_button);

/* Debug level spin button */
	config_debug_level_hbox = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(config_debug_vbox), config_debug_level_hbox);
	gtk_container_border_width(GTK_CONTAINER(config_debug_level_hbox), 2);
	gtk_widget_show(config_debug_level_hbox);

	cat_str = g_strconcat(_("Debug level"), ": ", NULL);
	config_debug_level_label = gtk_label_new(cat_str);
	g_free(cat_str);
	gtk_box_pack_start(GTK_BOX(config_debug_level_hbox),
		config_debug_level_label, FALSE, FALSE, 2);
	gtk_widget_show(config_debug_level_label);

	config_debug_level_spinbutton_adj = gtk_adjustment_new
		(scd->debugLevel, 0, 9, 1, 1, 1);
	config_debug_level_spinbutton = gtk_spin_button_new
		(GTK_ADJUSTMENT (config_debug_level_spinbutton_adj), 1, 0);
	gtk_entry_set_max_length(GTK_ENTRY(config_debug_level_spinbutton), 1);
	gtk_box_pack_start(GTK_BOX(config_debug_level_hbox),
		config_debug_level_spinbutton, FALSE, FALSE, 2);
	gtk_widget_show (config_debug_level_spinbutton);
#endif
}

static void singit_config_dialogs_page(SingitConfigData *scd)
{
	gchar *tmp_text = NULL;

	config_dialogs_hbox = gtk_hbox_new(FALSE, 1);
	gtk_container_set_border_width(GTK_CONTAINER(config_dialogs_hbox), 2);
	gtk_widget_show(config_dialogs_hbox);

	config_dialogs_label = gtk_label_new(_("Dialogs"));
	gtk_notebook_append_page(GTK_NOTEBOOK(config_notebook),
		config_dialogs_hbox, config_dialogs_label);
	gtk_widget_show(config_dialogs_label);

	config_dialogs_vbox = gtk_vbox_new(FALSE, 1);
	gtk_box_pack_start(GTK_BOX(config_dialogs_hbox),
		config_dialogs_vbox, TRUE, FALSE, 0);
	gtk_container_border_width(GTK_CONTAINER(config_dialogs_vbox), 2);
	gtk_widget_show(config_dialogs_vbox);

	config_about_button = gtk_button_new_with_label(_("Show about dialog..."));
	gtk_signal_connect(GTK_OBJECT(config_about_button), "clicked",
		GTK_SIGNAL_FUNC(singit_about_show), NULL);
	gtk_container_set_border_width(GTK_CONTAINER(config_about_button), 5);
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_about_button, TRUE, FALSE, 10);
	gtk_widget_show(config_about_button);

	config_about_hseparator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_about_hseparator, TRUE, FALSE, 4);
	gtk_widget_show(config_about_hseparator);

#ifdef HAVE_TAG_MANAGEMENT_SUPPORT
	config_tag_manager_button = gtk_button_new_with_label(_("Show tag manager..."));
	gtk_signal_connect(GTK_OBJECT(config_tag_manager_button), "clicked",
		GTK_SIGNAL_FUNC(singit_tag_manager_show), NULL);
	gtk_container_set_border_width(GTK_CONTAINER(config_tag_manager_button), 5);
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_tag_manager_button, TRUE, FALSE, 10);
	gtk_widget_show(config_tag_manager_button);

	config_tag_manager_hseparator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_tag_manager_hseparator, TRUE, FALSE, 4);
	gtk_widget_show(config_tag_manager_hseparator);
#endif

	config_editor_start_button =
		gtk_button_new_with_label(_("Start lyrics editor..."));
	gtk_signal_connect(GTK_OBJECT(config_editor_start_button),
		"clicked", GTK_SIGNAL_FUNC(start_singit_editor), NULL);
	gtk_container_set_border_width(GTK_CONTAINER(config_editor_start_button), 5);
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_editor_start_button, TRUE, FALSE, 10);
	gtk_widget_show(config_editor_start_button);

/* Reaction spin button */
	config_reaction_hbox = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(config_dialogs_vbox),
		config_reaction_hbox);
	gtk_container_border_width(GTK_CONTAINER(config_reaction_hbox), 2);
	gtk_widget_show(config_reaction_hbox);

	tmp_text = g_strconcat("(!) ",
		_("Reaction time [ms] (> slower)"), " (!)", 0);
	config_reaction_label = gtk_label_new (tmp_text);
	g_free(tmp_text);
	gtk_box_pack_start(GTK_BOX(config_reaction_hbox),
		config_reaction_label, FALSE, FALSE, 2);
	gtk_widget_show(config_reaction_label);

	config_reaction_spinbutton_adj = gtk_adjustment_new
		(scd->reactionTime, 0, 9999, 10, 50, 50);
	config_reaction_spinbutton = gtk_spin_button_new
		(GTK_ADJUSTMENT (config_reaction_spinbutton_adj), 1, 0);
	gtk_box_pack_start(GTK_BOX(config_reaction_hbox),
		config_reaction_spinbutton, TRUE, TRUE, 2);
	gtk_widget_show (config_reaction_spinbutton);

// Framerate limit spin button
	config_framerate_limit_hseparator = gtk_hseparator_new();
	gtk_box_pack_start(GTK_BOX(config_dialogs_vbox),
		config_framerate_limit_hseparator, TRUE, FALSE, 4);
	gtk_widget_show(config_framerate_limit_hseparator);

	config_framerate_limit_hbox = gtk_hbox_new(FALSE, 2);
	gtk_container_add(GTK_CONTAINER(config_dialogs_vbox),
		config_framerate_limit_hbox);
	gtk_container_border_width
		(GTK_CONTAINER(config_framerate_limit_hbox), 2);
	gtk_widget_show(config_framerate_limit_hbox);

	config_framerate_limit_label =
		gtk_label_new(_("Framerate limit (0 = no limit)"));
	gtk_box_pack_start(GTK_BOX(config_framerate_limit_hbox),
		config_framerate_limit_label, FALSE, FALSE, 2);
	gtk_widget_show(config_framerate_limit_label);

	config_framerate_limit_spinbutton_adj = gtk_adjustment_new
		(scd->framerateLimit, 0, 99, 1, 10, 10);
	config_framerate_limit_spinbutton = gtk_spin_button_new
		(GTK_ADJUSTMENT(config_framerate_limit_spinbutton_adj), 1, 0);
	gtk_box_pack_start(GTK_BOX(config_framerate_limit_hbox),
		config_framerate_limit_spinbutton, TRUE, TRUE, 2);
	gtk_widget_show (config_framerate_limit_spinbutton);
}

static void singit_config_plugins_page(SingitConfigData *scd)
{
	static const gchar *displayer_titles[] = 
		{ N_(" K "), N_(" displayer plugins ") };

	config_dis_plugin_vbox = gtk_vbox_new(FALSE, 3);
	gtk_container_set_border_width(GTK_CONTAINER(config_dis_plugin_vbox), 2);
	gtk_widget_show(config_dis_plugin_vbox);

	config_dis_plugin_label = gtk_label_new(_("Plugins"));
	gtk_notebook_append_page(GTK_NOTEBOOK(config_notebook), config_dis_plugin_vbox, config_dis_plugin_label);
	gtk_widget_show(config_dis_plugin_label);

	config_dis_plugin_clist = gtk_clist_new_with_titles(2, (gchar**) displayer_titles);
        gtk_clist_column_titles_passive(GTK_CLIST(config_dis_plugin_clist));
        gtk_clist_set_selection_mode(GTK_CLIST(config_dis_plugin_clist), GTK_SELECTION_SINGLE);
	gtk_clist_set_column_resizeable(GTK_CLIST(config_dis_plugin_clist), 0, FALSE);
	gtk_clist_set_column_resizeable(GTK_CLIST(config_dis_plugin_clist), 1, FALSE);
	gtk_signal_connect(GTK_OBJECT(config_dis_plugin_clist), 
			"select_row", GTK_SIGNAL_FUNC(config_dis_plugins_clist_clicked), NULL);
	gtk_signal_connect(GTK_OBJECT(config_dis_plugin_clist), 
			"unselect_row", GTK_SIGNAL_FUNC(config_dis_plugins_clist_clicked), NULL);
	gtk_widget_show(config_dis_plugin_clist);

        config_dis_plugin_scrolled_window = gtk_scrolled_window_new(NULL, NULL);
        gtk_container_add(GTK_CONTAINER(config_dis_plugin_scrolled_window), config_dis_plugin_clist);
        gtk_container_border_width(GTK_CONTAINER(config_dis_plugin_scrolled_window), 5);
        gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(config_dis_plugin_scrolled_window), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
        gtk_box_pack_start(GTK_BOX(config_dis_plugin_vbox), config_dis_plugin_scrolled_window, TRUE, TRUE, 0);
	gtk_widget_show(config_dis_plugin_scrolled_window);

	config_dis_plugin_hbox = gtk_hbox_new(FALSE, 5);
	gtk_container_set_border_width(GTK_CONTAINER(config_dis_plugin_hbox), 2);
	gtk_box_pack_start(GTK_BOX(config_dis_plugin_vbox), config_dis_plugin_hbox, FALSE, FALSE, 0);
	gtk_widget_show(config_dis_plugin_hbox);

	config_dis_plugin_configure_button = gtk_button_new_with_label(_("Configure"));
	gtk_signal_connect(GTK_OBJECT(config_dis_plugin_configure_button), 
		"clicked", GTK_SIGNAL_FUNC(config_dis_plugins_configure), NULL);
	gtk_box_pack_start(GTK_BOX(config_dis_plugin_hbox), config_dis_plugin_configure_button, TRUE, TRUE, 5);
	gtk_widget_set_sensitive(config_dis_plugin_configure_button, FALSE);
	gtk_widget_show(config_dis_plugin_configure_button);

	config_dis_plugin_about_button = gtk_button_new_with_label(_("About"));
	gtk_signal_connect(GTK_OBJECT(config_dis_plugin_about_button), "clicked", GTK_SIGNAL_FUNC(config_dis_plugins_about), NULL);
	gtk_box_pack_start(GTK_BOX(config_dis_plugin_hbox), config_dis_plugin_about_button, TRUE, TRUE, 5);
	gtk_widget_set_sensitive(config_dis_plugin_about_button, FALSE);
	gtk_widget_show(config_dis_plugin_about_button);

	config_dis_plugin_use_check_button = gtk_check_button_new_with_label(_("Enable displayer"));
	gtk_box_pack_start(GTK_BOX(config_dis_plugin_hbox), config_dis_plugin_use_check_button, TRUE, TRUE, 5);
	gtk_signal_connect(GTK_OBJECT(config_dis_plugin_use_check_button), "toggled", GTK_SIGNAL_FUNC(config_dis_plugins_use_cb), NULL);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(config_dis_plugin_use_check_button), FALSE);
	gtk_widget_set_sensitive(config_dis_plugin_use_check_button, FALSE);
	gtk_widget_show(config_dis_plugin_use_check_button);
}

void singit_config_show_page(gint page)
{
//	GdkColor *color;
//	gchar *tmp_text = NULL;
	SingitConfigData *scd;

#ifdef CODEDEBUG
	DEBUG(2, ("dlg_singit_config.c [singit_config_show]\n"));
#endif

	if (singit_config_win) {
		if (gtk_notebook_current_page(GTK_NOTEBOOK(config_notebook)) != page)
			gtk_notebook_set_page(GTK_NOTEBOOK(config_notebook), page);
		gdk_window_raise(singit_config_win->window);
		return;
	}

	if (!singit_config_gen_attach(singit_config)) {
		singit_config_new();
		singit_config_gen_load(singit_config);
	}
	scd = GET_SCD;
	if (!scd) { return; }

	singit_main_init(FALSE);

#ifdef ENABLE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALE_DIR);
#endif

	config_status.color_array = g_malloc(sizeof(guchar) * 3 * 5);
	config_status.updating_displayer_list = FALSE;

//	plugin_clist_tooltips = gtk_tooltips_new ();

	singit_config_win = gtk_window_new(GTK_WINDOW_DIALOG);
	gtk_signal_connect(GTK_OBJECT(singit_config_win), "delete_event",
		GTK_SIGNAL_FUNC(dlg_singit_config_delete_event), NULL);

	/* update singit_config_win when window is destroyed */
	gtk_signal_connect(GTK_OBJECT(singit_config_win), "destroy",
		       GTK_SIGNAL_FUNC(gtk_widget_destroyed), &singit_config_win);

	gtk_window_set_title(GTK_WINDOW(singit_config_win), SINGIT_VERSION_STRING " Config");
	gtk_window_set_policy(GTK_WINDOW(singit_config_win), FALSE, FALSE, FALSE);
	gtk_container_border_width(GTK_CONTAINER(singit_config_win), 5);
//	gtk_object_set_data (GTK_OBJECT(singit_config_win), "tooltips", plugin_clist_tooltips);

	config_vbox = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(singit_config_win), config_vbox);
	gtk_widget_show(config_vbox);

	config_notebook = gtk_notebook_new();
	gtk_notebook_set_tab_pos(GTK_NOTEBOOK(config_notebook), GTK_POS_TOP);
	gtk_box_pack_start(GTK_BOX(config_vbox), config_notebook, FALSE, FALSE, 0);
	gtk_widget_show(config_notebook);

/*
	Generate all notebook pages
*/
	singit_config_displayer_page(scd);
	singit_config_options_page(scd);
	singit_config_dialogs_page(scd);
	singit_config_plugins_page(scd);

/*
	Bottom button row ([Ok] [Cancel] [Apply])
*/
	bottom_button_hbox = gtk_hbutton_box_new();
	gtk_button_box_set_spacing(GTK_BUTTON_BOX(bottom_button_hbox), 5);
	gtk_box_pack_start(GTK_BOX(config_vbox), bottom_button_hbox, FALSE, FALSE, 0);
	gtk_widget_show(bottom_button_hbox);

	bottom_button_ok = gtk_button_new_with_label(_("Ok"));
	gtk_signal_connect_object(GTK_OBJECT(bottom_button_ok), "clicked", GTK_SIGNAL_FUNC(singit_config_ok_clicked), (GtkObject*) TRUE);
	GTK_WIDGET_SET_FLAGS(bottom_button_ok, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bottom_button_hbox), bottom_button_ok, TRUE, TRUE, 0);
	gtk_widget_show(bottom_button_ok);

	bottom_button_cancel = gtk_button_new_with_label(_("Cancel"));
	gtk_signal_connect_object(GTK_OBJECT(bottom_button_cancel), "clicked", GTK_SIGNAL_FUNC(singit_config_cancel_clicked), NULL);
	GTK_WIDGET_SET_FLAGS(bottom_button_cancel, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bottom_button_hbox), bottom_button_cancel, TRUE, TRUE, 0);
	gtk_widget_grab_default(bottom_button_cancel);
	gtk_widget_show(bottom_button_cancel);

	bottom_button_apply = gtk_button_new_with_label(_("Apply"));
	gtk_signal_connect_object(GTK_OBJECT(bottom_button_apply), "clicked", GTK_SIGNAL_FUNC(singit_config_apply_clicked), NULL);
	GTK_WIDGET_SET_FLAGS(bottom_button_apply, GTK_CAN_DEFAULT);
	gtk_box_pack_start(GTK_BOX(bottom_button_hbox), bottom_button_apply, TRUE, TRUE, 0);
	gtk_widget_show(bottom_button_apply);

	gtk_widget_show(singit_config_win);

	config_dis_plugins_plugins();

	if ((scd->configWinPosX < 0) || (scd->configWinPosY < 0)) {
		gtk_window_set_position(GTK_WINDOW(singit_config_win), GTK_WIN_POS_MOUSE);
	}
	else {
		gtk_widget_set_uposition(singit_config_win, scd->configWinPosX, scd->configWinPosY);
	}

	if (gtk_notebook_current_page(GTK_NOTEBOOK(config_notebook)) != page)
		gtk_notebook_set_page(GTK_NOTEBOOK(config_notebook), page);
}

void singit_config_hide(void)
{
#ifdef CODEDEBUG
	DEBUG(2, ("dlg_singit_config.c [singit_config_hide]\n"));
#endif

	if (singit_config_win) {
		singit_config_cancel_clicked(NULL, NULL);
	}
}

void singit_config_show(void)
{
#ifdef CODEDEBUG
	DEBUG(2, ("dlg_singit_config.c [singit_config_show]\n"));
#endif

	singit_config_show_page(0);
}
