/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <sys/stat.h>
#include <unistd.h>
#include <string.h>

#include <X11/Xlib.h>
#include <gtk/gtk.h>

#include <xmms/plugin.h>
#include <xmms/xmmsctrl.h>

#include <stdlib.h>
#include <gdk/gdkkeysyms.h>
#include <stdio.h>

#include "singit_macros.h"
#include "singit_dialogs.h"
#include "singit_config.h"
#include "singit_displayer_plugin.h"
#include "displayer_clist_config.h"
#include "displayer_clist_config_dlg.h"

#include "displayer_clist_status.h"

static GtkWidget *displayer_clist_win = NULL;
static GtkWidget
		*displayer_clist_vbox,
			*displayer_clist_scrlwnd, *displayer_clist_clst,
			*displayer_clist_hbox,
			*displayer_clist_timestamp_status_bar,
			*displayer_clist_song_gtk_event_box, *displayer_clist_song_status_bar;
static GtkTooltips
		*displayer_clist_tooltips;

extern SingitConfigGen *displayer_clist_config;

void displayer_clist_init(void);
void displayer_clist_show(SingitSong *cur_song);
void displayer_clist_hide(SingitSong *cur_song);
void displayer_clist_toggle(SingitSong *cur_song);
void displayer_clist_finish(void);
void displayer_clist_set_time(guint time, SingitSong *cur_song, GList *real_next);
void displayer_clist_config_update(const SingitConfigData *singit_config);
void displayer_clist_configure(void);

DisplayerPlugin clist_dp =
{
	NULL,
	NULL,
	0,

	"Navigation displayer",

	displayer_clist_init,
	displayer_clist_show,
	displayer_clist_hide,
	displayer_clist_toggle,
	displayer_clist_finish,
	NULL,
	displayer_clist_config_update,

	displayer_clist_set_time,

	NULL,
	displayer_clist_configure,
	NULL,
	NULL,

	NULL,
	NULL
};

DIPSPLAYER_SYMBOL(libdisplayer_clist, clist_dp)

void displayer_clist_config_update_event(SingitConfigGen *scg, CListConfigData *cfg_data, gpointer user_data);

static void displayer_clist_clst_jumpToTime(GtkWidget *clist, gint row, gint column, GdkEventButton *event, gpointer data)
{
	GList *item = NULL;
	gboolean isPlaying, doReturn = FALSE;
	SingitSong *cur_song = NULL;

	if ((!event) || (!displayer_clist_win) ||
		((cur_song = singit_song_attach(displayer_clist_status.song)) == NULL)) { return; }

// Prevent to show the focus row
	if (GTK_CLIST(clist)->focus_row != -1) {
		GTK_CLIST(clist)->focus_row = -1;
		if (GTK_CLIST(clist)->freeze_count == 0)
			gtk_widget_queue_draw (clist);
	}

	isPlaying = xmms_remote_is_playing(clist_dp.xmms_session);

	if (!isPlaying && cur_song) {
		if (!doReturn) { doReturn = (cur_song->active_token == NULL); }
		if (!doReturn) { doReturn = ((gint) tLine(cur_song->active_token) == row); }
		if (doReturn) {
			singit_song_detach(&cur_song);
			return;
		}
	}

#	ifdef CODEDEBUG
	DEBUG(("dlg_singit_navigator.c [clst_jumpToTime] : "));
#	endif

	if (displayer_clist_status.lastRow == row) {
		item = cur_song->first_token;
		while (item != cur_song->last_token) {
			if ((gint) tLine(item) == row) {
				if (!isPlaying) { xmms_remote_play(clist_dp.xmms_session); }
				xmms_remote_jump_to_time(clist_dp.xmms_session, tTime(item));
				singit_song_detach(&cur_song);
#				ifdef CODEDEBUG
				DEBUG(("Ok\n"));
#				endif
				return;
			}
			item = g_list_next(item);
		}
		if ((gint) tLine(item) == row) {
			if (!isPlaying) { xmms_remote_play(clist_dp.xmms_session); }
			xmms_remote_jump_to_time(clist_dp.xmms_session, tTime(item));
		}
	}
	else {
		displayer_clist_status.lastRow = row;
	}

	singit_song_detach(&cur_song);

#	ifdef CODEDEBUG
        DEBUG(("Ok\n"));
#	endif
}

static void set_statusbar_messages(gint time)
{
	gint new_message_count;
	gchar timeStamp[10];

	new_message_count = (time / 1000);
	if (new_message_count < 0) { new_message_count = 0; }
	else
		if (!(time % 1000 == 0)) { new_message_count++; }
	if (displayer_clist_status.messageCounter == new_message_count) { return; }

	timeStamp[9] = '\0';
	if (displayer_clist_status.timestamp_context_id != 0) {
		while (displayer_clist_status.messageCounter > new_message_count) {
			gtk_statusbar_pop(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
				displayer_clist_status.timestamp_context_id);
			displayer_clist_status.messageCounter--;
		}
	}
	else {
		displayer_clist_status.timestamp_context_id =
			gtk_statusbar_get_context_id(
				GTK_STATUSBAR(displayer_clist_timestamp_status_bar), "timestamps");
		displayer_clist_status.messageCounter = 0;
	}
	while (displayer_clist_status.messageCounter < new_message_count) {
		displayer_clist_status.messageCounter++;
		sprintf(timeStamp, " - %.2i:%.2i ", displayer_clist_status.messageCounter / 60,
			displayer_clist_status.messageCounter % 60);
		gtk_statusbar_push
			(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
			displayer_clist_status.timestamp_context_id, timeStamp);
	}
}

static void lyrics_not_found(void)
{
	gchar *fileNotFound, *nlbr[1][1];
	gint width, height;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [lyrics_not_found]\n"));
#	endif

	gtk_clist_freeze(GTK_CLIST(displayer_clist_clst));
	gtk_clist_clear(GTK_CLIST(displayer_clist_clst));
	fileNotFound = g_strconcat(_("Lyrics not found"), NULL);
	nlbr[0][0] = fileNotFound;
	gtk_clist_append(GTK_CLIST(displayer_clist_clst), nlbr[0]);
	if (GET_CFG(displayer_clist_config,autoResize) == TRUE) {
		gdk_window_get_size (displayer_clist_win->window, &width, &height);
		gtk_window_set_default_size(GTK_WINDOW(displayer_clist_win), 230, 80);
	}
	g_free(fileNotFound);
	gtk_clist_thaw(GTK_CLIST(displayer_clist_clst));

	if (displayer_clist_status.timestamp_context_id != 0)
		{ set_statusbar_messages(0); }
}

static void update_clist(SingitSong *new_song)
{
        gchar *lbr[2];
        gint i = 0;
	SingitSong *my_song = NULL;
	gint clist_width;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [update_clist]\n"));
#	endif

	gtk_clist_freeze(GTK_CLIST(displayer_clist_clst));
	gtk_clist_clear(GTK_CLIST(displayer_clist_clst));

	my_song = singit_song_attach(new_song);
	if (my_song) {
		if (displayer_clist_status.timestamp_context_id != 0)
			{ set_statusbar_messages(0); }

		// * Fill the clist *
		lbr[1] = NULL;
		i = 0;
		while (my_song->lyrics[i]) {
			lbr[0] = my_song->lyrics[i];
			gtk_clist_append(GTK_CLIST(displayer_clist_clst), lbr);
			i++;
		}

		// * Resize the clist *
		if (GET_CFG(displayer_clist_config,autoResize) == TRUE) {
			clist_width = gtk_clist_columns_autosize
				(GTK_CLIST(displayer_clist_clst));
			if (clist_width < 150) { clist_width = 150; }
			gtk_window_set_default_size(GTK_WINDOW(displayer_clist_win),
				clist_width + 45,
				displayer_clist_status.win_size_y);
		}
		singit_song_detach(&my_song);
	}
	gtk_clist_thaw(GTK_CLIST(displayer_clist_clst));
}

static void set_status_filename(gchar *fileName)
{
	gchar *tmp;
	if (!displayer_clist_win) { return; }
	if (displayer_clist_status.song_context_id) {
		gtk_statusbar_pop(GTK_STATUSBAR
			(displayer_clist_song_status_bar),
			displayer_clist_status.song_context_id);
	}
	else {
		displayer_clist_status.song_context_id =
			gtk_statusbar_get_context_id(GTK_STATUSBAR
				(displayer_clist_song_status_bar), "filename");
	}
	tmp = g_strconcat(" ", fileName, NULL);
	gtk_statusbar_push(GTK_STATUSBAR(displayer_clist_song_status_bar),
		displayer_clist_status.song_context_id, tmp);
	gtk_tooltips_set_tip(displayer_clist_tooltips,
		displayer_clist_song_gtk_event_box, fileName, NULL);
	g_free(tmp);
}

static void select_item(gint new_item, gboolean show)
{
	if (displayer_clist_status.line > -1)
		gtk_clist_unselect_row(GTK_CLIST(displayer_clist_clst),
			displayer_clist_status.line, 0);

	if (show) {
		gtk_clist_select_row(GTK_CLIST(displayer_clist_clst), new_item, 0);
		gtk_clist_moveto(GTK_CLIST(displayer_clist_clst), new_item, -1, 0.5, 0);
	}
	displayer_clist_status.line = new_item;
}

void displayer_clist_set_time(guint time, SingitSong *cur_song, GList *real_next)
{
	GList *item = NULL;
	SingitSong* my_song = NULL, *tmp_song;
	gboolean update_song = FALSE;
	gboolean update_selection = FALSE;
	gboolean update_timer = FALSE;
	gboolean show_item = FALSE;
	gint new_item = -1;
	gint timer;

	if (displayer_clist_win && GTK_WIDGET_VISIBLE(displayer_clist_win) &&
		(gdk_window_is_visible(displayer_clist_win->window)) &&
		(gdk_window_is_viewable(displayer_clist_win->window)))
	{
		my_song = singit_song_attach(cur_song);
		if (!my_song) { return; }

		if ((my_song != displayer_clist_status.song) ||
			(strcmp(my_song->song_filename, displayer_clist_status.song->song_filename) != 0))
		{
			displayer_clist_status.line = -1;
			tmp_song = displayer_clist_status.song;
			displayer_clist_status.song = singit_song_attach(my_song);
			singit_song_detach(&tmp_song);
			update_song = TRUE;
		}

		if (singit_song_text_found(my_song)) {
			// * Has the active token and ??? *
			if (singit_song_lyrics_found(my_song)) {
				if (my_song->active_token != NULL) {
					new_item = tLine(my_song->active_token);
					if (displayer_clist_status.line != new_item)
					{
						if (!singit_song_is_empty_item(my_song, my_song->active_token, FALSE)
							|| (GET_CFG(displayer_clist_config,showEmpty) == TRUE))
						{
							show_item = TRUE;
						}

						update_selection = TRUE;
					}
				}

				// * Has the timer status changed ??? *
				item = (real_next) ? real_next : my_song->last_token;
				if (item != NULL) {
					timer = (time / 1000);
					if (timer < 0) { timer = 0; }
					else
						if (!(time % 1000 == 0)) { timer++; }
					if (displayer_clist_status.messageCounter != timer)
						{ update_timer = TRUE; }
				}
			}

			// * Any updates needed ??? *
			if (update_song || update_selection || update_timer) {

				GDK_THREADS_ENTER();

				// * Run any updates *
				if (update_song) {
					set_status_filename(my_song->song_filename);
					update_clist(my_song);
				}
				if (update_selection)
					{ select_item(new_item, show_item); }
				if (update_timer)
					{ set_statusbar_messages(tTime(item) - time); }

				GDK_THREADS_LEAVE();
			}
		}
		else {
			GDK_THREADS_ENTER();
			lyrics_not_found();
			GDK_THREADS_LEAVE();
		}
		singit_song_detach(&my_song);
	}
}

static gboolean displayer_clist_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume, value;

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
                        clist_dp.disable(&clist_dp);
                        break;
		case GDK_a:
			if (event->state & GDK_CONTROL_MASK) { singit_about_show(); }
			break;
                case GDK_b:
                        xmms_remote_playlist_next(clist_dp.xmms_session);
                        break;
                case GDK_c:
			if (event->state & GDK_CONTROL_MASK) { displayer_clist_config_show(); }
			else { xmms_remote_pause(clist_dp.xmms_session); }
                        break;
		case GDK_e:
			if (event->state & GDK_CONTROL_MASK) { singit_editor_show(); }
			break;
		case GDK_i:
			if (event->state & GDK_CONTROL_MASK) { singit_tag_manager_show(); }
			break;
		case GDK_p:
			if (event->state & GDK_CONTROL_MASK) { singit_config_show(); }
			break;
                case GDK_v:
                        xmms_remote_stop(clist_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(clist_dp.xmms_session);
                        break;
		case GDK_y:
			if (GET_DCC(displayer_clist_config)->gblToggleYZ)
				{ xmms_remote_playlist_prev(clist_dp.xmms_session); }
			break;
                case GDK_z:
			if (!GET_DCC(displayer_clist_config)->gblToggleYZ)
				{ xmms_remote_playlist_prev(clist_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(clist_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(clist_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(clist_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(clist_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (displayer_clist_status.jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(clist_dp.xmms_session) !=
				(displayer_clist_status.jump_to_song - 1)) {
				xmms_remote_set_playlist_pos
					(clist_dp.xmms_session, (displayer_clist_status.jump_to_song - 1));
			}
		case GDK_KP_Decimal:
			displayer_clist_status.jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
		case GDK_0:
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			value = event->keyval - GDK_0;
			if (value < 0 || value > 9)
				value = event->keyval - GDK_KP_0;

			if (event->state & GDK_CONTROL_MASK) {
				enable_plugin(((value + 9) % 10));
			}
			else {
				displayer_clist_status.jump_to_song = (displayer_clist_status.jump_to_song * 10) + value;
				if (displayer_clist_status.jump_to_song > xmms_remote_get_playlist_length
					(clist_dp.xmms_session)) { displayer_clist_status.jump_to_song = value; }
			}
			break;
                default:
                        break;
        }

        return TRUE;
}

static void displayer_clist_resize_event (GtkWidget *widget, gpointer user_data)
{
	gint width, height;
	SingitSong *cur_song = NULL;

	if (!displayer_clist_win) { return; }
	cur_song = singit_song_attach(displayer_clist_status.song);
	if (!cur_song) { return; }
	if (singit_song_lyrics_found(cur_song) && displayer_clist_win->window)
		if (GET_CFG(displayer_clist_config,autoResize) == TRUE) {
			gdk_window_get_size (displayer_clist_win->window, &width, &height);
			displayer_clist_status.win_size_y = height;
		}
	if (displayer_clist_clst)
		if (cur_song->active_token)
			gtk_clist_moveto(GTK_CLIST(displayer_clist_clst), tLine(cur_song->active_token), -1, 0.5, 0);
	singit_song_detach(&cur_song);
}

void displayer_clist_finish(void)
{
	CListConfigData *ccd;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_finish]\n"));
#	endif

	if (displayer_clist_win) {
		ccd = GET_DCC(displayer_clist_config);
		if (displayer_clist_win->window && (ccd != NULL)) {
			gdk_window_get_root_origin(displayer_clist_win->window, &ccd->winPosX, &ccd->winPosY);
			singit_config_gen_save(SINGIT_CONFIG_GEN(displayer_clist_config));
		}
		gtk_widget_destroy(displayer_clist_win);
	}

	singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_clist_config));

	displayer_clist_status_finish();
}

static gint displayer_clist_delete_event(GtkWidget *widget, GdkEvent  *event, gpointer data)
{
	clist_dp.disable(&clist_dp);
	return (FALSE);
}

void displayer_clist_init(void)
{
	gchar* title;
	CListConfigData *ccd;

#	ifdef CODEDEBUG
	DEBUG(("dlg_singit_navigator.c [displayer_clist_init]\n"));
#	endif

	g_return_if_fail(displayer_clist_win == NULL);

	displayer_clist_status_init();

	if (!singit_config_gen_attach(displayer_clist_config)) {
		displayer_clist_config = displayer_clist_config_new();
		singit_config_gen_load(SINGIT_CONFIG_GEN(displayer_clist_config));
	}

	ccd = GET_DCC(displayer_clist_config);
	if (!ccd) {
		singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_clist_config));
		return;
	}

	gtk_signal_connect(GTK_OBJECT(displayer_clist_config), "update",
		GTK_SIGNAL_FUNC(displayer_clist_config_update_event), NULL);

	displayer_clist_tooltips = gtk_tooltips_new ();

	displayer_clist_win = gtk_window_new(GTK_WINDOW_DIALOG);
	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("Lyrics navigator"), NULL);
	gtk_window_set_title(GTK_WINDOW(displayer_clist_win), title);
	g_free(title);
	gtk_window_set_policy(GTK_WINDOW(displayer_clist_win), TRUE, TRUE, TRUE);
	gtk_widget_set_usize(displayer_clist_win, 180, 200);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win),
		"destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_clist_win);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win),
		"delete_event", GTK_SIGNAL_FUNC(displayer_clist_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win),
		"key-press-event", GTK_SIGNAL_FUNC(displayer_clist_win_keypress), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_win),
		"size-allocate", GTK_SIGNAL_FUNC(displayer_clist_resize_event), NULL);
	gtk_container_border_width(GTK_CONTAINER(displayer_clist_win), 5);

	displayer_clist_vbox = gtk_vbox_new(FALSE, 0);
	gtk_container_add(GTK_CONTAINER(displayer_clist_win), displayer_clist_vbox);
	gtk_widget_show(displayer_clist_vbox);

	displayer_clist_scrlwnd = gtk_scrolled_window_new(NULL, NULL);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(displayer_clist_scrlwnd), GTK_POLICY_AUTOMATIC, GTK_POLICY_ALWAYS);
	gtk_box_pack_start(GTK_BOX(displayer_clist_vbox), displayer_clist_scrlwnd, TRUE, TRUE, 0);
	gtk_widget_show (displayer_clist_scrlwnd);

	displayer_clist_clst = gtk_clist_new(1);
	gtk_signal_connect(GTK_OBJECT(displayer_clist_clst),
		"select_row", GTK_SIGNAL_FUNC(displayer_clist_clst_jumpToTime), NULL);
	gtk_clist_set_selection_mode(GTK_CLIST(displayer_clist_clst), GTK_SELECTION_SINGLE);
	gtk_clist_set_reorderable (GTK_CLIST(displayer_clist_clst), FALSE);
	gtk_container_add(GTK_CONTAINER(displayer_clist_scrlwnd), displayer_clist_clst);
	gtk_clist_set_column_width(GTK_CLIST(displayer_clist_clst), 0, 1);
	gtk_widget_show(displayer_clist_clst);

	displayer_clist_hbox = gtk_hbox_new(FALSE, 0);
	gtk_box_pack_start(GTK_BOX(displayer_clist_vbox), displayer_clist_hbox, FALSE, FALSE, 1);
	gtk_widget_show(displayer_clist_hbox);

	displayer_clist_timestamp_status_bar = gtk_statusbar_new();
	gtk_box_pack_start (GTK_BOX(displayer_clist_hbox),
		displayer_clist_timestamp_status_bar, FALSE, FALSE, 0);
	gtk_widget_show(displayer_clist_timestamp_status_bar);

	displayer_clist_song_gtk_event_box = gtk_event_box_new ();
	gtk_container_add (GTK_CONTAINER(displayer_clist_hbox),
		displayer_clist_song_gtk_event_box);
	gtk_widget_show (displayer_clist_song_gtk_event_box);

	displayer_clist_song_status_bar = gtk_statusbar_new();
	gtk_container_add (GTK_CONTAINER (displayer_clist_song_gtk_event_box),
		displayer_clist_song_status_bar);
	gtk_widget_show(displayer_clist_song_status_bar);

	gtk_object_set_data (GTK_OBJECT (displayer_clist_win),
		"tooltips", displayer_clist_tooltips);

// * Normally a window is created when it's shown first, so if not shown *
// * we must explicit realize it to get a GDK_WINDOW *
	gtk_widget_realize(displayer_clist_win);

	displayer_clist_status.song_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(displayer_clist_song_status_bar), "filename");

	displayer_clist_status.timestamp_context_id =
		gtk_statusbar_get_context_id(GTK_STATUSBAR(displayer_clist_timestamp_status_bar), "timestamps");
	gtk_statusbar_push(GTK_STATUSBAR(displayer_clist_timestamp_status_bar),
		displayer_clist_status.timestamp_context_id, "         ");

	if ((ccd->winPosX >= 0) && (ccd->winPosY >= 0))
		gdk_window_move(displayer_clist_win->window, ccd->winPosX, ccd->winPosY);
	else
		gtk_window_set_position
			(GTK_WINDOW(displayer_clist_win), GTK_WIN_POS_CENTER);

	displayer_clist_config_update_event(NULL, ccd, NULL);
}

void displayer_clist_show(SingitSong *cur_song)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_show]\n"));
#	endif

	if (displayer_clist_win) {
		if (!GTK_WIDGET_VISIBLE(displayer_clist_win))
			{ gtk_widget_show(displayer_clist_win); }
		else { gdk_window_raise(displayer_clist_win->window); }
	}
}

void displayer_clist_hide(SingitSong *cur_song)
{
	gboolean hide_it = TRUE;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_hide]\n"));
#	endif

	if (displayer_clist_config != NULL) {
		hide_it = GET_CFG(displayer_clist_config, hideIfNotFound);
		if ((hide_it == TRUE) && (GET_DCC(displayer_clist_config)->showOnText == TRUE)) {
			if (cur_song != NULL) {
				hide_it = (singit_song_text_found(cur_song) == FALSE);
			}
		}
	}

	if (displayer_clist_win) {
		if (GTK_WIDGET_VISIBLE(displayer_clist_win) && hide_it)
			{ gtk_widget_hide(displayer_clist_win); }
	}
}

void displayer_clist_toggle(SingitSong *cur_song)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_toggle]\n"));
#	endif

	if (displayer_clist_win) {
		if (GTK_WIDGET_VISIBLE(displayer_clist_win))
			{ gtk_widget_hide(displayer_clist_win); }
		else  { gtk_widget_show(displayer_clist_win); }
	}
}

void displayer_clist_config_update(const SingitConfigData *singit_config)
{
	CListConfigData *ccd = GET_DCC(displayer_clist_config);

	ccd->gblToggleYZ = singit_config->toggleYZ;

	if (ccd->forceGlobals == TRUE)
		{ displayer_clist_config_update_event(displayer_clist_config, ccd, NULL); }
}

void displayer_clist_configure(void)
{
#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_configure]\n"));
#	endif

	displayer_clist_config_show();
}

void displayer_clist_config_update_event(SingitConfigGen *scg, CListConfigData *cfg_data, gpointer user_data)
{
#	define REAL_CFG(val) \
		((cfg_data->forceGlobals == TRUE) ? GET_SCD->val : cfg_data->val)

	GtkRcStyle *rc_style;
	GdkFont *newFont;
	gchar *font_name;
	GtkCList *clist;

#	ifdef CODEDEBUG
	DEBUG(("displayer_clist.c [displayer_clist_config_update_event]\n"));
#	endif

	if (displayer_clist_win == NULL) { return; }

	if ((cfg_data->forceGlobals == TRUE) && GET_SCD) {
		cfg_data->autoResize = GET_SCD->autoResize;
		cfg_data->debugEnable = GET_SCD->debugEnable;
		cfg_data->showEmpty = GET_SCD->showEmpty;
	}

	clist = GTK_CLIST(displayer_clist_clst);
	rc_style = gtk_rc_style_new ();

	gtk_clist_freeze(clist);

// * Check if we could load a font with this name
	font_name = REAL_CFG(lyricFontName);
	if ((newFont = gdk_font_load(font_name)) != NULL) {
		gdk_font_unref(newFont);
		rc_style->font_name = g_strdup(font_name);
		rc_style->fontset_name = g_strdup(font_name);
	}

	/* Ok - is there any logic to use these colors ??? */

// color of a selected row
	gdk_color_parse(REAL_CFG(progressBarColor), &rc_style->bg[GTK_STATE_SELECTED]);
	rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_BG;

// color of the normal text
	gdk_color_parse(REAL_CFG(normalTextColor), &rc_style->fg[GTK_STATE_NORMAL]);
	rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_FG;

// color of the a selected row text
	gdk_color_parse(REAL_CFG(progressTextColor), &rc_style->fg[GTK_STATE_SELECTED]);
	rc_style->color_flags[GTK_STATE_SELECTED] |= GTK_RC_FG;

// background color of the row
	gdk_color_parse(REAL_CFG(normalBarColor), &rc_style->base[GTK_STATE_NORMAL]);
	rc_style->color_flags[GTK_STATE_NORMAL] |= GTK_RC_BASE;

// "background" color of the space between the rows
	gdk_color_parse(REAL_CFG(backgroundColor), &rc_style->base[GTK_STATE_ACTIVE]);
	rc_style->color_flags[GTK_STATE_ACTIVE] |= GTK_RC_BASE;

	gtk_widget_modify_style(displayer_clist_clst, rc_style);

	gtk_rc_style_unref(rc_style);

// * Reseize the window *
	if (REAL_CFG(autoResize) == TRUE) {
		gtk_window_set_default_size(GTK_WINDOW(displayer_clist_win),
			gtk_clist_columns_autosize(clist) + 45,
			displayer_clist_status.win_size_y);
	}

	gtk_clist_thaw(clist);

#	undef REAL_CFG
}
