/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2002 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <assert.h>
#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <gltt/FTEngine.h>
#include <gltt/FTFace.h>
#include <gltt/FTInstance.h>
#include <gltt/FTGlyph.h>
#include <gltt/FTGlyphBitmap.h>
#include <gltt/FTGlyphPixmap.h>
#include <gltt/FTBitmapFont.h>
#include <gltt/GLTTBitmapFont.h>
#include <gltt/GLTTPixmapFont.h>
#include <gltt/GLTTOutlineFont.h>

#include <GL/glu.h>
#include <GL/glut.h>

#include "displayer_opengl.h"
#include "clyrictextengine.h"

class GLTTGlyphTriangles: public GLTTGlyphTriangulator
{
public:
	// Type
	struct Triangle {

		FTGlyphVectorizer::POINT* p1;
		FTGlyphVectorizer::POINT* p2;
		FTGlyphVectorizer::POINT* p3;
	};

	// Data
	Triangle *triangles;
	int nTriangles;
	GLTTboolean count_them;

	// Constructor
	GLTTGlyphTriangles (FTGlyphVectorizer* vectorizer) :GLTTGlyphTriangulator(vectorizer)
	{
		triangles = 0;
		nTriangles = 0;
		count_them = GLTT_TRUE;
	}

	virtual ~GLTTGlyphTriangles()
	{
		delete[] triangles;
		triangles = 0;
	}

	void alloc()
	{
		delete[] triangles;
		triangles= new Triangle [ nTriangles + 1 ];
	}

	virtual void triangle( FTGlyphVectorizer::POINT* p1,
		FTGlyphVectorizer::POINT* p2, FTGlyphVectorizer::POINT* p3 )
	{
		if (count_them) {
			++nTriangles;
			return;
		}
		triangles[nTriangles].p1 = p1;
		triangles[nTriangles].p2 = p2;
		triangles[nTriangles].p3 = p3;
		++nTriangles;
	}
};

class GLGlyphData {
public:

	FTGlyphVectorizer *vec;
	GLTTGlyphTriangles *tri;

	GLGlyphData()
	{
		vec = new FTGlyphVectorizer;
		tri = new GLTTGlyphTriangles(vec);
	}

	~GLGlyphData()
	{
		delete tri;
		for (int c = 0; c < vec->getNContours(); ++c) {
			FTGlyphVectorizer::Contour* contour = vec->getContour(c);
			if (contour == 0) continue;
			for (int i = 0; i < contour->nPoints; ++i ) {
				FTGlyphVectorizer::POINT* point= contour->points + i;
				delete [] (double*) point->data;
				point->data= 0;
			}
		}
		delete vec;

	}
};

/* Main class */

CLyricTextEngine::CLyricTextEngine()
{
	lines = 0;
	song = 0;
	start_line = start_char = 0;
	line_quantity = 0;
	glyths = 0;
	font = 0;
	max_width = max_height = 0;
	current = 0;
	show_empty_lines = true;

	face = new FTFace;
	if (!face->open(SINGIT_DATA_DIR "/vixar.ttf")) {
		delete face;
		face = 0;
		return;
	}

	font = new GLTTFont(face);
	if (!font->create(166)) {
		delete font;
		delete face;
		font = 0;
		face = 0;
		return;
	}

	g_datalist_init(&glyph_data_list);
}

void data_delete_func (GQuark key_id, gpointer data, gpointer user_data)
{
	if (data != 0) {
		delete (GLGlyphData *) data;
		((CLyricTextEngine *) user_data)->glyths--;
	}
}

CLyricTextEngine::~CLyricTextEngine()
{
	if (font == 0) { return; }

	g_datalist_foreach(&glyph_data_list, data_delete_func, this);
	g_datalist_clear(&glyph_data_list);

//	printf("Stored glypths: %i\n", glyths);

	delete font;
	delete face;
	font = NULL;
	face = NULL;

	free_lines();
	lines = 0;
	if (song != NULL)
		{ singit_song_detach(&song); }
}

void CLyricTextEngine::set_positions(guint time)
{
	GList *new_item = NULL, *next_item;
	SingitSong *my_song = NULL;
	gchar *text = NULL, *startPos, *endPos = NULL;
	gchar oldChar;
	gboolean recheck = FALSE;
	double multi = 0.0;

	if (font == 0) { return; }

	my_song = singit_song_attach(song);
	if (my_song) {
		new_item = my_song->active_token;
		next_item = inl_singit_song_get_next_token(my_song);
		if ((new_item) && (new_item != current)) {
			text = my_song->lyrics[tLine(new_item)];
			if (new_item != my_song->last_token) { recheck = (tPos(g_list_next(new_item))); }
			if ((tPos(new_item) > 0) || recheck) {
				startPos = text + tPos(new_item);
				oldChar = startPos[0];
				startPos[0] = '\0';
				pbp_start = font->getWidth(text);
				startPos[0] = oldChar;
				recheck = TRUE;
				if (next_item) {
					recheck = (tLine(next_item) != tLine(new_item));
					if (recheck) { recheck = (tPos(next_item) == strlen(tText(my_song, new_item))); }
				}
				if (recheck) { pbp_offset_max = font->getWidth(startPos); }
				else {
					if (next_item) {
						endPos = text + tPos(next_item);
						oldChar = endPos[0];
						endPos[0] = '\0';
					}
					pbp_offset_max = font->getWidth(startPos);
					if (new_item != my_song->last_token) { endPos[0] = oldChar; }
				}
			}
			else {
				pbp_start = pbp_offset = 0;
				pbp_offset_max = font->getWidth(text);
			}
			if (current != new_item) {
				recheck = TRUE;
				if (current && new_item)
					if (tLine(new_item) == tLine(current)) { recheck = FALSE; }
				if (recheck) {
					if ((current == g_list_previous(new_item)) && (current != NULL) &&
						(!show_empty_lines)) {
						if (!singit_song_is_empty_item(my_song, current, FALSE)) {
							start_line = tLine(new_item);
						}
					}
					else {
						start_line = tLine(new_item);
					}
				}
				current = new_item;
			}
		}

		if (new_item) {
			if (new_item != my_song->last_token) {
				multi = (time - tTime(new_item)) / (tTime(next_item) - tTime(new_item));
				if ((multi > 1.0) || (multi < 0.0)) { pbp_offset = pbp_offset_max; }
				else {
					pbp_offset = pbp_offset_max *
						(time - tTime(new_item)) / (tTime(next_item) - tTime(new_item));
				}
			}
			else { pbp_offset = pbp_offset_max; }
		}
		else {
			if ((my_song->first_token) && (new_item != current)) {
				start_line = tLine(new_item);
			}
			current = new_item;
		}
/*		if (use_ball) {
			singit_karaoke_widget_set_curr_y_pos(skw);
			singit_karaoke_widget_update_ball (skw);
		}*/
		if (pbp_offset_last != pbp_offset) {
			pbp_offset_last = pbp_offset;
//			singit_karaoke_widget_update_progess_bar(skw, tText(my_song, new_item));
		}
		singit_song_detach(&my_song);
	}
}

void draw_colored_char(GLGlyphData *glyph, GLfloat r, GLfloat g,
	GLfloat b, GLfloat a = 1.0, double pos_z = 0.0)
{
	glBegin(GL_TRIANGLES);
	glColor4f(r,g,b,a);
	for( int j = 0; j < glyph->tri->nTriangles; j++ ) {
		GLTTGlyphTriangles::Triangle *t = &glyph->tri->triangles[j];
		double* p1= ((double*) t->p1->data);
		p1[2] += pos_z;
		double* p2= ((double*) t->p2->data);
		p2[2] += pos_z;
		double* p3= ((double*) t->p3->data);
		p3[2] += pos_z;
		double* n1= p1 + 3;
		n1[2] += pos_z;
		double* n2= p2 + 3;
		n2[2] += pos_z;
		double* n3= p3 + 3;
		n3[2] += pos_z;

		// Dreiecke eines Buchstabes
		glNormal3dv( n1 );
		glVertex3dv( p1 );

		glNormal3dv( n2 );
		glVertex3dv( p2 );

		glNormal3dv( n3 );
		glVertex3dv( p3 );
	}
	glEnd();
}

void CLyricTextEngine::drawLine(gchar *text, bool enlighten, gdouble border)
{
	gfloat decender = font->getDescender() / (gfloat) max_width - border;
	gfloat height = font->getHeight() / (gdouble) max_width + decender + 2 * border;

	glDisable(GL_BLEND);
	glDisable(GL_DEPTH_TEST);
	glBegin(GL_QUADS);
		glColor4f(0.0f, 0.0f, 0.0f, 1.0f);
		glVertex3f(-1 * border, decender, -0.001f);
		glVertex3f(-1 * border, height, -0.001f);
		glVertex3f(1.0f + border, height, -0.001f);
		glVertex3f(1.0f + border, decender, -0.001f);
	glEnd();
	glEnable(GL_BLEND);
	glEnable(GL_DEPTH_TEST);

	if (!text) { return; }

	gchar *test = g_strdup(" ");
	guint ichar = 0;
	double base_x = 0;

	while (text[ichar] != 0) {
		// Konturpunkte im Hash suchen
		test[0] = text[ichar];
		GLGlyphData *glyph = (GLGlyphData*) g_datalist_get_data(&glyph_data_list, test);
		FTGlyphVectorizer *vc = glyph->vec;

		// 2D Konturpunkte des Buchstaben nach 3D "mappen"
		for( int c = 0; c < vc->getNContours(); ++c ) {
			FTGlyphVectorizer::Contour* contour = vc->getContour(c);
			if( contour == 0 ) continue;
			for( int j = 0; j < contour->nPoints; ++j ) {
				FTGlyphVectorizer::POINT* point = contour->points + j;

				double* p= (double*) point->data;
				double* n= p + 3;

				p[0]= (base_x + point->x) / max_width;
				p[1]= point->y / max_width;
				p[2]= 0;

				n[0]= 0;
				n[1]= 0.;
				n[2]= -1;
			}
		}

		// 3D Konturpunke in Dreiecke umsetzen und farbigen Buchstaben malen
		draw_colored_char(glyph, 0.4, 0.4, 0.8, 1.0, -0.0001f);
		draw_colored_char(glyph, 0.6, 0.6, 1.0, 0.7);

		// Enlighten an active line
		if (enlighten) {
			GLfloat alpha = 0.84;
			double pos_z = 0.0001f;
			for (int i = 0; i < 4; i++) {
				draw_colored_char(glyph, 1.0, 1.0, 1.0, alpha, pos_z);
				alpha *= 0.84f;
				pos_z += 0.0001f;
			}
		}

		base_x += vc->getAdvance();
		ichar++;
	}
	g_free(test);
}

void CLyricTextEngine::draw(guint time)
{
//	GLfloat light1_position[] = { 0.0, 0.0, 3.0, 0.0 };

#define OVERALL_LINE_DISTANCE 0.065f
#define ACTIVE_LINE_DISTANCE 1.3f

	if (!(font && lines && (start_line < line_quantity)))  { return; }

	/* Funtions On */
	glEnable(GL_NORMALIZE);
	glEnable(GL_BLEND);

	/* Options */
	glBlendFunc(GL_SRC_ALPHA,GL_ONE);

	glPushMatrix();
	glLoadIdentity();
	glScalef(3.0f, 3.0f, 3.0f);
	glTranslatef(-0.5f,0.25f,-1.0f);

//	glLightfv(GL_LIGHT1, GL_POSITION, light1_position);

	set_positions(time);

	double base_x = 0;

	// Zeilenabstand
	gdouble step_y = -1.0 * ((gdouble) max_height / (gdouble) max_width + OVERALL_LINE_DISTANCE);

	// Anzahl der geschriebenen Zeilen
	guint printed_lines = 0;

	SingitSong *my_song = singit_song_attach(song);
	if ((my_song == NULL) || (my_song->first_token == NULL)) {
		singit_song_detach(&my_song);
		return;
	}

	GList *token = singit_song_find_prev_lyric_line(my_song, my_song->active_token, TRUE, NULL);

	if (!token) {
		token = my_song->first_token;
		if (token) {
			drawLine(0, false);
			glTranslatef(0.0, step_y + (step_y / ACTIVE_LINE_DISTANCE), 0.0);
			printed_lines = 1;
		}
	}

	while (printed_lines < 4) {
		gchar *text = (token) ? lines[tLine(token)] : 0;
		if (printed_lines == 1) {
			glBlendFunc(GL_SRC_ALPHA,GL_ONE);
			drawLine(text, true, 0.05);
		}
		else {
			glBlendFunc(GL_SRC_ALPHA,GL_ONE);
			drawLine(text, false);
		}
		base_x = 0.0;
		printed_lines++;
		glTranslatef(0.0, step_y, 0.0);
		if ((printed_lines == 1) || (printed_lines == 2))
			{ glTranslatef(0.0, (step_y / ACTIVE_LINE_DISTANCE), 0.0); }
		token = singit_song_find_next_lyric_line(my_song, token, TRUE, NULL);
	}
	glPopMatrix();

	/* Funtions Off */
	glDisable(GL_NORMALIZE);
	glDisable(GL_BLEND);

	singit_song_detach(&my_song);
}

void CLyricTextEngine::set_max_height_and_width()
{
	#ifdef CODEDEBUG
	DEBUG(("clyrictextengine.c [set_max_height_and_width]\n"));
	#endif
	max_width = max_height = 0;
	if (lines != 0) {
		max_height = font->getHeight();
		gint i = 0;
		while (lines[i] != 0) {
			guint new_width = font->getWidth(lines[i]);
			if (new_width > max_width) { max_width = new_width; }
			i++;
		}
	}
}

bool CLyricTextEngine::create_glyth_data(gchar character)
{
	gchar *test = g_strdup(" ");
	test[0] = character;
	GQuark quark = g_quark_from_string(test);

	if (!g_datalist_id_get_data(&glyph_data_list, quark)) {

		#ifdef CODEDEBUG
		DEBUG(("clyrictextengine.c [create_glyth_data - gchar] : %s\n", test));
		#endif

		g_free(test);
		GLGlyphData *gl_glyth_data = new GLGlyphData;
		int ch = (unsigned char) character;
		FTGlyph* glyph = font->getFont()->getGlyph(ch);
		if (!glyph) return false;

		FTGlyphVectorizer *vec = gl_glyth_data->vec;
		vec->setPrecision(20.);
		if (!vec->init(glyph)) return false;
		if (!vec->vectorize()) return false;

		for (int c = 0; c < vec->getNContours(); ++c ) {
			FTGlyphVectorizer::Contour* contour= vec->getContour(c);
			if (contour == 0) return false;
			for (int j= 0; j < contour->nPoints; ++j) {
				FTGlyphVectorizer::POINT* point= contour->points + j;
				point->data= (void*) new double [6];
			}
		}

		GLTTGlyphTriangles* tri = gl_glyth_data->tri;
		if (!tri->init(glyph)) return false;
		tri->count_them = GLTT_TRUE;
		tri->nTriangles = 0;
		tri->triangulate();
		tri->count_them = GLTT_FALSE;
		tri->alloc();
		tri->nTriangles = 0;
		tri->triangulate();

		g_datalist_id_set_data(&glyph_data_list, quark, gl_glyth_data);

		glyths++;
	}
	else { g_free(test); }
	return true;
}

void CLyricTextEngine::free_lines()
{
	#ifdef CODEDEBUG
	DEBUG(("clyrictextengine.c [free_lines]\n"));
	#endif
	if (lines != 0) {
		if (song != 0) {
			if (song->lyrics != lines) { g_strfreev(lines); }
		}
		else { g_strfreev(lines); }
	}
}

void CLyricTextEngine::create_glyth_data(gchar **new_lines)
{
	gint counter_strings = 0, counter_chars;

	#ifdef CODEDEBUG
	DEBUG(("clyrictextengine.c [create_glyth_data - gchar**]\n"));
	#endif

	free_lines();
	lines = new_lines;
	set_max_height_and_width();
	while (lines[counter_strings]) {
		counter_chars = 0;
		while (lines[counter_strings][counter_chars]) {
			if (!create_glyth_data(lines[counter_strings][counter_chars])) {
				#ifdef CODEDEBUG
				DEBUG(("clyrictextengine.c [create_glyth_data - gchar] : Error\n"));
				#endif
			}
			counter_chars++;
		}
		counter_strings++;
	}
	line_quantity = counter_strings;
}

void CLyricTextEngine::create_glyth_data(gchar *new_text)
{
	gint counter_strings = 0, counter_chars;

	#ifdef CODEDEBUG
	DEBUG(("clyrictextengine.c [create_glyth_data - gchar*]\n"));
	#endif

	free_lines();
	lines = g_strsplit(new_text, "\n", 1000);
	set_max_height_and_width();

	while (lines[counter_strings]) {
		counter_chars = 0;
		while (lines[counter_strings][counter_chars]) {
			if (!create_glyth_data(lines[counter_strings][counter_chars])) {
				#ifdef CODEDEBUG
				DEBUG(("clyrictextengine.c [create_glyth_data - gchar] : Error\n"));
				#endif
			}
			counter_chars++;
		}
		counter_strings++;
	}
	line_quantity = counter_strings;
}

void CLyricTextEngine::set_song(SingitSong *new_song)
{
	#ifdef CODEDEBUG
	DEBUG(("clyrictextengine.c [set_song]\n"));
	#endif

	if (font == 0) { return; }

	start_line = start_char = 0;

	if ((song != NULL) && (song->lyrics == lines)) 
		{ lines = 0; }
	singit_song_detach(&song);
	song = singit_song_attach(new_song);
	if (song != 0) {
		if (singit_song_lyrics_found(song)) {
			create_glyth_data(song->lyrics);
			#ifdef CODEDEBUG
			DEBUG(("Stored glypths: %i\n", glyths));
			#endif
		}
	}
}

void CLyricTextEngine::set_text(gchar *new_text)
{
	if (font == 0) { return; }

	start_line = start_char = 0;

	create_glyth_data(new_text);
}
