/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <GL/gl.h>
#include <GL/glu.h>

#include <gtk/gtk.h>
#include <gdk/gdkkeysyms.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>

#include "singit_displayer_plugin.h"
#include "displayer_opengl.h"

#include "gtkglarea.h"
#include "c_wrapper.h"
#include "singit_config.h"

#include "displayer_opengl_config.h"
#include "displayer_opengl_config_dlg.h"

#include "singit_dialogs.h"

void displayer_opengl_init(void);
void displayer_opengl_finish(void);

void displayer_opengl_show(SingitSong *cur_song);
void displayer_opengl_hide(SingitSong *cur_song);
void displayer_opengl_toggle(SingitSong *cur_song);
void displayer_opengl_render_freq(gint16 freq_data[2][256], SingitSoundPrecalcs *ssp);
void displayer_opengl_render_pcm(gint16 pcm_data[2][512], SingitSoundPrecalcs *ssp);
void displayer_opengl_configure(void);

void displayer_opengl_set_time(guint time, SingitSong *cur_song, GList *real_next);
void displayer_opengl_config_update(const SingitConfigData *singit_config);

// static void displayer_opengl_limit_fps(gboolean init);

static gboolean displayer_opengl_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data);

DisplayerPlugin opengl_dp =
{
	NULL,
	NULL,
	0,

	"OpenGL displayer",

	displayer_opengl_init,
	NULL,
	NULL,
	NULL,
	displayer_opengl_finish,
	NULL,
	displayer_opengl_config_update,

	displayer_opengl_set_time,

	NULL,
	displayer_opengl_configure,
	NULL,
	NULL,

	displayer_opengl_render_pcm,
	displayer_opengl_render_freq
};

DIPSPLAYER_SYMBOL(libdisplayer_opengl, opengl_dp)

static GtkWidget *displayer_opengl_window = NULL;

gint16 static_pcm_data[2][512];
gint16 static_freq_data[2][256];
float madspin_data[256];

GtkWidget* create_glarea         (void);
gint       glarea_button_release (GtkWidget*, GdkEventButton*);
gint       glarea_button_press   (GtkWidget*, GdkEventButton*);
gint       glarea_motion_notify  (GtkWidget*, GdkEventMotion*);
gint       glarea_draw           (GtkWidget*, GdkEventExpose*);
gint       glarea_reshape        (GtkWidget*, GdkEventConfigure*);
gint       glarea_init           (GtkWidget*);
gint       glarea_destroy        (GtkWidget*);

static gint displayer_opengl_delete_event( GtkWidget *widget, GdkEvent  *event, gpointer data )
{
	opengl_dp.disable(&opengl_dp);
	return (FALSE);
}

static gint displayer_opengl_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	opengl_dp.disable(&opengl_dp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

void displayer_opengl_init(void)
{
	gchar *title;
	GtkWidget* box_main;
	gint posX, posY;
	OpenGLConfigData *oglcd;

	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_init]\n"));
	#endif

	g_return_if_fail(displayer_opengl_window == NULL);

	displayer_opengl_status_init();

	if (!singit_config_gen_attach(displayer_opengl_config)) {
		displayer_opengl_config = displayer_opengl_config_new();
		singit_config_gen_load(SINGIT_CONFIG_GEN(displayer_opengl_config));
	}

	oglcd = GET_OGLCD(displayer_opengl_config);
	if (!oglcd) {
		singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_opengl_config));
		return;
	}

	displayer_opengl_status.gl_area_widget = create_glarea();
	if (displayer_opengl_status.gl_area_widget == NULL) {
		gtk_idle_add (displayer_opengl_disable_func, NULL);
		return;
	}
	gtk_signal_connect(GTK_OBJECT(displayer_opengl_status.gl_area_widget),
		"destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_opengl_status.gl_area_widget);

	displayer_opengl_window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_container_border_width(GTK_CONTAINER(displayer_opengl_window), 2);
	gtk_signal_connect(GTK_OBJECT(displayer_opengl_window), "delete_event", GTK_SIGNAL_FUNC(displayer_opengl_delete_event), NULL);
	gtk_signal_connect(GTK_OBJECT(displayer_opengl_window), "destroy", GTK_SIGNAL_FUNC(gtk_widget_destroyed), &displayer_opengl_window);
	gtk_signal_connect(GTK_OBJECT(displayer_opengl_window), "key-press-event", GTK_SIGNAL_FUNC(displayer_opengl_win_keypress), NULL);
	title = g_strconcat(SINGIT_VERSION_STRING, " - ", _("OpenGL displayer"), NULL);
	gtk_window_set_title(GTK_WINDOW(displayer_opengl_window), title);
	g_free(title);
	gtk_window_set_policy(GTK_WINDOW(displayer_opengl_window), TRUE, TRUE, TRUE);

	box_main = gtk_vbox_new(FALSE, 10);
	gtk_container_add(GTK_CONTAINER(displayer_opengl_window), box_main);
	gtk_widget_show(box_main);

	gtk_box_pack_start (GTK_BOX(box_main), displayer_opengl_status.gl_area_widget, TRUE, TRUE, 0);
	gtk_widget_show(displayer_opengl_status.gl_area_widget);

	gtk_widget_show(displayer_opengl_window);

	if (singit_config_load_plugin_position("opengl_window", &posX, &posY, NULL, NULL)) {
		gtk_widget_set_uposition(displayer_opengl_window, posX, posY);
	}
	else
		gtk_window_set_position
			(GTK_WINDOW(displayer_opengl_window), GTK_WIN_POS_CENTER);

//	displayer_opengl_limit_fps(TRUE);
}

void displayer_opengl_finish(void)
{
	OpenGLConfigData *oglcd;

	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_finish]\n"));
	#endif

	if (displayer_opengl_window) {
		oglcd = GET_OGLCD(displayer_opengl_config);
		if (displayer_opengl_window->window && (oglcd != NULL)) {
			gdk_window_get_root_origin(displayer_opengl_window->window,
				&oglcd->winPosX, &oglcd->winPosY);
			singit_config_gen_save(SINGIT_CONFIG_GEN(displayer_opengl_config));
		}
		gtk_widget_destroy(displayer_opengl_window);
	}

	singit_config_gen_detach(SINGIT_CONFIG_GEN(displayer_opengl_config));

	displayer_opengl_status_finish();
}

void displayer_opengl_show(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_show]\n"));
	#endif

	if (displayer_opengl_window) {
		if (!GTK_WIDGET_VISIBLE(displayer_opengl_window))
			{ gtk_widget_show(displayer_opengl_window); }
		else { gdk_window_raise(displayer_opengl_window->window); }
	}
}

void displayer_opengl_hide(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_hide]\n"));
	#endif

	if (displayer_opengl_window) {
 		if (GTK_WIDGET_VISIBLE(displayer_opengl_window))
			gtk_widget_hide(displayer_opengl_window);
	}
}

void displayer_opengl_toggle(SingitSong *cur_song)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_toggle]\n"));
	#endif

	if (displayer_opengl_window) {
		if (GTK_WIDGET_VISIBLE(displayer_opengl_window)) {
			gtk_widget_hide(displayer_opengl_window);
		}
		else { gtk_widget_show(displayer_opengl_window); }
	}
}

void displayer_opengl_set_time(guint time, SingitSong *cur_song, GList *real_next)
{
	SingitSong *my_song;

	if (displayer_opengl_window != NULL)
	{
		my_song = singit_song_attach(cur_song);
		if ((my_song != displayer_opengl_status.song) ||
			(strcmp(my_song->song_filename, displayer_opengl_status.song->song_filename) != 0))
		{
			singit_song_detach(&displayer_opengl_status.song);
			displayer_opengl_status.song = singit_song_attach(my_song);
			displayer_opengl_status.rotate = !singit_song_lyrics_found(my_song);
			GDK_THREADS_ENTER();
			opengl_lyrics_set_song(my_song);
			gtk_widget_queue_draw(displayer_opengl_status.gl_area_widget);
			GDK_THREADS_LEAVE();
		}
		singit_song_detach(&my_song);
		displayer_opengl_status.time = time;
	}
}

void displayer_opengl_config_update(const SingitConfigData *singit_config)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_config_update]\n"));
	#endif

	displayer_opengl_status.toggle_yz = singit_config->toggleYZ;
}

void displayer_opengl_configure(void)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [displayer_opengl_configure]\n"));
	#endif

	displayer_opengl_config_show();
}

void displayer_opengl_render_pcm(gint16 pcm_data[2][512], SingitSoundPrecalcs *ssp)
{
	if (displayer_opengl_status.gl_area_widget != NULL) {
		memcpy(static_pcm_data, pcm_data, 2 * 2 * 512);
		displayer_opengl_status.beat = ssp->beat;

		gtk_widget_queue_draw(displayer_opengl_status.gl_area_widget);
	}
}

void displayer_opengl_render_freq(gint16 freq_data[2][256], SingitSoundPrecalcs *ssp)
{
	gint i;
	if (displayer_opengl_window) {

		memcpy(static_freq_data, freq_data, 2 * 2 * 256);

		for (i = 0; i < 256; i++)
		{
			madspin_data[i] = (float)static_freq_data[0][i] / 5000.0;
			if (madspin_data[i] > 1.0)
				madspin_data[i] = 1.0;
		}

		displayer_opengl_status.level[0] = ssp->level[0];
		displayer_opengl_status.level[1] = ssp->level[1];
		displayer_opengl_status.level[2] = ssp->level[2];

//		g_print("levels - l: %f / r: %f / c: %f\n", ssp->level[0], ssp->level[1], ssp->level[2]);
	}

/*
	const gint scale[] =
		{ 0, 1, 2, 3, 5, 7, 10, 14, 20, 28, 40, 54, 74, 101, 137, 187, 255 };

	int i, j, h;
	for (j = 0;j != 2;j++) {
		for (i = 0;i != 16;i++) {
			int tmp = 0;
			for ( h = scale[i];h < scale[i + 1];h++) {
				if (freq_data[j][h] > tmp)
					{ tmp = freq_data[j][h]; }
			}
			static_freq_data[j][i] = tmp;
		}
	}
*/
}

static gboolean displayer_opengl_win_keypress(GtkWidget * w, GdkEventKey * event, gpointer data)
{
	gint volume, value;

        switch(event->keyval) {
                case GDK_Escape: // exit plugin
                        opengl_dp.disable(&opengl_dp);
                        break;
		case GDK_a:
			if (event->state & GDK_CONTROL_MASK) { singit_about_show(); }
			break;
                case GDK_b:
                        xmms_remote_playlist_next(opengl_dp.xmms_session);
                        break;
                case GDK_c:
			if (event->state & GDK_CONTROL_MASK) {  }
			else { xmms_remote_pause(opengl_dp.xmms_session); }
                        break;
		case GDK_e:
			if (event->state & GDK_CONTROL_MASK) { singit_editor_show(); }
			break;
		case GDK_i:
			if (event->state & GDK_CONTROL_MASK) { singit_tag_manager_show(); }
			break;
		case GDK_p:
			if (event->state & GDK_CONTROL_MASK) { singit_config_show(); }
			break;
                case GDK_v:
                        xmms_remote_stop(opengl_dp.xmms_session);
                        break;
                case GDK_x:
                        xmms_remote_play(opengl_dp.xmms_session);
                        break;
		case GDK_y:
			if (displayer_opengl_status.toggle_yz)
				{ xmms_remote_playlist_prev(opengl_dp.xmms_session); }
			break;
                case GDK_z:
			if (!displayer_opengl_status.toggle_yz)
				{ xmms_remote_playlist_prev(opengl_dp.xmms_session); }
                        break;
		case GDK_KP_Add:
			volume = xmms_remote_get_main_volume(opengl_dp.xmms_session);
			if (volume < 100) { volume++; }
			xmms_remote_set_main_volume(opengl_dp.xmms_session, volume);
			break;
		case GDK_KP_Subtract:
			volume = xmms_remote_get_main_volume(opengl_dp.xmms_session);
			if (volume > 0) { volume--; }
			xmms_remote_set_main_volume(opengl_dp.xmms_session, volume);
			break;
		case GDK_KP_Enter:
			if (displayer_opengl_status.jump_to_song == 0) { break; }
			if (xmms_remote_get_playlist_pos(opengl_dp.xmms_session) !=
				(displayer_opengl_status.jump_to_song - 1)) {
				xmms_remote_set_playlist_pos
					(opengl_dp.xmms_session, (displayer_opengl_status.jump_to_song - 1));
			}
		case GDK_KP_Decimal:
			displayer_opengl_status.jump_to_song = 0;
			break;
		case GDK_KP_0:
		case GDK_KP_1:
		case GDK_KP_2:
		case GDK_KP_3:
		case GDK_KP_4:
		case GDK_KP_5:
		case GDK_KP_6:
		case GDK_KP_7:
		case GDK_KP_8:
		case GDK_KP_9:
		case GDK_0:
		case GDK_1:
		case GDK_2:
		case GDK_3:
		case GDK_4:
		case GDK_5:
		case GDK_6:
		case GDK_7:
		case GDK_8:
		case GDK_9:
			value = event->keyval - GDK_0;
			if (value < 0 || value > 9)
				value = event->keyval - GDK_KP_0;

			if (event->state & GDK_CONTROL_MASK) {
				enable_plugin(((value + 9) % 10));
			}
			else {
				displayer_opengl_status.jump_to_song = (displayer_opengl_status.jump_to_song * 10) + value;
				if (displayer_opengl_status.jump_to_song > xmms_remote_get_playlist_length
					(opengl_dp.xmms_session)) { displayer_opengl_status.jump_to_song = value; }
			}
			break;
                default:
			// printf("%x - %x\n", event->state, event->keyval);
                        break;
        }

        return TRUE;
}


GtkWidget* create_glarea (void) {

	GtkWidget* glarea;

	static int attrlist[] = {
		GDK_GL_RGBA,
		GDK_GL_DOUBLEBUFFER,
		GDK_GL_DEPTH_SIZE, 1,
		GDK_GL_NONE
	};

	if(gdk_gl_query() == FALSE) {
		g_print("displayer_opengl.c [create_glarea] - OpenGL not supported!\n");
		return NULL;
	}

	if ((glarea = gtk_gl_area_new(attrlist)) == NULL) {
		g_print("displayer_opengl.c [create_glarea] - Error creating GtkGLArea!\n");
		return NULL;
	}

/*	gtk_widget_set_events(GTK_WIDGET(glarea),
		GDK_EXPOSURE_MASK |
		GDK_BUTTON_PRESS_MASK |
		GDK_BUTTON_RELEASE_MASK |
		GDK_POINTER_MOTION_MASK |
		GDK_POINTER_MOTION_HINT_MASK);*/
	gtk_widget_set_events(GTK_WIDGET(glarea), GDK_EXPOSURE_MASK);
/*
	gtk_signal_connect (GTK_OBJECT(glarea), "button_release_event",
		GTK_SIGNAL_FUNC(glarea_button_release), NULL);

	gtk_signal_connect (GTK_OBJECT(glarea), "button_press_event",
		GTK_SIGNAL_FUNC(glarea_button_press), NULL);

	gtk_signal_connect (GTK_OBJECT(glarea), "motion_notify_event",
		GTK_SIGNAL_FUNC(glarea_motion_notify), NULL);
*/
	gtk_signal_connect (GTK_OBJECT(glarea), "expose_event",
		GTK_SIGNAL_FUNC(glarea_draw), NULL);

	gtk_signal_connect (GTK_OBJECT(glarea), "configure_event",
		GTK_SIGNAL_FUNC(glarea_reshape), NULL);

	gtk_signal_connect (GTK_OBJECT(glarea), "realize",
		GTK_SIGNAL_FUNC(glarea_init), NULL);

	gtk_signal_connect (GTK_OBJECT(glarea), "destroy",
		GTK_SIGNAL_FUNC (glarea_destroy), NULL);

	gtk_widget_set_usize(GTK_WIDGET(glarea), 200, 200);

	return (glarea);
}

gint glarea_draw (GtkWidget* widget, GdkEventExpose* event)
{
	/* Draw only on the last expose event. */
	if (event->count > 0) { return(TRUE); }

	if (gtk_gl_area_make_current(GTK_GL_AREA(widget))) {

		opengl_lyrics_draw(displayer_opengl_status.time,
			displayer_opengl_status.rotate,
			displayer_opengl_status.beat,
			displayer_opengl_status.level);
		gtk_gl_area_swap_buffers(GTK_GL_AREA(widget));

//		displayer_opengl_limit_fps(FALSE);
	}

	return (TRUE);
}

gint glarea_reshape (GtkWidget* widget, GdkEventConfigure* event)
{
	int w = widget->allocation.width;
	int h = widget->allocation.height;

	if (gtk_gl_area_make_current(GTK_GL_AREA(widget))) {
		glViewport(0, 0, w, h);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	}

	return (TRUE);
}

gint glarea_init (GtkWidget* widget)
{
//	const float light_pos[] = { -1.0, -1.0, 2.0, 0.0 };

	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [glarea_init] - GTK GL Area Realize Event\n"));
	#endif

	if (gtk_gl_area_make_current (GTK_GL_AREA(widget))) {

//		const GLfloat light0_position[] = { 1.0, 1.0, 1.0, 0.0 };
//		const GLfloat light1_position[] = { 0.0, 1.0, 3.0, 0.0 };

		opengl_lyrics_init();

		glViewport(0, 0, (GLsizei)widget->allocation.width, (GLsizei)widget->allocation.height);

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();
		gluPerspective(60,1.0,1.5,64.0);

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		glClearColor(0.1, 0.1, 0.45, 0.0);
		glClearAccum(0.1, 0.1, 0.45, 0.0);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		glShadeModel(GL_FLAT);

//		glLightfv(GL_LIGHT0, GL_POSITION, light0_position);
//		glEnable(GL_LIGHT0);

//		glLightfv(GL_LIGHT1, GL_POSITION, light1_position);
//		glEnable(GL_LIGHT1);

		glEnable(GL_DEPTH_TEST);
		glDepthFunc(GL_LESS);

//		glEnable(GL_LIGHTING);
		glEnable(GL_BLEND);
		glBlendFunc(GL_SRC_ALPHA,GL_ONE);
		glColorMaterial(GL_FRONT, GL_AMBIENT_AND_DIFFUSE);
//		glEnable(GL_CULL_FACE);

		gluLookAt(0.0, 0.0, 10.0, 0.0, 0.0, 0.0, 0.0, 1.0, 0.0);
	}

	return TRUE;
}

gint glarea_destroy (GtkWidget* widget)
{
	#ifdef CODEDEBUG
	DEBUG(("displayer_opengl.c [glarea_destroy] - GTK GL Area Destroy Event\n"));
	#endif

	glDeleteLists(displayer_opengl_status.display_list, 1);
	opengl_lyrics_free();

	return TRUE;
}


/**********************************
	Unused templates
**********************************/

gint glarea_button_release (GtkWidget* widget, GdkEventButton* event)
{
#	ifdef CODEDEBUG
	int x = event->x;
	int y = event->y;
#	endif

	if (event->button == 1) {
		/* Mouse button 1 was released */
#		ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_button_release] - Button 1 release  (%d, %d)\n", x, y));
#		endif
		return TRUE;
	}

	if (event->button == 2) {
		/* Mouse button 2 was released */
#		ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_button_release] - Button 2 release  (%d, %d)\n", x, y));
#		endif
		return TRUE;
	}

	return FALSE;
}

gint glarea_button_press (GtkWidget* widget, GdkEventButton* event)
{
#	ifdef CODEDEBUG
	int x = event->x;
	int y = event->y;
#	endif

	if (event->button == 1) {
		/* Mouse button 1 was engaged */
#		ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_button_release] - Button 1 press  (%d, %d)\n", x, y));
#		endif
		return TRUE;
	}

	if (event->button == 2) {
		/* Mouse button 2 was engaged */
#		ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_button_release] - Button 2 press  (%d, %d)\n", x, y));
#		endif
		return TRUE;
	}

	return FALSE;
}

gint glarea_motion_notify (GtkWidget* widget, GdkEventMotion* event)
{
	int x;
	int y;
	GdkModifierType state;

	if (event->is_hint) {
		gdk_window_get_pointer(event->window, &x, &y, &state);
	}
	else {
		x = event->x;
		y = event->y;
		state = event->state;
	}

	if (state & GDK_BUTTON1_MASK) {
		/* Mouse button 1 is engaged */
		#ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_motion_notify] - Button 1 motion  (%d, %d)\n", x, y));
		#endif
	}

	if (state & GDK_BUTTON2_MASK) {
		/* Mouse button 2 is engaged */
		#ifdef CODEDEBUG
		DEBUG(("displayer_opengl.c [glarea_motion_notify] - Button 2 motion  (%d, %d)\n", x, y));
		#endif
	}

	return TRUE;
}
/*
static void displayer_opengl_limit_fps (gboolean init)
{
	static float fps = 0;
	static gint usec = 0;
	static GTimeVal last_rtc_time;

	GTimeVal new_rtc_time;
	float fps_config = 50;

	if (init) { g_get_current_time(&last_rtc_time); }
	else {
		long t;

		g_get_current_time(&new_rtc_time);
		t = (new_rtc_time.tv_sec - new_rtc_time.tv_sec) * 10000000 +
			(new_rtc_time.tv_usec - new_rtc_time.tv_usec);
		fps = (float) 1000000 / (float) t;
		if (fps >= (float) fps_config) { usec += 200; }
		else {
			if (usec > 0)
				usec -= 200;
		}
		if (usec > 0)
			xmms_usleep(usec);
		last_rtc_time = new_rtc_time;
	}
}
*/
