/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Based on xmms visualization plugin mechanism */


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sys/stat.h>
#include <dirent.h>
#include <stdio.h>
#include <string.h>

#include <gdk/gdk.h>
#include <gtk/gtkmain.h>

#include <xmms/plugin.h>

#include "ltdl_wrapper.h"

#include "singit_debug.h"
#include "singit_config_gen.h"
#include "singit_config.h"
#include "singit_config_private.h"
#include "singit_main.h"
#include "singit_plugin_data.h"
#include "singit_plugin_scanner.h"
#include "dlg_singit_config.h"
#include "singit_sound_precalcs_private.h"
#include "singit_displayer_plugin.h"
#include "singit_wgt_karaoke.h"

DisplayerPluginData *dp_data = NULL;

#define PLUGINSUBDIR "xmms-singit"

extern VisPlugin singit_vp;
extern SingitStatus singit_status;
extern SingitConfigGen *singit_config;

static void scan_dis_plugins(gchar *dirname);
static gboolean plugins_finalize_real(gboolean has_lock);
void dis_plugin_disable(DisplayerPlugin *dp);

static GList *get_dis_plugin_enabled_list(gboolean initialized);

static gint singit_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	singit_vp.disable_plugin(&singit_vp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

static gint dislist_compare_func(const void *a, const void *b)
{
	return strcasecmp(((DisplayerPlugin *) a)->description, ((DisplayerPlugin *) b)->description);
}

void plugins_init()
{
	gchar *dir;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [plugins_init]\n"));
#endif

	if (displayer_plugin_data_attach(dp_data) == TRUE)
		{ return; }

	if (wrp_dlinit() != 0) {
		g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			"dlopen error: %s", wrp_dlerror());
		return;
	}

	dp_data = DISPLAYER_PLUGIN_DATA(displayer_plugin_data_new());

#ifndef DISABLE_USER_PLUGIN_DIR
	dir = g_strconcat(g_get_home_dir(), "/.xmms/Plugins/Visualization", NULL);
	scan_dis_plugins(dir);
	g_free(dir);
	/*
	 * This is in a separate loop so if the user puts them in the
	 * wrong dir we'll still get them in the right order (home dir
	 * first                                                - Zinx
	 */
	dir = g_strconcat(g_get_home_dir(), "/.xmms/Plugins/Visualization/", PLUGINSUBDIR, NULL);
	scan_dis_plugins(dir);
	g_free(dir);
#endif

	dir = g_strconcat(PLUGIN_DIR, "/", PLUGINSUBDIR, NULL);
	scan_dis_plugins(dir);
	g_free(dir);

	dp_data->displayer_list = g_list_sort(dp_data->displayer_list, dislist_compare_func);
	if (singit_config_gen_attach(singit_config)) {
		dis_plugin_enable_from_stringified_list(GET_SCD->enabled_dplugins);
		singit_config_gen_detach(SINGIT_CONFIG_GEN(singit_config));
	}
}

static gboolean dis_plugin_is_duplicate_libname(gchar* filename)
{
	GList *pl = dp_data->displayer_list;
	gchar *base_filename = g_basename(filename);

	while (pl != NULL) {
		if (wrp_is_same_libname
			(((DisplayerPlugin*)pl->data)->handle, base_filename))
		{
			return TRUE;
		}
		pl = g_list_next(pl);
	}

	return FALSE;
}

// libltdl is smart enought to return the same handle for the same lib
// Our wrapper is not, so we just return FALSE - no dup
static gboolean dis_plugin_check_duplicate_handle(lt_dlhandle h)
{
#ifdef HAVE_LTDL_H
	GList *pl = dp_data->displayer_list;

	while (pl != NULL) {
		if (h == ((lt_dlhandle) ((DisplayerPlugin*)pl->data)->handle))
			{ return TRUE; }
		pl = g_list_next(pl);
	}
#endif
	return FALSE;
}

static void add_dis_plugin(gchar *filename)
{
	lt_dlhandle h;
	void *(*gpi) (void);

	if (dis_plugin_is_duplicate_libname(filename) == TRUE)
		{ return; }
	
	if ((h = wrp_dlopen(filename)) == NULL)
	{
		g_log (G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
			"dlopen error: %s", wrp_dlerror());
		return;
	}

	if ((dis_plugin_check_duplicate_handle(h) == FALSE) &&
		((gpi = wrp_dlsym(h, "get_dplugin_info")) != NULL))
	{
		DisplayerPlugin *p;

		p = (DisplayerPlugin *) gpi();
		p->handle = (void*) h;
		p->filename = g_strdup(filename);
		p->xmms_session = singit_vp.xmms_session;
		p->disable = dis_plugin_disable;
		dp_data->displayer_list = g_list_prepend(dp_data->displayer_list, p);
	}
	else { wrp_dlclose(h); }
}

static void scan_dis_plugins(gchar *dirname)
{
	gchar *filename;
	DIR *dir;
	struct dirent *ent;
	struct stat statbuf;

	dir = opendir(dirname);
	if (!dir)
		return;

	while ((ent = readdir(dir)) != NULL)
	{
		// We skip "." and ".." directory entries
		if ((strcmp(ent->d_name, ".") == 0) || (strcmp(ent->d_name, "..") == 0))
			{ continue; }

		filename = g_strdup_printf("%s/%s", dirname, ent->d_name);
		if (!stat(filename, &statbuf) && S_ISREG(statbuf.st_mode) && 
			wrp_is_libname(filename))
		{
			add_dis_plugin(filename);
		}
		g_free(filename);
	}
	closedir(dir);
}

void plugins_finish(void)
{
	DisplayerPlugin *dp;
	GList *node;
	gboolean init;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [plugins_finish] : "));
#endif

	g_return_if_fail(dp_data != NULL);

	if (GTK_OBJECT(dp_data)->ref_count > 1) {
		displayer_plugin_data_detach(&dp_data);

#ifdef CODEDEBUG
		DEBUG(8, ("Just detached\n"));
#endif
		return;
	}
#ifdef CODEDEBUG
	DEBUG(8, ("Real finished\n"));
#endif

	init = displayer_plugin_data_lock_init_ext(dp_data, TRUE, FALSE, FALSE);

	if (singit_config_gen_attach(singit_config)) {
		g_free(GET_SCD->enabled_dplugins);
		GET_SCD->enabled_dplugins = dis_plugin_stringify_enabled_list();
		if ((GET_SCD->enabled_dplugins == NULL) &&
			(dp_data->displayer_last != NULL))
		{
			GET_SCD->enabled_dplugins = g_strdup(wrp_get_libname
				(((DisplayerPlugin *) dp_data->displayer_last)->handle));
		}
		singit_config_save_plugins();
		if (init) { singit_config_save_positions(); }
		singit_config_gen_detach(SINGIT_CONFIG_GEN(singit_config));
	}

	if (init) { plugins_finalize_real(TRUE); }

	node = get_dis_plugin_list();
	while (node)
	{
		dp = (DisplayerPlugin *) node->data;
		wrp_dlclose(dp->handle);
		node = g_list_next(node);
	}

	displayer_plugin_data_detach(&dp_data);

	wrp_dlexit();

	if (init && (dp_data != NULL))
		displayer_plugin_data_unlock_init(dp_data);
}

void plugins_set_time(gint time, SingitSong *cur_song, GList *real_next)
{
	DisplayerPlugin *dp;
	GList *node;

	if (displayer_plugin_data_lock_plugins(dp_data, TRUE) == FALSE)
		{ return; }

	node = get_dis_plugin_enabled_list(TRUE);
	while (node != NULL)
	{
		dp = (DisplayerPlugin *) node->data;
		if ((dp != NULL) && (dp->set_time != NULL)) {
			dp->set_time(time, cur_song, real_next);
		}
		node = g_list_next(node);
	}

	displayer_plugin_data_unlock_plugins(dp_data);
}

typedef enum {

	dpc_show = 0,
	dpc_hide,
	dpc_toggle,
	dpc_update,
	dpc_playback_start,
	dpc_playback_stop,
	dpc_about,
	dpc_configure
}
DPCallback;

// Emitter for "seldom" used callbacks
static inline gboolean real_emitter
	(GList *node, DPCallback callback, SingitSong *cur_song, gboolean enabled)
{
	DisplayerPlugin *dp;

	g_return_val_if_fail(node != NULL, FALSE);
	g_return_val_if_fail(node->data != NULL, FALSE);
	g_return_val_if_fail(dp_data != NULL, FALSE);

	dp = (DisplayerPlugin *) node->data;

	switch (callback) {
	case dpc_about:
		if (dp->about)
			dp->about();
		return TRUE;
	case dpc_configure:
		if (dp->configure)
			dp->configure();
		return TRUE;
	default: // To suppress compiler warnings
		if (enabled == FALSE)
			{ return FALSE; }
	}
	
	switch (callback) {
	case dpc_show:
		if (!g_list_find(dp_data->visible_list, node->data)) {
			dp_data->visible_list = g_list_append
				(dp_data->visible_list, node->data);
			if (dp->show)
				dp->show(cur_song);
		}
		break;
	case dpc_hide:
		if (g_list_find(dp_data->visible_list, node->data)) {
			dp_data->visible_list = g_list_remove
				(dp_data->visible_list, node->data);
			if (dp->hide)
				dp->hide(cur_song);
		}
		break;
	case dpc_toggle:
		if (dp->toggle)
			dp->toggle(cur_song);
		break;
	case dpc_update:
		if (dp->update)
			dp->update(GET_SCD);
		break;
	case dpc_playback_start:
		if (dp->playback_start)
			dp->playback_start();
		break;
	case dpc_playback_stop:
		if (dp->playback_stop)
			dp->playback_stop();
		break;
	default: // To suppress compiler warnings
		return FALSE;
	}

	return TRUE;
}

// if (plugin < 0) ==> for all enabled
static void emit_callback(DPCallback callback, gint plugin, SingitSong *cur_song)
{
	GList *node;
	gboolean plugins, initialized;

	g_return_if_fail(dp_data != NULL);

	switch (callback) {
	case dpc_about:
	case dpc_configure:
		if (displayer_plugin_data_lock_init(dp_data, &initialized) == FALSE)
			{ return; }
		plugins = FALSE;
		break;
	default:
		if (displayer_plugin_data_lock_plugins(dp_data, FALSE) == FALSE)
			{ return; }
		plugins = TRUE;
		break;
	}
		
	if (plugin < 0) {
		node = get_dis_plugin_enabled_list(FALSE);
		while (node != NULL)
		{
			if (real_emitter(node, callback, cur_song, initialized) == FALSE)
				{ break; }
			node = g_list_next(node);
		}
	}
	else {
		node = g_list_nth(dp_data->displayer_list, plugin);
		if (node != NULL)
			real_emitter(node, callback, cur_song, initialized && 
				is_dis_plugin_enabled(plugin));
		else
			g_log(G_LOG_DOMAIN, G_LOG_LEVEL_CRITICAL,
				"Plugin not available");
	}

	if (plugins == FALSE) 
		{ displayer_plugin_data_unlock_init(dp_data); }
	else
		{ displayer_plugin_data_unlock_plugins(dp_data); }
}

void dis_plugin_show(gint plugin, SingitSong *cur_song)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_show]\n"));
#endif

	emit_callback(dpc_show, plugin, cur_song);
}

void dis_plugin_hide(gint plugin, SingitSong *cur_song)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_hide]\n"));
#endif

	emit_callback(dpc_hide, plugin, cur_song);
}

void dis_plugin_toggle(gint plugin, SingitSong *cur_song)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_toggle]\n"));
#endif

	emit_callback(dpc_toggle, plugin, cur_song);
}

void dis_plugin_update(gint plugin)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_update]\n"));
#endif

	emit_callback(dpc_update, plugin, NULL);
}

void dis_plugin_playback_start(gint plugin)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_playback_start]\n"));
#endif

	if (!dp_data || dp_data->playback_started) 
		{ return; }

	emit_callback(dpc_playback_start, plugin, NULL);
	dp_data->playback_started = TRUE;
}

void dis_plugin_playback_stop(gint plugin)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_playback_stop]\n"));
#endif

	if (!dp_data || !dp_data->playback_started) { return; }
	emit_callback(dpc_playback_stop, plugin, NULL);
	dp_data->playback_started = FALSE;
}

void dis_plugin_about(gint plugin)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_about]\n"));
#endif

	emit_callback(dpc_about, plugin, NULL);
}

void dis_plugin_configure(gint plugin)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_configure]\n"));
#endif

	emit_callback(dpc_configure, plugin, NULL);
}

void dis_plugin_render_freq(gint16 freq_data[2][256])
{
	GList *node;
	node = get_dis_plugin_enabled_list(TRUE);

	if (node && (singit_status.singit_sound_precalcs != NULL)) {
		sigit_sound_precalcs_freq(singit_status.singit_sound_precalcs, &freq_data[0][0]);
	}
	while (node)
	{
		if (((DisplayerPlugin *) node->data)->render_freq)
			((DisplayerPlugin *) node->data)->render_freq(freq_data, singit_status.singit_sound_precalcs);
		node = g_list_next(node);
	}
}

void dis_plugin_render_pcm(gint16 pcm_data[2][512])
{
	GList *node;
	node = get_dis_plugin_enabled_list(TRUE);

	if (node && (singit_status.singit_sound_precalcs != NULL)) {
		sigit_sound_precalcs_pcm(singit_status.singit_sound_precalcs, &pcm_data[0][0]);
	}
	while (node)
	{
		if (((DisplayerPlugin *) node->data)->render_pcm)
			((DisplayerPlugin *) node->data)->render_pcm(pcm_data, singit_status.singit_sound_precalcs);
		node = g_list_next(node);
	}
}

GList *get_dis_plugin_list(void)
{
	if (!dp_data) { return NULL; }
	return dp_data->displayer_list;
}

static GList *get_dis_plugin_enabled_list(gboolean initialized)
{
	if (!dp_data) { return NULL; }
	if (initialized && !dp_data->initialized) { return NULL; }
	return dp_data->enabled_list;
}

gboolean is_dis_plugin_enabled(gint plugin)
{
	gboolean result;

	g_return_val_if_fail(dp_data != NULL, FALSE);

	result = (dp_data->enabled_list != NULL);

	if ((result == TRUE) && (plugin >= 0))
		result = (g_list_find(dp_data->enabled_list,
			(g_list_nth(dp_data->displayer_list, plugin)->data)) != NULL);

	return result;
}

gboolean is_dis_plugin_running(gint plugin)
{
	g_return_val_if_fail(dp_data != NULL, FALSE);

	if (displayer_plugin_data_lock_init_ext(dp_data, TRUE, FALSE, TRUE) == FALSE)
		{ return FALSE; }

	displayer_plugin_data_unlock_init(dp_data);

	return is_dis_plugin_enabled(plugin);
}

// This is the callback the plugin can use to disable itself
void dis_plugin_disable(DisplayerPlugin *dp)
{
	gint i;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [dis_plugin_disable]\n"));
#endif

	g_return_if_fail(dp_data != NULL);
	g_return_if_fail(dp != NULL);
	g_return_if_fail(dp_data->displayer_list != NULL);

	i = g_list_index(dp_data->displayer_list, dp);

	set_dis_plugin_status(i, FALSE);

	config_dis_plugins_rescan();
}

gboolean set_dis_plugin_status(gint plugin, gboolean enable)
{
	GList *node;
	DisplayerPlugin *dp;
	gboolean search;
	gboolean initialized;

	g_return_val_if_fail(dp_data != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [set_dis_plugin_status] : "));
	if (enable) { DEBUG(DLV_ALL, ("Enable\n")); }
	else { DEBUG(DLV_ALL, ("Disable\n")); }
#endif

	// Find the plugin, look and look if it is (!enable)
	if ((guint) plugin >= g_list_length(dp_data->displayer_list))
		{ return FALSE; }

	node = g_list_nth(dp_data->displayer_list, plugin);

	g_return_val_if_fail(node != NULL, FALSE);
	g_return_val_if_fail(node->data != NULL, FALSE);

	dp = (DisplayerPlugin *) node->data;

	search = (g_list_find(dp_data->enabled_list, dp) != NULL);
	if (enable == search)
		{ return FALSE; }

	if (displayer_plugin_data_lock_lower(dp_data, &initialized) == FALSE)
		{ return FALSE; }

	if (enable) {
		// Start the displayer plugin
		dp_data->enabled_list = g_list_append(dp_data->enabled_list, dp);

		if (initialized == FALSE)
			{ goto unlock_lower; }

		if (dp->init) {
			dp->init();
			if (dp->update) {
				dp->update(GET_SCD);
			}
		}
		singit_status.initialize_plugins = TRUE;
	}
	else {
		// Stop the displayer plugin
		if (dp_data->enabled_list->next == NULL)
			{ dp_data->displayer_last = dp_data->enabled_list->data; }

		dp_data->enabled_list = g_list_remove(dp_data->enabled_list, dp);

		if (g_list_find(dp_data->visible_list, dp))
			{ dp_data->visible_list = g_list_remove(dp_data->visible_list, dp); }

		if (initialized == FALSE)
			{ goto unlock_lower; }

		if (dp->finish) {
			dp->finish();
		}

		singit_status.initialize_plugins = TRUE;
		singit_config_save_positions();

		if (!dp_data->enabled_list) {
			dp_data->initialized = FALSE;
			gtk_idle_add(singit_disable_func, NULL);
		}
	}

unlock_lower:
	if (initialized == TRUE)
		{ displayer_plugin_data_unlock_plugins(dp_data); }
	else 
		{ displayer_plugin_data_unlock_init(dp_data); }

	return TRUE;
}

gchar *dis_plugin_stringify_enabled_list(void)
{
	gchar *enabled_str = NULL, *dest_pos;
	const gchar *src_pos;
	GList *node;
	gint len = 0;

	g_return_val_if_fail(dp_data != NULL, NULL);

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_plugin_scanner.c [dis_plugin_stringify_enabled_list]\n"));
#endif

	node = dp_data->enabled_list;
	if (node == NULL)
		{ return NULL; }

	while (node) {
		len += strlen(wrp_get_libname
			(((DisplayerPlugin *) node->data)->handle)) + 1;
		node = g_list_next(node);
	}
	
	dest_pos = enabled_str = g_new(gchar, len);
	enabled_str[0] = '\0';

	node = dp_data->enabled_list;
	while (node) {
		src_pos = wrp_get_libname
			(((DisplayerPlugin *) node->data)->handle);
		len = strlen(src_pos);
		memcpy(dest_pos, src_pos, len);
		dest_pos += len;
		dest_pos[0] = ','; 
		dest_pos++;
		node = g_list_next(node);
	}
	dest_pos--;
	dest_pos[0] = '\0';
	
	return enabled_str;
}

void dis_plugin_enable_from_stringified_list(gchar *list)
{
	gchar **plugins;
	GList *node;
	gint i;
	DisplayerPlugin *dp;

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [dis_plugin_enable_from_stringified_list]\n"));
#endif

	if (!list || !strcmp(list, ""))
		return;
	plugins = g_strsplit(list, ",", 0);
	i = 0;
	while (plugins[i])
	{
		node = dp_data->displayer_list;
 		while (node)
		{
			if (wrp_is_same_libname(((DisplayerPlugin *) node->data)->handle, plugins[i]))
			{
				dp = node->data;
				dp_data->enabled_list = g_list_append
					(dp_data->enabled_list, (DisplayerPlugin *) dp);
			}
			node = g_list_next(node);
		}
		i++;
	}
	g_strfreev(plugins);
}

gboolean plugins_initialize()
{
	GList *node;
	DisplayerPlugin *dp;

	g_return_val_if_fail(dp_data != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [plugins_initialize]\n"));
#endif

	if (displayer_plugin_data_lock_init_ext(dp_data, FALSE, FALSE, FALSE) == FALSE)
		{ return FALSE; }

	node = dp_data->enabled_list;
	while (node)
	{
		dp = node->data;
		if(dp->init) {
			dp->init();
			if (dp->update) {
				dp->update(GET_SCD);
			}
		}
		node = g_list_next(node);
	}

	dp_data->initialized = TRUE;

	// We have finished the initialization - unlock the plugins
	displayer_plugin_data_unlock_plugins(dp_data);

	displayer_plugin_data_unlock_init(dp_data);

	return TRUE;
}

static gboolean plugins_finalize_real(gboolean has_lock)
{
	GList *node;
	DisplayerPlugin *dp;

	g_return_val_if_fail(dp_data != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [plugins_finalize]\n"));
#endif

	if (displayer_plugin_data_lock_init_ext(dp_data, TRUE, has_lock, FALSE) == FALSE)
		{ return FALSE; }

	// We are going to shut down the plugins - lock them
	if (displayer_plugin_data_lock_plugins(dp_data, FALSE) == FALSE)
		{ return FALSE; }

	node = dp_data->enabled_list;
	while (node)
	{
		dp = node->data;
		if (dp->finish) {
			dp->finish();
		}
		node = g_list_next(node);
	}

	dp_data->initialized = FALSE;
	g_list_free(dp_data->visible_list);
	dp_data->visible_list = NULL;

	if (has_lock == FALSE)
		displayer_plugin_data_unlock_init(dp_data);

	return TRUE;
}

gboolean plugins_finalize()
{
	g_return_val_if_fail(dp_data != NULL, FALSE);

#ifdef CODEDEBUG
	DEBUG(8, ("singit_plugin_scanner.c [plugins_finalize]\n"));
#endif

	return plugins_finalize_real(FALSE);
}
