

#include "SoundInputPort.h"

#include "EgOSUtils.h"
#include <math.h>

#define __err( msg )		EgOSUtils::ShowMsg( msg );

#if EG_MAC
#define __blockmove( srce, dest, num )  ::BlockMove( (char*) (srce), (char*) (dest), (num) )
#define __getBufInfo \
	long start, n, i;			\
	n = mParams.mDimSamples;		\
	start = mParams.mBufEnd - inN;
#endif


#if EG_WIN
#define __blockmove( srce, dest, num )  ::MoveMemory( (dest), (srce), (num) )

#define __getBufInfo																		\
		if ( mHdr1.dwFlags & WHDR_DONE ) {									\
			mHdr1.dwBytesRecorded = 0;										\
			waveInAddBuffer( mWaveDevice, &mHdr1, sizeof( mHdr1 ) );		\
		}																	\
		if ( mHdr2.dwFlags & WHDR_DONE  ) {									\
			mHdr2.dwBytesRecorded = 0;										\
			waveInAddBuffer( mWaveDevice, &mHdr2, sizeof( mHdr2 ) );		\
		}																	\
		\
		/* See where the most recent recorded data ends */					\
		MMTIME pos;		pos.wType = TIME_BYTES;								\
		waveInGetPosition( mWaveDevice, &pos, sizeof( pos ) );				\
		\
		/* Calc the starting element number		 */							\
		long n = mHdr1.dwBufferLength / sizeof( short );					\
		long i, start = ( pos.u.cb / sizeof( short ) ) % n - 900 - inN;		\

#endif

#if EG_POSIX
#include <string.h>
#define __blockmove( srce, dest, num )  memmove( (char*) (srce), (char*) (dest), (num) )
#define __getBufInfo \
	long start, n, i;			\
	n = mParams.mDimSamples;		\
	start = mParams.mBufEnd - inN;
#endif


SoundInput::SoundInput() :
	mAnalyzer( true ) {

	mIsRecording			= false;
	mParams.mNumSampleSkip	= 0;
	mParams.mSamples		= nil;
	mParams.mDimSamples		= 0;

	DimBufSize( 16 );
}


SoundInput::~SoundInput() {

	StopRecording();
}



void SoundInput::DimBufSize( long inNumSamples ) {

	// The interrupt will write as we read data, so make the buf big enough so it won't write over while we're still reading
	long i, need, samplesNeeded = inNumSamples + 100;
	bool wasRecording = mIsRecording;

	if ( samplesNeeded > mParams.mDimSamples ) {

		// times 2 b/c we wrap our data
		need = 2 * sizeof( short ) * samplesNeeded;

		#if EG_WIN
		if ( need < 0x4000 )
			need = 0x4000;
		#endif

		StopRecording();
		mParams.mSamples		= (short*) mParamBuf.Dim( need );
		mParams.mDimSamples		= samplesNeeded;
		mParams.mBufEnd 		= 0;
		mParams.mBufRemainder	= 0;

		#if EG_WIN
		mHdr1.lpData			= (char*) mParams.mSamples;
		mHdr1.dwBufferLength	= need;
		mHdr1.dwBytesRecorded	= 0;
		mHdr1.dwFlags			= 0;
		mHdr1.dwUser			= 0;
		mHdr1.dwLoops			= 0;

		mHdr2 = mHdr1;
		#endif

		for ( i = 0; i < inNumSamples; i++ )
			mParams.mSamples[ i ] = 0;

		if ( wasRecording )
			StartRecording();
	}
}


bool SoundInput::GetSamples( short outSamples[], long inN ) {

	if ( inN < 32 )
		inN = 32;

	DimBufSize( inN );

	__getBufInfo

	// We use a circular buffer, so mind it...
	for ( i = 0; i < - start && i < inN; i++ )
		outSamples[ i ] = mParams.mSamples[ n + start + i ];
	for ( ; i < inN; i++ )
		outSamples[ i ] = mParams.mSamples[     start + i ];

	return true;
}



bool SoundInput::GetSamples( float outSamples[], long inN, float inScale ) {

	if ( inN < 32 )
		inN = 32;

	DimBufSize( inN );

	__getBufInfo

	// We use a circular buffer, so mind it...
	for ( i = 0; i < - start && i < inN; i++ )
		outSamples[ i ] = inScale * mParams.mSamples[ n + start + i ];
	for ( ; i < inN; i++ )
		outSamples[ i ] = inScale * mParams.mSamples[     start + i ];

	return true;
}



#define FFT_SAMPLE_SIZE  400

bool SoundInput::GetSpectrum( long inBins, float inBinRange, float inFreqStart, float outFT[] ) {
	long inN = FFT_SAMPLE_SIZE;

	DimBufSize( FFT_SAMPLE_SIZE );

	__getBufInfo

	// Don't bother moving samples if we're not wrapping
	if ( start + inN < 0 )
		start += n;

	// Is our data wrapped around the edge of our circular buffer?
	if ( start < 0 ) {

		// Paste the last portion after the first portion (that's why we allocated extra)
		for ( i = 0; i < start + inN; i++ )
			mParams.mSamples[ n + i ] = mParams.mSamples[ i ];

		// Make start contain a valid starting index
		start += n;

	}

	// Turn a sequence of samples into a freq spectrum
	mAnalyzer.TransformPolar( &mParams.mSamples[ start ], inN, inBins, inBinRange, inFreqStart, outFT, nil );

	return true;
}




#if EG_MAC && STANDALONE
static OSErr	SetupDevice (long inputDevice,
							short *numChannels,
							short *sampleSize,
							Fixed *sampleRate,
							OSType *compression,
							long *devBuffer) {

	OSErr				err;
	Fixed				gain = 0x00008000;
	short				on = 1;

	#if EG_MAC
	err = SPBSetDeviceInfo (inputDevice, siSampleRate, (Ptr) sampleRate);
	if (err != noErr)
		__err( "Couldn't set sample rate" );

	err = SPBSetDeviceInfo (inputDevice, siSampleSize, (Ptr) sampleSize);
	if (err != noErr)
		__err( "Couldn't set sample size");

	err = SPBSetDeviceInfo (inputDevice, siTwosComplementOnOff, (Ptr) &on);
	if (err != noErr)
		__err( "Couldn't set twos complement");

	err = SPBSetDeviceInfo (inputDevice, siNumberChannels, (Ptr) numChannels);
	if (err != noErr)
		__err( "Couldn't set number of channels");

	err = SPBSetDeviceInfo (inputDevice, siCompressionType, (Ptr) compression);
	if (err != noErr)
		__err( "Couldn't set compression type");

	//turn on continuous recording to "warm up" the input device
	err = SPBSetDeviceInfo (inputDevice, siContinuous, (Ptr) &on);
	if (err != noErr)
		__err( "Couldn't turn on continuous recording");

	//turn on Automatic Gain Control
	err = SPBSetDeviceInfo (inputDevice, siAGCOnOff, (Ptr) &on);
	if (err != noErr) {
		//If AGC isn't available, just turn it all the way down to avoid over driving
		err = SPBSetDeviceInfo (inputDevice, siInputGain, (Ptr) &gain);
		if (err != noErr)
			__err( "Couldn't get siInputGain");
	}

	//check to see what we really got
	err = SPBGetDeviceInfo (inputDevice, siSampleRate, (Ptr) sampleRate);
	if (err != noErr)
		__err( "Couldn't get sample rate");

	err = SPBGetDeviceInfo (inputDevice, siSampleSize, (Ptr) sampleSize);
	if (err != noErr)
		__err( "Couldn't get sample size");

	err = SPBGetDeviceInfo (inputDevice, siNumberChannels, (Ptr) numChannels);
	if (err != noErr)
		__err( "Couldn't get number of channels");

	err = SPBGetDeviceInfo (inputDevice, siDeviceBufferInfo, (Ptr) devBuffer);
	if (err != noErr)
		__err( "Couldn't get number of channels");

	err = SPBGetDeviceInfo (inputDevice, siCompressionType, (Ptr) compression);
	if (err != noErr)
		__err( "Couldn't get compression type");
	#endif

	return err;
}
#endif



OSErr SoundInput::StartRecording () {
	OSErr				err = 0;

	if ( ! mIsRecording ) {

#if EG_WIN
		mParams.recordRec.wFormatTag		= 1;
		mParams.recordRec.nChannels		= 1;
		mParams.recordRec.nSamplesPerSec	= 11025;
		mParams.recordRec.wBitsPerSample	= 16;
		mParams.recordRec.nAvgBytesPerSec	= mParams.recordRec.nSamplesPerSec * mParams.recordRec.wBitsPerSample;
		mParams.recordRec.nBlockAlign		= mParams.recordRec.nChannels * mParams.recordRec.wBitsPerSample / 8;
		mParams.recordRec.cbSize		= 0;
		mWaveDevice				= 0;

		if ( waveInOpen( &mWaveDevice, 0, &mParams.recordRec, 0, 0, 0) == MMSYSERR_NOERROR ) {

			if (	waveInPrepareHeader( mWaveDevice, &mHdr1, sizeof( mHdr1 ) ) == MMSYSERR_NOERROR &&
					waveInPrepareHeader( mWaveDevice, &mHdr2, sizeof( mHdr2 ) ) == MMSYSERR_NOERROR ) {

				if ( 	waveInAddBuffer( mWaveDevice, &mHdr1, sizeof( mHdr1 ) )	== MMSYSERR_NOERROR &&
						waveInAddBuffer( mWaveDevice, &mHdr2, sizeof( mHdr2 ) ) == MMSYSERR_NOERROR ) {

					// Start the recording
					if ( waveInStart( mWaveDevice ) == MMSYSERR_NOERROR ) {
						err = 0;
						mIsRecording = true;
					}

				}

			}

			// Cleanup if we opened the device and then had probs
			if ( ! mIsRecording )
				waveInStop( mWaveDevice );
		}
#endif

#if EG_MAC && STANDALONE
		err = SPBOpenDevice ( nil, siWritePermission, &mParams.soundRefNum );
		if (err != noErr) {
			__err( "Couldn't open the device");
		}

		mParams.numChannels	= 1;
		mParams.sampleSize	= 16;
		mParams.sampleRate	= rate44khz;
		mParams.compression	= 'NONE';
		err = SetupDevice (mParams.soundRefNum, &mParams.numChannels,
			&mParams.sampleSize, &mParams.sampleRate,
			&mParams.compression, &mParams.devBuffer);

		if (err == noErr) {
		/*	IOCompletionUPP 	MyPB0WriteCompUPP,
								MyPB1WriteCompUPP;*/
			//SICompletionUPP 	MyRecCompUPP;
			SIInterruptUPP		MyIntCompUPP;

			//mParams->sanitycheck = 'SANE';
			//myVars->whichBuffer = 0;
			mParams.myA5 = SetCurrentA5 ();

			//set up the record parameters
			//MyRecCompUPP = NewSICompletionProc (MyRecComp);
			MyIntCompUPP = NewSIInterruptProc (MyRecInterrupt);
			mParams.recordRec.inRefNum = mParams.soundRefNum;
			mParams.recordRec.count = 0;
			mParams.recordRec.milliseconds = 0;
			mParams.recordRec.bufferLength = 0 ;//buffSize;
			mParams.recordRec.bufferPtr = nil; //myVars->recBuffer;
			mParams.recordRec.completionRoutine = nil; // MyRecCompUPP;
			mParams.recordRec.interruptRoutine = MyIntCompUPP;
			mParams.recordRec.userLong = (long) &mParams;
			mParams.recordRec.error = 0;
			mParams.recordRec.unused1 = 0;

		}

		//mParams->totalBytes = 0;
		err = SPBRecord (&mParams.recordRec, true);
		if (err != noErr) {
			__err( "SPBRecord failed");
		}

		if ( err == noErr )
			mIsRecording = true;

#endif

#if GFORCE_PRO && EG_MAC
		// Temp
		if ( err == noErr )
			mIsRecording = true;

#endif
	}

	return err;
}

void SoundInput::StopRecording () {

	if ( mIsRecording ) {

#if EG_MAC && STANDALONE
		OSErr err;
		err = SPBStopRecording (mParams.soundRefNum);
		if (err != noErr)
			__err( "SPBStopRecording failed");

		err = SPBCloseDevice (mParams.soundRefNum);
		if (err != noErr)
			__err( "SPBCloseDevice failed");
#endif

#if EG_WIN
		waveInReset( mWaveDevice );
		waveInUnprepareHeader( mWaveDevice, &mHdr1, sizeof( mHdr1 ) );
		waveInUnprepareHeader( mWaveDevice, &mHdr2, sizeof( mHdr2 ) );
		if ( waveInStop( mWaveDevice ) != MMSYSERR_NOERROR ) {
			__err( "waveInStop() failed"); }
		else if ( waveInClose( mWaveDevice ) != MMSYSERR_NOERROR ) {
			__err( "waveInClose() failed");
		}
#endif

#if GFORCE_PRO && EG_MAC
#endif

		mIsRecording = false;
	}

}


#if EG_MAC && STANDALONE
pascal void		MyRecInterrupt( SPB* inPB, short* buf, long peak, long size ) {

	//SPBPtr inPB:__a0;
	#pragma unused( peak )

	SoundInputRec*			recPtr;
	int i;
	long oldA5, offset, skip;

	recPtr = (SoundInputRec*)inPB->userLong;

	oldA5 = SetA5 (recPtr->myA5);

	offset = recPtr -> mBufEnd;
	skip = 1 + recPtr -> mNumSampleSkip;

	for ( i = recPtr -> mBufRemainder; i < size; i += skip ) {
		recPtr -> mSamples[ offset ] = buf[ i ];
		offset++;
		if ( offset == recPtr -> mDimSamples )
			offset = 0;
	}

	recPtr -> mBufRemainder = i - size;
	recPtr -> mBufEnd = offset;

	oldA5 = SetA5 (oldA5);
}
#endif
