
#include "WC_WaveShape.h"

#include "ArgList.h"
#include "WhiteCap.h"

#include <time.h>
#include <stdlib.h>

WC_WaveShape::WC_WaveShape() {

	mPrefWidth = 1;
	mPrefHeight = 1;
}

#define _assignOrig( field )  field##Orig = field;

void WC_WaveShape::Assign( const ArgList& inArgs, ExpressionDict& ioDict ) {

	UtilStr str;
	Expression expr;
	float* addr;

	// Mix up the random number generator
	srand( clock() );

	// Extract, compile, and link in order
	mA_Var.Compile( inArgs, 'A', ioDict );
	mB_Var.Compile( inArgs, 'B', ioDict );
	mC_Var.Compile( inArgs, 'C', ioDict );
	mD_Var.Compile( inArgs, 'D', ioDict );

	// Compile mNum_S_Steps -- How many pieces we chop the 0-1 s interval into
	inArgs.GetArg( MCC4_TO_INT("Stps"), str );
	expr.Compile( str, ioDict );
	mNum_S_Steps = expr.Evaluate();

	/* It's legal for AVars to access NUM_S_STEPS.  However, the float accessed by NUM_S_STEPS is
	mNum_S_Steps by the owning WhiteCapWorld. So, if the 'Stps' param in this config defines an
	overriding number of S steps, then we must store this value into mNum_S_Steps. */
	addr = ioDict.LookupVar( "NUM_S_STEPS" );
	if ( mNum_S_Steps > 0 )
		*addr = mNum_S_Steps;

	// AVars are evaluated when the config loads
	mA_Var.Evaluate();

	inArgs.GetArg( 'X', str );	mX.Compile( str, ioDict );
	inArgs.GetArg( 'Y', str );	mY.Compile( str, ioDict );
	inArgs.GetArg( 'Z', str );	mZ.Compile( str, ioDict );

	// Set some basic params
	mBlurVal		= inArgs.GetArg( MCC4_TO_INT("BlrB") );
	mNumBlurs		= inArgs.GetArg( MCC4_TO_INT("Blrs") );
	mConnectSamples		= inArgs.GetArg( MCC4_TO_INT("ConL") );
	mConnectBins		= inArgs.GetArg( MCC4_TO_INT("ConB") );
	mConnectFirstLast	= inArgs.GetArg( MCC4_TO_INT("ConB") ) > 1;
	mCanIncreaseScale	= inArgs.GetArg( MCC4_TO_INT("ScSz") );

	// Change sample expiration/duration from secs to ticks
	inArgs.GetArg( MCC4_TO_INT("Durn"), str );
	expr.Compile( str, ioDict );
	mSampleDuration		= expr.Evaluate();

	// Know the preferred X to Y ratio
	mConfigScale		= mXYScale = inArgs.GetArg( MCC4_TO_INT("Scal") );
	mPrefWidth		= inArgs.GetArg( MCC4_TO_INT("widt") );
	mPrefHeight		= inArgs.GetArg( MCC4_TO_INT("heig") );
	mPerspectiveInt		= inArgs.GetFloat( MCC4_TO_INT("Pers") );

	inArgs.GetArg( MCC4_TO_INT("LWdt"), str );
	if ( str.length() == 0 )
		str.Assign( "1" );
	mLineWidth.Compile( str, ioDict );

	// Read and compile the camera xyz pos, dir, and angle exprs
	inArgs.GetArg( MCC4_TO_INT("CamX"), str );	mCamX.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CamY"), str );	mCamY.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CamZ"), str );	mCamZ.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CmLX"), str );	mCamLX.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CmLY"), str );	mCamLY.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CmLZ"), str );	mCamLZ.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CUpX"), str );	mCamUpX.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CUpY"), str );	mCamUpY.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("CUpZ"), str );	mCamUpZ.Compile( str, ioDict );

	// Read and compile color expressions
	inArgs.GetArg( 'R', str );	mR.Compile( str, ioDict );
	inArgs.GetArg( 'G', str );	mG.Compile( str, ioDict );
	inArgs.GetArg( 'B', str );	mB.Compile( str, ioDict );

	// The colors for the head of the waveform (ie, t=0) are overridden by LvlRGB args...
	if ( inArgs.ArgExists( MCC4_TO_INT("LvlR") ) ) {
		inArgs.GetArg( MCC4_TO_INT("LvlR"), str ); mLvlR.Compile( str, ioDict ); }
	else
		mLvlR.Assign( mR );
	if ( inArgs.ArgExists( MCC4_TO_INT("LvlG") ) ) {
		inArgs.GetArg( MCC4_TO_INT("LvlG"), str ); mLvlG.Compile( str, ioDict ); }
	else
		mLvlG.Assign( mG );
	if ( inArgs.ArgExists( MCC4_TO_INT("LvlB") ) ) {
		inArgs.GetArg( MCC4_TO_INT("LvlB"), str ); mLvlB.Compile( str, ioDict ); }
	else
		mLvlB.Assign( mB );

	inArgs.GetArg( MCC4_TO_INT("BckR"), str );	mBackR.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("BckG"), str );	mBackG.Compile( str, ioDict );
	inArgs.GetArg( MCC4_TO_INT("BckB"), str );	mBackB.Compile( str, ioDict );

	mX_Dep_S = mX.IsDependent( "s" ) || mX.IsDependent( "d" ) || mX.IsDependent( "rnd" );
	mY_Dep_S = mY.IsDependent( "s" ) || mY.IsDependent( "d" ) || mY.IsDependent( "rnd" );
	mZ_Dep_S = mZ.IsDependent( "s" ) || mZ.IsDependent( "d" ) || mZ.IsDependent( "rnd" );

	mR_Dep_S = mR.IsDependent( "s" ) || mR.IsDependent( "d" ) || mR.IsDependent( "rnd" );
	mG_Dep_S = mG.IsDependent( "s" ) || mG.IsDependent( "d" ) || mG.IsDependent( "rnd" );
	mB_Dep_S = mB.IsDependent( "s" ) || mB.IsDependent( "d" ) || mB.IsDependent( "rnd" );

	// Make copies of the original values
	_assignOrig( mSampleDuration )
	_assignOrig( mNumBlurs )
	_assignOrig( mBlurVal )
	_assignOrig( mPerspectiveInt )
	_assignOrig( mXYScale )
	_assignOrig( mNum_S_Steps )
	_assignOrig( mConnectBins )
	_assignOrig( mConnectSamples )
	_assignOrig( mConnectFirstLast )
}


void WC_WaveShape::SetScaleToFit( long inWidth, long inHeight ) {

	float s1, s2;
	float visFactor = 1, side;

	// As the pane starts to get very small, artifically, scale things up a bit
	side = sqrt( (float) inWidth * inHeight );
	if ( side < 300 ) {
		visFactor = 300 / ( side + 1 );
		if ( visFactor > 1.6 )
			visFactor = 1.6;
	}

	// Calc the scale factor for all drawing in this pane
	s1 = visFactor * mConfigScale * inWidth / mPrefWidth;
	s2 = visFactor * mConfigScale * inHeight / mPrefHeight;
	if ( s2 < s1 )
		s1 = s2;
	if ( ! mCanIncreaseScale && s1 > mConfigScale )
		mXYScaleOrig = mConfigScale;
	else
		mXYScaleOrig = s1;

	mXYScale = mXYScaleOrig;
}

void WC_WaveShape::SetupTransition( WC_WaveShape* inDest, float* inC1, float* inC2 ) {

	mX.Weight( inDest -> mX, inC1, inC2 );
	mY.Weight( inDest -> mY, inC1, inC2 );
	mZ.Weight( inDest -> mZ, inC1, inC2 );

	mCamX.Weight( inDest -> mCamX, inC1, inC2 );
	mCamY.Weight( inDest -> mCamY, inC1, inC2 );
	mCamZ.Weight( inDest -> mCamZ, inC1, inC2 );

	mCamLX.Weight( inDest -> mCamLX, inC1, inC2 );
	mCamLY.Weight( inDest -> mCamLY, inC1, inC2 );
	mCamLZ.Weight( inDest -> mCamLZ, inC1, inC2 );

	mCamUpX.Weight( inDest -> mCamUpX, inC1, inC2 );
	mCamUpY.Weight( inDest -> mCamUpY, inC1, inC2 );
	mCamUpZ.Weight( inDest -> mCamUpZ, inC1, inC2 );

	mR.Weight( inDest -> mR, inC1, inC2 );
	mG.Weight( inDest -> mG, inC1, inC2 );
	mB.Weight( inDest -> mB, inC1, inC2 );

	mLvlR.Weight( inDest -> mLvlR, inC1, inC2 );
	mLvlG.Weight( inDest -> mLvlG, inC1, inC2 );
	mLvlB.Weight( inDest -> mLvlB, inC1, inC2 );

	mBackR.Weight( inDest -> mBackR, inC1, inC2 );
	mBackG.Weight( inDest -> mBackG, inC1, inC2 );
	mBackB.Weight( inDest -> mBackB, inC1, inC2 );

	mX_Dep_S = mX_Dep_S || inDest -> mX_Dep_S;
	mY_Dep_S = mY_Dep_S || inDest -> mY_Dep_S;
	mZ_Dep_S = mZ_Dep_S || inDest -> mZ_Dep_S;

	mR_Dep_S = mR_Dep_S || inDest -> mR_Dep_S;
	mG_Dep_S = mG_Dep_S || inDest -> mG_Dep_S;
	mB_Dep_S = mB_Dep_S || inDest -> mB_Dep_S;

	mLineWidth.Weight( inDest -> mLineWidth, inC1, inC2 );

}

#define __weightFLT( field ) \
	field = ( inW * ( (float) field##Orig ) + w1 * ( (float) inDest -> field##Orig ) );
#define __weightINT( field ) \
	field = ( 0.5 + inW * ( (float) field##Orig ) + w1 * ( (float) inDest -> field##Orig ) );
#define __weightBOL( field ) \
	field = .5 < ( inW * ( field##Orig ? 1.0 : 0.0 ) + w1 * ( inDest -> field##Orig ? 1.0 : 0.0 ) );

#define __weightBOL2( field, K ) \
	if ( field##Orig != inDest -> field##Orig )			\
		field = ( field##Orig ) ? inW > K : inW < ( 1.0 - K );	\
	else								\
		field = field##Orig;

void WC_WaveShape::SetupFrame( WC_WaveShape* inDest, float inW, long inNum_FFT_Bins ) {
	float w1 = 1.0 - inW;

	__weightFLT( mSampleDuration )
	__weightINT( mNumBlurs )
	__weightINT( mBlurVal )
	__weightFLT( mPerspectiveInt )
	__weightFLT( mXYScale )

	// To avoid a mass of yucky lines when we're morphing from lines to no line (or vice versa), don't draw lines until we're almost done
	__weightBOL2( mConnectBins, .65 )
	__weightBOL2( mConnectFirstLast, .65 )
	__weightBOL2( mConnectSamples, .75 )

	// If the Configs has an invalid num of sample bins, it means use the default/current num sample bins
	long s1, s2;
	if ( mNum_S_StepsOrig < 1 )
		s1 = inNum_FFT_Bins;
	else
		s1 = mNum_S_StepsOrig;

	if ( inDest -> mNum_S_StepsOrig < 1 )
		s2 = inNum_FFT_Bins;
	else
		s2 = inDest -> mNum_S_StepsOrig;
	mNum_S_Steps = ( 0.5 + inW * ( (float) s1 ) + w1 * ( (float) s2 ) );
}
