/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gtk/gtk.h>

#include <stdlib.h>
#include <stdio.h>

#include <xmms/xmmsctrl.h>
#include <xmms/util.h>
#include <xmms/titlestring.h>

#include <SDL/SDL_types.h>
#include <SDL/SDL_syswm.h>
#include <SDL/SDL_ttf.h>

#include "singit/macros.h"
#include "singit/control.h"
#include "singit/displayer_plugin.h"

#include "displayer_sdl_gforce.h"

#include "sdl_helpers.h"
#include "sdl_lyrics.h"
#include "sdl_lyrics_skd.h"

#include "sdl_gforce_config.h"
#include "sdl_gforce_config_dlg.h"
#include "sdl_gforce_status.h"
#include "sdl_gforce_handlers.h"

#include "SDL_prim.h"

static void displayer_sdl_init(void);
static void displayer_sdl_finish(void);

static void displayer_sdl_set_song(SingitSong *cur_song);
static void displayer_sdl_set_time(guint time, GList *real_next);
static void displayer_sdl_config_update(void);

static void displayer_sdl_playback_start(void);
static void displayer_sdl_playback_stop(void);

static void displayer_sdl_configure(void);

DisplayerPlugin sdl_gforce_dp =
{
	NULL,
	NULL,
	0,
	0,
	NULL,

	"SDL displayer (GForce)",

	displayer_sdl_init,
	displayer_sdl_finish,
	displayer_sdl_config_update,

	displayer_sdl_set_song,
	displayer_sdl_set_time,

	NULL,
	displayer_sdl_configure,
	displayer_sdl_playback_start,
	displayer_sdl_playback_stop,

	displayer_sdl_render_pcm,
	displayer_sdl_render_freq
};

DIPSPLAYER_SYMBOL(libdisplayer_sdl_gforce, sdl_gforce_dp)

static void displayer_sdl_config_update_event(SingitConfigGen *scg, SDLGForceConfigData *cfg_data, gpointer user_data);

gint displayer_sdl_disable_func(gpointer data)
{
	GDK_THREADS_ENTER();
	sdl_gforce_dp.disable(&sdl_gforce_dp);
	GDK_THREADS_LEAVE();
	return (FALSE);
}

static void displayer_sdl_process_events(guint time)
{
	SDL_Event sdl_event;

	if (STATUS->sdl_inuse == TRUE)
		{ return; }

	sdl_lyrics_set_text_colors(getPaletteColor(255),
		getPaletteColor(128), getPaletteColor(0));

	if (STATUS->screen != SDL_GetVideoSurface())
		{ return; }

	sdl_lyrics_render_text(time);
	displayer_sdl_render();
	displayer_sdl_draw_text();
	SDL_UpdateRect(STATUS->screen, 0, 0, 0, 0);

	if (!STATUS->kill_render_thread && (SDL_PollEvent(&sdl_event) > 0)) {
		switch (sdl_event.type) {
		case SDL_KEYDOWN:
			sdl_handler_key_down(&sdl_event);
			break;
		case SDL_QUIT:
			// Prevent double shutdown
			if (STATUS->shutdown == TRUE)
				break;
			STATUS->shutdown = TRUE;
			sdl_helper_blanking(BLANK_RESET);
			if (STATUS->kill_render_thread == FALSE) {
				gtk_idle_add(displayer_sdl_disable_func, NULL);
			}
			break;
		case SDL_VIDEORESIZE:
			// After a resize directly redraw the screen so return
			sdl_handler_resize(&sdl_event);
			return;
		case SDL_ACTIVEEVENT: // Application visibility event structure
			sdl_handler_active(&sdl_event);
			break;
		case SDL_VIDEOEXPOSE:
			// Since the screen is rendered quite often expose
			// events don't need to be handled expicit
			break;
		}
	}

	singit_framerate_counter_new_frame(STATUS->sfc);
}

static void *displayer_sdl_event_pthread_func(void *args)
{
	singit_framerate_counter_set_limit(STATUS->sfc, 32);

	while (!STATUS->kill_render_thread)
	{
		displayer_sdl_process_events(0);
	}

	singit_framerate_counter_set_limit(STATUS->sfc, 0);

#ifdef CODEDEBUG
	DEBUG(("  Quit render thread\n"));
#endif

	STATUS->kill_render_thread = FALSE;
	STATUS->render_thread = 0;

	pthread_exit(0);
}

static gboolean sdl_render_pthread_quit()
{
	if (STATUS->render_thread != 0)
	{
#ifdef CODEDEBUG
		DEBUG(("  Kill render thread\n"));
#endif
		STATUS->kill_render_thread = TRUE;
		pthread_join(STATUS->render_thread, NULL);
		STATUS->kill_render_thread = FALSE;
		STATUS->render_thread = 0;
		return TRUE;
	}
	return FALSE;
}

static gboolean set_sdl_render_pthread()
{
	gboolean result = FALSE;

	gboolean need_thread =
		((STATUS->xmms_playing == FALSE) ||
		 (STATUS->singit_playing == FALSE) ||
		 (xmms_remote_is_paused(sdl_gforce_dp.xmms_session) == TRUE));

	if ((STATUS->render_thread == 0) && (need_thread == TRUE))
	{
#ifdef CODEDEBUG
		DEBUG(("  Start render thread\n"));
#endif
		if (pthread_create(&STATUS->render_thread, NULL,
			displayer_sdl_event_pthread_func, NULL) != 0)
		{
			STATUS->render_thread = 0;
			g_print("Unable to create sdl render thread\n");
		}
		else { result = TRUE; }
	}
	else if (need_thread == FALSE)
		{ result = sdl_render_pthread_quit(); }

	return result;
}

static void displayer_sdl_init(void)
{
	if (sdl_gforce_status_ref() == NULL) {
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		return;
	}

#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_init]\n"));
#endif

	if (sdl_handler_init() == FALSE) {
		gtk_idle_add (displayer_sdl_disable_func, NULL);
		return;
	}

	sdl_lyrics_init();
	sdl_lyrics_set_optimize_font(TRUE);

	if (!((GET_DSCD->lyricFontName != NULL) &&
		sdl_lyrics_set_font(GET_DSCD->lyricFontName, GET_DSCD->lyricFontSize, TRUE)))
	{
		sdl_lyrics_set_font(SINGIT_DATA_DIR "/vixar.ttf", 32, TRUE);
	}

	sdl_skd_init();

	gtk_signal_connect(GTK_OBJECT(STATUS->config), "update",
		GTK_SIGNAL_FUNC(displayer_sdl_config_update_event), NULL);

	STATUS->xmms_playing = FALSE;

	STATUS->sfc = SINGIT_FRAMERATE_COUNTER
		(singit_framerate_counter_new(15));
	singit_framerate_counter_set_ticktype(STATUS->sfc, SDL_TICKS);
	singit_framerate_counter_start(STATUS->sfc);

	set_sdl_render_pthread();
}

static void displayer_sdl_finish(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_finish]\n"));
#endif

	STATUS->shutdown = TRUE;

	sdl_render_pthread_quit();

	if (STATUS->sdl_inuse == FALSE) {
		sdl_skd_finish();
		sdl_lyrics_finish();
	}

	sdl_gforce_status_unref(NULL);
}

static void displayer_sdl_set_time(guint time, GList *real_next)
{
	SingitSong *my_song;

	my_song = singit_song_attach(STATUS->song);

	set_sdl_render_pthread();

	if (STATUS->screen == SDL_GetVideoSurface())
	{
		singit_karaoke_data_set_time(STATUS->skd, time);
		STATUS->time = time;
		if (STATUS->render_thread == 0)
			{ displayer_sdl_process_events(time); }
	}

	singit_song_detach(&my_song);
}

static void displayer_sdl_playback_start(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_playback_start]\n"));
#endif

	STATUS->xmms_playing = TRUE;
}

static void displayer_sdl_playback_stop(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_playback_stop]\n"));
#endif

	STATUS->xmms_playing = FALSE;
	set_sdl_render_pthread();
}

static void displayer_sdl_configure(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_configure]\n"));
#endif

	if (!STATUS->is_fullscreen) {
		displayer_sdl_config_show();
	}
}

static void displayer_sdl_config_update_event(SingitConfigGen *scg, SDLGForceConfigData *cfg_data, gpointer user_data)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_config_update_event]\n"));
#endif

	if (!((cfg_data->lyricFontName != NULL) &&
		sdl_lyrics_set_font(cfg_data->lyricFontName, cfg_data->lyricFontSize, FALSE)))
	{
		sdl_lyrics_set_font(SINGIT_DATA_DIR "/vixar.ttf", 32, FALSE);
	}

	sdl_lyrics_set_optimize_font(cfg_data->optimizeFontSize);
}

static void displayer_sdl_config_update(void)
{
#ifdef CODEDEBUG
	DEBUG(("displayer_sdl.c [displayer_sdl_config_update]\n"));
#endif

	STATUS->toggle_yz = singit_remote_cfg_toggle_yz
		(sdl_gforce_dp.singit_session);

	if (STATUS->render_thread != 0)
		singit_framerate_counter_set_limit(STATUS->sfc,
			singit_remote_cfg_framerate_limit
			(sdl_gforce_dp.singit_session));
}

static void displayer_sdl_set_song(SingitSong *new_song)
{
	SingitSong *tmp_song;
	gboolean hide;

	tmp_song = STATUS->song;
	STATUS->song = singit_song_attach(new_song);
	singit_song_detach(&tmp_song);

	STATUS->singit_playing = new_song ?
		singit_song_lyrics_found(new_song) : FALSE;

	STATUS->xmms_playing =
		xmms_remote_is_playing(sdl_gforce_dp.xmms_session);

	sdl_lyrics_set_song(new_song);

	if (new_song && singit_song_lyrics_found(new_song)) {
		if (set_sdl_render_pthread() == TRUE)
			{ sdl_helper_map(TRUE); }
	}
	else {
		hide = ((GET_DSCD->hideIfNotFound &&
				!STATUS->is_fullscreen) ||
			(GET_DSCD->hideIfNotFoundFullscreen &&
				STATUS->is_fullscreen));

		if (hide) { sdl_helper_map(FALSE); }
		set_sdl_render_pthread();
	}
}
