/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <string.h>
#include <stdio.h>
#include <gtk/gtk.h>

#include "singit_tools.h"
#include "singit_debug.h"
#include "editor_config.h"
#include "editor_status.h"

#include "singit_marshallers.h"

#include "editor_query.h"
#include "editor_query_callbacks.h"

enum {
	GET_TEXT,
	SET_TEXT,
	ERROR_INFO,
	LAST_SIGNAL,
};

#define FREE(var) \
	if (var != NULL) { g_free(var); }

#define FREE_N(var) \
	if (var != NULL) { g_free(var); var = NULL; }

static GtkBinClass *parent_class = NULL;
static guint eqw_signals[LAST_SIGNAL] = { 0 };

static void editor_query_widget_destroy (GtkObject *object);

static void editor_query_widget_size_request(GtkWidget *widget, GtkRequisition *requisition);
static void editor_query_widget_size_allocate(GtkWidget *widget, GtkAllocation *allocation);
static void editor_query_widget_realize(GtkWidget *widget);
static void editor_query_widget_unrealize(GtkWidget *widget);

static void editor_query_widget_class_init (EditorQueryWidgetClass *klass)
{
	GtkObjectClass *object_klass;
	GtkWidgetClass *widget_klass;

	object_klass = (GtkObjectClass*) klass;
	widget_klass = (GtkWidgetClass*) klass;

	parent_class = gtk_type_class(GTK_TYPE_BIN);

	eqw_signals[GET_TEXT] =
		gtk_signal_new("get_text",
			GTK_RUN_LAST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (EditorQueryWidgetClass, get_text),
			gtk_marshal_STRING__NONE,
			GTK_TYPE_POINTER, 0);

	eqw_signals[SET_TEXT] =
		gtk_signal_new("set_text",
			GTK_RUN_FIRST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (EditorQueryWidgetClass, set_text),
			gtk_marshal_NONE__STRING,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_STRING);

	eqw_signals[ERROR_INFO] =
		gtk_signal_new("error_info",
			GTK_RUN_FIRST,
			object_klass->type,
			GTK_SIGNAL_OFFSET (EditorQueryWidgetClass, error_info),
			gtk_marshal_NONE__STRING,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_STRING);


	gtk_object_class_add_signals(object_klass, eqw_signals, LAST_SIGNAL);

	object_klass->destroy = editor_query_widget_destroy;

	widget_klass->size_request = editor_query_widget_size_request;
	widget_klass->size_allocate = editor_query_widget_size_allocate;
	widget_klass->realize = editor_query_widget_realize;
	widget_klass->unrealize = editor_query_widget_unrealize;

	klass->get_text = NULL;
	klass->set_text = NULL;
}

static void editor_query_widget_init (EditorQueryWidget *eqw)
{
	GtkWidget *query_vbox;
	GtkWidget *query_input_table;
	GtkWidget *artist_entry_label;
	GtkWidget *album_entry_label;
	GtkWidget *title_entry_label;
	GtkWidget *query_action_frame;
	GtkWidget *action_button_table;
	guint submit_query_button_key;
	GtkWidget *submit_query_button;
	guint clear_button_key;
	GtkWidget *clear_button;
	guint submit_button_key;
	GtkWidget *submit_button;
	guint save_button_key;
	GtkWidget *save_button;
	GtkWidget *action_hseparator;
	GtkWidget *query_result_scrolledwindow;
	GtkAccelGroup *accel_group;

	accel_group = gtk_accel_group_new();

	query_vbox = gtk_vbox_new (FALSE, 2);
	gtk_container_add(GTK_CONTAINER(eqw), query_vbox);
	gtk_container_set_border_width (GTK_CONTAINER (query_vbox), 2);
	gtk_widget_show(query_vbox);

	query_input_table = gtk_table_new (3, 3, FALSE);
	gtk_box_pack_start (GTK_BOX (query_vbox), query_input_table, FALSE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER (query_input_table), 2);
	gtk_table_set_row_spacings (GTK_TABLE (query_input_table), 2);
	gtk_table_set_col_spacings (GTK_TABLE (query_input_table), 4);
	gtk_widget_show(query_input_table);

	artist_entry_label = gtk_label_new (transadd(_("Artist"), ":"));
	gtk_table_attach (GTK_TABLE (query_input_table), artist_entry_label, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (artist_entry_label), 1, 0.5);
	gtk_widget_show(artist_entry_label);

	eqw->artist_combo = gtk_combo_new ();
	gtk_combo_disable_activate(GTK_COMBO(eqw->artist_combo));
	gtk_combo_set_use_arrows_always(GTK_COMBO(eqw->artist_combo), TRUE);
	gtk_combo_set_case_sensitive(GTK_COMBO(eqw->artist_combo), TRUE);
	gtk_table_attach (GTK_TABLE (query_input_table), eqw->artist_combo, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_show(eqw->artist_combo);

	album_entry_label = gtk_label_new (transadd(_("Album"), ":"));
	gtk_table_attach (GTK_TABLE (query_input_table), album_entry_label, 0, 1, 1, 2,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (album_entry_label), 1, 0.5);
	gtk_widget_show(album_entry_label);

	eqw->album_combo = gtk_combo_new ();
	gtk_combo_disable_activate(GTK_COMBO(eqw->album_combo));
	gtk_combo_set_use_arrows_always(GTK_COMBO(eqw->album_combo), TRUE);
	gtk_combo_set_case_sensitive(GTK_COMBO(eqw->album_combo), TRUE);
	gtk_table_attach (GTK_TABLE (query_input_table), eqw->album_combo, 1, 2, 1, 2,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_show(eqw->album_combo);

	title_entry_label = gtk_label_new (transadd(_("Title"), ":"));
	gtk_table_attach (GTK_TABLE (query_input_table), title_entry_label, 0, 1, 2, 3,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_misc_set_alignment (GTK_MISC (title_entry_label), 1, 0.5);
	gtk_widget_show(title_entry_label);

	eqw->title_combo = gtk_combo_new ();
	gtk_combo_disable_activate(GTK_COMBO(eqw->title_combo));
	gtk_combo_set_use_arrows_always(GTK_COMBO(eqw->title_combo), TRUE);
	gtk_combo_set_case_sensitive(GTK_COMBO(eqw->title_combo), TRUE);
	gtk_table_attach (GTK_TABLE (query_input_table), eqw->title_combo, 1, 2, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_show(eqw->title_combo);

	query_action_frame = gtk_frame_new (_("Actions"));
	gtk_table_attach (GTK_TABLE (query_input_table), query_action_frame, 2, 3, 0, 3,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_FILL), 1, 0);
	gtk_frame_set_label_align (GTK_FRAME (query_action_frame), 0.1, 0.5);
	gtk_widget_show(query_action_frame);

	action_button_table = gtk_table_new (3, 2, FALSE);
	gtk_container_add (GTK_CONTAINER (query_action_frame), action_button_table);
	gtk_container_set_border_width (GTK_CONTAINER (action_button_table), 2);
	gtk_table_set_row_spacings (GTK_TABLE (action_button_table), 3);
	gtk_table_set_col_spacings (GTK_TABLE (action_button_table), 5);
	gtk_widget_show(action_button_table);

	submit_query_button = gtk_button_new_with_label ("");
	submit_query_button_key = gtk_label_parse_uline
		(GTK_LABEL (GTK_BIN (submit_query_button)->child), _("_Query"));
	gtk_widget_add_accelerator (submit_query_button, "clicked", accel_group,
		submit_query_button_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
	gtk_table_attach (GTK_TABLE (action_button_table), submit_query_button, 0, 1, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_show(submit_query_button);

	clear_button = gtk_button_new_with_label ("");
	clear_button_key = gtk_label_parse_uline (GTK_LABEL
		(GTK_BIN (clear_button)->child), _("_Clear"));
	gtk_widget_add_accelerator (clear_button, "clicked", accel_group,
		clear_button_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
	gtk_table_attach (GTK_TABLE (action_button_table), clear_button, 1, 2, 0, 1,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_show(clear_button);

	submit_button = gtk_button_new_with_label ("");
	submit_button_key = gtk_label_parse_uline (GTK_LABEL
		(GTK_BIN (submit_button)->child), _("S_ubmit"));
	gtk_widget_add_accelerator (submit_button, "clicked", accel_group,
		submit_button_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
	gtk_table_attach (GTK_TABLE (action_button_table), submit_button, 0, 1, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive(submit_button, FALSE);
	gtk_widget_show(submit_button);

	save_button = gtk_button_new_with_label ("");
	save_button_key = gtk_label_parse_uline (GTK_LABEL
		(GTK_BIN (save_button)->child), _("_Save"));
	gtk_widget_add_accelerator (save_button, "clicked", accel_group,
		save_button_key, GDK_MOD1_MASK, (GtkAccelFlags) 0);
	gtk_table_attach (GTK_TABLE (action_button_table), save_button, 1, 2, 2, 3,
		(GtkAttachOptions) (GTK_EXPAND | GTK_FILL),
		(GtkAttachOptions) (0), 0, 0);
	gtk_widget_set_sensitive(save_button, FALSE);
	gtk_widget_show(save_button);

	action_hseparator = gtk_hseparator_new ();
	gtk_table_attach (GTK_TABLE (action_button_table), action_hseparator, 0, 2, 1, 2,
		(GtkAttachOptions) (GTK_FILL),
		(GtkAttachOptions) (GTK_FILL), 0, 0);
	gtk_widget_show(action_hseparator);

	query_result_scrolledwindow = gtk_scrolled_window_new (NULL, NULL);
	gtk_box_pack_start (GTK_BOX (query_vbox),
		query_result_scrolledwindow, TRUE, TRUE, 0);
	gtk_widget_set_usize (query_result_scrolledwindow, -2, 150);
	gtk_widget_show(query_result_scrolledwindow);

	gchar* clist_titles[5];
	clist_titles[0] = _("Artist");
	clist_titles[1] = _("Album");
	clist_titles[2] = _("Title");
	clist_titles[3] = _("Last modification");
	clist_titles[4] = _("Added by");

	eqw->query_result_clist = gtk_clist_new_with_titles
		(5, (gchar**) clist_titles);
	gtk_container_add (GTK_CONTAINER
		(query_result_scrolledwindow), eqw->query_result_clist);
	gtk_clist_column_titles_show(GTK_CLIST(eqw->query_result_clist));
	gtk_clist_set_column_width(GTK_CLIST(eqw->query_result_clist), 0, 92);
	gtk_clist_set_column_width(GTK_CLIST(eqw->query_result_clist), 1, 91);
	gtk_clist_set_column_width(GTK_CLIST(eqw->query_result_clist), 2, 95);
	gtk_clist_set_column_width(GTK_CLIST(eqw->query_result_clist), 3, 106);
	gtk_clist_set_column_width(GTK_CLIST(eqw->query_result_clist), 4, 80);
	gtk_widget_show(eqw->query_result_clist);

//	gtk_window_add_accel_group (GTK_WINDOW(eqw), accel_group);

	gtk_signal_connect(GTK_OBJECT(submit_query_button), "clicked",
		GTK_SIGNAL_FUNC(on_submit_query_button_clicked), eqw);

	gtk_signal_connect(GTK_OBJECT(clear_button), "clicked",
		GTK_SIGNAL_FUNC(on_clear_button_clicked), eqw);

	gtk_signal_connect(GTK_OBJECT(eqw->query_result_clist), "select_row",
		GTK_SIGNAL_FUNC(on_query_result_clist_select_row), eqw);

	gtk_signal_connect(GTK_OBJECT(eqw->query_result_clist), "click_column",
		GTK_SIGNAL_FUNC(on_query_result_clist_click_column), eqw);

	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(eqw->artist_combo)->entry),
		"activate",
		GTK_SIGNAL_FUNC(on_submit_query_button_clicked), eqw);
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(eqw->album_combo)->entry),
		"activate",
		GTK_SIGNAL_FUNC(on_submit_query_button_clicked), eqw);
	gtk_signal_connect(GTK_OBJECT(GTK_COMBO(eqw->title_combo)->entry),
		"activate",
		GTK_SIGNAL_FUNC(on_submit_query_button_clicked), eqw);

	eqw->servers = NULL;
	eqw->request = NULL;
	eqw->query_res = NULL;
	eqw->default_proxy = NULL;
	eqw->query_thread = 0;
	eqw->artist_match = MATCH_PERFECT;
	eqw->title_match = MATCH_FIRST;
}

GtkType editor_query_widget_get_type (void)
{
	static guint eqw_type = 0;

	if (!eqw_type) {
		GtkTypeInfo eqw_info = {
			(gchar*) "EditorQueryWidget",
			sizeof (EditorQueryWidget),
			sizeof (EditorQueryWidgetClass),
			(GtkClassInitFunc) editor_query_widget_class_init,
			(GtkObjectInitFunc) editor_query_widget_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL
		};

		eqw_type = gtk_type_unique (GTK_TYPE_BIN, &eqw_info);
	}

	return eqw_type;
}

static void editor_query_widget_destroy (GtkObject *object)
{
	EditorQueryWidget *eqw;

	g_return_if_fail (eqw = EDITOR_QUERY_WIDGET (object));

	if (eqw->request != NULL) {
		lyrix_request_unlink_http_data(eqw->request);
		lyrix_request_free(eqw->request);
	}

	if (eqw->servers != NULL) {
		GList *serv = eqw->servers;
		while (serv != NULL) {
			lyrix_request_data_free
				((LyrixRequestHTTP*) serv->data);
			serv = g_list_next(serv);
		}
		g_list_free(eqw->servers);
	}

	if (eqw->query_res != NULL)
		lyrix_query_result_free(eqw->query_res);

	if (GTK_OBJECT_CLASS (parent_class)->destroy)
		(* GTK_OBJECT_CLASS (parent_class)->destroy) (object);
}

static void editor_query_widget_size_request(GtkWidget *widget, GtkRequisition *requisition)
{
	GtkBin *bin;

	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_EDITOR_QUERY_WIDGET (widget));
	g_return_if_fail (requisition != NULL);

	bin = GTK_BIN (widget);

	requisition->width = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->xthickness) * 2;

	requisition->height = (GTK_CONTAINER (widget)->border_width +
		GTK_WIDGET (widget)->style->klass->ythickness) * 2;

	if (bin->child && GTK_WIDGET_VISIBLE (bin->child)) {
		GtkRequisition child_requisition;
		gtk_widget_size_request (bin->child, &child_requisition);

		requisition->width += child_requisition.width;
		requisition->height += child_requisition.height;
	}
}

static void editor_query_widget_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
	g_return_if_fail (widget != NULL);
	g_return_if_fail (IS_EDITOR_QUERY_WIDGET (widget));
	g_return_if_fail (allocation != NULL);

	if (GTK_WIDGET_MAPPED (widget) &&
		((widget->allocation.x != allocation->x) ||
		(widget->allocation.y != allocation->y) ||
		(widget->allocation.width != allocation->width) ||
		(widget->allocation.height != allocation->height)) &&
		(widget->allocation.width != 0) &&
		(widget->allocation.height != 0))
	{
		gtk_widget_queue_clear (widget);
	}

	widget->allocation = *allocation;

	if (GTK_BIN(widget)->child && GTK_WIDGET_VISIBLE (GTK_BIN (widget)->child))
	{
		GtkAllocation child_allocation;

		gint border_width = GTK_CONTAINER (widget)->border_width;
		gint xthickness = GTK_WIDGET (widget)->style->klass->xthickness;
		gint ythickness = GTK_WIDGET (widget)->style->klass->ythickness;

		child_allocation.x = border_width + xthickness;
		child_allocation.width = MAX(1,
			((gint) allocation->width - child_allocation.x * 2));

		child_allocation.y = border_width + ythickness;
		child_allocation.height = MAX (1,
			((gint) allocation->height - child_allocation.y -
			 border_width - ythickness));
		child_allocation.x += allocation->x;
		child_allocation.y += allocation->y;

		gtk_widget_size_allocate (GTK_BIN (widget)->child, &child_allocation);
	}
}

static void editor_query_widget_realize(GtkWidget *widget)
{
	if (GTK_WIDGET_CLASS(parent_class)->realize)
		GTK_WIDGET_CLASS(parent_class)->realize (widget);

	gtk_clist_columns_autosize
		(GTK_CLIST(EDITOR_QUERY_WIDGET(widget)->query_result_clist));
}

static void editor_query_widget_unrealize(GtkWidget *widget)
{
	pthread_join(EDITOR_QUERY_WIDGET(widget)->query_thread, NULL);

	if (GTK_WIDGET_CLASS(parent_class)->unrealize)
		GTK_WIDGET_CLASS(parent_class)->unrealize (widget);
}

GtkWidget* editor_query_widget_new(void)
{
#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("editor_query_widget.c [editor_query_widget_new]\n"));
#endif

	return gtk_type_new(TYPE_EDITOR_QUERY_WIDGET);
}

gboolean editor_query_widget_add_server
	(EditorQueryWidget *eqw, LyrixRequestHTTP *server)
{
	g_return_val_if_fail (eqw != NULL, FALSE);
	g_return_val_if_fail (server != NULL, FALSE);
	g_return_val_if_fail (IS_EDITOR_QUERY_WIDGET(eqw), FALSE);
	g_return_val_if_fail (lyrix_request_data_is_valid(server), FALSE);

	eqw->servers = g_list_append(eqw->servers, server);

	return TRUE;
}

void editor_query_widget_set_default_proxy
	(EditorQueryWidget *eqw, HTTPProxy *proxy)
{
	g_return_if_fail (eqw != NULL);
	g_return_if_fail (IS_EDITOR_QUERY_WIDGET(eqw));

	if (eqw->default_proxy != NULL) {
		FREE(eqw->default_proxy->name);
		FREE(eqw->default_proxy->user);
		FREE(eqw->default_proxy->passwd);

		if (proxy == NULL) {
			FREE_N(eqw->default_proxy);
			return;
		}
	}
	else {
		if (proxy == NULL)
			{ return; }
		eqw->default_proxy = g_malloc(sizeof(HTTPProxy));
	}

	eqw->default_proxy->name = g_strdup(proxy->name);
	eqw->default_proxy->port = proxy->port;
	eqw->default_proxy->user = g_strdup(proxy->name);
	eqw->default_proxy->passwd = g_strdup(proxy->passwd);
}

void  editor_query_widget_set_search_matches
	(EditorQueryWidget *eqw, SearchMatch artist, SearchMatch title)
{
	g_return_if_fail(eqw != NULL);
	g_return_if_fail(IS_EDITOR_QUERY_WIDGET(eqw));
	g_return_if_fail((artist > 0) && (artist < MATCH_LAST));
	g_return_if_fail((title > 0) && (title < MATCH_LAST));

	eqw->artist_match = artist;
	eqw->title_match = title;
}
