/*
 *  SingIt Lyrics Displayer
 *  Copyright (C) 2000 - 2003 Jan-Marek Glogowski <glogow@stud.fbi.fh-darmstadt.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */


#include <gdk/gdkx.h>
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "singit_debug.h"

#include "singit/config_gen.h"
#include "singit_tools.h"
#include "singit_marshallers.h"

enum {
	INIT_DATA,
	FREE_DATA,
	OPEN,
	CLOSE,
	LOAD,
	SAVE,
	SAVE_PART,
	SYNC,
	UPDATE,
	LAST_SIGNAL
};

static GtkObjectClass *parent_class = NULL;
static guint scg_signals[LAST_SIGNAL] = { 0 };

static void singit_config_gen_class_init (SingitConfigGenClass *klass);
static void singit_config_gen_init (SingitConfigGen *scg);
static void singit_config_gen_destroy(GtkObject *object);

GtkType singit_config_gen_get_type (void)
{
	static GtkType singit_config_gen_type = 0;

	if (!singit_config_gen_type) {

		static const GtkTypeInfo singit_config_gen_info =
		{
			(gchar*) "SingitConfigGen",
			sizeof (SingitConfigGen),
			sizeof (SingitConfigGenClass),
			(GtkClassInitFunc) singit_config_gen_class_init,
			(GtkObjectInitFunc) singit_config_gen_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		singit_config_gen_type = gtk_type_unique(GTK_TYPE_OBJECT, &singit_config_gen_info);
	}

	return singit_config_gen_type;
}

static void singit_config_gen_class_init (SingitConfigGenClass *klass)
{
	GtkObjectClass *object_class = (GtkObjectClass*) klass;
	parent_class = gtk_type_class(GTK_TYPE_OBJECT);

	object_class->destroy = singit_config_gen_destroy;

	scg_signals[INIT_DATA] =
		gtk_signal_new("init_data",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, init_data),
			gtk_marshal_POINTER__NONE,
			GTK_TYPE_POINTER, 0);

	scg_signals[FREE_DATA] =
		gtk_signal_new("free_data",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, free_data),
			gtk_marshal_NONE__POINTER,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	scg_signals[OPEN] =
		gtk_signal_new("open",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, open),
			gtk_marshal_POINTER__STRING,
			GTK_TYPE_POINTER, 1,
			GTK_TYPE_STRING);

	scg_signals[CLOSE] =
		gtk_signal_new("close",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, close),
			gtk_marshal_NONE__POINTER_STRING,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_STRING);

	scg_signals[LOAD] =
		gtk_signal_new("load",
			GTK_RUN_LAST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, load),
			gtk_marshal_NONE__POINTER_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER);

	scg_signals[SAVE] =
		gtk_signal_new("save",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, save),
			gtk_marshal_NONE__POINTER_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER);

	scg_signals[SAVE_PART] =
		gtk_signal_new("save_part",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, save_part),
			gtk_marshal_NONE__POINTER_POINTER_INT,
			GTK_TYPE_NONE, 3,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER,
			GTK_TYPE_INT);

	scg_signals[SYNC] =
		gtk_signal_new("sync",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, sync),
			gtk_marshal_NONE__POINTER_POINTER,
			GTK_TYPE_NONE, 2,
			GTK_TYPE_POINTER,
			GTK_TYPE_POINTER);

	scg_signals[UPDATE] =
		gtk_signal_new("update",
			GTK_RUN_FIRST,
			object_class->type,
			GTK_SIGNAL_OFFSET (SingitConfigGenClass, update),
			gtk_marshal_NONE__POINTER,
			GTK_TYPE_NONE, 1,
			GTK_TYPE_POINTER);

	klass->init_data = NULL;
	klass->free_data = NULL;
	klass->open = NULL;
	klass->close = NULL;
	klass->load = NULL;
	klass->save = NULL;
	klass->save_part = NULL;
	klass->sync = NULL;
	klass->update = NULL;

	gtk_object_class_add_signals (object_class, scg_signals, LAST_SIGNAL);
}

static void singit_config_gen_init (SingitConfigGen *scg)
{
	scg->filename = NULL;
	scg->config_data = NULL;
	scg->config_file = NULL;
	scg->properties = NULL;
	scg->create_new = TRUE;
}

static void singit_config_gen_destroy (GtkObject *object)
{
	SingitConfigGen *scg;

	g_return_if_fail (object != NULL);
	g_return_if_fail (IS_SINGIT_CONFIG_GEN (object));

	scg = SINGIT_CONFIG_GEN (object);

	if (scg->config_data != NULL)
		gtk_signal_emit (GTK_OBJECT(scg), scg_signals[FREE_DATA], scg->config_data);

	if (scg->filename != NULL)
		g_free(scg->filename);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(object);
}

GtkObject *singit_config_gen_new(void)
{
	return GTK_OBJECT(gtk_type_new(TYPE_SINGIT_CONFIG_GEN));
}

void singit_config_gen_set_filename(SingitConfigGen *scg, const gchar *filename)
{
	gchar *temp_filename;
	gboolean is_resolved_filename_absolut;

	g_return_if_fail(singit_config_gen_attach(scg));

	if (filename[0] == '~') {
		temp_filename = g_strconcat(g_get_home_dir(), &filename[1], NULL);
	}
	else { temp_filename = g_strdup(filename); }

	is_resolved_filename_absolut = g_path_is_absolute(temp_filename);

	if (is_resolved_filename_absolut == FALSE) {
		g_free(temp_filename);
		goto detach;
	}

	if (scg->filename != NULL)
		g_free(scg->filename);
	scg->filename = temp_filename;

detach:
	singit_config_gen_detach(&scg);
	g_return_if_fail(is_resolved_filename_absolut == TRUE);
}

void singit_config_gen_set_cfg_data(SingitConfigGen *scg, const gpointer data)
{
	g_return_if_fail (singit_config_gen_attach(scg));

	if (scg->config_data != NULL)
		gtk_signal_emit (GTK_OBJECT(scg),
			scg_signals[FREE_DATA], scg->config_data);

	scg->config_data = data;
	singit_config_gen_detach(&scg);
}

void singit_config_gen_set_create_new(SingitConfigGen *scg, gboolean create)
{
	g_return_if_fail (singit_config_gen_attach(scg));
	scg->create_new = create;
	singit_config_gen_detach(&scg);
}


// Two macros to ensure that all the actions are done on really opened configs
// otherwise set the appropriate error codes
#define __singit_config_gen_open_file \
	if (scg->config_file == NULL) { \
		gtk_signal_emit(GTK_OBJECT(scg), scg_signals[OPEN], \
			scg->filename, &scg->config_file); \
		if (scg->config_file == NULL) \
			singit_config_gen_set_error_code(scg, SCGS_NO_CFG); \
	} \
	\
	if (singit_config_gen_get_error_code(scg) == SCGS_OK) {

#define __singit_config_gen_close_file \
		if (scg->config_file != NULL) { \
			gtk_signal_emit (GTK_OBJECT(scg), \
				scg_signals[CLOSE], \
				scg->config_file, \
				scg->filename); \
			scg->config_file = NULL; \
		} \
		else { singit_config_gen_set_error_code(scg, SCGS_NO_CFG); } \
	}


// We have to prevent unlimited recursice calls to singit_config_gen_attach
// throught the singit_config_gen_get_data code, so we lock output while
// inside the attach code
SingitConfigGen* singit_config_gen_attach(SingitConfigGen *scg)
{
#ifdef CODEDEBUG
	static gboolean in_attach = FALSE;

	if (in_attach == FALSE) {
		in_attach = TRUE;
		DEBUG(DLV_ALL, ("singit_config_gen.c [singit_config_gen_attach] : "));
		in_attach = FALSE;
	}
#endif

	if (scg != NULL) {

		g_return_val_if_fail (IS_SINGIT_CONFIG_GEN (scg), NULL);

		gtk_object_ref(GTK_OBJECT(scg));

#ifdef CODEDEBUG
		if (in_attach == FALSE) {
			in_attach = TRUE;
			DEBUG(DLV_ALL, ("Attached\n"));
			in_attach = FALSE;
		}
#endif
		return scg;
	}

#ifdef CODEDEBUG
	if (in_attach == FALSE) {
		in_attach = TRUE;
		DEBUG(DLV_ALL, ("Failed\n"));
		in_attach = FALSE;
	}
#endif
	return scg;
}

void singit_config_gen_detach(SingitConfigGen **scg_p)
{
	SingitConfigGen *_scg;

	g_return_if_fail(scg_p != NULL);

	_scg = *scg_p;

	if (_scg == NULL)
		{ return; }

	g_return_if_fail(IS_SINGIT_CONFIG_GEN(_scg));

#ifdef CODEDEBUG
	DEBUG(DLV_ALL, ("singit_config_gen.c [singit_config_gen_detach]\n"));
#endif

	gtk_object_unref(GTK_OBJECT(_scg));
	_scg = NULL;
}

void singit_config_gen_load(SingitConfigGen *scg)
{
	g_return_if_fail (singit_config_gen_attach(scg));

#ifdef CODEDEBUG
	DEBUG(8, ("singit_config_gen.c [singit_config_gen_load] : "));
#endif

	if (scg->config_data != NULL)
		gtk_signal_emit (GTK_OBJECT(scg),
			scg_signals[FREE_DATA], scg->config_data);

	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[INIT_DATA],
		&scg->config_data);

	__singit_config_gen_open_file

	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[LOAD],
		scg->config_file, scg->config_data);
	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[UPDATE],
		scg->config_data);

	__singit_config_gen_close_file

#ifdef CODEDEBUG
	if (singit_config_gen_get_error_code(scg) == SCGS_OK)
		{ DEBUG(8, ("Ok\n")); }
	else { DEBUG(8, ("Failed\n")); }
#endif
	singit_config_gen_detach(&scg);
}

void singit_config_gen_save(SingitConfigGen *scg)
{
	g_return_if_fail (singit_config_gen_attach(scg));

#ifdef CODEDEBUG
	DEBUG(8, ("singit_config_gen.c [singit_config_gen_save] : "));
#endif
	__singit_config_gen_open_file

	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[SAVE],
		scg->config_file, scg->config_data);

	__singit_config_gen_close_file

#ifdef CODEDEBUG
	if (singit_config_gen_get_error_code(scg) == SCGS_OK)
		{ DEBUG(8, ("Ok\n")); }
	else { DEBUG(8, ("Failed\n")); }
#endif
	singit_config_gen_detach(&scg);
}

gpointer singit_config_gen_get_data(SingitConfigGen *scg)
{
	g_return_val_if_fail (scg != NULL, NULL);
	g_return_val_if_fail (IS_SINGIT_CONFIG_GEN (scg), NULL);

	return scg->config_data;
}

SCGStatus singit_config_gen_emit_update(SingitConfigGen *scg)
{
	g_return_val_if_fail(singit_config_gen_attach(scg),
		SCGS_PARAM_INVALID);

#ifdef CODEDEBUG
	DEBUG(8, ("singit_config_gen.c [singit_config_gen_emit_update]\n"));
#endif

	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[UPDATE], scg->config_data);

	singit_config_gen_detach(&scg);

	return SCGS_OK;
}

void singit_config_gen_save_part(SingitConfigGen *scg, gint part)
{
	gint result;

	g_return_if_fail (singit_config_gen_attach(scg));

#ifdef CODEDEBUG
	DEBUG(8, ("singit_config_gen.c [singit_config_gen_save_part] : "));
#endif

	__singit_config_gen_open_file

	gtk_signal_emit (GTK_OBJECT(scg), scg_signals[SAVE_PART],
		scg->config_file, scg->config_data, part);

	__singit_config_gen_close_file

#ifdef CODEDEBUG
	if (result == SCGS_OK) { DEBUG(8, ("Ok\n")); }
	else { DEBUG(8, ("Failed\n")); }
#endif
	singit_config_gen_detach(&scg);
}

SCGStatus singit_config_gen_get_error_code (SingitConfigGen *scg)
{
	SCGStatus code;

	g_return_val_if_fail(singit_config_gen_attach(scg),
		SCGS_PARAM_INVALID);

	code = scg->error_code;

	singit_config_gen_detach(&scg);

	return code;
}

SCGStatus singit_config_gen_set_error_code (SingitConfigGen *scg, SCGStatus error)
{
	g_return_val_if_fail(singit_config_gen_attach(scg),
		SCGS_PARAM_INVALID);

	if (scg->error_code == SCGS_OK)
		scg->error_code = error;

	singit_config_gen_detach(&scg);

	return SCGS_OK;
}

void singit_config_gen_clear_error_code (SingitConfigGen *scg)
{
	g_return_if_fail(singit_config_gen_attach(scg));
	scg->error_code = SCGS_OK;
	singit_config_gen_detach(&scg);
}

static void singit_config_gen_destroyed(gpointer data)
{
	*((gpointer*) data) = NULL;
}

void singit_config_gen_clear_initial(SingitConfigGen **scg_p)
{
	g_return_if_fail(scg_p != NULL);
	g_return_if_fail(*scg_p != NULL);
	g_return_if_fail(IS_SINGIT_CONFIG_GEN(*scg_p));
	g_return_if_fail(GTK_OBJECT_FLOATING(*scg_p));

	gtk_object_ref(GTK_OBJECT(*scg_p));
	gtk_object_weakref(GTK_OBJECT(*scg_p),
		singit_config_gen_destroyed, scg_p);
	gtk_object_sink(GTK_OBJECT(*scg_p));
}
