/*
 * $Id: xf86audio.c 138 2003-04-17 00:00:41Z aqua $
 * 
 * XF86Audio keys plugin for XMMS
 *
 * This is a "general" XMMS plugin (that is, one that doesn't provide any
 * audio processing functions).  When enabled, it grabs the XF86Audio*
 * keys for play/stop/next/previous, then translates keyrelease events
 * on those keys into XMMS actions.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * Copyright (c) 2003 by Devin Carraway <xf86audio-plugin@devin.com>.
 *
 */

#include <xmms/plugin.h>
#include <xmms/util.h>
#include <xmms/xmmsctrl.h>
#include <gtk/gtk.h>
#include <gdk/gdkx.h>
#include <gdk/gdk.h>
#include <X11/Xlib.h>
#include <X11/Xatom.h>

#define VERSION "0.4.1"

enum xf86audio_value {
       XF86AUDIO_PLAY = 0,
       XF86AUDIO_PAUSE,
       XF86AUDIO_STOP,
       XF86AUDIO_NEXT,
       XF86AUDIO_PREV,
       XF86AUDIO_RAISEVOLUME,
       XF86AUDIO_LOWERVOLUME,
       XF86AUDIO_MUTE,
       XF86AUDIO_MEDIA,
       XF86AUDIO_MAX,
};

static KeyCode map[XF86AUDIO_MAX];
static GeneralPlugin gpi;

static void grab_keys();
static void ungrab_keys();



/* Public API routines -- the general plugin API exposes these four
 * methods.  Most actual entries to the plugin happen through GTK
 * X event callbacks.
 */

/* plugin_init(): called by XMMS at plugin activation time; GTK is
 * initialized by that point and GDK is available.
 */
static void plugin_init()
{
	grab_keys();
}

/* plugin_cleanup(): called by XMMS ad plugin unload time (disable or
 * shutdown).
 */
static void plugin_cleanup()
{
	ungrab_keys();
}

/* plugin_about(): called when the "about" button in the plugins
 * configuration is pressed.  Typical behavior here is to present
 * a dialog explaining what the plugin does.
 */
static void plugin_about()
{
	static GtkWidget *about;

	if (about != NULL)
		return;

	about = xmms_show_message(
			"About XF86Audio Keys Control",
			"XF86Audio Keys Control Plugin " VERSION "\n\n"
			  "This plugin enables the XF86Audio keysyms produced by\n"
			  "multimedia keyboards to control XMMS playback.\n\n"
			  "Note that this plugin will not set up the initial keysym\n"
			  "mapping -- you should use xmodmap or Acme for that.\n\n"
			  "Copyright (c) 2003 by Devin Carraway <xf86audio-plugin@devin.com>.\n"
			  "This plugin is free software, released under the terms of the GNU\n"
			  "General Public License, v2.  You should have received a copy of\n"
			  "the license with this software.\n",
			"OK", 1, NULL, NULL);
	gtk_signal_connect(GTK_OBJECT(about), "destroy",
			GTK_SIGNAL_FUNC(gtk_widget_destroyed), &about);

}

/* gst_gplugin_info(): General plugin query function.  Oddly, XMMS does
 * not supply a pointer to a GeneralPlugin structure, so we keep a static
 * one around.
 */
GeneralPlugin *get_gplugin_info()
{
	gpi.description = "XF86Audio Keys Control";
	gpi.xmms_session = -1;
	gpi.init = plugin_init;
	gpi.about = plugin_about;
	gpi.cleanup = plugin_cleanup;
	return &gpi;
}


/* XMMS-public API methods end here; private code commences.
 */

static GdkFilterReturn xf86audio_filter(GdkXEvent *xevent, GdkEvent *event, gpointer data)
{
	XEvent *xev = (XEvent *)xevent;
	XKeyEvent *keyevent = (XKeyEvent *)xevent;
	KeyCode *k = (KeyCode *)data;
	gint i;
	gint vl, vr;
	static gint saved_vl, saved_vr;
	
	if (xev->type != KeyRelease)
		return GDK_FILTER_CONTINUE;

	for (i=0; i<XF86AUDIO_MAX; i++) {
		if (k[i] == keyevent->keycode)
			break;
	}
	if (i == XF86AUDIO_MAX) {
		g_warning("Received KeyRelease event for unrequested keycode %d", keyevent->keycode);
		return GDK_FILTER_CONTINUE;
	}

	switch (i) {
		case XF86AUDIO_PLAY:
			xmms_remote_play(gpi.xmms_session);
			break;
		case XF86AUDIO_STOP:
			xmms_remote_stop(gpi.xmms_session);
			break;
		case XF86AUDIO_PREV:
			xmms_remote_playlist_prev(gpi.xmms_session);
			break;
		case XF86AUDIO_NEXT:
			xmms_remote_playlist_next(gpi.xmms_session);
			break;
		case XF86AUDIO_PAUSE:
			if (xmms_remote_is_playing(gpi.xmms_session))
				xmms_remote_pause(gpi.xmms_session);
			else
				xmms_remote_play(gpi.xmms_session);
			break;
		case XF86AUDIO_RAISEVOLUME:
			xmms_remote_get_volume(gpi.xmms_session, &vl, &vr);
			vl = MIN(100, vl + 5);
			vr = MIN(100, vr + 5);
			xmms_remote_set_volume(gpi.xmms_session, vl, vr);
			break;
		case XF86AUDIO_LOWERVOLUME:
			xmms_remote_get_volume(gpi.xmms_session, &vl, &vr);
			vl = MAX(0, vl - 5);
			vr = MAX(0, vr - 5);
			xmms_remote_set_volume(gpi.xmms_session, vl, vr);
			break;
		case XF86AUDIO_MUTE:
			xmms_remote_get_volume(gpi.xmms_session, &vl, &vr);
			if (vl == 0 && vr == 0) {
				xmms_remote_set_volume(gpi.xmms_session,
						saved_vl, saved_vr);
			} else {
				xmms_remote_get_volume(gpi.xmms_session,
						&saved_vl, &saved_vr);
				xmms_remote_set_volume(gpi.xmms_session, 0, 0);
			}
			break;
		case XF86AUDIO_MEDIA:
			xmms_remote_eject(gpi.xmms_session);
			break;
		default: return GDK_FILTER_CONTINUE;
	}
	return GDK_FILTER_REMOVE;
}


static KeyCode grab_key(char *keystring)
{
	KeySym sym;
	KeyCode code;

	sym = XStringToKeysym(keystring);
	if (sym == NoSymbol)
		return 0;
	if ((code = XKeysymToKeycode(GDK_DISPLAY(), sym)) == 0)
		return 0;

	gdk_error_trap_push();
	XGrabKey(GDK_DISPLAY(), code,
			AnyModifier, GDK_ROOT_WINDOW(),
			1, GrabModeAsync, GrabModeAsync);
	gdk_flush();
	if (gdk_error_trap_pop()) {
		g_warning("Couldn't grab %s: another client may already have done so",
				keystring);
		return 0;
	}
	return code;
}

static void grab_keys()
{
	KeyCode code;
	
	if ((code = grab_key("XF86AudioNext")) != 0)
		map[XF86AUDIO_NEXT] = code;
	if ((code = grab_key("XF86AudioPrev")) != 0)
		map[XF86AUDIO_PREV] = code;
	if ((code = grab_key("XF86AudioPlay")) != 0)
		map[XF86AUDIO_PLAY] = code;
	if ((code = grab_key("XF86AudioStop")) != 0)
		map[XF86AUDIO_STOP] = code;
	if ((code = grab_key("XF86AudioPause")) != 0)
		map[XF86AUDIO_PAUSE] = code;
	if ((code = grab_key("XF86AudioRaiseVolume")) != 0)
		map[XF86AUDIO_RAISEVOLUME] = code;
	if ((code = grab_key("XF86AudioLowerVolume")) != 0)
		map[XF86AUDIO_LOWERVOLUME] = code;
	if ((code = grab_key("XF86AudioMute")) != 0)
		map[XF86AUDIO_MUTE] = code;
	if ((code = grab_key("XF86AudioMedia")) != 0)
		map[XF86AUDIO_MEDIA] = code;

	gdk_window_add_filter(GDK_ROOT_PARENT(),
			xf86audio_filter,
			map);
}


static void ungrab_key(KeyCode code)
{
	XUngrabKey(GDK_DISPLAY(), code,
			AnyModifier, GDK_ROOT_WINDOW());
}

static void ungrab_keys()
{
	int i;

	for (i = 0; i < XF86AUDIO_MAX; i++) {
		if (map[i] != 0)
			ungrab_key(map[i]);
	}
	gdk_window_remove_filter(GDK_ROOT_PARENT(),
			xf86audio_filter,
			map);

}


