#!/usr/bin/env python
# -*- coding: utf-8 -*-
# License:: GPL
# Author:: Alberto Milone (aka tseliot) (mailto:albertomilone@alice.it)
# Website:: http://albertomilone.com
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.


import gtk
import pango
import dbus
import logging
import sys
import os

import XorgOptionsEditor
from XorgOptionsEditor.policykit import PolicyKitAuthentication
from XorgOptionsEditor import xorgabstraction
from XorgOptionsEditor import XorgOptionsEditorStrings


POLICY_KIT_ACTION = 'com.ubuntu.xorgoptionseditor.mechanism.configure'
SERVICE_NAME   = 'com.ubuntu.XorgOptionsEditor.Mechanism'
OBJECT_PATH    = '/'
INTERFACE_NAME = 'com.ubuntu.XorgOptionsEditor.Mechanism'

import pygtk
pygtk.require('2.0')

import gtk
import gtk.glade
import gobject

from XorgOptionsEditor import SimpleGladeApp
from XorgOptionsEditor.SimpleGladeApp import SimpleGladeApp


#website = 'http://albertomilone.com/wordpress'
#def gotoUrl(d, link, data):
        #webbrowser.open(website)

#gtk.about_dialog_set_url_hook(gotoUrl, None)

def get_xkit_service(widget=None):
    '''
    returns a dbus interface to the grub.conf mechanism
    '''
    policy_auth = PolicyKitAuthentication()    

    granted = policy_auth.obtain_authorization(POLICY_KIT_ACTION, widget)
    logging.debug("granted = %s" % granted)

    if not granted:
        return None
    
    service_object = dbus.SystemBus().get_object(SERVICE_NAME, OBJECT_PATH)
    service = dbus.Interface(service_object, INTERFACE_NAME)

    return service

def gui_dialog(message, parent_dialog,
                      message_type=None,
                      widget=None, page=0, 
                      broken_widget=None):
    '''
    Displays an error dialog.
    '''
    if message_type == 'error':
        message_type = gtk.MESSAGE_ERROR
    elif message_type == 'info':
        message_type = gtk.MESSAGE_INFO


    logging.debug(message)
        
    dialog = gtk.MessageDialog(parent_dialog,
                               gtk.DIALOG_MODAL|gtk.DIALOG_DESTROY_WITH_PARENT,
                               message_type, gtk.BUTTONS_OK,
                               message)
    dialog.set_title(parent_dialog.get_title())#'Xorg Options Editor')
    dialog.set_icon_from_file('/usr/share/XorgOptionsEditor/gnome-display-properties.svg')
    
    if widget != None:
        if isinstance (widget, gtk.CList):
            widget.select_row (page, 0)
        elif isinstance (widget, gtk.Notebook):
            widget.set_current_page (page)
    if broken_widget != None:
        broken_widget.grab_focus ()
        if isinstance (broken_widget, gtk.Entry):
            broken_widget.select_region (0, -1)

    if parent_dialog:
        dialog.set_position (gtk.WIN_POS_CENTER_ON_PARENT)
        dialog.set_transient_for(parent_dialog)
    else:
        dialog.set_position (gtk.WIN_POS_CENTER)

    ret = dialog.run ()
    dialog.destroy()    

    return ret






class XorgOptionsEditor(SimpleGladeApp):
    def __init__(self, xorgSource=None, xorgDestination=None):
        #Set the Glade file
        self.translatedStrings = XorgOptionsEditorStrings.XorgOptionsEditorStrings()
        
        SimpleGladeApp.__init__(self, "/usr/share/XorgOptionsEditor/xorg-options-editor-gtk.glade")
        
        # self.globalDetails is a global index used for the interface
        # 
        # self.globalDetails = {
        #                       card_index (in tree): {
        #                                        'cardName': card_name,
        #                                        'listStore': object,
        #                                        'screen': screen_position (in XKit),
        #                                        'monitor': monitor_position (in XKit),
        #                                        'device': monitor_position (in XKit),
        #                                        'driver': driver in the combobox,
        #                                       }
        #                      }
        self.globalDetails = {}
        
        # self.optionsRef stores the details of the options detected from the xorg.conf
        # 
        # self.optionsRef = {
        #                    card_index: {
        #                                 'optionName': {
        #                                                 'optionType': 'Option',
        #                                                 'section': 'Device',
        #                                                 'position': 0,
        #                                                 'value': 'True',
        #                                                 'valueType': 'boolean'
        #                                                 },
        #                                                                                             
        #                                 'anotheroptionName': {},
        #                                 
        #                                }
        #                   }
        self.optionsRef = {}
        
        # self.manOptions stores the details of the options from the XML
        # 
        # self.manOptions = {
        #                    option1: {
        #                              'section': '',
        #                              'optionType': '',
        #                              'valueType': '',
        #                              'description': '',
        #                              'fullDescription': '',
        #                              }
        #                   }
        self.manOptions = {}
        
        # A dictionary containing the drivers to set for each card
        self.driversToUse = {}
        
        #Get the Main Window, and connect the destroy event
        #self.window = self.wTree.get_widget('MainWindow')
        self.window_main.realize()
        
        # Empty string which will hold the option name
        self.option = ''
        
        self.isUser = True
        
        # This filter will be set to the treemodel filter
        self.filter = None
        
        # List which will contain the options to filter out
        self.filteredOptions = []
        
        self.firstUpdate = True
        
        self.updatingFilter = False
        
        # Set Glade translations
        self.setGladeTranslations()
        
        # Pressing ENTER should mean OK in the edit_option_dialog
        self.text_entry.set_activates_default(True)
        self.edit_option_dialog.set_default_response(gtk.RESPONSE_OK)
        
        # Pressing ENTER should mean OK in the full_options_dialog
        self.full_options_dialog.set_default_response(gtk.RESPONSE_OK)
        
        
        # Interface to XKit, Hardware detection, etc.
        self.xorg = xorgabstraction.XorgAbstraction(xorgSource, xorgDestination)
        
        # Set the titles of column 1 and 2
        self.column1 = self.translatedStrings.main_treeview_column_header_1#('Option')
        self.column2 = self.translatedStrings.main_treeview_column_header_2#_('Value')
        
        #Add all of the List Columns to the driverView
        deviceColumn = 0
        deviceColumnTitle = "Device"
        self.addColumnToTree(deviceColumnTitle, deviceColumn, self.main_treeview)
        
        # Create the listStore Model to use with self.main_treeview
        self.deviceList = gtk.ListStore(str)
        
        # Fill the data structure which contain the options and the main liststore
        self.fillDetails()
        
        # Create listStores containing the options of each device
        # (or group, in the case of Misc) and listStores containing
        # the options from the XML for each device or group
        self.createListStores()
        
        # Fill the listStores (self.deviceXmlStore, self.miscXmlStore) with the
        # respective options from self.manOptions
        self.fillXmlListStores()
        
        # Fill the rows of self.options_treeview with the options of the device selected
        # in self.main_treeview
        self.fillOptionsTree()
        
        # Select the first entry in the device column
        # NOTE: this fails with a GTK warning if no rows are available
        self.main_treeview.set_cursor(0)
        
        self.window_main.show()
    
    def setBold(self, text):
        boldText = '<b>%s</b>' % text
        
        return boldText
    
    def setOptionName(self):
        selected_option_text = '%s "%s":' % (self.translatedStrings.enter_value_request, self.option)
        self.enter_value_request.set_text(selected_option_text)
        
    def setGladeTranslations(self):
        # Set the title of:
        #   window_main
        self.window_main.set_title(self.translatedStrings.main_title)
        #   edit_option_dialog
        self.edit_option_dialog.set_title(self.translatedStrings.edit_option_dialog_title)
        #   full_options_dialog
        self.full_options_dialog.set_title(self.translatedStrings.full_options_dialog_title)

        self.setOptionName()
        
        option_request_text = '%s:' % (self.translatedStrings.select_option_request)
        self.select_option_request.set_text(option_request_text)
        
        #self.screen_identifier_label.set_use_markup(True)
        
        screen_text = '%s:' % (self.translatedStrings.screen_identifier_label)
        
        self.screen_identifier_label.set_markup(self.setBold(screen_text))
        
        driver_text = '%s:' % (self.translatedStrings.driver_label)
        
        self.driver_label.set_markup(self.setBold(driver_text))
        
    
    def fillDeviceList(self, tree):
        '''Fill the left-most treeview'''
        # Fill the listStore (deviceList) with the devices
        # and the ServerFlags entry
        it = 0
        self.deviceList.append(['ServerFlags'])
        
        it += 1
        for card in self.xorg.cards:
            self.deviceList.append([card])
            it += 1

        #Attach the model to the treeView
        self.main_treeview.set_model(self.deviceList)
    
    def fillXorgOptionsList(self):
        '''Fill self.globalDetails and self.optionsRef with the options detected by XKit'''
        
        # Fill self.globalDetails and self.optionsRef
        self.xorg.fillXorgDetails(self.globalDetails, self.optionsRef, self.manOptions)
    
    def fillManOptions(self):
        '''Get the options from the man page'''
        # the sections you need
        # NOTE: subsections will be picked up too
        # the sections in the  must be sectionFilter lowercase
        sectionFilter = ['device', 'screen', 'monitor', 'serverflags', 'display subsection']
        self.manOptions = self.xorg.getManOptions(sectionFilter)#xorgmanparser.makeDict(sectionFilter)
    
    def fillDetails(self):
        '''Fill the data structure which contain the options and the main liststore'''
        
        # Get the options from the man page
        self.fillManOptions()
        
        # Get the options detected by XKit
        self.fillXorgOptionsList()
        
        # Fill the listStore (deviceList) with the devices
        # and the ServerFlags entry
        self.fillDeviceList(self.deviceList)
    
    
    def refreshCurrentOptions(self, index, data=None):
        '''Update the list of options in the treeview, the driver and the screen'''
        
        sectionIndex = self.globalDetails[index].get('device')
        
        # if ServerFlags, self.globalDetails['device'] will be ''
        if sectionIndex == None:
            sectionIndex = 0
        
        self.driverSelected = False
        
        self.options_treeview.set_model(self.globalDetails[index]['listStore'])
        
        # Make sure that the 1st row is selected
        self.options_treeview.set_cursor(0)
        cardName = self.globalDetails[index].get('cardName')
        
        # Set the current screen
        screen = self.globalDetails[index].get('screen')
        
        # if this fails, currentScreen will be set to 'Screen N/A'
        currentScreen = self.xorg.getIdentifier('Screen', screen, defaults='')#Screen N/A')
        
        self.screen_text.set_label('<small>%s</small>' % (currentScreen))
        
        
        # Set the current driver
        self.driver_combobox.get_model().clear()
        
        driver = self.globalDetails[index].get('driver')

        # If ServerFlags
        if not driver:
            self.driver_combobox.append_text('')#N/A')
            self.driverSelected = True
            self.driver_combobox.set_active(0)
        # If Device section or device related
        else:
            try:
                selected = self.xorg.allDrivers.index(driver)
                for driverEntry in self.xorg.allDrivers:
                    self.driver_combobox.append_text(driverEntry)
                self.driverSelected = True
                self.driver_combobox.set_active(selected)
            # if the driver is in xorg.conf but not in pciIds
            # isn't this dealt with in advance???
            # FIXME
            except (IndexError, ValueError):
                self.xorg.allDrivers.append(driver)
                selected = len(self.xorg.allDrivers) -1
                for driverEntry in self.xorg.allDrivers:
                    self.driver_combobox.append_text(driverEntry)
                self.driverSelected = True
                self.driver_combobox.set_active(selected)

        
            
    def createListStores(self):
        '''Create a ListStore per treeview item and create two ListStores for all the acceptable options
        
        self.globalDetails[item]['listStore'] contains all the options of a specific
        Device (or ServerFlags) section
        
        * self.deviceXmlStore contains all the options that (according to the man page) can be added to
          a Device or device-related section
        * self.miscXmlStore contains all the options that (according to the man page) can be added to a
          ServerFlags section
        '''

        # Create a gtk.ListStore for each item we want to put in
        # the left-most treeview
        for item in self.globalDetails:
            self.globalDetails[item]['listStore'] = gtk.ListStore(str, str)

        # self.deviceXmlStore contains the options from the Xorg xml
        # related to the following sections:
        # Device, Screen, Monitor, Display (subsection)
        self.deviceXmlStore = gtk.ListStore(str)
        self.deviceFilter = self.deviceXmlStore.filter_new()
#        self.deviceFilter.set_visible_column(1)
        

        # self.miscXmlStore contains the options from the Xorg xml
        # related to the sections in misc
        # (currently ServerFlags)
        self.miscXmlStore = gtk.ListStore(str)
        self.miscXmlFilter = self.miscXmlStore.filter_new()
#        self.miscXmlFilter.set_visible_column(1)

        
        
    
    def fillXmlListStores(self):
        '''Fill the listStores (self.deviceXmlStore, self.miscXmlStore) with the
        respective options from self.manOptions
        '''
        self.addColumnToTree(self.column1, 0, self.treeview_full)
        
        
        for option in self.manOptions.keys():
            description = self.manOptions[option].get('description', '')#'Description N/A')
            if self.manOptions[option]['section'] in ['Device', 'Screen', 'Monitor', 'display subsection']:
                self.deviceXmlStore.append(['<b>%s</b>\n <small>%s</small>' % (option, description)])
            elif self.manOptions[option]['section'] in ['ServerFlags']:
                self.miscXmlStore.append(['<b>%s</b>\n <small>%s</small>' % (option, description)])

    
    def fillOptionsTree(self):
        '''Fill the rows of self.options_treeview with the options of the device selected
        in self.main_treeview'''

        # Add the columns to the options tree
        self.addColumnToTree(self.column1, 0, self.options_treeview, True)
        self.addColumnToTree(self.column2, 1, self.options_treeview, True)#, editable=1)
        
        for index in self.globalDetails:
            for usedOption in self.optionsRef[index]:
                if usedOption not in self.xorg.optionsBlacklist:
                    self.globalDetails[index]['listStore'].append(['<b>%s</b>\n <small>%s</small>' \
                    % (usedOption, self.optionsRef[index][usedOption]['description']), \
                    '%s'%(self.optionsRef[index][usedOption]['value'])])
        

    
    def on_full_options_dialog_close(self, widget):
        self.full_options_dialog.hide()
        return True
        
    
    def on_full_options_dialog_delete_event(self, widget):
        self.full_options_dialog.hide()
        return True
    

    def on_main_treeview_cursor_changed(self, widget):
        '''Triggered when the leftmost treeview is changed'''
        self.isUser = False
        
        index = self.main_treeview.get_cursor()[0][0]
        
        # any device related section
        if self.globalDetails[index].get('device') != None:
            self.filter = self.deviceFilter
            self.driver_combobox.set_sensitive(1)
        # ServerFlags
        else:
            self.filter = self.miscXmlFilter
            self.driver_combobox.set_sensitive(0)
		
        self.refreshCurrentOptions(index)
        
        self.treeview_full.set_model(self.filter)
        # Update the filter
        self.updateFilterView()
        
        try:
            newDriver = self.globalDetails[index].get('driver')
        # if ServerFlags
        except AttributeError:
            newDriver = None
        
        
        
        
        # Set the driver in the combobox
        if newDriver:
            self.driver_combobox.set_active(self.xorg.allDrivers.index(newDriver))
        
        self.isUser = True
        

    
    def on_options_treeview_cursor_changed(self, widget):
        '''Triggered when the options treeview is changed'''
        fullName = ''
        fullDescription = ''
        
        hasDescription = False
        
        selection = self.options_treeview.get_selection()
        result = selection.get_selected()
        if result:
            model, iter = result
            if iter:
                option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
                value = model.get_value(iter, 1).replace('<i>', '').replace('</i>', '')
                
                strippedOption = option[:option.find('\n')]
                try:
                    fullDescription = self.manOptions.get(strippedOption).get('fullDescription').replace('\t', '').replace('. ', '.\n\n')
                
                except AttributeError:
                    pass
                else:
                    fullName = strippedOption
                    hasDescription = True
        
        textBuffer = self.description_textview.get_buffer()
        textBuffer.set_text('')
        
        #start, end = textBuffer.get_bounds()
        #textBuffer.delete(start, end)
        
        textBuffer = self.description_textview.get_buffer()
        if hasDescription:
            tag_table = textBuffer.get_tag_table()
            tag_table.foreach((lambda tag, table: table.remove(tag)), tag_table)
            
            tag_name = textBuffer.create_tag("bold-option", weight=pango.WEIGHT_BOLD)
            tag_name = textBuffer.create_tag("full-description", weight=pango.WEIGHT_NORMAL)
            
            bufIter = textBuffer.get_start_iter()
            textBuffer.insert_with_tags_by_name(bufIter, "%s\n" % fullName, "bold-option")
            textBuffer.insert_with_tags_by_name(bufIter, "%s" % fullDescription, "full-description")
            
        
        self.description_textview.set_buffer(textBuffer)
    
    def on_driver_combobox_changed(self, widget):
        '''Triggered when a new driver is selected'''
        if self.driverSelected:
            #print self.xorg.allDrivers[widget.get_active()]
            
            index = self.main_treeview.get_cursor()[0][0]
            oldDriver = self.globalDetails[index].get('driver')
            newDriver = self.xorg.allDrivers[widget.get_active()]
            
            # If the driver has changed and it is not a ServerFlags section
            if oldDriver != newDriver and self.globalDetails[index].get('device') != None:
                # Set the driver with XKit and add a reference to it
                # in self.globalDetails[index]['driver']
                if self.isUser:
                    operation = self.xorg.setDriver('Device', newDriver, self.globalDetails, index)
                    if not operation:
                        raise Exception('Could not set the driver')


    def on_add_button_clicked(self, widget):        
        '''Show the dialog to add options'''
        self.full_options_dialog.show()
    
    def on_edit_button_clicked(self, widget):
        # Clean the error label
        self.error_label.set_text('')
        
        # self.editOnly will let dlg_apply_button know that it doesn't
        # have to add another option but simply change the value of
        # an existing option
        self.editOnly = True
        
        selection = self.options_treeview.get_selection()
        result = selection.get_selected()
        if result:
            model, iter = result
            option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
            value = model.get_value(iter, 1).replace('<i>', '').replace('</i>', '')
            self.option = option[0: option.find('\n')].strip()
        
        #self.option_label.set_markup('<i>%s</i>' % (self.option))#set_text(self.option)
        self.setOptionName()
        self.text_entry.set_text(value)
        
        # Hide the dialog with the list of options
        self.full_options_dialog.hide()
        
        # Show the edit dialog
        self.edit_option_dialog.show()
    
    
    def on_remove_button_clicked(self, widget):
        '''Remove an option from the treeview and from X
        
        NOTE: if no row is selected, iter is None'''
        
        try:
            index = self.main_treeview.get_cursor()[0][0]
            
            # Remove the row from the treeview and get the option
            selection = self.options_treeview.get_selection()
            result = selection.get_selected()
            if result:
                model, iter = result
                option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
                option = option[0: option.find('\n')].strip()
                model.remove(iter)
            
            
                try:
                    # if it's in the xorg.conf
                    section = self.optionsRef[index][option]['section']
                except KeyError:
                    # if it was just added and it's not in the xorg.conf
                    section = self.manOptions[option]['section']
                
                if section.lower().find('subsection') != -1:
                    # only subsections in the Screen sections are supported
                    # such as "Display"
                    subName = section.lower().replace('subsection', '').strip().capitalize()
                    
                    try:
                        position = self.optionsRef[index][option]['position']
                    except KeyError:
                        # if it was just added and it's not in the xorg.conf
                        #index = self.main_treeview.get_cursor()[0][0]
                        position = self.globalDetails[index].get('screen')
                    
                    # Remove the option from the subsection with XKit
                    #print 'Removing %s from %s SubSection in Screen %d' % (option, subName, position)
                    self.xorg.removeSubOption('Screen', subName, option, position=position)
                
                # if it's a section
                else:
                    try:
                        position = self.optionsRef[index][option]['position']
                    except KeyError:
                        # if it was just added and it's not in the xorg.conf
                        if section.lower() == 'serverflags':
                            position = None
                        else:
                            #index = self.main_treeview.get_cursor()[0][0]
                            # valid only for device, screen, monitor sections
                            position = self.globalDetails[index].get(section.lower().strip())
                    # Remove the option from the section with XKit
                    self.xorg.removeOption(section, option, position=position, reference=None)
#                    print 'Removed option', option, 'from section', section, 'Nr.', position
                
                # Remove the option from self.optionsRef
                try:
                    del self.optionsRef[index][option]
                except KeyError:
                    pass

            # Update the filter
            self.updateFilterView()
        except TypeError:
            pass
            
            

    
    def addColumnToTree(self, title, columnId, treeview, fixed=False):#, editable=0, model=None, width=None):
        """This function adds a column to the list view.
        First it create the gtk.TreeViewColumn and then set
        some needed properties"""
#        if editable == 1:
#            # Spin column
#            #print 'columnId', columnId
#            column = gtk.CellRendererText()
#            column.set_property( 'editable', True )
#            xalign = 0.0
#            column.set_property('xalign', xalign)
#            column = gtk.TreeViewColumn(title, column, markup=columnId)
#            
#        elif editable == 2:
#            #exec 'cellname = "cell' + title + '"'
#            exec 'cell' + title + '= gtk.CellRendererToggle()'
#            exec 'cell' + title + '.set_property("activatable", True)'
#            xalign = 0.0
#            xpad = 0
#            exec 'cell' + title + '.set_property("xalign", ' + xalign + ')'
#            exec 'cell' + title + '.set_property("xpad", ' + xpad + ')'
#            exec 'cell' + title + '.connect( "toggled", self.col1_toggled_cb, model, cell' + title + ')'
#            exec 'cell' + title + '.set_active(0)'
#            
#            
#            exec 'column = gtk.TreeViewColumn(title, cell' + title + ', markup=' + columnId + ')'

#        else:
        mycell = gtk.CellRendererText()
        xalign = 0.0
        mycell.set_property('xalign', xalign)
        if fixed:
            # Add "..." to the option if it doesn't fit the column
            mycell.set_property("ellipsize", pango.ELLIPSIZE_END)
        column = gtk.TreeViewColumn(title, mycell, markup=columnId)
            
        if fixed:
            column.set_sizing(gtk.TREE_VIEW_COLUMN_FIXED)
            column.set_expand(True)
        
        column.set_resizable(True)
        
        #column.set_fixed_width(10)
#        if width != None:
#            print 'setting width for', title, 'to', width
#            column.set_max_width(width)
#            column.set_min_width(width)
        #column.set_max_width(20)
        column.set_sort_column_id(columnId)

        treeview.append_column(column)
        
    
    def on_exit_button_clicked(self, widget):
        gtk.main_quit()
    
    def on_apply_button_clicked(self, widget):
        '''Write the settings to the destination file'''
        self.conf = get_xkit_service()
        #"Could not connect to XorgOptionsEditor DBUS service."
        if not self.conf:
            gui_dialog(self.translatedStrings.dbus_connection_error,
                              self.window_main, message_type='error')
        else:
            tempFile = self.xorg.writeFile()
            status = self.conf.writeSettings(tempFile.name)
            
            #"Operation Complete. You will have to restart the Xserver in order to apply your settings."
            if status == True:
                gui_dialog(self.translatedStrings.operation_complete_success,
                                  self.window_main, message_type='info')
            else:
                #"Xorg Options Editor can\'t apply your settings."
                gui_dialog(self.translatedStrings.operation_complete_failure,
                                  self.window_main, message_type='error')
            
            
            tempFile.close()

        
    def on_window_main_delete_event(self, window, event):
        """Interpret a closing of the main window as a cancel event"""
        self.on_exitbutton_clicked(None)
        
    
    def on_dlg_apply_button_clicked(self, widget):
        '''
        validate the value and add it to the toadd list
        '''
        card = self.main_treeview.get_cursor()[0][0]

        try:
            optionType = self.manOptions[self.option]['optionType']
            section = self.manOptions[self.option]['section']
            valueType = self.manOptions[self.option]['valueType']
        except KeyError:
            optionType = self.optionsRef[card][self.option]['optionType']
            section = self.optionsRef[card][self.option]['section']
            valueType = self.optionsRef[card][self.option]['valueType']
        
        value = self.text_entry.get_text()
        
        valid, needsPrefix = self.xorg.hasValidType(value, valueType)#(value, valueType, optionType, section)
        
        if not valid:
            #'Error: entered value is not\n suitable for the option'
            notification = self.translatedStrings.option_validation_error
            colour = 'red'
            self.error_label.set_markup('<span color="%s"><b>%s</b></span>' % (colour, notification))
        else:
            index = self.main_treeview.get_cursor()[0][0]
            
            # See if it's ServerFlags
            if self.globalDetails[index].get('cardName') == '':#'ServerFlags':
                position = None
            else:
                position = self.globalDetails[index]['screen']
            
            # Hide the dialog
            self.edit_option_dialog.hide()
            
            # Add the option to the treeview
            try:
                description = (self.manOptions.get(self.option).get('description') or '')
            except (TypeError, AttributeError):
                description = ''#'Description N/A'
            
            # Make sure that optionType is set to None
            # e.g. if optionType == ''
            if not optionType:
                optionType = None
            
            if section.lower().find('subsection') != -1:
                # only subsections in the Screen sections are supported
                subName = section.lower().replace('subsection', '').strip().capitalize()
                #print 'position', position
                # Add the option to the SubSection with XKit
                self.xorg.addSubOption('Screen', subName, self.option, value, optiontype=optionType, position=position)
                #print 'added %s to %s subsection in Screen section %d' % (self.option, subName, position) 
            else:
                # Add the option with XKit
                if not needsPrefix:#e.g. if integer
                    self.xorg.addOption(section, self.option, value, optiontype=optionType, position=position, prefix='')
                else:
                    self.xorg.addOption(section, self.option, value, optiontype=optionType, position=position)

                #print 'added/edited %s (value=%s) in %s section %s' % (self.option, str(value), section, str(position))
                
            # Add an entry in self.optionsRef[index]
            self.optionsRef[index][self.option] = {}
            self.optionsRef[index][self.option]['optionType'] = optionType
            self.optionsRef[index][self.option]['section'] = section
                
            # If ServerFlags
            if position == None:
                position = 0
            self.optionsRef[index][self.option]['position'] = position
            self.optionsRef[index][self.option]['value'] = value
            try:
                self.optionsRef[index][self.option]['valueType'] = self.manOptions.get(self.option).get('valueType')
                self.optionsRef[index][self.option]['description'] = self.manOptions.get(self.option).get('description')
            except AttributeError:
                self.optionsRef[index][self.option]['valueType'] = None
                self.optionsRef[index][self.option]['description'] = ''#'Description N/A'
                
                #print 'Added this', self.optionsRef[index][self.option]
                
                
            if self.editOnly == True:
                selection = self.options_treeview.get_selection()
                result = selection.get_selected()
                if result:
                    model, iter = result
                    option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
                    option = option[0: option.find('\n')].strip()
                    model.set_value(iter, 1, value)
            else:
#                self.options_treeview.get_model().append(['<b>%s</b>\n <span font_family="monospace">%s</span>' % (self.option, description), '<i>%s</i>'%(value)])
                self.options_treeview.get_model().append(['<b>%s</b>\n <small>%s</small>' % (self.option, description), '%s'%(value)])
                
                # Update the filter
                self.updateFilterView()
        
        
    def on_edit_option_dialog_delete_event(self, *args):
        self.edit_option_dialog.hide()
        return True
    
    def on_dlg_back_button_clicked(self, widget):
        '''Hide the options dialog'''
        self.edit_option_dialog.hide()
        if self.editOnly == False:
            self.full_options_dialog.show()
    
    def on_treeview_full_cursor_changed(self, widget):
        '''If the left-most treeview changes'''
        selection = self.treeview_full.get_selection()
        result = selection.get_selected()
        if result:
            model, iter = result
            option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
            self.option = option[0: option.find('\n')].strip()
        
    def on_full_options_dialog_delete_event(self, *args):
        self.full_options_dialog.hide()
        return True
    
    def on_full_select_button_clicked(self, widget):
#        self.option_label.set_markup('<i>%s</i>' % (self.option))
        self.setOptionName()
        self.error_label.set_text('')
        self.full_options_dialog.hide()
        self.editOnly = False
        self.text_entry.set_text('')
        self.edit_option_dialog.show()
        
    def on_full_cancel_button_clicked(self, widget):
        self.full_options_dialog.hide()
    
    def updateFilterView(self):
        '''This method will update the filter'''
        if not self.updatingFilter:
            self.updatingFilter = True
            index = self.main_treeview.get_cursor()[0][0]
            del self.filteredOptions[:]
            
            # NOTE: keep this loop instead of doing something like:
            # self.filteredOptions = map(lambda n: n.lower().strip(), self.optionsRef[index].keys())
            # or it won't work
            for opt in self.optionsRef[index].keys():
                self.filteredOptions.append(opt.lower().strip())

            if self.firstUpdate:
                # Connect the filters to the visible_cb method
                self.deviceFilter.set_visible_func(self.visible_cb, self.filteredOptions)
                self.miscXmlFilter.set_visible_func(self.visible_cb, self.filteredOptions)
                self.firstUpdate = False

            self.filter.refilter()
            self.updatingFilter = False
        
        
    def visible_cb(self, model, iter, data):
        '''Method which determines whether to show a row or not'''
        option = model.get_value(iter, 0).replace('<b>', '').replace('</b>', '')
        option = option[0: option.find('\n')].strip().lower()
        return option not in data
        


def main():
    gtk.main()
    return 0

