/* C headers */
#include <stdio.h>
#include <limits.h>

/* Xlib headers */
#include <X11/Xlib.h>
#include <X11/keysym.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/Xlocale.h>
#include <X11/Xlibint.h>

/* internal Xlib headers */
#include "XlcGeneric.h"

/* program headers */
#include "range_tbl.h"
#include "conv_info.h"
#include "lookup_tbl.h"
#include "cstream.h"

#ifdef TIME_PRINT
#include "time.h"
#endif


/* Define */
#define LOCALE_DIR_ENV		"XLOCALEDIR"
#define WC_VALID_LENGTH_ENTRY		"wc_valid_length"
#define WC_TO_CS_CONV_TABLE_ENTRY	"wc_conversion_table"
#define CS_TO_WC_CONV_ENTRY		"cs_conversion"
#define CS_TO_WC_CONV_FILE_ENTRY	"cs_conversion_file"
#define CS_TO_WC_CONV_TABLE_ENTRY	"cs_conversion_table"

/* private functions */
static int generate_conversion_tables(XLCd lcd, boolean_t sysmode);
static int parse_range(char *str, RangeTblEntry *entry);
static int make_filename(char *str, char *filename);


char	*lc_name = NULL;
int	cs_num = 0;

int	main(int argc, char** argv)
{
	boolean_t	sys_mode;
	int	i, j;


	XLCd	lcd;
	CodeSet	*codesets;

	char	**value;
	int	num;

#ifdef TIME_PRINT
	float	start, end;
#endif

	/* argument parsing */
	sys_mode = B_TRUE;
	for(i = 1; i < argc; i++){
		if(!strcmp(argv[i], "+sys"))
			sys_mode = B_FALSE;
		else if(!strcmp(argv[i], "-locale"))
			lc_name = argv[++i];
		else if (!strcmp(argv[i], "-codeset_num"))
			cs_num = atoi(argv[++i]);
	}

	if (lc_name == NULL) {
	    fprintf(stderr, "Usage: %s -locale locale_name\n", argv[0]);
	    exit(1);
	}
	if(sys_mode == B_TRUE){
		/* create system tables */
		lcd = _XlcCreateLC(lc_name, _XlcGenericMethods);
		if (lcd == (XLCd)NULL){
			fprintf(stderr, "cannot create LC\n");
			return -1;
		}
		if (generate_conversion_tables(lcd, sys_mode)){
			fprintf(stderr, "%s : generating table failed\n", argv[0]);
			_XlcDestroyLC(lcd);
			return -1;
		}
		_XlcDestroyLC(lcd);
	}else{
		/* create user tables */
#ifdef TIME_PRINT
		start = put_time();
#endif
		if(!XSupportsLocale()) {
			fprintf(stderr, "X does not support locale\n");
			return -1;
		}
#ifdef TIME_PRINT
		end = put_time();
		fprintf(stdout, "time for XSupportsLocale() is ");
		fprintf(stdout, "\t%f\n", end - start);
#endif

#ifdef TIME_PRINT
		start = put_time();
#endif
		if(!XSetLocaleModifiers("")) {
			fprintf(stderr, "Warnimg: XSetLocaleModifiers() returns error\n");
			return -1;
		}
#ifdef TIME_PRINT
		end = put_time();
		fprintf(stdout, "time for XSetLocaleModifiers() is ");
		fprintf(stdout, "\t%f\n", end - start);
#endif

#ifdef TIME_PRINT
		start = put_time();
#endif
		lcd = _XOpenLC(NULL);
#ifdef TIME_PRINT
		end = put_time();
		fprintf(stdout, "time for XOpenLC() is ");
		fprintf(stdout, "\t%f\n", end - start);
#endif
		if (lcd == (XLCd)NULL){
			fprintf(stderr, "cannot create LC\n");
			return -1;
		}

#ifdef TIME_PRINT
		start = put_time();
#endif
		if (generate_conversion_tables(lcd, sys_mode)){
			fprintf(stderr, "%s : generating table failed\n", argv[0]);
			_XCloseLC(lcd);
			return -1;
		}
#ifdef TIME_PRINT
		end = put_time();
		fprintf(stdout, "time for generating tables is ");
		fprintf(stdout, "\t%f\n", end - start);
#endif
		_XCloseLC(lcd);
	}

	return 0;
}


static int
generate_conversion_tables(XLCd lcd, boolean_t sys_mode)
{
	int	i, j, k, l, num, codeset_num;
	int	wc_length, cs_length, cs_max_length;
	char	**str_list;
	LookupTable	*wtc_tbl = (LookupTable *)NULL;
	LookupTable	*ctw_tbl = (LookupTable *)NULL;
	RangeTbl	*byte_range_tbl = (RangeTbl *)NULL;
	RangeTbl	*usr_range_tbl = (RangeTbl *)NULL;
	ConvInfo	*conv_info_tbl = (ConvInfo *)NULL;
	char	resource_name[256];
	char	filename[PATH_MAX];
	FILE	*fp;

	if (cs_num)
		codeset_num = cs_num;
	else
		codeset_num = XLC_GENERIC(lcd, codeset_num);
	fprintf(stderr, "CODESETNUM=%d (%d)\n", codeset_num, cs_num);
	/* determin cs_length of the WC_to_CS conversion table */
	cs_max_length = 0;
	for(i = 0; i < codeset_num; i++){
		sprintf(resource_name, "cs%d.length", i);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
		if(num > 0){
			cs_length = atoi(str_list[0]);
			if(cs_max_length < cs_length)
				cs_max_length = cs_length;
		}
	}
	if(cs_max_length < 1)
		goto err_return;

	/* get wc_valid_len for WC_to_CS conversion table */
	_XlcGetResource(lcd, "XLC_XLOCALE", WC_VALID_LENGTH_ENTRY, &str_list, &num);
	if(num > 0)
		wc_length = atoi(str_list[0]);
	else
		goto err_return;

	/* create WC_to_CS conversion table */
	wtc_tbl = LookupTable_create(wc_length, cs_max_length, B_TRUE);
	if(wtc_tbl == (LookupTable *)NULL){
		goto err_return;
	}

	for(i = codeset_num - 1; i >= 0; i--){

		/* read Byte-Range info */
#ifdef VERBOSE
		fprintf(stdout, "reading byteM of cs%d ...\n", i);
#endif
		sprintf(resource_name, "cs%d.length", i);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
		if(num < 1)
			goto err_return;
		cs_length = atoi(str_list[0]);
		byte_range_tbl = RangeTbl_create(cs_length);
		if(byte_range_tbl == (RangeTbl *)NULL)
			goto err_return;
		for(j = 1; j <= cs_length; j++){
			sprintf(resource_name, "cs%d.byte%d", i, j);
			_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
			for(k = 0; k < num; k++){
				RangeTblEntry	entry;
				if(parse_range(str_list[k], &entry))
					goto err_return;
				if(RangeTbl_add(&(byte_range_tbl[cs_length - j]), &entry))
					goto err_return;
			}
		}

		/* read User-Range info */
#ifdef VERBOSE
		fprintf(stdout, "reading cs_range of cs%d ...\n", i);
#endif
		usr_range_tbl = RangeTbl_create(1);
		if(usr_range_tbl == (RangeTbl *)NULL)
			goto err_return;
		sprintf(resource_name, "cs%d.cs_range", i);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
		for(j = 0; j < num; j++){
			RangeTblEntry	entry;
			if(parse_range(str_list[j], &entry))
					goto err_return;
			if(RangeTbl_add(&(usr_range_tbl[0]), &entry))
					goto err_return;
		}

		/* read CS_to_WC conversion info */
#ifdef VERBOSE
		fprintf(stdout, "reading cs_conversion of cs%d ...\n", i);
#endif
		conv_info_tbl = ConvInfo_create();
		if(conv_info_tbl == (ConvInfo *)NULL)
			goto err_return;
		sprintf(resource_name, "cs%d.%s", i, CS_TO_WC_CONV_ENTRY);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
		if (num > 0){
			CStream	stream;
			ConvInfoEntry	entry;
			CStream_initString(&stream, str_list[0]);
			while(CStream_parseConvInfo(&stream, &entry) != EOF){
				if(ConvInfo_add(conv_info_tbl, &entry))
					goto err_return;
			}
		}

#ifdef VERBOSE
		fprintf(stdout, "reading cs_conversion_file of cs%d ...\n", i);
#endif
		sprintf(resource_name, "cs%d.%s", i, CS_TO_WC_CONV_FILE_ENTRY);
		_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
		if (num > 0){
			CStream	stream;
			ConvInfoEntry	entry;
			if(make_filename(str_list[0], filename))
				goto err_return;
			if ((fp = fopen(filename, "r")) == (FILE *)NULL)
			    goto err_return;

			CStream_initFile(&stream, fp);
			while(CStream_parseConvInfo(&stream, &entry) != EOF){
				if(ConvInfo_add(conv_info_tbl, &entry)){
					fclose(fp);
					goto err_return;
				}
			}
			fclose(fp);
		}
	
		/* create CS_to_WC conversion table */
#ifdef VERBOSE
		fprintf(stdout, "generating cs_conversion_table of cs%d ...\n", i);
#endif
		ctw_tbl = LookupTable_create(cs_length, wc_length, B_FALSE);
		if(ctw_tbl == (LookupTable *)NULL)
			goto err_return;

		/* write conversion info to conversion table */
		for(j = 0; j < conv_info_tbl->length; j++){
			unsigned long	cs_code;
			unsigned long	wc_code;
			int	loop_count;
			cs_code = conv_info_tbl->entry[j].cs_begin;
			wc_code = conv_info_tbl->entry[j].wc_begin;
			loop_count = conv_info_tbl->entry[j].cs_end - conv_info_tbl->entry[j].cs_begin + 1;
			for(k = 0; k < loop_count; k++){
				if(sys_mode == B_TRUE){
					if (LookupTable_add(ctw_tbl, cs_code, wc_code, -1))
						goto err_return;
				}
				if(RangeTbl_in(usr_range_tbl, cs_code) == B_FALSE)
					goto next_loop;
				for(l = 0; l < cs_length; l++){
					if(RangeTbl_in(&(byte_range_tbl[l]), (cs_code >> (l * 8)) & 0xff) == B_FALSE)
						goto next_loop;
				}
				if (LookupTable_add(wtc_tbl, wc_code, cs_code, i))
					goto err_return;
next_loop:
				cs_code++;
				wc_code++;
			}
		}

		if(sys_mode == B_TRUE){
			/* save CS_to_WC conversion file */
#ifdef VERBOSE
			fprintf(stdout, "writing cs_conversion_table of cs%d ...\n", i);
#endif
			sprintf(resource_name, "cs%d.%s", i, CS_TO_WC_CONV_TABLE_ENTRY);
			_XlcGetResource(lcd, "XLC_XLOCALE", resource_name, &str_list, &num);
			if (num > 0){
				if(make_filename(str_list[0], filename))
					goto err_return;
				if ((fp = fopen(filename, "w")) == (FILE *)NULL)
					goto err_return;
				if(LookupTable_save(ctw_tbl, fp))
					goto err_return;
			}
		}

		/* free resource */
		LookupTable_destroy(ctw_tbl);
		ctw_tbl = (LookupTable *)NULL;
		
		RangeTbl_destroy(usr_range_tbl, 1);
		usr_range_tbl = (RangeTbl *)NULL;

		RangeTbl_destroy(byte_range_tbl, cs_length);
		byte_range_tbl = (RangeTbl *)NULL;
	}

	/* save WC_to_CS conversion file */
#ifdef VERBOSE
	fprintf(stdout, "writing wc_conversion_table\n");
#endif
	_XlcGetResource(lcd, "XLC_XLOCALE", WC_TO_CS_CONV_TABLE_ENTRY, &str_list, &num);
	if (num < 1)
		goto err_return;
	if(make_filename(str_list[0], filename))
		goto err_return;
	if ((fp = fopen(filename, "w")) == (FILE *)NULL)
		goto err_return;
	if(LookupTable_save(wtc_tbl, fp))
		goto err_return;
	fclose(fp);

	/* free resource */
	LookupTable_destroy(wtc_tbl);
	wtc_tbl = (LookupTable *)NULL;

	return 0;

err_return:
	if(ctw_tbl)
		LookupTable_destroy(ctw_tbl);
	if(usr_range_tbl)
		RangeTbl_destroy(usr_range_tbl, 1);
	if(byte_range_tbl)
		RangeTbl_destroy(byte_range_tbl, cs_length);
	if(wtc_tbl)
		LookupTable_destroy(wtc_tbl);
	return -1;
}


static int
parse_range(char *str, RangeTblEntry *entry)
{
	int read_num = sscanf(str, "\\x%lx,\\x%lx", &(entry->begin), &(entry->end));
	switch(read_num){
		case 1:
			entry->end = entry->begin;
		case 2:
			break;
		default:
			return -1;
			break;
	}
	return 0;
}


static int
make_filename(char *str, char *filename)
{
    char	*dir_name;

    if(str == (char *)NULL)
	return -1;
    if(str[0] == '/'){
	if (strlen(str) >= PATH_MAX)
	    return -1;
	strcpy(filename, str);
    } else{
	dir_name = getenv("XLOCALEDIR");
	if (dir_name == (char *)NULL)
	    return -1;
	if (strlen(dir_name) + strlen(str) + 8 >= PATH_MAX)
	    return -1;
	strcpy(filename, dir_name);
	strcat(filename, "/common/");
	strcat(filename, str);
    }
    return 0;
}
