/*

    xpuyopuyo - pplayer.h     Copyright(c) 1999,2000 Justin David Smith
    justins(at)chaos2.org     http://chaos2.org/
    
    Header file for player information
    

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

*/


#ifndef __pplayer_h_defined
#define __pplayer_h_defined


#if TIME_WITH_SYS_TIME
#  include <sys/time.h>
#  include <time.h>
#else
#  if HAVE_SYS_TIME_H
#    include <sys/time.h>
#  else
#    include <time.h>
#  endif
#endif
#include <pfield.h>
#include <pai.h>


#define P_PLAYER_NUM_CLUMPS   8     /* We record multiple-match "clumps" from
                                       1 through this value.  */
#define P_PLAYER_NAME_SIZE    64    /* Maximum length of the player name */


/* Macros to select the appropriate player from the pconfig structure */
/* In each of these macros, c is a pconfig structure */
#define P_PLAYER(c, i)        ((c)->player[(i)])            /* Select player number i */
#define P_PLAYER_0(c)         P_PLAYER(c, 0)                /* Select player 0 */
#define P_PLAYER_1(c)         P_PLAYER(c, 1)                /* Select player 1 */
#define P_1_PLAYER(c)         ((c)->numplayers == 1)        /* truth if playing single-player */
#define P_2_PLAYER(c)         ((c)->numplayers == 2)        /* truth if playing two players (including AI's) */
#define P_AI_GAME(c)          ((c)->ai)                     /* truth if any AI's are playing */


/* The following macro is used to determine which player the player 2
   controls will affect.  If single-player or human-AI game, then the
   player 2 controls affect player 1's field.  In human-human games,
   the player 2 controls control the second player.  */
#define P_PLAYER__(c)         (P_1_PLAYER(c) || !P_PLAYER_HUMAN(c, P_PLAYER_1(c)) ? 0 : 1)


/* Possible player states */
#define P_STATE_ACTIVE        0x0100               /* Player has control */
#define P_STATE_PASSIVE       0x0200               /* Player waiting (matches running, etc) */
#define P_STATE_INACTIVE      0x0400               /* Player has no control (player won, dead) */
#define P_STATE_WINLOSE       0x0800               /* Player has decidedly won/lost */
#define P_STATE_PLAYING       (0x0001 | P_STATE_ACTIVE)     /* Playing (moving the piece) */
#define P_STATE_FALLING       (0x0002 | P_STATE_ACTIVE)     /* Player pressed "down" */
#define P_STATE_MATCHING      (0x0003 | P_STATE_PASSIVE)    /* Running matches */
#define P_STATE_MATCHING_2    (0x0004 | P_STATE_PASSIVE)    /* Clearing matches */
#define P_STATE_MATCH_FALL    (0x0005 | P_STATE_PASSIVE)    /* Dropping blobs which were hovering */
#define P_STATE_ROCKS         (0x0006 | P_STATE_PASSIVE)    /* Rocks being dumped on player */
#define P_STATE_ROCKS_FALL    (0x0007 | P_STATE_PASSIVE)    /* Rocks still being dumped ... */
#define P_STATE_IDLE          (0x0008 | P_STATE_PASSIVE)    /* Player is idling (delay before next piece) */
#define P_STATE_LOSER         (0x0009 | P_STATE_INACTIVE | P_STATE_WINLOSE)   /* Player lost :( */
#define P_STATE_WINNER        (0x000a | P_STATE_INACTIVE | P_STATE_WINLOSE)   /* Player won =)  */
#define P_STATE_DISABLED      (0x000b | P_STATE_INACTIVE | P_STATE_WINLOSE)   /* second player in 1-player game */
#define P_STATE_AIFALL        (0x000c | P_STATE_INACTIVE)   /* Player is AI, and is in freefall */
#define P_STATE_AIDROP        (0x000d | P_STATE_INACTIVE)   /* Player is AI, and AI "pressed" "down" */
#define P_STATE_LOSER_        (0x000e | P_STATE_INACTIVE)   /* Player sortof lost (?) */
#define P_STATE_NETWORK       (0x000f | P_STATE_INACTIVE)   /* Network player control */


/* Functions to check player status */
#define P_PLAYER_ACTIVE(p)    ((p)->state & P_STATE_ACTIVE)
#define P_PLAYER_PASSIVE(p)   ((p)->state & P_STATE_PASSIVE)
#define P_PLAYER_INACTIVE(p)  ((p)->state & P_STATE_INACTIVE)
#define P_PLAYER_WINLOSE(p)   ((p)->state & P_STATE_WINLOSE)
#define P_PLAYER_AI(c, p)     ((p)->aiactive)
#define P_PLAYER_LOCAL(c, p)  (!P_PLAYER_NETWORK(c, p))
#define P_PLAYER_NETWORK(c, p)((p)->network)
#define P_PLAYER_HUMAN(c, p)  (P_PLAYER_LOCAL(c, p) && !P_PLAYER_AI(c, p))


/* Return index of other player 1->0  and 0->1 */
#define P_OTHER_PLAYER(i)     (1 - (i))


typedef struct _pplayer {
   int            number;     /* Player ID */
   int            wins;       /* Number of wins */
   char           name[P_PLAYER_NAME_SIZE];  /* Player name */
   pfield         *field;     /* Player's field */
   nextpiece      next;       /* Next piece */
   struct timeval statetime;  /* Time to next state */
   int            state;      /* Next state */
   int            rocks;      /* Rocks in queue for us */
   int            accum;      /* Rocks in queue for them */
   int            score;      /* Total player score */
   int            scorerocks; /* Rocks sent over */
   int            aiactive;   /* AI is active if nonzero */
   int            network;    /* Network player if nonzero */
   int            airule;     /* AI ruleset to use */
   int            f_pieces;   /* Number of pieces seen */
   int            f_rkrecv;   /* Number of rocks received */
   int            f_rksent;   /* Number of rocks sent */
   airule         *ai;        /* AI rules (if aiactive, then AI is controlling) */
   int            num_clumps[P_PLAYER_NUM_CLUMPS]; /* Number of multi-matches */
   int            num_at_once;                     /* matches all at once! */
} pplayer;


/* Interface functions */
void p_player_resize_field(pplayer *p, int width, int height, int numcolors);
pplayer *p_player_new(int num, int nc, int width, int height, int numcolors);
void p_player_free(pplayer **p);
void p_player_calc_next(pplayer *p, int numcolors);
void p_player_set_name(pplayer *p, const char *name);


#endif
