/*-
# X-BASED DINOSAUR CUBE
#
#  Dino.c
#
###
#
#  Copyright (c) 1995 - 2003	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Dino */

#include "DinoP.h"
#include "Dino2dP.h"
#include "Dino3dP.h"

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wdino.ini"
#endif

#define SECTION "setup"

static const char *faceColorString[MAXFACES] =
{
	"255 0 0",
	"0 0 255",
	"255 255 0",
	"0 255 0",
	"255 255 255",
	"255 165 0",
};

static const char faceColorChar[MAXFACES] =
{'R', 'B', 'Y', 'G', 'W', 'O'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesDino(Widget current, Widget request, Widget renew);
static void DestroyDino(Widget old);
static void InitializeDino(Widget request, Widget renew);

DinoClassRec dinoClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Dino",		/* class name */
		sizeof (DinoRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeDino,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		NULL,		/* actions */
		0,		/* num actions */
		NULL,		/* resources */
		0,		/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroyDino,	/* destroy */
		NULL,		/* resize */
		NULL,		/* expose */
		(XtSetValuesFunc) SetValuesDino,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		NULL,		/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass dinoWidgetClass = (WidgetClass) & dinoClassRec;

void
SetDino(DinoWidget w, int reason)
{
	dinoCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

static void
SetDinoMove(DinoWidget w, int reason, int face, int position, int direction,
		int style)
{
	dinoCallbackStruct cb;

	cb.reason = reason;
	cb.face = face;
	cb.position = position;
	cb.direction = direction;
	cb.style = style;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(DinoWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->dino.fontInfo) {
		XUnloadFont(XtDisplay(w), w->dino.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->dino.fontInfo);
	}
	if ((w->dino.fontInfo = XLoadQueryFont(display,
			w->dino.font)) == NULL) {
		(void) sprintf(buf,
			"Cannot open %s font.\nAttempting %s font as alternate\n",
			w->dino.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->dino.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Cannot open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->dino.fontInfo) {
		w->dino.letterOffset.x = XTextWidth(w->dino.fontInfo, "8", 1)
			/ 2;
		w->dino.letterOffset.y = w->dino.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->dino.letterOffset.x = 3;
		w->dino.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "dino.log"
#endif

static DinoLoc slideCorner[MAXFACES][MAXORIENT][MAXORIENT / 2] =
{
	{
		{
			{3, TR, 0},
			{5, BR, 0}},
		{
			{3, TL, 1},
			{2, TR, 0}},
		{
			{2, TL, 1},
			{1, TR, 0}},
		{
			{5, BL, 1},
			{1, TL, 1}}
	},
	{
		{
			{2, TL, 0},
			{0, BL, 0}},
		{
			{2, BL, 0},
			{4, TL, 0}},
		{
			{4, BL, 0},
			{5, TL, 0}},
		{
			{0, TL, 0},
			{5, BL, 0}}
	},
	{
		{
			{3, TL, 0},
			{0, BR, 0}},
		{
			{3, BL, 0},
			{4, TR, 0}},
		{
			{4, TL, 1},
			{1, BR, 1}},
		{
			{0, BL, 1},
			{1, TR, 1}}
	},
	{
		{
			{5, BR, 1},
			{0, TR, 1}},
		{
			{5, TR, 1},
			{4, BR, 1}},
		{
			{4, TR, 1},
			{2, BR, 1}},
		{
			{0, BR, 1},
			{2, TR, 1}}
	},
	{
		{
			{3, BL, 1},
			{2, BR, 0}},
		{
			{3, BR, 0},
			{5, TR, 0}},
		{
			{5, TL, 1},
			{1, BL, 1}},
		{
			{2, BL, 1},
			{1, BR, 0}}
	},
	{
		{
			{3, BR, 1},
			{4, BR, 0}},
		{
			{3, TR, 1},
			{0, TR, 0}},
		{
			{0, TL, 1},
			{1, TL, 0}},
		{
			{4, BL, 1},
			{1, BL, 0}}
	}
};

static int  oppFace[MAXFACES] =
{4, 3, 5, 1, 0, 2};

static DinoCornerLoc oppCorner[MAXFACES][MAXORIENT] =
{
	{
		{4, 3},
		{4, 2},
		{4, 1},
		{4, 0}},
	{
		{3, 1},
		{3, 0},
		{3, 3},
		{3, 2}},
	{
		{5, 3},
		{5, 2},
		{5, 1},
		{5, 0}},
	{
		{1, 1},
		{1, 0},
		{1, 3},
		{1, 2}},
	{
		{0, 3},
		{0, 2},
		{0, 1},
		{0, 0}},
	{
		{2, 3},
		{2, 2},
		{2, 1},
		{2, 0}}
};

/* static int slideNextFace2[MAXFACES] = {4, 3, 5, 1, 0, 2}; */

static DinoCornerLoc slideNextFace[MAXFACES][MAXORIENT] =
{
	{
		{5, STRT},
		{3, CW},
		{2, STRT},
		{1, CCW}},
	{
		{0, CW},
		{2, STRT},
		{4, CCW},
		{5, HALF}},
	{
		{0, STRT},
		{3, STRT},
		{4, STRT},
		{1, STRT}},
	{
		{0, CCW},
		{5, HALF},
		{4, CW},
		{2, STRT}},
	{
		{2, STRT},
		{3, CCW},
		{5, STRT},
		{1, CW}},
	{
		{4, STRT},
		{3, HALF},
		{0, STRT},
		{1, HALF}}
};

static int  faceToRotate2[MAXFACES][MAXORIENT][2] =
{
	{
		{3, 5},
		{2, 3},
		{1, 2},
		{1, 5}},
	{
		{0, 2},
		{2, 4},
		{4, 5},
		{0, 5}},
	{
		{3, 0},
		{4, 3},
		{1, 4},
		{0, 1}},
	{
		{0, 5},
		{4, 5},
		{2, 4},
		{0, 2}},
	{
		{2, 3},
		{3, 5},
		{1, 5},
		{1, 2}},
	{
		{4, 3},
		{3, 0},
		{0, 1},
		{1, 4}}
};

static int  faceToRotate[MAXFACES][MAXORIENT] =
{
	{3, 2, 1, 5},
	{2, 4, 5, 0},
	{3, 4, 1, 0},
	{5, 4, 2, 0},
	{3, 5, 1, 2},
	{3, 0, 1, 4}
};

void
intCat(char ** string, const char * var1, const int var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + 21))) {
        	DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%d", var1, var2);
}

void
stringCat(char ** string, const char * var1, const char * var2)
{
	if (!(*string = (char *) malloc(strlen(var1) + strlen(var2) + 1))) {
        	DISPLAY_ERROR("Not enough memory, exiting.");
	}
	(void) sprintf(*string, "%s%s", var1, var2);
}

static void
CheckPieces(DinoWidget w)
{
	char *buf1 = NULL;

	if (w->dino.mode < PERIOD2 || w->dino.mode > BOTH) {
		 intCat(&buf1, "Mode is in error, use 2 for Period2, 3 for Period3, 4 for Both, defaulting to ",
		DEFAULTMODE);
		DISPLAY_WARNING(buf1);
		free(buf1);
		w->dino.mode = DEFAULTMODE;
	}
}

static int
GetStyle(DinoWidget w, int shift, int control, int alt)
{
	if (w->dino.mode != BOTH) {
		if (control && shift) {
			if (w->dino.mode == PERIOD3)
				return EDGE;
			else if (alt)
				return MIDDLE;
			else
				return CORNER;
		} else if (w->dino.mode == PERIOD2)
			return EDGE;
		else if (alt)
			return MIDDLE;
		else
			return CORNER;
	} else {
		if (shift)
			return EDGE;
		else {
			if (alt)
				return MIDDLE;
			else
				return CORNER;
		}
	}
}

Boolean
CheckSolved(DinoWidget w)
{
	int         face, position;
	DinoCornerLoc test;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < MAXORIENT; position++) {
			if (!position) {
				test.face = w->dino.cubeLoc[face][position].face;
				test.rotation = w->dino.cubeLoc[face][position].rotation;
			} else if (test.face !=		/*face */
				   w->dino.cubeLoc[face][position].face ||
				   (w->dino.orient && test.rotation !=	/*STRT - MAXORIENT */
				    w->dino.cubeLoc[face][position].rotation))
				return False;
		}
	return True;
}

#if DEBUG
void
PrintCube(DinoWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++) {
		for (position = 0; position < MAXORIENT; position++)
			(void) printf("%d %d  ",
				w->dino.cubeLoc[face][position].face,
				w->dino.cubeLoc[face][position].rotation);
		(void) printf("\n");
	}
	(void) printf("\n");
}
#endif

#ifdef WINVER
void
POLYGON(DinoWidget w, GC color, GC lineColor, const POINT * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	POINT      *temp = NULL;
	int         pt;

	if (!origin) {
		if (!(temp = (POINT *) malloc(sizeof (POINT) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	w->dino.hPen = CreatePen(PS_SOLID, 1, lineColor);
	w->dino.hOldPen = (HPEN) SelectObject(w->core.hDC, w->dino.hPen);
	w->dino.hBrush = CreateSolidBrush(color);
	w->dino.hOldBrush = (HBRUSH) SelectObject(w->core.hDC, w->dino.hBrush);
	(void) Polygon(w->core.hDC, (origin) ? poly : temp, n);
	(void) SelectObject(w->core.hDC, w->dino.hOldBrush);
	(void) DeleteObject(w->dino.hBrush);
	(void) SelectObject(w->core.hDC, w->dino.hOldPen);
        (void) DeleteObject(w->dino.hPen);
	if (!origin) {
		free(temp);
	}
}

void
POLYLINE(DinoWidget w, GC color, const POINT * poly, int n,
		Boolean origin)
{
	/* CoordModePrevious -> CoordModeOrigin */
	POINT      *temp = NULL;
	int         pt;

	if (!origin) {
		if (!(temp = (POINT *) malloc(sizeof (POINT) * n))) {
			DISPLAY_ERROR("Not enough memory, exiting.");
		}
		temp[0] = poly[0];
		for (pt = 1; pt < n; pt++) {
			temp[pt].x = temp[pt - 1].x + poly[pt].x,
			temp[pt].y = temp[pt - 1].y + poly[pt].y;
		}
	}
	w->dino.hPen = CreatePen(PS_SOLID, 1, color);
	w->dino.hOldPen = (HPEN) SelectObject(w->core.hDC, w->dino.hPen);
	(void) Polygon(w->core.hDC, (origin) ? poly : temp, n);
	(void) SelectObject(w->core.hDC, w->dino.hOldPen);
        (void) DeleteObject(w->dino.hPen);
	if (!origin) {
		free(temp);
	}
}
#endif

static void
DrawTriangle(DinoWidget w, int face, int position, int offset)
{
	if (w->dino.dim == 2)
		DrawTriangle2D((Dino2DWidget) w, face, position, offset);
	else if (w->dino.dim == 3)
		DrawTriangle3D((Dino3DWidget) w, face, position, offset);
}

void
DrawAllPieces(DinoWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < MAXORIENT; position++)
			DrawTriangle(w, face, position, FALSE);
}

static void
DrawFrame(const DinoWidget w, const Boolean focus)
{
	if (w->dino.dim == 2)
		DrawFrame2D((Dino2DWidget) w, focus);
	else if (w->dino.dim == 3)
		DrawFrame3D((Dino3DWidget) w, focus);
}

static void
MoveNoPieces(DinoWidget w)
{
	SetDino(w, DINO_ILLEGAL);
}

static void
RotateFace(DinoWidget w, int face, int direction)
{
	int         side;

	/* Read Face */
	for (side = 0; side < MAXORIENT; side++)
		w->dino.faceLoc[side] = w->dino.cubeLoc[face][side];
	/* Write Face */
	for (side = 0; side < MAXORIENT; side++) {
		w->dino.cubeLoc[face][side] = (direction == CW) ?
			w->dino.faceLoc[(side + MAXORIENT - 1) % MAXORIENT] :
			w->dino.faceLoc[(side + 1) % MAXORIENT];
		w->dino.cubeLoc[face][side].rotation =
			(w->dino.cubeLoc[face][side].rotation + direction) % MAXORIENT;
		DrawTriangle(w, face, side, FALSE);
	}
}

static void
ReadDiagonal(DinoWidget w, int face, int corner, int h)
{
	w->dino.spindleLoc[h][0] = w->dino.cubeLoc[face][corner];
	w->dino.spindleLoc[h][1] = w->dino.cubeLoc[face][(corner + 1) % MAXORIENT];
}

static void
WriteDiagonal(DinoWidget w, int face, int corner, int rotate, int h)
{
	w->dino.spindleLoc[h][0].rotation =
		(w->dino.spindleLoc[h][0].rotation + rotate) % MAXORIENT;
	w->dino.spindleLoc[h][1].rotation =
		(w->dino.spindleLoc[h][1].rotation + rotate) % MAXORIENT;
	w->dino.cubeLoc[face][corner] = w->dino.spindleLoc[h][0];
	DrawTriangle(w, face, corner, FALSE);
	w->dino.cubeLoc[face][(corner + 1) % MAXORIENT] = w->dino.spindleLoc[h][1];
	DrawTriangle(w, face, (corner + 1) % MAXORIENT, FALSE);
}

static void
ReadFace(DinoWidget w, int face, int h)
{
	int         side;

	for (side = 0; side < MAXORIENT; side++)
		w->dino.rowLoc[h][side] = w->dino.cubeLoc[face][side];
}

static void
WriteFace(DinoWidget w, int face, int rotate, int h)
{
	int         side, newSide;

	for (side = 0; side < MAXORIENT; side++) {
		newSide = (side + rotate) % MAXORIENT;
		w->dino.cubeLoc[face][newSide] = w->dino.rowLoc[h][side];
		w->dino.cubeLoc[face][newSide].rotation =
			(w->dino.cubeLoc[face][newSide].rotation + rotate) % MAXORIENT;
		DrawTriangle(w, face, (side + rotate) % MAXORIENT, FALSE);
	}
}

static void
MoveEdges(DinoWidget w, int face, int corner, int direction)
{
	int         k, newFace, rotate, newCorner, newDirection;

	ReadDiagonal((DinoWidget) w, face, corner, 0);
	for (k = 1; k <= 2; k++) {
		newFace = oppFace[face];
		/*rotate = (((face == 1 || face == 3) ? 1 : 3) + 3 * direction) %
		   MAXORIENT; */
		newCorner = ((((face == 1 || face == 3) + corner) % 2) ?
			     (corner + 3) : (corner + 1)) % MAXORIENT;
		rotate = (newCorner - corner + MAXORIENT) % MAXORIENT;
		newDirection = (rotate + direction) % MAXORIENT;
		if (k != 2)
			ReadDiagonal((DinoWidget) w, newFace, newCorner, k);
		WriteDiagonal(w, newFace, newCorner, rotate, k - 1);
		face = newFace;
		corner = newCorner;
		direction = newDirection;
	}
}

static void
MoveFaces(DinoWidget w, int f, int d, int rotate)
{
	int         k, face, newFace;

	face = faceToRotate2[f][d][0];
	ReadFace((DinoWidget) w, face, 0);
	for (k = 1; k <= 2; k++) {
		newFace = faceToRotate2[f][d][k % 2];
		rotate = MAXORIENT - rotate;
		if (k != 2)
			ReadFace((DinoWidget) w, newFace, k);
		WriteFace(w, newFace, rotate, k - 1);
		face = newFace;
	}
}

static void
MoveInsideCorners(DinoWidget w, int face, int corner, int direction)
{
	int         newFace, newCorner, newDirection, dir, k;

	ReadDiagonal((DinoWidget) w, face, corner, 0);
	for (k = 1; k <= MAXROTATE; k++) {
		dir = direction / 2;
		newFace = slideCorner[face][corner][dir].face;
		newCorner = slideCorner[face][corner][dir].side;
		newDirection = 2 * slideCorner[face][corner][dir].dir + !(newCorner % 2);
		if (k != MAXROTATE)
			ReadDiagonal((DinoWidget) w, newFace, newCorner, k);
		WriteDiagonal(w, newFace, newCorner,
		  (newDirection - direction + MAXORIENT) % MAXORIENT, k - 1);
		face = newFace;
		corner = newCorner;
		direction = newDirection;
	}
}

static void
MoveOutsideCorners(DinoWidget w, int face, int corner, int direction)
{
	int         newFace, newCorner, newDirection, dir, k;

	ReadDiagonal((DinoWidget) w, face, corner, 0);
	for (k = 1; k <= MAXROTATE; k++) {
		corner = (corner + 2) % MAXORIENT;
		dir = direction / 2;
		newFace = slideCorner[face][corner][dir].face;
		newCorner = (slideCorner[face][corner][dir].side + 2) % MAXORIENT;
		newDirection = 2 * slideCorner[face][corner][dir].dir + !(newCorner % 2);
		if (k != MAXROTATE)
			ReadDiagonal((DinoWidget) w, newFace, newCorner, k);
		WriteDiagonal(w, newFace, newCorner,
		  (newDirection - direction + MAXORIENT) % MAXORIENT, k - 1);
		face = newFace;
		corner = newCorner;
		direction = newDirection;
	}
}

static void
MovePieces(DinoWidget w, int face, int position, int direction, int style)
{
	int         corner, newCorner;

	corner = (position - !((position + direction) % 2) + MAXORIENT) % MAXORIENT;
	if (style == CORNER) {
		MoveInsideCorners(w, face, corner, direction);
	} else if (style == MIDDLE) {
		MoveOutsideCorners(w, face, corner, direction);
		newCorner = oppCorner[face][corner].rotation;
		face = oppCorner[face][corner].face;
		if (((face != 1 && face != 3) + corner) % 2)
			direction = (direction + 1) % MAXORIENT;
		else
			direction = (direction + 3) % MAXORIENT;
		corner = newCorner;
		MoveOutsideCorners(w, face, corner, direction);
	} else if (style == EDGE) {
		MoveEdges(w, face, corner, direction);
		MoveFaces(w, face, corner,
			((face == 2 || face == 5) ? CCW : HALF) % MAXORIENT);
	} else {
		int         k, newFace, rotate, newDirection;

		RotateFace(w, faceToRotate[face][direction % MAXORIENT], CW);
		RotateFace(w, faceToRotate[face][(direction + 2) % MAXORIENT], CCW);
		ReadFace((DinoWidget) w, face, 0);
		for (k = 1; k <= MAXORIENT; k++) {
			newFace = slideNextFace[face][direction % MAXORIENT].face;
			rotate = slideNextFace[face][direction % MAXORIENT].rotation;
			newDirection = (rotate + direction) % MAXORIENT;
			if (k != MAXORIENT)
				ReadFace((DinoWidget) w, newFace, k);
			WriteFace(w, newFace, rotate, k - 1);
			face = newFace;
			direction = newDirection;
		}
	}
}

static void
MoveControlCb(DinoWidget w, int face, int position, int direction, int style)
{
	int         newFace, newSide, newDirection, corner, newCorner;

	MovePieces(w, face, position, direction, style);
	SetDinoMove(w, DINO_CONTROL, face, position, direction, style);
	if (style == CORNER) {
		newSide = (position + 2) % MAXORIENT;
		MovePieces(w, face, newSide, direction, MIDDLE);
		SetDinoMove(w, DINO_CONTROL, face, newSide, direction, MIDDLE);
		corner = (position - !((position + direction) % 2) + MAXORIENT) % MAXORIENT;
		newFace = oppCorner[face][corner].face;
		newCorner = oppCorner[face][corner].rotation;
		newDirection = 2 * ((direction / 2 +
			(face != 1 && face != 3)) % 2) + !(newCorner % 2);
		newSide = newCorner;
		MovePieces(w, newFace, newSide, newDirection, CORNER);
		SetDinoMove(w, DINO_CONTROL, newFace, newSide, newDirection,
			CORNER);
	} else if (style == MIDDLE) {
		newSide = (position + 2) % MAXORIENT;
		MovePieces(w, face, newSide, direction, CORNER);
		SetDinoMove(w, DINO_CONTROL, face, newSide, direction,
			CORNER);
		corner = (position - !((position + direction) % 2) +
			MAXORIENT) % MAXORIENT;
		newCorner = oppCorner[face][corner].rotation;
		newFace = oppCorner[face][corner].face;
		if (newFace != 1 && newFace != 3)
			newDirection = (direction + 2) % MAXORIENT;
		else
			newDirection = direction;
		newDirection = 2 * (newDirection / 2) + !(newDirection % 2);
		newSide = (newCorner + 2) % MAXORIENT;
		MovePieces(w, newFace, newSide, newDirection, CORNER);
		SetDinoMove(w, DINO_CONTROL, newFace, newSide, newDirection,
			CORNER);
	} else if (style == EDGE) {
		newSide = (position + 2) % MAXORIENT;
		MovePieces(w, face, newSide, direction, EDGE);
		SetDinoMove(w, DINO_CONTROL, face, newSide, direction, EDGE);
	}
}

void
MoveDino(DinoWidget w, int face, int position, int direction, int style,
		int control)
{
	if (control)
		MoveControlCb(w, face, position, direction, style);
	else {
		MovePieces(w, face, position, direction, style);
		SetDinoMove(w, DINO_MOVED, face, position, direction, style);
	}
	PutMove(face, position, direction, style, control);
}

static      Boolean
SelectPieces(DinoWidget w, int x, int y, int *face, int *position)
{
	if (w->dino.dim == 2)
		return SelectPieces2D((Dino2DWidget) w, x, y,
			face, position);
	else if (w->dino.dim == 3)
		return SelectPieces3D((Dino3DWidget) w, x, y,
			face, position);
	return False;
}

static int
CheckMoveDir(int position1, int position2, int *direction)
{
	if (!((position1 - position2 + MAXORIENT) % 2))
		return FALSE;
	switch (position1) {
		case 0:
			*direction = (position2 == 1) ? 1 : 2;
			break;
		case 1:
			*direction = (position2 == 2) ? 2 : 3;
			break;
		case 2:
			*direction = (position2 == 3) ? 3 : 0;
			break;
		case 3:
			*direction = (position2 == 0) ? 0 : 1;
			break;
		default:
			return FALSE;
	}
	return TRUE;
}

static      Boolean
NarrowSelection(DinoWidget w, int *face, int *direction)
{
	if (w->dino.dim == 2)
		return NarrowSelection2D((Dino2DWidget) w, face, direction);
	else if (w->dino.dim == 3)
		return NarrowSelection3D((Dino3DWidget) w, face, direction);
	return False;
}

static      Boolean
PositionPieces(DinoWidget w, int x, int y, int *face, int *position, int *direction)
{
	if (!SelectPieces(w, x, y, face, position))
		return False;
	return NarrowSelection(w, face, direction);
}

void
MoveDinoInput(DinoWidget w, int x, int y, int direction, int shift,
		int control, int alt)
{
	int         face, position, style;

	if (!w->dino.practice && !control && CheckSolved(w)) {
		MoveNoPieces(w);
		return;
	}
	if (!PositionPieces(w, x, y, &face, &position, &direction))
		return;
	if (direction >= 2 * MAXORIENT) {
		if (control)
			style = FACE;
		else
			return;
	} else
		style = GetStyle(w, shift, control, alt);
	control = (control) ? 1 : 0;
	MoveDino(w, face, position, direction, style, control);
	if (!control && CheckSolved(w)) {
		SetDino(w, DINO_SOLVED);
	}
}

static void
ResetPieces(DinoWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < MAXORIENT; position++) {
			w->dino.cubeLoc[face][position].face = face;
			w->dino.cubeLoc[face][position].rotation = STRT -
				MAXORIENT;
		}
	FlushMoves(w);
	w->dino.started = False;
}

static void
GetPieces(DinoWidget w)
{
	FILE       *fp;
	int         c, mode, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &mode);
	switch (mode) {
		case PERIOD2:
			SetDino(w, DINO_PERIOD2);
			break;
		case PERIOD3:
			SetDino(w, DINO_PERIOD3);
			break;
		case BOTH:
			SetDino(w, DINO_BOTH);
			break;
		default:
			stringCat(&buf1, name, " corrupted: mode ");
			intCat(&buf2, buf1, mode);
			free(buf1);
			stringCat(&buf1, buf2, " should be between ");
			free(buf2);
			intCat(&buf2, buf1, PERIOD2);
			free(buf1);
			stringCat(&buf1, buf2, " and ");
			free(buf2);
			intCat(&buf1, buf2, BOTH);
			free(buf1);
			DISPLAY_WARNING(buf2);
			free(buf2);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->dino.orient != (Boolean) orient) {
		SetDino(w, DINO_ORIENT);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->dino.practice != (Boolean) practice) {
		SetDino(w, DINO_PRACTICE);
	}
#ifdef WINVER
	ResetPieces(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetDino(w, DINO_RESTORE);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: mode %d, orient %d, practice %d, moves %d.\n",
		name, mode, orient, practice, moves);
	free(lname);
	free(fname);
	w->dino.cheat = True; /* Assume the worst. */
}

static void
WritePieces(DinoWidget w)
{
	FILE       *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "mode%c %d\n", SYMBOL, w->dino.mode);
	(void) fprintf(fp, "orient%c %d\n", SYMBOL, (w->dino.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->dino.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearPieces(DinoWidget w)
{
	SetDino(w, DINO_CLEAR);
}

static void
UndoPieces(DinoWidget w)
{
	if (MadeMoves()) {
		int         face, position, direction, style, control;

		GetMove(&face, &position, &direction, &style, &control);
		direction = (direction < MAXORIENT) ?
			(direction + MAXORIENT / 2) % MAXORIENT :
			5 * MAXORIENT - direction;
		if (control)
			MoveControlCb(w, face, position, direction, style);
		else {
			MovePieces(w, face, position, direction, style);
			SetDinoMove(w, DINO_UNDO, face, position, direction,
				style);
		}
	}
}

static void
PracticePieces(DinoWidget w)
{
	SetDino(w, DINO_PRACTICE);
}

static void
RandomizePieces(DinoWidget w)
{
	int         face, position, direction, style;
	int         big = MAXORIENT * 3 + NRAND(2);

	w->dino.cheat = False;
	if (w->dino.practice)
		PracticePieces(w);
	SetDino(w, DINO_RESET);

#ifdef DEBUG
	big = 3;
#endif

	while (big--) {
		face = NRAND(MAXFACES);
		position = NRAND(MAXORIENT);
		direction = ((NRAND(2)) ? position + 1 : position + 3) %
			MAXORIENT;
		if (w->dino.mode == PERIOD2)
			style = EDGE;
		else if (w->dino.mode == BOTH)
			style = NRAND(3);
		else
			style = NRAND(2);
		MoveDino(w, face, position, direction, style, FALSE);
	}
	FlushMoves(w);
	SetDino(w, DINO_RANDOMIZE);
	if (CheckSolved(w)) {
		SetDino(w, DINO_SOLVED);
	}
}

static void
SolvePieces(DinoWidget w)
{
	if (CheckSolved(w))
		return;
	{
		SetDino(w, DINO_SOLVE_MESSAGE);
	}
}

static void
OrientizePieces(DinoWidget w)
{
	SetDino(w, DINO_ORIENT);
}


#ifdef WINVER
static void
SetValuesDino(DinoWidget w)
{
	struct tagColor {
		int         red, green, blue;
	} color;
	char        szBuf[80], buf[20], charbuf[2];
	int         face;

	w->dino.mode = GetPrivateProfileInt(SECTION, "mode",
		DEFAULTMODE, INIFILE);
	w->dino.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULTORIENT, INIFILE);
	w->dino.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULTPRACTICE, INIFILE);
	w->dino.dim = GetPrivateProfileInt(SECTION, "dim", 3, INIFILE);
	w->dino.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->dino.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	/* gray25 */
	w->dino.frameGC = RGB(color.red, color.green, color.blue);
	(void) GetPrivateProfileString(SECTION, "pieceBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->dino.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->dino.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAXFACES; face++) {
		(void) sprintf(buf, "faceColor%d", face);
		(void) GetPrivateProfileString(SECTION, buf,
			faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->dino.faceGC[face] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "faceChar%d", face);
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->dino.faceChar[face] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "name", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->dino.username, szBuf);
		w->dino.username[80] = 0;

}

void
DestroyDino(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

void
ResizeDino(DinoWidget w)
{
	if (w->dino.dim == 2)
		ResizeDino2D((Dino2DWidget) w);
	else if (w->dino.dim == 3)
		ResizeDino3D((Dino3DWidget) w);
}

void
SizeDino(DinoWidget w)
{
	ResetPieces(w);
	ResizeDino(w);
}

void
ExposeDino(DinoWidget w)
{
	if (w->dino.dim == 2)
		ExposeDino2D((Dino2DWidget) w);
	else if (w->dino.dim == 3)
		ExposeDino3D((Dino3DWidget) w);
}

#else
static void
GetColor(DinoWidget w, int face)
{
	XGCValues   values;
	XtGCMask    valueMask;
	XColor      colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->dino.reverse) {
		values.background = w->dino.foreground;
	} else {
		values.background = w->dino.background;
	}
	if (!w->dino.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				  DefaultColormapOfScreen(XtScreen(w)),
				 w->dino.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->dino.faceColor[face] =
				colorCell.pixel;
			if (w->dino.faceGC[face])
				XtReleaseGC((Widget) w, w->dino.faceGC[face]);
			w->dino.faceGC[face] = XtGetGC((Widget) w, valueMask,
				&values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->dino.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->dino.reverse) {
		values.background = w->dino.foreground;
		values.foreground = w->dino.background;
	} else {
		values.background = w->dino.background;
		values.foreground = w->dino.foreground;
	}
	if (w->dino.faceGC[face])
		XtReleaseGC((Widget) w, w->dino.faceGC[face]);
	w->dino.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

void
SetAllColors(DinoWidget w)
{
	XGCValues   values;
	XtGCMask    valueMask;
	int         face;

	valueMask = GCForeground | GCBackground;

	if (w->dino.reverse) {
		values.background = w->dino.background;
		values.foreground = w->dino.foreground;
	} else {
		values.foreground = w->dino.background;
		values.background = w->dino.foreground;
	}
	if (w->dino.inverseGC)
		XtReleaseGC((Widget) w, w->dino.inverseGC);
	w->dino.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->dino.mono) {
		if (w->dino.reverse) {
			values.background = w->dino.foreground;
			values.foreground = w->dino.background;
		} else {
			values.foreground = w->dino.foreground;
			values.background = w->dino.background;
		}
	} else {
		values.foreground = w->dino.frameColor;
		values.background = w->dino.background;
	}
	if (w->dino.frameGC)
		XtReleaseGC((Widget) w, w->dino.frameGC);
	w->dino.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->dino.mono) {
		if (w->dino.reverse) {
			values.background = w->dino.foreground;
			values.foreground = w->dino.background;
		} else {
			values.foreground = w->dino.foreground;
			values.background = w->dino.background;
		}
	} else {
		values.foreground = w->dino.borderColor;
		values.background = w->dino.background;
	}
	if (w->dino.borderGC)
		XtReleaseGC((Widget) w, w->dino.borderGC);
	w->dino.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAXFACES; face++)
		GetColor(w, face);
	if (w->dino.fontInfo)
		XSetFont(XtDisplay(w), w->dino.borderGC,
			w->dino.fontInfo->fid);
}

static      Boolean
SetValuesDino(Widget current, Widget request, Widget renew)
{
	DinoWidget  c = (DinoWidget) current, w = (DinoWidget) renew;
	Boolean     redraw = False, setColors = False;
	int         face;

	for (face = 0; face < MAXFACES; face++) {
		if (strcmp(w->dino.faceName[face], c->dino.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->dino.font != c->dino.font ||
			w->dino.borderColor != c->dino.borderColor ||
			w->dino.reverse != c->dino.reverse ||
			w->dino.mono != c->dino.mono) {
		loadFont(w);
		SetAllColors(w);
		redraw = True;
	} else if (w->dino.background != c->dino.background ||
			w->dino.foreground != c->dino.foreground ||
			setColors) {
		SetAllColors(w);
		redraw = True;
	}
	if (w->dino.orient != c->dino.orient) {
		ResetPieces(w);
		redraw = TRUE;
	} else if (w->dino.practice != c->dino.practice) {
		ResetPieces(w);
		redraw = TRUE;
	}
	if (w->dino.mode != c->dino.mode) {
		ResetPieces(w);
		redraw = TRUE;
	}

	if (w->dino.menu != -1) {
		switch (w->dino.menu) {
		case 0:
			w->dino.menu = -1;
			GetPieces(w);
			break;
		case 1:
			w->dino.menu = -1;
			WritePieces(w);
			break;
		case 3:
			w->dino.menu = -1;
			ClearPieces(w);
			break;
		case 4:
			w->dino.menu = -1;
			UndoPieces(w);
			break;
		case 5:
			w->dino.menu = -1;
			RandomizePieces(w);
			break;
		case 6:
			w->dino.menu = -1;
			SolvePieces(w);
			break;
		case 7:
			w->dino.menu = -1;
			OrientizePieces(w);
			break;
		case 8:
			w->dino.menu = -1;
			PracticePieces(w);
			break;
		default:
			w->dino.menu = -1;
			break;
		}
	}
	if (w->dino.currentDirection == DINO_RESTORE) {
		SetStartPosition(w);
		w->dino.currentDirection = DINO_IGNORE;
	} else if (w->dino.currentDirection == DINO_CLEAR) {
		ResetPieces(w);
		redraw = TRUE;
		w->dino.currentDirection = DINO_IGNORE;
	} else if (w->dino.currentDirection != DINO_IGNORE) {
		MovePieces(w, w->dino.currentFace, w->dino.currentPosition,
				w->dino.currentDirection, w->dino.style);
		w->dino.currentDirection = DINO_IGNORE;
	}
	return redraw;
}

static void
DestroyDino(Widget old)
{
	DinoWidget  w = (DinoWidget) old;
	int         face;

	for (face = 0; face < MAXFACES; face++)
		XtReleaseGC(old, w->dino.faceGC[face]);
	XtReleaseGC(old, w->dino.borderGC);
	XtReleaseGC(old, w->dino.frameGC);
	XtReleaseGC(old, w->dino.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->dino.select);
}

void
QuitDino(DinoWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->dino.fontInfo) {
		XUnloadFont(display, w->dino.fontInfo->fid);
		XFreeFont(display, w->dino.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}
#endif

#ifndef WINVER
static
#endif
void
InitializeDino(
#ifdef WINVER
DinoWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesDino(w);
#else
	DinoWidget w = (DinoWidget) renew;
	int face;

	w->dino.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->dino.mono);
	w->dino.fontInfo = NULL;
	for (face = 0; face < MAXFACES; face++)
		w->dino.faceGC[face] = NULL;
	w->dino.borderGC = NULL;
	w->dino.frameGC = NULL;
	w->dino.inverseGC = NULL;
#endif
	w->dino.focus = False;
	loadFont(w);
	CheckPieces(w);
	InitMoves();
	w->dino.cheat = False;
	ResetPieces(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->dino.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
#endif
}

void
HideDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetDino(w, DINO_HIDE);
}

void
SelectDino(DinoWidget w
#ifdef WINVER
, const int x, const int y, const int control
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int  x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
#endif

	if (SelectPieces(w, x, y,
		       &(w->dino.currentFace), &(w->dino.currentPosition))) {
		if (control || w->dino.practice || !CheckSolved(w))
			DrawTriangle(w, w->dino.currentFace,
				w->dino.currentPosition, TRUE);
	} else {
		w->dino.currentFace = DINO_IGNORE;
		w->dino.currentDirection = DINO_IGNORE;
	}
}

void
ReleaseDino(DinoWidget w
#ifdef WINVER
, const int x, const int y, const int shift, const int control, const int alt
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int  x = event->xbutton.x, y = event->xbutton.y;
	int shift = (int) (event->xbutton.state & (ShiftMask | LockMask));
	int control = (int) (event->xkey.state & ControlMask);
	int alt = (int) (event->xkey.state &
		     (Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask));
#endif
	int         style, face, position, count = -1;
	int         direction = 0;

	if (w->dino.currentFace == DINO_IGNORE)
		return;
	DrawTriangle(w, w->dino.currentFace, w->dino.currentPosition, FALSE);
	style = GetStyle(w, shift, control, alt);
	if (!control && !w->dino.practice && CheckSolved(w))
		MoveNoPieces(w);
	else if (SelectPieces(w, x, y,
		  &face, &position) && position != w->dino.currentPosition) {
		if (face == w->dino.currentFace)
			count = CheckMoveDir(w->dino.currentPosition, position,
				&direction);
		if (count == 1) {
			MoveDino(w, face, w->dino.currentPosition, direction,
				style, (control) ? 1 : 0);
			if (!control && CheckSolved(w)) {
				SetDino(w, DINO_SOLVED);
			}
		} else if (count == 0)
			MoveNoPieces(w);
	}
	w->dino.currentFace = DINO_IGNORE;
	w->dino.currentDirection = DINO_IGNORE;
}

void
PracticeDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	PracticePieces(w);
}

#ifndef WINVER
void
PracticeDinoMaybe(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->dino.started)
		PracticePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetDino(w, DINO_PRACTICE_QUERY);
	}
#endif
}

void
PracticeDino2(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->dino.started)
#endif
		PracticePieces(w);
}
#endif

void
RandomizeDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizePieces(w);
}

#ifndef WINVER
void
RandomizeDinoMaybe(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->dino.started)
		RandomizePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetDino(w, DINO_RANDOMIZE_QUERY);
	}
#endif
}

void
RandomizeDino2(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->dino.started)
#endif
		RandomizePieces(w);
}
#endif

void
GetDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetPieces(w);
}

void
WriteDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WritePieces(w);
}

void
ClearDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearPieces(w);
}

void
UndoDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoPieces(w);
}

void
SolveDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolvePieces(w);
}

void
OrientizeDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	OrientizePieces(w);
}

#ifndef WINVER
void
Period2ModeDino(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetDino(w, DINO_PERIOD2);
}

void
Period3ModeDino(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetDino(w, DINO_PERIOD3);
}

void
BothModeDino(DinoWidget w
, XEvent * event, char **args, int nArgs
)
{
	SetDino(w, DINO_BOTH);
}
#endif

void
EnterDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->dino.focus = True;
	DrawFrame(w, w->dino.focus);
}

void
LeaveDino(DinoWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->dino.focus = False;
	DrawFrame(w, w->dino.focus);
}

#ifdef WINVER
void
PeriodModeDino(DinoWidget w, const int mode)
{
	SetDino(w, mode + DINO_PERIOD2);
}

void
DimDino(DinoWidget w)
{
	SetDino(w, DINO_DIM);
}
#endif
