/*-
# MOTIF/X-BASED DINOSAUR CUBE
#
#  xdino.c
#
###
#
#  Copyright (c) 1995 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/12/15 Xt/Motif
  Version 5: 95/10/06 Xt/Motif
  Version 4: 94/05/30 Xt
*/

#ifdef WINVER
#include "DinoP.h"
#include "wdino.h"
#define TITLE "wdino"

static DinoRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->dino.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Dino.h"
#include "Dino2d.h"
#include "Dino3d.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/dino.xpm"
#endif
#include "icons/dino.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}] \n"
	"[-{mode {int} | both}] [-[no]orient] [-[no]practice] "
	"[-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The period 2 turning (i.e. the edges turn with 180 "
	"degree intervals) was suggested by Derek Bosch\n"
	"<bosch@sgi.com>.  The physical puzzle has period 3 "
	"turning (i.e. the points turn with 120 degree\n"
	"intervals) is called a Triangle - 4 Cube - Dinosaur "
	"with colored dinosaurs on it, (or Triangle - 4 Cube -\n"
	"6 Colors with solid colors, (a 4 color and a 2 color "
	"are also available with point oriented coloring))\n"
	"manufactured by The International Puzzle & Games "
	"(Great Animation International Corporation).\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a piece.  Release "
	"\"mouse-left\" button on a piece on the same face.\n"
	"in the same row.  The pieces will then turn towards "
	"where the mouse button was released.\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle (this must be done first\n"
	"to set a new record).\n"
	"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
	"has to orient the faces in orient mode, besides\n"
	"getting all the faces to be the same color.  To do this "
	"one has to get the lines to be oriented in\n"
	"the same direction.  This does add complexity (ever so "
	"slightly, there are only 2 possibilities) so there\n"
	"are 2 sets of records.\n"
	"Press \"2\", \"3\", \"B\", or \"b\" keys (not the keypad "
	"2, 3) to change modes to Period 2, Period 3, or Both.\n"
	"\"S\" or \"s\" keys reserved for the auto-solver "
	"(not implemented).\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"clicks.\n"
	"Key pad is defined for Dino2d as:\n"
	"7 8 9  Upper Left, Up, Upper Right\n"
	"  ^\n"
	"4< >6  Left, Right\n"
	"  v\n"
	"1 2 3  Lower Left, Down, Lower Right\n"
	"Note: Up, Left, Right, and Down only work when the "
	"control key is pressed and there is no analog for\n"
	"Dino3d.\n"
	"Key pad for Dino3d, use must use your intuition (is this "
        "a cop out or what?).  The key pad is defined\n"
        "differently depending on which side of the cube your mouse "
        "is pointing at.\n"
	"Use the alt key and the left mouse button, keypad, or arrow "
	"keys to move the center of the cube.  The 2 opposite\n"
	"corners do not move.\n"
	"Use the shift keys and the left mouse button, keypad, "
	"or arrow key to access \"Period 2\" turns from \"Both\" mode,\n"
	"otherwise it assumes \"Period 3\" turning.  Edges turn in "
	"\"Period 2\" and corners turn in \"Period 3\".  The\n"
	"\"Period 2\" mode has extra cuts around the faces.\n"
        "Use the control key and the left mouse button, keypad, or "
        "arrow keys to move the whole cube.  This is\n"
        "not recorded as a turn.\n"
};
static const char referencesHelp[] = {
	"International Puzzles & Games Catalog.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, not implemented.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "dino.scores"
#endif

#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord dinoRecord[MAXMODES][2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";
#ifndef HAVE_MOTIF
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
#ifdef HAVE_MOTIF
static Widget moves, record, message, modes[MAXMODES],
	orientSwitch, practiceSwitch;
static char buff[21];
static const char *modeString[] =
{
	"Period 2", "Period 3", "Both"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
Widget shell;
#endif
static Pixmap dinoIcon = None;
static Widget topLevel, dino2d, dino3d;
static Arg arg[5];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";

/* There's probably a better way to assure that they are the same but I don't
   know it off hand. */
static void
MakeEquivalent(String * username, int *mode, Boolean * orient, Boolean * practice)
{
	Boolean mono, reverse;
	Pixel foreground, background, pieceBorder;
	String faceColor[MAXFACES];

	XtVaGetValues(dino2d,
		XtNuserName, username,
		XtNmode, mode,
		XtNorient, orient,
		XtNpractice, practice,
		XtNmono, &mono,
		XtNreverse, &reverse,
		XtNforeground, &foreground,
		XtNbackground, &background,
		XtNpieceBorder, &pieceBorder,
		XtNfaceColor0, &(faceColor[0]),
		XtNfaceColor1, &(faceColor[1]),
		XtNfaceColor2, &(faceColor[2]),
		XtNfaceColor3, &(faceColor[3]),
		XtNfaceColor4, &(faceColor[4]),
		XtNfaceColor5, &(faceColor[5]), NULL);
	XtVaSetValues(dino2d,
		XtNdirection, DINO_IGNORE,
		XtNstart, False, NULL);
	XtVaSetValues(dino3d,
		XtNuserName, *username,
		XtNmode, *mode,
		XtNorient, *orient,
		XtNpractice, *practice,
		XtNmono, mono,
		XtNreverse, reverse,
		XtNdirection, DINO_IGNORE,
		XtNstart, False,
		XtNforeground, foreground,
		XtNbackground, background,
		XtNpieceBorder, pieceBorder,
		XtNfaceColor0, faceColor[0],
		XtNfaceColor1, faceColor[1],
		XtNfaceColor2, faceColor[2],
		XtNfaceColor3, faceColor[3],
		XtNfaceColor4, faceColor[4],
		XtNfaceColor5, faceColor[5], NULL);
}
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int dim, int mode, int moves, char *msg)
{
	char mb[10];

	if (mode == BOTH)
		(void) strcpy(mb, "both");
	else
		(void) sprintf(mb, "%d", mode);
	(void) sprintf(titleDsp, "%s%dd.%s: (%d/%s) - %s",
		progDsp, dim, mb, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int mode, orient;

	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			dinoRecord[mode][orient].score = NEVER;
			(void) strncpy(dinoRecord[mode][orient].name, NOACCESS,
				USERNAMELEN);
		}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int n, mode, orient;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++) {
			(void) fscanf(fp, "%d %s\n", &n, username);
			if (n <= dinoRecord[mode][orient].score ||
			    dinoRecord[mode][orient].score <= NEVER) {
				dinoRecord[mode][orient].score = n;
				(void) strncpy(dinoRecord[mode][orient].name,
					username, USERNAMELEN);
			}
		}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int mode, orient;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAXMODES; mode++) {
			for (orient = 0; orient < 2; orient++)
				(void) fprintf(fp, "%d %s\n",
					dinoRecord[mode][orient].score,
					dinoRecord[mode][orient].name);
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int mode, Boolean orient, Boolean practice)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (dinoRecord[i][j].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			dinoRecord[i][j].score, dinoRecord[i][j].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int mode, Boolean orient)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;

	if (counter < dinoRecord[i][j].score ||
	    dinoRecord[i][j].score <= NEVER) {
		ReadRecords();	/* Maybe its been updated by another */
		dinoRecord[i][j].score = counter;
		(void) strncpy(dinoRecord[i][j].name, usernameDsp, USERNAMELEN);
		if (orient && (counter < dinoRecord[!i][j].score ||
		    dinoRecord[!i][j].score <= NEVER)) {
			dinoRecord[!i][j].score = counter;
			(void) strncpy(dinoRecord[!i][j].name, usernameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(mode, orient, False);
		return True;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
DinoWidget w, HBRUSH brush
#else
void
#endif
)
{
	int mode;
	Boolean orient, practice;
	char *username;
#ifdef WINVER
	int dim;

	InitializeDino(w, brush);

	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	username = w->dino.username;
	dim = w->dino.dim;
	SET_STARTED(w, False);
#else
	MakeEquivalent(&username, &mode, &orient, &practice);
#ifdef HAVE_MOTIF
	XmToggleButtonSetState(modes[mode - PERIOD2], True, False);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		/* The NOACCESS is not necessary, but it may stop silliness. */
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", NOBODY);
#endif
	}
	PrintRecord(mode, orient, practice);
#ifdef WINVER
	PrintState(dim, mode, movesDsp, messageDsp);
#else
#ifndef HAVE_MOTIF
	PrintState(XtParent(dino2d), 2, mode, movesDsp, messageDsp);
	PrintState(XtParent(dino3d), 3, mode, movesDsp, messageDsp);
#endif
#endif
}

#ifdef WINVER
void
SetDino(DinoWidget w, int reason)
#else
static void
CallbackDino(Widget w, caddr_t clientData, dinoCallbackStruct * callData)
#endif
{
	int mode;
	Boolean orient, practice, start, cheat;
#ifdef WINVER
	int dim;
#else
	int reason = callData->reason;
	Widget otherw;
#ifndef HAVE_MOTIF
	int dim, otherdim;
#endif

	if (w == dino2d) {
		otherw = dino3d;
#ifndef HAVE_MOTIF
		dim = 2, otherdim = 3;
#endif
	} else {	/* (w == dino3d) */
		otherw = dino2d;
#ifndef HAVE_MOTIF
		dim = 3, otherdim = 2;
#endif
	}
#endif
	(void) strcpy(messageDsp, "");
#ifdef WINVER
	mode = w->dino.mode;
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
	start = w->dino.started;
	dim = w->dino.dim;
#else
	XtVaGetValues(w,
		XtNorient, &orient,
		XtNmode, &mode,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case DINO_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#ifndef HAVE_MOTIF
			(void) XIconifyWindow(XtDisplay(shell),
				XtWindow(shell),
				XScreenNumberOfScreen(XtScreen(shell)));
#endif
#endif
			break;
#ifndef WINVER
		case DINO_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
			XtSetValues(dino2d, arg, 1);
#endif
			break;
		case DINO_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(dino2d, arg, 1);
#endif
			break;
#endif
		case DINO_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case DINO_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, DINO_RESTORE);
			XtSetValues(otherw, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case DINO_CLEAR:
			movesDsp = 0;
#ifndef WINVER
			XtSetArg(arg[0], XtNdirection, DINO_CLEAR);
			XtSetValues(otherw, arg, 1);
			XtSetValues(w, arg, 1);
#endif
			break;
		case DINO_RESET:
			movesDsp = 0;
			break;
		case DINO_ILLEGAL:
			if (practice || start)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			break;
#ifndef WINVER
		case DINO_MOVED:
			movesDsp++;
			XtSetArg(arg[0], XtNstart, True);
			XtSetArg(arg[1], XtNface, callData->face);
			XtSetArg(arg[2], XtNpos, callData->position);
			XtSetArg(arg[3], XtNdirection, callData->direction);
			XtSetArg(arg[4], XtNstyle, callData->style);
			XtSetValues(otherw, arg, 5);
			XtSetValues(w, arg, 1);
			break;
		case DINO_CONTROL:
			XtSetArg(arg[0], XtNface, callData->face);
			XtSetArg(arg[1], XtNpos, callData->position);
			XtSetArg(arg[2], XtNdirection, callData->direction);
			XtSetArg(arg[3], XtNstyle, callData->style);
			XtSetValues(otherw, arg, 4);
			return;
#endif
		case DINO_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, mode, orient))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!", MESSAGELEN);
#ifdef WINVER
			SET_STARTED(w, False);
#else

			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
			XtSetValues(otherw, arg, 1);
#endif
			break;
		case DINO_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.practice = practice;
			w->dino.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
			XtSetValues(otherw, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case DINO_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->dino.practice = practice;
			w->dino.started = False;
#else
			XtSetArg(arg[0], XtNpractice, False);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
			XtSetValues(otherw, arg, 2);
#endif
			break;
		case DINO_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
			XtSetValues(otherw, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case DINO_PERIOD2:
		case DINO_PERIOD3:
		case DINO_BOTH:
			movesDsp = 0;
			mode = reason - DINO_PERIOD2 + PERIOD2;
			PrintRecord(mode, orient, practice);
#ifdef WINVER
			w->dino.mode = mode;
#else
			XtSetArg(arg[0], XtNmode, mode);
			XtSetValues(w, arg, 1);
			XtSetValues(otherw, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(modes[mode - PERIOD2], True, True);
#endif
#endif
			break;
		case DINO_COMPUTED:
#ifdef WINVER
			SET_STARTED(w, False);
#else
			XtSetArg(arg[0], XtNstart, False);
			XtSetValues(w, arg, 1);
			XtSetValues(otherw, arg, 1);
#endif
			break;
#ifndef WINVER
		case DINO_UNDO:
			movesDsp--;
			XtSetArg(arg[0], XtNstart, True);
			XtSetArg(arg[1], XtNface, callData->face);
			XtSetArg(arg[2], XtNpos, callData->position);
			XtSetArg(arg[3], XtNdirection, callData->direction);
			XtSetArg(arg[4], XtNstyle, callData->style);
			XtSetValues(otherw, arg, 5);
			XtSetValues(w, arg, 1);
			break;
#endif
#ifdef WINVER
		case DINO_DIM:
			dim++;
			if (dim > 3)
				dim = 2;
			w->dino.dim = dim;
			break;
#endif
	}
#ifdef WINVER
	PrintState(dim, mode, movesDsp, messageDsp);
#else
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(XtParent(w), dim, mode, movesDsp, messageDsp);
	PrintState(XtParent(otherw), otherdim, mode, movesDsp, messageDsp);
#endif
#endif
}

#ifdef WINVER
void
SetDinoMove(DinoWidget w, int reason, int face, int position, int direction,
		int style)
{
	int mode;
	/* Boolean orient, practice, cheat; */
	int dim;

	(void) strcpy(messageDsp, "");
	mode = w->dino.mode;
#if 0
	orient = w->dino.orient;
	practice = w->dino.practice;
	cheat = w->dino.cheat;
#endif
	dim = w->dino.dim;
	switch (reason) {
		case DINO_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case DINO_CONTROL:
			break;
		case DINO_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
	}
	PrintState(dim, mode, movesDsp, messageDsp);
}

static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	int shift = 0;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterDino(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.dino.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeaveDino(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyDino(brush);
			break;
		case WM_SIZE:
			ResizeDino(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposeDino(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizeDino(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectDino(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseDino(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0,
				(GetKeyState(VK_TAB) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetDino(&widget);
					ResizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WriteDino(&widget);
					break;
				case IDM_EXIT:
					DestroyDino(brush);
					break;
				case IDM_HIDE:
					HideDino(&widget);
					break;
				case IDM_CLEAR:
					ClearDino(&widget);
					SizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoDino(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizeDino(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticeDino(&widget);
					SizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolveDino(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizeDino(&widget);
					SizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIM:
					(void) DimDino(&widget);
					ResizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SHIFT_TR:
				case IDM_SHIFT_BR:
				case IDM_SHIFT_BL:
				case IDM_SHIFT_TL:
				case IDM_SHIFT_CW:
				case IDM_SHIFT_CCW:
				case IDM_SHIFT_TOP:
				case IDM_SHIFT_RIGHT:
				case IDM_SHIFT_BOTTOM:
				case IDM_SHIFT_LEFT:
					shift = 1;
				case IDM_TR:
				case IDM_BR:
				case IDM_BL:
				case IDM_TL:
				case IDM_CW:
				case IDM_CCW:
				case IDM_TOP:
				case IDM_RIGHT:
				case IDM_BOTTOM:
				case IDM_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						shift = shift || (GetKeyState(VK_CAPITAL) & 1);
						(void) MoveDinoInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFT_TR : IDM_TR),
							shift, FALSE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_SHIFTCONTROL_TR:
				case IDM_SHIFTCONTROL_BR:
				case IDM_SHIFTCONTROL_BL:
				case IDM_SHIFTCONTROL_TL:
				case IDM_SHIFTCONTROL_CW:
				case IDM_SHIFTCONTROL_CCW:
				case IDM_SHIFTCONTROL_TOP:
				case IDM_SHIFTCONTROL_RIGHT:
				case IDM_SHIFTCONTROL_BOTTOM:
				case IDM_SHIFTCONTROL_LEFT:
					shift = 1;
				case IDM_CONTROL_TR:
				case IDM_CONTROL_BR:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_TL:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveDinoInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFTCONTROL_TR : IDM_CONTROL_TR),
							shift, TRUE, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_CONTROLALT_TR:
				case IDM_CONTROLALT_BR:
				case IDM_CONTROLALT_BL:
				case IDM_CONTROLALT_TL:
				case IDM_CONTROLALT_CW:
				case IDM_CONTROLALT_CCW:
				case IDM_CONTROLALT_TOP:
				case IDM_CONTROLALT_RIGHT:
				case IDM_CONTROLALT_BOTTOM:
				case IDM_CONTROLALT_LEFT:
					shift = 1;
				case IDM_ALT_TR:
				case IDM_ALT_BR:
				case IDM_ALT_BL:
				case IDM_ALT_TL:
				case IDM_ALT_CW:
				case IDM_ALT_CCW:
				case IDM_ALT_TOP:
				case IDM_ALT_RIGHT:
				case IDM_ALT_BOTTOM:
				case IDM_ALT_LEFT:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MoveDinoInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_CONTROLALT_TR : IDM_ALT_TR),
							FALSE, shift, TRUE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}
					break;
				case IDM_PERIOD2:
				case IDM_PERIOD3:
				case IDM_BOTH:
					PeriodModeDino(&widget, (int) LOWORD(wParam) - IDM_PERIOD2);
					SizeDino(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else
static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-face{0|1|2|3|4|5} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{mode {int} | both}] [-[no]orient] [-[no]practice]\n");
	(void) fprintf(stderr,
		"\t[-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*dino.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*dino.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*dino.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*dino.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*dino.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*dino.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*dino.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*dino.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-face4", (char *) "*dino.faceColor4", XrmoptionSepArg, NULL},
	{(char *) "-face5", (char *) "*dino.faceColor5", XrmoptionSepArg, NULL},
	{(char *) "-mode", (char *) "*dino.mode", XrmoptionSepArg, NULL},
	{(char *) "-both", (char *) "*dino.mode", XrmoptionNoArg, (char *) "4"},
	{(char *) "-orient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*dino.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*dino.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fn", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*dino.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*dino.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackDinoPractice(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 8); /* menu choice */
		XtSetValues(dino2d, arg, 1);
	}
}

static void
CallbackDinoRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(dino2d, arg, 1);
	}
}
static void
ModeToggle(Widget w, int mode, XmToggleButtonCallbackStruct * cbs)
{
	Boolean orient, practice;

	if (cbs->set) {
		XtVaGetValues(dino2d,
			XtNorient, &orient,
			XtNpractice, &practice, NULL);
		XtVaSetValues(dino2d,
			XtNmode, mode + PERIOD2, NULL);
		XtVaSetValues(dino3d,
			XtNmode, mode + PERIOD2, NULL);
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(mode + PERIOD2, orient, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int mode;
	Boolean orient = cbs->set, practice;

	XtVaGetValues(dino2d,
		XtNmode, &mode,
		XtNpractice, &practice, NULL);
	XtVaSetValues(dino2d,
		XtNorient, orient, NULL);
	XtVaSetValues(dino3d,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(mode, orient, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int mode;
	Boolean orient, practice = cbs->set;

	XtVaSetValues(dino2d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaSetValues(dino3d,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaGetValues(dino2d,
		XtNmode, &mode,
		XtNpractice, &orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(mode, orient, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(dino2d, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(dino2d, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, labelRowCol, radioRowCol, switchRowCol;
	Widget messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString orientString, practiceString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Dino",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) dino_xpm, &dinoIcon, NULL,
			&xpmAttributes);
	}
	if (dinoIcon == (Pixmap) NULL)
#endif
		dinoIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) dino_bits, dino_width, dino_height);
	XtSetArg(arg[0], XtNiconPixmap, dinoIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "(Solve)");
	orientString = XmStringCreateSimple((char *) "Orientize");
	practiceString = XmStringCreateSimple((char *) "Practice");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, orientString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(orientString);
	XmStringFree(practiceString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackDinoPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackDinoRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("Rowcol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	(void) XtVaCreateManagedWidget("PeriodText",
		xmLabelGadgetClass, labelRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Periods:", 9, NULL);
	radioRowCol = XtVaCreateManagedWidget("radioRowCol",
		xmRowColumnWidgetClass, labelRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) ModeToggle, (XtPointer) i);
	}
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);

	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	message = XtVaCreateManagedWidget("Play Dino! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	dino2d = XtCreateManagedWidget("dino",
		dino2dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(dino2d,
		XtNheight, 200, NULL);
	XtAddCallback(dino2d, XtNselectCallback,
		(XtCallbackProc) CallbackDino, (XtPointer) NULL);
	dino3d = XtCreateManagedWidget("dino",
		dino3dWidgetClass, mainPanel, NULL, 0);
	XtVaSetValues(dino3d,
		XtNheight, 200, NULL);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	shell = XtCreateApplicationShell(argv[0],
		topLevelShellWidgetClass, NULL, 0);
	XtSetValues(shell, arg, 2);
	dino2d = XtCreateManagedWidget("dino",
		dino2dWidgetClass, topLevel, NULL, 0);
	XtAddCallback(dino2d, XtNselectCallback,
		(XtCallbackProc) CallbackDino, (XtPointer) NULL);
	dino3d = XtCreateManagedWidget("dino",
		dino3dWidgetClass, shell, NULL, 0);
#endif
	XtAddCallback(dino3d, XtNselectCallback,
		(XtCallbackProc) CallbackDino, (XtPointer) NULL);
	Initialize();
	XtRealizeWidget(topLevel);
#ifndef HAVE_MOTIF
	XtRealizeWidget(shell);
#endif
	XGrabButton(XtDisplay(dino2d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(dino2d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(dino2d),
		XCreateFontCursor(XtDisplay(dino2d), XC_crosshair));
	XGrabButton(XtDisplay(dino3d), (unsigned int) AnyButton, AnyModifier,
		XtWindow(dino3d), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(dino3d),
		XCreateFontCursor(XtDisplay(dino3d), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
