/*-
# MOTIF/X-BASED PYRAMINX(tm)
#
#  xpyraminx.c
#
###
#
#  Copyright (c) 1993 - 2004	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*-
  Version 7: 03/12/15 X/Windows
  Version 5: 95/10/04 Xt/Motif
  Version 4: 94/05/31 Xt
  Version 3: 93/04/01 Motif
  Version 2: 92/01/29 XView
  Version 1: 91/03/19 SunView
*/

#ifdef WINVER
#include "PyraminxP.h"
#include "wpyraminx.h"
#define TITLE "wpyraminx"

static PyraminxRec widget;
static HWND Wnd;

#ifndef SCOREPATH
#define SCOREPATH "c:\\Windows"
#endif
#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->pyraminx.started = b
#else
#include <stdlib.h>
#include <stdio.h>
#include <errno.h>
#ifdef VMS
#include <unixlib.h>
#define getlogin() cuserid(NULL)
#else
#if HAVE_UNISTD_H
#include <unistd.h>
#endif
#endif
#if HAVE_FCNTL_H
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#endif
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Shell.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/ToggleBG.h>
#ifdef MOUSEBITMAPS
#include "icons/mouse-l.xbm"
#include "icons/mouse-r.xbm"
#endif
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Pyraminx.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/pyraminx.xpm"
#endif
#include "icons/pyraminx.xbm"
#ifndef SCOREPATH
#define SCOREPATH "/var/games/xpuzzles"
#endif
#endif

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n"
	"[-{foreground|fg} {color}] [-{background|bg} {color}] "
	"[-{border|bd} {color}] [-face{0|1|2|3} {color}]\n"
	"[-{size {int} | sticky}] [-{mode {int} | both}] "
	"[-[no]orient] [-[no]practice] [-delay msecs]\n"
	"[-{font|fn} {fontname}] [-username {string}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"The original puzzle has 9 triangles per face (size = 3) "
	"and has period 3 turning (i.e. the face or points turn\n"
	"in 120 degree intervals).  The puzzle was designed by Uwe "
	"Meffert and called the Pyraminx.  This has 2^5*3^8*6!/2\n"
	"or 75,582,720 different comibinations.\n"
	"Another puzzle Senior Pyraminx 3x3x3 exists only on "
	"paper, it has period 2 turning (i.e.  edges turn with\n"
	"180 degree intervals) but the corners would fall off "
	"unless it had some tricky mechanism.  (This may be the\n"
	"same as the Master Pyraminx which has 446,965,972,992,000 "
	"different combinations).\n"
	"Another puzzle (which was not widely distributed), the "
	"Junior Pyraminx (and similarly the Junior Pyraminx Star,\n"
	"a octahedron formed by two tetrahedra, this has 7!*3^6 or "
	"3,674,160 different combinations), has 4 triangles\n"
	"(size = 2) per face.  This puzzle has been recently "
	"reissued by Meffert as Pyramorphix\n"
	"(http://www.mefferts-puzzles.com). At the time I designed "
	"this computer puzzle thought that it had only period 2\n"
	"turning (i.e the edges rotate).  It turns out the puzzle "
	"has a period 4 turning (edges turn with 90 degree\n"
	"intervals) which makes it analogous to the 2x2x2 Rubik's "
	"cube.  This puzzle makes various non-tetrahedral shapes.\n"
	"The puzzle contained here has no period 4 turning "
	"flexability.\n"
	"One is able to simulate Halpern's Tetrahedron or Pyraminx "
	"Tetrahedron (period 3 turning and sticky mode).   Also one\n"
	"is able to simulate one with variant turning (period 2 "
	"turning and sticky mode).\n"
};
static const char featuresHelp[] = {
	"Press \"mouse-left\" button to move a piece.  Release "
	"\"mouse-left\" button on a piece on the same face and\n"
	"in the same row (but not an adjacent piece or the move "
	"is ambiguous).  The pieces will then turn towards\n"
	"where the mouse button was released.\n"
	"Click \"mouse-center\" button, or press \"P\" or \"p\" "
	"keys, to toggle the practice mode (in practice mode the\n"
	"record should say \"practice\").  This is good for learning "
	"moves and experimenting.\n"
	"Click \"mouse-right\" button, or press \"R\" or \"r\" "
	"keys, to randomize the puzzle (this must be done first\n"
	"to set a new record).\n"
	"Press \"I\" or \"i\" keys to increase the number of \"facets\".\n"
	"Press \"D\" or \"d\" keys to decrease the number of \"facets\".\n"
	"Press \"O\" or \"o\" keys to toggle the orient mode.  One "
	"has to orient the faces in orient mode, besides\n"
	"getting all the faces to be the same color.  To do this "
	"one has to get the lines to be oriented in\n"
	"the same direction, this only matters with center "
	"\"facets\", if at all (i.e. those \"cubies\" not on a\n"
	"corner or edge).  This does add complexity so there are "
	"2 sets of records.\n"
	"Press \"2\", \"3\", \"B\", or \"b\" keys (not the keypad 2, 3) "
	"to change modes to Period 2, Period 3, or Both.\n"
	"Press \"Y\" or \"y\" keys to toggle sticky mode (increase/decrease "
	"is disabled here if sticky mode is on).\n"
	"\"Sticky\" and \"Period 2\" turning allows only the edges to "
	"turn, and the 2 center rows turn together. It is as if\n"
	"the middle cut of the three cuts did not exist.\n"
	"\"Sticky\" and \"Period 3\" turning allows only the faces to "
	"turn, it is as if the middle cut of the three cuts\n"
	"did not exist.\n"
	"Beware, the \"Sticky\" mode is a hack and much could be done "
	"to improve its look.\n"
	"Press \"S\" or \"s\" keys to start auto-solver.  Only works on "
	"1x1x1, 2x2x2, and 3x3x3 pyrmaminxs in Period 3 mode.\n"
	"Press \"U\" or \"u\" keys to undo a move.\n"
	"Press \"G\" or \"g\" keys to get a saved puzzle.\n"
	"Press \"W\" or \"w\" keys to save a puzzle.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"q\", or \"CTRL-C\" keys to kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to move without "
	"mouse clicks.\n"
	"Key pad is defined for Pyraminx as:\n"
	"  /    Counterclockwise\n"
	"  8 9  Up, Upper Right\n"
	"  ^\n"
	"4<5>6  Left, Clockwise, Right\n"
	"  v\n"
	"1 2    Lower Left, Down\n"
	"Use the shift keys to access \"Period 3\" turns from "
	"\"Both\" mode, otherwise it assumes \"Period 2\" turning.\n"
	"Faces and points turn in \"Period 3\" and edges (2 points) "
	"turn in \"Period 2\".\n"
	"Use the control key and the left mouse button, keypad, or "
	"arrow keys to move the whole tetrahedron.  This is not\n"
	"recorded as a turn.\n"
};
static const char referencesHelp[] = {
	"James G. Nourse, The Simple Solutions to Cubic Puzzles, "
	"Bantam Books, New York, 1981. pp 8-15.\n"
	"Tom Werneck, Mastering the Magic Pyramid, Evans Brothers "
	"Limited, London, 1981. pp 109-111.\n"
	"Douglas R. Hofstadter, Beyond Rubik's Cube: spheres, pyramids, "
	"dodecahedrons and God knows what else, Scientific American,\n"
	"July 1982, pp 16-31.\n"
	"John Ewing & Czes Kosniowski, Puzzle it Out: Cubes Groups and "
	"Puzzles, Cambridge University Press, New York, 1982, pp 60-61.\n"
	"Magic Cubes 1996 Catalog of Dr. Christoph Bandelow.\n"
};
#endif
static const char solveHelp[] = {
	"Auto-solver: sorry, only implemented for size < 4 where "
	"period = 3.\n"
};

#ifndef SCOREFILE
#define SCOREFILE "pyraminx.scores"
#endif

#define MAXFACETS 7
#define NEVER -1
#define FILENAMELEN 1024
#define USERNAMELEN 120
#define MESSAGELEN (USERNAMELEN+64)
#define TITLELEN 2048
#define NOACCESS "noaccess"
#define NOBODY "nobody"

typedef struct {
	int score;
	char name[USERNAMELEN];
} GameRecord;

static GameRecord pyraminxRecord[MAXMODES][2][MAXFACETS - MINFACETS + 2];
static int movesDsp = 0;
static char messageDsp[MESSAGELEN] = "Welcome";
static char recordDsp[MESSAGELEN] = "NOT RECORDED";
static int oldSize;

#ifdef HAVE_MOTIF
#define MINSPEED 1
#define MAXSPEED 50
#define MULTSPEED 20
static Widget moves, record, message, modes[MAXMODES], orientSwitch,
	stickySwitch, practiceSwitch, facets, speed;
static char buff[21];
static const char *modeString[] =
{
	"Period 2", "Period 3", "Both"
};
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
static Widget solveDialog, practiceDialog, randomizeDialog;
#else
static char titleDsp[TITLELEN] = "";
#endif
#ifdef WINVER
#define MAXPROGNAME 80
static char progDsp[MAXPROGNAME] = TITLE;
static char usernameDsp[USERNAMELEN] = "Guest";
#else
static Pixmap pyraminxIcon = None;
static Widget topLevel, pyraminx;
static Arg arg[5];
static char *progDsp;
static char usernameDsp[USERNAMELEN] = "";
#endif

#ifdef HAVE_MOTIF
static void
PrintState(Widget w, char *msg)
{
	XmString xmstr;

	if (!XtIsSubclass(w, xmLabelWidgetClass))
		XtError("PrintState() requires a Label Widget");
	xmstr = XmStringCreateLtoR(msg, XmSTRING_DEFAULT_CHARSET);
	XtSetArg(arg[0], XmNlabelString, xmstr);
	XtSetValues(w, arg, 1);
}
#else
static void
PrintState(
#ifndef WINVER
Widget w,
#endif
int mode, int size, Boolean sticky, int moves, char *msg)
{
	char ss[10], mb[10];

	if (sticky)
		(void) strcpy(ss, "sticky");
	else
		(void) sprintf(ss, "%d", size);
	if (mode == BOTH)
		(void) strcpy(mb, "both");
	else
		(void) sprintf(mb, "%d", mode);
	(void) sprintf(titleDsp, "%s.%s: %s@ (%d/%s) - %s",
		progDsp, mb, ss, moves, recordDsp, msg);
#ifdef WINVER
	SetWindowText(Wnd, (LPSTR) titleDsp);
#else
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
#endif
}
#endif

static void
InitRecords(void)
{
	int i, mode, orient;

	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAXFACETS - MINFACETS + 1; i++) {
				pyraminxRecord[mode][orient][i].score = NEVER;
				(void) strncpy(pyraminxRecord[mode][orient][i].name,
					NOACCESS, USERNAMELEN);
			}
}

static void
ReadRecords(void)
{
	FILE *fp;
	int i, n, mode, orient;
	char username[USERNAMELEN];
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
        fname = buf2;
	name = fname;
	if ((fp = fopen(name, "r")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	free(lname);
	free(fname);
	for (mode = 0; mode < MAXMODES; mode++)
		for (orient = 0; orient < 2; orient++)
			for (i = 0; i <= MAXFACETS - MINFACETS + 1; i++) {
				(void) fscanf(fp, "%d %s\n", &n, username);
				if (n <= pyraminxRecord[mode][orient][i].score ||
				    pyraminxRecord[mode][orient][i].score <= NEVER) {
					pyraminxRecord[mode][orient][i].score = n;
					(void) strncpy(pyraminxRecord[mode][orient][i].name,
						username, USERNAMELEN);
				}
			}
	(void) fclose(fp);
}

static void
WriteRecords(void)
{
	FILE *fp;
	int i, mode, orient;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, SCOREFILE);
	lname = buf1;
	stringCat(&buf1, SCOREPATH, FINALDELIM);
	stringCat(&buf2, buf1, SCOREFILE);
	free(buf1);
	fname = buf2;
	name = fname;
	if ((fp = fopen(name, "w")) == NULL) {
		/* Try current directory in case its not installed yet. */
		name = lname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", fname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, lname);
			free(buf2);
			PRINT_MESSAGE(buf1);
			free(buf1);
		}
#endif
	}
	{
#if HAVE_FCNTL_H
		int lfd;
		char lockfile[FILENAMELEN];

		(void) strncpy(lockfile, name, FILENAMELEN - 6);
		(void) strcat(lockfile, ".lock");
		while (((lfd = open(lockfile, O_CREAT | O_EXCL, 0644)) < 0) &&
				errno == EEXIST)
			(void) sleep(1);
		if (lfd < 0) {
#if 1
			(void) fprintf(stderr,
			  "Lock file exists... guessing its an old one.\n");
#else
			(void) fprintf(stderr,
			  "Lock file exists... score not recorded - sorry.\n");
			return;
#endif
		}
#endif
		for (mode = 0; mode < MAXMODES; mode++) {
			for (orient = 0; orient < 2; orient++) {
				for (i = 0; i <= MAXFACETS - MINFACETS + 1; i++)
					(void) fprintf(fp, "%d %s\n",
					pyraminxRecord[mode][orient][i].score,
					pyraminxRecord[mode][orient][i].name);
				(void) fprintf(fp, "\n");
			}
			(void) fprintf(fp, "\n");
		}
#if HAVE_FCNTL_H
		(void) close(lfd);
		(void) unlink(lockfile);
#endif
		(void) fclose(fp);
	}
	free(lname);
	free(fname);
}

static void
PrintRecord(int size, int mode, Boolean orient, Boolean sticky, Boolean practice)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;
	int k = (sticky) ? MAXFACETS - MINFACETS + 1 : size - MINFACETS;

	if (practice)
		(void) strncpy(recordDsp, "practice", MESSAGELEN);
	else if (!sticky && size > MAXFACETS)
		(void) strncpy(recordDsp, "NOT RECORDED", MESSAGELEN);
	else if (pyraminxRecord[i][j][k].score <= NEVER) {
		(void) sprintf(recordDsp, "NEVER %s", NOACCESS);
	} else {
		(void) sprintf(recordDsp, "%d %s",
			pyraminxRecord[i][j][k].score,
			pyraminxRecord[i][j][k].name);
	}
#ifdef HAVE_MOTIF
	PrintState(record, recordDsp);
#endif
}

static Boolean
HandleSolved(int counter, int size, int mode, Boolean orient, Boolean sticky)
{
	int i = mode - PERIOD2;
	int j = (orient) ? 1 : 0;
	int k = (sticky) ? MAXFACETS - MINFACETS + 1 : size - MINFACETS;

	if ((sticky || size <= MAXFACETS) &&
	    (counter < pyraminxRecord[i][j][k].score ||
	    pyraminxRecord[i][j][k].score <= NEVER)) {
		ReadRecords();	/* Maybe its been updated by another */
		pyraminxRecord[i][j][k].score = counter;
		(void) strncpy(pyraminxRecord[i][j][k].name, usernameDsp,
			USERNAMELEN);
		if (size < 4 || mode == PERIOD2 || (orient &&
			       (counter < pyraminxRecord[i][!j][k].score ||
			       pyraminxRecord[i][!j][k].score <= NEVER))) {
			pyraminxRecord[i][!j][k].score = counter;
			(void) strncpy(pyraminxRecord[i][!j][k].name, usernameDsp,
				USERNAMELEN);
		}
		WriteRecords();
		PrintRecord(size, mode, orient, sticky, False);
		return TRUE;
	}
	return False;
}

static void
Initialize(
#ifdef WINVER
PyraminxWidget w, HBRUSH brush
#else
Widget w
#endif
)
{
	int size, mode;
	Boolean orient, sticky, practice;
	char *username;

#ifdef WINVER
	InitializePyraminx(w, brush);

	size = w->pyraminx.size;
	mode = w->pyraminx.mode;
	orient = w->pyraminx.orient;
	sticky = w->pyraminx.sticky;
	practice = w->pyraminx.practice;
	username = w->pyraminx.username;
#else
	int delay;

	XtVaGetValues(w,
		XtNuserName, &username,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice,
		XtNdelay, &delay, NULL);
#ifdef HAVE_MOTIF
	if (size > MAXFACETS)
		XtVaSetValues(facets, XmNmaximum, size, NULL);
	XmScaleSetValue(facets, size);
	XmToggleButtonSetState(modes[mode - PERIOD2], True, False);
	XmToggleButtonSetState(orientSwitch, orient, True);
	XmToggleButtonSetState(stickySwitch, sticky, True);
	XmToggleButtonSetState(practiceSwitch, practice, True);
	XmScaleSetValue(speed, MAXSPEED + MINSPEED - delay / MULTSPEED - 1);
#endif
#endif
	SET_STARTED(w, False);
	InitRecords();
	ReadRecords();
#ifndef WINVER
	(void) strncpy(usernameDsp, username, USERNAMELEN);
#endif
	if (!strcmp(username, "") || !strcmp(username, "(null)") ||
	    !strcmp(username, NOACCESS) || !strcmp(username, NOBODY)) {
#ifdef WINVER
		(void) strncpy(usernameDsp, username, USERNAMELEN);
#else
		/* The NOACCESS is not necessary, but it may stop silliness. */
		(void) sprintf(usernameDsp, "%s", getlogin());
		if (!strcmp(usernameDsp, "") ||
				!strcmp(usernameDsp, "(null)") ||
				!strcmp(usernameDsp, NOACCESS) ||
				!strcmp(usernameDsp, NOBODY))
			/* It really IS nobody */
			(void) sprintf(usernameDsp, "%s", "guest");
#endif
	}
	PrintRecord(size, mode, orient, sticky, practice);
	oldSize = size;
#ifndef HAVE_MOTIF
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		mode, size, sticky, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
void
SetPyraminx(PyraminxWidget w, int reason)
#else
static void
CallbackPyraminx(Widget w, caddr_t clientData,
		pyraminxCallbackStruct * callData)
#endif
{
#ifndef WINVER
	int reason = callData->reason;
#endif
	int size, mode;
	Boolean orient, sticky, practice, start, cheat;

	(void) strcpy(messageDsp, "");
#ifdef WINVER
	size = w->pyraminx.size;
	mode = w->pyraminx.mode;
	orient = w->pyraminx.orient;
	sticky = w->pyraminx.sticky;
	practice = w->pyraminx.practice;
	cheat = w->pyraminx.cheat;
	start = w->pyraminx.started;
#else
	XtVaGetValues(w,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice,
		XtNstart, &start,
		XtNcheat, &cheat, NULL);
#endif
	switch (reason) {
		case PYRAMINX_HIDE:
#ifdef WINVER
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
#else
			(void) XIconifyWindow(XtDisplay(topLevel),
				XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
#endif
			break;
#ifndef WINVER
		case PYRAMINX_PRACTICE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(practiceDialog);
#else
			XtSetArg(arg[0], XtNmenu, 11); /* menu choice */
			XtSetValues(pyraminx, arg, 1);
#endif
			break;
		case PYRAMINX_RANDOMIZE_QUERY:
#ifdef HAVE_MOTIF
			XtManageChild(randomizeDialog);
#else
			XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
			XtSetValues(pyraminx, arg, 1);
#endif
			break;
#endif
		case PYRAMINX_SOLVE_MESSAGE:
#ifdef WINVER
			(void) MessageBox(w->core.hWnd, solveHelp,
				"Auto-Solve", MB_OK);
#else
#ifdef HAVE_MOTIF
			XtManageChild(solveDialog);
#else
			(void) strncpy(messageDsp, solveHelp, MESSAGELEN);
#endif
#endif
			break;
		case PYRAMINX_RESTORE:
			if (practice) {
				(void) strncpy(recordDsp, "practice", MESSAGELEN);
#ifdef HAVE_MOTIF
				PrintState(record, recordDsp);
#endif
			}
			movesDsp = 0;
			break;
		case PYRAMINX_RESET:
			movesDsp = 0;
			break;
		case PYRAMINX_AMBIGUOUS:
			(void) strncpy(messageDsp, "Ambiguous move", MESSAGELEN);
			break;
		case PYRAMINX_ILLEGAL:
			if (practice || start)
				(void) strncpy(messageDsp, "Illegal move",
					MESSAGELEN);
			else
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			break;
		case PYRAMINX_MOVED:
			movesDsp++;
			SET_STARTED(w, True);
			break;
		case PYRAMINX_CONTROL:
			return;
		case PYRAMINX_SOLVED:
			if (practice)
				movesDsp = 0;
			else if (cheat)
				(void) sprintf(messageDsp,
					"No cheating %s!!", usernameDsp);
			else if (HandleSolved(movesDsp, size, mode, orient, sticky))
				(void) sprintf(messageDsp,
					"Congratulations %s!!", usernameDsp);
			else
				(void) strncpy(messageDsp, "Solved!", MESSAGELEN);
			SET_STARTED(w, False);
			break;
		case PYRAMINX_PRACTICE:
			movesDsp = 0;
			practice = !practice;
			if (!practice)
				(void) strncpy(messageDsp, "Randomize to start",
					MESSAGELEN);
			PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->pyraminx.practice = practice;
			w->pyraminx.started = False;
#else
			XtSetArg(arg[0], XtNpractice, practice);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(practiceSwitch, practice, True);
#endif
#endif
			break;
		case PYRAMINX_RANDOMIZE:
			movesDsp = 0;
#ifdef WINVER
			w->pyraminx.practice = practice;
			w->pyraminx.started = False;
#else
			XtSetArg(arg[0], XtNpractice, False);
			XtSetArg(arg[1], XtNstart, False);
			XtSetValues(w, arg, 2);
#endif
			break;
		case PYRAMINX_DEC:
			if (!sticky) {
				movesDsp = 0;
				size--;
				oldSize = size;
				PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
				w->pyraminx.size = size;
#else
				XtSetArg(arg[0], XtNsize, size);
				XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
				XmScaleSetValue(facets, size);
				if (size >= MAXFACETS)
					XtVaSetValues(facets, XmNmaximum, size, NULL);
#endif
#endif
			}
			break;
		case PYRAMINX_ORIENT:
			movesDsp = 0;
			orient = !orient;
			PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->pyraminx.orient = orient;
#else
			XtSetArg(arg[0], XtNorient, orient);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(orientSwitch, orient, True);
#endif
#endif
			break;
		case PYRAMINX_INC:
			if (!sticky) {
				movesDsp = 0;
				size++;
				oldSize = size;
				PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
				w->pyraminx.size = size;
#else
				XtSetArg(arg[0], XtNsize, size);
				XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
				if (size > MAXFACETS)
					XtVaSetValues(facets, XmNmaximum, size, NULL);
				XmScaleSetValue(facets, size);
#endif
#endif
			}
			break;
		case PYRAMINX_PERIOD2:
		case PYRAMINX_PERIOD3:
		case PYRAMINX_BOTH:
			movesDsp = 0;
			mode = reason - PYRAMINX_PERIOD2 + PERIOD2;
			PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->pyraminx.mode = mode;
#else
			XtSetArg(arg[0], XtNmode, mode);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(modes[mode - PERIOD2], True, True);
#endif
#endif
			break;
		case PYRAMINX_STICKY:
			movesDsp = 0;
			sticky = !sticky;
			if (sticky)
				size = 4;
			else
				size = oldSize;
			PrintRecord(size, mode, orient, sticky, practice);
#ifdef WINVER
			w->pyraminx.sticky = sticky;
			w->pyraminx.size = size;
#else
			XtSetArg(arg[0], XtNsticky, sticky);
			XtSetArg(arg[1], XtNsize, size);
			XtSetValues(w, arg, 2);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(stickySwitch, sticky, True);
			XmScaleSetValue(facets, size);
#endif
#endif
			break;
		case PYRAMINX_COMPUTED:
			SET_STARTED(w, False);
			break;
		case PYRAMINX_UNDO:
			movesDsp--;
			SET_STARTED(w, True);
			break;
	}
#ifdef HAVE_MOTIF
	PrintState(message, messageDsp);
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
#else
	PrintState(
#ifndef WINVER
		XtParent(w),
#endif
		mode, size, sticky, movesDsp, messageDsp);
#endif
}

#ifdef WINVER
static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	PAINTSTRUCT paint;
	int shift = 0;

	Wnd = widget.core.hWnd = hWnd;
	if (GetFocus()) {
		if (!widget.pyraminx.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			EnterPyraminx(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	} else {
		if (widget.pyraminx.focus) {
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			LeavePyraminx(&widget);
			(void) EndPaint(hWnd, &paint);
		}
	}
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyPyraminx(brush);
			break;
		case WM_SIZE:
			ResizePyraminx(&widget);
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ExposePyraminx(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_RBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			RandomizePyraminx(&widget);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONDOWN:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectPyraminx(&widget, LOWORD(lParam), HIWORD(lParam),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleasePyraminx(&widget, LOWORD(lParam), HIWORD(lParam),
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_GET:
					GetPyraminx(&widget);
					ResizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_WRITE:
					WritePyraminx(&widget);
					break;
				case IDM_EXIT:
					DestroyPyraminx(brush);
					break;
				case IDM_HIDE:
					HidePyraminx(&widget);
					break;
				case IDM_CLEAR:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ClearPyraminx(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_UNDO:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					UndoPyraminx(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_RANDOMIZE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					RandomizePyraminx(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_PRACTICE:
					PracticePyraminx(&widget);
					SizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SOLVE:
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					SolvePyraminx(&widget);
					(void) ReleaseDC(hWnd, widget.core.hDC);
					break;
				case IDM_ORIENT:
					OrientizePyraminx(&widget);
					SizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_STICKY:
					(void) StickyModePyraminx(&widget);
					SizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_SHIFT_TOP:
				case IDM_SHIFT_TR:
				case IDM_SHIFT_RIGHT:
				case IDM_SHIFT_BOTTOM:
				case IDM_SHIFT_BL:
				case IDM_SHIFT_LEFT:
				case IDM_SHIFT_CW:
				case IDM_SHIFT_CCW:
					shift = 1;
				case IDM_TOP:
				case IDM_TR:
				case IDM_RIGHT:
				case IDM_BOTTOM:
				case IDM_BL:
				case IDM_LEFT:
				case IDM_CW:
				case IDM_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						shift = shift || (GetKeyState(VK_CAPITAL) & 1);
						(void) MovePyraminxInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFT_TOP : IDM_TOP),
							shift, FALSE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_SHIFTCONTROL_TOP:
				case IDM_SHIFTCONTROL_TR:
				case IDM_SHIFTCONTROL_RIGHT:
				case IDM_SHIFTCONTROL_BOTTOM:
				case IDM_SHIFTCONTROL_BL:
				case IDM_SHIFTCONTROL_LEFT:
				case IDM_SHIFTCONTROL_CW:
				case IDM_SHIFTCONTROL_CCW:
					shift = 1;
				case IDM_CONTROL_TOP:
				case IDM_CONTROL_TR:
				case IDM_CONTROL_RIGHT:
				case IDM_CONTROL_BOTTOM:
				case IDM_CONTROL_BL:
				case IDM_CONTROL_LEFT:
				case IDM_CONTROL_CW:
				case IDM_CONTROL_CCW:
					{
						POINT cursor, origin;

						widget.core.hDC = GetDC(hWnd);
						(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
						origin.x = 0, origin.y = 0;
						ClientToScreen(hWnd, &origin);
						(void) GetCursorPos(&cursor);
						(void) MovePyraminxInput(&widget,
							cursor.x - origin.x, cursor.y - origin.y,
							(int) LOWORD(wParam) - ((shift) ? IDM_SHIFTCONTROL_TOP : IDM_CONTROL_TOP),
							shift, TRUE);
						(void) ReleaseDC(hWnd, widget.core.hDC);
					}

					break;
				case IDM_DEC:
					if (DecrementPyraminx(&widget)) {
						SizePyraminx(&widget);
						(void) InvalidateRect(hWnd, NULL, TRUE);
					}
					break;
				case IDM_INC:
					IncrementPyraminx(&widget);
					SizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PERIOD2:
				case IDM_PERIOD3:
				case IDM_BOTH:
					PeriodModePyraminx(&widget, (int) LOWORD(wParam) - IDM_PERIOD2);
					SizePyraminx(&widget);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
					 	"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
					 	"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
					 	"References", MB_OK);
					break;
			}
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]] [-[no]mono] [-[no]{reverse|rv}]\n");
	(void) fprintf(stderr,
		"\t[-{foreground|fg} {color}] [-{background|bg} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{border|bd} {color}] [-face{0|1|2|3} {color}]\n");
	(void) fprintf(stderr,
		"\t[-{size {int} | sticky}] [-{mode {int} | both}]\n");
	(void) fprintf(stderr,
		"\t[-[no]orient] [-[no]practice] [-delay msecs]\n");
	(void) fprintf(stderr,
		"\t[-{font|fn} {fontname}] [-username {string}]\n");
	exit(1);
}

static XrmOptionDescRec options[] =
{
	{(char *) "-mono", (char *) "*pyraminx.mono", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nomono", (char *) "*pyraminx.mono", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-rv", (char *) "*pyraminx.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-reverse", (char *) "*pyraminx.reverse", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-norv", (char *) "*pyraminx.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-noreverse", (char *) "*pyraminx.reverse", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-fg", (char *) "*pyraminx.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*pyraminx.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*pyraminx.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-border", (char *) "*pyraminx.pieceBorder", XrmoptionSepArg, NULL},
	{(char *) "-face0", (char *) "*pyraminx.faceColor0", XrmoptionSepArg, NULL},
	{(char *) "-face1", (char *) "*pyraminx.faceColor1", XrmoptionSepArg, NULL},
	{(char *) "-face2", (char *) "*pyraminx.faceColor2", XrmoptionSepArg, NULL},
	{(char *) "-face3", (char *) "*pyraminx.faceColor3", XrmoptionSepArg, NULL},
	{(char *) "-size", (char *) "*pyraminx.size", XrmoptionSepArg, NULL},
	{(char *) "-sticky", (char *) "*pyraminx.sticky", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-mode", (char *) "*pyraminx.mode", XrmoptionSepArg, NULL},
	{(char *) "-both", (char *) "*pyraminx.mode", XrmoptionNoArg, (char *) "4"},
	{(char *) "-orient", (char *) "*pyraminx.orient", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-noorient", (char *) "*pyraminx.orient", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-practice", (char *) "*pyraminx.practice", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nopractice", (char *) "*pyraminx.practice", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-delay", (char *) "*pyraminx.delay", XrmoptionSepArg, NULL},
	{(char *) "-fn", (char *) "*pyraminx.font", XrmoptionSepArg, NULL},
	{(char *) "-font", (char *) "*pyraminx.font", XrmoptionSepArg, NULL},
	{(char *) "-username", (char *) "*pyraminx.userName", XrmoptionSepArg, NULL}
};

#ifdef HAVE_MOTIF
static void
CallbackPyraminxPractice(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 11); /* menu choice */
		XtSetValues(pyraminx, arg, 1);
	}
}

static void
CallbackPyraminxRandomize(Widget w, XtPointer clientData, XmAnyCallbackStruct * cbs)
{
	if (cbs->reason == XmCR_OK) {
		XtSetArg(arg[0], XtNmenu, 5); /* menu choice */
		XtSetValues(pyraminx, arg, 1);
	}
}
static void
FacetSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int size = cbs->value, mode;
	Boolean orient, sticky, practice;

	XtVaGetValues(pyraminx,
		XtNsize, &oldSize,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky,
		XtNpractice, &practice, NULL);
	if (sticky)
		XmScaleSetValue(w, oldSize);
	else if (oldSize != size) {
		XtVaSetValues(pyraminx,
			XtNsize, size, NULL);
		oldSize = size;
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(size, mode, orient, sticky, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
ModeToggle(Widget w, int mode, XmToggleButtonCallbackStruct * cbs)
{
	int size;
	Boolean orient, sticky, practice;

	if (cbs->set) {
		XtVaGetValues(pyraminx,
			XtNsize, &size,
			XtNorient, &orient,
			XtNsticky, &sticky,
			XtNpractice, &practice, NULL);
		XtVaSetValues(pyraminx,
			XtNmode, mode + PERIOD2, NULL);
		movesDsp = 0;
		(void) sprintf(buff, "%d", movesDsp);
		PrintState(moves, buff);
		PrintRecord(size, mode + PERIOD2, orient, sticky, practice);
		(void) strcpy(messageDsp, "");
		PrintState(message, messageDsp);
	}
}

static void
OrientToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int size, mode;
	Boolean orient = cbs->set, sticky, practice;

	XtVaGetValues(pyraminx,
		XtNsize, &size,
		XtNmode, &mode,
		XtNsticky, &sticky,
		XtNpractice, &practice, NULL);
	XtVaSetValues(pyraminx,
		XtNorient, orient, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(size, mode, orient, sticky, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
StickyToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int size, mode;
	Boolean orient, sticky = cbs->set, practice;

	XtVaGetValues(pyraminx,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNpractice, &practice, NULL);
	XtVaSetValues(pyraminx,
		XtNsticky, sticky, NULL);
	if ((sticky && size != 4) || (!sticky && oldSize != 4)) {
		if (sticky)
			size = 4;
		else
			size = oldSize;
		if (size <= MAXFACETS)
			XmScaleSetValue(facets, size);
		XtVaSetValues(pyraminx,
			XtNsize, size, NULL);
	}
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(size, mode, orient, sticky, practice);
	(void) strcpy(messageDsp, "");
	PrintState(message, messageDsp);
}

static void
PracticeToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct * cbs)
{
	int  size, mode;
	Boolean orient, sticky, practice = cbs->set;

	XtVaSetValues(pyraminx,
		XtNpractice, practice,
		XtNstart, False, NULL);
	XtVaGetValues(pyraminx,
		XtNsize, &size,
		XtNmode, &mode,
		XtNorient, &orient,
		XtNsticky, &sticky, NULL);
	movesDsp = 0;
	(void) sprintf(buff, "%d", movesDsp);
	PrintState(moves, buff);
	PrintRecord(size, mode, orient, sticky, practice);
	if (practice)
		(void) strcpy(messageDsp, "");
	else
		(void) strncpy(messageDsp, "Randomize to start", MESSAGELEN);
	PrintState(message, messageDsp);
}

static void
SpeedSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct * cbs)
{
	int delay = MULTSPEED * (MAXSPEED + MINSPEED - cbs->value - 1), oldDelay;

	XtVaGetValues(pyraminx,
		XtNdelay, &oldDelay, NULL);
	if (oldDelay != delay) {
		XtVaSetValues(pyraminx, XtNdelay, delay, NULL);
	}
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;

	if (val == 2)
		exit(0);
	XtSetArg(arg[0], XtNmenu, val);
	XtSetValues(pyraminx, arg, 1);
}

static void
playCB(Widget w, void *value, void *clientData)
{
	int val = (int) value;
	XtSetArg(arg[0], XtNmenu, val + 3); /* GWQ */
	XtSetValues(pyraminx, arg, 1);
}

static Widget
createQuery(Widget w, char *text, char *title, XtCallbackProc callback)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNmessageString, messageString);
	messageBox = XmCreateWarningDialog(w, (char *) "queryBox",
		arg, 2);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(messageString);
	XtAddCallback(messageBox, XmNokCallback, callback, (XtPointer) NULL);
	XtAddCallback(messageBox, XmNcancelCallback, callback,
		(XtPointer) NULL);
	return messageBox;
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}
static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d\n", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif
int
main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget menuBarPanel, mainPanel, controlPanel;
	Widget movesRowCol, facetsRowCol, labelRowCol, radioRowCol;
	Widget switchRowCol, messageRowCol;
	XmString fileString, playString;
	XmString getString, writeString, quitString;
	XmString clearString, undoString, randomizeString, solveString;
	XmString incrementString, decrementString;
	XmString orientizeString, stickyString, practiceString;
	unsigned int i;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Pyraminx",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) pyraminx_xpm, &pyraminxIcon, NULL,
			&xpmAttributes);
	}
	if (pyraminxIcon == (Pixmap) NULL)
#endif
		pyraminxIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) pyraminx_bits,
			pyraminx_width, pyraminx_height);
	XtSetArg(arg[0], XtNiconPixmap, pyraminxIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	menuBarPanel = XtVaCreateManagedWidget("menuBarPanel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	playString = XmStringCreateSimple((char *) "Play");
	menuBar = XmVaCreateSimpleMenuBar(menuBarPanel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		XmVaCASCADEBUTTON, playString, 'P',
		NULL);
	XmStringFree(fileString);
	XmStringFree(playString);
	getString = XmStringCreateSimple((char *) "Get");
	writeString = XmStringCreateSimple((char *) "Write");
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, getString, 'G', NULL, NULL,
		XmVaPUSHBUTTON, writeString, 'W', NULL, NULL,
		XmVaSEPARATOR,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(getString);
	XmStringFree(writeString);
	XmStringFree(quitString);
	clearString = XmStringCreateSimple((char *) "Clear");
	undoString = XmStringCreateSimple((char *) "Undo");
	randomizeString = XmStringCreateSimple((char *) "Randomize");
	solveString = XmStringCreateSimple((char *) "Solve");
	incrementString = XmStringCreateSimple((char *) "Increment");
	decrementString = XmStringCreateSimple((char *) "Decrement");
	orientizeString = XmStringCreateSimple((char *) "Orientize");
	stickyString = XmStringCreateSimple((char *) "StickY");
	practiceString = XmStringCreateSimple((char *) "Practice");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "play_menu", 1, playCB,
		XmVaPUSHBUTTON, clearString, 'C', NULL, NULL,
		XmVaPUSHBUTTON, undoString, 'U', NULL, NULL,
		XmVaPUSHBUTTON, randomizeString, 'R', NULL, NULL,
		XmVaPUSHBUTTON, solveString, 'S', NULL, NULL,
		XmVaPUSHBUTTON, incrementString, 'I', NULL, NULL,
		XmVaPUSHBUTTON, decrementString, 'D', NULL, NULL,
		XmVaPUSHBUTTON, orientizeString, 'O', NULL, NULL,
		XmVaPUSHBUTTON, stickyString, 'Y', NULL, NULL,
		XmVaPUSHBUTTON, practiceString, 'P', NULL, NULL,
		NULL);
	XmStringFree(clearString);
	XmStringFree(undoString);
	XmStringFree(randomizeString);
	XmStringFree(solveString);
	XmStringFree(incrementString);
	XmStringFree(decrementString);
	XmStringFree(orientizeString);
	XmStringFree(stickyString);
	XmStringFree(practiceString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	solveDialog = createHelp(menuBar, (char *) solveHelp,
		(char *) "Solve");
	practiceDialog = createQuery(topLevel,
		(char *) "Are you sure you want to toggle the practice mode?",
		(char *) "Practice Query",
		(XtCallbackProc) CallbackPyraminxPractice);
	randomizeDialog = createQuery(topLevel,
		(char *) "Are you sure you want to randomize?",
		(char *) "Randomize Query",
		(XtCallbackProc) CallbackPyraminxRandomize);
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, menuBarPanel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	movesRowCol = XtVaCreateManagedWidget("movesRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 2,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
#ifdef MOUSEBITMAPS
	{
		/* Takes up valuable real estate. */
		Pixmap mouseLeftCursor, mouseRightCursor;
		Pixel fg, bg;

		(void) XtVaGetValues(movesRowCol,
			XmNforeground, &fg,
			XmNbackground, &bg, NULL);
		mouseLeftCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_left_bits,
			mouse_left_width, mouse_left_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		mouseRightCursor = XCreatePixmapFromBitmapData(
			XtDisplay(movesRowCol),
			RootWindowOfScreen(XtScreen(movesRowCol)),
			(char *) mouse_right_bits,
			mouse_right_width, mouse_right_height, fg, bg,
			DefaultDepthOfScreen(XtScreen(movesRowCol)));
		(void) XtVaCreateManagedWidget("mouseLeftText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Move", 5, NULL);
		(void) XtVaCreateManagedWidget("mouseLeft",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseLeftCursor, NULL);
		(void) XtVaCreateManagedWidget("mouseRightText",
			xmLabelGadgetClass, movesRowCol,
			XtVaTypedArg, XmNlabelString,
			XmRString, "Randomize", 10, NULL);
		(void) XtVaCreateManagedWidget("mouseRight",
			xmLabelGadgetClass, movesRowCol,
			XmNlabelType, XmPIXMAP,
			XmNlabelPixmap, mouseRightCursor, NULL);
	}
#endif
	(void) XtVaCreateManagedWidget("movesText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Moves", 6, NULL);
	moves = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);
	(void) XtVaCreateManagedWidget("recordText",
		xmLabelGadgetClass, movesRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Record", 7, NULL);
	record = XtVaCreateManagedWidget("0",
		xmLabelWidgetClass, movesRowCol, NULL);

	facetsRowCol = XtVaCreateManagedWidget("facetsRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	facets = XtVaCreateManagedWidget("facets",
		xmScaleWidgetClass, facetsRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Size", 5,
		XmNminimum, MINFACETS,
		XmNmaximum, MAXFACETS,
		XmNvalue, MINFACETS,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(facets, XmNvalueChangedCallback,
		(XtCallbackProc) FacetSlider, (XtPointer) NULL);
	speed = XtVaCreateManagedWidget("speed",
		xmScaleWidgetClass, facetsRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Animation Speed", 16,
		XmNminimum, MINSPEED,
		XmNmaximum, MAXSPEED,
		XmNvalue, MAXSPEED - 10,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(speed, XmNvalueChangedCallback,
		(XtCallbackProc) SpeedSlider, (XtPointer) NULL);
	labelRowCol = XtVaCreateManagedWidget("labelRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);
	(void) XtVaCreateManagedWidget("PeriodText",
		xmLabelWidgetClass, labelRowCol,
		XtVaTypedArg, XmNlabelString, XmRString, "Periods:", 9, NULL);
	radioRowCol = XtVaCreateManagedWidget("radioRowCol",
		xmRowColumnWidgetClass, labelRowCol,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN,
		XmNradioBehavior, True, NULL);
	for (i = 0; i < XtNumber(modeString); i++) {
		modes[i] = XtVaCreateManagedWidget(modeString[i],
			xmToggleButtonGadgetClass, radioRowCol,
			XmNradioBehavior, True, NULL);
		XtAddCallback(modes[i], XmNvalueChangedCallback,
			(XtCallbackProc) ModeToggle, (XtPointer) i);
	}
	switchRowCol = XtVaCreateManagedWidget("switchRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	orientSwitch = XtVaCreateManagedWidget("Oriented",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(orientSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) OrientToggle, (XtPointer) NULL);
	stickySwitch = XtVaCreateManagedWidget("Sticky",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(stickySwitch, XmNvalueChangedCallback,
		(XtCallbackProc) StickyToggle, (XtPointer) NULL);
	practiceSwitch = XtVaCreateManagedWidget("Practice",
		xmToggleButtonWidgetClass, switchRowCol, NULL);
	XtAddCallback(practiceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) PracticeToggle, (XtPointer) NULL);
	messageRowCol = XtVaCreateManagedWidget("messageRowCol",
		xmRowColumnWidgetClass, controlPanel, NULL);

	message = XtVaCreateManagedWidget("Play Pyraminx! (use mouse and keypad)",
		xmLabelWidgetClass, messageRowCol, NULL);

	pyraminx = XtCreateManagedWidget("pyraminx",
		pyraminxWidgetClass, mainPanel, NULL, 0);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	pyraminx = XtCreateManagedWidget("pyraminx",
		pyraminxWidgetClass, topLevel, NULL, 0);
#endif
	XtAddCallback(pyraminx, XtNselectCallback,
		(XtCallbackProc) CallbackPyraminx, (XtPointer) NULL);
	Initialize(pyraminx);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(pyraminx), (unsigned int) AnyButton, AnyModifier,
		XtWindow(pyraminx), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(pyraminx),
		XCreateFontCursor(XtDisplay(pyraminx), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
