/*
# MOTIF/X-BASED THREED
#
#  xthreed.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "useful",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/*
  Version 7: 03/11/15 X/Windows
  Version 5: 95/10/02 Xt/Motif
  Version 4: 94/11/17 Xt
  Version 3: 93/05/17 Motif
  Version 2: 93/05/14 XView
  Version 1: 93/03/01 MS Windows by Thomas W. Olsen "The C Users Journal"
*/

#ifdef HAVE_MOTIF
static const char aboutHelp[] = {
	"Version 7.1.3\n"
	"Send bugs (reports or fixes) to the author: "
	"David Bagley <bagleyd@tux.org>\n"
	"The latest version is at: "
	"http://www.tux.org/~bagleyd/puzzles.html\n"
	"Originally written by Thomas W. Olsen "
	"in the \"The C Users Journal\".\n"
};
static const char optionsHelp[] = {
	"[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]] "
	"[-display [{host}]:[{vs}]][-fg {color}] [-bg {color}]\n"
	"[-[no]surface] [-object {int}] [-x {int}] [-y {int}] [-z {int}] "
	"[-theta {int}] [-phi {int}] [psi {int}] [-white {color}]\n"
	"[-[lt|dk]gray {color}] [-black {color}] [-another {color}]\n"
};
#endif
#if defined(HAVE_MOTIF) || defined(WINVER)
static const char descriptionHelp[] = {
	"A simple 3D viewer."
};
static const char featuresHelp[] = {
	"Press \"i\" keys to move object in.\n"
	"Press \"o\" keys to move object out.\n"
	"Press \"S\" or \"s\" keys to change surface.\n"
	"Press \"B\" or \"b\" keys to change object.\n"
	"Press \"Esc\" key to hide program.\n"
	"Press \"Q\", \"fBq\", or \"CTRL-C\" keys kill program.\n"
	"Use the key pad, \"R\" keys, or arrow keys to rotate object.\n"
	"Key pad is defined for Threed as:\n"
	"  /     Out\n"
	"  8     Up\n"
	"  ^\n"
	"4<5>6   Left, In, Right\n"
	"  v\n"
	"  2     Down\n"
	"I figured there was other 3D viewers out there.  "
        "Well, this one is real simple, it only uses basic\n"
        "drawing routines, so it might be useful to someone "
	"out there.  If you have any improvements to make,\n"
	"let me know.\n"
	"In the sample data, Cubes and F16 are big, so the "
	"polyhedrons are scaled up to them in the data file.\n"
};
static const char referencesHelp[] = {
	"Original code by\n"
	"Thomas W. Olsen \"The C Users Journal\".\n"
	"Andre LaMothe \"Black Art of 3D Game Programming\".\n"
	"Arash Partow \"http://www.partow.net\".\n"
	
};
#endif

#ifdef WINVER
#include "ThreedP.h"
#include "wthreed.h"
#define TITLE "wthreed"

static ThreeDRec widget;
static HWND Wnd;

#define PRINT_MESSAGE(b) (void) MessageBox(Wnd, (LPCSTR) b, "Warning", MB_OK);
#define SET_STARTED(w,b) w->triangles.started = b

static void
Initialize(ThreeDWidget w, HBRUSH brush)
{
	HMENU hMenu;

	InitializeThreeD(w, brush);

	hMenu = GetMenu(w->core.hWnd);
	(void) CheckMenuItem(hMenu, (unsigned int)
		w->threed.object + IDM_TETRA, MF_CHECKED);
	(void) CheckMenuItem(hMenu, (unsigned int)
		w->threed.surface + IDM_WIRE, MF_CHECKED);
}

void
SetThreeD(ThreeDWidget w, int reason)
{
	switch (reason) {
		case THREED_HIDE:
			ShowWindow(w->core.hWnd, SW_SHOWMINIMIZED);
			break;
#if 0
		case THREED_SURFACE:
			w->threed.surface = !w->threed.surface;
			break;
		case THREED_OBJECT:
			w->threed.object = (w->threed.object + 1) %
				w->threed.numObjects;
			break;
#endif
	}
}

static LRESULT CALLBACK
About(HWND hDlg, UINT message, WPARAM wParam, LPARAM lParam)
{
	switch (message) {
		case WM_COMMAND:
			if (LOWORD(wParam) == IDOK) {
				(void) EndDialog(hDlg, TRUE);
				return TRUE;
			}
			break;
	}
	return FALSE;
}

static LRESULT CALLBACK
WindowProc(HWND hWnd, UINT message, WPARAM wParam, LPARAM lParam)
{
	HBRUSH brush = (HBRUSH) NULL;
	HMENU hMenu;
	PAINTSTRUCT paint;
	int vPos, hPos;
	static Boolean mousePressed = False;

	Wnd = widget.core.hWnd = hWnd;
	switch (message) {
		case WM_CREATE:
			Initialize(&widget, brush);
			break;
		case WM_DESTROY:
			DestroyThreeD(brush);
			break;
		case WM_SIZE:
			(void) InvalidateRect(widget.core.hWnd, NULL, TRUE);
			break;
		case WM_PAINT:
			widget.core.hDC = BeginPaint(widget.core.hWnd, &paint);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_BRUSH));
			ExposeThreeD(&widget);
			(void) EndPaint(hWnd, &paint);
			break;
		case WM_LBUTTONDOWN:
			mousePressed = True;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			SelectThreeD(&widget, LOWORD(lParam), HIWORD(lParam)
				/*, (GetKeyState(VK_CONTROL) >> 1) ? 1 : 0*/);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_MOUSEMOVE:
			if (!(wParam & MK_LBUTTON)) {
				if (mousePressed) {
					mousePressed = False;
					widget.core.hDC = GetDC(hWnd);
					(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
					ReleaseThreeD(&widget/*, LOWORD(lParam), HIWORD(lParam)
					((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
					(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0*/);
					(void) ReleaseDC(hWnd, widget.core.hDC);
				}
				break;
			}
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			MotionThreeD(&widget, LOWORD(lParam), HIWORD(lParam)
				/*, ((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0*/);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
		case WM_LBUTTONUP:
			mousePressed = False;
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			ReleaseThreeD(&widget/*, LOWORD(lParam), HIWORD(lParam)
				((GetKeyState(VK_SHIFT) >> 1) || (GetKeyState(VK_CAPITAL) & 1)),
				(GetKeyState(VK_CONTROL) >> 1) ? 1 : 0*/);
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#if (_WIN32_WINNT >= 0x0400) || (_WIN32_WINDOWS > 0x0400)
		case WM_MOUSEWHEEL:
			widget.core.hDC = GetDC(hWnd);
			(void) SelectObject(widget.core.hDC, GetStockObject(NULL_PEN));
			{
				int zDelta = ((short) HIWORD(wParam));
				POINT cursor, origin;

				origin.x = 0, origin.y = 0;
				ClientToScreen(hWnd, &origin);
				(void) GetCursorPos(&cursor);
				if (zDelta > (WHEEL_DELTA >> 1)) {
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.phi = (widget.threed.angle.phi +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
				} else if (zDelta < -(WHEEL_DELTA >> 1)) {
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.phi = (NUM_DEGREES + widget.threed.angle.phi -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
				}
			}
			(void) ReleaseDC(hWnd, widget.core.hDC);
			break;
#endif
		case WM_COMMAND:
			switch (LOWORD(wParam)) {
				case IDM_EXIT:
					DestroyThreeD(brush);
					break;
				case IDM_TETRA:
				case IDM_HEXA:
				case IDM_OCTA:
				case IDM_DODECA:
				case IDM_ICOSA:
				case IDM_RHOMBIC:
				case IDM_CONTA:
				case IDM_CUBES:
				case IDM_F16:
					hMenu = GetMenu(hWnd);
					(void) CheckMenuItem(hMenu,
						(unsigned int) widget.threed.object + IDM_TETRA,
						MF_UNCHECKED);
					(void) CheckMenuItem(hMenu,
						LOWORD(wParam), MF_CHECKED);
					SetObjectThreeD(&widget, LOWORD(wParam) - IDM_TETRA);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_HIDE:
					HideThreeD(&widget);
					break;
				case IDM_WIRE:
				case IDM_SOLID:
					hMenu = GetMenu(hWnd);
					(void) CheckMenuItem(hMenu,
						(unsigned int) widget.threed.surface + IDM_WIRE,
						MF_UNCHECKED);
					(void) CheckMenuItem(hMenu, LOWORD(wParam), MF_CHECKED);
					SetSurfaceThreeD(&widget, (BOOL) (LOWORD(wParam) - IDM_WIRE));
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DISTXD:
					if (widget.threed.distance.x >= MINDISTANCE + DELTADISTANCE)
						widget.threed.distance.x -= DELTADISTANCE;
					(void) SetScrollPos(hWnd, SB_HORZ, widget.threed.distance.x, TRUE);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DISTXI:
					if (widget.threed.distance.x <= MAXDISTANCE - DELTADISTANCE)
						widget.threed.distance.x += DELTADISTANCE;
					(void) SetScrollPos(hWnd, SB_HORZ, widget.threed.distance.x, TRUE);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DISTYD:
					if (widget.threed.distance.y >= MINDISTANCE + DELTADISTANCE)
						widget.threed.distance.y -= DELTADISTANCE;
					(void) SetScrollPos(hWnd, SB_VERT, widget.threed.distance.y, TRUE);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DISTYI:
					if (widget.threed.distance.y <= MAXDISTANCE - DELTADISTANCE)
						widget.threed.distance.y += DELTADISTANCE;
					(void) SetScrollPos(hWnd, SB_VERT, widget.threed.distance.y, TRUE);
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DEPTHZD:
					if (widget.threed.distance.z >= MINDEPTH + DELTADEPTH)
						widget.threed.distance.z -= DELTADEPTH;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DEPTHZI:
					if (widget.threed.distance.z <= MAXDEPTH - DELTADEPTH)
						widget.threed.distance.z += DELTADEPTH;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_THETAD:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.angle.theta = (NUM_DEGREES +
								widget.threed.angle.theta - DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_THETAI:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.angle.theta = (widget.threed.angle.theta +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PHID:
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.phi = (NUM_DEGREES + widget.threed.angle.phi -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PHII:
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.phi = (widget.threed.angle.phi +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PSID:
					widget.threed.deltaAngle.psi =
						-widget.threed.angle.psi;
					widget.threed.angle.psi = (NUM_DEGREES + widget.threed.angle.psi -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.psi +=
						widget.threed.angle.psi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_PSII:
					widget.threed.deltaAngle.psi =
						-widget.threed.angle.psi;
					widget.threed.angle.psi = (widget.threed.angle.psi +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.psi +=
						widget.threed.angle.psi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIAG0:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.theta = (NUM_DEGREES + widget.threed.angle.theta -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.angle.phi = (widget.threed.angle.phi +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIAG1:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.theta = (NUM_DEGREES + widget.threed.angle.theta -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.angle.phi = (NUM_DEGREES + widget.threed.angle.phi -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIAG2:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.theta = (widget.threed.angle.theta +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.angle.phi = (NUM_DEGREES + widget.threed.angle.phi -
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_DIAG3:
					widget.threed.deltaAngle.theta =
						-widget.threed.angle.theta;
					widget.threed.deltaAngle.phi =
						-widget.threed.angle.phi;
					widget.threed.angle.theta = (widget.threed.angle.theta +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.angle.phi = (widget.threed.angle.phi +
						DELTADEGREES) % NUM_DEGREES;
					widget.threed.deltaAngle.theta +=
						widget.threed.angle.theta;
					widget.threed.deltaAngle.phi +=
						widget.threed.angle.phi;
					(void) InvalidateRect(hWnd, NULL, TRUE);
					break;
				case IDM_ABOUT:
					(void) DialogBox(widget.core.hInstance,
						"About", hWnd, (DLGPROC) About);
					break;
				case IDM_DESCRIPTION:
					(void) MessageBox(hWnd, descriptionHelp,
						"Description", MB_OK);
					break;
				case IDM_FEATURES:
					(void) MessageBox(hWnd, featuresHelp,
						"Features", MB_OK);
					break;
				case IDM_REFERENCES:
					(void) MessageBox(hWnd, referencesHelp,
						"References", MB_OK);
					break;
			}
			break;
		case WM_HSCROLL:
			if (wParam == SB_THUMBTRACK)
				break;
			hPos = GetScrollPos(hWnd, SB_HORZ);
			switch (wParam) {
				case SB_TOP:
					hPos = MAXDISTANCE;
					break;
				case SB_BOTTOM:
					hPos = MINDISTANCE;
					break;
				case SB_LINEUP:
				case SB_PAGEUP:
					hPos -= DELTADISTANCE;
					break;
				case SB_PAGEDOWN:
				case SB_LINEDOWN:
					hPos += DELTADISTANCE;
					break;
				case SB_THUMBPOSITION:
					hPos = LOWORD(lParam);
					break;
			}
			if (hPos < MINDISTANCE)
				hPos = MINDISTANCE;
			else if (hPos > MAXDISTANCE)
				hPos = MAXDISTANCE;
			(void) SetScrollPos(hWnd, SB_HORZ, hPos, TRUE);
			widget.threed.distance.x = hPos;
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		case WM_VSCROLL:
			if (wParam == SB_THUMBTRACK)
				break;
			vPos = GetScrollPos(hWnd, SB_VERT);
			switch (wParam) {
				case SB_TOP:
					vPos = MINDISTANCE;
					break;
				case SB_BOTTOM:
					vPos = MAXDISTANCE;
					break;
				case SB_LINEUP:
				case SB_PAGEUP:
					vPos -= DELTADISTANCE;
					break;
				case SB_PAGEDOWN:
				case SB_LINEDOWN:
					vPos += DELTADISTANCE;
					break;
				case SB_THUMBPOSITION:
					vPos = LOWORD(lParam);
					break;
			}
			if (vPos < MINDISTANCE)
				vPos = MINDISTANCE;
			else if (vPos > MAXDISTANCE)
				vPos = MAXDISTANCE;
			(void) SetScrollPos(hWnd, SB_VERT, vPos, TRUE);
			widget.threed.distance.y = -vPos;
			(void) InvalidateRect(hWnd, NULL, TRUE);
			break;
		default:
			return (DefWindowProc(hWnd, message, wParam, lParam));
	}
	return FALSE;
}

int WINAPI
WinMain(HINSTANCE hInstance, HINSTANCE hPrevInstance, LPSTR lpCmdLine,
		int numCmdShow)
{
	HWND hWnd;
	MSG msg;
	WNDCLASS wc;
	HACCEL hAccel;

	if (!hPrevInstance) {
		wc.style = CS_HREDRAW | CS_VREDRAW;
		wc.lpfnWndProc = WindowProc;
		wc.cbClsExtra = 0;
		wc.cbWndExtra = 0;
		wc.hInstance = hInstance;
		wc.hIcon = LoadIcon(hInstance, TITLE);
		wc.hCursor = LoadCursor((HINSTANCE) NULL, IDC_ARROW);
		wc.hbrBackground = (HBRUSH) GetStockObject(GRAY_BRUSH);
		wc.lpszMenuName = TITLE;
		wc.lpszClassName = TITLE;
		if (!RegisterClass(&wc))
			return FALSE;
	}
	widget.core.hInstance = hInstance;
	hWnd = CreateWindow(
		TITLE,
		TITLE,
		WS_OVERLAPPEDWINDOW | WS_VSCROLL | WS_HSCROLL,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		(signed) CW_USEDEFAULT,
		HWND_DESKTOP,
		(HMENU) NULL,
		hInstance,
		(void *) NULL);
	if (!hWnd)
		return FALSE;
	(void) SetScrollRange(hWnd, SB_HORZ, MINDISTANCE, MAXDISTANCE, TRUE);
	(void) SetScrollRange(hWnd, SB_VERT, MINDISTANCE, MAXDISTANCE, TRUE);
	(void) SetScrollPos(hWnd, SB_HORZ, widget.threed.distance.x, TRUE);
	(void) SetScrollPos(hWnd, SB_VERT, widget.threed.distance.y, TRUE);
	hAccel = (HACCEL) LoadAccelerators(hInstance, TITLE);
	(void) ShowWindow(hWnd, numCmdShow);
	(void) UpdateWindow(hWnd);
	while (GetMessage(&msg, (HWND) NULL, 0, 0))
		if (!TranslateAccelerator(hWnd, hAccel, &msg)) {
			(void) TranslateMessage(&msg);
			(void) DispatchMessage(&msg);
		}
	return (msg.wParam);
}

#else

#include <stdlib.h>
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/Shell.h>
#include <X11/StringDefs.h>
#include <X11/cursorfont.h>
#ifdef HAVE_MOTIF
#include <Xm/PanedW.h>
#include <Xm/RowColumn.h>
#include <Xm/Label.h>
#include <Xm/LabelG.h>
#include <Xm/MessageB.h>
#include <Xm/PushBG.h>
#include <Xm/CascadeB.h>
#include <Xm/Scale.h>
#include <Xm/ToggleB.h>
#include <Xm/List.h>
#define PRINT_MESSAGE(b) PrintState(message, b)
#else
#define PRINT_MESSAGE(b) XtWarning(b)
#endif
#define SET_STARTED(w,b) XtSetArg(arg[0], XtNstart, b); XtSetValues(w, arg, 1)
#include "Threed.h"
#ifdef HAVE_XPM
#include <X11/xpm.h>
#include "icons/threed.xpm"
#endif
#include "icons/threed.xbm"

#define HIDDENSURFACEREMOVAL True
#define WIREFRAME False

#define FILENAMELEN 1024
#define TITLELEN 2048

#ifdef HAVE_MOTIF
static Widget xDist, yDist, zDist, thetaAng, phiAng, psiAng;
static Widget surfaceSwitch, objectList;
static Widget descriptionDialog, featuresDialog;
static Widget optionsDialog, referencesDialog, aboutDialog;
#else
static char titleDsp[TITLELEN]="";
#endif
static Pixmap threedIcon = None;
static Widget topLevel, threed;
static Arg arg[3];
static char *progDsp;

static void
Usage(char * programName)
{
	(void) fprintf(stderr, "usage: %s\n", programName);
	(void) fprintf(stderr,
		"\t[-geometry [{width}][x{height}][{+-}{xoff}[{+-}{yoff}]]]\n");
	(void) fprintf(stderr,
		"\t[-display [{host}]:[{vs}]][-fg {color}] [-bg {color}]\n");
	(void) fprintf(stderr,
		"\t[-[no]surface] [-object {int}] [-x {int}] [-y {int}]\n");
	(void) fprintf(stderr,
		"\t[-z {int}] [-theta {int}] [-phi {int}] [-psi {int}]\n");
	(void) fprintf(stderr,
		"\t[-white {color}] [-[lt|dk]gray {color}] [-black {color}]\n");
	(void) fprintf(stderr,
		"\t[-another {color}]\n");
	exit(1);
}

static XrmOptionDescRec options[] = {
	{(char *) "-fg", (char *) "*cubes.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bd", (char *) "*cubes.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-bg", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-foreground", (char *) "*cubes.Foreground", XrmoptionSepArg, NULL},
	{(char *) "-background", (char *) "*Background", XrmoptionSepArg, NULL},
	{(char *) "-surface", (char *) "*threed.surface", XrmoptionNoArg, (char *) "TRUE"},
	{(char *) "-nosurface", (char *) "*threed.surface", XrmoptionNoArg, (char *) "FALSE"},
	{(char *) "-object", (char *) "*threed.object", XrmoptionSepArg, (char *) "0"},
	{(char *) "-x", (char *) "*threed.distance.x", XrmoptionSepArg, (char *) "0.0"},
	{(char *) "-y", (char *) "*threed.distance.y", XrmoptionSepArg, (char *) "0.0"},
	{(char *) "-z", (char *) "*threed.distance.z", XrmoptionSepArg, (char *) "50.0"},
	{(char *) "-theta", (char *) "*threed.angle.theta", XrmoptionSepArg, (char *) "0.0"},
	{(char *) "-phi", (char *) "*threed.angle.phi", XrmoptionSepArg, (char *) "0.0"},
	{(char *) "-psi", (char *) "*threed.angle.psi", XrmoptionSepArg, (char *) "0.0"},
	{(char *) "-white", (char *) "*threed.whiteBrush", XrmoptionSepArg, NULL},
	{(char *) "-ltgray", (char *) "*threed.ltgrayBrush", XrmoptionSepArg, NULL},
	{(char *) "-gray", (char *) "*threed.grayBrush", XrmoptionSepArg, NULL},
	{(char *) "-dkgray", (char *) "*threed.dkgrayBrush", XrmoptionSepArg, NULL},
	{(char *) "-black", (char *) "*threed.blackBrush", XrmoptionSepArg, NULL},
	{(char *) "-another", (char *) "*threed.anotherBrush", XrmoptionSepArg, NULL},
};

#ifndef HAVE_MOTIF
static void PrintState(Widget w, char *name, Boolean surface, int object,
		IntPoint3D distance, IntAngle3D angle)
{
	(void) sprintf(titleDsp,
		"%s %d %s:x %d:y %d:z %d:theta %d:phi %d:psi %d", name,
		object, (surface) ? "surface" : "wire",
		distance.x, distance.y, distance.z,
		angle.theta, angle.phi, angle.psi);
	XtSetArg(arg[0], XtNtitle, titleDsp);
	XtSetValues(w, arg, 1);
}
#endif

static void Initialize(Widget w)
{
	IntPoint3D distance;
	IntAngle3D angle;
	int object;
	Boolean surface;
	char *name;
#ifndef HAVE_MOTIF
	char **list;
	int number, i;
#endif

	XtVaGetValues(w,
		XtNxDistance, &(distance.x),
		XtNyDistance, &(distance.y),
		XtNzDistance, &(distance.z),
		XtNthetaDegrees, &(angle.theta),
		XtNphiDegrees, &(angle.phi),
		XtNpsiDegrees, &(angle.psi),
		XtNobject, &object,
		XtNsurface, &surface,
		XtNobjectName, &name,
#ifndef HAVE_MOTIF
		XtNobjectNumber, &number,
		XtNobjectList, &list,
#endif
		NULL);
#ifdef HAVE_MOTIF
	XmScaleSetValue(xDist, distance.x);
	XmScaleSetValue(yDist, distance.y);
	XmScaleSetValue(zDist, distance.z);
	XmScaleSetValue(thetaAng, angle.theta);
	XmScaleSetValue(phiAng, angle.phi);
	XmScaleSetValue(psiAng, angle.psi);
	XmToggleButtonSetState(surfaceSwitch, surface, True);
#else
	PrintState(XtParent(w), name, surface, object, distance, theta, phi);
	(void) printf("Objects:\n");
	for (i = 0; i < number; i++)
		(void) printf("\t%s\n", list[i]);
#endif
}

#ifdef HAVE_MOTIF
static void makePosVisible(Widget list_w, int item_no)
{
	int top, visible;

	XtVaGetValues(list_w,
		XmNtopItemPosition, &top,
		XmNvisibleItemCount, &visible,
		NULL);
	if (item_no < top)
		XmListSetPos(list_w, item_no);
	else if (item_no >= top + visible)
		XmListSetBottomPos(list_w, item_no);
}
#endif

static void CallbackThreeD(Widget w, caddr_t clientData,
		threedCallbackStruct *callData)
{
	Boolean surface;
	IntPoint3D distance;
	IntAngle3D angle;
	int object, numObjects;
	char *name;

	XtVaGetValues(w,
		XtNobject, &object,
		XtNsurface, &surface,
		XtNxDistance, &(distance.x),
		XtNyDistance, &(distance.y),
		XtNzDistance, &(distance.z),
		XtNthetaDegrees, &(angle.theta),
		XtNphiDegrees, &(angle.phi),
		XtNpsiDegrees, &(angle.psi),
		XtNobjectNumber, &numObjects,
		XtNobjectName, &name,
		NULL);
	switch (callData->reason) {
		case THREED_HIDE:
			(void) XIconifyWindow(XtDisplay(topLevel), XtWindow(topLevel),
				XScreenNumberOfScreen(XtScreen(topLevel)));
			break;
		case THREED_SURFACE:
			surface = !surface;
			XtSetArg(arg[0], XtNsurface, surface);
			XtSetValues(w, arg, 1);
#ifdef HAVE_MOTIF
			XmToggleButtonSetState(surfaceSwitch, surface, True);
#endif
			break;
		case THREED_OBJECT:
			object = (object + 1) % numObjects;
			XtSetArg(arg[0], XtNobject, object);
			XtSetValues(w, arg, 1);
			XtVaGetValues(w,
				XtNobjectName, &name,
				NULL);
#ifdef HAVE_MOTIF
			makePosVisible(objectList, object + 1);
			XmListSelectPos(objectList, object + 1, True);
#endif
			break;
	}
#ifdef HAVE_MOTIF
	XmScaleSetValue(xDist, distance.x);
	XmScaleSetValue(yDist, distance.y);
	XmScaleSetValue(zDist, distance.z);
	XmScaleSetValue(thetaAng, angle.theta);
	XmScaleSetValue(phiAng, angle.phi);
	XmScaleSetValue(psiAng, angle.psi);
#else
	PrintState(XtParent(w), name, surface, object,
		x, y, z, theta, phi, psi);
#endif
}

#ifdef HAVE_MOTIF
static void
XDistSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNxDistance, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNxDistance, value,
			NULL);
}

static void
YDistSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNyDistance, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNyDistance, value,
			NULL);
}

static void
ZDistSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNzDistance, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNzDistance, value,
			NULL);
}

static void
ThetaSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNthetaDegrees, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNthetaDegrees, value,
			NULL);
}

static void
PhiSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNphiDegrees, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNphiDegrees, value,
			NULL);
}


static void
PsiSlider(Widget w, XtPointer clientData, XmScaleCallbackStruct *cbs)
{
	int value = cbs->value, old;

	XtVaGetValues(threed,
		XtNpsiDegrees, &old,
		NULL);
	if (old != value)
		XtVaSetValues(threed,
			XtNpsiDegrees, value,
			NULL);
}


static void
SurfaceToggle(Widget w, XtPointer clientData, XmToggleButtonCallbackStruct *cbs)
{
	XtVaSetValues(threed,
		XtNsurface, cbs->set,
		NULL);
}

static void
ObjectResponse(Widget listItem, XtPointer clientData, XmListCallbackStruct *cbs)
{
	char	*name;
	int object = cbs->item_position - 1, old;

	XtVaGetValues(threed,
		XtNobject, &old,
		NULL);
	if (old != object)
		XtVaSetValues(threed,
			XtNobject, object,
			NULL);
	XtVaGetValues(threed,
		XtNobjectName, &name,
		NULL);
}

#if 0
static void
createMenu(Widget *menu, Widget rowcol, int init)
{
	XmString choice, tetra, hexa, octa, dodeca, icosa, cubes, f16;

	choice = XmStringCreateSimple("Object:");
	tetra = XmStringCreateSimple("Tetrahedron");
	hexa = XmStringCreateSimple("Hexahedron");
	octa = XmStringCreateSimple("Octahedron");
	dodeca = XmStringCreateSimple("Dodecahedron");
	icosa = XmStringCreateSimple("Icosahedron");
	cubes = XmStringCreateSimple("Cubes");
	f16 = XmStringCreateSimple("F16");
	*menu = XmVaCreateSimpleOptionMenu(rowcol,
		"mode_menu", choice, 'M', init, object_response,
		XmVaPUSHBUTTON, tetra, 'T', NULL, NULL,
		XmVaPUSHBUTTON, hexa, 'H', NULL, NULL,
		XmVaPUSHBUTTON, octa, 'O', NULL, NULL,
		XmVaPUSHBUTTON, dodeca, 'D', NULL, NULL,
		XmVaPUSHBUTTON, icosa, 'I', NULL, NULL,
		XmVaPUSHBUTTON, cubes, 'C', NULL, NULL,
		XmVaPUSHBUTTON, f16, 'F', NULL, NULL,
		NULL);
	XmStringFree(tetra);
	XmStringFree(hexa);
	XmStringFree(octa);
	XmStringFree(dodeca);
	XmStringFree(icosa);
	XmStringFree(cubes);
	XmStringFree(f16);
	XtManageChild(*menu);
}
#endif

static void
createList(Widget *rowcol, char **list, int init, int num)
{
	XmString label;
	XmStringTable table;
	int i;

	label = XmStringCreateSimple((char *) "Object list:");
	(void) XtVaCreateManagedWidget("listLabel",
		xmLabelWidgetClass, *rowcol,
		XmNlabelString, label,
		NULL);
	XmStringFree(label);

	table = (XmStringTable) XtMalloc (num * sizeof (XmString *));
	for (i = 0; i < num; i++)
		table[i] = XmStringCreateSimple(list[i]);

	objectList = XmCreateScrolledList(*rowcol, (char *) "scrolledList",
		NULL, 0);
	XtVaSetValues(objectList,
		XmNvisibleItemCount, 5,
		XmNitemCount, num,
		XmNitems, table,
		NULL);
	XtManageChild(objectList);
	XtAddCallback(objectList, XmNbrowseSelectionCallback,
		(XtCallbackProc) ObjectResponse, (XtPointer) NULL);
	XmListSelectPos(objectList, init + 1, True);

	for (i = 0; i < num; i++)
		XmStringFree(table[i]);
	XtFree((char *) table);
}

static Widget
createHelp(Widget w, char *text, char *title)
{
	Widget button, messageBox;
	char titleDsp[FILENAMELEN + 8];
	XmString titleString = NULL, messageString = NULL, buttonString = NULL;
	static XmStringCharSet charSet =
		(XmStringCharSet) XmSTRING_DEFAULT_CHARSET;

	messageString = XmStringCreateLtoR(text, charSet);
	(void) sprintf(titleDsp, "%s: %s\n", progDsp, title);
	titleString = XmStringCreateSimple((char *) titleDsp);
	buttonString = XmStringCreateSimple((char *) "OK");
	XtSetArg(arg[0], XmNdialogTitle, titleString);
	XtSetArg(arg[1], XmNokLabelString, buttonString);
	XtSetArg(arg[2], XmNmessageString, messageString);
	messageBox = XmCreateInformationDialog(w, (char *) "helpBox",
		arg, 3);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_CANCEL_BUTTON);
	XtUnmanageChild(button);
	button = XmMessageBoxGetChild(messageBox, XmDIALOG_HELP_BUTTON);
	XtUnmanageChild(button);
	XmStringFree(titleString);
	XmStringFree(buttonString);
	XmStringFree(messageString);
	return messageBox;
}

static void
fileCB(Widget w, void *value, void *clientData)
{
	exit(0);
}


static void
helpCB(Widget w, XtPointer value, XtPointer clientData)
{
	int val = (int) value;

	switch (val) {
	case 0:
		XtManageChild(descriptionDialog);
		break;
	case 1:
		XtManageChild(featuresDialog);
		break;
	case 2:
		XtManageChild(optionsDialog);
		break;
	case 3:
		XtManageChild(referencesDialog);
		break;
	case 4:
		XtManageChild(aboutDialog);
		break;
	default:
		{
			char *buf;

			intCat(&buf, "helpCB: %d", val);
			XtWarning(buf);
			free(buf);
		}
	}
}
#endif

int main(int argc, char **argv)
{
#ifdef HAVE_MOTIF
	Widget menuBar, pullDownMenu, widget;
	Widget panel, mainPanel, controlPanel;
	Widget distRowCol, angleRowCol, shapeRowCol;
	XmString fileString, quitString;
	char **list;
	int object, number;
#endif

	progDsp = argv[0];
	topLevel = XtInitialize(argv[0], "Threed",
		options, XtNumber(options), &argc, argv);
	if (argc != 1)
		Usage(argv[0]);

#if HAVE_XPM
	{
		XpmAttributes xpmAttributes;
		XpmColorSymbol transparentColor[1] = {{NULL,
			(char *) "none", 0 }};
		Pixel bg;

		xpmAttributes.valuemask = XpmColorSymbols | XpmCloseness;
		xpmAttributes.colorsymbols = transparentColor;
		xpmAttributes.numsymbols = 1;
		xpmAttributes.closeness = 40000;
		XtVaGetValues(topLevel, XtNbackground, &bg, NULL);
		transparentColor[0].pixel = bg;
		(void) XpmCreatePixmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char **) threed_xpm, &threedIcon, NULL,
			&xpmAttributes);
	}
	if (threedIcon == (Pixmap) NULL)
#endif
		threedIcon = XCreateBitmapFromData(XtDisplay(topLevel),
			RootWindowOfScreen(XtScreen(topLevel)),
			(char *) threed_bits,
			threed_width, threed_height);
	XtSetArg(arg[0], XtNiconPixmap, threedIcon);
#ifdef HAVE_MOTIF
	/* not XmEXPLICIT */
	XtSetArg(arg[1], XmNkeyboardFocusPolicy, XmPOINTER);
	XtSetValues(topLevel, arg, 2);
	panel = XtVaCreateManagedWidget("panel",
		xmPanedWindowWidgetClass, topLevel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	fileString = XmStringCreateSimple((char *) "File");
	menuBar = XmVaCreateSimpleMenuBar(panel, (char *) "menuBar",
		XmVaCASCADEBUTTON, fileString, 'F',
		NULL);
	XmStringFree(fileString);
	quitString = XmStringCreateSimple((char *) "Quit");
	XmVaCreateSimplePulldownMenu(menuBar, (char *) "file_menu", 0, fileCB,
		XmVaPUSHBUTTON, quitString, 'Q', NULL, NULL,
		NULL);
	XmStringFree(quitString);
	pullDownMenu = XmCreatePulldownMenu(menuBar,
		(char *) "helpPullDown", NULL, 0);
	widget = XtVaCreateManagedWidget("Help",
		xmCascadeButtonWidgetClass, menuBar,
		XmNsubMenuId, pullDownMenu,
		XmNmnemonic, 'H', NULL);
	XtVaSetValues(menuBar, XmNmenuHelpWidget, widget, NULL);
	widget = XtVaCreateManagedWidget("Description",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'D', NULL);
		XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 0);
	widget = XtVaCreateManagedWidget("Features",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'F', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 1);
	widget = XtVaCreateManagedWidget("Options",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'O', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 2);
	widget = XtVaCreateManagedWidget("References",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'R', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 3);
	widget = XtVaCreateManagedWidget("About",
		xmPushButtonGadgetClass, pullDownMenu,
		XmNmnemonic, 'A', NULL);
	XtAddCallback(widget, XmNactivateCallback, helpCB, (char *) 4);
	XtManageChild(menuBar);
	descriptionDialog = createHelp(menuBar, (char *) descriptionHelp,
		(char *) "Description");
	featuresDialog = createHelp(menuBar, (char *) featuresHelp,
		(char *) "Features");
	optionsDialog = createHelp(menuBar, (char *) optionsHelp,
		(char *) "Options");
	referencesDialog = createHelp(menuBar, (char *) referencesHelp,
		(char *) "References");
	aboutDialog = createHelp(menuBar, (char *) aboutHelp,
		(char *) "About");
	mainPanel = XtCreateManagedWidget("mainPanel",
		xmPanedWindowWidgetClass, panel,
		NULL, 0);
	controlPanel = XtVaCreateManagedWidget("controlPanel",
		xmPanedWindowWidgetClass, mainPanel,
		XmNseparatorOn, False,
		XmNsashWidth, 1,
		XmNsashHeight, 1, NULL);
	distRowCol = XtVaCreateManagedWidget("distRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	xDist = XtVaCreateManagedWidget("xDist",
		xmScaleWidgetClass, distRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "X", 2,
		XmNminimum, MINDISTANCE,
		XmNmaximum, MAXDISTANCE,
		XmNvalue, (MINDISTANCE + MAXDISTANCE) / 2 ,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(xDist, XmNvalueChangedCallback,
		(XtCallbackProc) XDistSlider, (XtPointer) NULL);
	yDist = XtVaCreateManagedWidget("yDist",
		xmScaleWidgetClass, distRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Y", 2,
		XmNminimum, MINDISTANCE,
		XmNmaximum, MAXDISTANCE,
		XmNvalue, (MINDISTANCE + MAXDISTANCE) / 2 ,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(yDist, XmNvalueChangedCallback,
		(XtCallbackProc) YDistSlider, (XtPointer) NULL);
	zDist = XtVaCreateManagedWidget("zDist",
		xmScaleWidgetClass, distRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Z", 2,
		XmNminimum, MINDEPTH,
		XmNmaximum, MAXDEPTH,
		XmNvalue, (MINDEPTH + MAXDEPTH) / 2 ,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(zDist, XmNvalueChangedCallback,
		(XtCallbackProc) ZDistSlider, (XtPointer) NULL);
	angleRowCol = XtVaCreateManagedWidget("angleRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);
	thetaAng = XtVaCreateManagedWidget("thetaAng",
		xmScaleWidgetClass, angleRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Theta", 6,
		XmNminimum, MINDEGREES,
		XmNmaximum, MAXDEGREES,
		XmNvalue, MINDEGREES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(thetaAng, XmNvalueChangedCallback,
		(XtCallbackProc) ThetaSlider, (XtPointer) NULL);
	phiAng = XtVaCreateManagedWidget("phiAng",
		xmScaleWidgetClass, angleRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Phi", 4,
		XmNminimum, MINDEGREES,
		XmNmaximum, MAXDEGREES,
		XmNvalue, MINDEGREES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(phiAng, XmNvalueChangedCallback,
		(XtCallbackProc) PhiSlider, (XtPointer) NULL);
	psiAng = XtVaCreateManagedWidget("psiAng",
		xmScaleWidgetClass, angleRowCol,
		XtVaTypedArg, XmNtitleString, XmRString, "Psi", 4,
		XmNminimum, MINDEGREES,
		XmNmaximum, MAXDEGREES,
		XmNvalue, MINDEGREES,
		XmNshowValue, True,
		XmNorientation, XmHORIZONTAL, NULL);
	XtAddCallback(psiAng, XmNvalueChangedCallback,
		(XtCallbackProc) PsiSlider, (XtPointer) NULL);
	shapeRowCol = XtVaCreateManagedWidget("shapeRowCol",
		xmRowColumnWidgetClass, controlPanel,
		XmNnumColumns, 1,
		XmNorientation, XmHORIZONTAL,
		XmNpacking, XmPACK_COLUMN, NULL);

	threed = XtCreateManagedWidget("threed", threedWidgetClass, panel,
		NULL, 0);
	XtAddCallback(threed, XtNselectCallback,
		(XtCallbackProc) CallbackThreeD, (XtPointer) NULL);
	XtVaGetValues(threed,
		XtNobject, &object,
		XtNobjectNumber, &number,
		XtNobjectList, &list, NULL);
	/*createMenu(&objectMenu, shapeRowCol, object);*/
	createList(&shapeRowCol, list, object, number);
	surfaceSwitch = XtVaCreateManagedWidget("Surface",
		xmToggleButtonWidgetClass, shapeRowCol, NULL);
	XtAddCallback(surfaceSwitch, XmNvalueChangedCallback,
		(XtCallbackProc) SurfaceToggle, (XtPointer) NULL);
#else
	XtSetArg(arg[1], XtNinput, True);
	XtSetValues(topLevel, arg, 2);
	threed = XtCreateManagedWidget("threed",
		threedWidgetClass, topLevel,
		NULL, 0);
#endif
	XtAddCallback(threed,
		XtNselectCallback, (XtCallbackProc) CallbackThreeD,
		(XtPointer) NULL);
	Initialize(threed);
	XtRealizeWidget(topLevel);
	XGrabButton(XtDisplay(threed), (unsigned int) AnyButton, AnyModifier,
		XtWindow(threed), TRUE, (unsigned int) (ButtonPressMask |
		ButtonMotionMask | ButtonReleaseMask),
		GrabModeAsync, GrabModeAsync, XtWindow(threed),
		XCreateFontCursor(XtDisplay(threed), XC_crosshair));
	XtMainLoop();

#ifdef VMS
	return 1;
#else
	return 0;
#endif
}
#endif
