/*-
# X-BASED SKEWB
#
#  Skewb.c
#
###
#
#  Copyright (c) 1994 - 2005	David Albert Bagley, bagleyd@tux.org
#
#                   All Rights Reserved
#
#  Permission to use, copy, modify, and distribute this software and
#  its documentation for any purpose and without fee is hereby granted,
#  provided that the above copyright notice appear in all copies and
#  that both that copyright notice and this permission notice appear in
#  supporting documentation, and that the name of the author not be
#  used in advertising or publicity pertaining to distribution of the
#  software without specific, written prior permission.
#
#  This program is distributed in the hope that it will be "playable",
#  but WITHOUT ANY WARRANTY; without even the implied warranty of
#  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
#
*/

/* Methods file for Skewb */

#include "file.h"
#include "rngs.h"
#include "SkewbP.h"
#include "Skewb2dP.h"
#include "Skewb3dP.h"
#ifdef HAVE_OPENGL
#include "SkewbGLP.h"
#endif

#ifdef WINVER
#ifndef LOGPATH
#define LOGPATH "/usr/tmp"
#endif
#ifndef INIFILE
#define INIFILE "wskewb.ini"
#endif

#define SECTION "setup"

static const char *faceColorString[MAXFACES] =
{
        "255 0 0",
        "0 0 255",
        "255 255 255",
        "0 255 0",
        "255 192 203",
        "255 255 0"
};

static const char faceColorChar[MAXFACES] =
{'R', 'B', 'W', 'G', 'P', 'Y'};
#else

#ifndef LOGPATH
#ifdef VMS
#define LOGPATH "SYS$SCRATCH:"
#else
#define LOGPATH "/usr/tmp"
#endif
#endif

static Boolean SetValuesSkewb(Widget current, Widget request, Widget renew);
static void DestroySkewb(Widget old);
static void InitializeSkewb(Widget request, Widget renew);

SkewbClassRec skewbClassRec =
{
	{
		(WidgetClass) & widgetClassRec,		/* superclass */
		(char *) "Skewb",	/* class name */
		sizeof (SkewbRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) InitializeSkewb,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		NULL,		/* actions */
		0,		/* num actions */
		NULL,		/* resources */
		0,		/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		(XtWidgetProc) DestroySkewb,	/* destroy */
		NULL,		/* resize */
		NULL,		/* expose */
		(XtSetValuesFunc) SetValuesSkewb,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		NULL,		/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		NULL,		/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	}
};

WidgetClass skewbWidgetClass = (WidgetClass) & skewbClassRec;

void
SetSkewb(SkewbWidget w, int reason)
{
	skewbCallbackStruct cb;

	cb.reason = reason;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}

static void
SetSkewbMove(SkewbWidget w, int reason, int face, int position, int direction)
{
	skewbCallbackStruct cb;

	cb.reason = reason;
	cb.face = face;
	cb.position = position;
	cb.direction = direction;
	XtCallCallbacks((Widget) w, (char *) XtNselectCallback, &cb);
}
#endif

static void
loadFont(SkewbWidget w)
{
#ifndef WINVER
	Display *display = XtDisplay(w);
	const char *altfontname = "-*-times-*-r-*-*-*-180-*";
	char buf[512];

	if (w->skewb.fontInfo) {
		XUnloadFont(XtDisplay(w), w->skewb.fontInfo->fid);
		XFreeFont(XtDisplay(w), w->skewb.fontInfo);
	}
	if ((w->skewb.fontInfo = XLoadQueryFont(display,
			w->skewb.font)) == NULL) {
		(void) sprintf(buf,
			"Can not open %s font.\nAttempting %s font as alternate\n",
			w->skewb.font, altfontname);
		DISPLAY_WARNING(buf);
		if ((w->skewb.fontInfo = XLoadQueryFont(display,
				altfontname)) == NULL) {
			(void) sprintf(buf,
				"Can not open %s alternate font.\nUse the -font option to specify a font to use.\n",
				altfontname);
			DISPLAY_WARNING(buf);
		}
	}
	if (w->skewb.fontInfo) {
		w->skewb.letterOffset.x = XTextWidth(w->skewb.fontInfo, "8", 1)
			/ 2;
		w->skewb.letterOffset.y = w->skewb.fontInfo->max_bounds.ascent
			/ 2;
	} else
#endif
	{
		w->skewb.letterOffset.x = 3;
		w->skewb.letterOffset.y = 4;
	}
}

#ifndef LOGFILE
#define LOGFILE "skewb.log"
#endif

static SkewbLoc slideNextRow[MAXFACES][MAXORIENT][MAXORIENT / 2] =
{
	{
		{
			{2, CW},
			{1, HALF}},
		{
			{5, CCW},
			{1, STRT}},
		{
			{3, STRT},
			{5, CW}},
		{
			{3, HALF},
			{2, CCW}}
	},
	{
		{
			{4, STRT},
			{5, CW}},
		{
			{0, STRT},
			{5, CCW}},
		{
			{2, CCW},
			{0, HALF}},
		{
			{2, CW},
			{4, HALF}}
	},
	{
		{
			{4, CW},
			{1, CCW}},
		{
			{0, CCW},
			{1, CW}},
		{
			{3, CCW},
			{0, CW}},
		{
			{3, CW},
			{4, CCW}}
	},
	{
		{
			{4, HALF},
			{2, CCW}},
		{
			{0, HALF},
			{2, CW}},
		{
			{5, CW},
			{0, STRT}},
		{
			{5, CCW},
			{4, STRT}}
	},
	{
		{
			{5, CW},
			{1, STRT}},
		{
			{2, CCW},
			{1, HALF}},
		{
			{3, HALF},
			{2, CW}},
		{
			{3, STRT},
			{5, CCW}}
	},
	{
		{
			{0, CW},
			{1, CW}},
		{
			{4, CCW},
			{1, CCW}},
		{
			{3, CW},
			{4, CW}},
		{
			{3, CCW},
			{0, CCW}}
	}
};
static SkewbLoc minToMaj[MAXFACES][MAXORIENT] =
{				/* other equivalent mappings possible */
	{
		{3, CW},
		{2, STRT},
		{1, CCW},
		{5, STRT}},
	{
		{2, STRT},
		{4, CCW},
		{5, HALF},
		{0, CW}},
	{
		{3, STRT},
		{4, STRT},
		{1, STRT},
		{0, STRT}},
	{
		{5, HALF},
		{4, CW},
		{2, STRT},
		{0, CCW}},
	{
		{3, CCW},
		{5, STRT},
		{1, CW},
		{2, STRT}},
	{
		{3, HALF},
		{0, STRT},
		{1, HALF},
		{4, STRT}}
};

static SkewbLoc slideNextFace[MAXFACES][MAXORIENT] =
{
	{
		{5, STRT},
		{3, CW},
		{2, STRT},
		{1, CCW}},
	{
		{0, CW},
		{2, STRT},
		{4, CCW},
		{5, HALF}},
	{
		{0, STRT},
		{3, STRT},
		{4, STRT},
		{1, STRT}},
	{
		{0, CCW},
		{5, HALF},
		{4, CW},
		{2, STRT}},
	{
		{2, STRT},
		{3, CCW},
		{5, STRT},
		{1, CW}},
	{
		{4, STRT},
		{3, HALF},
		{0, STRT},
		{1, HALF}}
};

static int  faceToRotate[MAXFACES][MAXORIENT] =
{
	{3, 2, 1, 5},
	{2, 4, 5, 0},
	{3, 4, 1, 0},
	{5, 4, 2, 0},
	{3, 5, 1, 2},
	{3, 0, 1, 4}
};

static SkewbLocPos orthToDiag[MAXFACES][MAXORIENT][MAXORIENT] =
{
	{
		{
			{3, 0, 1},
			{5, 1, 0},
			{3, 0, 3},
			{5, 1, 2}},
		{
			{3, 3, 0},
			{2, 0, 1},
			{3, 3, 2},
			{2, 0, 3}},
		{
			{1, 0, 3},
			{2, 3, 0},
			{1, 0, 1},
			{2, 3, 2}},
		{
			{1, 3, 2},
			{5, 2, 1},
			{1, 3, 0},
			{5, 2, 3}}
	},
	{
		{
			{2, 3, 0},
			{0, 2, 1},
			{2, 3, 2},
			{0, 2, 3}},
		{
			{2, 2, 3},
			{4, 3, 0},
			{2, 2, 1},
			{4, 3, 2}},
		{
			{5, 3, 2},
			{4, 2, 3},
			{5, 3, 0},
			{4, 2, 1}},
		{
			{5, 2, 1},
			{0, 3, 2},
			{5, 2, 3},
			{0, 3, 0}}
	},
	{
		{
			{3, 3, 0},
			{0, 1, 0},
			{3, 3, 2},
			{0, 1, 2}},
		{
			{3, 2, 3},
			{4, 0, 1},
			{3, 2, 1},
			{4, 0, 3}},
		{
			{1, 1, 0},
			{4, 3, 0},
			{1, 1, 2},
			{4, 3, 2}},
		{
			{1, 0, 3},
			{0, 2, 1},
			{1, 0, 1},
			{0, 2, 3}}
	},
	{
		{
			{5, 1, 2},
			{0, 0, 3},
			{5, 1, 0},
			{0, 0, 1}},
		{
			{5, 0, 1},
			{4, 1, 2},
			{5, 0, 3},
			{4, 1, 0}},
		{
			{2, 1, 0},
			{4, 0, 1},
			{2, 1, 2},
			{4, 0, 3}},
		{
			{2, 0, 3},
			{0, 1, 0},
			{2, 0, 1},
			{0, 1, 2}}
	},
	{
		{
			{3, 2, 3},
			{2, 1, 0},
			{3, 2, 1},
			{2, 1, 2}},
		{
			{3, 1, 2},
			{5, 0, 1},
			{3, 1, 0},
			{5, 0, 3}},
		{
			{1, 2, 1},
			{5, 3, 0},
			{1, 2, 3},
			{5, 3, 2}},
		{
			{1, 1, 0},
			{2, 2, 1},
			{1, 1, 2},
			{2, 2, 3}}
	},
	{
		{
			{3, 1, 2},
			{4, 1, 0},
			{3, 1, 0},
			{4, 1, 2}},
		{
			{3, 0, 1},
			{0, 0, 1},
			{3, 0, 3},
			{0, 0, 3}},
		{
			{1, 3, 2},
			{0, 3, 0},
			{1, 3, 0},
			{0, 3, 2}},
		{
			{1, 2, 1},
			{4, 2, 1},
			{1, 2, 3},
			{4, 2, 3}}
	}
};

Boolean
CheckSolved(SkewbWidget w)
{
	int         face, position;
	SkewbLoc    test;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < MAXCUBES; position++) {
			if (!position) {
				test.face = w->skewb.cubeLoc[face][position].face;
				test.rotation = w->skewb.cubeLoc[face][position].rotation;
			} else if (test.face !=		/*face */
				   w->skewb.cubeLoc[face][position].face ||
				   (w->skewb.orient && test.rotation !=		/*STRT - MAXORIENT */
				  w->skewb.cubeLoc[face][position].rotation))
				return False;
		}
	return True;
}

#ifdef DEBUG

void
PrintCube(SkewbWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++) {
		for (position = 0; position < MAXCUBES; position++)
			(void) printf("%d %d  ", w->skewb.cubeLoc[face][position].face,
				  w->skewb.cubeLoc[face][position].rotation);
		(void) printf("\n");
	}
	(void) printf("\n");
}

#endif

static void
DrawDiamond(SkewbWidget w, int face, int offset)
{
	if (w->skewb.dim == 2)
		DrawDiamond2D((Skewb2DWidget) w, face, offset);
	else if (w->skewb.dim == 3)
		DrawDiamond3D((Skewb3DWidget) w, face, offset);
}

static void
DrawTriangle(SkewbWidget w, int face, int position, int offset)
{
	if (w->skewb.dim == 2)
		DrawTriangle2D((Skewb2DWidget) w, face, position, offset);
	else if (w->skewb.dim == 3)
		DrawTriangle3D((Skewb3DWidget) w, face, position, offset);
}

void
DrawAllPieces(SkewbWidget w)
{
	int         face, position;

#ifdef HAVE_OPENGL
	if (w->skewb.dim == 4) {
		DrawAllPiecesGL((SkewbGLWidget) w);
	}
#endif
	for (face = 0; face < MAXFACES; face++) {
		DrawDiamond(w, face, FALSE);
		for (position = 0; position < MAXORIENT; position++)
			DrawTriangle(w, face, position, FALSE);
	}
}

static void
DrawFrame(const SkewbWidget w, const Boolean focus)
{
	if (w->skewb.dim == 2)
		DrawFrame2D((Skewb2DWidget) w, focus);
	else if (w->skewb.dim == 3)
		DrawFrame3D((Skewb3DWidget) w, focus);
#ifdef HAVE_OPENGL
	else if (w->skewb.dim == 4)
		DrawFrameGL((SkewbGLWidget) w, focus);
#endif
}

static void
MoveNoPieces(SkewbWidget w)
{
	SetSkewb(w, SKEWB_ILLEGAL);
}

static void
RotateFace(SkewbWidget w, int face, int direction)
{
	int         corner;

	/* Read Face */
	for (corner = 0; corner < MAXORIENT; corner++)
		w->skewb.faceLoc[corner] = w->skewb.cubeLoc[face][corner];
	/* Write Face */
	for (corner = 0; corner < MAXORIENT; corner++) {
		w->skewb.cubeLoc[face][corner] = (direction == CW) ?
			w->skewb.faceLoc[(corner + MAXORIENT - 1) % MAXORIENT] :
			w->skewb.faceLoc[(corner + 1) % MAXORIENT];
		w->skewb.cubeLoc[face][corner].rotation =
			(w->skewb.cubeLoc[face][corner].rotation + direction) % MAXORIENT;
		DrawTriangle(w, face, corner, FALSE);
	}
	w->skewb.cubeLoc[face][MAXORIENT].rotation =
		(w->skewb.cubeLoc[face][MAXORIENT].rotation + direction) % MAXORIENT;
	DrawDiamond(w, face, FALSE);
#ifdef HAVE_OPENGL
	if (w->skewb.dim == 4) {
		DrawAllPiecesGL((SkewbGLWidget) w);
	}
#endif
}

static void
ReadFace(SkewbWidget w, int face, int h)
{
	int         position;

	for (position = 0; position < MAXCUBES; position++)
		w->skewb.rowLoc[h][position] = w->skewb.cubeLoc[face][position];
}

static void
WriteFace(SkewbWidget w, int face, int rotate, int h)
{
	int         corner, newCorner;

	for (corner = 0; corner < MAXORIENT; corner++) {
		newCorner = (corner + rotate) % MAXORIENT;
		w->skewb.cubeLoc[face][newCorner] = w->skewb.rowLoc[h][corner];
		w->skewb.cubeLoc[face][newCorner].rotation =
			(w->skewb.cubeLoc[face][newCorner].rotation + rotate) % MAXORIENT;
		DrawTriangle(w, face, (corner + rotate) % MAXORIENT, FALSE);
	}
	w->skewb.cubeLoc[face][MAXORIENT] = w->skewb.rowLoc[h][MAXORIENT];
	w->skewb.cubeLoc[face][MAXORIENT].rotation =
		(w->skewb.cubeLoc[face][MAXORIENT].rotation + rotate) % MAXORIENT;
	DrawDiamond(w, face, FALSE);
}

static void
ReadDiagonal(SkewbWidget w, int face, int corner, int orient, int size)
{
	int         g;

	if (size == MINOR)
		w->skewb.minorLoc[orient] = w->skewb.cubeLoc[face][corner];
	else {			/* size == MAJOR */
		for (g = 1; g < MAXORIENT; g++)
			w->skewb.majorLoc[orient][g - 1] =
				w->skewb.cubeLoc[face][(corner + g) % MAXORIENT];
		w->skewb.majorLoc[orient][MAXORIENT - 1] =
			w->skewb.cubeLoc[face][MAXORIENT];
	}
}

static void
RotateDiagonal(SkewbWidget w, int rotate, int orient, int size)
{
	int         g;

	if (size == MINOR)
		w->skewb.minorLoc[orient].rotation =
			(w->skewb.minorLoc[orient].rotation + rotate) % MAXORIENT;
	else			/* size == MAJOR */
		for (g = 0; g < MAXORIENT; g++)
			w->skewb.majorLoc[orient][g].rotation =
				(w->skewb.majorLoc[orient][g].rotation + rotate) % MAXORIENT;
}

static void
WriteDiagonal(SkewbWidget w, int face, int corner, int orient, int size)
{
	int         g, h;

	if (size == MINOR) {
		w->skewb.cubeLoc[face][corner] = w->skewb.minorLoc[orient];
		DrawTriangle(w, face, corner, FALSE);
	} else {		/* size == MAJOR */
		w->skewb.cubeLoc[face][MAXORIENT] =
			w->skewb.majorLoc[orient][MAXORIENT - 1];
		DrawDiamond(w, face, FALSE);
		for (g = 1; g < MAXORIENT; g++) {
			h = (corner + g) % MAXORIENT;
			w->skewb.cubeLoc[face][h] = w->skewb.majorLoc[orient][g - 1];
			DrawTriangle(w, face, h, FALSE);
		}
	}
#ifdef HAVE_OPENGL
	if (w->skewb.dim == 4) {
		DrawAllPiecesGL((SkewbGLWidget) w);
	}
#endif

}

static void
MovePieces(SkewbWidget w, int face, int position, int direction)
{
	int         newFace, newDirection, newCorner, k, size, rotate;

	if (direction < 2 * MAXORIENT) {
		/* position as MAXORIENT is ambiguous */
		for (size = MINOR; size <= MAJOR; size++) {
			ReadDiagonal((SkewbWidget) w, face, position, 0, size);
			for (k = 1; k <= MAXROTATE; k++) {
				newFace = slideNextRow[face][position][direction / 2].face;
				rotate = slideNextRow[face][position][direction / 2].rotation %
					MAXORIENT;
				newDirection = (rotate + direction) % MAXORIENT;
				newCorner = (rotate + position) % MAXORIENT;
				if (k != MAXROTATE)
					ReadDiagonal((SkewbWidget) w, newFace, newCorner, k, size);
				RotateDiagonal((SkewbWidget) w, rotate, k - 1, size);
				WriteDiagonal(w, newFace, newCorner, k - 1, size);
				face = newFace;
				position = newCorner;
				direction = newDirection;
			}
			if (size == MINOR) {
				newFace = minToMaj[face][position].face;
				rotate = minToMaj[face][position].rotation % MAXORIENT;
				direction = (rotate + direction) % MAXORIENT;
				position = (position + rotate + 2) % MAXORIENT;
				face = newFace;
			}
		}
	} else {
		RotateFace(w, faceToRotate[face][direction % MAXORIENT], CW);
		RotateFace(w, faceToRotate[face][(direction + 2) % MAXORIENT], CCW);
		ReadFace((SkewbWidget) w, face, 0);
		for (k = 1; k <= MAXORIENT; k++) {
			newFace = slideNextFace[face][direction % MAXORIENT].face;
			rotate = slideNextFace[face][direction % MAXORIENT].rotation;
			newDirection = (rotate + direction) % MAXORIENT;
			if (k != MAXORIENT)
				ReadFace((SkewbWidget) w, newFace, k);
			WriteFace(w, newFace, rotate, k - 1);
			face = newFace;
			direction = newDirection;
		}
	}
}

static void
MoveControlCb(SkewbWidget w, int face, int position, int direction)
{
	int         newFace, rotate;

	if (direction >= 2 * MAXORIENT) {
		SkewbLocPos newpos;

		newpos = orthToDiag[face][position][direction % MAXORIENT];
		face = newpos.face;
		position = newpos.position;
		direction = newpos.direction;
	}
	MovePieces(w, face, position, direction);
	SetSkewbMove(w, SKEWB_CONTROL, face, position, direction);
	newFace = minToMaj[face][position].face;
	rotate = minToMaj[face][position].rotation % MAXORIENT;
	direction = (rotate + direction) % MAXORIENT;
	position = (position + rotate + 2) % MAXORIENT;
	MovePieces(w, newFace, position, direction);
	SetSkewbMove(w, SKEWB_CONTROL, newFace, position, direction);
}

static void
MoveAltCb(SkewbWidget w, int face, int position, int direction)
{
	MovePieces(w, face, position, direction);
	SetSkewbMove(w, SKEWB_CONTROL, face, position, direction);
}

void
MoveSkewb(SkewbWidget w, int face, int position, int direction, int control)
{
	if (control == 2)
		MoveAltCb(w, face, position, direction);
	else if (control)
		MoveControlCb(w, face, position, direction);
	else {
		SkewbLocPos newpos;

		newpos.face = face;
		newpos.position = position;
		newpos.direction = direction;
		if (direction >= 2 * MAXORIENT)
			newpos = orthToDiag[face][position][direction % MAXORIENT];
		MovePieces(w, newpos.face, newpos.position, newpos.direction);
		SetSkewbMove(w, SKEWB_MOVED, newpos.face, newpos.position,
			newpos.direction);
	}
	PutMove(face, position, direction, control);
}

static      Boolean
SelectPieces(SkewbWidget w, int x, int y, int *face, int *position)
{
	if (w->skewb.dim == 2)
		return SelectPieces2D((Skewb2DWidget) w, x, y,
					   face, position);
	else if (w->skewb.dim == 3)
		return SelectPieces3D((Skewb3DWidget) w, x, y,
					   face, position);
	return False;
}

static      Boolean
CheckMoveDir(int position1, int position2, int *direction)
{
	if (!((position1 - position2 + MAXORIENT) % 2))
		return False;
	switch (position1) {
		case 0:
			*direction = (position2 == 1) ? 2 : 3;
			break;
		case 1:
			*direction = (position2 == 2) ? 3 : 0;
			break;
		case 2:
			*direction = (position2 == 3) ? 0 : 1;
			break;
		case 3:
			*direction = (position2 == 0) ? 1 : 2;
			break;
		default:
			return False;
	}
	*direction += 2 * MAXORIENT;
	return True;
}

static      Boolean
NarrowSelection(SkewbWidget w, int *face, int *position, int *direction)
{
	if (w->skewb.dim == 2)
		return NarrowSelection2D((Skewb2DWidget) w, face, position, direction);
	else if (w->skewb.dim == 3)
		return NarrowSelection3D((Skewb3DWidget) w, face, position, direction);
	return False;
}

static      Boolean
PositionPieces(SkewbWidget w, int x, int y, int *face, int *position, int *direction)
{
	if (!SelectPieces(w, x, y, face, position))
		return False;
	return NarrowSelection(w, face, position, direction);
}

void
MoveSkewbInput(SkewbWidget w, int x, int y, int direction, int control, int alt)
{
	int         face, position;

	if (!w->skewb.practice && !control && !alt && CheckSolved(w)) {
		MoveNoPieces(w);
		return;
	}
	if (!PositionPieces(w, x, y, &face, &position, &direction))
		return;
	if (alt)
		control = 2;
	else
		control = (control) ? 1 : 0;
	if (position == MAXORIENT)
		return;
	MoveSkewb(w, face, position, direction, control);
	if (!control && CheckSolved(w)) {
		SetSkewb(w, SKEWB_SOLVED);
	}
}

static void
ResetPieces(SkewbWidget w)
{
	int         face, position;

	for (face = 0; face < MAXFACES; face++)
		for (position = 0; position < MAXCUBES; position++) {
			w->skewb.cubeLoc[face][position].face = face;
			w->skewb.cubeLoc[face][position].rotation = STRT - MAXORIENT;
		}
	FlushMoves(w);
	w->skewb.started = False;
}

static void
GetPieces(SkewbWidget w)
{
	FILE       *fp;
	int         c, orient, practice, moves;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "r")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "r")) == NULL) {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not read (get) ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	FlushMoves(w);
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &orient);
	if (w->skewb.orient != (Boolean) orient) {
		SetSkewb(w, SKEWB_ORIENT);
	}
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &practice);
	if (w->skewb.practice != (Boolean) practice) {
		SetSkewb(w, SKEWB_PRACTICE);
	}
#ifdef WINVER
	ResetPieces(w);
#endif
	while ((c = getc(fp)) != EOF && c != SYMBOL);
	(void) fscanf(fp, "%d", &moves);
	ScanStartPosition(fp, w);
	SetSkewb(w, SKEWB_RESTORE);
	ScanMoves(fp, w, moves);
	(void) fclose(fp);
	(void) printf("%s: orient %d, practice %d, moves %d.\n",
		name, orient, practice, moves);
	free(lname);
	free(fname);
	w->skewb.cheat = True; /* Assume the worst. */
}

static void
WritePieces(SkewbWidget w)
{
	FILE       *fp;
	char *buf1 = NULL, *buf2 = NULL;
	char *fname, *lname, *name;

	stringCat(&buf1, CURRENTDELIM, LOGFILE);
	lname = buf1;
	stringCat(&buf1, LOGPATH, FINALDELIM);
	stringCat(&buf2, buf1, LOGFILE);
	free(buf1);
	fname = buf2;
	/* Try current directory first. */
	name = lname;
	if ((fp = fopen(name, "w")) == NULL) {
		name = fname;
		if ((fp = fopen(name, "w")) == NULL) {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, " or ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
			free(lname);
			free(fname);
			return;
		}
/* Probably annoying */
#if 0
		else {
			stringCat(&buf1, "Can not write to ", lname);
			stringCat(&buf2, buf1, ", falling back to ");
			free(buf1);
			stringCat(&buf1, buf2, fname);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
#endif
	}
	(void) fprintf(fp, "orient%c %d\n", SYMBOL,
		(w->skewb.orient) ? 1 : 0);
	(void) fprintf(fp, "practice%c %d\n", SYMBOL,
		(w->skewb.practice) ? 1 : 0);
	(void) fprintf(fp, "moves%c %d\n", SYMBOL, NumMoves());
	PrintStartPosition(fp, w);
	PrintMoves(fp);
	(void) fclose(fp);
	(void) printf("Saved to %s.\n", name);
	free(lname);
	free(fname);
}

static void
ClearPieces(SkewbWidget w)
{
	SetSkewb(w, SKEWB_CLEAR);
}

static void
UndoPieces(SkewbWidget w)
{
	if (MadeMoves()) {
		int         face, position, direction, control;

		GetMove(&face, &position, &direction, &control);
		if (direction < 2 * MAXORIENT) {
			direction = (direction < MAXORIENT) ?
				(direction + MAXORIENT / 2) % MAXORIENT :
				3 * MAXORIENT - direction;
		} else {
			direction = (direction + MAXORIENT / 2) % MAXORIENT +
			       2 * MAXORIENT;
		}
		if (control == 2)
			MoveAltCb(w, face, position, direction);
		else if (control)
			MoveControlCb(w, face, position, direction);
		else {
			if (direction >= 2 * MAXORIENT) {
				SkewbLocPos newpos;

				newpos = orthToDiag[face][position][direction % MAXORIENT];
				face = newpos.face;
				position = newpos.position;
				direction = newpos.direction;
			}
			MovePieces(w, face, position, direction);
			SetSkewbMove(w, SKEWB_UNDO, face, position, direction);
		}
	}
}

static void
PracticePieces(SkewbWidget w)
{
	SetSkewb(w, SKEWB_PRACTICE);
}

static void
RandomizePieces(SkewbWidget w)
{
	int         face, position, direction;
	int         big = MAXCUBES * 3 + NRAND(2);

	w->skewb.cheat = False;
	if (w->skewb.practice)
		PracticePieces(w);
	SetSkewb(w, SKEWB_RESET);

#ifdef DEBUG
	big = 3;
#endif

	while (big--) {
		face = NRAND(MAXFACES);
		position = NRAND(MAXORIENT);
		direction = ((NRAND(2)) ? position + 1 : position + 3) % MAXORIENT;
		MoveSkewb(w, face, position, direction, FALSE);
	}
	FlushMoves(w);
	SetSkewb(w, SKEWB_RANDOMIZE);
	if (CheckSolved(w)) {
		SetSkewb(w, SKEWB_SOLVED);
	}
}

static void
SolvePieces(SkewbWidget w)
{
	if (CheckSolved(w))
		return;
	{
		SetSkewb(w, SKEWB_SOLVE_MESSAGE);
	}
}

static void
OrientizePieces(SkewbWidget w)
{
	SetSkewb(w, SKEWB_ORIENT);
}

#ifdef WINVER
static void
SetValuesSkewb(SkewbWidget w)
{
	struct tagColor {
		int         red, green, blue;
	} color;
	char        szBuf[80], buf[20], charbuf[2];
	int         face;

	w->skewb.orient = (BOOL) GetPrivateProfileInt(SECTION, "orient",
		DEFAULTORIENT, INIFILE);
	w->skewb.practice = (BOOL) GetPrivateProfileInt(SECTION, "practice",
		DEFAULTPRACTICE, INIFILE);
	w->skewb.dim = GetPrivateProfileInt(SECTION, "dim", 3, INIFILE);
	w->skewb.mono = (BOOL) GetPrivateProfileInt(SECTION, "mono",
		DEFAULTMONO, INIFILE);
	w->skewb.reverse = (BOOL) GetPrivateProfileInt(SECTION, "reverse",
		DEFAULTREVERSE, INIFILE);
	/* cyan */
	(void) GetPrivateProfileString(SECTION, "frameColor", "0 255 255",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->skewb.frameGC = RGB(color.red, color.green, color.blue);
	/* gray25 */
	(void) GetPrivateProfileString(SECTION, "pieceBorder", "64 64 64",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->skewb.borderGC = RGB(color.red, color.green, color.blue);
	/* #AEB2C3 */
	(void) GetPrivateProfileString(SECTION, "background", "174 178 195",
		szBuf, sizeof (szBuf), INIFILE);
	(void) sscanf(szBuf, "%d %d %d",
		&(color.red), &(color.green), &(color.blue));
	w->skewb.inverseGC = RGB(color.red, color.green, color.blue);
	for (face = 0; face < MAXFACES; face++) {
		(void) sprintf(buf, "faceColor%d", face);
		(void) GetPrivateProfileString(SECTION, buf,
			faceColorString[face],
			szBuf, sizeof (szBuf), INIFILE);
		(void) sscanf(szBuf, "%d %d %d",
			&(color.red), &(color.green), &(color.blue));
		w->skewb.faceGC[face] =
			RGB(color.red, color.green, color.blue);
		(void) sprintf(buf, "faceChar%d", face);
		charbuf[0] = faceColorChar[face];
		charbuf[1] = '\0';
		(void) GetPrivateProfileString(SECTION, buf, charbuf,
			szBuf, sizeof (szBuf), INIFILE);
		w->skewb.faceChar[face] = szBuf[0];
	}
	(void) GetPrivateProfileString(SECTION, "userName", "Guest",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->skewb.userName, szBuf);
		w->skewb.userName[80] = 0;
	(void) GetPrivateProfileString(SECTION, "scoreFile", "",
		szBuf, sizeof (szBuf), INIFILE);
	(void) strcpy(w->skewb.scoreFile, szBuf);
		w->skewb.scoreFile[80] = 0;
}

void
DestroySkewb(HBRUSH brush)
{
	(void) DeleteObject(brush);
	PostQuitMessage(0);
}

void
ResizeSkewb(SkewbWidget w)
{
	if (w->skewb.dim == 2)
		ResizeSkewb2D((Skewb2DWidget) w);
	else if (w->skewb.dim == 3)
		ResizeSkewb3D((Skewb3DWidget) w);
}

void
SizeSkewb(SkewbWidget w)
{
	ResetPieces(w);
	ResizeSkewb(w);
}

void
ExposeSkewb(SkewbWidget w)
{
	if (w->skewb.dim == 2)
		ExposeSkewb2D((Skewb2DWidget) w);
	else if (w->skewb.dim == 3)
		ExposeSkewb3D((Skewb3DWidget) w);
}

#else
static void
GetColor(SkewbWidget w, int face)
{
	XGCValues   values;
	XtGCMask    valueMask;
	XColor      colorCell, rgb;

	valueMask = GCForeground | GCBackground;
	if (w->skewb.reverse) {
		values.background = w->skewb.foreground;
	} else {
		values.background = w->skewb.background;
	}
	if (!w->skewb.mono) {
		if (XAllocNamedColor(XtDisplay(w),
				  DefaultColormapOfScreen(XtScreen(w)),
				w->skewb.faceName[face], &colorCell, &rgb)) {
			values.foreground = w->skewb.faceColor[face] = colorCell.pixel;
			if (w->skewb.faceGC[face])
				XtReleaseGC((Widget) w, w->skewb.faceGC[face]);
			w->skewb.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
			return;
		} else {
			char *buf1, *buf2;

			stringCat(&buf1, "Color name \"",
				w->skewb.faceName[face]);
			stringCat(&buf2, buf1, "\" is not defined for face ");
			free(buf1);
			intCat(&buf1, buf2, face);
			free(buf2);
			DISPLAY_WARNING(buf1);
			free(buf1);
		}
	}
	if (w->skewb.reverse) {
		values.background = w->skewb.foreground;
		values.foreground = w->skewb.background;
	} else {
		values.background = w->skewb.background;
		values.foreground = w->skewb.foreground;
	}
	if (w->skewb.faceGC[face])
		XtReleaseGC((Widget) w, w->skewb.faceGC[face]);
	w->skewb.faceGC[face] = XtGetGC((Widget) w, valueMask, &values);
}

void
SetAllColors(SkewbWidget w)
{
	XGCValues   values;
	XtGCMask    valueMask;
	int         face;

	valueMask = GCForeground | GCBackground;

	if (w->skewb.reverse) {
		values.background = w->skewb.background;
		values.foreground = w->skewb.foreground;
	} else {
		values.foreground = w->skewb.background;
		values.background = w->skewb.foreground;
	}
	if (w->skewb.inverseGC)
		XtReleaseGC((Widget) w, w->skewb.inverseGC);
	w->skewb.inverseGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->skewb.mono) {
		if (w->skewb.reverse) {
			values.background = w->skewb.foreground;
			values.foreground = w->skewb.background;
		} else {
			values.foreground = w->skewb.foreground;
			values.background = w->skewb.background;
		}
	} else {
		values.foreground = w->skewb.frameColor;
		values.background = w->skewb.background;
	}
	if (w->skewb.frameGC)
		XtReleaseGC((Widget) w, w->skewb.frameGC);
	w->skewb.frameGC = XtGetGC((Widget) w, valueMask, &values);
	if (w->skewb.mono) {
		if (w->skewb.reverse) {
			values.background = w->skewb.foreground;
			values.foreground = w->skewb.background;
		} else {
			values.foreground = w->skewb.foreground;
			values.background = w->skewb.background;
		}
	} else {
		values.foreground = w->skewb.borderColor;
		values.background = w->skewb.background;
	}
	if (w->skewb.borderGC)
		XtReleaseGC((Widget) w, w->skewb.borderGC);
	w->skewb.borderGC = XtGetGC((Widget) w, valueMask, &values);
	for (face = 0; face < MAXFACES; face++)
		GetColor(w, face);
	if (w->skewb.fontInfo)
		XSetFont(XtDisplay(w), w->skewb.borderGC,
			w->skewb.fontInfo->fid);
}

static      Boolean
SetValuesSkewb(Widget current, Widget request, Widget renew)
{
	SkewbWidget c = (SkewbWidget) current, w = (SkewbWidget) renew;
	Boolean     redraw = False, setColors = False;
	int         face;

	for (face = 0; face < MAXFACES; face++) {
		if (strcmp(w->skewb.faceName[face], c->skewb.faceName[face])) {
			setColors = True;
			break;
		}
	}
	if (w->skewb.font != c->skewb.font ||
			w->skewb.borderColor != c->skewb.borderColor ||
			w->skewb.reverse != c->skewb.reverse ||
			w->skewb.mono != c->skewb.mono) {
		loadFont(w);
		SetAllColors(w);
		redraw = True;
	} else if (w->skewb.background != c->skewb.background ||
			w->skewb.foreground != c->skewb.foreground ||
			setColors) {
		SetAllColors(w);
		redraw = True;
	}
	if (w->skewb.orient != c->skewb.orient) {
		ResetPieces(w);
		redraw = True;
	} else if (w->skewb.practice != c->skewb.practice) {
		ResetPieces(w);
		redraw = True;
	}
	if (w->skewb.menu != -1) {
		switch (w->skewb.menu) {
		case 0:
			w->skewb.menu = -1;
			GetPieces(w);
			break;
		case 1:
			w->skewb.menu = -1;
			WritePieces(w);
			break;
		case 3:
			w->skewb.menu = -1;
			ClearPieces(w);
			break;
		case 4:
			w->skewb.menu = -1;
			UndoPieces(w);
			break;
		case 5:
			w->skewb.menu = -1;
			RandomizePieces(w);
			break;
		case 6:
			w->skewb.menu = -1;
			SolvePieces(w);
			break;
		case 7:
			w->skewb.menu = -1;
			OrientizePieces(w);
			break;
		case 8:
			w->skewb.menu = -1;
			PracticePieces(w);
			break;
		default:
			w->skewb.menu = -1;
			break;
		}
	}
	if (w->skewb.currentDirection == SKEWB_RESTORE) {
		SetStartPosition(w);
		w->skewb.currentDirection = SKEWB_IGNORE;
	} else if (w->skewb.currentDirection == SKEWB_CLEAR) {
		ResetPieces(w);
		redraw = True;
		w->skewb.currentDirection = SKEWB_IGNORE;
	} else if (w->skewb.currentDirection != SKEWB_IGNORE) {
		MovePieces(w, w->skewb.currentFace, w->skewb.currentPosition,
				w->skewb.currentDirection);
		w->skewb.currentDirection = SKEWB_IGNORE;
	}
	return redraw;
}

static void
DestroySkewb(Widget old)
{
	SkewbWidget w = (SkewbWidget) old;
	int         face;

	for (face = 0; face < MAXFACES; face++)
		XtReleaseGC(old, w->skewb.faceGC[face]);
	XtReleaseGC(old, w->skewb.borderGC);
	XtReleaseGC(old, w->skewb.frameGC);
	XtReleaseGC(old, w->skewb.inverseGC);
	XtRemoveCallbacks(old, XtNselectCallback, w->skewb.select);
}

void
QuitSkewb(SkewbWidget w, XEvent * event, char **args, int nArgs)
{
	Display *display = XtDisplay(w);

	if (w->skewb.fontInfo) {
		XUnloadFont(display, w->skewb.fontInfo->fid);
		XFreeFont(display, w->skewb.fontInfo);
	}
	XtCloseDisplay(display);
	exit(0);
}
#endif

#ifndef WINVER
static
#endif
void
InitializeSkewb(
#ifdef WINVER
SkewbWidget w, HBRUSH brush
#else
Widget request, Widget renew
#endif
)
{
#ifdef WINVER
	SetValuesSkewb(w);
#else
	SkewbWidget w = (SkewbWidget) renew;
	int face;

	w->skewb.mono = (DefaultDepthOfScreen(XtScreen(w)) < 2 ||
		w->skewb.mono);
	w->skewb.fontInfo = NULL;
	for (face = 0; face < MAXFACES; face++)
		w->skewb.faceGC[face] = NULL;
	w->skewb.borderGC = NULL;
	w->skewb.frameGC = NULL;
	w->skewb.inverseGC = NULL;
#endif
	w->skewb.focus = False;
	loadFont(w);
	InitMoves();
	w->skewb.cheat = False;
	ResetPieces(w);
#ifdef WINVER
	brush = CreateSolidBrush(w->skewb.inverseGC);
	SETBACK(w->core.hWnd, brush);
	(void) SRAND(time(NULL));
#else
	(void) SRAND(getpid());
#endif
}

void
HideSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SetSkewb(w, SKEWB_HIDE);
}

void
SelectSkewb(SkewbWidget w
#ifdef WINVER
, const int x, const int y, const int control, const int alt
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int  x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
	int alt = (int) (event->xkey.state &
		(Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask));
#endif

	if (SelectPieces(w, x, y,
		     &(w->skewb.currentFace), &(w->skewb.currentPosition))) {
		if (w->skewb.currentPosition == MAXORIENT) {
			if (control || alt || w->skewb.practice || !CheckSolved(w)) {
				DrawDiamond(w, w->skewb.currentFace, TRUE);
				FLUSH(w);
				Sleep(100);
				DrawDiamond(w, w->skewb.currentFace, FALSE);
				w->skewb.currentFace = SKEWB_IGNORE;
				w->skewb.currentDirection = SKEWB_IGNORE;
			}
		} else if (control || alt || w->skewb.practice || !CheckSolved(w)) {
			DrawTriangle(w, w->skewb.currentFace, w->skewb.currentPosition,
				     TRUE);
		}
	} else {
		w->skewb.currentFace = SKEWB_IGNORE;
		w->skewb.currentDirection = SKEWB_IGNORE;
	}
}

void
ReleaseSkewb(SkewbWidget w
#ifdef WINVER
, const int x, const int y, const int control, const int alt
#else
, XEvent * event, char **args, int nArgs
#endif
)
{
#ifndef WINVER
	int  x = event->xbutton.x, y = event->xbutton.y;
	int control = (int) (event->xkey.state & ControlMask);
	int alt = (int) (event->xkey.state &
		(Mod1Mask | Mod2Mask | Mod3Mask | Mod4Mask | Mod5Mask));
#endif
	int         face, position, count = -1, direction = 0, ctrl = control;

	if (w->skewb.currentFace == SKEWB_IGNORE)
		return;
	if (w->skewb.currentPosition != MAXORIENT)
		DrawTriangle(w, w->skewb.currentFace, w->skewb.currentPosition,
			FALSE);
	if (!control && !alt && !w->skewb.practice && CheckSolved(w))
		MoveNoPieces(w);
	else if (SelectPieces(w, x, y,
				&face, &position) && position != MAXORIENT &&
		 position != w->skewb.currentPosition) {
		if (alt)
			ctrl = 2;
		else
			ctrl = (ctrl) ? 1 : 0;
		if (face == w->skewb.currentFace)
			count = CheckMoveDir(w->skewb.currentPosition,
				position, &direction);
		if (count == 1) {
			MoveSkewb(w, face, w->skewb.currentPosition, direction,
				ctrl);
			if (!ctrl && CheckSolved(w)) {
				SetSkewb(w, SKEWB_SOLVED);
			}
		} else if (count == 0)
			MoveNoPieces(w);
	}
	w->skewb.currentFace = SKEWB_IGNORE;
	w->skewb.currentDirection = SKEWB_IGNORE;
}

void
PracticeSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	PracticePieces(w);
}

#ifndef WINVER
void
PracticeSkewbMaybe(SkewbWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->skewb.started)
		PracticePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetSkewb(w, SKEWB_PRACTICE_QUERY);
	}
#endif
}

void
PracticeSkewb2(SkewbWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->skewb.started)
#endif
		PracticePieces(w);
}
#endif

void
RandomizeSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	RandomizePieces(w);
}

#ifndef WINVER
void
RandomizeSkewbMaybe(SkewbWidget w
, XEvent * event, char **args, int nArgs
)
{
	if (!w->skewb.started)
		RandomizePieces(w);
#ifdef HAVE_MOTIF
	else {
		SetSkewb(w, SKEWB_RANDOMIZE_QUERY);
	}
#endif
}

void
RandomizeSkewb2(SkewbWidget w
, XEvent * event, char **args, int nArgs
)
{
#ifdef HAVE_MOTIF
	if (!w->skewb.started)
#endif
		RandomizePieces(w);
}
#endif

void
GetSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	GetPieces(w);
}

void
WriteSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	WritePieces(w);
}

void
ClearSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	ClearPieces(w);
}

void
UndoSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	UndoPieces(w);
}

void
SolveSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	SolvePieces(w);
}

void
OrientizeSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	OrientizePieces(w);
}

void
EnterSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->skewb.focus = True;
	DrawFrame(w, w->skewb.focus);
}

void
LeaveSkewb(SkewbWidget w
#ifndef WINVER
, XEvent * event, char **args, int nArgs
#endif
)
{
	w->skewb.focus = False;
	DrawFrame(w, w->skewb.focus);
}

#ifdef WINVER
void
DimSkewb(SkewbWidget w)
{
	SetSkewb(w, SKEWB_DIM);
}

#else
void
MoveSkewbCcw(SkewbWidget w, XEvent * event, char **args, int nArgs)
{
	MoveSkewbInput(w, event->xbutton.x, event->xbutton.y, CCW,
		       (int) (event->xbutton.state & ControlMask), FALSE);
}

void
MoveSkewbCw(SkewbWidget w, XEvent * event, char **args, int nArgs)
{
	MoveSkewbInput(w, event->xbutton.x, event->xbutton.y, CW,
		       (int) (event->xkey.state & ControlMask), FALSE);
}
#endif
