/*
 * Copyright 2005-2007 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

/*
 *
 * Trivial CRT ViaOutput implementation.
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <errno.h>
#endif

#include "via_vgahw.h"
#include "via_id.h"
#include "via_output.h"

/*
 * Output->Private
 */
struct CRTPrivate {
    Bool Sense; /* used for init only */

    int BandWidth; /* Limit modes - Config or EDID provided. */
};

/*
 *
 */
static void
CRTPrivateDestroy(struct ViaOutput *Output)
{
    xfree(Output->Private);

    /* So we won't try to destroy this twice */
    Output->PrivateDestroy = NULL;
}

/*
 *
 */
static struct CRTPrivate *
CRTPrivateCreate(struct ViaOutput *Output)
{
    VIAFUNC(Output->scrnIndex);

    Output->PrivSize = sizeof(struct CRTPrivate);
    Output->Private =  xnfcalloc(1, Output->PrivSize);
    memset(Output->Private, 0, Output->PrivSize);

    Output->PrivateDestroy = CRTPrivateDestroy;

    return Output->Private;
}

/*
 * Option handling.
 */
enum CRTOpts {
    OPTION_CRTSENSE,
    OPTION_CRTBANDWIDTH,
    OPTION_CRTEDID
};

static OptionInfoRec CRTOptions[] =
{
    {OPTION_CRTSENSE,     "CRTSense",     OPTV_BOOLEAN, {0}, FALSE},
    {OPTION_CRTBANDWIDTH, "CRTBandWidth", OPTV_INTEGER, {0}, FALSE},
    {OPTION_CRTEDID,      "CRTEDID",      OPTV_ANYSTR,  {0}, FALSE},
    {-1,  NULL,  OPTV_NONE,  {0},  FALSE }
};

/*
 *
 */
static OptionInfoPtr
CRTGetOptions(ScrnInfoPtr pScrn, struct ViaOutput *Output)
{
    struct CRTPrivate *Private = (struct CRTPrivate *) Output->Private;
    OptionInfoPtr  Options;
    char *EDIDFileName;

    Options = xnfalloc(sizeof(CRTOptions));
    memcpy(Options, CRTOptions, sizeof(CRTOptions));

    xf86ProcessOptions(pScrn->scrnIndex, pScrn->options, Options);

    if (!xf86ReturnOptValBool(Options, OPTION_CRTSENSE, TRUE)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Disabling CRT Sensing."
                   " CRT is considered always attached.\n");
        Private->Sense = FALSE;
    } else
        Private->Sense = TRUE;

    /* TODO, clean this up -- move to Output->MaxClock */
    Private->BandWidth = 0;
    xf86GetOptValInteger(Options, OPTION_CRTBANDWIDTH, &Private->BandWidth);
    if (Private->BandWidth && (Private->BandWidth < 1000)) {
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "%s: Option CRTBandWidth needs "
                   "to be provided in kHz not MHz.\n", __func__);
        Private->BandWidth = 0;
    }


    EDIDFileName = xf86GetOptValString(Options, OPTION_CRTEDID);
    if (EDIDFileName) {
        FILE *EDIDFile;

        EDIDFile = fopen(EDIDFileName, "r");
        if (!EDIDFile)
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "%s: Unable to open EDID file \"%s\": %s\n",
                       __func__, EDIDFileName, strerror(errno));
        else {
            unsigned char Block[128];

            if (fread(Block, 1, 128, EDIDFile) != 128)
                 xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                       "%s: Unable to read from EDID file \"%s\": %s\n",
                            __func__, EDIDFileName, strerror(errno));
            else
                Output->EDID = xf86InterpretEDID(pScrn->scrnIndex, Block);
        }
    }

    xfree(Options);
    return NULL;
}

/*
 * Not functional on the VT3122Ax.
 */
static Bool
CRTSense(struct ViaOutput *Output)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);
    CARD8 SR01, CR36;
    Bool Found = FALSE;

    VIAFUNC(pScrn->scrnIndex);

    SR01 = hwp->readSeq(hwp, 0x01);
    CR36 = hwp->readCrtc(hwp, 0x36);

    ViaSeqMask(hwp, 0x01, 0x00, 0x20);
    ViaCrtcMask(hwp, 0x36, 0x00, 0xF0);

    ViaSeqMask(hwp, 0x40, 0x80, 0x80);

    usleep(1);

    if (hwp->readST00(hwp) & 0x10)
        Found = TRUE;

    ViaSeqMask(hwp, 0x40, 0x00, 0x80);

    hwp->writeSeq(hwp, 0x01, SR01);
    hwp->writeCrtc(hwp, 0x36, CR36);

    return Found;
}

/*
 * Sync/Unsync
 */
static void
CRTPower(struct ViaOutput *Output, Bool On)
{
    ScrnInfoPtr pScrn = xf86Screens[Output->scrnIndex];
    vgaHWPtr hwp = VGAHWPTR(pScrn);

    VIAFUNC(pScrn->scrnIndex);

    if (On)
        ViaCrtcMask(hwp, 0x36, 0x00, 0x30);
    else
        ViaCrtcMask(hwp, 0x36, 0x30, 0x30);
}

/*
 * Empty.
 */
static ModeStatus
CRTModeValid(struct ViaOutput *Output, DisplayModePtr mode)
{
    return MODE_OK;
}

/*
 * Empty.
 */
static void
CRTMode(struct ViaOutput *Output, DisplayModePtr mode)
{
    ;
}

/*
 *
 */
struct ViaOutput *
ViaCRTInit(ScrnInfoPtr pScrn, I2CDevPtr pDev)
{
    VIAPtr pVia = VIAPTR(pScrn);
    struct ViaOutput *Output;
    struct CRTPrivate *Private;

    VIAFUNC(pScrn->scrnIndex);

    if (pDev)
        xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
                   "%s: Ignoring I2C Device.\n", __func__);

    Output = xnfcalloc(1, sizeof(struct ViaOutput));
    memset(Output, 0, sizeof(struct ViaOutput));

    Output->scrnIndex = pScrn->scrnIndex;
    Output->Type = OUTPUT_CRT;
    Output->Name = "CRT";

    Output->Active = TRUE;
    Output->ClockMaster = FALSE;

    Private = CRTPrivateCreate(Output);

    Output->Options = CRTGetOptions(pScrn, Output);

    /* Sensing doesn't work on VT3122Ax */
    if (Private->Sense && ((pVia->Chipset > VT3122) || VT3122_REV_IS_CX(pVia->HostRev)))
        Output->Sense = CRTSense;

    Output->ModeValid = CRTModeValid;
    Output->Mode = CRTMode;
    Output->Power = CRTPower;

    if (!Output->EDID && pVia->pI2CBus1)
        Output->EDID = xf86DoEDID_DDC2(pScrn->scrnIndex, pVia->pI2CBus1);

    if (Output->EDID)
        ViaOutputEDIDSet(Output, Output->EDID);

    return Output;
}
