/*
 * Copyright 1998-2003 VIA Technologies, Inc. All Rights Reserved.
 * Copyright 2001-2003 S3 Graphics, Inc. All Rights Reserved.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NON-INFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "xf86.h"
#include "xf86_OSproc.h"

#ifndef _XF86_ANSIC_H
#include <string.h>
#include <stdio.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>
#endif

#include "xf86Priv.h"

#include "xf86PciInfo.h"
#include "xf86Pci.h"

#define _XF86DRI_SERVER_
#include "GL/glxtokens.h"
#include "sarea.h"

#include "via_driver.h"
#include "via_dri.h"
#include "via_id.h"
#include "xf86drm.h"

extern void GlxSetVisualConfigs(
    int nconfigs,
    __GLXvisualConfig *configs,
    void **configprivs
);

#define VIDEO	0
#define AGP		1
#define AGP_PAGE_SIZE 4096
#define AGP_PAGES 8192
#define AGP_SIZE (AGP_PAGE_SIZE * AGP_PAGES)
#define AGP_CMDBUF_PAGES 512
#define AGP_CMDBUF_SIZE (AGP_PAGE_SIZE * AGP_CMDBUF_PAGES)

static char VIAKernelDriverName[] = "via";
static char VIAClientDriverName[] = "unichrome";
int test_alloc_FB(ScreenPtr pScreen, VIAPtr pVia, int Size);
int test_alloc_AGP(ScreenPtr pScreen, VIAPtr pVia, int Size);
static Bool VIAInitVisualConfigs(ScrnInfoPtr pScrn);
static Bool VIADRIAgpInit(VIAPtr pVia);
static Bool VIADRIPciInit(VIAPtr pVia);
static Bool VIADRIFBInit(VIAPtr pVia);
static Bool VIADRIKernelInit(VIAPtr pVia);
static Bool VIADRIMapInit(VIAPtr pVia);

static Bool VIACreateContext(ScreenPtr pScreen, VisualPtr visual,
                   drm_context_t hwContext, void *pVisualConfigPriv,
                   DRIContextType contextStore);
static void VIADestroyContext(ScreenPtr pScreen, drm_context_t hwContext,
                   DRIContextType contextStore);
static void VIADRISwapContext(ScreenPtr pScreen, DRISyncType syncType,
                   DRIContextType readContextType,
                   void *readContextStore,
                   DRIContextType writeContextType,
                   void *writeContextStore);
static void VIADRIInitBuffers(WindowPtr pWin, RegionPtr prgn, CARD32 index);
static void VIADRIMoveBuffers(WindowPtr pParent, DDXPointRec ptOldOrg,
                   RegionPtr prgnSrc, CARD32 index);


static void
VIADRIIrqInit(VIAPtr pVia, VIADRIPtr pVIADRI)
{
    pVIADRI->irqEnabled = drmGetInterruptFromBusID
	(pVia->drmFD,
	 ((pciConfigPtr)pVia->PciInfo->thisCard)->busnum,
	 ((pciConfigPtr)pVia->PciInfo->thisCard)->devnum,
	 ((pciConfigPtr)pVia->PciInfo->thisCard)->funcnum);
    if ((drmCtlInstHandler(pVia->drmFD, pVIADRI->irqEnabled))) {
	xf86DrvMsg(pVia->scrnIndex, X_WARNING,
		   "[drm] Failure adding irq handler. "
		   "Falling back to irq-free operation.\n");
	pVIADRI->irqEnabled = 0;
    }

    if (pVIADRI->irqEnabled)
	xf86DrvMsg(pVia->scrnIndex, X_INFO,
		   "[drm] Irq handler installed, using IRQ %d.\n",
		   pVIADRI->irqEnabled);
}

static void
VIADRIIrqExit(VIAPtr pVia, VIADRIPtr pVIADRI)
{
    if (pVIADRI->irqEnabled) {
	if (drmCtlUninstHandler(pVia->drmFD)) {
	    xf86DrvMsg(pVia->scrnIndex, X_INFO,
                       "[drm] Irq handler uninstalled.\n");
	} else {
	    xf86DrvMsg(pVia->scrnIndex, X_ERROR,
		       "[drm] Could not uninstall irq handler.\n");
	}
    }
}

void
VIADRIRingBufferCleanup(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIADRIPtr pVIADRI = pVia->pDRIInfo->devPrivate;

    if (pVIADRI->ringBufActive) {
	drm_via_dma_init_t ringBufInit;

	xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		   "[drm] Cleaning up DMA ring-buffer.\n");
	ringBufInit.func = VIA_CLEANUP_DMA;
	if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
			    sizeof(ringBufInit))) {
	    xf86DrvMsg(pScrn->scrnIndex, X_WARNING,
		       "[drm] Failed to clean up DMA ring-buffer: %d\n", errno);
	}
	pVIADRI->ringBufActive = 0;
    }
}

Bool
VIADRIRingBufferInit(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIADRIPtr pVIADRI = pVia->pDRIInfo->devPrivate;

    if (pVIADRI->ringBufActive)
	return TRUE;

    if (pVia->agpEnable) {
	drm_via_dma_init_t ringBufInit;

        if (pVia->drmVersion < 0x010400)
            return FALSE;

	/*
	 * Info frome code-snippet on DRI-DEVEL list; Erdi Chen.
	 */

        if (pVia->Chipset == VT3118)
	    pVIADRI->reg_pause_addr = 0x40c;
        else
	    pVIADRI->reg_pause_addr = 0x418;

	ringBufInit.offset = pVia->agpSize;
	ringBufInit.size = AGP_CMDBUF_SIZE;
	ringBufInit.reg_pause_addr = pVIADRI->reg_pause_addr;
	ringBufInit.func = VIA_INIT_DMA;
	if (drmCommandWrite(pVia->drmFD, DRM_VIA_DMA_INIT, &ringBufInit,
			    sizeof(ringBufInit))) {
	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		       "[drm] Failed to initialize DMA ring-buffer: %d\n", errno);
	    return FALSE;
	}
	xf86DrvMsg(pScrn->scrnIndex, X_INFO,
		   "[drm] Initialized AGP ring-buffer, size 0x%lx at AGP offset 0x%lx.\n",
		   ringBufInit.size, ringBufInit.offset);

	pVIADRI->ringBufActive = 1;
    }
    return TRUE;
}

static Bool
VIADRIAgpInit(VIAPtr pVia)
{
    DRIInfoPtr pDRIInfo = pVia->pDRIInfo;
    VIADRIPtr pVIADRI = pDRIInfo->devPrivate;
    drmAddress agpaddr;
    unsigned long agp_phys;

    pVia->agpSize = 0;

    if (drmAgpAcquire(pVia->drmFD) < 0) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "[drm] drmAgpAcquire failed %d\n", errno);
        return FALSE;
    }

    if (drmAgpEnable(pVia->drmFD, drmAgpGetMode(pVia->drmFD)&~0x0) < 0) {
         xf86DrvMsg(pVia->scrnIndex, X_ERROR, "[drm] drmAgpEnable failed\n");
        return FALSE;
    }

    xf86DrvMsg(pVia->scrnIndex, X_INFO, "[drm] drmAgpEnabled succeeded\n");

    if (drmAgpAlloc(pVia->drmFD, AGP_SIZE, 0, &agp_phys, &pVia->agpHandle) < 0) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "[drm] drmAgpAlloc failed\n");
        drmAgpRelease(pVia->drmFD);
        return FALSE;
    }

    if (drmAgpBind(pVia->drmFD, pVia->agpHandle, 0) < 0) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR, "[drm] drmAgpBind failed\n");
        drmAgpFree(pVia->drmFD, pVia->agpHandle);
        drmAgpRelease(pVia->drmFD);

        return FALSE;
    }

    /*
     * Place the ring-buffer last in the AGP region, and restrict the
     * public map not to include the buffer for security reasons.
     */

    pVia->agpSize = AGP_SIZE - AGP_CMDBUF_SIZE;
    pVia->agpAddr = drmAgpBase(pVia->drmFD);
    xf86DrvMsg(pVia->scrnIndex, X_INFO, "[drm] agpAddr = 0x%08lx\n",
	       pVia->agpAddr);

    pVIADRI->agp.size = pVia->agpSize;
    if (drmAddMap(pVia->drmFD, (drm_handle_t)0,
                 pVIADRI->agp.size, DRM_AGP, 0,
                 &pVIADRI->agp.handle) < 0) {
	xf86DrvMsg(pVia->scrnIndex, X_ERROR,
	    "[drm] Failed to map public agp area\n");
        pVIADRI->agp.size = 0;
	drmAgpUnbind(pVia->drmFD, pVia->agpHandle);
	drmAgpFree(pVia->drmFD, pVia->agpHandle);
	drmAgpRelease(pVia->drmFD);
	return FALSE;
    }

    /* Map AGP from kernel to Xserver - Not really needed */
    /* Why are we not passing agpMapped more directly? -- Luc */
    drmMap(pVia->drmFD, pVIADRI->agp.handle, pVIADRI->agp.size, &agpaddr);
    pVia->agpMappedAddr = agpaddr;

    xf86DrvMsg(pVia->scrnIndex, X_INFO,
                "[drm] agpBase = %p\n", pVia->agpBase);
    xf86DrvMsg(pVia->scrnIndex, X_INFO,
                "[drm] agpAddr = 0x%08lx\n", pVia->agpAddr);
    xf86DrvMsg(pVia->scrnIndex, X_INFO,
                "[drm] agpSize = 0x%08x\n", pVia->agpSize);
    xf86DrvMsg(pVia->scrnIndex, X_INFO,
                "[drm] agp physical addr = 0x%08lx\n", agp_phys);

    {
	drm_via_agp_t agp;
	agp.offset = 0;
	agp.size = AGP_SIZE-AGP_CMDBUF_SIZE;
	if (drmCommandWrite(pVia->drmFD, DRM_VIA_AGP_INIT, &agp,
			    sizeof(drm_via_agp_t)) < 0) {
	    drmUnmap(agpaddr,pVia->agpSize);
	    drmRmMap(pVia->drmFD,pVIADRI->agp.handle);
	    drmAgpUnbind(pVia->drmFD, pVia->agpHandle);
	    drmAgpFree(pVia->drmFD, pVia->agpHandle);
	    drmAgpRelease(pVia->drmFD);
	    return FALSE;
	}
    }

    return TRUE;
}

/*
 *
 */
static Bool
VIADRIFBInit(VIAPtr pVia)
{
    int FBSize = pVia->FBFreeEnd - pVia->FBFreeStart;
    int FBOffset = pVia->FBFreeStart;
    VIADRIPtr pVIADRI = pVia->pDRIInfo->devPrivate;

    /* Guesswork: make sure we at least have 1 page? This should be sane-ish,
     * even though a single page will be pretty much useless */
    if ((pVia->FBFreeStart + 4096) >= pVia->FBFreeEnd) {
        xf86DrvMsg(pVia->scrnIndex, X_ERROR,
                   "%s: Not enough FB memory left to use DRI.\n", __func__);
        return FALSE;
    }

    pVIADRI->fbOffset = FBOffset;
    pVIADRI->fbSize = pVia->videoRambytes;

    {
	drm_via_fb_t fb;
	fb.offset = FBOffset;
	fb.size = FBSize;

	if (drmCommandWrite(pVia->drmFD, DRM_VIA_FB_INIT, &fb,
			    sizeof(drm_via_fb_t)) < 0) {
	    xf86DrvMsg(pVia->scrnIndex, X_ERROR,
		       "[drm] failed to init frame buffer area\n");
	    return FALSE;
	} else {
	    xf86DrvMsg(pVia->scrnIndex, X_INFO,
		       "[drm] Using FB from 0x%08x to 0x%08x (0x%08x)\n",
		       pVia->FBFreeStart, pVia->FBFreeEnd, FBSize);
	    return TRUE;
	}
    }
}

static Bool VIADRIPciInit(VIAPtr pVia)
{
    return TRUE;
}

static Bool
VIAInitVisualConfigs(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    __GLXvisualConfig *pConfigs = 0;
    VIAConfigPrivPtr pVIAConfigs = 0;
    VIAConfigPrivPtr *pVIAConfigPtrs = 0;
    int numConfigs = 0;
    int i, db, stencil, accum;

    switch (pScrn->bitsPerPixel) {
	case 8:
	case 24:
	    break;
	case 16:
	    numConfigs = 12;
	    if (!(pConfigs = (__GLXvisualConfig*)xcalloc(sizeof(__GLXvisualConfig),
						   numConfigs)))
		return FALSE;
	    if (!(pVIAConfigs = (VIAConfigPrivPtr)xcalloc(sizeof(VIAConfigPrivRec),
						    numConfigs))) {
    		xfree(pConfigs);
    		return FALSE;
	    }
	    if (!(pVIAConfigPtrs = (VIAConfigPrivPtr*)xcalloc(sizeof(VIAConfigPrivPtr),
							  numConfigs))) {
    		xfree(pConfigs);
    		xfree(pVIAConfigs);
    		return FALSE;
	    }
	    for (i=0; i<numConfigs; i++)
    		pVIAConfigPtrs[i] = &pVIAConfigs[i];

	    i = 0;
	    for (accum = 0; accum <= 1; accum++) {
		/* 32bpp depth buffer disabled, as Mesa has limitations */
    		for (stencil=0; stencil<=2; stencil++) {
    		    for (db = 0; db <= 1; db++) {
        		pConfigs[i].vid = -1;
        		pConfigs[i].class = -1;
        		pConfigs[i].rgba = TRUE;
        		pConfigs[i].redSize = -1;
        		pConfigs[i].greenSize = -1;
        		pConfigs[i].blueSize = -1;
        		pConfigs[i].redMask = -1;
        		pConfigs[i].greenMask = -1;
        		pConfigs[i].blueMask = -1;
			pConfigs[i].alphaSize = 0;
        		pConfigs[i].alphaMask = 0;

			if (accum) {
        		    pConfigs[i].accumRedSize = 16;
        		    pConfigs[i].accumGreenSize = 16;
        		    pConfigs[i].accumBlueSize = 16;
        		    pConfigs[i].accumAlphaSize = 0;
        		}
			else {
        		    pConfigs[i].accumRedSize = 0;
        		    pConfigs[i].accumGreenSize = 0;
        		    pConfigs[i].accumBlueSize = 0;
        		    pConfigs[i].accumAlphaSize = 0;
        		}
        		if (!db)
        		    pConfigs[i].doubleBuffer = TRUE;
        		else
        		    pConfigs[i].doubleBuffer = FALSE;

			pConfigs[i].stereo = FALSE;
        		pConfigs[i].bufferSize = -1;

			switch (stencil) {
        		    case 0:
            			pConfigs[i].depthSize = 24;
            			pConfigs[i].stencilSize = 8;
            			break;
        		    case 1:
            			pConfigs[i].depthSize = 16;
            			pConfigs[i].stencilSize = 0;
            			break;
        		    case 2:
            			pConfigs[i].depthSize = 0;
            			pConfigs[i].stencilSize = 0;
            			break;
        		    case 3:
            			pConfigs[i].depthSize = 32;
            			pConfigs[i].stencilSize = 0;
            			break;
        		}

			pConfigs[i].auxBuffers = 0;
        		pConfigs[i].level = 0;
			if (accum) {
        		    pConfigs[i].visualRating = GLX_SLOW_VISUAL_EXT;
			} else {
        		    pConfigs[i].visualRating = GLX_NONE_EXT;
			}
        		pConfigs[i].transparentPixel = GLX_NONE_EXT;
        		pConfigs[i].transparentRed = 0;
        		pConfigs[i].transparentGreen = 0;
        		pConfigs[i].transparentBlue = 0;
	        	pConfigs[i].transparentAlpha = 0;
        		pConfigs[i].transparentIndex = 0;
        		i++;
		    }
    		}
	    }

	if (i != numConfigs) {
	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		"[dri] Incorrect initialization of visuals.  Disabling DRI.\n");
    		return FALSE;
	}
	break;

	case 32:
	    numConfigs = 12;
	    if (!(pConfigs = (__GLXvisualConfig*)xcalloc(sizeof(__GLXvisualConfig),
						   numConfigs)))
		return FALSE;
	    if (!(pVIAConfigs = (VIAConfigPrivPtr)xcalloc(sizeof(VIAConfigPrivRec),
						    numConfigs))) {
    		xfree(pConfigs);
    		return FALSE;
	    }
	    if (!(pVIAConfigPtrs = (VIAConfigPrivPtr*)xcalloc(sizeof(VIAConfigPrivPtr),
							  numConfigs))) {
    		xfree(pConfigs);
    		xfree(pVIAConfigs);
    		return FALSE;
	    }
	    for (i=0; i<numConfigs; i++)
    		pVIAConfigPtrs[i] = &pVIAConfigs[i];

	    i = 0;
	    for (accum = 0; accum <= 1; accum++) {
		/* 32bpp depth buffer disabled, as Mesa has limitations */
    		for (stencil=0; stencil<=2; stencil++) {
    		    for (db = 0; db <= 1; db++) {
        		pConfigs[i].vid = -1;
        		pConfigs[i].class = -1;
        		pConfigs[i].rgba = TRUE;
        		pConfigs[i].redSize = -1;
        		pConfigs[i].greenSize = -1;
        		pConfigs[i].blueSize = -1;
        		pConfigs[i].redMask = -1;
        		pConfigs[i].greenMask = -1;
        		pConfigs[i].blueMask = -1;
			pConfigs[i].alphaSize = 8;
        		pConfigs[i].alphaMask = 0xFF000000;

			if (accum) {
        		    pConfigs[i].accumRedSize = 16;
        		    pConfigs[i].accumGreenSize = 16;
        		    pConfigs[i].accumBlueSize = 16;
        		    pConfigs[i].accumAlphaSize = 16;
        		}
			else {
        		    pConfigs[i].accumRedSize = 0;
        		    pConfigs[i].accumGreenSize = 0;
        		    pConfigs[i].accumBlueSize = 0;
        		    pConfigs[i].accumAlphaSize = 0;
        		}
        		if (!db)
        		    pConfigs[i].doubleBuffer = TRUE;
        		else
        		    pConfigs[i].doubleBuffer = FALSE;

			pConfigs[i].stereo = FALSE;
        		pConfigs[i].bufferSize = -1;

			switch (stencil) {
        		    case 0:
            			pConfigs[i].depthSize = 24;
            			pConfigs[i].stencilSize = 8;
            			break;
        		    case 1:
            			pConfigs[i].depthSize = 16;
            			pConfigs[i].stencilSize = 0;
            			break;
        		    case 2:
            			pConfigs[i].depthSize = 0;
            			pConfigs[i].stencilSize = 0;
            			break;
        		    case 3:
            			pConfigs[i].depthSize = 32;
            			pConfigs[i].stencilSize = 0;
            			break;
        		}

			pConfigs[i].auxBuffers = 0;
        		pConfigs[i].level = 0;
			if (accum)
        		    pConfigs[i].visualRating = GLX_SLOW_VISUAL_EXT;
			else
        		    pConfigs[i].visualRating = GLX_NONE_EXT;
        		pConfigs[i].transparentPixel = GLX_NONE_EXT;
        		pConfigs[i].transparentRed = 0;
        		pConfigs[i].transparentGreen = 0;
        		pConfigs[i].transparentBlue = 0;
	        	pConfigs[i].transparentAlpha = 0;
        		pConfigs[i].transparentIndex = 0;
        		i++;
		    }
    		}
	    }

	if (i != numConfigs) {
	    xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
		"[dri] Incorrect initialization of visuals.  Disabling DRI.\n");
    		return FALSE;
	}

	break;
    }

    pVia->numVisualConfigs = numConfigs;
    pVia->pVisualConfigs = pConfigs;
    pVia->pVisualConfigsPriv = pVIAConfigs;
    GlxSetVisualConfigs(numConfigs, pConfigs, (void**)pVIAConfigPtrs);

    return TRUE;
}

/*
 * Get DRM version, and do some checks.
 */
static void
ViaDRMGetVersion(ScrnInfoPtr pScrn)
{
    VIAPtr pVia = VIAPTR(pScrn);
    drmVersionPtr drmVersion = drmGetVersion(pVia->drmFD);

    VIAFUNC(pScrn->scrnIndex);

    if (!drmVersion) {
        xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "[drm] unable to retrieve DRM"
                   " version.\n");
        pVia->drmVersion = 0;
    } else {
        xf86DrvMsg(pScrn->scrnIndex, X_INFO,
                   "[drm] using DRM driver version %d.%d.%d (%s)\n",
                   drmVersion->version_major, drmVersion->version_minor,
                   drmVersion->version_patchlevel, drmVersion->date);

        pVia->drmVersion = (drmVersion->version_major << 16) |
            (drmVersion->version_minor << 8) | drmVersion->version_patchlevel;
    }

    /* Check wether we can use DMA for copying */
    if (pVia->UseDMACopy) {
#ifdef __linux__
#ifdef DRM_VIA_DMA_BLIT
        if (pVia->drmVersion >= 0x020705)
            xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Using DMA for "
                       "copying to FB\n");
        else {
            pVia->UseDMACopy = FALSE;
            xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "[drm] Not using DMA for "
                       "copying to FB (drm driver older than 2.7.5).\n");
        }
#else
        pVia->UseDMACopy = FALSE;
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Not using DMA for "
                   "copying to FB (via_drm.h is too old).\n");
#endif /* DRM_VIA_DMA_BLIT */
#else
        pVia->UseDMACopy = FALSE;
        xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Not using DMA for "
                   "copying to FB (only present in linux drm).\n");
#endif /* __linux__ */
    }
}

/*
 * These only came into existence in xorg in may 2005, so provide
 * a fallback. See dri.h for more information.
 */
#ifndef DRIINFO_MAJOR_VERSION
#define DRIINFO_MAJOR_VERSION 4
#endif

#ifndef DRIINFO_MINOR_VERSION
#define DRIINFO_MINOR_VERSION 1
#endif

Bool
VIADRIScreenInit(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    DRIInfoPtr pDRIInfo;
    VIADRIPtr pVIADRI;

    /* if symbols or version check fails, we still want this to be NULL */
    pVia->pDRIInfo = NULL;

    /* Check that the GLX, DRI, and DRM modules have been loaded by testing
    * for canonical symbols in each module. */
    if (!xf86LoaderCheckSymbol("GlxSetVisualConfigs")) return FALSE;
    if (!xf86LoaderCheckSymbol("DRIScreenInit"))       return FALSE;
    if (!xf86LoaderCheckSymbol("drmAvailable"))        return FALSE;
    if (!xf86LoaderCheckSymbol("DRIQueryVersion")) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                 "[dri] VIADRIScreenInit failed (libdri.a too old)\n");
        return FALSE;
    }

    /* Check the DRI version */
    {
        int major, minor, patch;
        DRIQueryVersion(&major, &minor, &patch);
        if ((major != DRIINFO_MAJOR_VERSION) || (minor < DRIINFO_MINOR_VERSION)) {
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                       "DRI version mismatch. Disabling DRI.\n");
            xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
                       "libDRI %d.%d.%d is present while %d.%d.0 is needed.\n",
                       major, minor, patch, DRIINFO_MAJOR_VERSION,
                       DRIINFO_MINOR_VERSION);
            return FALSE;
        }
    }

    pVia->pDRIInfo = DRICreateInfoRec();
    if (!pVia->pDRIInfo)
	return FALSE;

    pDRIInfo = pVia->pDRIInfo;
    pDRIInfo->drmDriverName = VIAKernelDriverName;
    pDRIInfo->clientDriverName = VIAClientDriverName;
    pDRIInfo->busIdString = xalloc(64);
    sprintf(pDRIInfo->busIdString, "PCI:%d:%d:%d",
        ((pciConfigPtr)pVia->PciInfo->thisCard)->busnum,
        ((pciConfigPtr)pVia->PciInfo->thisCard)->devnum,
        ((pciConfigPtr)pVia->PciInfo->thisCard)->funcnum);
    pDRIInfo->ddxDriverMajorVersion = VIA_DRIDDX_VERSION_MAJOR;
    pDRIInfo->ddxDriverMinorVersion = VIA_DRIDDX_VERSION_MINOR;
    pDRIInfo->ddxDriverPatchVersion = VIA_DRIDDX_VERSION_PATCH;

#if DRIINFO_MAJOR_VERSION < 5
    pDRIInfo->frameBufferPhysicalAddress = pVia->FrameBufferBase;
#else
    pDRIInfo->frameBufferPhysicalAddress = (pointer) pVia->FrameBufferBase;
#endif

    pDRIInfo->frameBufferSize = pVia->videoRambytes;

    pDRIInfo->frameBufferStride = (pScrn->displayWidth *
					    pScrn->bitsPerPixel / 8);
    pDRIInfo->ddxDrawableTableEntry = VIA_MAX_DRAWABLES;

    if (SAREA_MAX_DRAWABLES < VIA_MAX_DRAWABLES)
	pDRIInfo->maxDrawableTableEntry = SAREA_MAX_DRAWABLES;
    else
	pDRIInfo->maxDrawableTableEntry = VIA_MAX_DRAWABLES;

#ifdef NOT_DONE
    /* FIXME need to extend DRI protocol to pass this size back to client
    * for SAREA mapping that includes a device private record
    */
    pDRIInfo->SAREASize =
	((sizeof(XF86DRISAREARec) + 0xfff) & 0x1000); /* round to page */
    /* + shared memory device private rec */
#else
    /* For now the mapping works by using a fixed size defined
    * in the SAREA header
    */
    if ((sizeof(XF86DRISAREARec) + sizeof(drm_via_sarea_t)) > SAREA_MAX) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "Data does not fit in SAREA\n");
	DRIDestroyInfoRec(pVia->pDRIInfo);
	pVia->pDRIInfo = NULL;
	return FALSE;
    }
    pDRIInfo->SAREASize = SAREA_MAX;
#endif

    if (!(pVIADRI = (VIADRIPtr)xcalloc(sizeof(VIADRIRec),1))) {
	DRIDestroyInfoRec(pVia->pDRIInfo);
	pVia->pDRIInfo = NULL;
	return FALSE;
    }
    pDRIInfo->devPrivate = pVIADRI;
    pDRIInfo->devPrivateSize = sizeof(VIADRIRec);
    pDRIInfo->contextSize = sizeof(VIADRIContextRec);

    pDRIInfo->CreateContext = VIACreateContext;
    pDRIInfo->DestroyContext = VIADestroyContext;
    pDRIInfo->SwapContext = VIADRISwapContext;
    pDRIInfo->InitBuffers = VIADRIInitBuffers;
    pDRIInfo->MoveBuffers = VIADRIMoveBuffers;
    pDRIInfo->bufferRequests = DRI_ALL_WINDOWS;

    if (!DRIScreenInit(pScreen, pDRIInfo, &pVia->drmFD)) {
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR,
	    "[dri] DRIScreenInit failed.  Disabling DRI.\n");
	xfree(pDRIInfo->devPrivate);
	pDRIInfo->devPrivate = NULL;
	DRIDestroyInfoRec(pVia->pDRIInfo);
	pVia->pDRIInfo = NULL;
	pVia->drmFD = -1;
	return FALSE;
    }

    if (!(VIAInitVisualConfigs(pScrn))) {
	VIADRICloseScreen(pScrn, pScreen);
	return FALSE;
    }
    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] visual configs initialized.\n" );

    /* DRIScreenInit doesn't add all the common mappings.  Add additional mappings here. */
    if (!VIADRIMapInit(pVia)) {
	VIADRICloseScreen(pScrn, pScreen);
	return FALSE;
    }
    pVIADRI->regs.size = VIA_MMIO_REGSIZE;
    pVIADRI->regs.handle = pVia->registerHandle;
    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] mmio Registers = 0x%08lx\n",
               (unsigned long) pVIADRI->regs.handle);

    pVIADRI->drixinerama = pVia->drixinerama;

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] mmio mapped.\n" );

    ViaDRMGetVersion(pScrn);

    return TRUE;
}

void
VIADRICloseScreen(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIADRIPtr pVIADRI;

    VIADRIRingBufferCleanup(pScrn);
    if (pVia->agpSize) {
	drmUnmap(pVia->agpMappedAddr,pVia->agpSize);
	drmRmMap(pVia->drmFD,pVia->agpHandle);
	drmAgpUnbind(pVia->drmFD, pVia->agpHandle);
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Freeing agp memory\n");
	drmAgpFree(pVia->drmFD, pVia->agpHandle);
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[drm] Releasing agp module\n");
	drmAgpRelease(pVia->drmFD);
    }

    DRICloseScreen(pScreen);

    if (pVia->pDRIInfo) {
	if ((pVIADRI = (VIADRIPtr) pVia->pDRIInfo->devPrivate)) {
	    VIADRIIrqExit(pVia, pVIADRI);
    	    xfree(pVIADRI);
    	    pVia->pDRIInfo->devPrivate = NULL;
	}
	DRIDestroyInfoRec(pVia->pDRIInfo);
	pVia->pDRIInfo = NULL;
    }

    if (pVia->pVisualConfigs) {
	xfree(pVia->pVisualConfigs);
	pVia->pVisualConfigs = NULL;
    }
    if (pVia->pVisualConfigsPriv) {
	xfree(pVia->pVisualConfigsPriv);
	pVia->pVisualConfigsPriv = NULL;
    }
}

/* TODO: xserver receives driver's swapping event and does something
 *       according the data initialized in this function.
 */
static Bool
VIACreateContext(ScreenPtr pScreen, VisualPtr visual,
          drm_context_t hwContext, void *pVisualConfigPriv,
          DRIContextType contextStore)
{
    return TRUE;
}

static void
VIADestroyContext(ScreenPtr pScreen, drm_context_t hwContext,
           DRIContextType contextStore)
{
}

Bool
VIADRIFinishScreenInit(ScrnInfoPtr pScrn, ScreenPtr pScreen)
{
    VIAPtr pVia = VIAPTR(pScrn);
    VIADRIPtr pVIADRI;

    pVia->pDRIInfo->driverSwapMethod = DRI_HIDE_X_CONTEXT;

    pVia->IsPCI = !VIADRIAgpInit(pVia);

    if (pVia->IsPCI) {
	VIADRIPciInit(pVia);
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] use pci.\n" );
    }
    else
	xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] use agp.\n" );

    if (!(VIADRIFBInit(pVia))) {
	VIADRICloseScreen(pScrn, pScreen);
	xf86DrvMsg(pScrn->scrnIndex, X_ERROR, "[dri] frame buffer initialization failed.\n" );
	return FALSE;
    }

    xf86DrvMsg(pScrn->scrnIndex, X_INFO, "[dri] frame buffer initialized.\n" );

    DRIFinishScreenInit(pScreen);

    if (!VIADRIKernelInit(pVia)) {
	VIADRICloseScreen(pScrn, pScreen);
	return FALSE;
    }
    xf86DrvMsg(pScreen->myNum, X_INFO, "[dri] kernel data initialized.\n");

    /* set SAREA value */
    {
        drm_via_sarea_t *saPriv;

	saPriv = (drm_via_sarea_t *) DRIGetSAREAPrivate(pScreen);
	assert(saPriv);
	memset(saPriv, 0, sizeof(*saPriv));
	saPriv->ctxOwner = -1;
    }
    pVIADRI=(VIADRIPtr)pVia->pDRIInfo->devPrivate;
    pVIADRI->deviceID=pVia->Chipset;
    pVIADRI->width=pScrn->virtualX;
    pVIADRI->height=pScrn->virtualY;
    pVIADRI->mem=pScrn->videoRam*1024;
    pVIADRI->bytesPerPixel= (pScrn->bitsPerPixel+7) / 8;
    pVIADRI->sarea_priv_offset = sizeof(XF86DRISAREARec);
    /* TODO */
    pVIADRI->scrnX=pVIADRI->width;
    pVIADRI->scrnY=pVIADRI->height;

    /* Initialize IRQ */
    if (pVia->DRIIrqEnable)
	VIADRIIrqInit(pVia, pVIADRI);

    pVIADRI->ringBufActive = 0;
    VIADRIRingBufferInit(pScrn);
    return TRUE;
}

static void
VIADRISwapContext(ScreenPtr pScreen, DRISyncType syncType,
           DRIContextType oldContextType, void *oldContext,
           DRIContextType newContextType, void *newContext)
{
  /*ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */
  return;
}

static void
VIADRIInitBuffers(WindowPtr pWin, RegionPtr prgn, CARD32 index)
{
  /*ScreenPtr pScreen = pWin->drawable.pScreen;
  ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */
  return;
}

static void
VIADRIMoveBuffers(WindowPtr pParent, DDXPointRec ptOldOrg,
           RegionPtr prgnSrc, CARD32 index)
{
  /*ScreenPtr pScreen = pParent->drawable.pScreen;
  ScrnInfoPtr pScrn = xf86Screens[pScreen->myNum];
  VIAPtr pVia = VIAPTR(pScrn);
  */
  return;
}

/* Initialize the kernel data structures. */
static Bool VIADRIKernelInit(VIAPtr pVia)
{
    drm_via_init_t drmInfo;
    memset(&drmInfo, 0, sizeof(drm_via_init_t));
    drmInfo.func = VIA_INIT_MAP;
    drmInfo.sarea_priv_offset   = sizeof(XF86DRISAREARec);
    drmInfo.fb_offset           = pVia->FrameBufferBase;
    drmInfo.mmio_offset         = pVia->registerHandle;
    if (pVia->IsPCI)
	drmInfo.agpAddr = (CARD32)NULL;
    else
	drmInfo.agpAddr = (CARD32)pVia->agpAddr;

	if ((drmCommandWrite(pVia->drmFD, DRM_VIA_MAP_INIT,&drmInfo,
			     sizeof(drm_via_init_t))) < 0)
	    return FALSE;

    return TRUE;
}

/* Add a map for the MMIO registers */
static Bool VIADRIMapInit(VIAPtr pVia)
{
    int flags = DRM_READ_ONLY;

    if (drmAddMap(pVia->drmFD, pVia->MmioBase, VIA_MMIO_REGSIZE,
		  DRM_REGISTERS, flags, &pVia->registerHandle) < 0) {
	return FALSE;
    }

    xf86DrvMsg(pVia->scrnIndex, X_INFO, "[drm] register handle = 0x%08lx\n",
               (unsigned long) pVia->registerHandle);

    return TRUE;
}
