/*
 * Copyright 2004-2005 Luc Verhaegen.
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sub license,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice (including the
 * next paragraph) shall be included in all copies or substantial portions
 * of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
 * THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 * DEALINGS IN THE SOFTWARE.
 */

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include "via_driver.h"
#include "via_id.h"

/*
 * Since we are going to hold a rather big structure with
 * basic card-id information, we might as well seperate this
 * into its own file.
 *
 */

/*
 * Known missing devices:
 * VT3122:
 * Biostar M6VLQ Grand
 * Biostar M6VLQ Pro
 * PcChips M791G
 * Soltek SL-B6A-F800 (C3 800Mhz onboard)
 * Soltek SL-B6A-F1000 (Qbic IQ3601 | C3 1Ghz onboard)
 * + loads of semi-embedded devices.
 *
 * VT7205:
 * ECS KM400-M
 * ECS KM400A-M2
 * FIC K7M-400
 * PcChips M851G
 * PcChips M851AG
 * Soltek SL-B7C-FGR (Qbic EQ3704 | VT7205A)
 * Soyo SY-K7VMP
 * Soyo SY-K7VMP2
 *
 * VT3108:
 * Abit KV8-MAX3
 * Abit KV8
 * Biostar Ideq 210V (or VT7205A?)
 * Biostar M7VIZ
 * Chaintech MK8M800
 * Epox EP-8KMM5I (or VT7205A?)
 * FIC K8M-800M
 * Packard Bell Easynote B35x0 (?== B3 800??)
 * PcChips A21G
 * Soltek SL-B9C-FGR (Qbic EQ3802-300P)
 * Soltek SL-K8M800I-R
 *
 * VT3118:
 * Twinhead 12BL
 * Biostar Ideq 210M
 * FIC P4M800
 * FIC P4M-800M
 * Gigabyte 8VM800M
 * PcChips M955G
 * PcChips M957G
 * Soltek SL-PM800I
 * Soltek SL-PM800I-R
 * Soltek SL-PM800
 * Soyo SY-P4VM800
 *
 */
/*
 * There's no reason for this to be known outside of via_id.o
 * Only a pointer to an single entry will ever be used outside.
 *
 */
static struct ViaCardId ViaCardIds[] = {
    /* VT3122 */
    {"LT21 VA28",                                   VT3122, 0x1019, 0x1B44, FALSE },
    {"ECS G320",                                    VT3122, 0x1019, 0xB320, TRUE  },
    {"Asustek Terminator C3V",                      VT3122, 0x1043, 0x8155, FALSE },
    /* 1106:3122 shared with: All CLE266 based EPIAs; Jetway 627FW; PcChips 789CG */
    {"VIA EPIA M/MII/...",                          VT3122, 0x1106, 0x3122, FALSE },
    {"Clevo T200/210V",                             VT3122, 0x1558, 0x200A, TRUE  }, /* aka "Element Computers Helium 2100" */
    /* VT7205 */
    {"ECS KM400-M2",                                VT7205, 0x1019, 0x1842, FALSE },
    {"Acer Aspire 135x",                            VT7205, 0x1025, 0x0033, TRUE  },
    {"Asustek A7V8X-MX",                            VT7205, 0x1043, 0x80ED, FALSE },
    {"Asustek A7V8X-LA",                            VT7205, 0x1043, 0x80F9, FALSE },
    {"Asustek A7V8X-MX SE/A7V400-MX",               VT7205, 0x1043, 0x8118, FALSE },
    {"Asustek Terminator A7VT",                     VT7205, 0x1043, 0x813E, FALSE },
    {"Mitac 8375X",                                 VT7205, 0x1071, 0x8375, TRUE  }, /* aka "UMAX 585T" */
    {"Soltek SL-75MIV2",                            VT7205, 0x1106, 0x0000, FALSE }, /* VIA/0x0000 */
    {"Soltek SL-B7C-FGR",                           VT7205, 0x1106, 0x3205, FALSE }, /* ids are: VIA and KM400 hostbridge. */
    {"Biostar iDEQ 200V/Chaintech 7VIF4",           VT7205, 0x1106, 0x7205, FALSE }, /* VIA/VT7205 -- 2 distinct devices */
    {"Shuttle FX43",                                VT7205, 0x1297, 0xF643, FALSE },
    {"Giga-byte 7VM400(A)M",                        VT7205, 0x1458, 0xD000, FALSE }, /* 7VM400M, GA-7VM400AM */
    {"MSI KM4(A)M-V",                               VT7205, 0x1462, 0x7061, FALSE }, /* aka "DFI KM400-MLV" */
    {"MSI PM8M2-V",                                 VT7205, 0x1462, 0x7071, FALSE },
    {"MSI KM4(A)M-L",                               VT7205, 0x1462, 0x7348, FALSE },
    {"Abit VA-10 (1)",                              VT7205, 0x147B, 0x140B, FALSE },
    {"Abit VA-10 (2)",                              VT7205, 0x147B, 0x140C, FALSE },
    {"Abit VA-20",                                  VT7205, 0x147B, 0x1411, FALSE },
    {"Averatec 322x",                               VT7205, 0x14FF, 0x030D, TRUE  }, /* unknown twinhead rebadged */
    {"FIC K7M-400A",                                VT7205, 0x1509, 0x9233, FALSE },
    {"Biostar P4M800-M7",                           VT7205, 0x1565, 0x1202, FALSE },
    {"Uniwill 755CI",                               VT7205, 0x1584, 0x800A, TRUE  }, /* aka "Gericom hummer advance", "Maxdata M-Book 1200X" */
    {"Packard Bell Quasar2 (MSI MS6786)",           VT7205, 0x1631, 0xD002, FALSE },
    {"Epox EP-8KMM3I",                              VT7205, 0x1695, 0x9023, FALSE },
    {"ASRock Inc. K7VM2/3/4",                       VT7205, 0x1849, 0x7205, FALSE },
    {"Acorp 7KM400QP",                              VT7205, 0x1915, 0x1100, FALSE }, /* optional VT1623 */
    {"Soyo K7VME",                                  VT7205, 0xA723, 0x10FD, FALSE },
    /* VT3108 */
    {"ECS 536",                                     VT3108, 0x1019, 0x0F60, TRUE  },
    {"ECS K8M800-M2 (1.0)",                         VT3108, 0x1019, 0x1828, FALSE },
    {"ECS K8M800-M2 (2.0)",                         VT3108, 0x1019, 0x1B45, FALSE },
    {"Acer Aspire 136x",                            VT3108, 0x1025, 0x006E, TRUE  },
    {"Asustek K8V-MX",                              VT3108, 0x1043, 0x8129, FALSE },
    {"Mitac 8399",                                  VT3108, 0x1071, 0x8399, TRUE  }, /* aka "pogolinux konabook 3100" */
    {"Mitac 8889",                                  VT3108, 0x1071, 0x8889, TRUE  }, /* aka "BenQ Joybook R23" */
    {"PCChips M861G",                               VT3108, 0x1106, 0x0204, FALSE }, /* ids are: VIA and K8M800 hostbridge. -- bloody PCChips */
    {"Asustek A8V-MX",                              VT3108, 0x1106, 0x3108, FALSE }, /* VIA/VT3108: First ASUS id misstep. */
    {"Shuttle FX21",                                VT3108, 0x1297, 0x3052, FALSE },
    {"Shuttle FX83",                                VT3108, 0x1297, 0xF683, FALSE },
    {"Sharp Actius AL27",                           VT3108, 0x13BD, 0x1044, TRUE  },
    {"Giga-byte GA-K8VM800M",                       VT3108, 0x1458, 0xD000, FALSE },
    {"MSI K8M Neo-V",                               VT3108, 0x1462, 0x0320, FALSE },
    {"MSI K8MM-V",                                  VT3108, 0x1462, 0x7142, FALSE },
    {"MSI K8MM3-V",                                 VT3108, 0x1462, 0x7181, FALSE },
    {"MSI K8MM-ILSR",                               VT3108, 0x1462, 0x7410, FALSE },
    {"Abit KV-80",                                  VT3108, 0x147B, 0x1419, FALSE },
    {"Abit KV-81",                                  VT3108, 0x147B, 0x141A, FALSE },
    {"Twinhead 12K(64)",                            VT3108, 0x14FF, 0x0315, TRUE  }, /* aka "Averatec 327x" */
    {"Twinhead 14K",                                VT3108, 0x14FF, 0x0321, TRUE  },
    {"Twinhead 12KF",                               VT3108, 0x14FF, 0x0322, TRUE  }, /* aka "Averatec 37xx" */
    {"FIC-PCG AN5W",                                VT3108, 0x1509, 0x3930, TRUE  }, /* aka "Averatec 54xx" */
    {"FIC K8M-800M",                                VT3108, 0x1509, 0x6001, FALSE },
    {"Clevo L570W",                                 VT3108, 0x1558, 0x0570, TRUE  },
    {"Clevo D400J",                                 VT3108, 0x1558, 0x4702, TRUE  },
    {"Biostar K8VGA-M",                             VT3108, 0x1565, 0x1203, FALSE },
    {"DFI K8M800-MLVF",                             VT3108, 0x15BD, 0x1002, FALSE },
    {"Packard Bell Easynote E6116/E63xx",           VT3108, 0x1631, 0xC008, TRUE  },
    {"Packard Bell Easynote B3 800/B3340",          VT3108, 0x1631, 0xC009, TRUE  },
    {"Packard Bell Bali (Giga-byte GA-K8VM800MNF)", VT3108, 0x1631, 0xD007, FALSE },
    {"Fujitsu/Siemens Amilo K7610",                 VT3108, 0x1734, 0x10B3, TRUE  },
    {"Asrock K8Upgrade-VM800",                      VT3108, 0x1849, 0x3108, FALSE },
    /* VT3118 */
    {"ECS PM800-M2",                                VT3118, 0x1106, 0x3118, FALSE }, /* VIA/VT3118 */
    {"Mitac 8666",                                  VT3118, 0x1071, 0x8666, TRUE  },
    {"Mitac 8965",                                  VT3118, 0x1071, 0x8965, TRUE  }, /* aka "Medion MIM2080"; could very well be the "*/
    {"TwinHead E12BL",                              VT3118, 0x14FF, 0x0314, TRUE  },
    {"Clevo M5xxV",                                 VT3118, 0x1558, 0x5402, TRUE  }, /* M540V/M550V/M541V/M551V/M545V/M555V aka "MaxSelect Optima C4" == M55[05]V */
    {"Biostar P4VMA-M",                             VT3118, 0x1565, 0x1202, FALSE },
    {"Sotec Winbook WA2330S5",                      VT3118, 0x161F, 0x2037, TRUE  },
    {"Packard Bell Easynote R1100",                 VT3118, 0x1631, 0xC015, TRUE  },
    {"Fujitsu/Siemens Amilo Pro V2010",             VT3118, 0x1734, 0x1078, TRUE  },
    {"Fujitsu/Siemens Amilo L73x0",                 VT3118, 0x1734, 0x10AB, TRUE  }, /* L7300, L7310 */
    {"Asrock P4VM8",                                VT3118, 0x1849, 0x3118, FALSE },
    {"Chaintech MPM800",                            VT3118, 0x270F, 0x7671, FALSE },
    /* VT3344 */
    {"ECS P4M800PRO-M",                             VT3344, 0x1019, 0x2122, FALSE },
    {"Gateway MX3210",                              VT3344, 0x107B, 0x0216, TRUE  },
    {"Giga-byte GA-8VM800M",                        VT3344, 0x1458, 0xD000, FALSE },
    {"MSI PM8M-V",                                  VT3344, 0x1462, 0x7104, FALSE },
    {"MSI PM8M3-V",                                 VT3344, 0x1462, 0x7211, FALSE },
    {"MSI PM8PM",                                   VT3344, 0x1462, 0x7222, FALSE },
    {"Biostar P4M80-M4",                            VT3344, 0x1565, 0x1202, FALSE },
    {"Fujitsu/Siemens Amilo Pro V2030",             VT3344, 0x1734, 0x109B, TRUE  },
    {"Asrock Inc. P4VM800",                         VT3344, 0x1849, 0x3344, FALSE },
    {"Asustek P5V800-MX",                           VT3344, 0x3344, 0x1122, FALSE },
    /* keep this */
    {NULL, VIA_UNKNOWN, 0x0000, 0x0000, FALSE}
};

/*
 *
 */
void
ViaCheckCardId(ScrnInfoPtr pScrn)
{
    struct ViaCardId *Id;
    VIAPtr pVia = VIAPTR(pScrn);
    
    if ((pVia->PciInfo->subsysVendor == pVia->PciInfo->vendor) &&
	(pVia->PciInfo->subsysCard == pVia->PciInfo->chipType))
	xf86DrvMsg(pScrn->scrnIndex, X_WARNING, "Manufacturer plainly copied main PCI"
		   " ids to Subsystem/Card ids.\n");
    
    for (Id = ViaCardIds; Id->String; Id++) {
	if ((Id->Chip == pVia->Chipset) && 
	    (Id->Vendor == pVia->PciInfo->subsysVendor) &&
	    (Id->Device == pVia->PciInfo->subsysCard)) {
	    xf86DrvMsg(pScrn->scrnIndex, X_PROBED, "Detected %s.\n", Id->String);
	    pVia->Id = Id;
	    return;
	}
    }
    
    xf86DrvMsg(pScrn->scrnIndex, X_ERROR, 
	       "Unknown Card-Ids (%4X|%4X), report this to unichrome-devel@lists.sf.net ASAP\n"
	       , pVia->PciInfo->subsysVendor, pVia->PciInfo->subsysCard);
    pVia->Id = NULL;
}
