/* validate.c -- verify that an alphabet file matches a file of golden strokes

   Copyright (C) 2000 Carl Worth

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#include "args.h"
#include "rec.h"

static void usage(char *argv[]);
static int validate_golden_file(rec_t *rec, char *filename);
static char *strip_whitespace(char *str);

int main(int argc, char *argv[])
{
    int err, optind;
    args_t args;
    rec_t rec;

    args_init(&args);
    err = args_parse(&args, argc, argv);
    optind = args_parse(&args, argc, argv);
    if (optind < 0 || optind >= argc) {
	usage(argv);
	exit(1);
    }

    err = rec_init(&rec, &args, 0, 0);
    if (err) {
	fprintf(stderr, "%s: An error occurred during rec_init\n", __FUNCTION__);
	return err;
    }
    args_deinit(&args);

    err = 0;
    while (optind < argc) {
	err += validate_golden_file(&rec, argv[optind++]);
    }

    if (err) {
	fprintf(stderr, "%s: *** ERROR: %d Inconsistenc%s found, (see above).\n",
		argv[0], err, err > 1 ? "ies" : "y");
	return err;
    } else {
	fprintf(stderr, "%s: Perfect. No inconsistencies found.\n", argv[0]);
	return 0;
    }
}

static void usage(char *argv[])
{
    fprintf(stderr, "Usage: %s [-a <alphabet_file>] <golden_stroke_file> [...]\n",
	    argv[0]);
}

#define MAX_LINE_LENGTH 255
static int validate_golden_file(rec_t *rec, char *filename)
{
    FILE *file;
    int line_num = 0;
    int errors;
    char buf[MAX_LINE_LENGTH];
    char *a_str, *seq, *rec_a_str;
    char *mode_str;
    rec_mode_t *mode = rec->global_mode;
    action_t *action;
    int done;
    double probability;
    char *classifications[1];

    file = fopen(filename, "r");
    if (!file) {
	fprintf(stderr, "%s: failed to open file %s\n", __FUNCTION__, filename);
	perror(__FUNCTION__);
	return 1;
    }

    fprintf(stderr, "Validating %s\n", filename);

    errors = 0;
    while (fgets(buf, MAX_LINE_LENGTH, file)) {
	char *p;
	line_num++;

	p = buf;
	while (isspace(p[0]))
	    p++;
	if (p[0] == '\0' || p[0] == '#')
	    goto NEXT_LINE;

	mode_str = strip_whitespace(buf);
	if (mode_str[0] == '[') {
	    mode_str = strtok(buf + 1, ":]");
	    if (mode_str && strlen(mode_str)) {
		mode = rec_get_mode(rec, mode_str);
	    }
	    goto NEXT_LINE;
	}

        seq = strtok(buf, "=");
	a_str = strtok(NULL, "\n");
	if (a_str == NULL) {
	    fprintf(stderr, "%s: Syntax error on line: %s\n", __FUNCTION__, buf);
	    goto NEXT_LINE;
	}
	if (a_str[0] == '>')
	    a_str++;

	a_str = strip_whitespace(a_str);
	seq = strip_whitespace(seq);
	while (*a_str == '*' || isspace(*a_str)) {
	    a_str++;
	}

	classifications[0] = seq;
	probability = 0.0;
	done = rec_mode_recognize(mode, (void **) classifications,
				  &action, &probability);
	if (! done) {
	    rec_mode_recognize(rec->global_mode, (void **) classifications,
			       &action, &probability);
	}

	rec_a_str = action_str(action);
	if (strcmp(a_str, rec_a_str)) {
	    fprintf(stderr, "ERROR: %s => `%s' instead of `%s'\n",
		    seq, action ? rec_a_str : "<unknown>", a_str);
	    errors++;
	}
	free(rec_a_str);
	
      NEXT_LINE: ;
    }

    return errors;
}

/* Return a pointer the the given string with initial and trailing
 * whitespace removed. Warning: This function modifies the original
 * string.
 */
static char *strip_whitespace(char *str)
{
    if (str && *str) {
	int n;
	while (*str && isspace (*str)) str++;
	
	n = strlen (str) - 1;
	while (n > 0 && isspace (str [n])) n--;
	str [n+1] = '\0';
    }
    return str;
}
