/*******************************************************************
 * File: cardif_hostap_driver.c
 *
 * A significant portion of this code is either based on, or taken directly
 * from WPA Supplicant.  The copyright and license information for WPA
 * Supplicant is reproduced below.  Code taken from WPA Supplicant is notated
 * in comments above the function definitions.
 *
 * All other code is licensed under a dual GPL/BSD license.  (See LICENSE file 
 * for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_hostap_driver.c,v 1.13 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_hostap_driver.c,v $
 * Revision 1.13  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.12  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

/*
 * WPA Supplicant - driver interaction with Linux Host AP driver
 * Copyright (c) 2003-2005, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */

#ifdef LINUX_FRAMER

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>

typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;
typedef int64_t s64;
typedef int32_t s32;
typedef int8_t s8;

#include "xsup_debug.h"
#include "xsup_err.h"
#include "config.h"
#include "profile.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/linux/hostap_common.h"
#include "wpa.h"

// Taken from WPA supplicant
static int hostapd_ioctl(const char *dev, struct prism2_hostapd_param *param,
			 int len, int show_err)
{
	int s;
	struct iwreq iwr;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		debug_printf(DEBUG_NORMAL, "Error getting socket for %s!\n",
			     dev);
		return -1;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, dev, IFNAMSIZ);
	iwr.u.data.pointer = (caddr_t) param;
	iwr.u.data.length = len;

	if (ioctl(s, PRISM2_IOCTL_HOSTAPD, &iwr) < 0) {
		int ret;
		close(s);
		ret = errno;
		if (show_err) 
			debug_printf(DEBUG_NORMAL, "ioctl[PRISM2_IOCTL_HOSTAPD]\n");
		return ret;
	}
	close(s);

	return 0;
}

// Taken from wpa_supplicant.
static int prism2param(const char *ifname, int param, int value)
{
	struct iwreq iwr;
	int *i, s, ret = 0;

	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
		debug_printf(DEBUG_NORMAL, "socket[PF_INET,SOCK_DGRAM]\n");
		return -1;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ);
	i = (int *) iwr.u.name;
	*i++ = param;
	*i++ = value;

	if (ioctl(s, PRISM2_IOCTL_PRISM2_PARAM, &iwr) < 0) {
		debug_printf(DEBUG_NORMAL, "ioctl[PRISM2_IOCTL_PRISM2_PARAM]\n");
		ret = -1;
	}
	close(s);
	return ret;
}

// Derived from WPA supplicant
int cardif_hostap_driver_set_wpa_ie(char *dev, const char *wpa_ie, 
				    size_t wpa_ie_len)
{
	struct prism2_hostapd_param *param;
	int res;
       	size_t blen = PRISM2_HOSTAPD_GENERIC_ELEMENT_HDR_LEN + wpa_ie_len;

	if (blen < sizeof(*param))
		blen = sizeof(*param);

	param = (struct prism2_hostapd_param *) malloc(blen);
	if (param == NULL)
		return -1;

	memset(param, 0, blen);
	param->cmd = PRISM2_HOSTAPD_SET_GENERIC_ELEMENT;
      	param->u.generic_elem.len = wpa_ie_len;
       	memcpy(param->u.generic_elem.data, wpa_ie, wpa_ie_len);
	res = hostapd_ioctl(dev, param, blen, 1);

	free(param);

	return res;
}

int cardif_hostap_driver_get_wpa_ie(struct interface_data *intdata,
				    char *iedata, int *ielen)
{
  // Use an IE that doesn't include the capabilities bytes.
  wpa_gen_ie(intdata, iedata);
  *ielen = 24;
  return XENONE;
}

int cardif_hostap_driver_get_wpa2_ie(struct interface_data *intdata,
				     char *iedata, int *ielen)
{
  debug_printf(DEBUG_NORMAL, "WPA2/802.11i is not implemented for hostap "
	       "driver.\n");

  return XENOTSUPPORTED;
}

int cardif_hostap_driver_wpa(struct interface_data *intdata,  char endis)
{
  if (prism2param(intdata->intName, PRISM2_PARAM_WPA, endis) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable WPA on interface %s!\n",
		   intdata->intName);
    }

  return XENONE;
}

int cardif_hostap_driver_drop_unencrypted(struct interface_data *intdata,
					  char endis)
{
  return prism2param(intdata->intName, PRISM2_PARAM_DROP_UNENCRYPTED, endis);
}

int cardif_hostap_driver_wpa_state(struct interface_data *thisint,  char endis)
{
  char wpaie[24];
  int len;

   // DKS -- if endis is FALSE, we need to turn off the wpa_ie, if we turned it on.
   // normally we'd reset the ie to whatever it was before, but as we may have stomped
   // on it already, let's just stomp on it again.
  if (endis) // enable
  {
	  cardif_hostap_driver_get_wpa_ie(thisint, wpaie, &len);

	  if (cardif_hostap_driver_set_wpa_ie(thisint->intName, wpaie, len) < 0)
	  {
		  debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE on device %s!\n",
					   thisint->intName);
	  }
   } else { // disable
	   memset(wpaie,0,sizeof(wpaie));
	   if (cardif_hostap_driver_set_wpa_ie(thisint->intName, wpaie, 0) < 0)
	   {
		   debug_printf(DEBUG_NORMAL, "Couldn't clear WPA IE on device %s!\n",
						thisint->intName);
	   }
   }

  /*
  if (prism2param(dev, PRISM2_PARAM_HOST_ROAMING, 2) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't change roaming on interface %s!\n",
		   dev);
    }
  */
  if (prism2param(thisint->intName, PRISM2_PARAM_PRIVACY_INVOKED, endis) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable/disable privacy on interface %s!\n",
		   thisint->intName);
    }
  
  if (prism2param(thisint->intName, PRISM2_PARAM_WPA, endis) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable/disable WPA on interface %s!\n",
		   thisint->intName);
    }

  if (cardif_hostap_driver_drop_unencrypted(thisint, 0) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable/disable \"drop unencrypted"
		   " packets\" on interface %s!\n", thisint->intName);
    }

  return XENONE;
}

int cardif_hostap_driver_countermeasures(struct interface_data *intdata,
					 char endis)
{
  return prism2param(intdata->intName, PRISM2_PARAM_TKIP_COUNTERMEASURES,
		     endis);
}

int cardif_hostap_driver_wpa_disable(char *intname)
{
  // XXX Finish!
  return XENONE;
}

// Taken from wpa_supplicant.
static void show_set_key_error(struct prism2_hostapd_param *param)
{
	switch (param->u.crypt.err) {
	case HOSTAP_CRYPT_ERR_UNKNOWN_ALG:
		debug_printf(DEBUG_NORMAL, "Unknown algorithm '%s'.",
			     param->u.crypt.alg);
		debug_printf(DEBUG_NORMAL, "You may need to load kernel module to "
			   "register that algorithm.");
		debug_printf(DEBUG_NORMAL, "E.g., 'modprobe hostap_crypt_wep' for "
			   "WEP.");
		break;
	case HOSTAP_CRYPT_ERR_UNKNOWN_ADDR:
		debug_printf(DEBUG_NORMAL, "Unknown address " MACSTR ".",
			   MAC2STR(param->sta_addr));
		break;
	case HOSTAP_CRYPT_ERR_CRYPT_INIT_FAILED:
		debug_printf(DEBUG_NORMAL, "Crypt algorithm initialization failed.");
		break;
	case HOSTAP_CRYPT_ERR_KEY_SET_FAILED:
		debug_printf(DEBUG_NORMAL, "Key setting failed.");
		break;
	case HOSTAP_CRYPT_ERR_TX_KEY_SET_FAILED:
		debug_printf(DEBUG_NORMAL, "TX key index setting failed.");
		break;
	case HOSTAP_CRYPT_ERR_CARD_CONF_FAILED:
		debug_printf(DEBUG_NORMAL, "Card configuration failed.");
		break;
	default:
	  debug_printf(DEBUG_NORMAL, "Unknown error (%d)\n", param->u.crypt.err);
	  break;
	}
}

int cardif_hostap_driver_set_key(char *dev, int alg, unsigned char *addr,
			      int key_idx, int set_tx, char *seq, int seq_len,
			      char *key, int key_len)
{
  struct prism2_hostapd_param *param;
  int blen;
  int ret = XENONE;
  char *buf;
  char *alg_name;

  switch (alg)
    {
    case WPA_NONE:
      alg_name = "none";
      break;

    case WPA_WEP:
      alg_name = "WEP";
      break;

    case WPA_TKIP:
      alg_name = "TKIP";
      break;

    case WPA_CCMP:
      alg_name = "CCMP";
      break;
      
    default:
      debug_printf(DEBUG_NORMAL, "Invalid encryption type requested! (%d)\n",
		   alg);
      return -1;
    }

  debug_printf(DEBUG_NORMAL, "alg=%s  key_idx=%d set_tx=%d seq_len=%d key_len=%d\n", alg_name, key_idx, set_tx, seq_len, key_len);

  debug_printf(DEBUG_NORMAL, "KeyRSC = ");
  debug_hex_printf(DEBUG_NORMAL, seq, seq_len);

  if (seq_len > 8)
    {
      debug_printf(DEBUG_NORMAL, "seq_len > 8!\n");
      return -1;
    }

  blen = sizeof(*param) + key_len;
  buf = malloc(blen);
  if (buf == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  bzero(buf, blen);

  param = (struct prism2_hostapd_param *)buf;
  param->cmd = PRISM2_SET_ENCRYPTION;

  memset(param->sta_addr, 0xff, ETH_ALEN);

  strncpy(param->u.crypt.alg, alg_name, HOSTAP_CRYPT_ALG_NAME_LEN);
  param->u.crypt.flags = set_tx ? HOSTAP_CRYPT_FLAG_SET_TX_KEY : 0;
  param->u.crypt.idx = key_idx;
  memcpy(param->u.crypt.seq, seq, seq_len);
  param->u.crypt.key_len = key_len;
  memcpy((u8 *)(param+1), key, key_len);

  if (hostapd_ioctl(dev, param, blen, 1))
    {
      debug_printf(DEBUG_NORMAL, "Unable to set encryption key! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      show_set_key_error(param);
      ret = -1;
    }

  free(buf);

  return ret;
}

int cardif_hostap_driver_set_tkip(struct interface_data *intdata, 
				  unsigned char *addr, int key_idx, int set_tx,
				  char *seq, int seq_len, char *key, 
				  int key_len)
{
  return cardif_hostap_driver_set_key(intdata->intName, WPA_TKIP, addr, 
				      key_idx, set_tx, seq, seq_len, key, 
				      key_len);
}

int cardif_hostap_driver_set_ccmp(struct interface_data *intdata,
				  unsigned char *addr, int key_idx, int set_tx,
				  char *seq, int seq_len, char *key, 
				  int key_len)
{
  return cardif_hostap_driver_set_key(intdata->intName, WPA_CCMP, addr,
				      key_idx, set_tx, seq, seq_len, key,
				      key_len);
}

int cardif_hostap_driver_delete_key(struct interface_data *intdata,
				    int key_idx, int set_tx)
{
  return cardif_hostap_driver_set_key(intdata->intName, WPA_NONE, NULL,
				      key_idx, set_tx, NULL, 0, NULL, 0);
}

// Taken from WPA supplicant.
int cardif_hostap_driver_reset(const char *ifname, int type)
{
	struct iwreq iwr;
	int *i, s, ret = 0;

	debug_printf(DEBUG_INT, "Issuing reset to card!\n");
	s = socket(PF_INET, SOCK_DGRAM, 0);
	if (s < 0) {
	  debug_printf(DEBUG_NORMAL, "Error getting socket! (%s:%d)\n",
		       __FUNCTION__, __LINE__);
	  return -1;
	}

	memset(&iwr, 0, sizeof(iwr));
	strncpy(iwr.ifr_name, ifname, IFNAMSIZ);
	i = (int *) iwr.u.name;
	*i++ = type;

	if (ioctl(s, PRISM2_IOCTL_RESET, &iwr) < 0) {
	  debug_printf(DEBUG_NORMAL, "Error in reset IOCTL! (%s:%d)\n",
		       __FUNCTION__, __LINE__);
	  ret = -1;
	}
	close(s);
	return ret;
}

// Derived from WPA supplicant code.
int cardif_hostap_driver_disassociate(struct interface_data *intdata,
				      int reason_code)
{
  struct prism2_hostapd_param param;
  int ret;

  memset(&param, 0, sizeof(param));
  param.cmd = PRISM2_HOSTAPD_MLME;
  memcpy(param.sta_addr, intdata->source_mac, 6);
  param.u.mlme.cmd = MLME_STA_DISASSOC;
  param.u.mlme.reason_code = reason_code;
  ret = hostapd_ioctl(intdata->intName, &param, sizeof(param), 1);
  if (ret == 0)
    {
      usleep(100000);
      ret = cardif_hostap_driver_reset(intdata->intName, 2);
    }

  return ret;
}

void cardif_hostap_driver_associate(struct interface_data *intdata, 
				    char *newessid)
{
  intdata->flags |= DONTSCAN;

  cardif_linux_wext_set_ssid(intdata, newessid);
   // We sometimes roam without catching it -- currently rtnetlink is looking for a
   // switch to unassociated in order to signal a roam. Unfortunately, as we don't
   // need to turn off keys in order to associate, only to do 1x, we don't actually
   // spend enough time unassociated for it to be caught -- we transition straight
   // to another association. So catch it here.
  /*
  debug_printf(DEBUG_INT, "ASSOCIATING: turning encryption off.\n");
  cardif_linux_wext_enc_disable(intdata);
  */
}

int cardif_hostap_driver_roam(struct interface_data *intdata, 
				    int zero_keys)
{
   // global zero_keys sent in. we should presumably *not* set zero keys; that
   // is currently the default (comments to the contrary). If we do want to set zero
   // keys, use -z on the xsupplicant command line. We don't need a separate
   // roam function if this actually works (it was added to force zero_keys=0).
  cardif_linux_wext_roam(intdata, zero_keys);
  return XENOTHING_TO_DO;
}



struct cardif_funcs cardif_hostap_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_hostap_driver_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_hostap_driver_set_tkip,
  .set_ccmp_key = cardif_hostap_driver_set_ccmp,
  .delete_key = cardif_hostap_driver_delete_key,
  .set_ssid = cardif_linux_wext_set_ssid,
  .associate = cardif_hostap_driver_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_hostap_driver_wpa_state,
  .wpa = cardif_hostap_driver_wpa,
  .roam = cardif_hostap_driver_roam,
  .countermeasures = cardif_hostap_driver_countermeasures,
  .drop_unencrypted = cardif_hostap_driver_drop_unencrypted,
  .get_wpa_ie = cardif_hostap_driver_get_wpa_ie,
  .get_wpa2_ie = cardif_hostap_driver_get_wpa2_ie,
  .enc_disable = cardif_linux_wext_enc_disable
};

#endif
