/*******************************************************************
 * Linux card interface implementation.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: cardif_linux.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_linux.c,v 1.98 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_linux.c,v $
 * Revision 1.98  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.97  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#ifdef LINUX_FRAMER

#include <fcntl.h>
#include <string.h>
#include <stdlib.h>
#include <sys/ioctl.h>
#include <netinet/in.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <linux/if_packet.h>
#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <net/if_arp.h>

#include "config.h"
#include "profile.h"
#include "config_ssid.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/cardif.h"
#include "xsup_debug.h"
#include "xsup_err.h"
#include "snmp.h"
#include "statemachine.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_hostap_driver.h"
#include "cardif/linux/cardif_atmel_driver.h"
#include "cardif/linux/cardif_ndiswrapper_driver.h"
#include "cardif/linux/cardif_ipw_driver.h"
#include "cardif/linux/cardif_madwifi_driver.h"
#include "timer.h"

#ifndef ETH_P_EAPOL
#define ETH_P_EAPOL 0x888e
#endif

// Define this, so the compiler doesn't complain.
extern unsigned int if_nametoindex(const char *);

// This contains a pointer to the functions needed for wireless.  
struct cardif_funcs *wireless;

/***********************************************
 *
 * Determine if we are currently associated. 
 *
 ***********************************************/
void cardif_check_associated(struct interface_data *intdata)
{
  char newmac[6], curbssid[6];

  // If we are wired, this function doesn't do anything useful.
  if (!TEST_FLAG(intdata->flags, IS_WIRELESS)) return;

  cardif_GetBSSID(intdata, curbssid);

  debug_printf(DEBUG_INT, "Current BSSID is ");
  debug_hex_printf(DEBUG_INT, curbssid, 6);

  memset(newmac, 0x00, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      UNSET_FLAG(intdata->flags, ASSOCIATED);
      return;
    }

  memset(newmac, 0x44, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      UNSET_FLAG(intdata->flags, ASSOCIATED);
      return;
    }

  memset(newmac, 0xFF, 6);
  if (memcmp(newmac, curbssid, 6) == 0)
    {
      UNSET_FLAG(intdata->flags, ASSOCIATED);
      return;
    }

  //Otherwise, we are associated.
  SET_FLAG(intdata->flags, ASSOCIATED);
}

/***********************************************
 *
 * Set up the wireless cardif_funcs structure to the driver that the user
 * has requested.
 *
 ***********************************************/
void cardif_set_driver(char driver)
{
  switch (driver)
    {
    case DRIVER_NONE:
      wireless = NULL;
      break;

    case DRIVER_WEXT:
      wireless = &cardif_linux_wext_driver;
      break;

    case DRIVER_HOSTAP:
      wireless = &cardif_hostap_driver;
      break;

    case DRIVER_NDISWRAPPER:
      wireless = &cardif_ndiswrapper_driver;
      break;

    case DRIVER_IPW:
      wireless = &cardif_ipw_driver;
      break;

#ifdef ENABLE_MADWIFI
    case DRIVER_MADWIFI:
      wireless = &cardif_madwifi_driver;
      break;
#endif
      
    case DRIVER_ATMEL:
      wireless = &cardif_atmel_driver;
    }
}

/***********************************************
 *
 * Do whatever is needed to get the interface in to a state that we can send
 * and recieve frames on the network.  Any information that we need to later
 * use should be stored in the interface_data structure.
 *
 ***********************************************/
int cardif_init(struct interface_data *thisint, char driver, int block_wpa)
{
  struct ifreq ifr;
  struct lin_sock_data *sockData;
  int sockopts, sockerr, retval;

  debug_printf(DEBUG_INT, "Initializing socket for interface %s..\n",
	       thisint->intName);

  // Keep track of which driver we were assigned.
  thisint->driver_in_use = driver;

  // Find out what the interface index is.
  thisint->intIndex = if_nametoindex(thisint->intName);

  // Allocate memory for the things we need.
  thisint->sockData = (void *)malloc(sizeof(struct lin_sock_data));
  if (thisint->sockData == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error allocating memory!\n");
      return XEMALLOC;
    }

  sockData = thisint->sockData;

  // Establish a socket handle.
  sockData->sockInt = socket(PF_PACKET, SOCK_RAW, htons(ETH_P_EAPOL));
  if (sockData->sockInt < 0)
    {
      debug_printf(DEBUG_NORMAL, 
		   "Couldn't initialize raw socket for interface %s!\n",
		   thisint->intName);
      return XENOSOCK;
    }        

  // Tell the ifreq struct which interface we want to use.
  strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

  retval = ioctl(sockData->sockInt, SIOCGIFINDEX, &ifr);
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting interface index value for interface %s\n",
		   thisint->intName);
      return XESOCKOP;
    }

  // Build our link layer socket struct, so we can bind it to a specific
  // interface.
  sockData->sll.sll_family = PF_PACKET;
  sockData->sll.sll_ifindex = ifr.ifr_ifindex;
  sockData->sll.sll_protocol = htons(ETH_P_EAPOL);

  // Bind to the interface.
  retval = bind(sockData->sockInt, (const struct sockaddr *)&sockData->sll, 
		sizeof(struct sockaddr_ll));
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error binding raw socket to interface %s!\n",
		   thisint->intName);
      return XESOCKOP;
    }

  // Get our MAC address.  (Needed for sending frames out correctly.)
  retval = ioctl(sockData->sockInt, SIOCGIFHWADDR, &ifr);
  if (retval < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting hardware (MAC) address for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  // Store a copy of our source MAC for later use.
  memcpy((char *)&thisint->source_mac[0], (char *)&ifr.ifr_hwaddr.sa_data[0], 6);

  // Check if we want ALLMULTI mode, and enable it.
  if (config_get_allmulti() == 1)
    {
      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't determine if ALLMULTI is enabled!\n");
	} else {
	  if (ifr.ifr_flags & IFF_ALLMULTI)
	    {
	      debug_printf(DEBUG_INT, "Allmulti mode is already enabled on this device!\n");
	      thisint->flags |= ALLMULTI;
	    } else {
	      debug_printf(DEBUG_INT, "Allmulti is currently disabled on this device!\n");
	      thisint->flags &= ~ALLMULTI;
	    }
	}

      ifr.ifr_flags |= IFF_ALLMULTI;
      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
	}
    }

  // Set our socket to non-blocking.
  sockopts = fcntl(sockData->sockInt, F_GETFL, 0);
  if (sockopts < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting socket options for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  sockerr = fcntl(sockData->sockInt, F_SETFL, sockopts | O_NONBLOCK);
  if (sockerr < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting socket options for interface %s!\n",
		   thisint->intName);
      return XENOTINT;
    }

  // Set up wireless card drivers.
  cardif_set_driver(driver);

  if (!block_wpa) 
  {
    cardif_enable_wpa(thisint);
  } else {
    debug_printf(DEBUG_NORMAL, "Not turning on WPA support!\n");
  }

  if (cardif_int_is_wireless(thisint->intName) == TRUE)
    {
      // If we have our destination set to AUTO, then preset our destination
      // address.
      if (config_get_destination() == DEST_AUTO)
	{
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	}
    }
	  
  return XENONE;
}

/**************************************************************
 *
 * Tell the wireless card to start scanning for wireless networks.
 *
 **************************************************************/
int cardif_do_wireless_scan(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  SET_FLAG(thisint->flags, SCANNING);
  config_ssid_clear();

  return wireless->scan(thisint);
}

/**************************************************************
 *
 * Send a disassociate message.
 *
 **************************************************************/
int cardif_disassociate(struct interface_data *thisint, int reason_code)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "Called %s\n", __FUNCTION__);
  return wireless->disassociate(thisint, reason_code);
}

/**************************************************************
 *
 * Check to see if the BSSID value is valid.  If it is, return TRUE. If
 * it isn't return FALSE.
 *
 **************************************************************/
int cardif_valid_dest(struct interface_data *thisint)
{
  char baddest[6];

  if ((thisint->flags & IS_WIRELESS))
    {
      memset((char *)&baddest, 0x00, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 0s for dest mac!\n");
	  return FALSE;

	}

      memset((char *)&baddest, 0x44, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All 4s for dest mac!\n");
	  return FALSE;
	}

      memset((char *)&baddest, 0xff, 6);
      if (memcmp(thisint->dest_mac, baddest, 6) == 0)
	{
	  debug_printf(DEBUG_INT, "All Fs for dest mac!\n");
	  return FALSE;
	}
    }  
  return TRUE;
}

/******************************************
 *
 * Return the socket number for functions that need it.
 *
 ******************************************/
int cardif_get_socket(struct interface_data *thisint)
{
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  return sockData->sockInt;
}

/******************************************
 *
 * Clean up anything that was created during the initialization and operation
 * of the interface.  This will be called before the program terminates.
 *
 ******************************************/
int cardif_deinit(struct interface_data *thisint)
{
  struct ifreq ifr;
  uint16_t int16;
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  debug_printf(DEBUG_EVERYTHING, "Cleaning up interface %s...\n",thisint->intName);

  if (TEST_FLAG(thisint->flags, IS_WIRELESS))
  {
    debug_printf(DEBUG_INT, "Turning off WPA support/state.\n");
    // DKS -- we should undo all the changes we did to the interface. Currently this
    // does not do things like remove the WPA ie we added. This leaves the card in
    // a potentially inconsistent state.
    cardif_disable_wpa_state(thisint);
  }

  // Check if we want ALLMULTI mode, and enable it.
  if (config_get_allmulti() == 1)
    {
      // Tell the ifreq struct which interface we want to use.
      strncpy((char *)&ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));

      if (ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr) < 0)
	{
	  debug_printf(DEBUG_NORMAL, "Couldn't get interface flags!\n");
	} else {
	  // Check if allmulti was disabled when we started.  If it was,
	  // then disable it again, so everything is good.
	  if (!(thisint->flags & ALLMULTI))
	    {
	      debug_printf(DEBUG_INT, "Turning off ALLMULTI mode!\n");

	      int16 = ifr.ifr_flags;

	      // ANDing the flags with 0xfdff will turn off the ALLMULTI flag.
	      ifr.ifr_flags = (int16 & 0xfdff);
	      if (ioctl(sockData->sockInt, SIOCSIFFLAGS, &ifr) < 0)
		{
		  debug_printf(DEBUG_NORMAL, "Couldn't set ALLMULTI mode on this interface!  We will continue anyway!\n");
		}
	    }
	}
    }

  close(sockData->sockInt);

  // Now clean up the memory.
  if (thisint->sockData != NULL)
    {
      free(thisint->sockData);
      thisint->sockData = NULL;
    }

  return XENONE;
}

/******************************************
 *
 * Set a WEP key.  Also, based on the index, we may change the transmit
 * key.
 *
 ******************************************/
int cardif_set_wep_key(struct interface_data *thisint, u_char *key, 
		       int keylen, int index)
{
  if (wireless == NULL) return -1;

  return wireless->set_wep_key(thisint, key, keylen, index);
}

/**********************************************************
 *
 * Set a TKIP key. 
 *
 **********************************************************/
int cardif_set_tkip_key(struct interface_data *thisint, char *addr, 
			      int keyidx, int settx, char *seq, int seqlen, 
			      char *key, int keylen)
{
  if (wireless == NULL) return -1;

  return wireless->set_tkip_key(thisint, addr, keyidx, settx, seq, seqlen,
				key, keylen);
}

/**********************************************************
 *
 * Set a CCMP (AES) key
 *
 **********************************************************/
int cardif_set_ccmp_key(struct interface_data *thisint, char *addr, int keyidx,
			int settx, char *seq, int seqlen, char *key,
			int keylen)
{
  if (wireless == NULL) return -1;

  return wireless->set_ccmp_key(thisint, addr, keyidx, settx, seq, seqlen,
				key, keylen);
}

/**********************************************************
 *
 * Delete a key
 *
 **********************************************************/
int cardif_delete_key(struct interface_data *intdata, int key_idx, int set_tx)
{
  if (wireless == NULL) return -1;

  return wireless->delete_key(intdata, key_idx, set_tx);
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_SetSSID(struct interface_data *thisint, char *ssid_name)
{
  if (wireless == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No valid call to set SSID on this driver!"
		   "\n");
      return -1;
    }

  return wireless->set_ssid(thisint, ssid_name);
}

/******************************************
 *
 * If our association timer expires, we need to attempt to associate again.
 *
 ******************************************/
void cardif_association_timeout_expired(struct interface_data *intdata)
{
  // And try to associate again.
  cardif_associate(intdata, intdata->cur_essid);
}

/******************************************
 *
 * Set the association timeout.
 *
 ******************************************/
void cardif_set_association_timeout()
{
  if (timer_check_existing(ASSOCIATION_TIMER))
    {
      timer_reset_timer_count(ASSOCIATION_TIMER, ASSOCIATION_TIMEOUT);
    } else {
      timer_add_timer(ASSOCIATION_TIMER, ASSOCIATION_TIMEOUT, NULL,
		      &cardif_association_timeout_expired);
    }
}

/******************************************
 *
 * Do whatever we need to do in order to associate based on the flags in
 * the ssids_list struct.
 *
 ******************************************/
void cardif_associate(struct interface_data *intdata, char *newssid)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface struct passed to %s!\n",
		   __FUNCTION__);
      return;
    }

  // Set up our association timeout.
  cardif_set_association_timeout();

  return wireless->associate(intdata, newssid);
}

/******************************************
 *
 * Ask the wireless card for the ESSID that we are currently connected to.  If
 * this is not a wireless card, or the information is not available, we should
 * return an error.
 *
 ******************************************/
int cardif_GetSSID(struct interface_data *thisint, char *ssid_name)
{
  if (wireless == NULL) 
    {
      debug_printf(DEBUG_NORMAL, "No valid call to get SSID for this driver!"
		   "\n");
      return -1;
    }

  if ((thisint == NULL) || (ssid_name == NULL)) 
  {
    debug_printf(DEBUG_INT, "NULL value passed to %s!\n", __FUNCTION__);
    return -1;
  }

  return wireless->get_ssid(thisint, ssid_name);
}

/******************************************
 *
 * Get the Broadcast SSID (MAC address) of the Access Point we are connected 
 * to.  If this is not a wireless card, or the information is not available,
 * we should return an error.
 *
 ******************************************/
int cardif_GetBSSID(struct interface_data *thisint, char *bssid_dest)
{
  if (wireless == NULL) return -1;

  if (thisint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid interface data structure passed to %s!\n", __FUNCTION__);
      return -1;
    }

  if (bssid_dest == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bssid_dest in %s!\n", __FUNCTION__);
      return -1;
    }

  return wireless->get_bssid(thisint, bssid_dest);
}

/******************************************
 *
 * Set the flag in the state machine that indicates if this interface is up
 * or down.  If there isn't an interface, we should return an error.
 *
 ******************************************/
int cardif_get_if_state(struct interface_data *thisint)
{
  int retVal;
  struct ifreq ifr;
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  strncpy(ifr.ifr_name, thisint->intName, sizeof(ifr.ifr_name));
  retVal = ioctl(sockData->sockInt, SIOCGIFFLAGS, &ifr);
  if (retVal < 0)
    {
      debug_printf(DEBUG_NORMAL, "Interface %s not found!\n", thisint->intName);
      return FALSE;
    }
  
  if ((ifr.ifr_flags & IFF_UP) == IFF_UP)
    {
      return TRUE;
    } else {
      SET_FLAG(thisint->flags, WAS_DOWN);
      return FALSE;
    }
  return XENONE;
}

/******************************************
 *
 * Send a frame out of the network card interface.  If there isn't an 
 * interface, we should return an error.  We should return a different error
 * if we have a problem sending the frame.
 *
 ******************************************/
int cardif_sendframe(struct interface_data *thisint)
{
  char nomac[] = {0x00, 0x00, 0x00, 0x00, 0x00, 0x00};
  int retval;
  struct lin_sock_data *sockData;
  struct config_network *network_data;

  sockData = thisint->sockData;

  if (thisint == NULL) return XEMALLOC;

  if (thisint->send_size == 0) return XENONE;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEBADCONFIG;
    }

  // The frame we are handed in shouldn't have a src/dest, so put it in.
  memcpy(&thisint->sendframe[0], &thisint->dest_mac[0], 6);
  memcpy(&thisint->sendframe[6], &thisint->source_mac[0], 6);

  if (memcmp(nomac, (char *)&network_data->dest_mac[0], 6) != 0)
    {
      debug_printf(DEBUG_INT, "Static MAC address defined!  Using it!\n");
      memcpy(&thisint->sendframe[0], &network_data->dest_mac[0], 6);
    }

  debug_printf(DEBUG_EVERYTHING, "Frame to be sent : \n");
  debug_hex_dump(DEBUG_EVERYTHING, thisint->sendframe, thisint->send_size);

  snmp_dot1xSuppEapolFramesTx();
  retval = sendto(sockData->sockInt, thisint->sendframe, thisint->send_size, 0,
		  (struct sockaddr *)&sockData->sll, sizeof(sockData->sll));
  if (retval <= 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't send frame! (%s)\n", strerror(errno));
    }

  thisint->send_size = 0;
  
  // Clear out the receive buffer so we don't accidently try to process it
  // again.
  bzero(thisint->recvframe, 1520);
  thisint->recv_size = 0;

  return retval;
}

/******************************************
 * 
 * Get a frame from the network.  Make sure to check the frame, to determine 
 * if it is something we care about, and act accordingly.
 *
 ******************************************/
int cardif_getframe(struct interface_data *thisint)
{
  int newsize=0;
  char dot1x_default_dest[6] = {0x01, 0x80, 0xc2, 0x00, 0x00, 0x03};
  struct lin_sock_data *sockData;
  u_char resultframe[1520];
  int resultsize;

  sockData = thisint->sockData;

  errno = 0;
  resultsize = 1550;

  newsize = recvfrom(sockData->sockInt, resultframe, resultsize, 0, 0, 0);
  if (newsize <= 0)
    {
      if (errno != EAGAIN)
	{
	  debug_printf(DEBUG_NORMAL, "Error (%d) : %s  (%s:%d)\n", errno,
		       strerror(errno), __FUNCTION__, __LINE__);
	}
      return XENOFRAMES;
    } else {
      debug_printf(DEBUG_EVERYTHING, "Got Frame : \n");
      debug_hex_dump(DEBUG_EVERYTHING, resultframe, newsize);
    }

  snmp_dot1xSuppEapolFramesRx();

  // Make sure that the frame we got is for us..
  if ((memcmp(&thisint->source_mac[0], &resultframe[0], 6) == 0) ||
      ((memcmp(&resultframe[0], &dot1x_default_dest[0], 6) == 0) &&
       (memcmp(&resultframe[6], &thisint->source_mac[0], 6) != 0)))
    {
      // Since we now know this frame is for us, record the address it
      // came from.
      snmp_dot1xSuppLastEapolFrameSource((char *)&resultframe[6]);

      resultsize = newsize;

      switch (config_get_destination())
	{
	case DEST_AUTO:
	  // If it is a wired interface, only change the destination if
	  // the recieved frame destination isn't the multicast address.
	  if (!TEST_FLAG(thisint->flags, IS_WIRELESS))
	    {
	      if (memcmp(&resultframe[0], dot1x_default_dest, 6) == 0)
		{
		  break;
		}
	      // Otherwise, fall through.
	    }
	case DEST_SOURCE:
	  if (memcmp(thisint->dest_mac, &resultframe[6], 6) != 0)
	    {
	      debug_printf(DEBUG_INT, "Changing destination mac to source.\n");
	    }
	  memcpy(thisint->dest_mac, &resultframe[6], 6);
	  break;

	case DEST_MULTICAST:
	  memcpy(thisint->dest_mac, dot1x_default_dest, 6);
	  break;

	case DEST_BSSID:
	  cardif_GetBSSID(thisint, thisint->dest_mac);
	  break;

	default:
	  debug_printf(DEBUG_NORMAL, "Unknown destination mode!\n");
	  break;
	}

      thisint->recv_size = newsize;

      memcpy(thisint->recvframe, resultframe, newsize);
      return newsize;
    }

  // Otherwise it isn't for us. 
  debug_printf(DEBUG_INT, "Got a frame, not for us.\n");
  return XENOFRAMES;
}

/**************************************************************
 *
 * Set the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_enable_wpa_state(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA state on interface %s.\n",thisint->intName);

  return wireless->wpa_state(thisint, TRUE);
}

/**************************************************************
 *
 * Clear the state needed to associate to a WPA enabled AP, and actually
 * do a WPA authentication.
 *
 **************************************************************/
int cardif_disable_wpa_state(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  return wireless->wpa_state(thisint, FALSE);
}

/**************************************************************
 *
 * Enable WPA (if it is supported.)
 *
 **************************************************************/
int cardif_enable_wpa(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  debug_printf(DEBUG_INT, "WPA: Enabling WPA on interface %s.\n",thisint->intName);

  return wireless->wpa(thisint, TRUE);
}

/**************************************************************
 *
 * Call this when we roam to a different AP, or disassociate from an AP.
 *
 **************************************************************/
int cardif_roam(struct interface_data *thisint)
{
  if (wireless == NULL) return -1;

  if (config_ssid_get_ssid_abilities() & WPA_IE)
    {
      debug_printf(DEBUG_INT, "Doing WPA mode! Not setting encryption to Open!\n");
      return 0;
    }

  return wireless->roam(thisint, (thisint->flags & ZEROONROAM)? 1:0);
}

/******************************************
 * 
 * Return true if there is a frame in the queue to be processed.
 *
 ******************************************/
int cardif_frameavail(struct interface_data *thisint)
{
  int newsize=0;
  char resultframe[1520];
  struct lin_sock_data *sockData;

  sockData = thisint->sockData;

  newsize = recvfrom(sockData->sockInt, &resultframe, 1520, MSG_PEEK, 0, 0);
  if (newsize > 0) return TRUE;

  return FALSE;
}

/******************************************
 *
 * Validate an interface, based on if it has a MAC address.
 *
 ******************************************/
int cardif_validate(char *interface)
{
  int sd, res;
  struct ifreq ifr;

  strncpy(ifr.ifr_name, interface, sizeof(interface)+1);

  sd = socket(PF_PACKET, SOCK_RAW, 0);
  if (sd < 0)
    return FALSE;
  res = ioctl(sd, SIOCGIFHWADDR, &ifr);
  close(sd);
  if (res < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't get information for interface %s!\n",interface);
    } else {
      switch (ifr.ifr_hwaddr.sa_family)
	{
	case ARPHRD_ETHER:
	case ARPHRD_IEEE80211:
	  return TRUE;
	}
    }
  return FALSE;
}

/******************************************
 *
 * (en)/(dis)able countermeasures on this interface.
 *
 ******************************************/
int cardif_countermeasures(struct interface_data *intdata, char endis)
{
  if (wireless == NULL) return -1;

  return wireless->countermeasures(intdata, endis);
}

/******************************************
 *
 * (en)/(dis)able receiving of unencrypted frames on this interface.
 *
 ******************************************/
int cardif_drop_unencrypted(struct interface_data *intdata, char endis)
{
  if (wireless == NULL) return -1;
  
  return wireless->drop_unencrypted(intdata, endis);
}

/******************************************
 *
 * Get the name of an interface, based on an index value.
 *
 ******************************************/
#define PROC_DEV_FILE  "/proc/net/dev"

int cardif_get_int(int index, char *retInterface)
{
  FILE *fp;
  int hits;
  char line[1000], *lineptr;

  hits = 0;

  fp = fopen(PROC_DEV_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/dev!\n");
      exit(250);
    }

  bzero(line, 1000);

  while ((hits <= index) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr == NULL) continue;

      *lineptr = '\0';
      lineptr = &line[0];

      while (*lineptr == ' ') lineptr++;  // Strip out blanks.
      
      strcpy(retInterface, lineptr);
      hits++;
    }

  if (hits <= index)
    {
      debug_printf(DEBUG_INT, "No more interfaces to look at!\n");
      return XNOMOREINTS;
    }

  debug_printf(DEBUG_INT, "Found interface : %s\n",retInterface);

  fclose(fp);

  return XENONE;   // No errors.
}


/*******************************************************
 *
 * Check to see if an interface is wireless.  On linux, we look in
 * /proc/net/wireless to see if the interface is registered with the
 * wireless extensions.
 *
 *******************************************************/
#define PROC_WIRELESS_FILE  "/proc/net/wireless"

int cardif_int_is_wireless(char *interface)
{
  FILE *fp;
  char line[1000], *lineptr=NULL;
  int done;

  done = FALSE;

  fp = fopen(PROC_WIRELESS_FILE, "r");
  if (fp == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't access /proc/net/wireless!  (You probably don't have wireless extensions enabled!)\n");
      return -1;
    }

  bzero(line, 1000);

  while ((!done) && (fgets(line, 999, fp) != NULL))
    { 
      lineptr = strchr(line, ':');
      
      if (lineptr != NULL)
	{
	  
	  *lineptr = '\0';
	  lineptr = &line[0];
	  
	  while (*lineptr == ' ') lineptr++;  // Strip out blanks.
	  if (lineptr != NULL)
	    {
	      if (strcmp(lineptr, interface) == 0) done=TRUE;
	    }
	}
    }
  fclose(fp);
  
  if ((lineptr != NULL) && (strcmp(lineptr, interface) == 0))
    {
      debug_printf(DEBUG_INT, "Interface %s is wireless!\n",interface);
      return TRUE;
    } else {
      debug_printf(DEBUG_INT, "Interface %s is NOT wireless!\n",interface);
      return FALSE;
    }
  return XENONE;   // No errors.
}

int cardif_get_wpa_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for IE data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->get_wpa_ie(intdata, iedata, ielen);
}

int cardif_get_wpa2_ie(struct interface_data *intdata, char *iedata, int *ielen)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  if (iedata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid bucket for IE data! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->get_wpa2_ie(intdata, iedata, ielen);
}

/**************************************************************
 *
 * This function should clear out all keys that have been applied to the card.
 * It should be indepentant of the type (WEP/TKIP/CCMP) of key that was
 * applied.
 *
 **************************************************************/
int cardif_clear_keys(struct interface_data *intdata)
{
  int retVal = 0, i;

  debug_printf(DEBUG_INT, "Called %s!\n", __FUNCTION__);
  // Clear the TX key.
  retVal = cardif_delete_key(intdata, 0, 1);
  if (retVal != XENONE) 
    {
      debug_printf(DEBUG_NORMAL, "Error clearing default TX key!\n");
      return retVal;
    }

  // Now, clear out the broadcast/multicast/group key(s).
  for (i=0;i<4;i++)
    {
      retVal = cardif_delete_key(intdata, i, 0);
      if (retVal != XENONE)
	{
	  debug_printf(DEBUG_NORMAL, "Error clearing key %d!\n", i);
	  return retVal;
	}
    }

  return XENONE;
}

void cardif_reassociate(struct interface_data *intiface, u_char reason)
{
  if (config_ssid_get_ssid_abilities() & WPA_IE)
    {
      debug_printf(DEBUG_NORMAL, "WPA is enabled on this connection! AP is WPA capable.\n");
      
      // Since we are doing WPA, we need to disassociate from 
      // the network, and reassociate with WPA set up.
      cardif_enable_wpa_state(intiface);
     // In general, we need to set requirements for reassociation & force reassociation.
     cardif_disassociate(intiface, reason);  
     cardif_clear_keys(intiface);
    }
  cardif_associate(intiface, intiface->cur_essid);  
}

int cardif_enc_disable(struct interface_data *intdata)
{
  if (intdata == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Error!  Invalid interface data structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return XEMALLOC;
    }

  return wireless->enc_disable(intdata);
}

#endif
