/*******************************************************************
 * File: cardif_madwifi_driver.c
 *
 * A significant portion of this code is either based on, or taken directly
 * from WPA Supplicant.  The copyright and license information for WPA
 * Supplicant is reproduced below.  Code taken from WPA Supplicant is notated
 * in comments above the function definitions.
 *
 * All other code is licensed under a dual GPL/BSD license.  (See LICENSE file 
 * for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_madwifi_driver.c,v 1.18 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_madwifi_driver.c,v $
 * Revision 1.18  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.17  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

/*
 * WPA Supplicant - driver interaction with MADWIFI 802.11 driver
 * Copyright (c) 2004, Sam Leffler <sam@errno.com>
 * Copyright (c) 2004, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */

#ifdef ENABLE_MADWIFI

#ifdef LINUX_FRAMER

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>
#include <linux/if_packet.h>
#include <linux/netlink.h>
#include <linux/rtnetlink.h>

typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;
typedef int64_t s64;
typedef int32_t s32;
typedef int8_t s8;

#include <include/compat.h>
#include <net80211/_ieee80211.h>
#include <net80211/ieee80211.h>
#include <net80211/ieee80211_crypto.h>
#include <net80211/ieee80211_ioctl.h>
#include "xsup_debug.h"
#include "xsup_err.h"
#include "config.h"
#include "profile.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/linux/cardif_madwifi_driver.h"
#include "cardif/linux/cardif_linux.h"
#include "cardif/linux/cardif_linux_rtnetlink.h"
#include "wpa.h"
#include "wpa2.h"
#include "config_ssid.h"

// Derived from wpa_supplicant
int set80211priv(char *dev, int op, void *data, int len, int show_err)
{
  struct iwreq iwr;
  int sock;

  sock = socket(AF_INET, SOCK_DGRAM, 0);
  if (sock < 0) {
    debug_printf(DEBUG_NORMAL, "Error allocating socket! (%s:%d)\n",
		 __FUNCTION__, __LINE__);
    return -1;
  }

  bzero(&iwr, sizeof(iwr));
  strncpy(iwr.ifr_name, dev, IFNAMSIZ);
  if (len < IFNAMSIZ) {
    memcpy(iwr.u.name, data, len);
  } else {
    iwr.u.data.pointer = data;
    iwr.u.data.length = len;
  }

  if (ioctl(sock, op, &iwr) < 0) {
    if (show_err) {
      static const char *opnames[] = {
	"ioctl[IEEE80211_IOCTL_SETPARAM]",
	"ioctl[IEEE80211_IOCTL_GETPARAM]",
	"ioctl[IEEE80211_IOCTL_SETKEY]",
	"ioctl[IEEE80211_IOCTL_GETKEY]",
	"ioctl[IEEE80211_IOCTL_DELKEY]",
	NULL,
	"ioctl[IEEE80211_IOCTL_SETMLME]",
	NULL,
	"ioctl[IEEE80211_IOCTL_SETOPTIE]",
	"ioctl[IEEE80211_IOCTL_GETOPTIE]",
	"ioctl[IEEE80211_IOCTL_ADDMAC]",
	NULL,
	"ioctl[IEEE80211_IOCTL_DELMAC]",
	NULL,
	"ioctl[IEEE80211_IOCTL_CHANLIST]",
      };
      if (IEEE80211_IOCTL_SETPARAM <= op &&
	  op <= IEEE80211_IOCTL_CHANLIST)
	debug_printf(DEBUG_NORMAL, "%s\n", opnames[op - SIOCIWFIRSTPRIV]);
      else
	debug_printf(DEBUG_NORMAL, "ioctl[unknown???]");
    }
    return -1;
  }
  return 0;
}

// Derived from the wpa supplicant code.
int set80211param(const char *dev, int op, int arg, int show_err)
{
  struct iwreq iwr;
  int s;

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if (s < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error getting socket! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  memset(&iwr, 0, sizeof(iwr));
  strncpy(iwr.ifr_name, dev, IFNAMSIZ);
  iwr.u.mode = op;
  memcpy(iwr.u.name+sizeof(__u32), &arg, sizeof(arg));
  
  if (ioctl(s, IEEE80211_IOCTL_SETPARAM, &iwr) < 0) {
    if (show_err) 
      debug_printf(DEBUG_NORMAL, "ioctl[IEEE80211_IOCTL_SETPARAM]\n");
    return -1;
  }
  return 0;
}

int cardif_madwifi_driver_get_wpa_ie(struct interface_data *intdata,
				     char *iedata, int *ielen)
{
  wpa_gen_ie(intdata, iedata);
  *ielen = 24;
  return XENONE;
}

int cardif_madwifi_driver_get_wpa2_ie(struct interface_data *intdata,
				      char *iedata, int *ielen)
{
  wpa2_gen_ie(intdata, iedata, ielen);

  debug_printf(DEBUG_INT, "Setting WPA2 IE : ");
  debug_hex_printf(DEBUG_INT, iedata, *ielen);
  debug_printf(DEBUG_INT, "\n");

  return XENONE;
}

// Derived from WPA supplicant
int cardif_madwifi_driver_set_wpa_ie(char *dev, char *wpa_ie,  int wpa_ie_len)
{
  struct iwreq iwr;
  int s;

  s = socket(AF_INET, SOCK_DGRAM, 0);
  if (s < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate socket! (%s:%d)\n",
		   __FUNCTION__, __LINE__);
      return -1;
    }

  memset(&iwr, 0, sizeof(iwr));
  strncpy(iwr.ifr_name, dev, IFNAMSIZ);
  /* NB: SETOPTIE is not fixed-size so must not be inlined */
  iwr.u.data.pointer = (void *) wpa_ie;
  iwr.u.data.length = wpa_ie_len;
  
  if (ioctl(s, IEEE80211_IOCTL_SETOPTIE, &iwr) < 0) {
    debug_printf(DEBUG_NORMAL, "ioctl[IEEE80211_IOCTL_SETOPTIE]");
    return -1;
  }
  return 0;
}

int cardif_madwifi_driver_wpa(struct interface_data *intdata,  char endis)
{
   if (endis)
	   debug_printf(DEBUG_INT, "WPA: Enabling WPA on madwifi interface %s.\n",intdata->intName);
  else
	   debug_printf(DEBUG_INT, "WPA: Disabling WPA on madwifi interface %s.\n",intdata->intName);

  if (set80211param(intdata->intName, IEEE80211_PARAM_WPA, endis ? 3 : 0, 1)
      < 0)
    return -1;

  return XENONE;
}

// DKS -- if endis is FALSE, we are disabling WPA state.
int cardif_madwifi_driver_wpa_state(struct interface_data *thisint,  char endis)
{
  char wpaie[24];
  int ret = XENONE, len;

   if (endis)
	   debug_printf(DEBUG_INT, "WPA: Enabling WPA on madwifi interface %s.\n",thisint->intName);
  else
	   debug_printf(DEBUG_INT, "WPA: Disabling WPA on madwifi interface %s.\n",thisint->intName);

   // DKS -- if endis is FALSE, we need to turn off the wpa_ie, if we turned it on.
   // normally we'd reset the ie to whatever it was before, but as we may have stomped
   // on it already, let's just stomp on it again.
   if (endis) // enable
   {
	   cardif_madwifi_driver_get_wpa_ie(thisint, wpaie, &len);

	   if (cardif_madwifi_driver_set_wpa_ie(thisint->intName, wpaie, len) < 0)
	   {
		   debug_printf(DEBUG_NORMAL, "Couldn't set WPA IE! On device %s!\n",
						thisint->intName);
	   }
   } else { // disable
	   memset(wpaie,0,sizeof(wpaie));
	   if (cardif_madwifi_driver_set_wpa_ie(thisint->intName, wpaie, 0) < 0)
	   {
		   debug_printf(DEBUG_NORMAL, "Couldn't clear WPA IE! On device %s!\n",
						thisint->intName);
	   }
   }

  if (set80211param(thisint->intName, IEEE80211_PARAM_PRIVACY, endis, 1) < 0)
    ret = -1;

  if (set80211param(thisint->intName, IEEE80211_PARAM_WPA, endis ? 3 : 0, 1)
      < 0)
    ret = -1;

  return ret;
}

int cardif_madwifi_driver_wpa_disable(char *intname)
{
  // XXX Finish!
  return XENONE;
}

int cardif_madwifi_driver_del_key(char *ifname, int key_idx, 
				  unsigned char *addr)
{
	struct ieee80211req_del_key wk;

	debug_printf(DEBUG_INT, "%s: keyidx=%d\n", __FUNCTION__, key_idx);
	memset(&wk, 0, sizeof(wk));
	wk.idk_keyix = key_idx;
	if (addr != NULL)
		memcpy(wk.idk_macaddr, addr, IEEE80211_ADDR_LEN);

	return set80211priv(ifname, IEEE80211_IOCTL_DELKEY, &wk, sizeof(wk), 1);
}

int cardif_madwifi_driver_set_key(char *dev, int alg, unsigned char *addr,
			      int key_idx, int set_tx, char *seq, int seq_len,
			      char *key, int key_len)
{
  struct ieee80211req_key wk;
  char *alg_name;
  u_int8_t cipher;

  if (alg == WPA_NONE)
    {
      return cardif_madwifi_driver_del_key(dev, key_idx, addr);
    }

  switch (alg) {
  case WPA_WEP:
    alg_name = "WEP";
    cipher = IEEE80211_CIPHER_WEP;
    break;
  case WPA_TKIP:
    alg_name = "TKIP";
    cipher = IEEE80211_CIPHER_TKIP;
    break;
  case WPA_CCMP:
    alg_name = "CCMP";
    cipher = IEEE80211_CIPHER_AES_CCM;
    break;
  default:
    debug_printf(DEBUG_NORMAL, "%s: unknown/unsupported algorithm %d",
	       __FUNCTION__, alg);
    return -1;
  }
  
  if (seq_len > sizeof(u_int64_t)) {
    debug_printf(DEBUG_NORMAL, "%s: seq_len %d too big",
		 __FUNCTION__, seq_len);
    return -2;
  }
  if (key_len > sizeof(wk.ik_keydata)) {
    debug_printf(DEBUG_NORMAL, "%s: key length %d too big",
		 __FUNCTION__, key_len);
    return -3;
  }
  
  memset(&wk, 0, sizeof(wk));
  wk.ik_type = cipher;
  wk.ik_flags = IEEE80211_KEY_RECV;
  if (set_tx) {
    wk.ik_flags |= IEEE80211_KEY_XMIT | IEEE80211_KEY_DEFAULT;
    memcpy(wk.ik_macaddr, addr, IEEE80211_ADDR_LEN);
  } else
    memset(wk.ik_macaddr, 0, IEEE80211_ADDR_LEN);
  wk.ik_keyix = key_idx;
  wk.ik_keylen = key_len;
  memcpy(&wk.ik_keyrsc, seq, seq_len);
  debug_printf(DEBUG_NORMAL, "Key Seq = ");
  debug_hex_printf(DEBUG_NORMAL, seq, seq_len);
  memcpy(wk.ik_keydata, key, key_len);
  debug_printf(DEBUG_NORMAL, "Key = ");
  debug_hex_printf(DEBUG_NORMAL, key, key_len);

  return set80211priv(dev, IEEE80211_IOCTL_SETKEY, &wk, sizeof(wk), 1);
}

int cardif_madwifi_driver_set_tkip(struct interface_data *intdata, 
				  unsigned char *addr, int key_idx, int set_tx,
				  char *seq, int seq_len, char *key, 
				  int key_len)
{
  return cardif_madwifi_driver_set_key(intdata->intName, WPA_TKIP, addr, 
				      key_idx, set_tx, seq, seq_len, key, 
				      key_len);
}

int cardif_madwifi_driver_set_ccmp(struct interface_data *intdata,
				   unsigned char *addr, int key_idx,
				   int set_tx, char *seq, int seq_len,
				   char *key, int key_len)
{
  return cardif_madwifi_driver_set_key(intdata->intName, WPA_CCMP, addr,
				       key_idx, set_tx, seq, seq_len, key,
				       key_len);
}

int cardif_madwifi_driver_delete_key(struct interface_data *intdata,
				     int keyidx, int settx)
{
  return cardif_madwifi_driver_set_key(intdata->intName, WPA_NONE, NULL,
				       keyidx, settx, NULL, 0, NULL, 0);
}

// Taken from WPA supplicant.
int cardif_madwifi_driver_disassociate(struct interface_data *intdata,
				       int reason_code)
{
	struct ieee80211req_mlme mlme;

	mlme.im_op = IEEE80211_MLME_DISASSOC;
	mlme.im_reason = reason_code;
	memcpy(mlme.im_macaddr, intdata->source_mac, IEEE80211_ADDR_LEN);
	return set80211priv(intdata->intName, IEEE80211_IOCTL_SETMLME, &mlme, 
			    sizeof(mlme), 1);
}

int cardif_madwifi_driver_countermeasures(struct interface_data *intdata,
					  char endis)
{
  return set80211param(intdata->intName, IEEE80211_PARAM_COUNTERMEASURES,
		       endis, TRUE);
}

int cardif_madwifi_driver_drop_unencrypted(struct interface_data *intdata,
					   char endis)
{
  return set80211param(intdata->intName, IEEE80211_PARAM_DROPUNENCRYPTED,
		       endis, TRUE);
}

int cardif_madwifi_driver_roam(struct interface_data *intdata, int zero_keys)
{
  return XENONE;
}

void cardif_madwifi_driver_associate(struct interface_data *intdata,
				     char *newessid)
{
  u_char wpaie[255];
  int len;

  intdata->flags |= DONTSCAN;

  if (config_ssid_get_ssid_abilities() & RSN_IE)
    {
      debug_printf(DEBUG_NORMAL, "Getting WPA2 IE!\n");
      cardif_madwifi_driver_get_wpa2_ie(intdata, wpaie, &len);
    } else if (config_ssid_get_ssid_abilities() & WPA_IE)
      {
	cardif_madwifi_driver_get_wpa_ie(intdata, wpaie, &len);
      }

  if (cardif_madwifi_driver_set_wpa_ie(intdata->intName, wpaie, len) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't set WPA/RSN IE on device %s!\n",
		   intdata->intName);
    }

  cardif_madwifi_driver_disassociate(intdata, 0);

  cardif_linux_wext_set_ssid(intdata, newessid);
  // We need to force zero keys to cause association.
  //  debug_printf(DEBUG_INT, "ASSOCIATING: zeroing keys off.\n");
  //  cardif_linux_wext_zero_keys(intdata);

  cardif_linux_wext_enc_open(intdata);
  cardif_madwifi_driver_delete_key(intdata, 0, 1);
  cardif_madwifi_driver_delete_key(intdata, 0, 0);
  cardif_madwifi_driver_delete_key(intdata, 1, 0);
  cardif_madwifi_driver_delete_key(intdata, 2, 0);
  cardif_madwifi_driver_delete_key(intdata, 3, 0);

  
}

struct cardif_funcs cardif_madwifi_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_madwifi_driver_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_madwifi_driver_set_tkip,
  .set_ccmp_key = cardif_madwifi_driver_set_ccmp,
  .delete_key = cardif_madwifi_driver_delete_key,
  .set_ssid = cardif_linux_wext_set_ssid,
  .associate = cardif_madwifi_driver_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_madwifi_driver_wpa_state,
  .wpa = cardif_madwifi_driver_wpa,
  .roam = cardif_madwifi_driver_roam,
  .countermeasures = cardif_madwifi_driver_countermeasures,
  .drop_unencrypted = cardif_madwifi_driver_drop_unencrypted,
  .get_wpa_ie = cardif_madwifi_driver_get_wpa_ie,
  .enc_disable = cardif_linux_wext_enc_disable
};

#endif  // LINUX_FRAMER

#endif  // ENABLE_MADWIFI
