/*******************************************************************
 * File: cardif_ndiswrapper_driver.c
 *
 * A significant portion of this code is either based on, or taken directly
 * from WPA Supplicant.  The copyright and license information for WPA
 * Supplicant is reproduced below.  Code taken from WPA Supplicant is notated
 * in comments above the function definitions.
 *
 * All other code is licensed under a dual GPL/BSD license.  (See LICENSE file 
 * for more info.)
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: cardif_ndiswrapper_driver.c,v 1.18 2005/08/20 19:06:54 chessing Exp $
 * $Date: 2005/08/20 19:06:54 $
 * $Log: cardif_ndiswrapper_driver.c,v $
 * Revision 1.18  2005/08/20 19:06:54  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.17  2005/08/09 01:39:15  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

/*
 * WPA Supplicant - driver interaction with Linux ndiswrapper
 * Copyright (c) 2004-2005, Giridhar Pemmasani <giri@lmc.cs.sunysb.edu>
 * Copyright (c) 2004-2005, Jouni Malinen <jkmaline@cc.hut.fi>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2 as
 * published by the Free Software Foundation.
 *
 * Alternatively, this software may be distributed under the terms of BSD
 * license.
 *
 * See README and COPYING for more details.
 */

#ifdef LINUX_FRAMER

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <linux/types.h>
#include <linux/socket.h>
#include <linux/if.h>
#include <linux/wireless.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <stdint.h>

typedef uint64_t u64;
typedef uint32_t u32;
typedef uint16_t u16;
typedef uint8_t u8;
typedef int64_t s64;
typedef int32_t s32;
typedef int8_t s8;

#include "xsup_debug.h"
#include "xsup_err.h"
#include "config.h"
#include "profile.h"
#include "config_ssid.h"
#include "cardif/cardif.h"
#include "cardif/linux/cardif_linux_wext.h"
#include "cardif/linux/cardif_ndiswrapper_driver.h"
#include "wpa.h"

struct wpa_key
{
	uint32_t alg;
	u8 *addr;
	int key_index;
	int set_tx;
	u8 *seq;
	size_t seq_len;
	u8 *key;
	size_t key_len;
};

struct wpa_assoc_info
{
	u_char *bssid;
	u_char *ssid;
	unsigned int ssid_len;
	unsigned int freq;
	u_char *wpa_ie;
	int wpa_ie_len;
	u_int32_t pairwise_suite;
	u_int32_t group_suite;
	u_int32_t key_mgmt_suite;
};

#define PRIV_RESET	 		SIOCIWFIRSTPRIV+0
#define WPA_SET_WPA 			SIOCIWFIRSTPRIV+1
#define WPA_SET_KEY 			SIOCIWFIRSTPRIV+2
#define WPA_ASSOCIATE		 	SIOCIWFIRSTPRIV+3
#define WPA_DISASSOCIATE 		SIOCIWFIRSTPRIV+4
#define WPA_DROP_UNENCRYPTED 		SIOCIWFIRSTPRIV+5
#define WPA_SET_COUNTERMEASURES 	SIOCIWFIRSTPRIV+6
#define WPA_DEAUTHENTICATE	 	SIOCIWFIRSTPRIV+7
#define WPA_SET_AUTH_ALG	 	SIOCIWFIRSTPRIV+8

// Taken from wpa supplicant.
int get_socket(void)
{
	static const int families[] = {
		AF_INET, AF_IPX, AF_AX25, AF_APPLETALK
	};
	unsigned int i;
	int sock;

	for (i = 0; i < sizeof(families) / sizeof(int); ++i) {
		sock = socket(families[i], SOCK_DGRAM, 0);
		if (sock >= 0)
			return sock;
	}

	debug_printf(DEBUG_NORMAL, "Couldn't acquire a socket!  (%s:%d)\n",
		     __FUNCTION__, __LINE__);
	return -1;
}

// Taken from wpa supplicant
int iw_set_ext(const char *ifname, int request, struct iwreq *pwrq)
{
	int s = get_socket();
	int ret;

	if (s < 0)
		return -1;
	strncpy(pwrq->ifr_name, ifname, IFNAMSIZ);

	ret = ioctl(s, request, pwrq);
	close(s);
	return ret;
}

int cardif_ndiswrapper_driver_get_wpa_ie(struct interface_data *intdata,
					 char *iedata, int *ielen)
{
  wpa_gen_ie(intdata, iedata);
  *ielen = 24;
  return XENONE;
}

int cardif_ndiswrapper_driver_get_wpa2_ie(struct interface_data *intdata,
					  char *iedata, int *ielen)
{
  debug_printf(DEBUG_NORMAL, "WPA2/802.11i support is not implemented for "
	       "ndiswrapper driver!\n");
  return XENOTSUPPORTED;
}

int cardif_ndiswrapper_driver_wpa(struct interface_data *intdata,  char endis)
{
  struct iwreq priv_req;
  int ret = 0;
  
  memset(&priv_req, 0, sizeof(priv_req));
  
  priv_req.u.data.flags = endis;
  if (iw_set_ext(intdata->intName, WPA_SET_WPA, &priv_req) < 0)
    {
      ret = -1;
      debug_printf(DEBUG_NORMAL, "Error enabling WPA : Error (%d) - %s\n",
		   errno, strerror(errno));
    }
  return XENONE;
}

int cardif_ndiswrapper_driver_wpa_state(struct interface_data *thisint,  char endis)
{
  return cardif_ndiswrapper_driver_wpa(thisint, endis);
}

int cardif_ndiswrapper_driver_wpa_disable(char *intname)
{
  // XXX Finish!
  return XENONE;
}

// This function derived from WPA Supplicant.
int cardif_ndiswrapper_driver_set_key(char *dev, uint32_t alg, 
				      unsigned char *addr, int key_idx, 
				      int set_tx, char *seq, int seq_len,
				      char *key, int key_len)
{
  struct wpa_key wpa_key;
  int ret = 0;
  struct iwreq priv_req;
  
  bzero(&wpa_key, 0);
  memset(&priv_req, 0, sizeof(priv_req));

  wpa_key.alg = alg;
  wpa_key.addr = addr;
  wpa_key.key_index = key_idx;
  wpa_key.set_tx = set_tx;
  wpa_key.seq = seq;
  wpa_key.seq_len = seq_len;
  debug_printf(DEBUG_NORMAL, "SEQ : ");
  debug_hex_printf(DEBUG_NORMAL, seq, seq_len);
  wpa_key.key = key;
  wpa_key.key_len = key_len;

  debug_printf(DEBUG_INT, "Setting key.. alg=%d idx=%d tx=%d seqlen=%d "
	       "keylen=%d\n", wpa_key.alg, wpa_key.key_index, wpa_key.set_tx,
	       wpa_key.seq_len, wpa_key.key_len);

  debug_printf(DEBUG_NORMAL, "DUMP : \n");
  debug_hex_dump(DEBUG_NORMAL, (u_char *)&wpa_key, sizeof(wpa_key));

  debug_printf(DEBUG_NORMAL, "Key (%d) : ", wpa_key.key_len);
  debug_hex_printf(DEBUG_NORMAL, wpa_key.key, wpa_key.key_len);
  
  priv_req.u.data.pointer = (void *)&wpa_key;
  
  if (iw_set_ext(dev, WPA_SET_KEY, &priv_req) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error setting key! : Error (%d) - %s\n",
		   errno, strerror(errno));
      ret = -1;
    }
  return ret;
}

int cardif_ndiswrapper_driver_set_tkip(struct interface_data *intdata, 
				  unsigned char *addr, int key_idx, int set_tx,
				  char *seq, int seq_len, char *key, 
				  int key_len)
{
  return cardif_ndiswrapper_driver_set_key(intdata->intName, WPA_TKIP, addr, 
				      key_idx, set_tx, seq, seq_len, key, 
				      key_len);
}

int cardif_ndiswrapper_driver_set_ccmp(struct interface_data *intdata,
				       unsigned char *addr, int key_idx,
				       int set_tx, char *seq, int seq_len,
				       char *key, int key_len)
{
  return cardif_ndiswrapper_driver_set_key(intdata->intName, WPA_CCMP, addr,
					   key_idx, set_tx, seq, seq_len,
					   key, key_len);
}

int cardif_ndiswrapper_driver_delete_key(struct interface_data *intdata,
					 int keyidx, int settx)
{
  return cardif_ndiswrapper_driver_set_key(intdata->intName, WPA_NONE, NULL,
					   keyidx, settx, NULL, 0, NULL, 0);
}

// Derived from WPA Supplicant
int cardif_ndiswrapper_driver_disassociate(struct interface_data *intdata,
				      int reason_code)
{
  int ret = 0;
  struct iwreq priv_req;
  
  memset(&priv_req, 0, sizeof(priv_req));
  
  memcpy(&priv_req.u.ap_addr.sa_data, intdata->source_mac, 6);
  if (iw_set_ext(intdata->intName, WPA_DISASSOCIATE, &priv_req) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error disassociating! : Error (%d) - %s\n",
		   errno, strerror(errno));
      ret = -1;
    }
  return ret;
}

// Derived from WPA supplicant.
void cardif_ndiswrapper_driver_associate(struct interface_data *intdata,
					char *ssid)
{
  struct wpa_assoc_info wpa_inf;
  struct iwreq priv_req;
  char wpaie[26];
  char buf[IW_ESSID_MAX_SIZE];
  int ielen;
  struct config_network *network_data;

  network_data = config_get_network_config();
  
  if (network_data == NULL)
    {
      debug_printf(DEBUG_NORMAL, "Invalid network configuration structure! "
		   "(%s:%d)\n", __FUNCTION__, __LINE__);
      return;
    }

  intdata->flags |= DONTSCAN;

  cardif_ndiswrapper_driver_get_wpa_ie(intdata, wpaie, &ielen);

  bzero(&priv_req, sizeof(priv_req));
  bzero(&wpa_inf, sizeof(wpa_inf));
  
  memcpy(buf, ssid, strlen(ssid));
  wpa_inf.bssid = (char *)config_ssid_get_mac();
  wpa_inf.ssid = (char *)&buf;
  wpa_inf.ssid_len = strlen(ssid);
  wpa_inf.freq = config_ssid_get_freq();
  wpa_inf.wpa_ie = (char *)&wpaie;
  wpa_inf.wpa_ie_len = ielen;   

  wpa_inf.pairwise_suite = network_data->wpa_pairwise_crypt; 
  wpa_inf.group_suite = network_data->wpa_group_crypt;    
  
  if (network_data->methods->method_num == WPA_PSK)
    {
      wpa_inf.key_mgmt_suite = 2;   // WPA-PSK
    } else {
      wpa_inf.key_mgmt_suite = 1;   // WPA-EAP
    }

  priv_req.u.data.pointer = (void *)&wpa_inf;

  if (iw_set_ext(intdata->intName, WPA_ASSOCIATE, &priv_req) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Error attempting to associate! \n");
      debug_printf(DEBUG_NORMAL, "Error (%d) - %s (%s:%d)\n", errno, 
		   strerror(errno), __FUNCTION__, __LINE__);
      return;
    }

  return;
}

int cardif_ndiswrapper_driver_countermeasures(struct interface_data *intdata,
					      char endis)
{
  int ret = 0;
  struct iwreq priv_req;

  bzero(&priv_req, sizeof(priv_req));

  priv_req.u.param.value = endis;
  if (iw_set_ext(intdata->intName, WPA_SET_COUNTERMEASURES, &priv_req) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Could not set countermeasures! : Error (%d)"
		   " - %s\n", errno, strerror(errno));
      ret = -1;
    }

  return ret;
}

int cardif_ndiswrapper_driver_drop_unencrypted(struct interface_data *intdata,
					       char endis)
{
  int ret = 0;
  struct iwreq priv_req;

  bzero(&priv_req, sizeof(priv_req));
  
  priv_req.u.param.value = endis;
  if (iw_set_ext(intdata->intName, WPA_DROP_UNENCRYPTED, &priv_req) < 0)
    {
      debug_printf(DEBUG_NORMAL, "Couldn't enable 'drop unencrypted' mode! : "
		   "Error (%d) - %s\n", errno, strerror(errno));
      ret = -1;
    }

  return ret;
}

struct cardif_funcs cardif_ndiswrapper_driver = {
  .scan = cardif_linux_wext_scan,
  .disassociate = cardif_ndiswrapper_driver_disassociate,
  .set_wep_key = cardif_linux_wext_set_WEP_key,
  .set_tkip_key = cardif_ndiswrapper_driver_set_tkip,
  .set_ccmp_key = cardif_ndiswrapper_driver_set_ccmp,
  .delete_key = cardif_ndiswrapper_driver_delete_key,
  .set_ssid = cardif_linux_wext_set_ssid,
  .associate = cardif_ndiswrapper_driver_associate,
  .get_ssid = cardif_linux_wext_get_ssid,
  .get_bssid = cardif_linux_wext_get_bssid,
  .wpa_state = cardif_ndiswrapper_driver_wpa_state,
  .wpa = cardif_ndiswrapper_driver_wpa,
  .roam = cardif_linux_wext_roam,
  .countermeasures = cardif_ndiswrapper_driver_countermeasures,
  .drop_unencrypted = cardif_ndiswrapper_driver_drop_unencrypted,
  .get_wpa_ie = cardif_ndiswrapper_driver_get_wpa_ie,
  .enc_disable = cardif_linux_wext_enc_disable
};

#endif
