/*******************************************************************
 * Implementation for parsing configuration file, and storing needed data.
 *
 * Licensed under a dual GPL/BSD license.  (See LICENSE file for more info.)
 *
 * File: config.c
 *
 * Authors: Chris.Hessing@utah.edu
 *
 * $Id: config.c,v 1.76 2005/08/21 18:12:12 chessing Exp $
 * $Date: 2005/08/21 18:12:12 $
 * $Log: config.c,v $
 * Revision 1.76  2005/08/21 18:12:12  chessing
 * Added the ability to reload our configuration using a HUP.  Cleaned up a bunch of memory leaks in the config parse code.
 *
 * Revision 1.75  2005/08/20 19:06:52  chessing
 * Patch from Carsten Grohmann to fix a few things in xsup_get_state.c.  Also added the ability to define an empty network clause, that will set the card in to encryption disabled mode.  From there, anything short of changing the SSID will be ignored by Xsupplicant.
 *
 * Revision 1.74  2005/08/18 03:19:04  chessing
 * Added the ability to define an SSID with static WEP keys.  When we switch to a network that has this type of configuration we will set the keys, and stop the various association timers.
 *
 * Revision 1.73  2005/08/12 03:34:06  chessing
 * Fix to the TLS implementation, should help avoid some of the weird 'block cipher pad' errors.  Also includes a partial implementation of the ability to use static WEP keys based on the SSID in use.
 *
 * Revision 1.72  2005/08/09 01:39:13  chessing
 * Cleaned out old commit notes from the released version.  Added a few small features including the ability to disable the friendly warnings that are spit out.  (Such as the warning that is displayed when keys aren't rotated after 10 minutes.)  We should also be able to start when the interface is down.  Last, but not least, we can handle empty network configs.  (This may be useful for situations where there isn't a good reason to have a default network defined.)
 *
 *
 *******************************************************************/

#include <stdlib.h>
#include <strings.h>
#include <string.h>
#include <stdio.h>
#include <sys/types.h>
#include <grp.h>

#include "xsup_debug.h"
#include "xsup_err.h"
#include "config.h"
#include "wpa.h"

#ifdef EAP_SIM_ENABLE
#include "winscard.h"
#endif

// there has GOT to be a better way than this...
#include "eap_types/md5/eapmd5.h"
#include "eap_types/tls/eaptls.h"
#include "eap_types/ttls/eapttls.h"
#include "eap_types/mschapv2/eapmschapv2.h"
#include "eap_types/peap/eappeap.h"
#include "eap_types/leap/eapleap.h"
#include "eap_types/sim/eapsim.h"
#include "eap_types/aka/eapaka.h"

struct config_data *config_info = NULL; // the configuration data structure
struct config_network *current_network_data = NULL;  // Point to the config
                                                     // data we currently are
                                                     // using.

extern FILE *yyin;
extern int config_linenum;
extern int yyparse(void);

#define FREE_STRING(x) if (x != NULL) {free(x); x = NULL;}


/****************************************
 *
 * Load all of the configuration information in to memory.  We should
 * set a flag to make sure that we have loaded the config before we
 * call config_build().
 *
 ****************************************/
int config_setup(char *path_to_config)
{
  /* Make sure we got a parameter */
  if (path_to_config == NULL) {
    debug_printf(DEBUG_NORMAL, "Configuration file not given\n");
    debug_printf(DEBUG_NORMAL, "This should NEVER happen!\n");
    return XECONFIGFILEFAIL;
  }

  /* check to see if we can really open this file */
  yyin = fopen(path_to_config, "r");
  if (yyin == NULL) {
    debug_printf(DEBUG_NORMAL, "Failed to open configuration %s\n\n", path_to_config);
    return XECONFIGFILEFAIL;
  }
  
  /* check to see if the configuration is already set */
  if (config_info) {
    debug_printf(DEBUG_NORMAL, "config_setup called, but configuration is already loaded. Call config_destroy\n");
    return XECONFIGFILEFAIL;
  }

  /* parse the file */
  if (config_parse() != XENONE) {
    delete_config_data(&config_info);
    return XECONFIGPARSEFAIL;
  }

  // set the file name
  if (config_info)
    config_info->config_fname = strdup(path_to_config);

  return XENONE;
}


struct config_network *config_find_network(struct config_network *nethead, 
					   char *matchname)
{
  struct config_network *cur;

  cur = nethead;

  if ((nethead == NULL) || (matchname == NULL))
    {
      return NULL;
    }

  while ((cur != NULL) && (strcmp(cur->name, matchname) != 0))
    {
      cur = cur->next;
    }
  
  // If we got a match, return it.
  if (cur != NULL)
    {
      return cur;
    }
  
  // Otherwise, look against the essid.
  cur = nethead;

  while (cur != NULL)
  {
    if ((cur->ssid == NULL) || (strcmp(cur->ssid,matchname) != 0)) 
	{
	  cur = cur->next;
	} else {
  	  break;
	}
  }
  
  // Do we have a match on ssid?
  if (cur != NULL)
  {
	return cur;
  }

  return NULL;
}

/****************************************
 *
 * Return our network data structure.
 *
 ****************************************/
struct config_network *config_get_network_config()
{
  return current_network_data;
}

/****************************************
 *
 * Get configuration information out of memory, and 
 *
 ****************************************/
char config_build(char *network_name)
{
  struct config_network *result;

  if (config_info != NULL)
    {
      debug_printf(DEBUG_CONFIG, "Working from config file %s.\n",config_info->config_fname);

      // We were passed in a "network name".  First, look through the config
      // to see if it matches any friendly names.
      result = config_find_network(config_info->networks, network_name);

      if (result != NULL) 
	{
	  current_network_data = result;
	  return TRUE;
	}

      // This is not good.  We don't have any configuration information
      // for the requested network.  So, we need to return the default
      // information, and a warning.
      debug_printf(DEBUG_NORMAL, "No configuration information for network \"%s\" found.  Using default.\n", network_name);

      result = config_find_network(config_info->networks, 
				   config_info->globals->default_net);

      if (result != NULL) 
	{
	  current_network_data = result;
	  return TRUE;
	}

      // Uh oh..  We didn't find *anything*.
      debug_printf(DEBUG_NORMAL, "ERROR : No valid network profile could be located!  (Even tried default.)\n");
      debug_printf(DEBUG_NORMAL, "Abnormal program termination!!!!\n");
      exit(1);
    } else {
      debug_printf(DEBUG_CONFIG, "config_info == NULL!  No config to update!\n");
    }
  return FALSE;
}

/************************************
 *
 * Set statemachine/config related variables for this interface.
 *
 ************************************/
int config_set_globals(struct interface_data *myint)
{

  if (myint == NULL)
    {
      debug_printf(DEBUG_NORMAL, "No configuration information is available!\n");
      return -1;
    }
  // Start by seeing if we need to set any global values.
  if ((CONFIG_GLOBALS_AUTH_PER & config_info->globals->flags) ==
      CONFIG_GLOBALS_AUTH_PER)
    {
      myint->statemachine->authPeriod = config_info->globals->auth_period;
    }

  if ((CONFIG_GLOBALS_HELD_PER & config_info->globals->flags) ==
      CONFIG_GLOBALS_HELD_PER)
    {
      myint->statemachine->heldPeriod = config_info->globals->held_period;
    }

  if ((CONFIG_GLOBALS_MAX_STARTS & config_info->globals->flags) ==
      CONFIG_GLOBALS_MAX_STARTS)
    {
      myint->statemachine->maxStart = config_info->globals->max_starts;
    }
 
  return 0;
}

/************************************
 *
 * Return the GID of the group defined by ipc_group.
 *
 ************************************/
int config_get_ipc_gid()
{
  struct group *grdata = NULL;
  int result = -1;

  if (config_info->globals->ipc_group_name != NULL)
    {
      grdata = getgrnam(config_info->globals->ipc_group_name);
      if (grdata != NULL)
	{
	  result = grdata->gr_gid;
	  debug_printf(DEBUG_CONFIG, "IPC Group Name %s, number %d!\n",
		       grdata->gr_name, result);
	}
    }

  return result;
}

/************************************
 *
 * Return the way that we want to determine the destination for packets.
 *
 ************************************/
char config_get_destination()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "No configuration information available!  "
		   "Returning destination of AUTO.\n");
      return DEST_AUTO;
    }

  return config_info->globals->destination;
}

/************************************
 *
 * Get stale key timer.
 *
 ************************************/
int config_get_stale_key_timeout()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "Invalid configuration information!  Stale"
		   " key timer will be set to the default of %d seconds!\n",
		   STALE_KEY_WARN_TIMEOUT);
      return STALE_KEY_WARN_TIMEOUT;
    }
  return config_info->globals->stale_key_timeout;
}

/************************************
 *
 * Return the default interface that was in the configuration file.
 *
 ************************************/
char *config_get_default_int()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "No default interface set in the "
		   "configuration file!\n");
      return NULL;
    }

  return config_info->globals->default_int;
}

/************************************
 *
 * Return the log_facility value.
 *
 ************************************/
char *config_get_log_facility()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "No log facility defined!\n");
      return NULL;
    }

  return config_info->globals->log_facility;
}

/***********************************
 *
 * Return if we should display friendly warnings or not.
 *
 ***********************************/
char config_get_display_friendly_warnings()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "No configuration information available! "
		   "We will display all friendly warnings.\n");
      return TRUE;
    }

  if (TEST_FLAG(config_info->globals->flags, 
		CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
    {
      return FALSE;  // Don't display friendly warnings.
    } else {
      return TRUE;
    }

  return TRUE;
}

/***********************************
 *
 * Return if we should set allmulti mode, or not.
 *
 ***********************************/
char config_get_allmulti()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      debug_printf(DEBUG_CONFIG, "No configuration information available!  "
		   "ALLMULTI will return default value!\n");
      return 1;
    }

  if (TEST_FLAG(config_info->globals->flags, CONFIG_GLOBALS_ALLMULTI))
    {
      return TRUE;
    } else {
      return FALSE;
    }
}

/*****************************************************************************
 * NOTE: Do *NOT* debug_printf() in this function or you'll cause a recursive 
 * loop.
 *****************************************************************************/
char *config_get_logfile()
{
  if ((config_info == NULL) || (config_info->globals == NULL))
    {
      return NULL;
    }
  return config_info->globals->logfile;
}

/************************************
 *
 * Clean up any memory that we have used to store the configuration information
 * 
 ************************************/
void config_destroy()
{
  /* close the input file */
  if (yyin)
    fclose(yyin);

  /* see if there really is something to cleanup */
  delete_config_data(&config_info);
}

/************************************
 *
 * Temporary test function for parsing
 *
 ************************************/
int config_parse()
{
  if (yyparse() != XENONE) {
    return XECONFIGPARSEFAIL;
  }
  return XENONE;
}


//****************************************
// CONFIG QUERIES
//****************************************

/******************************************
 *
 * See if the network config is currently in memory
 *
 ******************************************/
int config_contains_network(char *netname) 
{
  if (!config_info || !config_info->networks)
    return FALSE;
  return config_network_contains_net(config_info->networks, netname);
}

/******************************************
 *
 * See if network config is  allowed
 * 
 ******************************************/
int config_allows_network(struct config_data *conf, char *netname)
{
  struct config_string_list *current;
  // make sure we have a config and globals
  if (!conf || !conf->globals)
    return FALSE;

  current = conf->globals->allowed_nets;
  
  // lack of an allowed list means all nets are allowed
  if (current == NULL) 
    return TRUE;

  if (config_string_list_contains_string(current, netname))
    return TRUE;

  return FALSE;
}


//**********************************************
// Private functions for config parsing. Do 
// not call these from outside config code
//**********************************************

  /*********************/
 /* CONFIG_STATIC_WEP */
/*********************/

void delete_config_static_wep(struct config_static_wep **tmp_static_wep)
{
  int i;

  if (*tmp_static_wep == NULL)
    return;

  for (i=1;i<5;i++)
    {
      FREE_STRING((*tmp_static_wep)->key[i]);
    }

  free(*tmp_static_wep);
  *tmp_static_wep = NULL;
}

void initialize_config_static_wep(struct config_static_wep **tmp_static_wep)
{
  if (*tmp_static_wep != NULL) {
    delete_config_static_wep(tmp_static_wep);
  }
  *tmp_static_wep = 
    (struct config_static_wep *)malloc(sizeof(struct config_static_wep));
  if (*tmp_static_wep)
    memset(*tmp_static_wep, 0, sizeof(struct config_static_wep));
}

void dump_config_static_wep(struct config_static_wep *tmp_static_wep)
{
  int i;

  if (!tmp_static_wep)
    return;
  debug_printf(DEBUG_NORMAL, "\t-------------static-wep-------------\n");
    debug_printf(DEBUG_NORMAL, "\t TX Key       : %d\n", tmp_static_wep->tx_key);

    for (i=1;i<5;i++)
      {
	if (tmp_static_wep->key[i])
	  debug_printf(DEBUG_NORMAL, "\t Static Key %d : %s\n",
		       tmp_static_wep->key[i]);
      }
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_WPA_PSK  */
/*******************/

void delete_config_wpa_psk(struct config_wpa_psk **tmp_wpa_psk)
{
  if (*tmp_wpa_psk == NULL)
    return;

  FREE_STRING((*tmp_wpa_psk)->key);
  FREE_STRING((*tmp_wpa_psk)->hex_key);

  free(*tmp_wpa_psk);
  *tmp_wpa_psk = NULL;
}

void initialize_config_wpa_psk(struct config_wpa_psk **tmp_wpa_psk)
{
  if (*tmp_wpa_psk != NULL) {
    delete_config_wpa_psk(tmp_wpa_psk);
  }
  *tmp_wpa_psk = 
    (struct config_wpa_psk *)malloc(sizeof(struct config_wpa_psk));
  if (*tmp_wpa_psk)
    memset(*tmp_wpa_psk, 0, sizeof(struct config_wpa_psk));
}

void dump_config_wpa_psk(struct config_wpa_psk *tmp_wpa_psk)
{
  if (!tmp_wpa_psk)
    return;
  debug_printf(DEBUG_NORMAL, "\t---------------wpa-psk--------------\n");
  debug_printf(DEBUG_NORMAL, "\t  ASCII key: \"%s\"\n", tmp_wpa_psk->key);
  debug_printf(DEBUG_NORMAL, "\t  HEX key: \"%s\"\n", tmp_wpa_psk->hex_key);
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_TLS      */
/*******************/

/* take a pointer to a config_eap_tls and cleanly delete the structure
   then make it NULL */
void delete_config_eap_tls(struct config_eap_tls **tmp_tls)
{
  if (*tmp_tls == NULL)
    return;

  FREE_STRING((*tmp_tls)->user_cert);
  FREE_STRING((*tmp_tls)->root_cert);
  FREE_STRING((*tmp_tls)->root_dir);  
  FREE_STRING((*tmp_tls)->crl_dir);  
  FREE_STRING((*tmp_tls)->user_key);
  FREE_STRING((*tmp_tls)->user_key_pass);
  FREE_STRING((*tmp_tls)->random_file);
  FREE_STRING((*tmp_tls)->sc.engine_id);
  FREE_STRING((*tmp_tls)->sc.opensc_so_path);
  FREE_STRING((*tmp_tls)->sc.key_id);
  
  free (*tmp_tls);
  *tmp_tls = NULL;
}

/* take a pointer to a config_eap_tls and put a blank one there */
void initialize_config_eap_tls(struct config_eap_tls **tmp_tls)
{
  if (*tmp_tls != NULL) {
    delete_config_eap_tls(tmp_tls);
  }
  *tmp_tls = 
    (struct config_eap_tls *)malloc(sizeof(struct config_eap_tls));
  if (*tmp_tls)
    {
      memset(*tmp_tls, 0, sizeof(struct config_eap_tls));
      bzero(&(*tmp_tls)->sc, sizeof((*tmp_tls)->sc));
    }
}

void dump_config_eap_tls(struct config_eap_tls *tls)
{
  if (!tls)
    return;
  debug_printf(DEBUG_NORMAL, "\t---------------eap-tls--------------\n");
  debug_printf(DEBUG_NORMAL, "\t  TLS Cert: \"%s\"\n", tls->user_cert);
  debug_printf(DEBUG_NORMAL, "\t  TLS Root Cert: \"%s\"\n", tls->root_cert);
  debug_printf(DEBUG_NORMAL, "\t  TLS Root Dir: \"%s\"\n", tls->root_dir);
  debug_printf(DEBUG_NORMAL, "\t  TLS CRL Dir: \"%s\"\n", tls->crl_dir);
  debug_printf(DEBUG_NORMAL, "\t  TLS Key: \"%s\"\n", tls->user_key);
  debug_printf(DEBUG_NORMAL, "\t  TLS Key Pass: \"%s\"\n", tls->user_key_pass);
  debug_printf(DEBUG_NORMAL, "\t  TLS Chunk Size: %d\n", tls->chunk_size);
  debug_printf(DEBUG_NORMAL, "\t  TLS Random Source: \"%s\"\n", 
	       tls->random_file);
  debug_printf(DEBUG_NORMAL, "\t  TLS Session Resumption: ");
  switch (tls->session_resume)
    {
    case RES_UNSET:
      debug_printf_nl(DEBUG_NORMAL, "UNSET\n");
      break;
    case RES_YES:
      debug_printf_nl(DEBUG_NORMAL, "YES\n");
      break;
    case RES_NO:
      debug_printf_nl(DEBUG_NORMAL, "NO\n");
      break;
    }
  debug_printf(DEBUG_NORMAL, "\tUsing Smartcard:\n");
  debug_printf(DEBUG_NORMAL, "\t\tEngine        : \"%s\"\n", tls->sc.engine_id);
  debug_printf(DEBUG_NORMAL, "\t\tOpensc SO_PATH: \"%s\"\n", tls->sc.opensc_so_path);
  debug_printf(DEBUG_NORMAL, "\t\tKey ID        : \"%s\"\n", tls->sc.key_id);
  //debug_printf(DEBUG_NORMAL, "\t\tCertificate ID: \"%s\"", tls.sc->cert_id);
  //debug_printf(DEBUG_NORMAL, "\t\tRoot Cert ID  : \"%s\"", tls.sc->root_id); 
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}


  /*******************/
 /* CONFIG_MD5      */
/*******************/
void delete_config_eap_md5(struct config_eap_md5 **tmp_md5)
{
  if (*tmp_md5 == NULL)
    return;

  FREE_STRING((*tmp_md5)->username);
  FREE_STRING((*tmp_md5)->password);

  free (*tmp_md5);
  *tmp_md5 = NULL;
}

void initialize_config_eap_md5(struct config_eap_md5 **tmp_md5)
{
  if (*tmp_md5 != NULL) {
    delete_config_eap_md5(tmp_md5);
  }
  *tmp_md5 = 
    (struct config_eap_md5 *)malloc(sizeof(struct config_eap_md5));  
  if (*tmp_md5)
    memset(*tmp_md5, 0, sizeof(struct config_eap_md5));
}

void dump_config_eap_md5(struct config_eap_md5 *md5, int level)
{
  if (!md5)
    return;
  if (level == 0) {
    debug_printf(DEBUG_NORMAL, "\t---------------eap-md5--------------\n");
    debug_printf(DEBUG_NORMAL, "\t  MD5 User: \"%s\"\n", md5->username);
    debug_printf(DEBUG_NORMAL, "\t  MD5 Pass: \"%s\"\n", md5->password);
    debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
  }else {
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  eap-md5  ^ ^ ^\n");
    debug_printf(DEBUG_NORMAL, "\t\t  MD5 User: \"%s\"\n", 
		 md5->username);
    debug_printf(DEBUG_NORMAL, "\t\t  MD5 Pass: \"%s\"\n", 
		 md5->password);
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");    
  }
  
}


  /*******************/
 /* CONFIG_TTLS     */
/*******************/

//--------
// PAP
//--------
void delete_config_pap(struct config_pap **tmp_pap)
{
  if (*tmp_pap == NULL)
    return;

  FREE_STRING((*tmp_pap)->username);
  FREE_STRING((*tmp_pap)->password);

  free (*tmp_pap);
  *tmp_pap = NULL;
}

void initialize_config_pap(struct config_pap **tmp_pap)
{
  if (*tmp_pap != NULL) {
    delete_config_pap(tmp_pap);
  }
  *tmp_pap = 
    (struct config_pap *)malloc(sizeof(struct config_pap));  
  if (*tmp_pap)
    memset(*tmp_pap, 0, sizeof(struct config_pap));
}

void dump_config_pap(struct config_pap *pap)
{
  if (!pap)
    return;
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  pap  ^ ^ ^\n");
  debug_printf(DEBUG_NORMAL, "\t\t  PAP User: \"%s\"\n", pap->username);
  debug_printf(DEBUG_NORMAL, "\t\t  PAP Pass: \"%s\"\n", pap->password);
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// CHAP
//--------
void delete_config_chap(struct config_chap **tmp_chap)
{
  if (*tmp_chap == NULL)
    return;

  FREE_STRING((*tmp_chap)->username);
  FREE_STRING((*tmp_chap)->password);

  free (*tmp_chap);
  *tmp_chap = NULL;
}

void initialize_config_chap(struct config_chap **tmp_chap)
{
  if (*tmp_chap != NULL) {
    delete_config_chap(tmp_chap);
  }
  *tmp_chap = 
    (struct config_chap *)malloc(sizeof(struct config_chap));  
  if (*tmp_chap)
    memset(*tmp_chap, 0, sizeof(struct config_chap));
}

void dump_config_chap(struct config_chap *chap)
{
  if (!chap)
    return;
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  chap  ^ ^ ^\n");
  debug_printf(DEBUG_NORMAL, "\t\t  CHAP User: \"%s\"\n", chap->username);
  debug_printf(DEBUG_NORMAL, "\t\t  CHAP Pass: \"%s\"\n", chap->password);
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// MSCHAP
//--------
void delete_config_mschap(struct config_mschap **tmp_mschap)
{
  if (*tmp_mschap == NULL)
    return;

  FREE_STRING((*tmp_mschap)->username);
  FREE_STRING((*tmp_mschap)->password);

  free (*tmp_mschap);
  *tmp_mschap = NULL;
}

void initialize_config_mschap(struct config_mschap **tmp_mschap)
{
  if (*tmp_mschap != NULL) {
    delete_config_mschap(tmp_mschap);
  }
  *tmp_mschap = 
    (struct config_mschap *)malloc(sizeof(struct config_mschap));  
  if (*tmp_mschap)
    memset(*tmp_mschap, 0, sizeof(struct config_mschap));
}

void dump_config_mschap(struct config_mschap *mschap)
{
  if (!mschap)
    return;
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  mschap  ^ ^ ^\n");
  debug_printf(DEBUG_NORMAL, "\t\t  MSCHAP User: \"%s\"\n", mschap->username);
  debug_printf(DEBUG_NORMAL, "\t\t  MSCHAP Pass: \"%s\"\n", mschap->password);
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}

//--------
// MSCHAPV2
//--------
void delete_config_mschapv2(struct config_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 == NULL)
    return;

  FREE_STRING((*tmp_mschapv2)->username);
  FREE_STRING((*tmp_mschapv2)->password);

  free (*tmp_mschapv2);
  *tmp_mschapv2 = NULL;
}

void initialize_config_mschapv2(struct config_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 != NULL) {
    delete_config_mschapv2(tmp_mschapv2);
  }
  *tmp_mschapv2 = 
    (struct config_mschapv2 *)malloc(sizeof(struct config_mschapv2));  
  if (*tmp_mschapv2)
    memset(*tmp_mschapv2, 0, sizeof(struct config_mschapv2));
}

void dump_config_mschapv2(struct config_mschapv2 *mschapv2)
{
  if (!mschapv2)
    return;
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  mschapv2  ^ ^ ^\n");
  debug_printf(DEBUG_NORMAL, "\t\t  MSCHAPv2 User: \"%s\"\n", 
	       mschapv2->username);
  debug_printf(DEBUG_NORMAL, "\t\t  MSCHAPv2 Pass: \"%s\"\n", 
	       mschapv2->password);
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^\n");
}


//-------------
// TTLS_PHASE2
//------------
// Be SURE to call config_ttls_phase2_contains_phase2 BEFORE adding.
// no such check will be done here.
void add_config_ttls_phase2(struct config_ttls_phase2 **phase2,
			   ttls_phase2_type phase2_type, void *phase2_data)
{
  struct config_ttls_phase2 *tmp, *newphase2;

  if (!phase2_data)
    return;

  newphase2 = 
    (struct config_ttls_phase2 *)malloc(sizeof(struct config_ttls_phase2));
  if (newphase2 == NULL)
    return;
  memset(newphase2, 0, sizeof(struct config_ttls_phase2));
  newphase2->phase2_type = phase2_type;
  newphase2->phase2_data = phase2_data;
  
  if (*phase2 == NULL) {
    *phase2 = newphase2;
    return;
  }

  tmp = *phase2;

  while (tmp->next != NULL) {
    tmp = tmp->next;
  }
  tmp->next = newphase2;
}

int config_ttls_phase2_contains_phase2(struct config_ttls_phase2 *phase2,
				      ttls_phase2_type new_type)
{
  struct config_ttls_phase2 *tmp;

  if (!phase2)
    return 0;
  
  tmp = phase2;
  while (tmp) {
    if (tmp->phase2_type == new_type)
      return 1;
    tmp = tmp->next;
  }

  return 0;
}

void delete_config_ttls_phase2 (struct config_ttls_phase2 **phase2)
{
  struct config_ttls_phase2 *temp;

  if (*phase2 == NULL)
    return;
  switch ((*phase2)->phase2_type) {
  case TTLS_PHASE2_PAP:
    delete_config_pap((struct config_pap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_CHAP: 
    delete_config_chap((struct config_chap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAP:
    delete_config_mschap((struct config_mschap **)&(*phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAPV2:
    delete_config_mschapv2((struct config_mschapv2 **)&(*phase2)->phase2_data);
    break;
  default:
    debug_printf(DEBUG_NORMAL, "AAAH! Trying to delete an undefined config"
		 " type.\nNotify developers. Type: 0x%x\n", 
		 (*phase2)->phase2_type);
  }
  temp = (*phase2)->next;
  free((*phase2));

  if (temp)
    delete_config_ttls_phase2(&temp);
}

void dump_config_ttls_phase2(struct config_ttls_phase2 *phase2) {
  if (phase2 == NULL)
    return;
  switch ((phase2)->phase2_type) {
  case TTLS_PHASE2_PAP:
    dump_config_pap((struct config_pap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_CHAP: 
    dump_config_chap((struct config_chap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAP:
    dump_config_mschap((struct config_mschap *)(phase2)->phase2_data);
    break;
  case TTLS_PHASE2_MSCHAPV2:
    dump_config_mschapv2((struct config_mschapv2 *)(phase2)->phase2_data);
    break;
  default:
    debug_printf(DEBUG_NORMAL, "AAAH! Trying to dump an undefined config"
		 " type.\nNotify developers. Type: 0x%x\n", 
		 (phase2)->phase2_type);
  }
  if ((phase2)->next)
    dump_config_ttls_phase2((phase2)->next);
}

void delete_config_eap_ttls(struct config_eap_ttls **tmp_ttls)
{
  if (*tmp_ttls == NULL)
    return;

  FREE_STRING((*tmp_ttls)->user_cert);
  FREE_STRING((*tmp_ttls)->root_cert);
  FREE_STRING((*tmp_ttls)->root_dir);
  FREE_STRING((*tmp_ttls)->crl_dir);
  FREE_STRING((*tmp_ttls)->user_key);
  FREE_STRING((*tmp_ttls)->user_key_pass);
  FREE_STRING((*tmp_ttls)->random_file);  
  FREE_STRING((*tmp_ttls)->cncheck);
  if ((*tmp_ttls)->phase2) 
    delete_config_ttls_phase2(&(*tmp_ttls)->phase2);

  free (*tmp_ttls);
  *tmp_ttls = NULL;
}

void initialize_config_eap_ttls(struct config_eap_ttls **tmp_ttls)
{
  if (*tmp_ttls != NULL) {
    delete_config_eap_ttls(tmp_ttls);
  }
  *tmp_ttls = 
    (struct config_eap_ttls *)malloc(sizeof(struct config_eap_ttls));  
  if (*tmp_ttls == NULL)
    return;
  memset(*tmp_ttls, 0, sizeof(struct config_eap_ttls));

  (*tmp_ttls)->phase2_type = TTLS_PHASE2_UNDEFINED;
}

void dump_config_eap_ttls(struct config_eap_ttls *ttls)
{
  if (!ttls) {
    return;
  }
  debug_printf(DEBUG_NORMAL, "\t---------------eap-ttls--------------\n");
  debug_printf(DEBUG_NORMAL, "\t  TTLS Cert: \"%s\"\n", ttls->user_cert);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Root Cert: \"%s\"\n", ttls->root_cert);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Root Dir: \"%s\"\n", ttls->root_dir);
  debug_printf(DEBUG_NORMAL, "\t  TTLS CRL Dir: \"%s\"\n", ttls->crl_dir);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Key: \"%s\"\n", ttls->user_key);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Key Pass: \"%s\"\n", ttls->user_key_pass);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Chunk Size: %d\n", ttls->chunk_size);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Random Source: \"%s\"\n", 
	       ttls->random_file);
  debug_printf(DEBUG_NORMAL, "\t  TTLS CN to Check : \"%s\"\n", ttls->cncheck);
  debug_printf(DEBUG_NORMAL, "\t  TTLS Exact CN Match : %s\n",  
	       ttls->cnexact ? "yes" : "no"); 
  debug_printf(DEBUG_NORMAL, "\t  TTLS Session Resumption: ");
  switch (ttls->session_resume)
    {
    case RES_UNSET:
      debug_printf_nl(DEBUG_NORMAL, "UNSET\n");
      break;
    case RES_YES:
      debug_printf_nl(DEBUG_NORMAL, "YES\n");
      break;
    case RES_NO:
      debug_printf_nl(DEBUG_NORMAL, "NO\n");
      break;
    }
  switch (ttls->phase2_type) {
  case TTLS_PHASE2_PAP:
    debug_printf(DEBUG_NORMAL, "\t  TTLS phase2: pap\n");    
    break;
  case TTLS_PHASE2_CHAP:
    debug_printf(DEBUG_NORMAL, "\t  TTLS phase2: chap\n");    
    break;
  case TTLS_PHASE2_MSCHAP:
    debug_printf(DEBUG_NORMAL, "\t  TTLS phase2: mschap\n");    
    break;
  case TTLS_PHASE2_MSCHAPV2:
    debug_printf(DEBUG_NORMAL, "\t  TTLS phase2: mschapv2\n");        
    break;
  default:
    debug_printf(DEBUG_NORMAL, "\t  TTLS phase2: UNDEFINED\n");    
    break;
  }
  if (ttls->phase2) dump_config_ttls_phase2(ttls->phase2);
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}

int check_config_eap_ttls(struct config_eap_ttls *tmp_ttls)
{
  int errno = 0;
  
  if (tmp_ttls->phase2_type == TTLS_PHASE2_UNDEFINED || !tmp_ttls->phase2) {
    debug_printf(DEBUG_NORMAL, "No phase2 defined for ttls\n");
    errno = -1;
  }
  
  if (!config_ttls_phase2_contains_phase2(tmp_ttls->phase2, 
					  tmp_ttls->phase2_type)) {
    debug_printf(DEBUG_NORMAL, "Phase2 type chosen, but not defined.\n");
    errno = -1;      
  }
  return errno;
}

  /*******************/
 /* CONFIG_LEAP     */
/*******************/
void delete_config_eap_leap(struct config_eap_leap **tmp_leap)
{
  if (*tmp_leap == NULL)
    return;

  FREE_STRING((*tmp_leap)->username);
  FREE_STRING((*tmp_leap)->password);

  free (*tmp_leap);
  *tmp_leap = NULL;
}

void initialize_config_eap_leap(struct config_eap_leap **tmp_leap)
{
  if (*tmp_leap != NULL) {
    delete_config_eap_leap(tmp_leap);
  }
  *tmp_leap = 
    (struct config_eap_leap *)malloc(sizeof(struct config_eap_leap));  
  if (*tmp_leap)
    memset(*tmp_leap, 0, sizeof(struct config_eap_leap));
}

void dump_config_eap_leap(struct config_eap_leap *leap)
{
  if (!leap)
    return;
  debug_printf(DEBUG_NORMAL, "\t---------------eap-leap--------------\n");
  debug_printf(DEBUG_NORMAL, "\t  LEAP User: \"%s\"\n", leap->username);
  debug_printf(DEBUG_NORMAL, "\t  LEAP Pass: \"%s\"\n", leap->password);
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}

  /*******************/
 /* CONFIG_MSCHAPV2 */
/*******************/
void delete_config_eap_mschapv2(struct config_eap_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 == NULL)
    return;

  FREE_STRING((*tmp_mschapv2)->username);
  FREE_STRING((*tmp_mschapv2)->password);

  free (*tmp_mschapv2);
  *tmp_mschapv2 = NULL;
}

void initialize_config_eap_mschapv2(struct config_eap_mschapv2 **tmp_mschapv2)
{
  if (*tmp_mschapv2 != NULL) {
    delete_config_eap_mschapv2(tmp_mschapv2);
  }
  *tmp_mschapv2 = 
    (struct config_eap_mschapv2 *)malloc(sizeof(struct config_eap_mschapv2));  
  if (*tmp_mschapv2)
    memset(*tmp_mschapv2, 0, sizeof(struct config_eap_mschapv2));
}

void dump_config_eap_mschapv2(struct config_eap_mschapv2 *mschapv2, int level)
{
  if (!mschapv2)
    return;
  if (level == 0) {
    debug_printf(DEBUG_NORMAL, "\t---------------eap-mschapv2--------------\n");
    debug_printf(DEBUG_NORMAL, "\t  MSCHAPV2 User: \"%s\"\n", 
		 mschapv2->username);
    debug_printf(DEBUG_NORMAL, "\t  MSCHAPV2 Pass: \"%s\"\n", 
		 mschapv2->password);
    debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
  }else {
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  eap-mschapv2  ^ ^ ^\n");
    debug_printf(DEBUG_NORMAL, "\t\t  MSCHAPV2 User: \"%s\"\n", 
		 mschapv2->username);
    debug_printf(DEBUG_NORMAL, "\t\t  MSCHAPV2 Pass: \"%s\"\n", 
		 mschapv2->password);
  debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }

}


  /*******************/
 /* CONFIG_PEAP     */
/*******************/
void delete_config_eap_peap(struct config_eap_peap **tmp_peap)
{
  if (*tmp_peap == NULL)
    return;

  FREE_STRING((*tmp_peap)->identity);
  FREE_STRING((*tmp_peap)->user_cert);
  FREE_STRING((*tmp_peap)->root_cert);
  FREE_STRING((*tmp_peap)->root_dir);
  FREE_STRING((*tmp_peap)->crl_dir);
  FREE_STRING((*tmp_peap)->user_key)
  FREE_STRING((*tmp_peap)->user_key_pass);
  FREE_STRING((*tmp_peap)->random_file);
  FREE_STRING((*tmp_peap)->cncheck);
  if ((*tmp_peap)->phase2)
    delete_config_eap_method(&(*tmp_peap)->phase2);

  free (*tmp_peap);
  *tmp_peap = NULL;
}

void initialize_config_eap_peap(struct config_eap_peap **tmp_peap)
{
  if (*tmp_peap != NULL) {
    delete_config_eap_peap(tmp_peap);
  }
  *tmp_peap = 
    (struct config_eap_peap *)malloc(sizeof(struct config_eap_peap));  
  if (*tmp_peap)
    {
      memset(*tmp_peap, 0, sizeof(struct config_eap_peap));
    }
}

void dump_config_eap_peap(struct config_eap_peap *peap)
{
  if (!peap)
    return;
  debug_printf(DEBUG_NORMAL, "\t---------------eap-peap--------------\n");
  debug_printf(DEBUG_NORMAL, "\t  PEAP phase2 identity: \"%s\"\n", 
	       peap->identity);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Cert: \"%s\"\n", peap->user_cert);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Root Cert: \"%s\"\n", peap->root_cert);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Root Dir: \"%s\"\n", peap->root_dir);
  debug_printf(DEBUG_NORMAL, "\t  PEAP CRL Dir: \"%s\"\n", peap->crl_dir);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Key: \"%s\"\n", peap->user_key);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Key Pass: \"%s\"\n", peap->user_key_pass);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Chunk Size: %d\n", peap->chunk_size);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Random Source: \"%s\"\n", 
	       peap->random_file);
  debug_printf(DEBUG_NORMAL, "\t  PEAP CN to Check : \"%s\"\n", peap->cncheck);
  debug_printf(DEBUG_NORMAL, "\t  PEAP Exact CN Match : %s\n",  
	       peap->cnexact ? "yes" : "no");  
  debug_printf(DEBUG_NORMAL, "\t  PEAP Session Resumption: ");
  switch (peap->session_resume)
    {
    case RES_UNSET:
      debug_printf_nl(DEBUG_NORMAL, "UNSET\n");
      break;
    case RES_YES:
      debug_printf_nl(DEBUG_NORMAL, "YES\n");
      break;
    case RES_NO:
      debug_printf_nl(DEBUG_NORMAL, "NO\n");
      break;
    }
  debug_printf(DEBUG_NORMAL, "\t  Broken PEAPv1 Keying : ");
  switch (peap->proper_peapv1)
    {
    case 0:
      debug_printf_nl(DEBUG_NORMAL, "NO\n");
      break;
    case 1:
      debug_printf_nl(DEBUG_NORMAL, "YES\n");
      break;
    }
  debug_printf(DEBUG_NORMAL, "\t  PEAP IAS_QUIRK: \"%s\"\n", 
	       peap->ias_quirk ? "Yes" : "No");
    
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_MSCV2))
    debug_printf(DEBUG_NORMAL,"\t   Allow Phase 2 Type: MSCHAPv2\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_MD5))
    debug_printf(DEBUG_NORMAL,"\t   Allow Phase 2 Type: MD5\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_SIM))
    debug_printf(DEBUG_NORMAL,"\t   Allow Phase 2 Type: SIM\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_GTC))
    debug_printf(DEBUG_NORMAL,"\t   Allow Phase 2 Type: GTC\n");
  if (TEST_FLAG(peap->flags, CONFIG_PEAP_ALLOW_OTP))
    debug_printf(DEBUG_NORMAL,"\t   Allow Phase 2 Type: OTP\n");
  if (peap->phase2) dump_config_eap_method(peap->phase2, 1);
  debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
}


  /*******************/
 /* CONFIG_SIM      */
/*******************/
void delete_config_eap_sim(struct config_eap_sim **tmp_sim)
{
  if (*tmp_sim == NULL)
    return;

  FREE_STRING((*tmp_sim)->username);
  FREE_STRING((*tmp_sim)->password);

  free (*tmp_sim);
  *tmp_sim = NULL;
}

void initialize_config_eap_sim(struct config_eap_sim **tmp_sim)
{
  if (*tmp_sim != NULL) {
    delete_config_eap_sim(tmp_sim);
  }
  *tmp_sim = 
    (struct config_eap_sim *)malloc(sizeof(struct config_eap_sim));  
  if (*tmp_sim)
    memset(*tmp_sim, 0, sizeof(struct config_eap_sim));
}

void dump_config_eap_sim(struct config_eap_sim *sim, int level)
{
  if (!sim)
    return;
  if (level == 0) {
    debug_printf(DEBUG_NORMAL, "\t---------------eap-sim--------------\n");
    debug_printf(DEBUG_NORMAL, "\t  SIM User: \"%s\"\n", sim->username);
    debug_printf(DEBUG_NORMAL, "\t  SIM Pass: \"%s\"\n", sim->password);
    debug_printf(DEBUG_NORMAL, "\t  SIM Auto Realm: %s\n",  
		 sim->auto_realm ? "yes" : "no");  
    debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
  } else {
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  eap-sim  ^ ^ ^\n");
    debug_printf(DEBUG_NORMAL, "\t\t  SIM User: \"%s\"\n", 
		 sim->username);
    debug_printf(DEBUG_NORMAL, "\t\t  SIM Pass: \"%s\"\n", 
		 sim->password);
    debug_printf(DEBUG_NORMAL, "\t\t  SIM Auto Realm: %s\n",  
		 sim->auto_realm ? "yes" : "no");  
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }
}

  /*******************/
 /* CONFIG_AKA      */
/*******************/
void delete_config_eap_aka(struct config_eap_aka **tmp_aka)
{
  if (*tmp_aka == NULL)
    return;

  FREE_STRING((*tmp_aka)->username);
  FREE_STRING((*tmp_aka)->password);

  free (*tmp_aka);
  *tmp_aka = NULL;
}

void initialize_config_eap_aka(struct config_eap_aka **tmp_aka)
{
  if (*tmp_aka != NULL) {
    delete_config_eap_aka(tmp_aka);
  }
  *tmp_aka = 
    (struct config_eap_aka *)malloc(sizeof(struct config_eap_aka));  
  if (*tmp_aka)
    memset(*tmp_aka, 0, sizeof(struct config_eap_aka));
}

void dump_config_eap_aka(struct config_eap_aka *aka, int level)
{
  if (!aka)
    return;
  if (level == 0) {
    debug_printf(DEBUG_NORMAL, "\t---------------eap-aka--------------\n");
    debug_printf(DEBUG_NORMAL, "\t  AKA User: \"%s\"\n", aka->username);
    debug_printf(DEBUG_NORMAL, "\t  AKA Pass: \"%s\"\n", aka->password);
    debug_printf(DEBUG_NORMAL, "\t  AKA Auto Realm: %s\n",  
		 aka->auto_realm ? "yes" : "no");  
    debug_printf(DEBUG_NORMAL, "\t------------------------------------\n");
  } else {
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^  eap-aka  ^ ^ ^\n");
    debug_printf(DEBUG_NORMAL, "\t\t  AKA User: \"%s\"\n", 
		 aka->username);
    debug_printf(DEBUG_NORMAL, "\t\t  AKA Pass: \"%s\"\n", 
		 aka->password);
    debug_printf(DEBUG_NORMAL, "\t\t  AKA Auto Realm: %s\n",  
		 aka->auto_realm ? "yes" : "no");  
    debug_printf(DEBUG_NORMAL, "\t\t^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^ ^\n");
  }
}


  /*********************/
 /* CONFIG_EAP_METHOD */
/*********************/

// Be SURE to call config_eap_method_contains_method BEFORE adding.
// no such check will be done here.
void add_config_eap_method(struct config_eap_method **method,
			   int method_num, void *method_data)
{
  struct config_eap_method *tmp, *newmethod;

  if (!method_data)
    return;

  newmethod = 
    (struct config_eap_method *)malloc(sizeof(struct config_eap_method));
  if (newmethod == NULL)
    return;
  memset(newmethod, 0, sizeof(struct config_eap_method));
  newmethod->method_num = method_num;
  newmethod->method_data = method_data;
  
  if (*method == NULL) {
    *method = newmethod;
    return;
  }

  tmp = *method;

  while (tmp->next != NULL) {
    tmp = tmp->next;
  }
  tmp->next = newmethod;
}

void delete_config_eap_method(struct config_eap_method **method)
{
  struct config_eap_method *temp;

  if (*method == NULL)
    return;
  switch ((*method)->method_num) {
  case STATIC_WEP_METHOD:
    delete_config_static_wep((struct config_static_wep **)&((*method)->method_data));
    break;
  case WPA_PSK:
    delete_config_wpa_psk((struct config_wpa_psk **)&((*method)->method_data));
    break;
  case EAP_TYPE_TLS:
    delete_config_eap_tls((struct config_eap_tls **)&((*method)->method_data));
    break;
  case EAP_TYPE_MD5:
    delete_config_eap_md5((struct config_eap_md5 **)&(*method)->method_data);
    break;
  case EAP_TYPE_PEAP:
    delete_config_eap_peap((struct config_eap_peap **)&(*method)->method_data);
    break;

#ifdef EAP_SIM_ENABLE
  case EAP_TYPE_SIM:
    delete_config_eap_sim((struct config_eap_sim **)&(*method)->method_data);
    break;
  case EAP_TYPE_AKA:
    delete_config_eap_aka((struct config_eap_aka **)&(*method)->method_data);
    break;
#endif

  case EAP_TYPE_TTLS:
    delete_config_eap_ttls((struct config_eap_ttls **)&(*method)->method_data);
    break; 
  case EAP_TYPE_LEAP:
    delete_config_eap_leap((struct config_eap_leap **)&(*method)->method_data);
    break;
  case EAP_TYPE_MSCHAPV2:
    delete_config_eap_mschapv2((struct config_eap_mschapv2 **)&(*method)->method_data);
    break;
    
  default:
    debug_printf(DEBUG_NORMAL, "AAAH! Trying to delete an undefined config"
		 " type.\nNotify developers. Type: 0x%x\n", 
		 (*method)->method_num);
  }

  temp = (*method)->next;
  free((*method));

  if (temp)
    delete_config_eap_method(&temp);
  
}

void dump_config_eap_method(struct config_eap_method *method, int dumplevel)
{
  if (method == NULL)
    return;
  switch ((method)->method_num) {
  case STATIC_WEP_METHOD:
    dump_config_static_wep((struct config_static_wep *)((method)->method_data));
    break;
  case WPA_PSK:
    dump_config_wpa_psk((struct config_wpa_psk *)((method)->method_data));
    break;
  case EAP_TYPE_TLS:
    dump_config_eap_tls((struct config_eap_tls *)((method)->method_data));
    break;
  case EAP_TYPE_MD5:
    dump_config_eap_md5((struct config_eap_md5 *)(method)->method_data, 
			dumplevel);
    break;
  case EAP_TYPE_PEAP:
    dump_config_eap_peap((struct config_eap_peap *)(method)->method_data);
    break;

#ifdef EAP_SIM_ENABLE
  case EAP_TYPE_SIM:
    dump_config_eap_sim((struct config_eap_sim *)(method)->method_data,
			dumplevel);
    break;
  case EAP_TYPE_AKA:
    dump_config_eap_aka((struct config_eap_aka *)(method)->method_data,
			dumplevel);
    break;
#endif

  case EAP_TYPE_TTLS:
    dump_config_eap_ttls((struct config_eap_ttls *)(method)->method_data);
    break; 
  case EAP_TYPE_LEAP:
    dump_config_eap_leap((struct config_eap_leap *)(method)->method_data);
  case EAP_TYPE_MSCHAPV2:
    dump_config_eap_mschapv2((struct config_eap_mschapv2 *)(method)->method_data,
			     dumplevel);
    break;
    
  default:
    debug_printf(DEBUG_NORMAL, "AAAH! Trying to dump an undefined config"
		 " type\n.Notify developers. Type: 0x%x\n", 
		 (method)->method_num);
  }

  dump_config_eap_method(method->next, dumplevel);
}

int config_eap_method_contains_method(struct config_eap_method *method,
				      int new_num)
{
  struct config_eap_method *tmp;

  if (!method)
    return 0;
  
  tmp = method;
  while (tmp) {
    if (tmp->method_num == new_num)
      return 1;
    tmp = tmp->next;
  }

  return 0;
}

  /*******************/
 /* CONFIG_NETWORK  */
/*******************/
void delete_config_network(struct config_network **tmp_network)
{
  if (*tmp_network == NULL)
    return;

  FREE_STRING((*tmp_network)->name);
  FREE_STRING((*tmp_network)->ssid);
  FREE_STRING((*tmp_network)->identity);

  if ((*tmp_network)->methods)
    delete_config_eap_method(&(*tmp_network)->methods);

  if ((*tmp_network)->next)
    delete_config_network(&(*tmp_network)->next);
      
  free (*tmp_network);
  *tmp_network = NULL;
}

void initialize_config_network(struct config_network **tmp_network)
{
  if (*tmp_network != NULL) {
    delete_config_network(tmp_network);
  }
  *tmp_network = 
    (struct config_network *)malloc(sizeof(struct config_network));  
  if (*tmp_network)
    memset(*tmp_network, 0, sizeof(struct config_network));
}

int config_network_contains_net(struct config_network *net, char *netname)
{
  while (net != NULL) {
    if (strcmp(net->name, netname) == 0)
      return TRUE;
    net = net->next;
  }
  return FALSE;
}

void config_network_add_net(struct config_network **list, 
			    struct config_network *toadd)
{
  struct config_network **current = list;
  while (*current != NULL) {
    if (strcmp((*current)->name, toadd->name) == 0) {
      return;
    }
    current = &(*current)->next;
  }
  (*current) = toadd;
}

void dump_config_network_wpa(u_char crypt)
{
  switch (crypt)
    {
    case CRYPT_WEP40:
      debug_printf(DEBUG_NORMAL, "WEP40\n");
      return;

    case CRYPT_TKIP:
      debug_printf(DEBUG_NORMAL, "TKIP\n");
      return;

    case CRYPT_WRAP:
      debug_printf(DEBUG_NORMAL, "WRAP\n");
      return;

    case CRYPT_CCMP:
      debug_printf(DEBUG_NORMAL, "CCMP\n");
      return;

    case CRYPT_WEP104:
      debug_printf(DEBUG_NORMAL, "WEP104\n");
      return;
    
    default:
      debug_printf(DEBUG_NORMAL, "NONE\n");
      return;
    }
}

void dump_config_network(struct config_network *net)
{
  if (!net)
    return;
  debug_printf(DEBUG_NORMAL, "+-+-+-+-+  Network Name: \"%s\" +-+-+-+-+\n",
	       net->name);
  if (net->type == UNSET)
    debug_printf(DEBUG_NORMAL, "  Type: UNSET\n");
  else if (net->type == WIRED)
    debug_printf(DEBUG_NORMAL, "  Type: WIRED\n");
  else
    debug_printf(DEBUG_NORMAL, "  Type: WIRELESS\n");

  switch (net->force_eapol_ver)
    {
    case 1:
      debug_printf(DEBUG_NORMAL, "  Forced EAPOL Version : 1\n");
      break;
      
    case 2:
      debug_printf(DEBUG_NORMAL, "  Forced EAPOL Version : 2\n");
      break;

    default:
      debug_printf(DEBUG_NORMAL, "  Forced EAPOL Version : Default\n");
      break;
    }

  if (net->wireless_ctrl == CTL_UNSET)
    debug_printf(DEBUG_NORMAL, "  Wireless Control: UNSET\n");
  else if (net->wireless_ctrl == CTL_YES)
    debug_printf(DEBUG_NORMAL, "  Wireless Control: YES\n");
  else
    debug_printf(DEBUG_NORMAL, "  Wireless Control: NO\n");

  debug_printf(DEBUG_NORMAL, "  WPA Group Crypt : ");
  dump_config_network_wpa(net->wpa_group_crypt);
  
  debug_printf(DEBUG_NORMAL, "  WPA Pairwise Crypt : ");
  dump_config_network_wpa(net->wpa_pairwise_crypt);

  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_WPA_PSK))
    debug_printf(DEBUG_NORMAL, "  Allow Type: WPA-PSK\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_TLS))
    debug_printf(DEBUG_NORMAL, "  Allow Type: TLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_MD5))
    debug_printf(DEBUG_NORMAL, "  Allow Type: MD5\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_TTLS))
    debug_printf(DEBUG_NORMAL, "  Allow Type: TTLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_LEAP))
    debug_printf(DEBUG_NORMAL, "  Allow Type: LEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_MSCV2))
    debug_printf(DEBUG_NORMAL, "  Allow Type: MSCHAPv2\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_PEAP))
    debug_printf(DEBUG_NORMAL, "  Allow Type: PEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_SIM))
    debug_printf(DEBUG_NORMAL, "  Allow Type: SIM\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_AKA))
    debug_printf(DEBUG_NORMAL, "  Allow Type: AKA\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_GTC))
    debug_printf(DEBUG_NORMAL, "  Allow Type: GTC\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_ALLOW_OTP))
    debug_printf(DEBUG_NORMAL, "  Allow Type: OTP\n");

  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_WPA_PSK))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: WPA-PSK\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_TLS))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: TLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_MD5))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: MD5\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_TTLS))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: TTLS\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_LEAP))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: LEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_MSCV2))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: MSCHAPv2\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_PEAP))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: PEAP\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_SIM))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: SIM\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_AKA))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: AKA\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_GTC))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: GTC\n");
  if (TEST_FLAG(net->flags, CONFIG_NET_PREFER_OTP))
    debug_printf(DEBUG_NORMAL, "  Prefer Type: OTP\n");

  debug_printf(DEBUG_NORMAL, "  SSID: \"%s\"\n", net->ssid);
  debug_printf(DEBUG_NORMAL, "  Identity: \"%s\"\n", net->identity);

  if (TEST_FLAG(net->flags, CONFIG_NET_DEST_MAC))
    debug_printf(DEBUG_NORMAL, "  DEST MAC: %.2x:%.2x:%.2x:%.2x:%.2x:%.2x\n",
		 net->dest_mac[0], net->dest_mac[1], net->dest_mac[2],
		 net->dest_mac[3], net->dest_mac[4], net->dest_mac[5]);

  dump_config_eap_method(net->methods, 0);

  if (net->next)
    dump_config_network(net->next);
      
}


  /*****************************/
 /* CONFIG_STRING_LIST       */
/*****************************/
void delete_config_string_list(struct config_string_list **tmp_string_list)
{
  if (*tmp_string_list == NULL)
    return;

  free (*tmp_string_list);
  *tmp_string_list = NULL;
}

void initialize_config_string_list(struct config_string_list **tmp_string_list)
{
  if (*tmp_string_list != NULL) {
    delete_config_string_list(tmp_string_list);
  }
  *tmp_string_list = 
    (struct config_string_list *)malloc(sizeof(struct config_string_list));  
  if (*tmp_string_list)
    memset(*tmp_string_list, 0, sizeof(struct config_string_list));
}

int config_string_list_contains_string(struct config_string_list *net_list,
				     char *netname)
{
  // if there is a list, we need to search for the net
  while (net_list != NULL) {
    if (strcmp(net_list->name, netname) == 0)
      return TRUE;
    net_list = net_list->next;
  }
  return FALSE;
}

void config_string_list_add_string(struct config_string_list **net_list,
				 char *netname)
{
  struct config_string_list **current = net_list;

  while (*current != NULL) {
    if (strcmp((*current)->name, netname) == 0)
      return;
    current = &(*current)->next;
  }
  initialize_config_string_list(current);
  (*current)->name = netname;
}

void dump_config_string_list(struct config_string_list *stringlist, 
			     char *title)
{
  if (!stringlist) {
    return;
  }

  debug_printf(DEBUG_NORMAL, "%s: \"%s\"\n", title,  stringlist->name);
  if (stringlist->next)
    dump_config_string_list(stringlist->next, title);

}

  /*******************/
 /* CONFIG_GLOBALS  */
/*******************/
void delete_config_globals(struct config_globals **tmp_globals)
{
  if (*tmp_globals == NULL)
    return;

  if ((*tmp_globals)->default_net)
    free((*tmp_globals)->default_net);

  if ((*tmp_globals)->allowed_nets)
    delete_config_string_list(&(*tmp_globals)->allowed_nets);

  FREE_STRING((*tmp_globals)->logfile);
  FREE_STRING((*tmp_globals)->ipc_group_name);
  FREE_STRING((*tmp_globals)->log_facility);
  FREE_STRING((*tmp_globals)->default_int);
  
  free (*tmp_globals);
  *tmp_globals = NULL;
}

void initialize_config_globals(struct config_globals **tmp_globals)
{
  if (*tmp_globals != NULL) {
    delete_config_globals(tmp_globals);
  }
  *tmp_globals = 
    (struct config_globals *)malloc(sizeof(struct config_globals));  
  if (*tmp_globals)
    memset(*tmp_globals, 0, sizeof(struct config_globals));

  // We want to set allmulti to have a default of 1.
  SET_FLAG((*tmp_globals)->flags, CONFIG_GLOBALS_ALLMULTI);

  (*tmp_globals)->stale_key_timeout = STALE_KEY_WARN_TIMEOUT;
}

void dump_config_globals(struct config_globals *globals)
{
  if (!globals) {
    debug_printf(DEBUG_NORMAL, "No Globals\n");
    return;
  }
  debug_printf(DEBUG_NORMAL, "Default Net: \"%s\"\n", globals->default_net);
  if (globals->allowed_nets)
    dump_config_string_list(globals->allowed_nets, "Network Allowed");
  else 
    debug_printf(DEBUG_NORMAL, "Allowed Nets: ALL\n");

  if (globals->logfile)
    debug_printf(DEBUG_NORMAL,
		 "Logfile: '%s'\n", globals->logfile);

  if (globals->log_facility)
    debug_printf(DEBUG_NORMAL,
		 "Log Facility: '%s'\n", globals->log_facility);

  if (globals->ipc_group_name)
    debug_printf(DEBUG_NORMAL, "IPC Group Name : '%s' \n",
		 globals->ipc_group_name);

  if (globals->default_int)
    debug_printf(DEBUG_NORMAL, "Default Interface : '%s' \n",
		 globals->default_int);

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_AUTH_PER))
    debug_printf(DEBUG_NORMAL, "Auth Period: %d\n", globals->auth_period);
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_HELD_PER))
    debug_printf(DEBUG_NORMAL, "Held Period: %d\n", globals->held_period);
  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_MAX_STARTS))
    debug_printf(DEBUG_NORMAL,"Max Starts: %d\n", globals->max_starts);

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_NO_FRIENDLY_WARNINGS))
    {
      debug_printf(DEBUG_NORMAL, "Friendly Warnings : Yes\n");
    } else {
      debug_printf(DEBUG_NORMAL, "Friendly Warnings : No\n");
    }

  if (TEST_FLAG(globals->flags, CONFIG_GLOBALS_ALLMULTI))
    debug_printf(DEBUG_NORMAL, "ALLMULTI is enabled!\n");
  else
    debug_printf(DEBUG_NORMAL, "ALLMULTI is disabled!\n");

  debug_printf(DEBUG_NORMAL, "Destination: ");
  switch (globals->destination)
    {
    case DEST_AUTO:
      debug_printf(DEBUG_NORMAL, "Auto\n");
      break;

    case DEST_BSSID:
      debug_printf(DEBUG_NORMAL, "BSSID\n");
      break;

    case DEST_MULTICAST:
      debug_printf(DEBUG_NORMAL, "Multicast\n");
      break;

    case DEST_SOURCE:
      debug_printf(DEBUG_NORMAL, "Source\n");
      break;
    }
}

  /*******************/
 /* CONFIG_DATA     */
/*******************/
void delete_config_data(struct config_data **tmp_data)
{
  if (*tmp_data == NULL)
    return;

  if ((*tmp_data)->config_fname)
    free((*tmp_data)->config_fname);
  if ((*tmp_data)->globals)
    delete_config_globals(&(*tmp_data)->globals);
  if ((*tmp_data)->networks)
    delete_config_network(&(*tmp_data)->networks);
  
  free (*tmp_data);
  *tmp_data = NULL;
}

void initialize_config_data(struct config_data **tmp_data)
{
  if (*tmp_data != NULL) {
    delete_config_data(tmp_data);
  }
  *tmp_data = 
    (struct config_data *)malloc(sizeof(struct config_data));  
  if (*tmp_data)
    {
      memset(*tmp_data, 0, sizeof(struct config_data));
    } else {
      debug_printf(DEBUG_NORMAL, "Couldn't allocate memory for configuration"
		   " data!\n");
    }
}

void dump_config_data(struct config_data *data)
{
  if (!data)
    return;
  debug_printf(DEBUG_NORMAL, "=-=-=-=-=-=-=-=-=-=-=-=-=\n");
  debug_printf(DEBUG_NORMAL, "Configuration File: %s\n", data->config_fname);
  dump_config_globals(data->globals);
  dump_config_network(data->networks);
  debug_printf(DEBUG_NORMAL, "=-=-=-=-=-=-=-=-=-=-=-=-=\n");
}
