/* 
 * xwave - an interactive audio player, recorder, editor 
 * for the XWindow System
 * 
 * Copyright (C) 1996 Kai Kollmorgen
 * (kkollmor@informatik.uni-rostock.de)
 *
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 * 
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <dirent.h>
#include <sys/types.h>

#ifdef linux
#include <endian.h>
#elif defined (FreeBSD)
#include <machine/endian.h>
#elif defined (sgi)
#include <sys/endian.h>
#elif defined (sun)
#include <sys/byteorder.h>
#endif

#include "types.h"
#include "audio_file.h"
#include "ieee/ieee.h"
#include "au.h"
#include "ccitt/g711.h"
#include "ccitt/g72x.h"
#include "endian.h"

static int au_new(Audio_File *af);

bool is_au(int fd)
{
   int magic;
   
   if ((read(fd,&magic,sizeof(magic)))==-1) return(False);
   if ((lseek(fd,0,SEEK_SET))==-1) return(False);
   
#ifdef little_endian
   M_32_SWAP(magic);
#endif
   if (magic!=SND_MAGIC) return(False);
   return(True);
}


int au_open(Audio_File *af,int mode)
{
   SNDSoundStruct snd;
   
   if (mode==AF_NEW) return(au_new(af));
 
   if (read(af->fd,&snd,sizeof(snd))!=sizeof(snd)) return(AF_ERROR);

#ifdef little_endian
   M_32_SWAP(snd.magic);
   M_32_SWAP(snd.dataLocation);
   M_32_SWAP(snd.dataSize);
   M_32_SWAP(snd.dataFormat);
   M_32_SWAP(snd.samplingRate);
   M_32_SWAP(snd.channelCount);
#endif
   
   if (snd.magic!=SND_MAGIC) return(AF_ERROR);
   
   af->freq=snd.samplingRate;
   af->channels=snd.channelCount;
   af->type=AF_AU;
   af->length=snd.dataSize;
   af->headoffs=snd.dataLocation;

   switch (snd.dataFormat) {
    case SND_FORMAT_MULAW_8:
      af->bps=16;
      af->length*=2;
      af->comp=AF_MULAW;
      break;
    case SND_FORMAT_LINEAR_8:
      af->bps=8;
      af->comp=AF_PCM;
      break;
    case SND_FORMAT_LINEAR_16:
      af->bps=16;
      af->comp=AF_PCM;
      break;
    default:
      return(AF_NOTSUPPORTED);
   }
   return(AF_AU);
}

int au_new(Audio_File *af)
{
   SNDSoundStruct snd;
   char info[]="Creator: XWave ";
   int ilength=strlen(info);
   
   snd.magic=SND_MAGIC;
   snd.samplingRate=af->freq;
   snd.channelCount=af->channels;
   snd.dataSize=af->length;
   snd.dataLocation=sizeof(snd)+ilength;

   switch (af->bps) {
    case 8:
      switch (af->comp) {
       case AF_PCM:
	 snd.dataFormat=SND_FORMAT_LINEAR_8;
	 break;
       default:
	 return(AF_NOTSUPPORTED);
      }
      break;
    case 16:
      switch (af->comp) {
       case AF_PCM:
	 snd.dataFormat=SND_FORMAT_LINEAR_16;
	 break;
       case AF_MULAW: 
	 snd.dataFormat=SND_FORMAT_MULAW_8;
	 break;
       default:
	 return(AF_NOTSUPPORTED);
      }
      break;
    default:
      return(AF_NOTSUPPORTED);
   }

#ifdef little_endian
   M_32_SWAP(snd.magic);
   M_32_SWAP(snd.dataLocation);
   M_32_SWAP(snd.dataSize);
   M_32_SWAP(snd.dataFormat);
   M_32_SWAP(snd.samplingRate);
   M_32_SWAP(snd.channelCount);
#endif
   
   if (write(af->fd,&snd,sizeof(snd))!=sizeof(snd)) return(AF_ERROR);
   info[ilength-1]=0;
   if (write(af->fd,&info,ilength)!=ilength) return(AF_ERROR);
   
   return(AF_AU);
}

int au_read(Audio_File af,char *buffer,int size)
{
   switch (af.comp) {
    case AF_PCM:
      return(read(af.fd,buffer,size));
    case AF_MULAW: {
       short *linear=(short *) buffer;
       byte *law;
       register int i;
       if ((law = (byte *) malloc(size/2)) == NULL) return(AF_ERROR);
       if ((size=read(af.fd,law,size/2))==-1) {
          free(law);
          return(AF_ERROR);
       }
       for(i=0;i<size;i++) linear[i]=ulaw2linear(law[i]);
       free(law);
       return(size*2);
    }
   }
   return(AF_ERROR);
}

int au_write(Audio_File af,char *buffer,int size)
{
   switch (af.comp) {
    case AF_PCM:
      return(write(af.fd,buffer,size));
    case AF_MULAW: {
       short *linear=(short *) buffer;
       byte *law;
       register int i;
       if ((law = (byte *) malloc(size/2)) == NULL) return(AF_ERROR);
       for(i=0;i<size/2;i++) law[i]=linear2ulaw(linear[i]);
       if ((size=write(af.fd,law,size/2))==-1) {
          free(law);
          return(AF_ERROR);
       }
       free(law);
       return(size);
    }
   }
   return(AF_ERROR);
}

int au_seek(Audio_File af,int pos,int mode)
{
   switch (af.comp) {
    case AF_PCM:
      return(lseek(af.fd,pos,mode));
    case AF_MULAW:
      return(lseek(af.fd,pos/2,mode));
   }
   return(AF_ERROR);
}

int au_close(Audio_File af)
{
   if (au_seek(af,0,SEEK_SET)==AF_ERROR) return(AF_ERROR);
   if (au_new (&af)==AF_ERROR) return(AF_ERROR);
   return(close(af.fd));
}

