/*------------------------------------------------------------------------------
*
* Copyright (c) 2011-2019, EURid vzw. All rights reserved.
* The YADIFA TM software product is provided under the BSD 3-clause license:
* 
* Redistribution and use in source and binary forms, with or without 
* modification, are permitted provided that the following conditions
* are met:
*
*        * Redistributions of source code must retain the above copyright 
*          notice, this list of conditions and the following disclaimer.
*        * Redistributions in binary form must reproduce the above copyright 
*          notice, this list of conditions and the following disclaimer in the 
*          documentation and/or other materials provided with the distribution.
*        * Neither the name of EURid nor the names of its contributors may be 
*          used to endorse or promote products derived from this software 
*          without specific prior written permission.
*
* THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
* AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE 
* IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE 
* ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR CONTRIBUTORS BE
* LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
* CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF 
* SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
* INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN 
* CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) 
* ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
* POSSIBILITY OF SUCH DAMAGE.
*
*------------------------------------------------------------------------------
*
*/
/** @defgroup dnsdbcollection Collections used by the database
 *  @ingroup dnsdb
 *  @brief Hash table used to hash the dns names.
 *
 *  Hash table used to hash the dns names.
 *
 * @{
 */

#include "dnsdb/dnsdb-config.h"
#include <dnscore/sys_types.h>

/**
 * @note I've tried the transposed hash table to see what would be faster:
 *
 * [0](0) : 33572985/s
 * [0](T) : 28201308/s
 * [1](0) : 17625817/s
 * [1](T) : 16023470/s
 *
 * [0](ORIGINAL) ayiuoez9 : 89285714/s
 * [0](TRANSPOS) ayiuoez9 : 76923076/s
 * [1](ORIGINAL) aeiouyz0 : 88495575/s
 * [1](TRANSPOS) aeiouyz0 : 78740157/s
 * [2](ORIGINAL) aeio : 178571428/s
 * [2](TRANSPOS) aeio : 126582278/s
 * [3](ORIGINAL) m : 434782608/s
 * [3](TRANSPOS) m : 357142857/s
 * [4](ORIGINAL)  : 714285714/s
 * [4](TRANSPOS)  : 666666666/s
 *
 * The original one is the fastest with both the old and the new algorithm.
 *
 */

/* Hash Values */

const u32 ZDB_HASH_TABLE[256][37] ={
    {0xD885253C, 0x099128C1, 0xED616222, 0xFBBDF883, 0x068A4C7C, 0xF67E891C, 0xC677F8A4, 0x66CA897F, 0x12E738AB, 0x72A0BE2B, 0x11993499, 0x721556B1, 0x705D9504, 0x1871489D, 0x368E67DF, 0x41529873, 0x3C543C6F, 0xF309AAF5, 0xCAB1BEA5, 0x68CE1DBF, 0x4AA2A113, 0xDC25905E, 0x62414D91, 0xC2F014D1, 0xE7626577, 0x7707D6B7, 0xFA082FCF, 0xD62E4D5D, 0xFA106A65, 0x7910F6E8, 0x852D4222, 0xD0BF0B0B, 0x9D9DED71, 0x81111D09, 0x8D789900, 0xC54486B5, 0x10D50D72},
    {0xA2491394, 0xCC2B1C8D, 0x150567B0, 0xC492F934, 0xD6F1BF73, 0x2D1E8548, 0x58422644, 0x716148CE, 0x78E95C1B, 0x173F5E66, 0xD3206569, 0xB62603D0, 0xF0EC1CFF, 0x0AA221A6, 0xF1664A6C, 0x16318678, 0x03E6D82A, 0x219FCA53, 0xC230E1AC, 0xC8EDB62C, 0x15E1898C, 0xB47F0AAE, 0x57049F3D, 0x6696233C, 0x0D5BA666, 0xCCFFA387, 0x00ACEC12, 0xF9834B21, 0x9E146B14, 0x1B5B813F, 0x979F0C04, 0x1E31AE5D, 0xFA9A6309, 0xDE2848E3, 0xF8E07777, 0xF183C9AA, 0xF946ACDF},
    {0x3D4AEC7E, 0x99A882B2, 0x48555C8B, 0x93F5A31F, 0x2E180A28, 0x16E840B8, 0x2482CEBC, 0x31A6ED1B, 0x230F22B0, 0x8DADB3C4, 0xA9C2AFB0, 0x077B0300, 0x088FF2F4, 0xB9947644, 0xB775505C, 0x2769A2DC, 0xD48966B4, 0x7D488DAA, 0x1C7A01EE, 0x785D9285, 0x2423CE62, 0x8443A91E, 0xE60C3BBA, 0xB426DCFB, 0xD9FE0411, 0x1D1D7569, 0x18A0E08E, 0xC367805F, 0xD9603669, 0x9608F952, 0xC9DB7D94, 0x335B6305, 0x21568518, 0xE4A3A305, 0xDE663512, 0xA373935E, 0x4C2BA61C},
    {0x5320BF91, 0x46B97147, 0xA661442E, 0x7B99D786, 0xA4992356, 0x63DFD36B, 0x8BDED2EA, 0xF4D581ED, 0x8EB9C4E1, 0xB25C94DD, 0xB9B6FD62, 0xCA117B01, 0x8ACF553E, 0x28E5C3A6, 0x69D8E108, 0xD6553C51, 0x89CC0635, 0x8E4D3109, 0xD0F2A698, 0x20C3B5CA, 0x6B185697, 0xF15EC427, 0x5143E302, 0xDC208CBF, 0x4CE40E31, 0x640D6365, 0x1DD81DA6, 0x3C19F9A7, 0xCBDF28EB, 0xDE235A7F, 0x1BD4F443, 0xC56317FA, 0x587EA1D3, 0x4744603D, 0xAF8C3D83, 0xD0FD7229, 0xDDE6EBCC},
    {0x4AEF7B3E, 0x67475872, 0xB9174BDE, 0xABDECF1A, 0xE486D79F, 0x0A8CF93E, 0xEE4E8E99, 0x5FEC125B, 0x9502FA02, 0xECD50BCA, 0x4DC485F2, 0x3E12FF9E, 0xFF74EC95, 0xFEEAE032, 0x3EA99AFD, 0x90A4DE00, 0xB0BFC059, 0x1609AA78, 0xF4BC7936, 0x4F202FE9, 0xA22992E7, 0x34A1267C, 0x406AB625, 0x009F8F47, 0x6C316B8D, 0xE40E8218, 0x407255B0, 0x12017B30, 0xB1BE5052, 0x6777EE94, 0x0C60B1B7, 0x313229FC, 0x8341F8FD, 0x57F8854E, 0xBEC773E8, 0x19662938, 0xB76E142B},
    {0x5CF777FB, 0x4E962996, 0x7ED09551, 0xE0AA3C02, 0x252BE4EB, 0x974EF663, 0x54DF80A7, 0x16115A3F, 0x2721838C, 0xCD813B11, 0xB1A52503, 0xF4F645FD, 0xBBE5DFAE, 0x2724C441, 0xEBC55FF6, 0x2984B0C0, 0x16BCD529, 0x2EB8DC8F, 0x02E5D931, 0x501F8047, 0xD0E6EA24, 0x0C6A014F, 0xDEA06675, 0xFAC56253, 0x67DD573A, 0xF6462E03, 0x1E28241A, 0xEBFBD5E1, 0xD825130B, 0xE339D315, 0x0DA46324, 0x5CE9B3D6, 0x81E0A9EF, 0xB1A9C3DB, 0xBF7AD38A, 0x039A8FB5, 0x4C9A77A4},
    {0x668C91BD, 0xDE6B5641, 0xBAA14EA8, 0x047FED27, 0xEB96B51D, 0x0BAB2C44, 0xF022C7DF, 0xAC9026C4, 0x67FE3EAA, 0xF2E994E9, 0x4825DBCA, 0xB037E49B, 0x2005DA9E, 0xE4873DA2, 0xE0F08ADE, 0x84C3153C, 0x58C3B7DA, 0xB1E59747, 0xD3694882, 0xDE570D81, 0x8E2B9FAE, 0xADEEA492, 0x54292E36, 0x580BD4C9, 0xDCBA370C, 0xE04A6FCB, 0x557C21C7, 0xF30371C0, 0x1043CD6B, 0xC5F01E33, 0x52A3AC37, 0xC7EC9ECA, 0x6D432432, 0x06CB13BB, 0x6C564C7A, 0x4A12148E, 0x4B7F404B},
    {0x9BDB1482, 0x048EC261, 0x3D1B8FDB, 0x78F05475, 0x1B80E5BA, 0xA890593B, 0xDF690678, 0x9F5EF12D, 0x28BFE37E, 0xFA491DAF, 0xAA1546DB, 0x069FA750, 0x3967BC58, 0x938DC293, 0x01E549D7, 0x34EE7EFB, 0xBF480AC3, 0x9FECFBD4, 0xECEFDB6F, 0x25E9AD0E, 0x5493A192, 0xC0AD627A, 0x62046635, 0xC1612124, 0x71AC3E8D, 0x805D5B6F, 0xA3E44C56, 0xAC192888, 0x7072B105, 0x58C504E0, 0x7B44F21C, 0xBBFDE1C2, 0x5602A037, 0x3D0341F6, 0xE2DD035F, 0x374021E7, 0x19105191},
    {0xD2134A33, 0xDE18E8A0, 0xB414D965, 0x229D9CEF, 0xAE925137, 0x31D1641A, 0x0DBB8E44, 0xADF0AE51, 0xA7D920A2, 0x7DB74E6A, 0xD4512728, 0x8B90EB06, 0xEE56F37B, 0x75BEC3F3, 0x96E95EBD, 0x6CED8699, 0x6CDF6345, 0x378E6405, 0xD33DDCFC, 0x9B5C98A0, 0x10358257, 0x2EDA506C, 0x3E7D70AE, 0x99C21853, 0x545A2DA8, 0xB446D08C, 0x93AFE46D, 0x72F8A9D5, 0xD29B25C5, 0x051E29B5, 0x309424B4, 0x0B6A6BB4, 0x5F861964, 0x770E74C2, 0xB1BD9EC9, 0x95224FA1, 0x515C04A9},
    {0x9000698C, 0x82B8B42C, 0x1E23E486, 0x676813B4, 0x08F291DB, 0x377F32C9, 0xED9FBED8, 0x1A6AB46E, 0x8C735C50, 0xB2B31FD5, 0xFA404624, 0x1071139D, 0x37B633C9, 0x943FC3AF, 0x2CBFBCBB, 0xC3D0D3CA, 0x9E625661, 0x973176D2, 0x2E10CBB6, 0x8D49AFBC, 0x62A5519E, 0x75DD0F0F, 0x43B7328F, 0xA408E248, 0xE94F9350, 0xCE860573, 0x9423CB42, 0x853A50CA, 0xE46D0292, 0xD44A52DB, 0xDBFDDCC7, 0xA4407C03, 0x2B4C425F, 0x2D29A766, 0x45F1E52C, 0x3AEB6F11, 0x5330CCAC},
    {0x76BFBF07, 0xE684CFD2, 0x05431C0F, 0x7EFA79B6, 0xDF3B2DDB, 0x3C3B3ABC, 0xA400BF3D, 0xDD095A2B, 0x7A455760, 0xC54D3A68, 0x17EE161D, 0x3F42E066, 0x2713B8A3, 0x0553DE0F, 0xBC3F48DD, 0x8865D9A7, 0x4CA2E449, 0x51F88F57, 0x7069CB5B, 0x4B14366A, 0x4BFA4D55, 0x3FF0981D, 0x923EC9CD, 0xD94AFCD5, 0xC9C0C874, 0xA0A06A9A, 0xF744A314, 0xFE383B39, 0xCEEE5993, 0xA86EB6A6, 0x102088B4, 0xDC1D72E6, 0x71F97D53, 0xD6C42CFF, 0xAC9FE0E7, 0x352114D1, 0xA64E38C8},
    {0x2FA7725C, 0x5BE73F35, 0x6EFE4BAD, 0x7184BF60, 0x9C29C2DB, 0x8DE48EB8, 0xEED5F58E, 0x330DA68C, 0x256F7E5A, 0x7419E8E3, 0x0BBD42B1, 0xAB17E7D5, 0x9B2E5472, 0x2CB5EAA4, 0x40DBEEE7, 0x3335CC7E, 0x06FF0A37, 0x1047797D, 0x25FCD2E1, 0xEABBCCDF, 0x1EB309B2, 0xF79FBE57, 0xB7467E44, 0x51D6EF89, 0x218CA588, 0xC850142C, 0x7C1A80BA, 0x34D39476, 0xB48922F0, 0x155B53AD, 0x27B53A4D, 0x464E7667, 0x23982914, 0xA94D3128, 0xEB560A54, 0x468C007A, 0xBF3F34C8},
    {0x23A52E98, 0x4E9B4773, 0x3C9F662C, 0xB517FEEC, 0x270F4188, 0xA7504D8E, 0x45C33387, 0x18F1E537, 0xD7B186E8, 0x5772A5AA, 0x361747FC, 0x0CC8A293, 0xA2DD99E1, 0x5E2CCBD7, 0x3AEF54BB, 0x432BB4E2, 0x03F54A89, 0x82BA0987, 0xD215E742, 0xE76C76AB, 0xC928216E, 0x7DAAAB61, 0xBF7B15F4, 0xC124A10E, 0xCCCB931D, 0xC7A9986F, 0x49C69A27, 0x8B46A085, 0x94FE667F, 0xD23C9C37, 0x9D71C5F8, 0xCF1AC6AB, 0x9D05D458, 0x897645EB, 0x6BC45A56, 0x502E204E, 0x732C5136},
    {0xCBA11C51, 0x8836F44B, 0x91C3980D, 0x42A82946, 0xA6FA2BDE, 0x6085A76F, 0x74D7452A, 0xA1C2E971, 0x0508DE5D, 0x8A3DBA4D, 0x3F8EEEB8, 0xB13D58A7, 0x27D72E34, 0x1AC7755E, 0x855F3E47, 0x1DEF7AFE, 0xF278D75D, 0x1BF0B875, 0x71C7BAA4, 0x375C2BCC, 0xE20360BD, 0xEA964E79, 0x38EBD6A6, 0x612613E6, 0xA9970CCF, 0x03B1484C, 0xCFD01C3A, 0xA645F8CB, 0x82980428, 0x7B68B294, 0x05597AFE, 0x5B641235, 0xD6B3DF3E, 0x899B896E, 0x771F9286, 0xAA9AEAF7, 0x4679FB37},
    {0x9CB57F89, 0xE91DE720, 0x8CFD7D3A, 0xDB4005DC, 0x0C8198D2, 0xA388B79D, 0xDA0D3862, 0x0DFFC4EA, 0xAC3096EE, 0x3AACAD45, 0x0B452C37, 0x6F6114A5, 0xF7EE9566, 0x10329CA7, 0x4BA654F9, 0x35492AEC, 0x3AC3A53C, 0xC8D14C2E, 0xC17E387B, 0xCC57F0EB, 0x842DA76A, 0x1F0CD704, 0x1608EBD5, 0xD4EFB465, 0xF50F72AD, 0xF27270FA, 0x8C40F36F, 0x2D225C13, 0x7AAAA1E9, 0x98916391, 0x3A8B64A5, 0x826660A7, 0x3BA3F8E5, 0x378A9815, 0xB396302B, 0x3E30B3C2, 0x4EE1A91F},
    {0x2DF0ECD0, 0x12B333C4, 0x9C541A74, 0x19421C37, 0x9DC3BC82, 0x4BF79A88, 0xCCFDF726, 0xEFF014FD, 0x0A7B8F73, 0x55EFFCF5, 0xFFEB81B7, 0x10D1651A, 0x51BBFEDA, 0xAD019A49, 0x4609E0FF, 0x89D1C25A, 0xF33A6007, 0x53EAC4F4, 0xCE40BD0D, 0x5EBD4D2D, 0xFE5D883C, 0x843C4F0D, 0x6681F4DF, 0x6A32CCDE, 0x137CE3EE, 0xAFEBD12F, 0xDC556527, 0x2C76EE4E, 0x4C7A062E, 0xD108610D, 0x6CC87CC6, 0x8352559E, 0x63C7297A, 0xDE76E562, 0x668F4E87, 0x78208FF5, 0x61EC64CE},
    {0x79C953D7, 0x36B0A00C, 0x47DE99BE, 0x6AF532FE, 0xD593F790, 0x7CF2E4A4, 0xBF7BAFE5, 0x186FB65F, 0x201760B5, 0xD228CC03, 0x19E8EC8B, 0x514346CC, 0xE84BA45F, 0x77C93EB6, 0xB30F30E5, 0x46D334FC, 0x9EE5D985, 0xAF31CB1B, 0x88D3C5F7, 0x65BF04BB, 0x0C788FCC, 0x442A4D73, 0x8B0325AD, 0x3C792A03, 0x060F192D, 0xAD5C2463, 0x141B2BBE, 0x292D9C47, 0x21B6D27B, 0x2D7A264B, 0x98EF0DB4, 0xA6ECBF71, 0xE6F9565A, 0x24FE9F33, 0xDFB875D1, 0x70121197, 0x6E4F8E16},
    {0x03BF81D0, 0xCE741340, 0x0FAEC483, 0x618EEE98, 0xE55C6259, 0x9C7EAAA9, 0x6351E282, 0x88FB4CC2, 0x8C460F1E, 0xBF8A1BC6, 0xE819D8D3, 0xD3BDF593, 0x7F1763CE, 0x41146EA2, 0xB0056809, 0x6CB6F389, 0x0FD5FCEE, 0xCF1D75D6, 0x01BD935E, 0xD89DCAB5, 0x314AD29C, 0x7DCA5CDB, 0xA91559B8, 0x49F5E92D, 0x0043CE81, 0x58355888, 0x697C051A, 0x94D0AF60, 0x69DB44C8, 0xCE438107, 0x133530C1, 0xF691C557, 0xCDD18634, 0xCBF8FDC3, 0xBC669C1A, 0x8BDFA83A, 0x17AFE9C8},
    {0xC4F29786, 0x55BC48FE, 0xE2F486C0, 0xA43AEC74, 0x3D574BD5, 0x308747BD, 0xCC254739, 0x0BB39211, 0xF7D472DC, 0x9B96AE41, 0x5BE53336, 0x6947FB44, 0xB3EC1B1D, 0x6557EB4E, 0x4A904192, 0x6039E9E9, 0xE7DF3721, 0x2D3DC0EF, 0x3DF37128, 0x52ADD676, 0x3A891C83, 0x955BA477, 0x7608920C, 0x63F66B84, 0x89245BF4, 0x84C35891, 0xEBB17BA2, 0x2EE23E6F, 0xD7E2AE29, 0x6F517BDF, 0x9E758F74, 0x6C31CA6F, 0xAE6EE2F1, 0xE39492F9, 0xE459308D, 0xEC5401DA, 0xDE273F8A},
    {0x76972B4A, 0x834C6929, 0x7FC6B6D4, 0x6938B3AF, 0x8AC97047, 0xA795A7FC, 0x68FED4B7, 0xAF7D6323, 0xC14938A3, 0x10448F83, 0xC23A71D5, 0x7C129684, 0x61258A43, 0x58E2687D, 0x3849C7DA, 0x887A27B5, 0x08D7BC75, 0x1FBD688C, 0xB8F1B363, 0xBA530790, 0x634D32A5, 0x5921F197, 0x29F5319D, 0x89F72CE9, 0x7A7F98E7, 0x59D79A52, 0x38F801C1, 0x9189A4C7, 0xB28F48B9, 0xB09774C5, 0x2FD7C8FF, 0xF6073A90, 0x4BA1B4F7, 0x87F8BD46, 0x5D93668C, 0xDB25E918, 0x6071FB3E},
    {0xDC8D5C7F, 0xE9B946C7, 0xC33F961E, 0x76B40B05, 0x7229BEE5, 0xA578494B, 0xD193EED2, 0x634B3767, 0xA1A65BA4, 0xB2A18235, 0x774E87A1, 0x6C8ADE9F, 0x78B46ADF, 0x407207E6, 0xB32BE3AC, 0x3C2AB33A, 0xC859B31C, 0x5455D4E3, 0x6A75F655, 0x30FF62CD, 0xBA638A5B, 0x29A0B2E4, 0xB956976C, 0x7F48B1C1, 0x4FFF27C2, 0x0E964ED4, 0x2F520E8F, 0x0FA8F0AB, 0x6CD557C4, 0xBB84490C, 0x276708EA, 0xE01D97C3, 0x24FB5A86, 0xED21132C, 0xD21372B0, 0x81F19CDD, 0x5E5E8460},
    {0xFFCAE46D, 0xB98757A7, 0x51D63497, 0xE5DF95A3, 0xBDD6478B, 0x2EA1FFD4, 0xABCBA456, 0xDC6ECA92, 0xEA167D0C, 0x3715AF06, 0x2B86317D, 0x535C8C2D, 0x397763D6, 0xFF949AA1, 0xAEBE24BB, 0x059BB8CA, 0x2C78A456, 0xA040F26F, 0x3D30E1E0, 0x9D13C549, 0x0E0CEB7D, 0x45956EE4, 0xF987526A, 0xB89F88ED, 0x028AE881, 0x76D914A5, 0x44237D8E, 0x56263328, 0x120D7AAC, 0xA2D7FC89, 0x485A2163, 0x7AB65F15, 0x49EA029E, 0xCD019A18, 0x3C954E8D, 0x41138BCE, 0xF1636DC9},
    {0xA8DA5479, 0xF274D9D3, 0x58584D14, 0x9DD05A48, 0x588391AA, 0x2D9C5D08, 0xBB2A229A, 0xAEDDCE59, 0x947E8C0C, 0x921AA6B0, 0x859077E0, 0xD3E07D58, 0x5811D762, 0x23B0636B, 0xF7CC5429, 0xE64B42D2, 0xEE5F9EA7, 0x879F94D8, 0xF770D695, 0xDB67E31D, 0xF00E3CB2, 0xC6F70C2D, 0xB71E95C2, 0x1A3836BF, 0x91405CE5, 0xF6E5AB42, 0xEA25B8A9, 0xC7FDFE5E, 0x9C9EDB98, 0xFCBEE52E, 0xA097519D, 0x597C2EB0, 0x3529A37D, 0xD1004D1B, 0x6D172B46, 0xD7CCE7CC, 0xF3244EAA},
    {0x7B84EB7E, 0x13EE01B3, 0x1999758A, 0x82E51C1A, 0x1DBC0715, 0x9F4E6A49, 0xB216CC8D, 0xD7D82BED, 0x2FF36E0E, 0x77107DD9, 0x0FEBD72E, 0x1283A686, 0xBD5A4935, 0x623E9F54, 0xB087F32D, 0xCFD89761, 0xBFFE31BA, 0x277E82EE, 0xDE0788DD, 0x5844BE6D, 0x134DF11A, 0x34DD936F, 0x92192354, 0xBC1C835B, 0x076ACF0C, 0xB4F48855, 0xAD3F367F, 0xCE8E104D, 0x1E0BFE6A, 0xB98D386F, 0x2A004912, 0x8D55C0C2, 0x089A10ED, 0xFC7F202C, 0xC969F5C4, 0xF181680B, 0x6D727DF1},
    {0x357786BB, 0xEBFFA61E, 0x43582E82, 0x5B315B5E, 0x93993D96, 0x5C4DD85A, 0x03ECD45B, 0x07A82EC4, 0xF8389AC6, 0x315636C5, 0xF196985B, 0x9AB97A91, 0xA3BF113F, 0xA5028C88, 0x4B665550, 0x765309E3, 0x54C5E6D0, 0xC96F43C5, 0x71B8D322, 0x22631A18, 0xA16E6C83, 0x431C3476, 0x638BEBE3, 0x8C7B2D40, 0x45864744, 0xE1C7A748, 0x7010429F, 0xFCE8ACA0, 0x4529CF75, 0x4941483D, 0x0E585CBD, 0x17EFD999, 0x644BEC64, 0x565FCE57, 0xDA9A2FF0, 0x1E95C6C5, 0x5D82260F},
    {0x9740DBF4, 0x5CAAFAFA, 0xE0781578, 0x2577EB5E, 0xE2203046, 0xA94BF2E5, 0x79A612DB, 0x10F94313, 0xF6983D99, 0xE47F681A, 0xBD6641EA, 0xB3620DEB, 0x92C503DE, 0x873CBA35, 0x6028D617, 0xD0929BD7, 0x5CA57C25, 0x70B2872C, 0x325F24B7, 0x9FEE662F, 0x655FAF73, 0x205D1E1F, 0x979D5D76, 0xBB39D090, 0xC74451CC, 0xF249B44D, 0x59413684, 0x78E99106, 0x3025110C, 0xCC602C86, 0x93CFFAAA, 0x8B3A3723, 0x5C46CF21, 0x1AE33E88, 0xEB9FD840, 0x8D56725A, 0x7DB70AED},
    {0x9B105C95, 0x90607C64, 0xA4C03E0A, 0x4501B61E, 0xB5AEC5BF, 0xFB6D4B84, 0xD8513A75, 0x45C3A38E, 0x4B229625, 0x1248A03D, 0xCD4B95BD, 0x39FAEFD1, 0x9512C46A, 0xAC15BAFD, 0x9C6EBBF5, 0x1D5C5A3E, 0xBCF07AB8, 0x773FAC88, 0xBA2AA4F2, 0x796DC8C9, 0x1579913B, 0xD2C94654, 0x56937ACB, 0x87ED1382, 0xF96B6734, 0xE538E83F, 0x2FBF6651, 0x99F906FF, 0x56A0DE52, 0x51C87788, 0x0E2E73F7, 0x311863C1, 0x88627FA4, 0x75BDAC47, 0x4F91DC25, 0x65EAB074, 0xFB223A35},
    {0x100A7345, 0xFBE038D2, 0x8D1F78AF, 0x22CCD951, 0x97672AAD, 0xD9F8A87A, 0x7C088D40, 0x2C370160, 0x90E14768, 0x55EB59E5, 0xE5CED485, 0x49C1D53B, 0x946FF420, 0xF6D692C3, 0xE975887A, 0x3F904803, 0x6309D281, 0xE45EA2E4, 0x8BC54B46, 0x9F5ABE00, 0xE86FC80A, 0x02617BBC, 0x0C99335D, 0x69133A45, 0x272C8A86, 0xE958EC72, 0x205803F5, 0x39467437, 0x87145369, 0xEC018595, 0xE64F2DA3, 0x70194301, 0xCD59F6ED, 0x7E4217EE, 0x648D093C, 0x2C294305, 0x16F697B7},
    {0xB1B3EFF1, 0x606D21AA, 0x431CBF9D, 0x5AED97C1, 0x9353520F, 0xF006DEF3, 0x5FDC34C5, 0xEEDB20C2, 0x841F7843, 0xD185480C, 0xE2032F52, 0x9D565FAA, 0x68FD323F, 0x96A649DD, 0x0B77AD33, 0x63B6B315, 0x2FE9D0E1, 0x84A92C8C, 0x6D3E8EDC, 0xA6DCA4E3, 0xED0C6BA0, 0x67CE7956, 0x8F6E1455, 0xCF61B06D, 0x3F874B5C, 0x2A233586, 0x2011A762, 0x4AE0C327, 0xD7983D3F, 0x62ACD15D, 0x900E9EA6, 0xEB749CB8, 0xD15AF322, 0x210B89D1, 0x064D6934, 0x3B0C8F57, 0x3420F11C},
    {0x7D8A9FF7, 0x08F1B9B4, 0x24ACBC49, 0x8AA2831D, 0xB8A8AE49, 0xB283ED38, 0x022FCDBE, 0xC243A109, 0x253C7F85, 0xC27DEA5F, 0xF417CC89, 0x82608798, 0xD4A7BBCB, 0x33700BA3, 0x901C3641, 0x64D8F603, 0x7C515EB3, 0xD1C7CE0B, 0xE7F4A2CD, 0x823AC760, 0xA7B0F961, 0x37D8BBDC, 0x603CF9C9, 0xD9F15C8C, 0x97764CF1, 0x2674800D, 0xB466FE93, 0x6F277CD4, 0x54FF4CA8, 0x2C7FD159, 0xD3CA6E35, 0x0A89316E, 0xE7AF6B75, 0x064E5DA4, 0x9EC433F1, 0x1976FE95, 0xE77F45ED},
    {0x3A60FD52, 0x95F413F8, 0x387CFBD3, 0xA4BEA704, 0x8C252DA4, 0x4E947E3A, 0x81E8BF58, 0x0A297D88, 0xEFCD2549, 0xEA107720, 0x3A729CAB, 0x5B00ADFD, 0x59192E1E, 0xD6E32A28, 0xF2F4AC0F, 0xA439D94E, 0x52C0DD0A, 0x2C0DB3ED, 0x65976D81, 0x7B74AFC9, 0x888A5225, 0xF3E9DD6C, 0xE4D950BC, 0x5B08D44E, 0xB8B89D02, 0x4C0F5BCD, 0x23E7528A, 0x6A74631C, 0x32CEBF8C, 0xB6B3F91A, 0x24F5E4EA, 0x91EEE30F, 0x703B9D96, 0x37EE5806, 0x59539C69, 0x16871230, 0x19518348},
    {0x92AAACF7, 0x963F837C, 0x9FFBBBB8, 0x267DBAB0, 0xE0355406, 0xCEA99FC7, 0xC0944C8F, 0x99A3D9B7, 0xC9C5AF82, 0xB421EA6F, 0x037D1FE2, 0xB4C819A7, 0x61719630, 0xB0D1FD01, 0x194A1121, 0xDA5B198A, 0xE0C245C1, 0xFA277523, 0x4ABDF545, 0x6F84062F, 0x168533B7, 0x8933835E, 0xC51012F8, 0xA0CD469F, 0x54DF3902, 0x5A5A8C4A, 0x9B24E67A, 0x209E5DBD, 0x8CDB97E7, 0xC9787ABA, 0x0B7E6C91, 0xF543DFB9, 0x28C2C779, 0x37DE8895, 0x920B99FC, 0x785FF132, 0x083204B8},
    {0xDDA027A3, 0xC51AFB37, 0xCD9A3345, 0xA846243D, 0x3C2F56B6, 0x62A34DAA, 0x5B0DE142, 0x471B3B91, 0x6AF4BC34, 0x785A4F5F, 0x4ACF9132, 0xBDB392A0, 0x048AC4F5, 0x760E3CBF, 0x1BED9EE7, 0xC93F365B, 0x6FFBEC8E, 0xD4BBA12D, 0x590D33F8, 0xA81AD3CE, 0xB9E47136, 0x45CDA3F6, 0x99525C3E, 0x789604C6, 0xC1C124E1, 0xE7F5F0CF, 0xF2F5C175, 0xBB05A900, 0xE3D2853D, 0x180A0ACF, 0x163286AB, 0x58FE58BF, 0x953DE4A8, 0x09C6B8E4, 0x96DE0C2E, 0xC8ACB2E7, 0x37244EA5},
    {0x4D594763, 0x56FC0696, 0xAC29E39B, 0x538FFF12, 0x164F3A03, 0x29B2B9D3, 0xA2C3FB7D, 0xBB14B942, 0xF88748DD, 0x365DADA6, 0xDEB734DE, 0x6ED860D4, 0x5E759942, 0x93E6BFA2, 0xF4B32040, 0x3651C389, 0x7F9EA5C5, 0xD7A6E0BE, 0x94D0CD8E, 0xA28289C8, 0xAAF5B3D2, 0x2C9EB3A4, 0x99BDA39C, 0xD6F7EE01, 0x0FCFB369, 0x459E5B83, 0x6262B205, 0x39707C6D, 0xA39293CC, 0xD6E737FC, 0x2163BCD6, 0xECFC5278, 0x11452A33, 0x5E1416BD, 0x6060B11A, 0x3E5B7005, 0xC2AC8CC2},
    {0x9F2F5D81, 0xE89ED7FD, 0x1790939A, 0xE08549B0, 0x0FFDC897, 0xFB892726, 0xFF4AF729, 0x112D7EBE, 0xC103F753, 0xEB01EBFC, 0x9E0F0815, 0xEBDCDABE, 0xBFACA699, 0x0599EDF2, 0xF4910EF0, 0xBEB65D73, 0xDB7ED377, 0x33362688, 0xBBC3E750, 0x6B1EE572, 0xB27A6884, 0x1C69992B, 0x2B27220B, 0xFF939791, 0xB98A99A0, 0xF5A6BB01, 0xE0E301E4, 0x058EB273, 0x5EC6AEBA, 0xE31BF8A1, 0x16D62D0F, 0x391CF6D0, 0x7F48A3E6, 0x69F19A2A, 0xB0BD07E2, 0x549B6D95, 0x495E9C28},
    {0xCE8465DF, 0x3651A299, 0x7AB47212, 0xB3EF0514, 0xF6392B79, 0x5A7A9CD0, 0x8DEEA124, 0x60639CB7, 0xA4294BA0, 0xACA4B328, 0x54F083EA, 0x7BEDD4DA, 0xE0235AB5, 0xBA60A198, 0x130025FE, 0x08AC3E8A, 0x8A089264, 0x1E1BB1B4, 0xD5DC988A, 0x3952CC3B, 0xAC33390F, 0xB6685FD2, 0xD5AF46DE, 0xED02EF9B, 0x4E08432E, 0x933C787D, 0x94EA3008, 0x26212B28, 0xEBD6E1A4, 0xE68E6F68, 0xC58378FE, 0xF0CC193F, 0x8824584A, 0x92861B78, 0x08FDC657, 0x82BABD9B, 0x9CB18A73},
    {0x4E761F73, 0xE33AFB8E, 0x0994B970, 0x81D447AE, 0x2ADB5C7E, 0x3C1E5938, 0xE6351A60, 0xAC5CF756, 0xA492CA00, 0x1C6BB463, 0x0B0DED0D, 0x86C267A2, 0x444E414F, 0xC8CFD8A9, 0x1BF8BA97, 0x6511E0A5, 0x3BA69841, 0x44E534B0, 0x4A689E5B, 0xAEC18F0C, 0x898710CB, 0xED426FF7, 0xB23EB8EE, 0x8F4632B0, 0x6982EB46, 0x689D7087, 0xDC57CCA5, 0x8AF1D313, 0xAA6D7148, 0x6CE3C820, 0x98804417, 0x65666359, 0x7A5AF3DA, 0xE1B65AB1, 0x6DD9341F, 0x032CFC0D, 0x41920723},
    {0x72A8128D, 0xC92433A8, 0xEEEC873D, 0x6A28C77A, 0x38BB6CE6, 0x191C21AC, 0xD85179AC, 0xC3F100C9, 0x289DE6A7, 0x1877D9E7, 0x9E8E8821, 0x5CAAD0A9, 0xE46CF441, 0x54588372, 0x29D9F2E3, 0x0360CB8A, 0x9CD52BED, 0xAC84821A, 0x2B9E4BCE, 0xA52F4A81, 0x7FFF30C9, 0x0F7CB7AB, 0x5AE37CEF, 0x6E19C8F3, 0x06DF9580, 0x380D8A7D, 0xD467D3A3, 0x2B966652, 0x15376B9E, 0xA3583E24, 0xF0770119, 0x3A23CB2B, 0xE28FC2E2, 0x4F69E1FF, 0x877FBDFA, 0x780A7186, 0x1FB4E106},
    {0xB0D15B88, 0xC1AC90EC, 0x32DF98C5, 0x4A307A34, 0x0DA93FC7, 0x40CAE13D, 0xF1E5EA82, 0xCD59163F, 0xBEFF4371, 0xE00FC8D0, 0x35ED09C8, 0x3B5CC867, 0x28B23D0A, 0x312E740D, 0xE1AB35B6, 0x7FDC599A, 0x9B5D2C7A, 0xE43D149B, 0xBCD05047, 0x132926F5, 0xBD1F1511, 0x2FBDAE43, 0xD363596A, 0x935F43D0, 0x8F54EEAF, 0xCCAC9CC5, 0xD7528F30, 0x602B5B86, 0x3250DF81, 0x1C44B613, 0xF877AD17, 0xAE67E4C6, 0x47BF87A8, 0xAB3CE250, 0x4DB81AF2, 0x1E9C9D74, 0x480924D0},
    {0xEA7F9E16, 0xA7516208, 0xC24A20A1, 0x17034D23, 0x6833978F, 0x7D953357, 0x7456A139, 0xB3097F4D, 0xBF41283E, 0x8F1EFB54, 0xF888AD56, 0x303CE840, 0x4031B220, 0xA475BA4B, 0x5E018F7B, 0x81D4720B, 0xE6414F31, 0xEF1CD8FE, 0x0E95910C, 0x09E92A73, 0x7C74DF69, 0x159E7E35, 0x0BFF973C, 0x841EB0DA, 0xA7822C9D, 0xA10BC00B, 0xA8D80B9C, 0x03727BB6, 0xD38FF5E3, 0xFC5957FB, 0xD75E52C9, 0x38D1B837, 0xB25E16F0, 0xA05B823E, 0xB6BF1EB3, 0x6BA9565E, 0xC43F258A},
    {0xF0B81CEE, 0xEF7F5717, 0xB818C8E4, 0x5F5F6372, 0x9C0DD3E5, 0xC14DB857, 0x07628392, 0x3AD947CC, 0xE2F0B311, 0x847D589C, 0xD059DA66, 0x90731CA6, 0x52934F7E, 0x4C5F8457, 0x8F2CFDE0, 0x1A2147B4, 0x10B672A4, 0x67E4E7C8, 0x09F8DCDD, 0xD45E5113, 0x030B3F9B, 0x7BDDADE1, 0x64E97C43, 0xDE7DDB80, 0x2B698569, 0x72E32384, 0x11996106, 0xE2F89055, 0xBB8A5041, 0x1A531F5B, 0x27C9B8EB, 0x432579D5, 0xE88D2669, 0x9DBB1754, 0xF48A1D20, 0x0FE376F9, 0x81E7CAE0},
    {0x313A0918, 0x06982C55, 0xBC8BFAFD, 0x093F5EC9, 0x3F4C421A, 0x015434C1, 0x5BE3B687, 0x8FA9609D, 0x1825F1B6, 0xB80F29A8, 0x5981631C, 0x8299145E, 0xED5B977B, 0xAF73D38E, 0xA523FACF, 0xACFA7987, 0xF0E2E124, 0x44EBF8D8, 0xEBFCEADE, 0x642E5399, 0xB9136F11, 0x3BE452F5, 0x8007A3F2, 0xD2B3CB5C, 0x6627F1BB, 0x663DB62A, 0x61A83DF8, 0x0EB2BC42, 0x3E2FBDFF, 0x529C0FA8, 0x0E37E309, 0xC7ADD149, 0xCA83D0D2, 0xD46478D1, 0x280E41F3, 0xC9515589, 0xB5E3637B},
    {0x58870877, 0xD3A6C2A4, 0x1ECA5CD3, 0x925CA0D5, 0xA7D0763C, 0x45D16D06, 0x924127D6, 0x7AFED91A, 0x1CFA06F7, 0xC59A9E11, 0x2DADA5EF, 0x50F4A3F1, 0x10256892, 0x126983E6, 0x7EA3A461, 0xBA733099, 0x31567D97, 0x7798DC85, 0x9138D0DE, 0x4C826E3C, 0x1B7AFD84, 0x39E989A0, 0x258BBAD2, 0x919B9453, 0x8B2D8A98, 0x8BA3068D, 0x53F544C0, 0x3631B4B0, 0x0ED8FBFF, 0x58CB4524, 0xA0E452C6, 0x16281413, 0x30E553C5, 0x32A11D45, 0x84DF1015, 0x6B800D2B, 0xF48087B4},
    {0x7A24E789, 0x65A88E86, 0x3A872AF5, 0xF8CE8F7C, 0xF9D044FD, 0xE8B474DA, 0x2495012B, 0x06D51F2F, 0x120D9F58, 0xDB2AC8A0, 0x1CE23804, 0xB3370DD7, 0x6BB4410C, 0xCC4199E6, 0x0C45B7A7, 0xD2A77EE1, 0x878C282C, 0x666E4D8C, 0x0A710531, 0x44B48A44, 0x3458945C, 0xD02CEE44, 0x46D0C196, 0xD7D5CF61, 0xB89B4080, 0xA370AF69, 0x65564C82, 0x131B197B, 0xDA58E4E1, 0xEDA59945, 0x4B43CB78, 0xB9084780, 0x806F00C9, 0xD9063E03, 0xA9A26CFC, 0x47982BA5, 0x47C46582},
    {0xF8B34C48, 0x29D6D682, 0xE8049DB1, 0xB596973E, 0x047357F1, 0x82856DC0, 0x97210791, 0x52800BE6, 0xDE870640, 0xCEA99DD5, 0xFB31F4DD, 0x549E94B0, 0xD7C6F8CD, 0xDDAF9165, 0xAD9623F2, 0x35397C6E, 0xE3FA49DF, 0x195197EC, 0x203D0ED9, 0x6A18E6C0, 0x31A436CC, 0x232B9408, 0x76C6B064, 0x2993D7AA, 0xE7618E50, 0x62BEA422, 0x7DBDD918, 0xAE0F7241, 0x5915EC0F, 0xB4CEADC5, 0x81C5E39D, 0x8C986044, 0xB965CAC3, 0xFBA4B214, 0x542961AF, 0x38D521C9, 0x5793CF2E},
    {0xCA5EF7A9, 0xADF4290D, 0xA9FF841F, 0x47662859, 0xFD5F54A5, 0xE159DA89, 0x233B9B3F, 0xDE7ACD16, 0xC58CAFB0, 0x71EDF824, 0x19EB8852, 0x682E64B4, 0xBAF48F30, 0xC429C9BD, 0x3D76A501, 0x5AF164C1, 0x12373AB6, 0xBBA5B4C8, 0x672B5905, 0x62BA39D5, 0xD849AF94, 0x8C13D1C5, 0x250D39F1, 0x79A0F29D, 0xB24988D6, 0x3CCB5825, 0x4E34F2BC, 0x3DDECA14, 0x56C1727A, 0xD661CA75, 0x65A92E05, 0xBAF6481F, 0x1689311B, 0xEB37D1C8, 0xB7F2992B, 0xF31E925A, 0x216D7238},
    {0x6313B4E3, 0x07C875B1, 0xBA76165D, 0xB70F5547, 0xF7C4C3A9, 0xB5253995, 0x7703C523, 0xEF436F91, 0xF9BC6DAB, 0xC0243580, 0xD77BF62B, 0xC3264635, 0x8DCDC1AF, 0xDE4F108B, 0x8CC2E295, 0xBDCEDC68, 0x52CA75D9, 0x975CE27D, 0x40EE466B, 0x7BDF6B56, 0x962AB313, 0x74409B19, 0x181774D4, 0x35F55052, 0x8B8F007B, 0xA25E9ABA, 0xF59A7118, 0xDA9E2C18, 0xDF41EE80, 0x48603A10, 0x62F629C5, 0x6A7513B7, 0x01151E73, 0x3CCCC716, 0x275BACF3, 0x1EEF767D, 0xD5FD94BC},
    {0x12C90D32, 0x144E86B7, 0x4CF4ACEF, 0x0804E0E3, 0x507DC67F, 0x462A2AE0, 0xDBC8C2EC, 0x0E7CB0B3, 0x7A794546, 0x724E01EA, 0x7ADAC0C9, 0xE14B57DD, 0x33F9317B, 0xBD780745, 0x618101AF, 0x7C849958, 0x44EAE313, 0x1F916CBD, 0x44B602E9, 0xCD419908, 0x92D62982, 0x076866AA, 0x9C0B81B2, 0xB2ED5462, 0x6785CBB3, 0xB6592909, 0xA284EA61, 0x88872834, 0x725BE6F7, 0x81A47789, 0x3F878EA6, 0x506E0603, 0x071034B3, 0x7A75A1CE, 0x9EA13F87, 0xA562C0AF, 0x2B65AB14},
    {0xC8B1DF78, 0x19EBB4DF, 0x655BEB14, 0x1CD47BFC, 0x320D3BD7, 0x7DDEC14A, 0x7CECD374, 0xBBF5FF62, 0x040FCF8D, 0x77F37889, 0xE1223C26, 0x43C9E773, 0x7B4691D2, 0xDA4B53A4, 0x2AC2AA66, 0x6479E22E, 0xDCA8FA0C, 0x4B3D627C, 0x1745AF1D, 0x035B9F22, 0x79C1C86C, 0x4300F4F8, 0x4476D08E, 0x34441892, 0xCCDB434C, 0x09AF5D82, 0xE0096AC7, 0xEA733FD1, 0x5BD50D70, 0x7A0BADE0, 0xE5E545F0, 0xDB1E177E, 0xA0E00F70, 0xA1D1B3D9, 0xE4907571, 0x14E26A89, 0x2718C5F2},
    {0xD3D440AE, 0xAA077F2D, 0xB6E49842, 0x99B0716A, 0x30820256, 0xB70BB020, 0x74FFD1AB, 0xA0661A1A, 0xBC59D72A, 0xD781BD62, 0x4C8BAC95, 0xAF44493E, 0x5E4D492D, 0x8E4D5B14, 0xD44742ED, 0xE6549603, 0x3EE732CE, 0xF6BD8C19, 0x0ED76D72, 0xB40B38F4, 0x2720A384, 0x8AC36F70, 0x80851F6B, 0xFA3F7875, 0xE31F7070, 0x6CEC7FBA, 0x0FE5C4A1, 0xBCBEFAE4, 0x53EDF6B3, 0x9AB2D6E2, 0x1413513E, 0x77D7270C, 0x12F56B41, 0xDF168243, 0xB2EC1700, 0x48202EE9, 0x8B538B0D},
    {0x7CF6BB0D, 0x576A54C2, 0x001CCACA, 0x49BD2167, 0xEB87E36F, 0xE3147E95, 0x34667F9F, 0x3DD2F9EA, 0x374E77C6, 0xB4FA792C, 0x88A0E977, 0x40253643, 0xE21D5035, 0x72E17F89, 0x041E51F2, 0x4AC1FF84, 0xB6E5DF23, 0x524F2E8C, 0xDD1B0551, 0x27F1C4EC, 0x4A1D995F, 0x223FD2D8, 0xCD2FC1A8, 0x0A0EDF55, 0x054736F8, 0xC2410B69, 0x36701F7B, 0x6FFF8087, 0x62E187FB, 0xCEFDF3DA, 0x01560CBA, 0x0E26F0D0, 0x386482D8, 0x5119162D, 0xB2976C94, 0x9E3A4F2D, 0xF3CAC2F0},
    {0x6CD946B5, 0x8C32236B, 0xB77DF4DD, 0xF1018A96, 0x678D46F5, 0xF29A818F, 0x9A32BF58, 0x08DC0188, 0xED8C8C77, 0xFF61817B, 0x23C8FFF3, 0x0676E69C, 0x79E41C5C, 0x3F1DB313, 0xDD15280C, 0x06188152, 0x9BD7F4AD, 0x4EA7B92E, 0x3BD6236D, 0xCA12A5AD, 0xDF9BF84C, 0x7C91445F, 0x0022A88A, 0x15ECBC73, 0x2A33876F, 0xD086B197, 0xF5093A03, 0xEF98B509, 0x09F30FCD, 0xEE776C5B, 0x25E6D73B, 0x093D3559, 0x969AAB95, 0xB26E7A2C, 0x00E84107, 0x82ECED3F, 0x542E9C40},
    {0x9AA143F0, 0xB76B2FF3, 0x14582467, 0x4B95D497, 0x5CBD2912, 0xFBCD9A21, 0x309951B0, 0x6E1574D2, 0xF901FAAC, 0x8BE17FC3, 0xFF929BBB, 0x450F0D77, 0x899E0C2E, 0x79AFBD00, 0x51907DB6, 0x0F4BD89D, 0x5D52FE3F, 0xA2134499, 0x79949B4C, 0x77CD1D94, 0x3B3D137C, 0xF194A31C, 0x8C9A330C, 0x0CE859C2, 0x13D9A3EF, 0x22E3C70A, 0xB037C39B, 0x79A822CF, 0x8DA360A0, 0x71ED2690, 0x1B18A8BD, 0xD2667330, 0x655C5133, 0x2D4B38F5, 0xA8212396, 0x930B2473, 0x46A7DF6B},
    {0xD3E13862, 0x96AD8A7B, 0xC93B026F, 0x2316A444, 0xE53DCF11, 0x141CFD83, 0x3CFD5FA6, 0xD46A8F76, 0x7BF42BCA, 0xF68AA65E, 0x9B24698E, 0xB2160E20, 0xD634EF29, 0x33726A9F, 0xD70459E9, 0x866E0C29, 0xE0B5B2A4, 0x777B0E34, 0x528C0E21, 0xEFEFE15C, 0xB64510FF, 0x0F2931E8, 0xDA8A8152, 0x0622C5B3, 0xB2466D7E, 0x1731C40C, 0x6066A9CB, 0x192D66FC, 0x3852FEF6, 0x55F7E1FF, 0xDAA995C4, 0xFFFDB3A2, 0xBBDBAC44, 0x1559F5DA, 0xA199FDE7, 0x65095CCE, 0x4CE9505B},
    {0xC28B7225, 0x14856E42, 0x4059EEF6, 0x0168BBDF, 0x72A25425, 0x10B7920B, 0xEC13B381, 0xEC285976, 0x56CB9345, 0xA7BED63A, 0x1D6E21B3, 0x44211D3D, 0xF8126934, 0x74EDE685, 0xE48EA508, 0x10D1E5CC, 0xA18BC7CE, 0xFCB04B31, 0x578D4509, 0x714DECD4, 0xD36A17DF, 0x2F911135, 0xF8D3557A, 0x8230671D, 0x08C0B856, 0xE0E699A2, 0xE0436F83, 0x1A4362AC, 0xBB272DA6, 0xE13D2CB5, 0x7A459563, 0x09779CF7, 0x3DF1EA05, 0x32486446, 0xD3BE168B, 0x7F3C83CE, 0x244C7AB6},
    {0x55BFC9D9, 0x48BC1867, 0xD55F6149, 0xAD38C7C5, 0xBCC48476, 0xAB8356C7, 0x59047F3D, 0x596983C5, 0xF87458A2, 0xBE0236D7, 0x5B20A311, 0x01E735AA, 0x89A881D5, 0xEA587335, 0xF30989CC, 0xBAAEB91F, 0x5D3FE815, 0x51B1E0A6, 0x5AA03AD9, 0x9FBF0803, 0x17115048, 0x3F0F7EDE, 0x0FFD1FB8, 0x6EE3BCF6, 0x5C235E91, 0x30E37686, 0x28B3FAD5, 0xC0FC5EF2, 0x50E89E68, 0x8D0EB044, 0xA0EE924C, 0xF4B5F98C, 0x5649015B, 0x27ECA424, 0xCE2ED97B, 0xABC7FCAB, 0xA13E4151},
    {0x1020CEF6, 0xBD689576, 0x2856C33D, 0xB8459FB0, 0x25905FA1, 0xE5D50F51, 0x0D0613EE, 0x2153B644, 0xD181B86F, 0x8F0CFA68, 0xBE385BCB, 0xA3BA53E2, 0xA974A6B7, 0x2DFA4A34, 0x948FA559, 0x1DFFD943, 0x858A8CB3, 0xB03FD94A, 0x46B47618, 0x64D816B5, 0x38F90501, 0xB94BBDDB, 0xAA9FE4B1, 0xDCDD38E0, 0x55889794, 0x676282E2, 0x157421B1, 0x6AEF205F, 0x47364182, 0x8C90335E, 0x618DC969, 0xB5A7B45B, 0x2926C70E, 0x4FD9CF20, 0x7560AF5B, 0x30C0F4A9, 0xB3C3FE19},
    {0xDFEA6F50, 0x2818CBDC, 0x64F0F0C4, 0xA016843D, 0x39A9EC1C, 0xE47945E2, 0xF6EB7E02, 0xE4706B9F, 0x76936CEA, 0x9F12F06E, 0x0D24A2EE, 0x5FFF8F62, 0xDD067681, 0x8D3D966A, 0xCE7B77E4, 0x1BDF6401, 0x77BCFA0B, 0xDD3930DF, 0x66299486, 0x85C4443C, 0x876B786C, 0x5B7830A2, 0x11CEC30B, 0x9EFAFF07, 0x6FFF4684, 0xE9084A98, 0xE65F3EA9, 0xBAD86267, 0x43D31557, 0xBD29D399, 0xBD308156, 0x975BBC74, 0x8A6636CD, 0xCDEF5351, 0x7E222850, 0x874132FA, 0x9667FEEF},
    {0xA0F573E9, 0x43F6C249, 0xF8CB1FC2, 0x85D38E5D, 0x5B46CE25, 0x54C6BED4, 0xCE421D59, 0x1BDBEB19, 0x54194712, 0x152CF8F8, 0x50D9DAF0, 0x40A46C30, 0xDC7E24E2, 0x632BC0F5, 0xD888702E, 0x0524D889, 0xEE780C51, 0xF1C8B145, 0x773B7C9D, 0x00D8DF6F, 0x0E7DDC14, 0xE4565037, 0x53CABD97, 0xBE38B7F2, 0xECC554C3, 0xE2E2F61B, 0xB19ADD23, 0x2691870B, 0x6BD53A5E, 0x37971655, 0x630B7BD7, 0xFA2C20E3, 0xFC25D8DA, 0xDFD79A20, 0xF3E1F036, 0x9F47A4EE, 0xB0B2FB61},
    {0x5E1DE654, 0xB95F1014, 0x7780B690, 0xD97D7DFE, 0x9A6192C5, 0x57071F56, 0x59B48225, 0x1157782B, 0xBD4CEBFC, 0x944A32F9, 0xF8D1BCDD, 0x1113796F, 0x13B15892, 0x4E960652, 0xB67F8027, 0xFF47AF65, 0x0C1379EF, 0x30D8B7B7, 0xCFC4C878, 0x0D0A2AD6, 0xB4C94DDB, 0x5F382A55, 0x5FC5B7F1, 0xCF01A8D5, 0x61874246, 0xA4EF39C8, 0xB5CD4DEA, 0xEB7D1C45, 0xB49DB456, 0x959388A8, 0xA58790E8, 0xC96B29E9, 0xC9CE2479, 0x3B259620, 0x5D1E7142, 0x5ECA651C, 0x608A597B},
    {0x6A4AF458, 0x393C7AED, 0x2980A266, 0x87624706, 0x91D65092, 0xD0B1F145, 0x871EEDF4, 0xC7BD2135, 0xDBFF3DAF, 0x519DBA5E, 0x3FC950DE, 0xFF7B52FF, 0x9FD3B28A, 0x2B0DD9DD, 0x18B79457, 0x623456AB, 0xF1FC67A4, 0x416ED857, 0xE739F8B4, 0x9E693A1E, 0x4A0A9D43, 0xA7750CBC, 0x6EBE7DDE, 0x74EB6372, 0xDB8B1935, 0xF3F0BD18, 0xCFF5FE4A, 0xE44BEF93, 0x3CEC7118, 0x416E0BD5, 0x8ABFFC23, 0xED382F4B, 0xC307B6D9, 0x4C3651A6, 0xBD4F0FAD, 0xB35A93D4, 0xAF111F15},
    {0x54DBBE22, 0xF64E8BD8, 0xC1445C83, 0x2BD859C5, 0x4BCBFEC2, 0x1E493E2B, 0x9CCBEBC8, 0xC240352B, 0x4C3B9F2C, 0x4C445523, 0x30A2FEAE, 0x8B274592, 0x75BDD1EE, 0x4C5F9A85, 0x053D17EA, 0xAD67ED4B, 0x6D9A6ABA, 0xF5286BCA, 0x12B45A48, 0x6F393424, 0xE5698639, 0x51B68BAF, 0x121788EA, 0x4CD37059, 0x6E5F3BB5, 0x4810773C, 0x60F24874, 0xC7D864C4, 0x6B9EB4E0, 0x7CE45BA0, 0xBB3ABDB7, 0xC09483B9, 0x0DA2A952, 0xDBB617A2, 0x8AA57D3D, 0xC6D82DB8, 0x205D8E42},
    {0x152DADE6, 0xD675E877, 0x8D8C114D, 0x7DF045E6, 0xCE4B4050, 0x87DD732C, 0x2580CDB6, 0xCBCBF5A9, 0x6C6181A0, 0xE9F7867C, 0xE6FB5F69, 0x2676FFFC, 0x902FEDBE, 0xB1234099, 0x26878C9B, 0x98B291EF, 0x83F45B7F, 0x08C51883, 0xC1FF7874, 0xF169BB70, 0x869631E6, 0x8CE026CA, 0x1F234BC6, 0x75F3EC67, 0x10E295C0, 0x8032541A, 0xEC16E7AC, 0x91982EE3, 0xCFB2891B, 0x4EA8A154, 0xDE79AAF5, 0x24885B9F, 0xB284E30A, 0xC162D760, 0x05967BC9, 0x7ABF38BC, 0x171B591F},
    {0xA81A654C, 0x1B8A2885, 0xB0900634, 0xEC569F2F, 0xFA0D173B, 0xB276E487, 0x012AAC52, 0x21BEB825, 0x0DD26D7B, 0x1BBFD031, 0xAE387C5A, 0xF20E5653, 0xB6657F21, 0xB323733C, 0xB1956D85, 0xA5115A52, 0x27D422A7, 0xA40A1D3A, 0xCF0BDD34, 0xE796C068, 0x8D78A06B, 0x3B317AF2, 0xB3E86B60, 0x24C8B023, 0x1378CED1, 0xB7EA96D9, 0x3138332C, 0x9E30465B, 0x8B841187, 0x2F5DA4DF, 0xAD22015C, 0xF0ECF1C5, 0x77C84A2C, 0x880035A6, 0xF01E1E54, 0x6505E592, 0xB9BE8AB8},
    {0x9DBFB3B8, 0x2279B27E, 0x480F3EA7, 0x2D85BFDA, 0x30682319, 0x6492C674, 0x2D736ABD, 0x3C40681B, 0xE19CF0EF, 0x409B55A8, 0xD040B7BF, 0x494D99BF, 0x5BAAC02A, 0x92A1AAAD, 0xFFC3A440, 0xA067C014, 0x45540DD4, 0x7FD1C560, 0x4983F62A, 0xE11C858D, 0xE42E537B, 0x1224D508, 0xD32A67BD, 0xF4BA1F59, 0x74820E48, 0xCD7B9712, 0x11083EA0, 0xCE3A4020, 0x97AA25B4, 0xD7B4BBAC, 0xA9B346CC, 0x30C47EE9, 0x0AB6F105, 0x912E8E98, 0x05C750FA, 0x8597C472, 0x09DCB75A},
    {0xBB031198, 0xB2DADB21, 0x08A5E184, 0xD7BB0AEA, 0xA14D4F1D, 0xAF5F03B2, 0x07109441, 0xC175F8B4, 0xF9EF0A05, 0xD8377F1D, 0xFFE57930, 0xC1BE6C3A, 0x4D450CD1, 0xF2A9E0CD, 0xE8E9BC1B, 0x86404CA8, 0xB3667DF0, 0xB8175350, 0x236C37A1, 0x1B242E2B, 0xA5280698, 0x800B4956, 0x6B26589A, 0x503F1068, 0x9149FCC1, 0x1A1B921A, 0xEF402BC7, 0x726327F8, 0xE84179AE, 0xEAC18012, 0x836C9489, 0xB3FEE097, 0xCC993D39, 0x35A269D4, 0xC7102AB3, 0x5341DBCF, 0xA55CC71B},
    {0xE9944ABE, 0x9B51F6BA, 0xA8872BEE, 0xFE88B2BE, 0x43A7AC04, 0x7ED6F4CF, 0xA727DA60, 0xF8681B3D, 0xEB507489, 0xC5951DE5, 0xFEFB92CC, 0x8BEAE2D6, 0xBA4F1C28, 0x2C48F619, 0xA4BA8A8C, 0x710341EC, 0xF75C9120, 0x8B85EA23, 0x718C3B23, 0x23524B50, 0x85DDBFD1, 0x1893977C, 0xC0D871A1, 0x70ADB4E3, 0x4980AA7F, 0x6BC712D8, 0x9E264ABD, 0x8BCB9D16, 0x53D15B3B, 0xACE47468, 0x94B73C97, 0x4731BF6C, 0x291FED9C, 0xEDDDBC4A, 0x642DCCD0, 0xEF5682AE, 0xA1D4A00C},
    {0x149705B6, 0x9B86E0CF, 0x01E51FDD, 0xB0CAB317, 0x1CEC9477, 0x00B7D4F6, 0xF6F7105A, 0xC229875D, 0x2A0E8AC1, 0xDC8EA9A1, 0x8187877B, 0x540069CC, 0xD2A6119F, 0x0EFD408E, 0x23489746, 0x84B9F4DB, 0xF530E119, 0x0EAB024F, 0xFB8DC5CD, 0x04004988, 0xF02C684E, 0x2847956F, 0xCF7A078F, 0xA93D7B85, 0x7CB2F899, 0xC882B8DC, 0xEB4050A8, 0x1995109E, 0x75726BE7, 0xF6301DDE, 0x5899873B, 0xAF4602BB, 0xA6E9E17D, 0x011CA231, 0xED257328, 0x73522956, 0xFF307703},
    {0x1A22722E, 0x86E35586, 0x09C74027, 0x733C3533, 0xE9CBB096, 0x1A6F14FE, 0x29837C91, 0xB0F81407, 0xF7B053D1, 0xD6E90714, 0x28EC5AF1, 0x31C9BD82, 0x65566546, 0x83DE8BD1, 0x960E02F9, 0x6305CC9E, 0x798627BC, 0x20F65A05, 0x3341B9AE, 0x00D8C52A, 0x28A156DD, 0x216F564D, 0x2B897C8B, 0xAAB3B316, 0x8B160826, 0xCB15C9B9, 0xA1D973BB, 0x2CDD66D4, 0x03FE8222, 0x082B523A, 0x903E3DE7, 0x8CED51ED, 0x278D6284, 0xB70F4D0D, 0x2FF0215F, 0x8324F666, 0xB775560D},
    {0x9FBB1F5C, 0x023268F1, 0xB1F35B0F, 0x3CBE6594, 0x20D27239, 0x62634C31, 0xAE1BC45F, 0x5780B3E3, 0x87B9E51E, 0xC8A27A48, 0x886A1E53, 0xB2BFCBEB, 0x1FACB990, 0xEABE010E, 0xA5380A74, 0xC91869AE, 0x75057E52, 0xA946AEE2, 0xC3A93A52, 0x777475F5, 0xB2CBD31C, 0x9F13CB84, 0xB2614852, 0x4DE5E973, 0x831F234D, 0xF33085B2, 0xB87F493F, 0x7A658C19, 0xA1198853, 0x37F77300, 0x295AAEB3, 0xA0901340, 0x8303E7FD, 0xBD916596, 0x060F8612, 0xAD9A6050, 0x8475A778},
    {0x20FE87C9, 0x13D1592E, 0x22CE4553, 0xF5D2A2DA, 0x6ACA8331, 0xB963FB92, 0x8C8A4385, 0x45646817, 0xA3922B88, 0x4C3C4C15, 0x2CDF0410, 0x7791BFE3, 0x8DF5EF4D, 0x85CF3FF3, 0xEAF22E29, 0x0DBBD744, 0x1A87C469, 0x4F268786, 0xB3E8370B, 0x75CBDEB6, 0x603B298B, 0x398BA81B, 0x44966028, 0xB9346698, 0xF3CF6DE0, 0x09E50280, 0xCD027A4F, 0x39EFF780, 0xC65FD3C0, 0xCED394E0, 0xFD4F353C, 0x6EE4AC25, 0x81AE2ED6, 0xAD057958, 0x9794F8DE, 0x34F8D3A2, 0x42BA7025},
    {0xE10DF81B, 0x78DA3879, 0xE141A1BE, 0xF98FFBE3, 0x9F45E869, 0x99DE2040, 0x55DF0E95, 0x527686FD, 0x9B2219E0, 0x389D4CB0, 0x0615D485, 0x4B42E23A, 0xC7560E98, 0xF7005C57, 0xB1B0E417, 0xB65D74AE, 0xA4C5B30C, 0x5D68FC1F, 0xE4C24E74, 0x8A72EB5C, 0xEE16DB7F, 0xBD005C59, 0xEC5AA7B6, 0x69148304, 0xB395F1DC, 0x9646CDB9, 0xC349362C, 0x3D62E3CC, 0x0CFA458E, 0x03B7F0F3, 0xCA3D1E73, 0xB690FC27, 0x9B8C7E36, 0x5EBE117A, 0xB0C80247, 0x777DD04E, 0x02D5B582},
    {0x55FB01B1, 0xA60C90F4, 0xADDDA069, 0x7AA6729D, 0x30FA2AF5, 0x375F4C8A, 0xD2CB2E57, 0x51336659, 0xFAFC94DA, 0x662183CC, 0x2A220CFD, 0x12E28D11, 0x1DD9E71E, 0xF0F72C38, 0xD64A3256, 0x984C29B9, 0xF9FA4CC2, 0x1799BB6C, 0xA4EE5BE7, 0x62F14571, 0x0B7D4E67, 0x2537FA9F, 0x4D429514, 0x36EBD5C3, 0x228BCB03, 0x772B7D7C, 0xDF4AEF46, 0xAD276A9B, 0x200621DB, 0xAD3BCFCD, 0x63877554, 0xD20A3269, 0xB6502B29, 0x9DBAAE88, 0x3E352CC3, 0xA0557AE2, 0xB3794E4C},
    {0xBCDF6FF9, 0x82FDDE11, 0x302BB814, 0x5A837560, 0x940A2BE5, 0x8D668B80, 0x0D6084A8, 0xC98492C5, 0x75BA5C68, 0xA55A8659, 0x6B82041E, 0xA1267439, 0x6428C7EE, 0x610FF88D, 0xDF34BF8A, 0x2E9102A3, 0xDBBD6D35, 0x90646016, 0xBE5CB362, 0x8E00E44D, 0x3CB48501, 0xD809D2F7, 0x2144FE74, 0x6C55DE64, 0x695FFEE4, 0xA9A10C69, 0x95125084, 0x038A229A, 0x2AE69CE1, 0xB149BADA, 0x118A5D40, 0xC6C9E5DC, 0x0F37BAE9, 0x5A72C716, 0x5AD980F4, 0xA5603797, 0x65ABAF1D},
    {0x2A2F5EE0, 0x4E6F3202, 0x84456AD0, 0x0D7B34E2, 0xE592B4D7, 0x6FEE012C, 0x8F24A63E, 0xA68CEE97, 0x74DCDEA9, 0x26E293F3, 0x22574790, 0xD71E9AFC, 0x4768A5C9, 0xFC0DC4A8, 0xEBC03C2F, 0xF2C34D37, 0xEA4E8E16, 0x0852DCB4, 0xCC4DD35C, 0x17107B88, 0x70242D12, 0xD435D13F, 0x678193B2, 0x08E8D469, 0x9D25A819, 0xC6CBDA13, 0xB9F993E9, 0x458C7F6F, 0xF76FE226, 0x15C817ED, 0x245AF255, 0xE879BEDF, 0xAAA72F2F, 0xD9D58F6C, 0xC5F43731, 0x86C79F88, 0x4AD7D179},
    {0x71C8E659, 0xF4898843, 0xCF89C432, 0xB9B71989, 0x581F0E0D, 0xD1618215, 0x922A86CE, 0xFBEF05F0, 0xB716E8BA, 0x6FC89A02, 0x0AFA1CB2, 0x6DCD4434, 0xCDF48DDF, 0xEA22EDE7, 0x8DF93388, 0x05C279ED, 0x876AE483, 0xA9F8F0A6, 0x9A3D4CC9, 0x16AE8F18, 0xAB6E9DBB, 0xEA8D2B42, 0x3CC0169C, 0x767B576A, 0x10E59937, 0x1AEE13A6, 0x5F7A14A3, 0xD71A9901, 0x56636BD8, 0xB4EC7391, 0x836980C5, 0x30995022, 0x53549581, 0xB563283C, 0x9887304B, 0xF227AD57, 0x9CADF482},
    {0xE6A3D964, 0xB989C82E, 0x7DFBD74E, 0xFEC4C9D3, 0x6C1BD1F7, 0x0EC84575, 0x34F78B9C, 0x798D0166, 0x1B102A40, 0x03688C95, 0x543E5812, 0x3AE6A363, 0xE21A9546, 0x72086FE6, 0x0C55CE76, 0x0BE2FD89, 0x7C6000B0, 0xC917F9A9, 0x71278D56, 0xCBAEEC53, 0x982E8180, 0x3C428A8E, 0xC9826D37, 0x56317D3B, 0x37641B8F, 0xD515D389, 0xC660F466, 0x30273115, 0x35A35DDA, 0x355FD4E7, 0x8CA77939, 0x263808C9, 0x3F2C5FB9, 0xAA824B33, 0x15BEC70F, 0xEACA78F4, 0xB866678C},
    {0xBE39631B, 0xDC0029C3, 0xBEE7B2F0, 0xE8035CCF, 0xB68BE6D1, 0xB9B7F1C1, 0x281851B5, 0x3ABA2C98, 0xB4BEABD2, 0x85D11E5A, 0xE0ABCA08, 0x1C07BF13, 0xCD8AD0F9, 0x886BE581, 0x9A331C21, 0x098BCEEB, 0xB23AC580, 0x939F6B5F, 0x4ABD19F6, 0x632BA5C1, 0xF9D8F978, 0x1CE65E90, 0x287E7CE9, 0xE40F2629, 0x207E0204, 0x0BF80BB8, 0x95066834, 0x3F0DC027, 0x1710865E, 0xFADCF580, 0x3BC5899F, 0x0148F042, 0x4FD09ECB, 0xAE08E78E, 0x5F0DA2B8, 0x80B04691, 0x30731ADD},
    {0x95A8E184, 0x984E3774, 0xB5F17E64, 0x30585900, 0x3BB2C27F, 0xB585D053, 0x12B8C399, 0x7107D164, 0x67C95C8C, 0x120A64E0, 0x2FADA4CA, 0x5EA2080F, 0x1BA6ED8F, 0x4A1852C3, 0x28049969, 0x838F8422, 0x773FA6A3, 0x25DD22D3, 0x6BF99E17, 0x661DD59C, 0xFD2E2265, 0x88D8EFB9, 0xAC44BD0C, 0xAB28313F, 0xB5E7F150, 0x0DAD3C08, 0xC2E6E380, 0x77BB3CA2, 0x7E34A58F, 0x488A9222, 0x76AEE784, 0xEEBFD7E5, 0x57676BCF, 0x75243656, 0x6D617D37, 0x1163DD18, 0xAFBCC060},
    {0x77E4EF37, 0x9BC73CA0, 0x3E27B28D, 0x8DBAA090, 0x59A321C1, 0xC2B74DDD, 0xD43918E2, 0x43A8F4BB, 0x6950E980, 0xBCD45367, 0x98FB8A75, 0xFF7B5FEF, 0x63A5A515, 0xDD2A4F63, 0x339AD8D9, 0x5EF6954E, 0x20D50381, 0x82D81CD5, 0x41EEF0BE, 0x107539CB, 0xDF39D64E, 0x7E500513, 0x03D41E23, 0xBECD598F, 0x3E311275, 0xC9A82474, 0xBB63EB86, 0xAB83A078, 0xC74284F6, 0xCE5C3CA1, 0x11BACD3A, 0xBB10EB20, 0xB72A6278, 0xC1BB7910, 0x99EFF903, 0x7C1D91D9, 0xA6D1AEDD},
    {0xD68B1CDD, 0xDBE65AA5, 0x5BFD3013, 0x6A52C4A8, 0x3CF6AE91, 0x2C8521E2, 0x9C0E742B, 0x1247BE0A, 0x8CDDA2DB, 0xBA04814E, 0x71861F03, 0x157CC265, 0x6A154937, 0x72DCCD63, 0xE0E50A5B, 0x4C8FAA8F, 0x52EDC973, 0x31B56B06, 0x67A6DF5E, 0x2E439197, 0xAF80903A, 0x6F45A28C, 0x5699D608, 0xAC80392C, 0xA27C5FE8, 0x6A0747A7, 0x8A03B6B4, 0xF4619E70, 0xFDA16280, 0x29585F37, 0x7E8359D3, 0xBD886D98, 0x3000A31A, 0xBA96BA0D, 0xA20A6BA7, 0xD57F8EC0, 0x138CCCF0},
    {0xD5B6CB91, 0x82B506D9, 0x07FBDE4C, 0x63382250, 0xB47904C1, 0xA73D837A, 0x6593FF1B, 0x1B04625E, 0xFAA2441D, 0x52743F3B, 0x5993A4F2, 0xF2E2AF51, 0xFCF26251, 0xE2A970F8, 0x3068B2DD, 0x2171EB4E, 0xE4EB64C0, 0xFC150B47, 0x65A14F2E, 0x8D1E5D40, 0x743573D9, 0x951FBD1D, 0x5393D468, 0x632922CA, 0xC101E9F6, 0xB66B32C7, 0x127F1A4A, 0x6F5D12FC, 0x58144991, 0x338A513B, 0x8CBA8F78, 0xFD606AAB, 0xFFC39EC1, 0xA283937E, 0x1CE135B4, 0x80320A72, 0x0F872B00},
    {0xBA2B66EB, 0xC571A72F, 0x55CA57B2, 0xD8F454E3, 0x17E72B73, 0x281244FA, 0x51E88500, 0x0A60AA8D, 0x0C3ED18A, 0x4DE70F90, 0xEE97F088, 0xE9FD6EAD, 0xEDAE2F10, 0x5D60DF88, 0x5B908F80, 0x59CDB7AB, 0x3254BC47, 0x2EBCB06D, 0x11AC6743, 0x9EECA2AB, 0xFA0D095C, 0xE4628FC9, 0xC3641C86, 0xEB056992, 0x0A1CC4A3, 0x616DF2CD, 0xA10F3132, 0x9A619167, 0xC7FD10BA, 0x1F9437EF, 0xE6BC3B64, 0x0E7CC07D, 0x000797AF, 0xE3B3C82B, 0x1CF3859D, 0x8D8CD8CA, 0x98ED93B3},
    {0x031E5197, 0xA752D6C5, 0x24682582, 0xFEFC9CCE, 0x97492147, 0x7D9AACB4, 0xC0BD8A14, 0x93E2B611, 0xE0F9B053, 0x289B34B1, 0xFFDF8A79, 0x5F75ADE7, 0xB3A13473, 0xD34AC7CB, 0x7D4C3B2F, 0xAF199350, 0x85C254F3, 0xFDDEFEED, 0x92BF01B5, 0xF4738727, 0x746E80BC, 0x75AE9A77, 0x94389B62, 0x36B3F199, 0xF7E2BEEC, 0x5962405F, 0xEC3A7CD4, 0x250B7B72, 0xC677528C, 0x3CCD569B, 0x00FB0388, 0x61F11273, 0xE9F7A0A8, 0x9813CD13, 0x57C4502D, 0xEFC4CB10, 0x51A9EBF8},
    {0xD7FF9B11, 0x744FCD0A, 0xF527AC9F, 0x05931F42, 0xC1A471A6, 0x5AA593F1, 0x15A3742B, 0xC4FAFCF5, 0xCC93AFA7, 0xC75C6198, 0x7F3BD7ED, 0x141C3B85, 0xCCB34A72, 0x6A25623B, 0x81132A99, 0xC6A65586, 0x31656505, 0x7C630898, 0x7030F582, 0x69F74E3E, 0x9E9976EF, 0xD858BAC2, 0x40A17057, 0xA92B9879, 0x20DD03F8, 0xE83C5201, 0x5E434304, 0x29E0B787, 0xE3A1A90D, 0x643A53E9, 0xB374FCA0, 0x72EF02F1, 0xA4606AFD, 0x4CE5C561, 0x96D005A6, 0x6B19E7E8, 0x5BC1F210},
    {0x8DE45E60, 0x138E996A, 0x557B91FD, 0x5A1D1954, 0xA30D7795, 0xF6F57B3B, 0x4E7A1CFF, 0xB729E83D, 0x06419B28, 0x5D9B6268, 0x06B9BFF1, 0xD0289A32, 0x66FCEAC2, 0x36CA984C, 0x2599BC4B, 0xF55A6AF5, 0xB8720CD4, 0xAFF14877, 0xE38C92EE, 0x205B1071, 0xE41A779D, 0x9BC303CF, 0xB35DE416, 0x8C44C606, 0xA2A62B00, 0xC307348E, 0xA030B6C1, 0xDAB10621, 0x95A4CD1B, 0x8A1EEEE9, 0x349296D4, 0xBEA73086, 0xCD4EE813, 0xBADA9E72, 0xE1483653, 0x92DBDAD5, 0x0EB804CD},
    {0xCA1E8E1F, 0xEC973A1D, 0x0609F8DD, 0x80D25A35, 0x23A8915D, 0x85F4DA95, 0xF7CEE84C, 0x15F7560A, 0xED4F3353, 0xC095F150, 0x90461656, 0xFD806545, 0xDB0DEC25, 0x81FB306B, 0x1916B13B, 0xC52D632A, 0x4810298A, 0xB14B423D, 0x30BC11C1, 0xB93D69C0, 0xEA8638EC, 0x84A0DBAD, 0x93B507FC, 0x35EF323C, 0x99FF9F6E, 0x926AB176, 0xEB6BCFF1, 0x78BEE4CC, 0x32A8BAE8, 0x08A29566, 0x7CFFAB78, 0x946A59FA, 0xFE953281, 0x8D453F1D, 0x32577972, 0xF20FE794, 0xFD59968E},
    {0xAAD94587, 0x6F6ADB0B, 0xCE309287, 0x55C99EF7, 0x72023F25, 0xE45D4412, 0x94206632, 0x6E87D8EB, 0x53DC34EF, 0x67462315, 0xC7030FDF, 0x6BE11570, 0x070B9636, 0x88B14325, 0xD2D5830A, 0xC067831E, 0x46D4852C, 0xB0FCE7F5, 0x5AA0B7E1, 0xDA9AA2A4, 0xDAE36F6C, 0xAD3C6099, 0xD72FA3C7, 0x51C0CC98, 0x6E55929C, 0x6F2CF934, 0x9182DEBA, 0xFBA122C5, 0x631A1CE4, 0xD55F269E, 0x1200239F, 0xD25F4B50, 0x872BDEBD, 0xFDC28608, 0xC06882AE, 0xD0F2361F, 0xDF42C009},
    {0xA1E4AE45, 0x4E8D39DC, 0xA2B2B308, 0x834219B3, 0x8DC299BA, 0xE57FEA1C, 0x55F57FA6, 0x59502DF5, 0x6B7BF64B, 0x9D25CF20, 0x4C968F71, 0xE58E0948, 0x9B8350F1, 0xDD003AC1, 0x6B6F66AB, 0xBAFA6841, 0xB15DFE3C, 0x0C8EC81F, 0x86B968F4, 0x9D1BD557, 0xF677DBEE, 0x047964B8, 0x645F330A, 0x9D0D9409, 0x22B857E6, 0x362C0BE1, 0xFD24479F, 0x9F04AE41, 0xA3454545, 0xAD26E466, 0x50529497, 0x166529FF, 0x8A6E4A2E, 0xBFBF1CC6, 0xC1E0FD79, 0xA86235D9, 0x95DF6894},
    {0x9785AEF1, 0x70D13941, 0x72740888, 0xA072159F, 0x1C3FE5F8, 0x494F9ADD, 0x4B1111B8, 0x9382346F, 0xFF67B053, 0xE5639A9B, 0xCCA10603, 0x1329A007, 0xF110F1CC, 0xD3325426, 0x50DC2AFF, 0xC5FB1163, 0x61059019, 0x9C47AC21, 0x56887B98, 0x7267F394, 0xED1E039C, 0x0563EC08, 0x733DA614, 0x2A3206A7, 0xB33458D7, 0x5CC11243, 0xFA4F7E97, 0x3C949109, 0x291F68C7, 0x6EE4DE70, 0xA8EFD620, 0x31C30B96, 0x6C9F60D4, 0x0820AC46, 0xE23CB2FA, 0x3F4DF95E, 0xCA0C6477},
    {0x04AE4641, 0x2EAAC8E6, 0xA21A2757, 0xEB89E797, 0xEC499B26, 0xDB8BE0BC, 0xF45F503D, 0x920368E0, 0x7D249561, 0x9E9A171A, 0x84DC3A8E, 0xAE99DF3A, 0x434E8599, 0x244D847D, 0x4C7143E3, 0xEF19492F, 0x5DD480D9, 0xEF62386C, 0x6F21D732, 0x4D3815CC, 0x42BF52DA, 0x82666524, 0x958E2C1B, 0x9685B619, 0x3E14D10C, 0x46B7F996, 0x46437CD6, 0x6F6A1994, 0xA27735F6, 0x0CEF03CC, 0xB7E3B27E, 0x2540FA3E, 0x7239032C, 0xC17EABC4, 0x1E676171, 0xD609A948, 0xE193B699},
    {0x77E31BF0, 0x02D74F07, 0xAA5997FC, 0xA652A675, 0x3A263F68, 0x4935A819, 0xAE813D17, 0x9743D6C6, 0xD55F370D, 0x3C5BD9BA, 0xD3BFB2AA, 0xE0333F2B, 0xDA23460A, 0x64EAC5D0, 0xD1F60C91, 0x5DAA38F6, 0xFE2698DB, 0x6E0F162E, 0x8B8AC782, 0x67FA4BD3, 0x0A5E239C, 0x2EC299EE, 0x33EC2F1B, 0xFB9D7380, 0x0D06E502, 0x8ACE5CCD, 0xC16CFA42, 0xDA588BE3, 0xD20D0AA5, 0x7A465010, 0xA88FD11C, 0x66E75FBC, 0x617556E7, 0x5BFE7437, 0xB10EB50F, 0x74809232, 0x2988A883},
    {0x37A34CE1, 0xD85776FD, 0xD474D839, 0x4D97F895, 0x709BFF5C, 0xF575B91C, 0xA55D19A0, 0x53FEF3CF, 0x5E56571D, 0x088DA979, 0x100F834D, 0x1870C068, 0x47523EBE, 0xEB3976BB, 0x19F2CABB, 0x1C34D9B7, 0xDE95F0A8, 0xB7613A64, 0x3E1728B5, 0xA8EB7C43, 0x1C74FA5C, 0xF9AB6317, 0x93D3DA75, 0x36814DD2, 0x762D8B53, 0x54D386DC, 0xE0742193, 0xA4414406, 0x46505E34, 0x991AE5D2, 0x40EB92DC, 0xAF32E681, 0x3D6A4192, 0x0802E36B, 0x55DBAF9D, 0x11B9BEE1, 0x865C9712},
    {0x35FC6D27, 0x8372D70C, 0xF23902E2, 0x1CCF21F1, 0x597643B3, 0xBB69A507, 0x41CA2AA4, 0x6FB2C701, 0x9F0861D1, 0xF33947BE, 0x9228728E, 0x57B46F8A, 0xED298496, 0x0857A65B, 0x8F0DEE19, 0x0D547A0A, 0xA8AEAE6F, 0x9643C874, 0x81443A02, 0x1192CC11, 0xAF714E11, 0x167FC846, 0x23579D4C, 0xEB75DE7F, 0x17F910E3, 0xFBD03FF9, 0x48A8B34E, 0xE2E774A4, 0x70BFB947, 0x6AE8C6BC, 0x368D9AB3, 0xFEE75395, 0x37EDB729, 0x39A67562, 0xFD276142, 0xBA32D7CF, 0xB77D23C5},
    {0x5D5B69AA, 0xC6348906, 0xF9F1C026, 0x17C55876, 0xFB2C6B16, 0x5516640A, 0xAACB2482, 0xCF625806, 0x708B273C, 0x8069068E, 0x2A8C61D5, 0x1E07BE12, 0xA711DF1D, 0x833F2D5C, 0x1ED0E475, 0xE4F1E41D, 0x3F70407D, 0x1272296F, 0x8FAC80FF, 0x7851C89D, 0x5C33458A, 0xD54C4CFB, 0x58389725, 0xAA927B5B, 0xB2157D56, 0x46DA2947, 0xC7408D83, 0x5E7FE305, 0xEBD8AF39, 0x55B5758B, 0x86E663F4, 0x5C87ED13, 0x688A6549, 0x54661492, 0x5A521EC1, 0xFA7213DB, 0x4CEE6A43},
    {0x4E9945D8, 0xD210AB74, 0x077E15D2, 0x9BA49B84, 0xA3515A82, 0x94655FC9, 0x23294878, 0x236E1CA5, 0x01531E9D, 0x9951A703, 0xD940FB2C, 0x95B48B26, 0xBD70DF91, 0xBDFDCD93, 0x28399A7B, 0xFD279D37, 0xA6483FB2, 0x47D373D4, 0xDF36B400, 0x60056314, 0x952F9FB0, 0x95F3CD51, 0x6359A012, 0x50249DA1, 0xD4F5AF1B, 0xE862ACE0, 0x7D6F80E6, 0x748E5D6D, 0x5DD8737F, 0x25DE3C07, 0x50E45166, 0x52DA0109, 0xDEFF1AE3, 0x103FB79F, 0xE1EDFF8D, 0x88E3A974, 0xE47BFD40},
    {0x6CE4825F, 0x48721D73, 0x4B0A2B88, 0x71BAA292, 0xA8254FA3, 0xBE54BAEA, 0xFDDF176D, 0x13B802A2, 0x36F63936, 0x51A803D5, 0xD6A990DB, 0x47745034, 0xF0DDC8A1, 0x7ECF212B, 0xB82EDC2E, 0xA56CC339, 0x2D31B7BC, 0x043B2179, 0xEEB3A6AD, 0x41D5FBA7, 0xAC8AC3C0, 0x3C5161CB, 0x50D45B9D, 0xB76AA4F3, 0x55944F6F, 0x8F0CBCA6, 0x0A3F86AD, 0xAB8D0892, 0x5A601630, 0x72AF6BC1, 0xB28EDED4, 0xA4CDED33, 0x74D77AE7, 0xDA576086, 0x1348306F, 0xBFAE5B05, 0x7D203F02},
    {0xFE7296A0, 0xAF7677D3, 0xF3506729, 0x323F5E82, 0x7B8D3650, 0xD319F17C, 0xFDEFA604, 0x2E19A191, 0x314F7C94, 0x00ED0AD5, 0x581A3AE2, 0xA62182A6, 0x5A1D8ED4, 0x60728B4F, 0xB8D0A409, 0xF40771D8, 0xBC743425, 0xE87F584C, 0x230BCAF6, 0xA4CD27BC, 0x9B624B3F, 0x335EFE85, 0x898C6708, 0xD6B41FE0, 0xE57A1CF3, 0x7D8E7865, 0x9933AF2B, 0xB7E5A3E3, 0xD473126D, 0x453E8D29, 0x3553F741, 0xA2D1EC55, 0xAD0FB55B, 0x860760ED, 0xDA8361DD, 0xF10F7069, 0xB7134699},
    {0x3AC8473F, 0x567E481E, 0x4973A6C3, 0x7391F93C, 0xB35C0152, 0x9F187C84, 0xBEE910B7, 0x0F3D7FA1, 0x9B789759, 0x379FEE58, 0x838D1817, 0x444C8BA8, 0x3141EEB8, 0x0FFC473C, 0xF8056826, 0xF29F245D, 0x4925E88B, 0x87C5D1B2, 0x07E31DD1, 0x9B521BFE, 0x840ADBFF, 0x4217662F, 0x92BB6545, 0x0D859DEA, 0x84B6C125, 0x496A9B0C, 0xCE7E52F4, 0x37821EA5, 0x6E114C8F, 0x4AFADC1B, 0x3EFB5A3A, 0xEC9C649D, 0x282921A5, 0xCB90D05D, 0x331900D5, 0xD384762D, 0x51C60340},
    {0x05A6A9A8, 0xF0921B7F, 0x90FC7748, 0x7072DCB1, 0x715FCD96, 0xFB10DB5B, 0xD78AFD3B, 0x4C6087A6, 0x565729BF, 0x184A8E81, 0xDB54D46A, 0x7E6D03BC, 0x82DA6547, 0x54E32905, 0x98638C14, 0xDC2C3A2F, 0xF4FB2F0C, 0x42FCC515, 0x3F7439DA, 0x3F06AF79, 0x699D0193, 0x32984E2C, 0x4BC7F213, 0xAD49166F, 0x1BF3778B, 0xC1870AF6, 0x59FB593C, 0x0F056271, 0xB09E66A5, 0xCA519618, 0xFAD1484A, 0xD6790857, 0x9D69D1FE, 0x18B87E4C, 0x0710C1A3, 0x99B2EB8E, 0xC85BBD01},
    {0x640B2527, 0xAEE3177E, 0x3B2A7E5D, 0x1CA96047, 0xBF57F6B7, 0xA98D87EB, 0x30C95D17, 0x81535D68, 0x91FC9ADE, 0xFAAB623B, 0xB2F5069B, 0x43053C72, 0x4445E536, 0xF88A4E19, 0x7BBEF10E, 0x561C6B74, 0xCC465462, 0xF7CE2A11, 0x5B41E8FF, 0x1ED272F7, 0x518D9306, 0x942C92A3, 0x4F17C875, 0x0799A4A8, 0x7FF141DD, 0x26D7CA82, 0xF8F1E455, 0x9CD4019D, 0x6F6857B5, 0x597AFE8C, 0xE64F862C, 0xC857D0D6, 0x3395E288, 0x570ADF0A, 0xBEA416F4, 0xFC297A67, 0x03C359BE},
    {0xFABBA607, 0x8DE6FE3B, 0x19635B3E, 0x905F65EA, 0xDA3B6A06, 0x345944DF, 0x7895B27D, 0xA2C1B783, 0xA6C96B1C, 0x6F33BED8, 0xB0BB6495, 0x7214E4FB, 0x1D9CA72A, 0x35B32E5B, 0x37EFB052, 0xD2DB596C, 0x862DBD87, 0xFB4B9283, 0xDD8062D8, 0x6C350BF4, 0xA7B131F0, 0xC2A19824, 0xCBD85198, 0x8C913509, 0xC5E4666A, 0x7294F15B, 0x1AE03D2C, 0xD0471B3C, 0x2C834A7A, 0x534EB482, 0x26CCB1FC, 0xA2C493CE, 0x26462806, 0xA8C6BA2F, 0x8F8EFC8B, 0x091FD578, 0xE76B152F},
    {0xA2153053, 0x331E1674, 0xCD15102F, 0xF403E59C, 0x2C1BBD16, 0x7E688DBC, 0xEAD5AC8D, 0x9E0DC719, 0x8E5952FA, 0x087494E8, 0x3D004BE0, 0xDEF85680, 0xB7C67713, 0x45EF015F, 0xAFFCB312, 0xF0FDAF84, 0x3CB8F3B1, 0xA1687293, 0xF15C90ED, 0x0D5347EE, 0xE2FB21A8, 0x60440FAB, 0x1F6B5D15, 0x1124B85F, 0x10EE3774, 0xFB18E8AE, 0x3E3AB86F, 0x8DC0B504, 0x0CC52922, 0x57EC96D1, 0x467DBE13, 0xB16F8511, 0x1E90564B, 0x4D659DB1, 0x4D8B5559, 0xACE315D7, 0x066E83D7},
    {0x984802D5, 0x880EF765, 0xC86C62A4, 0x4D149184, 0xDBDCDBE6, 0xFA25FD75, 0xD99BEDBD, 0x52B2FE5B, 0x4A92E976, 0x66D3F23A, 0x90919986, 0x18A084CE, 0x14FC5E1A, 0x99115594, 0xF3AE4150, 0xA0542257, 0x0C18128C, 0xFAAB30BF, 0x3C1FE2B8, 0xABD2E8EF, 0xFBEAD8B3, 0x91C70F61, 0xF86CEFB5, 0xBF6CDEFD, 0x6CE8A434, 0x69C62876, 0x4DD3B3DD, 0xB0A525C9, 0x9FDD0FB2, 0x9E93B77E, 0x44DA3ACC, 0xA3F9D36E, 0x6AF9D851, 0x180D6192, 0x7E7E169C, 0x74AE2FAF, 0x651E661A},
    {0xA38F7D74, 0x09672B86, 0xD801E609, 0x8BD1D538, 0x02453D0E, 0xE43D2B66, 0xE85B1B82, 0x2210895F, 0x2D1E358A, 0x84C39499, 0xC137C965, 0x31F538FC, 0x49C38062, 0xE5F7B263, 0x2B65E37E, 0x6FA685CA, 0x5C6231BF, 0x7DF09031, 0xDC4CE949, 0xF9ED6BAB, 0x5526F0E7, 0xDBCC0779, 0xB1C8A38E, 0xE5449682, 0x2A8BB0D1, 0x94515874, 0x5188D5A3, 0x287B8741, 0x88A015B8, 0x4B23CC72, 0x41E24DF8, 0xA48C82FA, 0x42F540B6, 0x728AA3DE, 0x6CDA023D, 0x31FA1887, 0x68A86EFD},
    {0x221E23E3, 0xD9019D7D, 0xD9DA4F9F, 0xFA159929, 0x4E2EC6E9, 0x96E848E4, 0x599D1BBD, 0x6D53E1A1, 0x8A90A63E, 0x14268279, 0xD92F0B38, 0x2D243352, 0x3A88AA5F, 0x88AEB8F3, 0x643037FA, 0x6CECC201, 0x12A1CAA8, 0x06C52D4F, 0xC956784A, 0x924066AC, 0x2B6435DB, 0x9A1C2E47, 0x0B05F4CC, 0x3D2AD9D5, 0xB7A018D0, 0x3D122C9D, 0x47483BD0, 0xBBBCDAC0, 0x182F57B5, 0x02B73F31, 0x9D4B9D56, 0x0707AC31, 0x0B7FE526, 0xB6243F18, 0x3A78E8D1, 0x75C76A63, 0x54703CBA},
    {0xB583A5AC, 0xF9E68596, 0xF3D9AC16, 0x13550FCA, 0x37D358BA, 0xC42C75D5, 0xAE169470, 0x79932F36, 0x8FC43E6B, 0xBE81991C, 0x95F1DACC, 0x6F19FDE8, 0xE99786CF, 0xDAB865F3, 0xA8D91C30, 0x8CE7BE1D, 0xE1F2C82A, 0xBBCFEBAE, 0x07973B0A, 0xE34EAECC, 0xAF1DF6EE, 0xFF024AD9, 0xE3400A6F, 0x3157B46A, 0xE307DABD, 0x8E63675C, 0x8459D697, 0x4898D46B, 0xCA9B210F, 0xC2C6D9CD, 0x328B8F0A, 0x7E77BDA5, 0x97E1E91E, 0xB8DB1D24, 0x16AC4269, 0x342536AC, 0x12CF47A1},
    {0x40C66BD8, 0xBE8FF639, 0xFE7F6B95, 0x8A231FF1, 0x561DCC90, 0xD39BDCD5, 0xC235D991, 0x9D53A4F8, 0x639B4A87, 0x51715067, 0xC0DD5B98, 0xFDA35981, 0x9FF34AB7, 0x4D2E46F7, 0x3277FC26, 0x8E10F50F, 0x07C1E535, 0xB6D42619, 0x89EA428D, 0x165B3FB0, 0x08258FD9, 0x564091F3, 0xD5716D0C, 0xC2EDEE98, 0x0A45317C, 0x2B9E87FE, 0xFF78D44E, 0x707E514A, 0xC50B0F88, 0x78BFFA56, 0xAF319824, 0x3F31041F, 0xBAC8440C, 0x0EF757EC, 0xBADDDB63, 0xC73500BF, 0x5896D57D},
    {0x2110F0F5, 0x33A4F2A1, 0x64C1B419, 0x5EA76769, 0xF97839E1, 0x0B94DD7D, 0x6D6EF85D, 0x4FA87BC5, 0xE48C8C13, 0x3D10C850, 0xFBE56279, 0x5C257F3E, 0xAC352BBA, 0xEB10CB53, 0x4169D04F, 0x96C0B9F5, 0x3280CEA8, 0xC4E3406B, 0x91C6D2AC, 0x4E3DAF19, 0x09DD09C8, 0xF66DDBDD, 0x9C49168E, 0xFE262E86, 0x7C842891, 0x3FBF2C3C, 0x8F92D062, 0x4169BB8C, 0xF9C9666A, 0x11489C25, 0x8F09126E, 0xA7955217, 0x558DC2ED, 0xCCD929F4, 0x478162FD, 0x6DD458C4, 0x06E46F9A},
    {0x267E907B, 0xB967E2C6, 0xD9361CC5, 0x4EDB8F7E, 0x717904BA, 0x01D6F356, 0xEDA02D48, 0x492D19D2, 0x0EA61A19, 0xF836FC98, 0x8FF9E574, 0x7227D0AE, 0x10EC1A8C, 0xE700882E, 0x2F485D1E, 0x2D2E4849, 0x8E0566BB, 0x00CDC5B8, 0x7445A207, 0x2EC1ABEF, 0xE7B43E6D, 0x5BF7CFD2, 0x3DB5659C, 0xA91763E7, 0x99A8D5C3, 0x29608C5D, 0x006E9103, 0x1E159167, 0xCAC176E2, 0xAE79217D, 0x5A65E39C, 0x8D07C419, 0xE2DE71FD, 0x71F59736, 0x52032BE2, 0xE7BB97D4, 0x14809E74},
    {0x45070BDD, 0x844E4E13, 0x47F0B0C8, 0x07DAD52E, 0xC04DF7A7, 0xF083846F, 0xF6C14CE1, 0xF906A0D0, 0x12581B92, 0x3710AD53, 0x2C31BA4E, 0xF7EF066C, 0xB57192AA, 0xBA9D9333, 0xB0A3C8F9, 0x67DC87BF, 0x431FDFED, 0xA4A8968E, 0x9BBFC898, 0x13439F79, 0x48CF29EF, 0x2DA04139, 0x0DA3BEBA, 0x9208ED36, 0xB06B15A5, 0xA897D064, 0x85199F32, 0x942E28DA, 0x099DFD2C, 0x9556E777, 0xCF1A869C, 0xE4A5D7D0, 0xBF5F919A, 0xF8794304, 0x0A81436C, 0x9446C6AD, 0xBBB67CA0},
    {0x3F02A4DC, 0x77F398D5, 0x5CA9E0D5, 0xA8DD3CC9, 0x41FC2887, 0xA7790ECA, 0x584C8248, 0xD7168B5E, 0x62F17C6A, 0x16293D55, 0x98D8FBE0, 0xF6A8A2BB, 0x8673181D, 0x55A40A87, 0x496D9CAC, 0xEE11ABC9, 0xB0D77707, 0x52671EFE, 0xB9F87AD8, 0x834D880D, 0xB06A3691, 0x585169AC, 0x2BB806B4, 0x414E1780, 0x1C2916B4, 0x568E55F7, 0x3BDA42C0, 0xCE92E0A5, 0x480C7D0E, 0xB6B67EAC, 0x690AAC22, 0xC9130908, 0xB104BEA0, 0xAE012092, 0xF698785B, 0x02148E32, 0xB5BF4EBB},
    {0x2FE5061A, 0xAAD70FC2, 0x13633F89, 0x94F4B949, 0x4F752D53, 0x0574A2E7, 0xF589AE92, 0xEB6D3F4A, 0x28751B82, 0x87969FA5, 0xDD927192, 0x070DDE93, 0x74025481, 0x6634D33A, 0x8503B221, 0x6D9B8E56, 0x4E74FC3C, 0x1F822737, 0x8908259A, 0xAE01B7D8, 0x4141CC94, 0xBFDC05D0, 0x68336B52, 0x4506D6E5, 0xB9133F0F, 0xBEFB8AF2, 0x833F8FAA, 0xC16EA47F, 0xCDE1DA7B, 0x027D8438, 0x192010D2, 0x2EE7E4E9, 0x75CFE418, 0x09EB064D, 0xB6594544, 0x00487A75, 0xF51D6C91},
    {0x837F9C78, 0x062E13F7, 0xB74216E8, 0x5A71DAE8, 0x059AA690, 0xE557038A, 0x50158372, 0xE674CDCF, 0x6334BFC8, 0x6369092B, 0xC0587620, 0xBD0C9D19, 0x1E13F6CC, 0x2F01FB91, 0x59BCC185, 0x860FD042, 0xC1ABFE73, 0x62C29FD3, 0x7CA184D4, 0x7ADC8927, 0xC753D37A, 0xCEEA2CCD, 0x73C631F0, 0x60F65443, 0x49337EDC, 0x8E046764, 0x188B7066, 0x44E41D51, 0x72C3BD05, 0xA36FBE26, 0x8391AF36, 0x7E761A1B, 0x2292C2B4, 0x285B495F, 0xBE5908A4, 0xDD278719, 0x29CA9F70},
    {0xF01C567B, 0xF3A54001, 0x701A3DA8, 0x0F5FBDC1, 0x45FCD76F, 0x850536ED, 0x751077C3, 0x3C7D1E57, 0xB3E3BB4E, 0x1ED414BD, 0x0CF9B9B6, 0xDD02DECF, 0x088B7A5A, 0xD2F56540, 0xF7F32438, 0x08CEB7A9, 0xE5EBAD56, 0x6209BCBA, 0x638EB6FC, 0x3A810C39, 0x7B6B9DC1, 0x2D6C87D5, 0x7B149D5D, 0x6703DFE6, 0xF61C1D07, 0x1C3E5997, 0x279DCDD8, 0xB9E78DC3, 0x60B617E3, 0x333C7AA1, 0xA99524C1, 0xB2DE73C5, 0xD5E11F42, 0x33A7816B, 0x4CF186A1, 0xC3C722D6, 0x6D9550C5},
    {0xB9AEF79B, 0x0155452B, 0x9DC7CC8F, 0x9057B700, 0x80528B4F, 0xD731656D, 0x72DEA06B, 0xB0FCB7DF, 0x5DC56826, 0xC772296C, 0x90FE3425, 0xE8DD2C4C, 0x08CAB581, 0x4E149D2D, 0xDF1D0EEF, 0xC8DA6AB6, 0x8239452D, 0xBB520C08, 0xD82659B6, 0x78ED101B, 0xBA07698F, 0xBE2801A0, 0xB27909E1, 0xA7B07FF6, 0xAB936A94, 0x14C6B239, 0x8DA6DC13, 0xF552E247, 0x4D0B4359, 0xBBC9BDEE, 0x3F45D604, 0xDDE7D292, 0xC1EA6692, 0x48C0C381, 0x2E711C2B, 0x64235F26, 0x2FCDC4F8},
    {0xF51B4F70, 0x157A8BEA, 0xC208EFDA, 0xD2867C25, 0xD3B08736, 0x001F9793, 0xD6070C0A, 0x0F6CA79B, 0xD866A587, 0x256F8506, 0x10847B01, 0x1D439474, 0xE2FEF733, 0xFF38FA7C, 0x41548195, 0xCC2F1864, 0x3E84CB66, 0x912FB755, 0x0C665371, 0x809FC42F, 0x5FD6142C, 0x1C0891A2, 0xC30C2315, 0xB4C6E990, 0xE3716E0E, 0xC4043AFC, 0x5FF81C19, 0xDACCFF0F, 0xFA83E1C8, 0x92038D4C, 0x58572776, 0xE0EB272C, 0xE1FE1B01, 0x4AE38BA7, 0x2E7B876F, 0x4D55BD25, 0x91EAC958},
    {0xE04F8D6C, 0x3646750A, 0x9D52E13E, 0xB45CDDB2, 0xDAB38DC0, 0x42387533, 0x111E2570, 0xC70441C3, 0xE3F094F8, 0x65756EAA, 0x91295044, 0xD94D666D, 0xF1427897, 0x2E445D42, 0x43E60447, 0x4DC27224, 0x6AB31C8F, 0xDCB4B7CD, 0xCF1A7CA4, 0x559C680D, 0xB4C402FD, 0xE086084A, 0x4B2A4BBE, 0x2CE47B68, 0x7C28FA36, 0x010D0B0B, 0xEA20ACC9, 0xF2ED977F, 0x6028D172, 0xFC728ACE, 0x85554585, 0x47DBF0B8, 0x4F89022C, 0xB0F9B62F, 0xEE9781D0, 0xCB552CCE, 0xE032C249},
    {0x79CD16F9, 0x854A611E, 0xD8250A5E, 0xC1E5A5A5, 0x7888D2B9, 0x7936796A, 0x673B5932, 0xE04D879D, 0x1951ECFD, 0xCBAEEB87, 0xED31B528, 0xE4B380D1, 0xF81E74E7, 0x2FD09F9C, 0x183C1575, 0xC0C5D480, 0x8C2F01C8, 0x969A8AF3, 0x402EAD48, 0x39424AAF, 0x033A8C84, 0x22AEE979, 0x72617515, 0xD8904503, 0x61AAA1CB, 0xEC1A9CE5, 0xC6A94A75, 0x40B32BF1, 0x3AC3AD64, 0x89E43378, 0xF4640E1E, 0x5DE8BEE9, 0x82E24FB8, 0xFD08C6EE, 0x00238A50, 0x73421DF4, 0x46AE8A12},
    {0x3E69A909, 0xB7A7DDE6, 0x38F7FD98, 0xDC399BD7, 0x81AF6BE5, 0xA21A5C34, 0x634C7303, 0xDC1A31AF, 0xB1C7F889, 0xA2BF2F43, 0x621A60A9, 0x3D30F0BF, 0xA0126B24, 0x9714F7FE, 0xBD459359, 0x275FFA75, 0xCE20C735, 0x5C16603E, 0x9D212EFD, 0x0AC6399D, 0x4C960FDA, 0x7BE606C2, 0x5973D334, 0x397EDA92, 0xD973B184, 0x386EDF7D, 0x0B40DB6B, 0xAA83D02C, 0x5FEDB174, 0x113133A7, 0x7B43A963, 0x241A2B2E, 0xAB4E5B4E, 0x8158DFEB, 0x8F24BE12, 0xE59897DD, 0xEF93CBA3},
    {0x53E48B71, 0x00DD1DCB, 0xE0969EFB, 0x25301CA9, 0x0A533CBA, 0x7E588A09, 0xEB316AA9, 0x0B52A036, 0xBAC2F5AE, 0xECDA52F1, 0x6673FF9B, 0xE039E95A, 0xC643DDF9, 0x5D8538E4, 0xCBF865D6, 0x3551DE84, 0x7A51E6E5, 0xBCBCF18B, 0x015B0E76, 0x94807C5E, 0xEFDBE9BC, 0xE60E7178, 0x9AE40E7E, 0x1E84BC10, 0x6718E7D7, 0x0E136F72, 0xFCC77B44, 0x2E4BF433, 0x7D292864, 0x776B9D41, 0xACAD2F19, 0xD1389A47, 0x8E691F1B, 0x61FE50BF, 0xE61B46B1, 0xD43231E8, 0x4695F372},
    {0xC28C1A75, 0xCD9E6AFA, 0xA36EDF5F, 0x614A0367, 0xEFE34CDC, 0x76755F32, 0x0CFF295F, 0x37B56B6A, 0xD0EF22A3, 0xF8DF2171, 0x010C556C, 0xB06B0F76, 0x9A7685CF, 0x5F5534F3, 0x4AAA0487, 0xFCFE632C, 0xCCC3E31E, 0xC5324E1A, 0x7566ECD5, 0x822FE12E, 0x19D53FFD, 0x55B5D8CC, 0x0EEA8855, 0x12EF55C1, 0xD9E20705, 0x4FC83381, 0x50F61379, 0xDAE2A7C6, 0xC4945EFB, 0x9DDB7E9F, 0xD88D9367, 0x7B7905FF, 0x10CE298A, 0x7A549BCA, 0xB7F86D0D, 0xB4822508, 0xF8CB24B7},
    {0x9E1ADC78, 0x2884E372, 0x54F85D27, 0x3A4380CF, 0x50B2BD2B, 0xF8C017B1, 0x098D6FFF, 0x5C4E3226, 0x5E4C3922, 0x557CD2FE, 0x490D854C, 0x61C9B0E6, 0x29920280, 0x0A08C7D3, 0x547CD385, 0xAB00EA1E, 0xF5DFDBAC, 0xCC09B215, 0xF9E5597D, 0xB666DA00, 0xE2355B6E, 0xE4F3D35B, 0xDE864228, 0xE0535CDE, 0x123CA186, 0xCA77365E, 0x8279A275, 0xDE365D4B, 0xE7C36736, 0x945F6C27, 0xE4D52B1D, 0x42070322, 0xF1FA9C05, 0x39BBA440, 0x2D66CFD1, 0x0D23B5F9, 0xF47D10B6},
    {0x8D89BA53, 0x85EA4347, 0xBC5F27A6, 0x0BBB1463, 0x53DD10A0, 0x2AB26708, 0x140F828A, 0x227AD6AF, 0x76D2644F, 0x9E513D54, 0x32E1851D, 0x06817CDD, 0x8664A2A7, 0x6ADDE963, 0xA27A583D, 0xD5E73FD6, 0x812BE542, 0xB3A4E592, 0x32211DA3, 0xC53FF8BA, 0x62C7D52D, 0x25D8443E, 0x3B16295B, 0x019CA2EF, 0xF4D46117, 0xCB5F1143, 0x4E577405, 0xE727E302, 0x5A716F40, 0x5F02D61E, 0x705E66CE, 0x98448DAA, 0x0208E262, 0x941462E3, 0x93699848, 0x100A5D91, 0xC0CF7392},
    {0x98E51D76, 0xBB268773, 0x31D68459, 0xD27B72B7, 0xDE3544AE, 0x9CB29868, 0xFB359269, 0xD3F441E2, 0x4FE14A59, 0x9CBF4EBE, 0xF7E8F1F8, 0x3072ECDF, 0xBFAF4935, 0x23D3589F, 0x2D52C10C, 0xE8823826, 0xAB9B10DE, 0xF69E7D60, 0x0A494C59, 0x91CEFAB5, 0xAD11FA07, 0xA39AEF0C, 0xEDB487ED, 0x18056499, 0xA643FE4F, 0x213F7AAA, 0x0BC5AA76, 0xC046D051, 0x2933FE79, 0xD8A25D93, 0x00AD6C0D, 0xCDE824FD, 0x94E1DB3E, 0xF477DB15, 0xE3A4FBB4, 0xF5CD5438, 0xA43A7F95},
    {0x80D8E585, 0xA2242B63, 0x5747A219, 0x58FC1857, 0x9D7701CC, 0x6AABC4AB, 0x02B0FD42, 0x7FF2341A, 0x7464899F, 0xB91A9E41, 0xC6165D05, 0xC3B9D4F8, 0x05A2D7CB, 0x63536E5F, 0xBB015C1B, 0xC91450D4, 0xE1215460, 0x06F74ACA, 0xE740A7F7, 0x0C2CD7BE, 0xEAE93A68, 0x7029902A, 0x94479C20, 0xD6C0E13F, 0xFF2B0C42, 0xB077C7BB, 0xDBF303EC, 0xB263FA3A, 0x09D95D6C, 0x4E892551, 0xD73BE643, 0x3CFA7262, 0x96C3B9E7, 0x75259973, 0x6DEC3BED, 0xA340192F, 0x768D28A6},
    {0xC61E7CCD, 0x83A2B467, 0xF00F7A30, 0x1DAB2EE9, 0x248EE246, 0x1D8A0E2C, 0x17881FA7, 0x302F4B5E, 0xE771BB2E, 0x59A5F091, 0x6D029A2D, 0x7C211FFF, 0x95E49092, 0xDA2B3CCE, 0x2D7C60F1, 0xC4B0D77D, 0x58164182, 0x494031F6, 0x65B53032, 0x62AB21E6, 0x86D1EAE5, 0x8C56ED70, 0xEEB19411, 0x2D923D73, 0x448C9DA4, 0x2E5235B6, 0x445EDDFB, 0x55E33BE5, 0x48A2CE92, 0x2B64DA27, 0x171DEB88, 0x648D58EF, 0x9D5509BB, 0x6C28571A, 0x7C3D044A, 0xC88FB99F, 0xDBCDFDB7},
    {0x8530E29B, 0x51C36566, 0x8F8EEBFC, 0xC920AB9E, 0x09F3C5F4, 0xD2B55A95, 0x6DC6F9FC, 0x33284F8E, 0x7941FD5C, 0x5B311964, 0x26BC851D, 0xB2DA0FE7, 0x3F53143A, 0x14269D46, 0xEBE73C6E, 0x0B3E5F89, 0xE90B9DFB, 0xFD201343, 0x5DFE2231, 0x31F44B04, 0x974739B5, 0xE64072A8, 0x9942400C, 0xE4426ED3, 0xF0C5EFC5, 0x5E827273, 0xDDCF2B1A, 0x12830ACC, 0xBBAA4B50, 0xAD5FF7ED, 0x2E1CBCD8, 0xDEE4DA5A, 0x98F54CAA, 0xAE01D6E6, 0xCBD3CBEE, 0xF79DF834, 0x661340F4},
    {0x69A6D127, 0x2AFCFE62, 0xF58EFD93, 0x56B14DDB, 0xA86CC22C, 0x6B563AE1, 0x5302944D, 0x33B4BE65, 0x8D9835DA, 0x48E8D973, 0xBC313962, 0xF06057C0, 0x643E71DD, 0x59EB908D, 0x64F123DD, 0xC0CAA67F, 0x16FC58D3, 0x4A30D556, 0xA34997E6, 0x092348A6, 0x6EEB1B6E, 0xFBDF96E6, 0xC24B4D9A, 0x42DA09B9, 0xE899C1F2, 0xB20D988B, 0x169AEB29, 0x15EB0E98, 0x779EA79B, 0xD4ADCB1B, 0x03CB9537, 0x6AE929C2, 0x0D89F9C0, 0x1B85B948, 0x4543334F, 0x6FF8E1EE, 0xB8330188},
    {0x7427052A, 0x79F760EF, 0x54CB3D37, 0x27DACA7A, 0xB5DC2ADB, 0xFE4EB5B5, 0x32A4081D, 0x7B39DAFD, 0x30CB4C7A, 0xDB036A8C, 0x0E765D34, 0x053D3422, 0x9F0B8404, 0x73B8D39B, 0x8EF77E0E, 0xB5F145EC, 0xE00CA0CE, 0xF7F1120C, 0x45AFA4FE, 0x872E6545, 0xABF8D9C5, 0x4C8A253C, 0x933FC365, 0x85DF7694, 0xE372AD8F, 0x218E80AC, 0x845E56BA, 0x5634716F, 0x5C7AE834, 0x05CC8BA4, 0x4F1E491F, 0x0B742F6B, 0xCAA48663, 0xF7185A6B, 0xCB4A975C, 0x2D323250, 0x63315415},
    {0x572B975B, 0x16F47525, 0xA0470A2A, 0x5CEBF471, 0x1055873B, 0x389995C9, 0x592EE10B, 0xBFC4A4B4, 0x673EDB8A, 0xD6890A67, 0xA33771C3, 0x9FB6D0B8, 0x1FA70E5E, 0x0F26EEFB, 0x1A641DED, 0x32D96D70, 0x1FCE400A, 0x41AA7952, 0x7EFD4EA5, 0xBB6F1FF3, 0xFD773644, 0xDFDC53E3, 0x8A00327B, 0x157D3AF8, 0x07DADA40, 0x4147D482, 0xD7D36E6C, 0xA63110AD, 0xA839CE60, 0x5110C355, 0x5E7A42E0, 0x6C324D06, 0x9135E247, 0x3AE10B3A, 0x85DEBFCE, 0x9D7AFDBC, 0x4FCB4661},
    {0x079B1127, 0x4D24C131, 0x8AF6FF6E, 0xF3B29BF5, 0x9A44A289, 0xCE1F8DCC, 0x0E0CD025, 0x35646A36, 0xCE8583DD, 0x3AA163E2, 0xE9C3D218, 0xD4A3216F, 0x603A5E0C, 0x6E66CE40, 0xCE006D79, 0xE25EB57C, 0xE7F62AE0, 0x0D5CD5B8, 0x46E72D87, 0x16B903AD, 0xEF215AED, 0x86D77253, 0x31716973, 0x4B015FD2, 0x6B1BBCEF, 0xFAA745C3, 0x29572222, 0x149ADB60, 0x06DC7868, 0x1D7A1BE4, 0x42F71B55, 0xDAA5303A, 0x2AB35EAE, 0xDD578B3C, 0x2EDA300E, 0x8A809E9A, 0xB12809BF},
    {0x1F94523F, 0x40EE4196, 0xCC1FEF2A, 0x6DAD0FE9, 0xEC8F2F9C, 0xE1F47258, 0x8234024C, 0x8494AF00, 0x5F06BD30, 0xC966A83F, 0x5F965E14, 0xC72B5B01, 0x89AE8118, 0x43515D9C, 0x7C0D7C69, 0xEC71E0A0, 0xB204284D, 0xF2437DD3, 0xF9021504, 0xA8D29CD7, 0x5D33FC9D, 0x310761EF, 0x70149CF7, 0x30CDB896, 0x9074CB73, 0x9FD9793E, 0x86291B7C, 0x0A9D67E5, 0x6C6C6410, 0x1B63143C, 0xD8EEA9F2, 0xB7BF1074, 0x21D4040B, 0x331BB797, 0x8AEA22B9, 0x399A0041, 0xC5253CF4},
    {0x775F9E49, 0x610C2997, 0xD89F5D87, 0x12D60B08, 0xA2B77D05, 0x595D63ED, 0x97232D62, 0x14387C50, 0x06D0AC1E, 0x6CFCB913, 0x553A9CAC, 0x2B5A5C53, 0x5976BB27, 0x6420F3C2, 0x75A2AF6E, 0xD60DD93A, 0xEF2D6AB0, 0xFF6D5F5B, 0x0614E0AE, 0x12C1945F, 0x10EA4EB8, 0x6BA6491A, 0x9FFBB2CC, 0x0B68E70A, 0xAA22DE93, 0x6EF6CFBD, 0xD6BF400C, 0x673EDFE9, 0xCE5FE3FE, 0x40013F16, 0x95BE70F0, 0xD9783744, 0x95BDDEF4, 0x7BE7E522, 0xAEA8F7F3, 0x1A29E609, 0x223F06EC},
    {0xB6F396DE, 0x7ADE9C5D, 0xDAEAC722, 0x9DBE635E, 0x8932C951, 0xCBCB26A5, 0xC311628E, 0xC16CF204, 0xACD48D8F, 0xF638CEDD, 0xAD1A7BC7, 0x7D002FA0, 0x19325A49, 0x47901639, 0xEA4ACD84, 0xA6638DCA, 0xB337B7FE, 0x9D31DB37, 0xE59D6E72, 0x146BE4D3, 0xD94372C5, 0xD4E34170, 0xD3DB2215, 0x3F7E49B3, 0xB2888CB4, 0x9E9BD7E5, 0xE33BD59C, 0x9A51C244, 0x1A6FCF42, 0xBA05E71E, 0x665EA467, 0x380D3A7E, 0xB2DCFD9A, 0x66834EA0, 0x0F65CB5B, 0x60D749A0, 0x12B1C14A},
    {0x69246CC6, 0x1421E3C3, 0x4F4AD5CB, 0x237C04A8, 0xF9F0D0DC, 0x030797AD, 0xAE0F3148, 0x1AE7F307, 0x297C1CDC, 0xBE4ECB0C, 0x43874E3B, 0x77E0F13C, 0x63BF1C43, 0x41994798, 0x7975FC4A, 0x798ADB3C, 0x01D4AF7F, 0xD5DA05A8, 0x0F0A924F, 0xB30263CF, 0x5CDCB9E5, 0xC5DF7D65, 0x6F44001E, 0x6C8D1A6E, 0x384016E1, 0x276C3EC2, 0xEE90E8D1, 0xF0A5AB45, 0xA994E23C, 0x6F72991D, 0xAB4A8723, 0x79B456F8, 0xF3235BD3, 0xF743DFCA, 0x53C641E1, 0x7ADA4240, 0xF72177BF},
    {0x7451D784, 0x18D318D5, 0x87A1DF65, 0xF87EEE6B, 0x399FF741, 0xA24FEBED, 0x610AE096, 0xA58D1070, 0x9742FCEA, 0x539E62DC, 0x90528258, 0x63B5907D, 0x13FD9BB7, 0x25E19480, 0x0D959C47, 0xCCF4050E, 0xD211B9A6, 0x022B5FB3, 0x983B358B, 0xB8CD1D91, 0x309C7F5A, 0xA453EC62, 0xDDC8E6B3, 0x11BA247C, 0xA6C741EB, 0x5F579501, 0x725C634A, 0x2AF323F9, 0x7A3D090E, 0xA03B28CE, 0xEFF9370C, 0xBABDC3DA, 0x221FE410, 0xB524D45C, 0x05B9B0D9, 0x9ED276F2, 0x99CB125B},
    {0x9184D5B1, 0xAE6990B3, 0x09FE6157, 0x6538AC90, 0x3466E51C, 0x411991ED, 0xF4ECB478, 0x754DC5B9, 0x96E63734, 0xADA6EAED, 0x5A333CDE, 0xDC8CEBDF, 0xEA4AE3B4, 0x6B034956, 0xBB1CFB44, 0x7BA35112, 0x4123592F, 0x51233293, 0xDBC20C36, 0xB54D6F5C, 0xB2776A57, 0x31BB48DC, 0xE7A57D46, 0xD1467D89, 0xAD5244BD, 0xE8FA9AD9, 0xFA6AA917, 0x4564E9BC, 0x0DD02FD2, 0x3920DC1B, 0x62D3423E, 0xD846080C, 0x626007BF, 0x54FC6558, 0x973D504E, 0x1999C0B5, 0x985A8EB0},
    {0xF437F9CD, 0xDF20F179, 0x05453F2D, 0xE86FB803, 0x2A8CD496, 0x5290BFFB, 0xE4993D5C, 0xB398DFB7, 0xE25E4456, 0x6D3D0A29, 0x3AD2DDA7, 0x6197079D, 0x37F35C9A, 0x02DF13F5, 0x5D2A054F, 0xD908948F, 0x4D1B49EF, 0xD96856EC, 0x2F38E1C7, 0x2847CA82, 0x10475AEF, 0xB36B3FD7, 0xBE0BAA2C, 0x7D4611D3, 0xDA76E37C, 0xD5CA94D1, 0xFCEA9AAB, 0xB0763FC6, 0x7C230A0B, 0x0B8A20A0, 0x0F189C69, 0xA57CF84A, 0xA87C6B06, 0xADC43164, 0x742D5882, 0x03C62337, 0xC0AA0528},
    {0xCCAF1C0A, 0xD3E25622, 0xCD10D341, 0x15FA73C9, 0x1294AF47, 0xAC227E12, 0x59C33FDF, 0x83ACB129, 0x3BEB3407, 0x89379EF9, 0xAD56EC26, 0x4230AB6A, 0x35B36957, 0x0320C9DA, 0x77C7BE15, 0x90085500, 0x432A38C8, 0x960563D4, 0xA0744D3D, 0x4984A7E5, 0xF31AD010, 0x0108789C, 0x8F6B3E08, 0x7BEA35B7, 0xAC369B64, 0xDD1E5B26, 0xAC357C7E, 0x2E92F104, 0xF9BC2499, 0x2E00F57E, 0xB7DF5AAD, 0xBEE86455, 0x45BE7F96, 0x9FF01B1B, 0xF5BEEFF3, 0x17952278, 0xEF1C3893},
    {0xA131BE6B, 0x04D37455, 0xED3D8FDF, 0x4CB17D2B, 0xC218B582, 0xD29E0E8D, 0x2E786AE5, 0xF36FBD70, 0x384154F1, 0x93D5D548, 0xFB6E1DDF, 0x0E0F5184, 0x4848F1DE, 0x7A9346FB, 0xC0F3B940, 0x8109275A, 0x2E97E643, 0xAAF0EAA8, 0x872821B8, 0xE102D09B, 0xB19733F8, 0x702F4DA2, 0x98BCA0AA, 0x81DC2392, 0x3FE91C23, 0x24C4EB3D, 0xF2C6B4CF, 0x3510EA6B, 0x1847174E, 0x67C9EA78, 0xB1046EA7, 0x03ACB007, 0xD811D8C6, 0xCF3E0C2A, 0xCA280A4E, 0x24995C8D, 0x855CBD89},
    {0x90C8BDB6, 0xA34A1094, 0x11D43443, 0xDA192944, 0x76FF83A2, 0xBA0D9485, 0xD5801790, 0x30FC2D2B, 0x4035D04E, 0x7B0219B7, 0xF89036FA, 0x7C317FCF, 0xAB22C4BF, 0xFEED3EE1, 0x512AAAD8, 0x526B3E8C, 0x5A20E80D, 0x3B4314C3, 0x34C1C2B3, 0x841FEB6A, 0x65BA0E16, 0x2F3DBEE8, 0xD49E3699, 0xCEEBA4C3, 0x51B77A62, 0x748E420F, 0x96C7F281, 0x60600CA9, 0x232BBF22, 0x7654B7C5, 0x11F45BC5, 0x99F89998, 0xACCFA29F, 0x36D9D53B, 0xF8BB2738, 0xBE6404DB, 0x0DCBE4E3},
    {0xB940E21C, 0x7C02ACC8, 0xF522292A, 0xB2E96260, 0x5691731A, 0x89FDAB2C, 0xDB8DD0D9, 0x7A29518E, 0x02D9E550, 0x5D95E2C5, 0x91FB6225, 0x8CD1E5C8, 0xB7BCCDD9, 0x917E8CA4, 0x6FDF062E, 0x29A8AC07, 0xF81C28E1, 0xF3BDB95C, 0x6B4E329B, 0xDBC18A31, 0x6E6A7F3C, 0x47C079CB, 0xAE72373D, 0x4920D95C, 0x5626ACFB, 0xAD52C44A, 0xBEE4DAD9, 0xC2EA88DA, 0xFD4CFF9B, 0xEAF4CB98, 0x8E0C0F3C, 0x721E5EA7, 0x40485D4F, 0xE2A4625E, 0xCC0D0437, 0xC5B23A7B, 0xD7B6012F},
    {0x3992B53D, 0xC7289CDA, 0x2B251944, 0x90B4B90A, 0xBB2D6B04, 0xD2DEE748, 0xFAD9D19A, 0x8CB129DA, 0x5A89E453, 0xC9272659, 0x92241CF8, 0xA60563FE, 0xF3B31E62, 0x61E37986, 0x5774AF4A, 0xB456FC85, 0xB2010731, 0x8B257545, 0x18E9DA09, 0x33EC0A71, 0x6495AA09, 0x19D0DD24, 0x223347ED, 0x4F7418EF, 0xB2FE109A, 0xEFE2CC4A, 0x46635A90, 0xB1E6E4B9, 0x2C0B0CD7, 0x2CDC00D7, 0xCF0325BC, 0x8D8EA776, 0xD5CF7D13, 0xF73D0C53, 0x08356C1A, 0xB666927E, 0xA1A2D320},
    {0xF1676F1D, 0x0384B46D, 0x039EF62A, 0x20A56EFC, 0x08CA70B7, 0x3DC3F9C0, 0xB71DF1F0, 0x2CBE2F34, 0xABD5C8C4, 0xF51C86F3, 0x0C0B3648, 0x6484608F, 0x529DAFA3, 0xA325C293, 0xADABC4DA, 0x447E1CB8, 0x599F87C0, 0xA296C9C4, 0xDC71CDAA, 0xA3A7E1C0, 0xAC307A3A, 0x0472662A, 0x269E60CE, 0xFB062D02, 0xD3891B9A, 0xEE01C9AE, 0x38C638F6, 0xD48A8D9A, 0x81EDB4D6, 0x66D0728A, 0x799286AD, 0x96A4AAAF, 0xE6087B80, 0xDF9A5851, 0xB2F519CC, 0xCFBC56CC, 0xD98C07FE},
    {0xF2110AC7, 0xE674EF39, 0x11E05ED1, 0x18E0A6F7, 0x84D48750, 0x7250A73F, 0x9799A488, 0x5CD2D94B, 0xC9541CAC, 0xD8B7F874, 0xED66BA86, 0x93B8EBA6, 0x9216BC10, 0x18F62CC0, 0x4769B4C7, 0x8D864A52, 0x91A5B304, 0xC8E79F3D, 0x8A11769E, 0xF94B27CA, 0x6A00A478, 0xEFDF65D7, 0x5031677B, 0x458684CE, 0xB1CA0C1E, 0x6411A339, 0xBF2B831E, 0x3F7B1EDC, 0xF74FE18A, 0xA34D53EA, 0xBB261E3E, 0x6C772D66, 0x5C60F986, 0x62202131, 0x260E1D22, 0x03DA2CF1, 0x668666C8},
    {0x7EF6DD43, 0x715F6B57, 0x3371014C, 0xBA3237EB, 0xE10CC0B0, 0x0CC59E77, 0xA5B3EF0F, 0xE2D01757, 0xD8245C53, 0xFEFD7C94, 0xE3B3C81A, 0x77E44729, 0xF0ABE12C, 0xE03F3D72, 0xDCD8B489, 0x0438FA90, 0x12F7298D, 0x43ED88A5, 0x3942EF5F, 0xED4913A3, 0x0FCE1B14, 0x7FA9AD22, 0xB0DE3238, 0x303C6A0C, 0x8FD9A158, 0xC62BFF7D, 0x065E4EBE, 0x404D7302, 0x91F9B4F6, 0x2B7EBDF3, 0x318B5922, 0xCFC788FA, 0x185344EA, 0xFE19F20F, 0xAC1CF775, 0xE7C62117, 0x9D6604BF},
    {0x9CBE1B12, 0xE6294FFD, 0xBB3E8C70, 0x48C654AE, 0x5C0977B1, 0xF92CA674, 0x626113F5, 0x4B2149C7, 0xE5B10CDD, 0x3FDE0C76, 0x8897724F, 0x321F032C, 0x75BB5617, 0x414735FC, 0xD2BAF17B, 0x25727DB2, 0x95D05717, 0xCCFD35C0, 0xD15DCA48, 0x8719EDF7, 0x4BE7B2D3, 0x9D08E019, 0x36707AB7, 0x70EB2AA5, 0x9CE6F85E, 0x965E540A, 0x74E7BE83, 0x45904731, 0x51F86D86, 0xCB7E6F2C, 0x5F0967EF, 0x7D150DBC, 0x04DB2A81, 0xD5660FF7, 0x2B65C6B1, 0x2079D22F, 0x8081B43F},
    {0x3BCB1D9D, 0x9B41B4BA, 0xE264204A, 0xBA8EC0F2, 0x9E29734E, 0x8EA21EFE, 0xAFE8723E, 0xC89E8189, 0xA14FC1AB, 0xAAD1CD2F, 0xE40793C2, 0xA8044BD4, 0xF227DD86, 0xA3A7E772, 0x2397F4AE, 0xCC72647A, 0x60436135, 0x075CB43C, 0x28B25CDA, 0xD034C5CE, 0xBD665B60, 0xE7717A62, 0x7F8D16A2, 0xEB68A53F, 0x1A86148D, 0xCFAC4021, 0x8B65220C, 0xF6C7BC7B, 0x1C316E8D, 0xDA0493E7, 0x57628938, 0x7CA0B3CF, 0x36A7DFC2, 0xEDF1B1C6, 0x7E660268, 0x8A133BFF, 0x7D2BA431},
    {0xAD495822, 0xF035F3D9, 0x7757AA36, 0xB1E214DB, 0xEE1D0E5F, 0x897CC5E9, 0xD269772F, 0xF985866F, 0xD6EC06F0, 0x64719CF7, 0x0ED33D98, 0xF031A97C, 0xBE79EEF1, 0x1A0DB802, 0x4F8D8158, 0xBB802376, 0x6E1E72C1, 0x91AEAC95, 0x92FD26D5, 0x6071D050, 0xA13BCBC9, 0x1E70CD33, 0xC2EBAE0D, 0x88FEE85E, 0x20526ECD, 0xDCC0C3EE, 0xAF184541, 0x5BB9CAB7, 0x2772EED0, 0x6495AAB2, 0xE670B18A, 0x8A9A9156, 0x61C26648, 0x7D574169, 0x343D2BBB, 0x957E9C54, 0x53C3836B},
    {0xFC111846, 0xBFD560EB, 0xEE9F8CD7, 0xEFEE20F5, 0xBB7584C9, 0x157E4079, 0x90DB0965, 0x47DB79B2, 0xA6EEB0E6, 0x696CF73C, 0xE7492B42, 0x69A3BBCB, 0x9A4AABCF, 0x07932D52, 0xF70DBDEC, 0xC1254B48, 0x14CBB77D, 0xAFBC3EB8, 0x24D6C7ED, 0xD9081975, 0xF3C1EC1F, 0xA0096AF5, 0x83713A88, 0xF5811E30, 0xB6A4C568, 0x0A59151C, 0x2AAF11E9, 0x9E2EBFD4, 0x7B276F0A, 0x17FED9CC, 0x04C69704, 0xF31CD56A, 0xBA0F3949, 0x3A1E061C, 0xBD598149, 0xBF76D75F, 0xA38E5926},
    {0x2A7A9254, 0xE2E5ABAF, 0x30E23420, 0x894A3DED, 0x4A4C4965, 0xE45FAEB3, 0xBD94747E, 0xA23447C4, 0xAFB7CDF4, 0xB856E7B4, 0x2B69428B, 0x72B446F8, 0xE4DCDA76, 0x5AA0ECE9, 0x36DC757C, 0x3A82C2DF, 0x5FCB6229, 0x6D5996ED, 0xB8368BE7, 0x697FE838, 0xD8BDEC25, 0x6E7EC630, 0x69AAC756, 0x0F727C23, 0x671252B9, 0x32CDA6F7, 0x9F2FF941, 0x1DA6498B, 0xC5783B45, 0xED207B82, 0x168E4265, 0xC9D62D31, 0x7BC9A0D5, 0x56EA875B, 0x9C8A4D66, 0xD5F00849, 0x3F38BA19},
    {0xD628591A, 0xBFF9AEED, 0x308F087F, 0x66D66E55, 0x666FA65F, 0x59FC30EF, 0xBFC2D69F, 0x0F7B136D, 0x5985189D, 0x2A8CD4D1, 0x5C8088BF, 0xD8F0525A, 0xAE0A8972, 0x78704DE4, 0xEEE46972, 0x19C3A8FF, 0x1727286C, 0x9C3AED51, 0x1079B161, 0x99614F04, 0x3EC31B57, 0x4354210C, 0x583544F9, 0x1CC8C5FA, 0xB704D274, 0x64147580, 0x5F3D9CA5, 0x21BD3590, 0x17D2536C, 0x77C72C05, 0x47D4E782, 0x512CE9BD, 0x539E4D3C, 0xB932E0DB, 0x1EEDEDF2, 0xC60FDEC5, 0xA141A4C1},
    {0x0EFD8B69, 0xBB7D1832, 0x40CE93EF, 0x0CAAF48D, 0x2B888E46, 0xA392042E, 0xF7802A3C, 0x8BDD0DBF, 0xC11BD12B, 0xC53C4C47, 0xDBDFAD37, 0x54B20285, 0x5D84EA3F, 0xBEA5F97A, 0x7449F045, 0x02DECA11, 0xB4EAFCB7, 0x48F75125, 0x33494243, 0x915997CE, 0xA4A3C86F, 0x87942D1C, 0x68814271, 0xF1B8DBC0, 0x37789DC1, 0xFB6D02D4, 0x3ECC20C8, 0x375F5D6C, 0xAC844756, 0xC622AF54, 0x4083A75F, 0xF60F10A2, 0xA0ECDD35, 0x7E915018, 0xCC0BA920, 0x596B0237, 0x7E208D1A},
    {0x3FD7CB00, 0x12FFCB64, 0x8A460E2E, 0x29E42524, 0x368396CE, 0x6489E25B, 0x8924B43D, 0x8796D41D, 0xA35D9186, 0xC7C36207, 0xB607F970, 0xC1FB9734, 0x1A5C4401, 0xAF24D106, 0xC34B8D9B, 0x1439CC3A, 0xDB482FC8, 0xF9E785E6, 0x5DADF604, 0xE05862B4, 0xC0A0AE34, 0xFB3C8990, 0x7765A1DF, 0x6F3D618E, 0x1D1149C1, 0xECC2A459, 0xB78A55F8, 0x2D6F3575, 0x02E2A241, 0x1B26E51E, 0xA1975E02, 0x2905E84C, 0x79022D04, 0xDA5C630E, 0xD0D5F648, 0xF239A92E, 0x97466814},
    {0xEE9D3520, 0xBAB69353, 0xA50A1A1F, 0x5ACEE1A8, 0xB050A106, 0x8266A03A, 0xBE0AE7C5, 0xEC73BAE9, 0x32804617, 0xA1FCFA6A, 0x8C67E7CC, 0xB80C39B6, 0x2DF53DF0, 0x82BC21AE, 0xFA0E8840, 0x40890F69, 0x57E65720, 0x58FEAE37, 0xC414C240, 0x88180834, 0xD37EA0FB, 0x314825E6, 0xD8A436B6, 0x4438B74E, 0x98E651FA, 0x372190C1, 0x081CFD49, 0xF45A1D56, 0x649902D0, 0x360296A9, 0x206D9A0F, 0x94646528, 0xA19D3701, 0x4E34A944, 0x91182CB0, 0x7809D5EB, 0x46A1B067},
    {0xB0FEF4FB, 0xB4513961, 0xBBCB86CB, 0x2E3BF8D0, 0x4E812154, 0x1F5DCE5D, 0x49DA4187, 0x07CA12A8, 0x303E7C7A, 0x37779F7C, 0x9794BA65, 0xF4A50052, 0x0A3B2B8E, 0xF6F69F5A, 0x349D25A6, 0x2F084B70, 0x596F4297, 0x435D2DA7, 0xC5812342, 0x4E0B6CAC, 0x78062509, 0xD0491DF1, 0x5A7C6E0D, 0x9FC45C7D, 0x10F6F9A6, 0x5A3CA7B8, 0xC60C5A41, 0x437AD272, 0xA00E2A35, 0xC91C8E2D, 0x79F48275, 0xDF434825, 0xE4766899, 0xDFCC9BC5, 0x4C64ADD5, 0xD6D946F7, 0x5483FDFB},
    {0x5DFCE4F4, 0xB3BC0E39, 0xD5686328, 0x88CC7C42, 0x1A90C59B, 0x79D42AB0, 0x56847A79, 0x002B3195, 0x382745FE, 0xC36F0857, 0xC91BE201, 0xD0B64C5F, 0xFFB8655B, 0xDF99494A, 0x61B9B984, 0x18E7DA96, 0x89E35296, 0xDFA6475F, 0xB74528D4, 0x32FF0EFC, 0xC0774F71, 0xED686E51, 0xB865CE5D, 0xDC34163B, 0x05F155EA, 0x9B1697F4, 0x03DD27C1, 0x57D66706, 0xA9B3245D, 0xDB20A949, 0x2AF9A49F, 0xAA21B848, 0xD9C87BAC, 0x3F6DEFEC, 0x508EB539, 0x5E9D979B, 0x31C7DBFB},
    {0xDC18222D, 0xFA7DE485, 0x4E6516FD, 0xB19C1246, 0x29951E62, 0xD42CA1AC, 0x0F21CF76, 0x2D040D67, 0x56EF3F4B, 0x6B101B07, 0xF98BFA0A, 0x88C94E5D, 0xB35DB34E, 0x9E8E3721, 0xFA3D959C, 0xAB98D7C1, 0xBC4AE2A8, 0x2A41B9BB, 0xFB514AB0, 0xA8F4F078, 0x55C39522, 0xECDB0BBE, 0x4498148A, 0x91E93750, 0x947F5AC2, 0x1496F490, 0x0E38AAB1, 0x9332B875, 0x84619046, 0xFE2FBE43, 0xD48F8657, 0xD0ED5155, 0x347D575D, 0x9979D411, 0x1B55758E, 0x4EC065E7, 0x0250E824},
    {0x4351A408, 0xABDAB10C, 0x31A4F70D, 0x73B7979C, 0xD60C55CD, 0xD2D681CD, 0xD321407E, 0x0A8BCA35, 0x5B2A3A67, 0xFBD568FA, 0xD486DBE7, 0xE41990A9, 0x305B2835, 0x0B45A04C, 0xED904CDB, 0xF3CBBE7E, 0x236F1FB6, 0xFD7AD037, 0xEF4C8CB6, 0x413320BF, 0xB96BF56F, 0xA22C1B54, 0x7F965953, 0x0641A14F, 0xA024A0CB, 0xE3AC760B, 0xB24EF75D, 0x035907D2, 0x312611FE, 0x97BD2A9E, 0x31A9BAE0, 0x9315EA40, 0x8441FC49, 0x0330A00B, 0x7C7AF1A5, 0xA3AC3CB7, 0xEE5F69B3},
    {0x3B77B924, 0xC24288C9, 0x04F53001, 0xBD5ED85E, 0xF2513282, 0x50EB67BF, 0x044746C5, 0x5163617D, 0xDCD788C8, 0x8756CCC5, 0x9BA7E61B, 0x3AA23923, 0x840708AC, 0xFB09E2B1, 0x07F93D25, 0x3B6A7E3E, 0x253B5669, 0x7841DC7D, 0x79725A0B, 0xBD82F5E5, 0xDBFDF536, 0x8A656D95, 0x210AB020, 0x61EB5C19, 0x2677FE69, 0x2C8B84E8, 0x4BD0F536, 0x3BDCBCBB, 0x59B8C8BC, 0xEFC7FBC7, 0xE059E1FE, 0x41FFAD4E, 0x9AFE8C69, 0x2B119B17, 0xE20A89C8, 0x732305E6, 0x20769BA7},
    {0x8B145FAF, 0x004E6933, 0x13DA7D6A, 0x7AC71A79, 0x62A9B67E, 0xC27DF902, 0x0290D9D8, 0xD25AD73F, 0x615A2A2C, 0xF8B8D005, 0xA099BDBD, 0xA46B5969, 0x333281BB, 0xEA8302AD, 0x9D7654CA, 0x22EDC9B4, 0xAA5B68BF, 0x6DE71742, 0xCD380BB5, 0xE63022E8, 0xEAC6DDA4, 0x486432C8, 0x15590868, 0x0A14D624, 0x47FECB99, 0x56D9F96E, 0x44236C82, 0xA9FC9BE6, 0x2BD2A718, 0x9DE93A96, 0x5520A386, 0xED2EB35B, 0x1F772456, 0x29D7B44D, 0xE055F18C, 0x2D982C39, 0x557F9637},
    {0xA29620CA, 0x5C12AEB0, 0x845A906B, 0x798305FA, 0xCC7F72D9, 0x02A84439, 0xEA59C663, 0xC4C34053, 0xF939E2D2, 0x87FFF0F9, 0x6182BA5E, 0x423C132C, 0x022FB1A9, 0xA095AFAB, 0x95A5AE63, 0x09084186, 0xE5CDFAC5, 0x9E6A8CB4, 0x874476CC, 0xF86B2806, 0xF33EE3EA, 0x86698DD2, 0x47335B7F, 0x94F9F723, 0x7AA447D0, 0x8919EE6B, 0xC93CA98D, 0x902EC307, 0xCA352BDB, 0xDAF9B52F, 0x02A51C4D, 0x27B05892, 0x3992E5A8, 0xF0C6E482, 0x26EC08D4, 0x0169F9B4, 0x018A9982},
    {0x6EBA4D7A, 0x146E8D7A, 0x3985F7A0, 0x0E696856, 0x6593D6D5, 0xDCA1ECBE, 0x39858C09, 0xC865AD51, 0x6F1D4645, 0x5425AE5E, 0xDF213897, 0x54D2805E, 0xFBC2FF3E, 0xF8D3B942, 0x704964FA, 0x27511138, 0xA6982C63, 0xB35624AD, 0x4B8F1229, 0xF73C6192, 0x24A66D44, 0xE5D96AF5, 0xF86B1C84, 0xA2400F12, 0x6A2C0316, 0x89B271C9, 0x78C9E756, 0xC0366782, 0xA50DD274, 0x7214AF0D, 0xF7671D63, 0x16AC7571, 0x555668C6, 0x8715E885, 0x280BBE87, 0xF531F58F, 0x9B89D7C8},
    {0x41EBE5EF, 0xDBDFEE46, 0x28BFF1D4, 0x03B10304, 0x8944BE2C, 0x28CA6066, 0x711544B6, 0xB5D80D49, 0x709D665E, 0x04F5B926, 0xBDF4EAE3, 0x4E8BB54E, 0xBE7FFBD3, 0xB28FDD1C, 0xE0AF523E, 0xC638426C, 0xDF50BC55, 0xB6726CC9, 0xC5545A19, 0x4C768C26, 0xCAF434C6, 0x31187B00, 0x49B7D7F0, 0x491C673F, 0x45AFB7A0, 0xCDC11E5D, 0x67B4D51C, 0xE6F4C79D, 0x2C5E1D11, 0xD356EE90, 0x4598CFAE, 0xF1271366, 0xC7CD1AA2, 0xC5800941, 0xED84DE5A, 0x71B966AC, 0x3E996898},
    {0x401D8FBE, 0x0FBAC0BE, 0x87F0DF66, 0xF704714A, 0x6A9882D6, 0xF2F0386D, 0x47890BD6, 0x52355601, 0x797BAB30, 0x83EDCC2F, 0x5AA6D2DD, 0x79279BF9, 0x0C22976E, 0x0EF0265E, 0x342176D5, 0x30E0CCA1, 0x8EEF51BE, 0x489C6182, 0x1E318618, 0x9D5CD930, 0xCCD31AB5, 0xE230FD5C, 0x93EDABEC, 0x000BB7BB, 0x37214977, 0x87EFD6C4, 0x272664F0, 0x0D74F10C, 0xC8FFEC76, 0xB9CD2B55, 0xA4C3F3BB, 0x577B0836, 0xF85F14E4, 0xE4137AEF, 0x2E494DF8, 0x15B4A72F, 0x57F818F3},
    {0x2A51AC64, 0x55EA1D25, 0x73F63ABD, 0x1393A0EF, 0xA41BD017, 0xEB53F626, 0xE59A5BBB, 0x93E8D794, 0xE5E56CFB, 0x6D6455D5, 0xDFA95ED7, 0xAC225ADD, 0x1F3C8A9A, 0x0C0EEA4B, 0x14D8453B, 0x2367CEC3, 0x8B968283, 0x981D9FC6, 0xFCAD1C5F, 0x33DC6568, 0x8C921FE8, 0xD8448018, 0x4F689F54, 0x2D644973, 0x75A3CEF4, 0x2FD03C12, 0xE5BA1C78, 0xB67F8964, 0x8C6D9C21, 0x22E3B11A, 0x97726796, 0xB33800E2, 0x152433E1, 0x80EA2788, 0x4C612B5A, 0xFDD45F6B, 0x0036773D},
    {0xB22C61D1, 0xD4A42BCE, 0x5EF01DC7, 0x62F9CEB6, 0xAA581620, 0x16796B17, 0xE78FF9C6, 0xF340FF57, 0xC36A5288, 0x25B7C151, 0x16AB4497, 0x73976A98, 0x7CF745FC, 0x8E83E6DD, 0x7F78E6E0, 0x0E86CD8C, 0xFA59F4FC, 0x652846FC, 0x70B86ED9, 0x356FB4A8, 0x13ACDDB7, 0x3E5AA108, 0xE24D7623, 0x670F4C96, 0x7C79B748, 0xFAC4A127, 0x5E4B8ECE, 0x52C7ABD0, 0xD2EAF1C5, 0x975FF704, 0x08703B09, 0x113E1A31, 0xB97B35FF, 0x01D87FE2, 0xAC51C60E, 0x7430093F, 0xB3497846},
    {0xEBF4C72B, 0x320A6BB1, 0x6B36911E, 0x4DEEF783, 0xBF6768DC, 0x8C58F99B, 0x922E383C, 0x2BE30204, 0x65CD3C81, 0xC3365F95, 0x4A29E97B, 0xADCEF01E, 0xAABB8AD8, 0x10C83346, 0x6EF7D292, 0xBF96453B, 0xDBB5D9C5, 0x2CABF9DA, 0x5C0B9180, 0x2ACAC4AE, 0xFBBE28EE, 0xEAEFFB5F, 0x8A1CE852, 0xB6E57C22, 0x48DB6796, 0x3152B861, 0x2A1B8BDB, 0xC6A327F4, 0x345CFBC7, 0xF88A409B, 0xC5F5EFAB, 0xFAFA126B, 0x3579E67B, 0x98293BFC, 0xB46E5053, 0x7FF28536, 0x1A3154BB},
    {0x70BB502D, 0x4AEC05FD, 0x32A06654, 0x5FA54917, 0xB2F20082, 0x02E9AAD4, 0xD9BBD33A, 0x086BD423, 0x7AB2731E, 0x810691FE, 0xFC7FD17C, 0x834F2BFD, 0x7E5CF291, 0xFF690C69, 0x48323915, 0x73F92343, 0x6ADB2A80, 0xBF274E56, 0x8A75C426, 0x8BCB9754, 0x85E3145F, 0x7C8DE9CF, 0x97C16BC8, 0xD03E0747, 0x9266E3DC, 0x997D01B5, 0xF4625D54, 0xC34D31B3, 0x13569E89, 0x8AF65908, 0x0ED221EF, 0x2356E831, 0xBEC005D1, 0xC315D354, 0x05574F86, 0x5FD8C68B, 0x3651053E},
    {0xA6EB2E46, 0xC57B4832, 0xB095B99F, 0x5B0435E2, 0xA27432DF, 0xFB170336, 0x7309BA75, 0x999C33A1, 0xEA2B6032, 0xDB245387, 0xCE89EDDA, 0x2DE5BAFE, 0xC276A632, 0x033B278C, 0xA6976078, 0xEFE96244, 0x904A6BA2, 0x6FDC3467, 0x9EC4BF67, 0xC43D1C74, 0x1740D627, 0xAF8A99DC, 0xDD297C57, 0x245E6863, 0xC6FBC7FB, 0xB0DE45B1, 0x78E5FAB9, 0xEBFF7AB4, 0x9298E679, 0xAEBA925C, 0xDF16138D, 0x94C8C6D9, 0x6C0F6CCF, 0xB5555874, 0xC1942FDA, 0xAEE24590, 0x91EA4A10},
    {0x9FC70E3F, 0xBB1BCC37, 0x425589C4, 0x821A1725, 0x3C67E94F, 0x8C6739A2, 0x2A4CC345, 0x93723CDC, 0xBB58B71F, 0xFEB4D91E, 0xE71B044B, 0xEC001111, 0x2BCEF86F, 0xEE950642, 0x4946DDB6, 0xDE3196EA, 0xD922AE7B, 0x2CF9A2A0, 0x6DC36BA3, 0xE91FB01F, 0x846921BB, 0xC95A6D7A, 0x30602F50, 0x70217F98, 0x049D0839, 0x6CFAE590, 0x8A2BCCA1, 0x60CD95DC, 0xEE946FC7, 0x1330A5F7, 0x0DED7CF1, 0x9742AD16, 0x39D4AE76, 0xEB37EC32, 0x002F2777, 0x202978C0, 0x0DF73BBD},
    {0x6FB35DF5, 0x52BA4246, 0x492C020D, 0xD14888E7, 0xF8FFF2A7, 0x10622E26, 0xF82FDF0B, 0x572BC4D3, 0x3E6EA194, 0xB5BC2D8B, 0x80E74BFD, 0x438EA3B4, 0x5CCA0DA3, 0x0E3C1585, 0xEE186446, 0xC459FDF5, 0x6FE7891B, 0x13C9585F, 0x1647A5F8, 0x5AF12514, 0x8AC87135, 0x016C7578, 0x134D5C1F, 0x9746051B, 0x0A24EA61, 0x4E284497, 0xD279DE99, 0x91255074, 0xEC67AE55, 0xF99C9DD7, 0xFE530769, 0xDE147527, 0xC8A67B28, 0x90668D3C, 0xC1C5EB50, 0x797506D0, 0xE37218F2},
    {0x12FA8749, 0x3E092444, 0xEECFAB98, 0x3B4DAC87, 0x6F793235, 0x4356602A, 0x2EB6CFFA, 0x25D4914B, 0xC51056D6, 0x1806A5C7, 0x6401E4CC, 0x9E544662, 0xBC2BFA25, 0x814558A9, 0x37B5CAFF, 0xD5BDCCD9, 0x96473844, 0x405DFECD, 0xFB6C9968, 0x70784E77, 0x72FD667C, 0x5EE44E70, 0xE67D0D9F, 0xC1279713, 0xF2912A9F, 0xF5A20056, 0x9A804815, 0x70B63B4D, 0x4D7190CC, 0x28718097, 0xA49013CB, 0xDF7368D0, 0xAD9B6AD3, 0xE3DD7BD1, 0xECF1098D, 0xB62CCD81, 0xBDC7CCD1},
    {0x42A7D845, 0xAE816DDF, 0xFD186BB1, 0xA929B325, 0xB1C28DB8, 0x33807D06, 0x68F7E569, 0xB920AC6E, 0xC749A738, 0x83C8FD76, 0x6E995FE9, 0xB4185D4C, 0x58112E28, 0x5DAA5A00, 0x99BEB688, 0xBAE16866, 0xAB316CF8, 0x84853B00, 0x721CAC7F, 0x0D12AA6B, 0xF1B05D1C, 0x5D2045E2, 0x8FDD7B2D, 0x0E4C034B, 0x37D503DE, 0xDD801F89, 0x8259F5C7, 0xF53FF251, 0x9AA2BCA4, 0x47B7CD0F, 0x2C1F8761, 0xB947BCEA, 0xC9394765, 0x29FB5B05, 0x91D4E7A2, 0x802B0004, 0x7F7C5789},
    {0x62324F61, 0x8B657126, 0xA37478B0, 0xBB686485, 0x029310B9, 0x60946B35, 0x8316048A, 0x02F66AAF, 0x1A0A0A31, 0xC231E951, 0x25A67284, 0x94379E14, 0x2FBF7C94, 0xF743D053, 0x55BB8AEC, 0xB26DBE19, 0x697F4FAB, 0x0465D0E0, 0x6B1B73FE, 0x10134F16, 0x9189D9C8, 0x1811C019, 0x45B97131, 0x4C131CFC, 0xC2A7642C, 0x57D7667D, 0xEA7FBF25, 0xEB22AE17, 0xADDEC146, 0x52AF48ED, 0xC416AD97, 0x4F265526, 0x5440C640, 0x1C96E8D2, 0x3B81D254, 0x4687E447, 0x2859C70F},
    {0x1A4E65F8, 0x0BCD1F17, 0xC0DC3A67, 0xA11D9870, 0x204ADED5, 0xF5DB1845, 0xDDCDE427, 0xD888B8E8, 0xDEEFF252, 0xC48C75C3, 0xCB3A2177, 0x54742A0D, 0xF65CA5B1, 0xF4A4B813, 0x67CECB8F, 0x2D14B087, 0x415C8A06, 0xB5966EA1, 0x1DAC2857, 0x653FD5B0, 0x260F52C1, 0x7788095B, 0x020BBA7F, 0x9358D07F, 0xD48617DC, 0xB26B4073, 0xAB6DBE8A, 0x9258F3EC, 0xC85E9447, 0xB5BA17E5, 0x7DB63962, 0xB5EA495D, 0xCB4C631D, 0x358FA9A3, 0x3D5E23EF, 0x3F09A770, 0xA8305620},
    {0x07C27742, 0x08FBF9CC, 0x0FAE7FC0, 0x679F5557, 0xE58B0291, 0xF53FB79F, 0x7A1B467D, 0xBE675AF9, 0x935991AF, 0x53D801B1, 0x12B683F2, 0x71489C77, 0xEA404266, 0xD2F59F5B, 0x436962CB, 0xDC41A387, 0x598A12E9, 0x393FD780, 0x22D7D0D7, 0xD4B7B3B2, 0x71C733C0, 0xB9294FC6, 0xEF82697E, 0x7B7480D3, 0x9A6DEE75, 0x1A2EF122, 0xAE86B34E, 0x9DD2A443, 0x03772D70, 0xFA5C32CC, 0x642220FF, 0xA1D0D227, 0x78EA9DFA, 0xA0AECEB1, 0x1C3BF351, 0xF65122BB, 0x3C707FFB},
    {0x8C2A542E, 0x712C76CE, 0x20F8C69E, 0xD813E9C5, 0x1CA0BB68, 0xEECDD3F2, 0xE8BD79CF, 0x5F35E1F4, 0xBA16B9B8, 0x5866BCE9, 0x5F575709, 0x377CA185, 0xC378B942, 0x71247643, 0xC7E26CF3, 0x3C32DDF0, 0xA3A834EE, 0x48D9F31C, 0x58D484E7, 0xBEE20A44, 0x5341C703, 0xF75C7BD6, 0x0D71EA00, 0xDC569EF8, 0xED1A8539, 0xACF1C672, 0x2D6AC356, 0x1D4E77FF, 0x978AD28F, 0x401DA261, 0xC91C0838, 0x37242F40, 0x08348236, 0x2DD66A25, 0xC2A92601, 0x8B1B3662, 0x50715199},
    {0x911E06FF, 0xC4026969, 0xE15195FA, 0xDCA8FCB3, 0xFB3ABEEB, 0xCCFADEA2, 0xE64FD99E, 0x4FC0A8A2, 0x4F760445, 0x6B5E8842, 0x0DE756BB, 0xA1C29CD1, 0x4DE78041, 0xE456F497, 0x22E99107, 0x98FF2237, 0x785E6C4B, 0x15F0CA98, 0x838CB878, 0x3A2EA31A, 0x93F418DF, 0xCB0DA3F4, 0x0F294C5A, 0x2277505E, 0xD4530493, 0xFC08D337, 0xB05E7D2A, 0x1061B79A, 0x4C76B691, 0x0DEB5EFF, 0xB5D2D35B, 0x35C14A00, 0x30A2E2ED, 0x7C506EA6, 0x07174596, 0x8A420FE0, 0xEC14306B},
    {0xBBF7EE2D, 0x51A00B0D, 0x4C4F0016, 0xFFE03EF7, 0xA91001AB, 0x42DAC641, 0x7FE2DB36, 0x86D40E7D, 0x226AAFEA, 0x2074F757, 0x5ADFCDD2, 0xF5566354, 0x9732E077, 0x82A2F6C4, 0x872DF023, 0xC49695D4, 0x4DC39A39, 0x2C858C99, 0xB17EC565, 0xAD9BFC95, 0xCB33C8E4, 0xA22E6087, 0x1D1B3EBF, 0x8FCD8803, 0x564761E8, 0x657A482D, 0xC81F441F, 0x16902B19, 0x4491AFFE, 0xF1EF34DE, 0x5217FF93, 0xEE5320AB, 0xBC05BC99, 0x13649675, 0x26F8CC82, 0x0640334D, 0xEAFE8B8F},
    {0xD91D5C30, 0xE3E9E02C, 0x445346D5, 0xB16317B6, 0xADF94844, 0x02E3CE3D, 0xE9942620, 0xFF850889, 0x9014A418, 0x0306BCC6, 0x5E666F42, 0xE7832C50, 0xA51ECE32, 0x177AD47C, 0x3F03FD8D, 0x922123A2, 0x28E99092, 0x0BC649B7, 0x22AF4FFD, 0x77A45AD4, 0x4707577A, 0xFF9BA95B, 0xE6CE1022, 0xAE52E672, 0xEDF149A2, 0x144491AE, 0xA73F03B2, 0x86135EA7, 0x56C4945C, 0x0F1D7F0E, 0xBEA107A5, 0xEC5F0B49, 0xFB3CA351, 0xF5432AE5, 0xD2286DA5, 0x842827D0, 0x25262C60},
    {0x96D1F5B6, 0xBB070F3B, 0x28CAD4C3, 0x95E4043E, 0x4D8DA18D, 0x30A380FD, 0x9B94B359, 0x324AC88E, 0x03197C48, 0xE70841D4, 0xC925ABFC, 0x3FF52919, 0x59CA146F, 0x9766043E, 0xF871EBCD, 0xC46EEAB2, 0x3314CA24, 0xD7F83508, 0x33200B58, 0xB216983B, 0x57CAA057, 0xF2A05E8B, 0x7589FCEE, 0x4811F7DD, 0xB5EA14D8, 0x303ACE6F, 0x0918F848, 0x09751CD9, 0x6A874323, 0x8D95EF20, 0x626D35C5, 0x8F5BA9F7, 0x41060F9A, 0x115CD712, 0x6EA16A8C, 0xFD6357FF, 0x9E922BAD},
    {0x29F03778, 0x6D3A6FDC, 0x7FE8404F, 0x14B4098F, 0x7933DFCB, 0x2775AF8E, 0x7AAFBABB, 0x9ECCAB4D, 0x987E8066, 0x4261E21A, 0x0750E975, 0xD5D3BA85, 0x70D34D98, 0x601417A8, 0x8E720C36, 0x75CBD64A, 0x7152C716, 0xD024CA9D, 0xB3121EF5, 0xA1032D43, 0xE7BDA34F, 0xE7FCF957, 0x8C3EE28D, 0x3D895552, 0xB0B466E0, 0x6480D4F9, 0xE3A31A0A, 0x2B8D41B0, 0x9FFB99AC, 0xC09C0239, 0x95CB3D8A, 0xB43A5F62, 0xA693DA5E, 0x6AB0EDCE, 0x4F7CDE51, 0xDB793ADC, 0x0573488B},
    {0x8F052C3D, 0x7D5B30AD, 0x6938465B, 0xD046F2C5, 0x7CEBD0FF, 0x5B41A4F3, 0xFAA481B4, 0x41776844, 0x49C089FE, 0x47E2E1E5, 0x4114E8B7, 0xDC2CB5B8, 0xD7B95677, 0x1DAFC1D7, 0x17AD5FE4, 0x1D6C9A5C, 0x89AD03CA, 0x62827ED5, 0x7F2E7004, 0xC5D09259, 0x78888430, 0xE7CA5D4C, 0x46722456, 0x27224063, 0x77AF8453, 0x17153294, 0xEE5CF911, 0xB6283C83, 0xA34BB723, 0xB2C79B61, 0xC7846E75, 0xA0F652F6, 0x55360874, 0xCFA197FF, 0xC8112E4F, 0x52931336, 0x6920EF6D},
    {0xB0346D53, 0x602B8423, 0x7A878432, 0x1E57C0AA, 0x533C9089, 0xA7497C76, 0x6F3A45B6, 0xDEC65599, 0x2375996C, 0x47A74EFF, 0x9E8C0AC6, 0xE538178A, 0x7272287B, 0x5D06845B, 0x8AAAD01E, 0xF185C646, 0xA6AFF9C6, 0x7866F191, 0x2C8FD825, 0xB46F53D0, 0xDE4FE828, 0xD684351A, 0xC45EFDEC, 0x823A43DB, 0x1615C5DA, 0xD8D045F1, 0x0AE67446, 0xB3417159, 0x65A02242, 0xC50B45FD, 0xDC158020, 0xF14D3A58, 0xDC3374E3, 0x71B75FC5, 0x7124E32F, 0xACC13DAA, 0x252FB4BE},
    {0x1D6BF290, 0xECDA0463, 0x03462A4A, 0x7AC220E7, 0x1A555C49, 0xD4FA7170, 0xAF570804, 0xD8D2B2AD, 0xBC22261B, 0x50B165A7, 0x6E9F5D06, 0xFFACFDF9, 0x5490CD01, 0x4DA8C562, 0xE9C10BD7, 0x4DCC7574, 0x6EF8C0F5, 0xEDD012F4, 0xE15BAC48, 0xF61F8BA0, 0xCEBB61AE, 0xECAE90BD, 0xCB809669, 0xFD18BD7D, 0x92943C15, 0x658D2F6F, 0x5F2FBEBA, 0x193F0850, 0x29B01C70, 0x06CD4FA4, 0xD8110260, 0x700535FA, 0xF025046D, 0xB776ACB5, 0x25E42E6F, 0x33CC46C4, 0xB65AC414},
    {0xF3E71979, 0xB104D5F9, 0xFAB1844C, 0x3FA9F592, 0x931F6BB9, 0x754E2361, 0x3BFC41CD, 0xB481F67B, 0x5FCAB99B, 0x8365B425, 0xC10DDB20, 0x94A94287, 0x86ADE790, 0xFB614A68, 0xE6AED9A6, 0xF018FF03, 0x3ECF7650, 0xDFBD85B3, 0x600DC9F0, 0x7A4C63BA, 0x7C3BF0C5, 0x850DCD86, 0xD67D48A1, 0x7815F195, 0xCE902CD9, 0xF91DA3D9, 0x844EDB6C, 0x1AE80057, 0x006F847D, 0xF41FC7B5, 0x2ADED5E8, 0x400F8F68, 0x927C2FDB, 0x191494B0, 0x7F4585CE, 0xDFCC7276, 0x1045493D},
    {0x6000AA82, 0x72981436, 0x94949516, 0x18441C9C, 0x29D88BB9, 0x0824CE75, 0x5C73C342, 0x2D0BBD81, 0x5088310A, 0xB25B950C, 0xC138A826, 0xD9F73D05, 0x7F35B0B9, 0x17DC9E04, 0x249AD5B9, 0xF45C018E, 0x8453DF7B, 0x1518D000, 0xEFA60740, 0x530560B5, 0x0D716455, 0xA2EDB6DF, 0x5D2DADD3, 0x0945F7EC, 0x509DC46D, 0x50716F31, 0xD87D4235, 0x7DA4FD4C, 0xBD7A189F, 0xE7EFB2D1, 0xD341026F, 0x11EF0D64, 0x45E164A5, 0x47087E9E, 0xFF2C5F5F, 0xFE9B3757, 0xB86AD126},
    {0xE02503FC, 0x41CCA4CB, 0xF12CCD50, 0x1AC7C6ED, 0xDF36D318, 0x57A748B2, 0x9BD0BF93, 0xC9E20162, 0x71A796B2, 0x6653FF5C, 0x8D3CB6F1, 0xC1E99228, 0xAD8B4DF8, 0xE09ACAE0, 0x50301789, 0x81D6284C, 0x7CFCC690, 0x9463E5B5, 0x286ACA45, 0xEA7CE9B4, 0x03EB2A99, 0xA3079B15, 0x499B1674, 0xC02ECFB6, 0x6E53D545, 0x1EE839BB, 0x3EA54F10, 0xFDE2819F, 0x97AF95D6, 0xCF757D38, 0x6C373715, 0x92F0A958, 0x1CEF9586, 0x8DD84F81, 0x5C669E1E, 0x01ECB720, 0x969909E6},
    {0x2D1FB241, 0x4F0E0C46, 0x4747304D, 0x7BCEE5CC, 0x63667506, 0xA6CA6BDB, 0x755E7929, 0x65CF2772, 0x310637BF, 0x205BAB6F, 0xAFF1ADCE, 0x64A1E05E, 0xF67A1D73, 0xE406B87A, 0x44F341E1, 0x2CBD9F95, 0x0FD0AF20, 0xF293EE03, 0x899FDA8A, 0xE67DD678, 0xBF376D9A, 0xBB069E3D, 0x0F88BF9E, 0x817994CC, 0x32943F3E, 0xF0C8E14F, 0x5BCF3011, 0x3FD2BEBB, 0x714F208C, 0xBD067165, 0x4A957439, 0xBC590C03, 0xCBA6B2B8, 0x22D5A77B, 0xBD71011B, 0xF8910497, 0x02A732C8},
    {0x7B8F04D2, 0x987AFD06, 0xD2649A8C, 0xA0141A26, 0x7D8A51FE, 0xC2C775E8, 0xC33B8030, 0xBA583DB8, 0x9A7718B8, 0xFBA77061, 0xB14FAC50, 0x94BE7F2A, 0x4FC21B53, 0xFB2B3008, 0xA8BA27A8, 0x5E889551, 0x6E458F30, 0xCF5D23D6, 0x0B7AE76E, 0x493AA165, 0xF5B66827, 0x0474298C, 0x69815307, 0x0BB955BF, 0x0AF44E2F, 0xAF09F7E0, 0xCB70DAD2, 0x1F533744, 0x0A95BA05, 0xCD5C9BDA, 0x6E9E046E, 0xDA5C524E, 0xA2FDCA13, 0x2CD88903, 0x985491BA, 0x0FAEE46B, 0x451E6BAD},
    {0x86E2D326, 0x97A927D5, 0x224E9F7C, 0x2E61EE1A, 0xFE52C52C, 0x1E0D8866, 0xAA921877, 0x16DD2BA2, 0x6C2A81B5, 0xA45879B5, 0xACF26BE4, 0x3C236B92, 0x2C63A6EA, 0x0B29669C, 0x164D1CB1, 0x59720A81, 0x2889A7DA, 0x0F23345E, 0x34BA951D, 0x10342C7C, 0x87DA6DB0, 0x0E4FB8C8, 0x2B4B9081, 0xA07DDE1E, 0x777F4984, 0xBD248EB2, 0xB787440D, 0xC000BDAE, 0xC0230404, 0x4A211BCE, 0x14DEB28F, 0x1DE2AAD6, 0xD7724A60, 0xE3125EFE, 0x8111E938, 0x67BAB9FA, 0x1D2B2EC0},
    {0x3C388BA5, 0xDCC673D8, 0x65AB8ED1, 0x63FAF73C, 0x3C738632, 0x6F100AE9, 0x75671766, 0xF885D05B, 0xBC9156D7, 0x69D2F8D2, 0xA734A4E4, 0x8B35F56E, 0xF23FEFD7, 0x18EB28BF, 0x0534D95E, 0xAB95C453, 0x4CB7B57D, 0x506C79E7, 0x68E5667F, 0x45D28F01, 0x097543D8, 0xC9CC3421, 0x316443E0, 0xE337FF96, 0x883075B9, 0xB2FCF3E8, 0x1320B2E5, 0xC00A6C1A, 0x81FC8F2A, 0x012DC719, 0x4895C61E, 0xCA8D27B7, 0xA2C6B219, 0xFE132FDE, 0x7D768D5A, 0xD8E3B807, 0xE518A515},
    {0x940CD418, 0x6ABB97E6, 0xF9688B86, 0xE63AE360, 0xF6932BBE, 0x1414A186, 0xF6F5EFC8, 0xC164AE8E, 0x1F648031, 0xAF12E70D, 0x29D3B2C8, 0xF3EDF9A8, 0xE8C0D0D0, 0x2C81A9EF, 0x05DEB9E6, 0x66803FEB, 0x542CEB28, 0xEB3811D8, 0x6B42E705, 0xDB1CEBDD, 0x00BE85B8, 0x88DC7E01, 0x69E20C12, 0x4EA346CB, 0xFBB0C3EB, 0xFC3FC996, 0x6B9B6EE7, 0xEF0F84E0, 0xC991FAA6, 0x09B4DA9E, 0xCBF85F7E, 0x20631336, 0x359D4DB7, 0x0B3D9270, 0x4F7F845D, 0x9BC56853, 0xEDEB1BCE},
    {0xA14790F7, 0x5673B0A0, 0xE3BED709, 0x3DCA5E6E, 0x392B4EEB, 0x7BB39FF4, 0x56BD376D, 0x1836D3AC, 0x6994E304, 0xB3B19F56, 0xFE972B4C, 0xC624A1E8, 0x4881257E, 0x6411A846, 0x36B967AE, 0xBB0C8014, 0xBED2218F, 0xEF9DEA26, 0x82EBCE2E, 0x93AD05CA, 0x5A2CD29F, 0x9A249BFC, 0x7AE88791, 0x28A5E503, 0x634F7929, 0xD2B6AB7E, 0x2E20612A, 0x8DE8FBEC, 0x2A95C9F2, 0x87B4E9C9, 0xB6F5F6BA, 0x5BAEFDC0, 0xD99C2FBF, 0x6F56A241, 0xE18F6716, 0xB2CB598C, 0x96898D80},
    {0x600C5B7B, 0x6B0FB589, 0x716D5A4E, 0x559881CE, 0x93C870DB, 0x4EDB62C5, 0xCAFA0E1D, 0xFFBF0ACE, 0xF1E3406E, 0x9099E608, 0x1C3A3BFF, 0x7C8FD481, 0x7E7FCED3, 0xA7DCE690, 0x62D25F77, 0xB2A4E431, 0xB907C58D, 0xE02914DE, 0xFA16AD08, 0x47EE3704, 0xF24D2899, 0xB8DE2CCF, 0xF99FBB6A, 0xFE041157, 0x89EA9D4E, 0xF6FACDAC, 0xB8D394D2, 0xDF495392, 0x54C1CE45, 0xF9D44BF3, 0x8E5A60DC, 0x66AA4AA9, 0x44A3E0DA, 0x8221CEB8, 0xF5CDDA85, 0xF1688C7D, 0x72EEBA6C},
    {0xE053BA71, 0xF342AEF9, 0x43B9611F, 0x8C536C10, 0xA90E858C, 0x8A825DC3, 0x44933D2B, 0xF2F53A99, 0xD2AAD392, 0xB286EB51, 0xF8C92CB0, 0x1E47F397, 0xDB2BC5A8, 0xE87AC0DF, 0x37B5F6CF, 0x1679CACF, 0x4A10EDA1, 0xCA4E05E3, 0x4E25E482, 0x2D23ECC5, 0x379DE179, 0x53B803A6, 0x39C6C484, 0xCA5AA813, 0x3F0D505C, 0x0C424AE4, 0x4F2EF279, 0x10E22958, 0x8017134A, 0xFEFC221E, 0x6A0C5262, 0x902557DC, 0x727583FC, 0x9FC69217, 0xD63A07B8, 0xB193AB86, 0x27E0CDD5},
    {0xE4ACC0D7, 0xA8DC3F76, 0xCB316F92, 0x6D43E961, 0xE0F83FE6, 0xD6B72780, 0xB255CF65, 0x78F2992F, 0xAC94D465, 0xE424EDAC, 0xD988D90D, 0xAAD096FF, 0x9A3AD365, 0xE49502B0, 0xA94A2490, 0xF5BCA82B, 0x3E6C47DE, 0x5F5F3760, 0x1A6302AE, 0xD2B9F7E6, 0x05D3231F, 0x92BCAEB1, 0x1B5B28B3, 0x1F528C4F, 0x8EB8B125, 0xCA09EAAF, 0x9ED58EEB, 0xDDBCE516, 0xA6874D12, 0xF46F9F7C, 0x4179D4B1, 0x44C4E4B4, 0x43489CDF, 0x684DED90, 0xF10A986A, 0xE8F879B7, 0x08B9BF73},
    {0x2A5895FE, 0x56CE6CC1, 0x5E7ED45E, 0xCA8C2331, 0x52EB2FD5, 0xE3D38FDB, 0x99403E77, 0x58995171, 0x8DF6E00F, 0x01937BA3, 0xA1F7AB7F, 0xE45BC89D, 0x9104F992, 0x83DC8307, 0x9FC35EF3, 0x734BF964, 0xA02D67F4, 0x7403AF0F, 0x2D9C5BED, 0x8106579E, 0x598312A1, 0x7EAF7F16, 0x61FD33AA, 0x201A97D1, 0xE89178CF, 0xD1CD20EA, 0x55C7A0EB, 0x2290B7D2, 0xF502AA37, 0x20B406D1, 0xB8ECDFE1, 0x1680F194, 0x4E248E3A, 0xE71994A5, 0xBC275FC7, 0x4A130313, 0xC88115A6},
    {0x646DF035, 0x52A0B8B3, 0x17CF09F0, 0x952A675E, 0xB7F51121, 0xDAAD4E86, 0x4E77BA5E, 0x138DD4C0, 0xEEAE0D91, 0x06818DA5, 0xD48B5B52, 0x9B2BB0BB, 0x5545D4E5, 0xC1752A6D, 0xFB1CB8CE, 0xB0BAFFCB, 0xB5B822F7, 0x8A2275E2, 0xFFF9ED8B, 0x21EE5F16, 0x81A0553B, 0xAC356172, 0x92178AEC, 0x604E8F86, 0x4E4A291F, 0xF8AB0966, 0x745DA2A1, 0x9B9F5965, 0x9C715F26, 0xBB47C27B, 0x2E4559D5, 0x7019D572, 0x06117836, 0x4516CB91, 0x54CAF4F2, 0xCE156DF5, 0x665E61D5},
    {0x9841114C, 0xC739EF2A, 0x14B83062, 0x243EAA97, 0x05016119, 0x236B6D51, 0x9F568A46, 0x4E7BA066, 0xCDB83963, 0x1DCEB36A, 0xC82D4452, 0x17DDFE65, 0x512F49B9, 0xAD49B7F5, 0x9AF0D8F0, 0x106974B6, 0x2BC99EDA, 0x23EEB3AB, 0x9FA054BD, 0xBEAFB817, 0x6410B667, 0xCF604B9C, 0x45963838, 0xDA94C599, 0xF5640582, 0x46D09940, 0x733068D9, 0xC8637801, 0x5E9710F1, 0x53B71D31, 0xCFD7B87D, 0x8D847D05, 0x8E680E06, 0x21949C66, 0x08DA9039, 0x9E0C3D49, 0xCDC76277},
    {0x0303B940, 0x03920AF7, 0xAB58346D, 0x58D75886, 0xF3E3190A, 0xBAD8520E, 0x623DBDA7, 0x69B025BF, 0xCADCC444, 0xB7500FC7, 0x696CFEB5, 0x8752B53D, 0x08DFD774, 0xCD1BEECD, 0x469B8BEF, 0x91E92609, 0xDC7A4E2A, 0x4ACC5E6D, 0xEA7E60ED, 0x8A2F8F23, 0x42A81674, 0x309CBAAD, 0xC2D2AACE, 0x676B6968, 0x1C325AD8, 0xABB3177C, 0xF3847A04, 0x9C24A27B, 0xDB39F99A, 0x2C94F092, 0x6A2953FD, 0xA8687F67, 0xD3B4EF39, 0x5B7A65DD, 0x3841CC06, 0xF1CD03C8, 0xE56EF64E},
    {0xAD7670D0, 0x863B8C42, 0xFBDE1207, 0x44AFEEFD, 0x1BF4B706, 0xF79E7152, 0x2F489BAA, 0x918C9807, 0xAD72FD00, 0x875CF555, 0xEC201427, 0xAF8C317B, 0x3EA42AFF, 0x39E1186A, 0xDB2613FF, 0x57725925, 0xDBD3BCAB, 0xCABB5927, 0xC35F1079, 0xFA3F72E5, 0x111C506A, 0x8CB431E1, 0x9D90EA60, 0xD93FDED1, 0x1AC5BF2A, 0xE4014502, 0xA9A43890, 0xA3D65D68, 0x13A5CC24, 0x6E3ABCC4, 0x9A513106, 0x5CB45B33, 0x3A4ABC3C, 0x28111F9A, 0x04880B4E, 0x740BE57E, 0xCD197B2B},
    {0xBFAE7AD4, 0xCF36EDA6, 0xB6EEA600, 0x574741FC, 0xB99B99F4, 0x97C321D8, 0x91402DE2, 0x95005F34, 0xFBDAAF56, 0xC521F253, 0x6D5EF39F, 0x34FD0D8D, 0x0E9E671F, 0xD9A3F670, 0x6594C416, 0x521B4AA1, 0x8EEA51BF, 0xB3F62301, 0x1234626A, 0x61FF1F3D, 0xAF4443A0, 0xEC77DE9B, 0xE4807B34, 0xE6F61F38, 0x01F3645B, 0x8ACA6258, 0x932D4DA0, 0x5C8F64BB, 0x275FC2AA, 0x102D0B98, 0x704BFF2D, 0x920A86EB, 0xDD3126A7, 0x15942B55, 0xC53B68BC, 0x1F181708, 0xBB2F2BAC},
    {0x6978C189, 0xE276AB06, 0x45CC0887, 0xE5166291, 0xDE5F1398, 0xD0F3413D, 0xB82A9AE9, 0xD541E85B, 0x9CC8BC79, 0x276119C3, 0x77E81288, 0xC593AEF2, 0xB7D2137F, 0xC2A27B58, 0xF658E3A2, 0xE0F7A50C, 0x1F789DF6, 0xBF461BDA, 0x992318C4, 0x55F7B611, 0x568DDF95, 0x40408342, 0xDF532AB5, 0xE4FCD050, 0xDC4D9C28, 0x13B10EFB, 0xD5471960, 0x2C56F542, 0x59CFCAD2, 0x53419764, 0x55D3CC78, 0x9A18E28B, 0x8372799C, 0xB7A810A9, 0x73B02096, 0xEBE88C9B, 0x8D142B07},
    {0x0ACB736B, 0x59C4B340, 0x98898F87, 0x71E26AB1, 0xE847F78F, 0xDF3ADCA5, 0x06144E9F, 0x012FD1D1, 0x87A7CCA7, 0xD7C28A65, 0x68023FEA, 0xBA79DD5F, 0x3D2F972C, 0xD188B10C, 0x3E9F2BCF, 0xB0F895A1, 0x7A533B37, 0x1C7B73AE, 0x84A3EC05, 0x68638A32, 0x9BC3CF73, 0x1D31043D, 0xCD841B3B, 0x680E4FF9, 0x7C3E6FAF, 0x41A2EDF2, 0x4DBF60F9, 0x5BE28884, 0x60002992, 0xD64CDC27, 0xEEB2DB46, 0x1009C715, 0xB3376EEB, 0xFC78EA63, 0x0F3F5DA2, 0x454316E3, 0x395D2C0A},
    {0x737848AC, 0x7D9D3DC1, 0xB9925CCE, 0x75BBB020, 0x656F525A, 0xA8991680, 0x2672D881, 0xA28E81E7, 0xD465C5C4, 0x7550726C, 0x931248D1, 0x00832875, 0x2539E86B, 0xE8DC0BBF, 0x48203C63, 0x96EE3C29, 0xFE82FCE1, 0x81BE894C, 0x7C78D9E9, 0x632C18D2, 0x36459F6D, 0x851E239A, 0xEEA230B7, 0x3803ADA1, 0x674AA785, 0x2B7C0445, 0x1770B321, 0xD30DC314, 0xC7FDD289, 0x7BB54A09, 0xB50141CA, 0xEAAB6F50, 0x3ED219BE, 0xB2451B7B, 0x59921640, 0x56A63082, 0xBE02A09A},
    {0xF1104B21, 0xD41E5659, 0x325B408F, 0x4C1F0B69, 0xCC28570E, 0x7F234110, 0xE6752CD2, 0x7325C0A7, 0x7CF454EB, 0xD53C85A2, 0x129F3E8A, 0x4D70F86C, 0xDC23328C, 0x1DE25626, 0xCF344FF9, 0xC49C3D22, 0x668441EC, 0xE286900D, 0x51F1C0BC, 0xAF07A355, 0x92E318A4, 0xE5575714, 0x95440706, 0xF5A1C83E, 0xC3458F16, 0xC323B7B7, 0x9EDA7B28, 0x89BAA3C5, 0xF044A6A1, 0x2B4FD669, 0xF397694B, 0x95FA0176, 0xF87C0EB1, 0x65207897, 0xBB9FF64D, 0xDD8919C2, 0x5EB0F04A},
    {0x9289DE6F, 0xF3F25CD4, 0x9E17CCBC, 0xF1B5E865, 0x0CE58250, 0x31A16A9A, 0xC9D0AC65, 0x448C5F53, 0x367B28EE, 0x5A76DFB2, 0xA9A8CAF3, 0x15DE8FD2, 0x90B72B33, 0x3A233E37, 0x58B58320, 0x3630EAFF, 0xDB457148, 0xF1D040F8, 0x677A7FBE, 0xA069CDDC, 0x3382CB00, 0xDC64DDA8, 0xC302B55F, 0xD8C609EF, 0x0CC666D5, 0x23B96DDA, 0x69E7CB6E, 0x4E6AFF1E, 0xC821772A, 0xBCD258CB, 0x617E713D, 0xFA87A14F, 0x6EDFA280, 0x053B9023, 0xC9F571AB, 0xCA868F17, 0x6F4075F9},
    {0x5D28BACA, 0xD8E8E550, 0xC1450196, 0x4A58BBFE, 0x05AD30ED, 0x2CA781C5, 0xFA9FC941, 0xB15F53CD, 0x5925EB67, 0x32175731, 0xA2E63AAA, 0x8BBF326B, 0xF34A9400, 0xAD7B7749, 0x5B93CDBD, 0x82E459B1, 0x994E5887, 0x2B89B597, 0xDFE0E4C5, 0xA11E8542, 0xF9BF2C1C, 0x58A460BC, 0x101C572D, 0x41829642, 0x6F6BC678, 0xCC50019A, 0x5CF0989C, 0x58EEB02E, 0x333A8EC2, 0x2FFE8B45, 0xE59B5647, 0x4ECCE197, 0xF1D0021D, 0xA4905513, 0xF08CA613, 0xF02529AA, 0x083F35C2},
    {0x52191549, 0x7E237119, 0x156ABCA5, 0xB6D68DB3, 0xD7EB8811, 0x82E114CD, 0x48588521, 0x460FB492, 0xEAA820AC, 0xFC510DB8, 0x5C808A2F, 0x643D135E, 0xD5581316, 0x3DE90520, 0xB0E337BB, 0xFB2BF84B, 0x183D48ED, 0xCED95E9D, 0x2C550933, 0x507BBFC5, 0x3C5BF2A0, 0x68197BA5, 0xC8847FDE, 0x50C07AE5, 0x612A2F1C, 0x1FB43D3D, 0x5812CED6, 0x4AB2E45C, 0x2A0E8AA3, 0x0D84E62F, 0x724CB6BC, 0x019DD736, 0xAF620A91, 0x8EBF0585, 0xC30B5191, 0x0E8B0846, 0x75C1DE88},
    {0x9CAA40F7, 0xD0AAF5C9, 0x8032F141, 0xADD08797, 0x30301AEE, 0x89284228, 0x0378ED2D, 0xCE75AE05, 0x164DA2F7, 0xDD75424D, 0x2BA7E112, 0x3673EC13, 0x13CDC6D3, 0x6D4AE72F, 0xB0F4B68F, 0x9263994F, 0x8A3A93AA, 0x30927E0F, 0x2FA41CFE, 0xE423BE50, 0x60BBE11C, 0x196DF642, 0x5B214C95, 0xAD67FF62, 0x1880925E, 0x14D24735, 0xE11D9268, 0x0758209E, 0xDFE17901, 0x9D77FBD4, 0xC45F2FEC, 0xA5491920, 0x66D7AF64, 0xF8D9427D, 0x610AFCC8, 0x537ED573, 0x0893A5A7},
    {0x5DFD7359, 0xA8C0316B, 0x233450E6, 0xA0916CAA, 0xB331FE40, 0x36803B22, 0xA07C70DB, 0x817D92DD, 0xE5F20876, 0xCA972A48, 0x31D6A608, 0x837445D3, 0xF86FD16C, 0x96217983, 0x75CB3F78, 0xA506A973, 0x587F9465, 0x9760238D, 0xD3C05E4B, 0x992C4429, 0xD140EF6F, 0x8F14FD45, 0x66BCB4D6, 0xBAB14670, 0x68F2EA45, 0x3A1246B2, 0x8B178611, 0x7860C8DA, 0x6F40B5E4, 0x2C515E4F, 0x5A315A85, 0x51E51745, 0x2CA115F8, 0xDDACE164, 0x232475E4, 0xEFAF52A4, 0x287D8744},
    {0x3C855DA8, 0xE288C105, 0x739583AF, 0xF71992C4, 0x4E437D6D, 0x86689586, 0xEE48412B, 0x5B71E654, 0xD21E49C8, 0xF17D6EC7, 0xA00074FF, 0xA69B7D4D, 0xC3145A7A, 0x89083E87, 0x947D35A6, 0xC04DBB6C, 0x60E9D531, 0xD93FBE15, 0x1B5B9BFD, 0x48B168F9, 0x76387167, 0x5A120655, 0xDDB846BC, 0x7A11F6B9, 0xFDF40EFC, 0xDDA8676C, 0x73BBED52, 0x69D3DF3F, 0x825A0597, 0xFE702C04, 0x80DFECAC, 0x7C282309, 0xA77864D6, 0x9E1635E0, 0x4064E46A, 0x8DCD9CDD, 0x4C9BD52B},
    {0xD2834820, 0x8BB31A14, 0x5B1BA700, 0xFA6F74A3, 0x5A7963DA, 0x70AF02DF, 0x67A3EE05, 0xACF66316, 0xED8810E5, 0x6D9C3A9B, 0xDE5700F2, 0xCAA15C04, 0x1216D779, 0x70EC52DD, 0x52E4848F, 0x113195C5, 0x2A767B66, 0x045996B2, 0x08F38E9D, 0x871D34A1, 0x3BDB0114, 0x1A5BBAC6, 0x4FB89653, 0x0116C79A, 0xE1DDC68E, 0x47A1B243, 0x86F00C39, 0xC220C567, 0x7FE08843, 0x007C712F, 0xD1FFC893, 0x06351AA0, 0x9F560023, 0x62A98700, 0xF8D57BCE, 0xACFC3AF4, 0xA9DFB6A2},
    {0x336F863F, 0x3A375259, 0xF5C6A327, 0x1E345224, 0xC372DD35, 0x2DB21041, 0x2911ACBF, 0x5388E8A1, 0x9E60D14F, 0xD9FFFBCF, 0xEBD01FC4, 0x65320FB5, 0xEA2A2A60, 0x9A7568DA, 0x4BB02D10, 0x4DDD9FB1, 0x4B65A72D, 0xFD7BDD39, 0xEA3B0F85, 0xB2E4375A, 0xC0427919, 0xCBB1E844, 0x2AAA5C33, 0xFEB4E727, 0x207A1F20, 0xA090CF8A, 0x5C727232, 0x5FECB8C3, 0x000B94E7, 0x76AB4A28, 0xB7213ABC, 0x12A89C26, 0xEEBE517D, 0x76CC2BC2, 0x03DE934C, 0x0DB13437, 0x2814CF39},
    {0xF98981CE, 0x38D0C86B, 0x5E50A50F, 0xAC0B8790, 0x9894ACFF, 0x6F6B74D5, 0xC1BBE67C, 0x7AF6D00F, 0x8CAE8C82, 0xFCE050D0, 0xECC57D74, 0xE078F27F, 0xD65F00E2, 0xCB4270DA, 0x0AE340F3, 0x3331C02F, 0x1B7CD2C8, 0x15B73279, 0xDB320A68, 0x001E7A86, 0xE060BCFA, 0x8429DFAF, 0x493D40F3, 0x07907BBB, 0x5A265D96, 0x36D26EDC, 0xA44E4DF8, 0x4CD28267, 0xF06FA170, 0x91620626, 0x253AC674, 0xBFC9864F, 0xD2D88AF5, 0x629677B2, 0xE752AB2D, 0xF622F3E7, 0x43DD2A68},
    {0xE46C53FD, 0xC42C1706, 0x660BDCBF, 0x8890B768, 0x80A67C66, 0x628E8B95, 0x9D8ACA7A, 0xBF062D7B, 0x75E415DD, 0x7F37BA75, 0x439A397B, 0x9D22952D, 0x4CC6A807, 0x411B21E0, 0x151C720C, 0x07F019AF, 0xF76792E1, 0x99450CE3, 0xBF921EF8, 0x581D7AFC, 0x78ED4C0E, 0xAD41E084, 0x767890DC, 0x78A245C4, 0x74E16729, 0x5225859E, 0x8806FEC8, 0xD1BE84C3, 0xBCA094F4, 0xB61610A1, 0x7F44F90F, 0xE5BF5E3C, 0xAC2B82CB, 0xBDA02D0D, 0xFE8113FF, 0x88385F45, 0x4DB9AE16},
    {0x5EA26E30, 0xE788AEF2, 0xA9399CB3, 0x8166483B, 0x2645FE21, 0xD71CDF7A, 0xCE8E50E1, 0x6EA02194, 0xC7FA3D6E, 0xC708FD21, 0x2546BA0E, 0x94FF591D, 0xDE301CCC, 0x188AFC49, 0xDC3F5BB9, 0x69F21397, 0x865E7ABB, 0x58EDCE62, 0x48DBF1BD, 0x4E45747F, 0xF7BF50B6, 0xF83EC20E, 0x0C6DC058, 0xFFBC4448, 0xC7F6518A, 0x6D2F97DD, 0x15497CAA, 0x2B54167D, 0x6DB67369, 0xECB74B69, 0x2B7818FA, 0x381AC6C3, 0xF3FF9E14, 0xB13B0827, 0xBEF94E1A, 0x9D14B3BD, 0x9E6B0E64},
    {0xC97DF2E3, 0x707DAC88, 0x9AFB9612, 0xF3C19831, 0xCBB71DED, 0xE28C99F3, 0x1B88CDB5, 0x4E6B5D08, 0xC8F9B95A, 0x4AF75C1F, 0x64987020, 0x866E0500, 0x76FA8B29, 0xA4D54A8C, 0x38CECD31, 0x73AE3B45, 0xD6B31263, 0x3ADFC51A, 0x97B8826F, 0xB1E561EB, 0xBA1E5902, 0x8A65F5AB, 0xFADE26F3, 0xD74D679B, 0xB6BD5834, 0x7B682E31, 0xC9401168, 0x8C255444, 0x6BFADC3C, 0x78A013A9, 0x92ED6649, 0x52C1BFEF, 0xDBC397A7, 0x37B3F311, 0xD75B948F, 0x2FED937E, 0x4724A822},
    {0x498E4B2B, 0x31ADC5E5, 0xAEE46216, 0x062D5A01, 0xAB785ECE, 0x729F8425, 0xE0A49126, 0x770E4EF5, 0xAC968C3C, 0x6F80635A, 0x34D7142E, 0x8A236044, 0xA75DB49E, 0xF8BF2F60, 0x8993ECFA, 0xFD139627, 0x18A5B6D2, 0x45089B0E, 0xD0863D4D, 0x75BBFD75, 0x7D2BCC27, 0x19D49389, 0x129ABCFC, 0x78CD7D08, 0xE162E158, 0xBD4E65AD, 0x338BD0A5, 0x2E3702C6, 0x63B726CE, 0x17334EB0, 0xE01432B8, 0x0EBD97D6, 0x09EB3286, 0x9CD3BECE, 0x62EF89B0, 0x65228160, 0xB5955FBA},
    {0xE5794628, 0x32B95161, 0x537EE11E, 0xBD46BFF2, 0x28A4E1A0, 0xBFAAB842, 0x034AD384, 0x8DEC7BB9, 0x9AEDDFE2, 0x18BA6C3B, 0x24FB1618, 0x7A0D674E, 0x6E9A4C16, 0xDA8E7078, 0xEB3E175B, 0x591FD375, 0x6698B75F, 0xD949D569, 0x2CDBB04C, 0x07E7FE4C, 0x50302076, 0x8E358F52, 0x73D792BF, 0x9A9C20FD, 0x0A16D061, 0x4085AB9E, 0x23BF06A1, 0xB3F8C223, 0xA7AD458D, 0x6FEA9A3D, 0x3F7CC0E3, 0xF430E240, 0x4AE4EA0F, 0x6C9FE91C, 0xB32D7B13, 0x4D926135, 0x77954D33},
    {0x8D4DEA61, 0x059B2D8B, 0x63B97A54, 0xBC8437B8, 0x3BFA3A82, 0x91B59107, 0xBA2E40A9, 0x5FACA42E, 0xF377D74C, 0xA4BCFABC, 0x4B9F462B, 0x68C5BE21, 0x1737D6BA, 0x030569F2, 0x200629F6, 0xF872C87D, 0x92783E02, 0x85BDF641, 0x8BD4D0AD, 0xDF326677, 0x443DB3C2, 0x02353E34, 0xB9759C19, 0xBE029608, 0x3F48E101, 0x70B59FE7, 0x25B32129, 0x9B2774CE, 0x8D4F9DE3, 0xB4B38F33, 0x65D5E77F, 0x34E6E767, 0xBCFCBB81, 0xB86FF9F0, 0xD68313E6, 0x3C7F330B, 0x58EBB0C4},
    {0x962534DE, 0xD43C0010, 0xA1276577, 0x9523283D, 0x2F98CF40, 0x54C45EEA, 0xB386C860, 0x2DB305F2, 0xF128C55D, 0x88138BBB, 0xFB139983, 0x7F71CD9F, 0x75074986, 0x5D6F0FF8, 0xF58FF8A4, 0xE96537A3, 0xC3B09F26, 0x737DCD6C, 0x8FFD581E, 0x9FC49B45, 0x2F13EA85, 0x63A6E555, 0xA4D85A94, 0x1974EA76, 0xF5DB0583, 0x59E0C514, 0xAACCA253, 0x88FA1E3D, 0xB0F2E158, 0x7DCDED4C, 0x2F3E02F6, 0x9D3D2401, 0x06A3E450, 0x9FF1B209, 0x8B5F2CD9, 0x3E49ED41, 0xB7469E24},
    {0x244D1C44, 0x7109E99C, 0xE8AB44BD, 0xB8757397, 0xE03C8B8F, 0x964DD476, 0x3B17E99B, 0x3ADFC2FC, 0x16841685, 0x34DC1841, 0x60712879, 0xDED4FD68, 0xDCCB55AB, 0x437BCF50, 0x1DBAC670, 0x74612094, 0x9EEE2401, 0xCD3CD201, 0xF86F7563, 0x19F24161, 0x09E5D659, 0xCD920B92, 0xE6E709CC, 0x7E0DFDF7, 0x8A37BD95, 0x1DDB269C, 0x8A194C6C, 0x9360FC48, 0xF160A93E, 0x6BE0EA2B, 0x4BEB1D39, 0x94AF218A, 0x02807D3B, 0xC3039D44, 0x34F83B7D, 0x4CB8EE84, 0xC7B49EE6},
    {0x7FD8A488, 0xAA50AEAC, 0x3BBC09F9, 0x0D6BCF62, 0x75DE5E1A, 0x27A2AA2E, 0x1E0E6778, 0xDBDAD274, 0x19D48B4C, 0x03FB284F, 0x07813D48, 0xCC1B638F, 0x6C54CC4F, 0x9D758586, 0x7DCA5ED2, 0x6B1BEC71, 0x5C290259, 0xE2C9A153, 0xC37E2808, 0x2D303C91, 0xEC29E6DC, 0x02C221FE, 0x16200655, 0x9CD178C2, 0x90B865B2, 0xF0085067, 0x69898A53, 0x64CCD518, 0x915C2D15, 0xE2C688DF, 0x3A8E9475, 0xEE619A24, 0x39AE75A9, 0x31CDFECA, 0x99144079, 0x26C9A62E, 0x92119094},
    {0x3FF53D66, 0x0FAACF12, 0x790170D3, 0x7948B7BB, 0xCC34309D, 0xA5A63240, 0x14DD44B5, 0xE0F20471, 0xE6865375, 0xFD595A36, 0xCE990BDA, 0x12C95E92, 0x5B06F963, 0x7D955183, 0x872B3654, 0x8D0665E4, 0xFC279DED, 0x4D5EF6BE, 0x9AD0C205, 0x3EDE3B65, 0x2A1AADF2, 0x835098B5, 0xB7E8D066, 0x068D3716, 0x7A8B483D, 0xC910922C, 0xC36F0FBB, 0xC43B2F63, 0xCCA819D1, 0x9E3E7C00, 0x3170DC11, 0x6D086538, 0xEC88005D, 0x00AED685, 0xA938B7A5, 0x452432B4, 0xC2780F91},
    {0x0F024247, 0x951ED321, 0x6D49574F, 0x598CAAED, 0xD09C0B49, 0x6B55D201, 0xE7F9E2DA, 0xC26ABD16, 0xC4C1A7EB, 0x1791D6A5, 0xB3637B71, 0x2A17657E, 0x95EE8417, 0x6E13CA76, 0x953BE33D, 0x888A1412, 0xFBDFB22C, 0x5409D489, 0xDFFCB1FD, 0x5C85CF46, 0x68E3F41F, 0x10916344, 0x3920DF7A, 0xD573D166, 0xA7C012AD, 0xCD21A277, 0xF340CAFC, 0xF1FD0E17, 0xD486AD06, 0x458ED915, 0x6DF127C9, 0xEDDF5E52, 0x5815AB3C, 0x32904192, 0xD5BF6802, 0xE8BC2880, 0xE3E01AE2},
    {0xF7969B39, 0x607E7701, 0xDE888A4D, 0x0C6163FB, 0x5D397EBF, 0x67B4D2A7, 0x085780A8, 0xC732C810, 0xE1E79EDF, 0xFBE072A5, 0x1FF90795, 0xD525CE15, 0x467725B1, 0xA003D89E, 0x4A845D2B, 0x6A8907DB, 0xB985A881, 0x2DE6E8AB, 0xB9FBBD44, 0x9D39DDBA, 0xB19A67FB, 0x78CBA1D7, 0x9B1C704D, 0x883EF871, 0x8BAEF511, 0x6D62D532, 0x2B693923, 0xE133B2AE, 0x868AA2B5, 0x531AD5DE, 0x2A9C4298, 0x696AE1AC, 0x1C2AB4F3, 0x3714E891, 0x0859CA95, 0xB9A49E56, 0xD344189F},
    {0xDBA34A46, 0x525E2F5D, 0x425AC32C, 0x5C8C2998, 0x857A327F, 0xBBAD7F54, 0xDF0B977B, 0x30A8F52B, 0x5D5A87ED, 0xF3FF5165, 0x04028D39, 0x2A013C6B, 0x99864956, 0x4C92514D, 0x9B0F688A, 0xA9B3862B, 0x6090BC63, 0x8CB6F3F6, 0xFC104E02, 0x79D74F76, 0x45098B12, 0x1EC99EBB, 0x0B25C8E2, 0x760375BB, 0xB370E685, 0x22615F26, 0x52FC67B5, 0x7DA33688, 0xB8255106, 0x7A05EF2F, 0x373CFB3B, 0x290A077E, 0x43C0B57D, 0x4A370821, 0x064C1B0D, 0x8BDB6399, 0x97071926},
    {0x48AD399F, 0x0787CCA3, 0x6E1B8AF8, 0x97282AD3, 0x14EB25B8, 0x686A15E1, 0xD3C7487B, 0xB4029D97, 0x1EB7AB3B, 0x25B9E580, 0xE089C35E, 0xF296B4CA, 0xBA9A3888, 0x4ACD6363, 0x3F4857E0, 0xEA0E6553, 0xD58EB585, 0x707C9C27, 0x2F07A081, 0xC4E55800, 0x2D97C374, 0xF13FB446, 0x5A05256E, 0x0C83AB1D, 0xB3E2C494, 0x42D49027, 0xF5291785, 0x278BE1A4, 0x72D15DC4, 0x45AB3260, 0xC083C6A0, 0xFCD1863E, 0x2EDF9796, 0x4392C20E, 0x4862EA8D, 0xCF403752, 0x49D2AD36},
    {0x279950BE, 0x53EC6405, 0x96165890, 0xB091A77B, 0x1CE39EA2, 0xDC46972E, 0x480C0B39, 0x79DD8A2E, 0x215FF5BB, 0x1B6A6B0C, 0x9E5501C9, 0x5008030B, 0xAB667687, 0x1763B54F, 0x104DCD25, 0xAFFAFFAF, 0x42B8FB93, 0x6161C266, 0xBBD17E3B, 0x3359C8E8, 0xA4B9F194, 0x67D1CE13, 0xC1B04259, 0x2E70E492, 0xDC5504AC, 0x07D7E04B, 0xB9EFE988, 0xDCF3C85B, 0x426DB7D8, 0xB9EE2CF9, 0x258AC253, 0xA2D138DF, 0x57190F78, 0xEFE2E960, 0xF669F638, 0xAD0D122D, 0x586FF633},
    {0x3AA445C5, 0x82121A38, 0xA53708B0, 0x65EBB7A7, 0x7E40F2A0, 0xD45C01E0, 0x4C135B64, 0x4483C683, 0x8B908363, 0x3060F6F4, 0x8DD7B5FA, 0x110EFCED, 0x7217D5C9, 0x1F42CA09, 0xA45A6A01, 0xF45068D3, 0x899DA5D3, 0x8D9C2A6E, 0xD6BBAC2E, 0x10DF6E91, 0x49FBC07B, 0x1B6BBFBF, 0x5916C6F2, 0x3CA37EE1, 0x561EA9D4, 0xCEEFDA2C, 0x05E82C9E, 0xB8B64F9A, 0x5B8E38A4, 0x3CA4266A, 0x816927C6, 0x4E3933DC, 0x85F8A507, 0xD07445B3, 0x2C4C169B, 0x6A2C6C33, 0xF80E95C6},
    {0x10668FFE, 0xABDD162E, 0x065E1F2A, 0x41D00404, 0x9FDA8555, 0xB78413E6, 0x038AEEE2, 0x8CCF48C2, 0xB67099FD, 0x7FAA17E1, 0x02159424, 0x86C66F3E, 0x85F81B0A, 0x5D6EE998, 0xEDF1CBEC, 0xD0853AC9, 0x15D68A47, 0x930249D4, 0x421CF02D, 0xA2474696, 0xDBCAA082, 0x71DD02F7, 0x419C99D9, 0x6C2EB812, 0xC55577DF, 0x46E2C20E, 0x12F8C3D2, 0xF51C5CDD, 0x638C6841, 0xFD41B200, 0x874010CB, 0xEA7691C8, 0xAFCFE37F, 0x67F61CE6, 0x19004414, 0x0FD9CD59, 0xCB4074DC},
    {0xDE9227D2, 0xB3795C38, 0x962A57AA, 0xC0E6A70F, 0xDAC95877, 0xA294F421, 0xBAAB6840, 0xC2BC1EA8, 0x801778B0, 0xE654E1B0, 0xF355C3D7, 0xC1F9C216, 0xBE234AE7, 0x76F738D5, 0x3892A48A, 0xD43E12FA, 0x3AF53B97, 0xFA7FB3EA, 0xC2CE1C25, 0xE80C854D, 0xDD5D55BA, 0x19CAC0B3, 0x70E634CA, 0x31D75E04, 0x511CDEE9, 0x8328D6B4, 0x09DE4759, 0x345B879D, 0x8DA9C94D, 0x1EE4004E, 0x759316E6, 0xA8491E35, 0x6445CCB8, 0x15B6F7E8, 0xCB639A2C, 0xEBE83DBD, 0x90AF65E6},
    {0xF4CA0086, 0x09DA11C5, 0x6C94C74D, 0x5C5E3080, 0xB7DF63A7, 0x5F212E84, 0xF32BACDC, 0xA446472C, 0x65814C27, 0x4417784A, 0x47D4D043, 0x179BA479, 0x74D39DA2, 0xDBDC4A4C, 0x2619ADC3, 0xF6D44E53, 0x33714F0A, 0xA2488ACF, 0x3B0971B3, 0x8A7DC9ED, 0x9B3CA3EE, 0xDFD95018, 0x6CB74E4B, 0xFE088ED7, 0xE685BD03, 0x8D3CD380, 0x93B25780, 0x8556CDE9, 0x24D9AC0F, 0xF727F9FC, 0x1E31F736, 0xB232B542, 0x85357F85, 0x25076674, 0x4D02443F, 0x98313DD6, 0xE69659EC},
    {0xF6843837, 0x91E6E323, 0xA7044159, 0xFC084237, 0x83B46317, 0xEEC42D73, 0xF887CE2A, 0xB867CBFB, 0x1048E50A, 0xDC531AA9, 0x5196D992, 0x9ADC4932, 0xD7B39954, 0xD1E5368F, 0x3965BAA3, 0x798CC085, 0xF5463286, 0x0CA49F86, 0xFAED57A4, 0x1662BD6A, 0x465880D5, 0x3DCEC41E, 0x6E0B831A, 0x14A220B2, 0x4559F9AE, 0x39EC421F, 0xB9757889, 0x5B7AC1BE, 0xF6B93106, 0xE18E0A82, 0xCFD2409A, 0x11EB2351, 0x5F8C898C, 0x51B7D124, 0xFE8213F5, 0xEF315329, 0x48065118},
    {0xFDEA39E4, 0xFC060DE0, 0x27F056DF, 0x3D412695, 0x557B6E30, 0x1053AB0E, 0xD8F2D8A1, 0x8033EB16, 0x4D9F8252, 0x92DB5EB6, 0x652124DD, 0x30D35B8B, 0x3BFC52BD, 0x5142E79C, 0x46C4A360, 0x58E7F1A2, 0x60A2489E, 0x15EB320F, 0x2E206A72, 0xC3D3D391, 0x20B00E02, 0x8FCF90E0, 0xA040B0E5, 0x9F6EB05C, 0x7A29BB05, 0x1EEEE732, 0xA1CD3E44, 0x5A8AE34A, 0x7FC98E40, 0x9D1361FE, 0xA76FA685, 0xBD825724, 0xCBED467E, 0x25CCC97D, 0x0C6DAED7, 0x04F9B50B, 0xB3FC465E},
    {0xA1E9ED49, 0xBD52045C, 0x6FC6321C, 0xB3D8CAFB, 0x9675AB8B, 0x5B92E81F, 0x927BBBD0, 0xBD651670, 0xDB23F118, 0x648AB78E, 0x8A9663CE, 0xEC8D4E2D, 0x977976D9, 0x0BE30247, 0x79451F8D, 0xDC985389, 0x02830FD5, 0x62CADFA6, 0x176DC515, 0xF261AAAF, 0xB74EAD64, 0x108DC582, 0x92843B18, 0x0DC14C32, 0x60685DE2, 0x558C7F82, 0xC4BB0A82, 0xAB33A47F, 0x3BC99611, 0x2BA78298, 0xE6970964, 0x41296FC5, 0x299C2548, 0xB7FDCD2A, 0xFEB4097E, 0xBED156C2, 0xCC458AEB},
    {0x678782BA, 0xE609F550, 0xDB413C47, 0xAFF47725, 0xF0EDE797, 0x3F22BA40, 0xA858550B, 0x3CE3F3DE, 0x1EB0737C, 0x89D7C765, 0x6BF2898C, 0x3149D8E5, 0x170E5471, 0x1B531BE7, 0x46F341D1, 0x67B9D033, 0xB777E210, 0xA0030F60, 0x004C65FA, 0xF45F4906, 0x6AACD5F2, 0x3DFC77AF, 0xF23FDE19, 0x873A25C1, 0xCDCCE709, 0x46039BF5, 0x7C80C056, 0x2C7F2015, 0x54E4C8D0, 0xA3EBAC3C, 0x34E51E4B, 0x7A9A2408, 0xDD077383, 0x5C139470, 0xDFC431C0, 0x4BD6C7A3, 0x24BE74D6},
    {0x907FB31B, 0xB2E9015B, 0x6AC9D8E9, 0xB293E235, 0x1F163074, 0xF6EE902C, 0xF45831C6, 0x00A140ED, 0xB3485BB9, 0x76AC8E58, 0xD5E9BA0F, 0x1EFF75C2, 0x418B67A0, 0xE43FEE88, 0xE4D04FD0, 0x8661A694, 0x30AADC99, 0x86DF6C37, 0xFDAB09CD, 0x78EAE5B4, 0xA0853281, 0xA15D0DEB, 0xC46B5D12, 0x8D2CFB47, 0xA79F9BD5, 0xFCBCA89F, 0x0615B8DD, 0x53E2734B, 0xCA9C2F8B, 0x7240F139, 0xF3533D4D, 0x82E611CE, 0x040D1675, 0x53B62A44, 0xB6D6ED64, 0x60A78388, 0xEBA54BBD},
    {0xB51B412E, 0x7C35E35B, 0xB639210E, 0x23F2693B, 0xD4CD19BA, 0xB0EE647A, 0xB726C1F6, 0x905D60B1, 0xF04E7ED2, 0x2426915F, 0xEECCC94C, 0x4C3E330B, 0x13826C9C, 0x9964F7DE, 0x3334544A, 0x4FFCA9AB, 0xE8EE0B55, 0x233C24A3, 0x6221B29F, 0xA977BB4F, 0xF6620B48, 0xD6E41FCC, 0x64D53A97, 0x98D3F870, 0x3C4BF89F, 0xC6F26354, 0xC1FA36D7, 0xF0D3F20A, 0x54145C69, 0x39898169, 0x7D1EB796, 0x08C7CCF1, 0x54DAB192, 0x70CB8984, 0xCA81E7B3, 0x26B5179A, 0x38BD2380},
    {0x11164ABA, 0x1E157B1C, 0x8F33977D, 0x75BE4C72, 0xEDAF817F, 0x2BC84FAB, 0xD26332B7, 0x0F6B7604, 0x39A4BB17, 0x60870C55, 0xAF809D3C, 0x5F20658E, 0xFC57A8A2, 0x5463055C, 0xCF9D841A, 0x60693904, 0x3EFA8818, 0xA0C6D057, 0x4EEEEBF6, 0xFA9E8465, 0x6B0F3B99, 0xC4AD3B5B, 0xB70D55A6, 0xB16600B1, 0xFD1B2D93, 0x8184BC2C, 0xBB45115C, 0x95AD621F, 0x998F4BE0, 0x57E469B8, 0x1D9D6947, 0x1739FDD2, 0xAC2F5207, 0x09A72355, 0x102C7990, 0xB36F38E8, 0xAEE3F95E},
    {0xF0BE5584, 0xE3BAC577, 0xA67C271F, 0xFE8AD256, 0x6557E671, 0x82900BD2, 0xD8989856, 0xC2F66CDB, 0xB3D9C85C, 0x12721679, 0x3ED9E157, 0x353B5D80, 0x01139C46, 0x2D8293C9, 0x50E84CD3, 0xA200796F, 0xE920F2B5, 0x0FEA17AB, 0x37AF7CC0, 0xA5C1A47D, 0xE432E5E5, 0x652244B4, 0x3B3F9B37, 0xB45A46A9, 0x6C5C102F, 0x8AAD0736, 0x17CDA8B5, 0xE1CE973D, 0xF26987DE, 0x44F60361, 0xE02BB20C, 0x47A7C27C, 0x071A4F93, 0x322DD615, 0xF90C5A41, 0x23E51C22, 0x6168FE01},
    {0xCBC7FD98, 0xC5D2E0F2, 0x6125DE3D, 0x8945FCDA, 0x3860E540, 0xF3D2C799, 0x818D4B65, 0x8EEF66FD, 0x7B329435, 0x9953BFEF, 0xE823AF52, 0xA2B97B26, 0x3EC9D193, 0x40734D58, 0xE9E571AC, 0x4D83C5A4, 0x3BAB0D12, 0x11437879, 0x598BEC34, 0xA3F118D1, 0xE52DCCDD, 0x35EC08DE, 0x9EA69A46, 0xAE28E4CE, 0x9F32658F, 0x107F53F5, 0x7276E445, 0x716D66ED, 0xF5E2E825, 0x1A0D1A43, 0xE8B36C19, 0x867D3FC4, 0xC64C1882, 0xEA2ECBC2, 0x901E929A, 0xDA847CC8, 0x2AFF7969},
    {0x9F9F67F4, 0x0D46956C, 0xF4A7C477, 0x21B98410, 0xBD1D7BE4, 0x791A5DF3, 0xC3E5D38C, 0xD93B36DB, 0x4552C062, 0x4BEE6B1F, 0xD021B6EB, 0x33B8E793, 0xBA9DC9CB, 0xADEC6B19, 0x81486D28, 0xD33A27D1, 0x8A7F6F7B, 0x10CDA228, 0xA12790B0, 0x9F751D20, 0x40AB17BC, 0x0760B657, 0xA34D3688, 0x7F85B329, 0x70AA1876, 0xE1A109A2, 0x7231E067, 0x68801124, 0x7C7D22EA, 0xAE92C39C, 0x6F98EEB4, 0x8EEB4ABF, 0xBFB8C8C7, 0xB15DCC46, 0x61F60440, 0xAC7BE5A7, 0x5AA1402E},
    {0x342415DD, 0x1D3B1C22, 0xBB19BA2F, 0xC58DC5B0, 0x11B36D5B, 0xB5B67B50, 0x6C01EED5, 0xB1610BFC, 0x05F23A4A, 0x474CE3E5, 0x08BE4B69, 0xCAB4A0DB, 0x290AA645, 0x97EE09AB, 0xA21A05FF, 0x7E803EDF, 0x3692A08B, 0xDA49464C, 0x233C62ED, 0x2B8BD2CB, 0x2BB1C60D, 0x984D9AA4, 0xD2B7DB97, 0xF7991BE9, 0x9558DB93, 0x7248EC4B, 0xB12D0831, 0xA85DDF18, 0x0B27B05B, 0x2B6DFE29, 0x0E1462F0, 0xC9CC99AF, 0xFF89F292, 0x30BEE433, 0x0D04BE76, 0x3CD3EFCB, 0x7A68D00F},
    {0x2AB9FD4E, 0xD388FBE6, 0x0815DCCA, 0xCA773140, 0x5F72763F, 0x5AD1581E, 0x61282867, 0x3932B16C, 0xB17E4429, 0x06001140, 0xBB3F2DD6, 0x747A1C9E, 0xECB29646, 0x1CF17CD5, 0x0F909FAF, 0x6E6764B0, 0xBDB4E233, 0x77A835BB, 0x1D49A8C6, 0xE0A0C588, 0x01169042, 0x8D0136C4, 0x21037751, 0x30902A5D, 0x59357B03, 0x2AF8CA37, 0x1AC84486, 0x6B7ADA98, 0xCA6934F4, 0x1E16C29E, 0xE9B91DDD, 0xB3204082, 0xB194000A, 0xAC8E3D27, 0x8AD441F1, 0x906B6DCF, 0x97265F85},
    {0x2E5A58D9, 0x6BDA61D6, 0x05DF64D0, 0xA2B97225, 0x25798FAF, 0x9C65625B, 0x65A6AE07, 0x04CB7F39, 0xA527774C, 0x33859061, 0xAD320F04, 0xF3218C5F, 0x0DDA64DC, 0x2827BE8B, 0x7C77F660, 0x7A4C3BBF, 0xFE927CB4, 0x190C280F, 0x42B634F5, 0xDEB84CE0, 0xF05DC4E8, 0x5FB57455, 0xF6F30E95, 0x377D41E9, 0x3215BA39, 0x1725E820, 0xBFA8A318, 0x2FBF89D7, 0x33DA7C71, 0xB26EC1F2, 0xA4DA15A4, 0x8000A783, 0xAB8587D4, 0x1806E80C, 0x03D1E2A1, 0x4348B12D, 0x18F4BF3C},
    {0x14389E67, 0x5016EEEC, 0x76B6F802, 0xE3840C8E, 0x2BF72A9B, 0xF1EF0C56, 0x8B630051, 0xD553E09C, 0x2B0FB4ED, 0xD37DF1B2, 0xC999CB73, 0xAB86E2E2, 0x9624180C, 0x06947CFB, 0x3FBF289D, 0x2F374651, 0xA913A1E2, 0xCEDE77B6, 0xD077929D, 0x958FA4E8, 0x3054F6A8, 0x96DB8CC9, 0xABF35F54, 0x03F4F512, 0xD40A434B, 0xA92F1DB9, 0xDFC9431B, 0x88A7008B, 0x891F4942, 0xC76BEBE1, 0xEF3E21D4, 0x5BDD570A, 0x45C0C894, 0x59DA07B7, 0xA48C011E, 0x3E82589F, 0x6F9F4EA0},
    {0x0EF7CBC4, 0xE4846CA8, 0x369D90E4, 0xF0B8B768, 0xDE0498B8, 0xAC998767, 0xCB8C3020, 0x32AFC1E6, 0x761113D4, 0x644D556F, 0xCA53911E, 0x43C12E68, 0x514A9448, 0xA42C3D3E, 0x22396BAF, 0xADB1AADC, 0x371F9A3E, 0x5B93123A, 0xDD32ECC7, 0xB1333F88, 0x961AABAE, 0x1F52C631, 0xC102AD12, 0x6D51F880, 0x6967E3E8, 0xC2C9120D, 0x54049859, 0xD564EE7E, 0x2259051C, 0xA4D94C7D, 0xCE7724D5, 0xFD8A5B70, 0x4C4A2794, 0x86676797, 0x0EFD17A1, 0xA61D8DD2, 0xA68F774A},
    {0xB2E07D39, 0x7C8670B6, 0x5DFABCF0, 0x526C8C2D, 0x4053E94D, 0x818563BB, 0x3EA42129, 0x3353FB57, 0x92C75C7E, 0x046C5D38, 0xB054331C, 0x3797AD10, 0xBB392667, 0x517B1A0C, 0xCBE0E6E7, 0x402FD418, 0x1DA297A2, 0xC657C508, 0x0308BB0C, 0x46E8F3A5, 0x4B7EB7F5, 0xEDB4D83A, 0xD430D0C7, 0x00D30EDA, 0xCBB4E36B, 0xBF98C913, 0x83600293, 0x217B73DF, 0xA2B07FF9, 0xC57989F7, 0x8DBD48EE, 0x4BA283CA, 0x4395C8AE, 0xB955ABB0, 0x1F003456, 0x72CC786A, 0xFF80AFDC},
    {0xE7DFD8A9, 0xA0C938EB, 0x2590445F, 0xD967BB6E, 0x2686734E, 0x7959B9F2, 0x091517CE, 0x28D7115E, 0x46904894, 0x9581AD38, 0x0EDC248D, 0x9C738C76, 0x3E02CD68, 0xC463760D, 0xFA387D69, 0x70FCF8FC, 0xA7C5EB2F, 0xF7CECB25, 0xE85DF7B5, 0x844C8F74, 0xB54B0E56, 0x954868BA, 0x9F06B3B4, 0xF0BE675D, 0x9DE1DD9E, 0x5EACB7A9, 0x1BF61CDE, 0x6B58D57B, 0xA60C60AC, 0x04F50D0D, 0xCD947E88, 0xDAAF407B, 0x403D6F08, 0x7F014E16, 0x926C1BC1, 0xDD30B3E8, 0x1A986C11},
    {0xAA930E29, 0x848683C9, 0xE3AA7B87, 0x3BE5B89D, 0xD39DC7B4, 0x30E7C58C, 0xF931E1AD, 0xA2783313, 0x20672D65, 0x3F5F6F1F, 0x4DAEE61A, 0xA8E34D70, 0x34779795, 0x74649E22, 0x7F9C9D2F, 0x3840CF0A, 0x0ABA5004, 0x1033B6E5, 0xE4350FF0, 0x0068F852, 0x3180E3A2, 0x320AF903, 0x17AE2116, 0xE0C5E6B5, 0x1DABCF69, 0xE721EE3C, 0x4D60FD82, 0xC95CAC40, 0x6D43DA02, 0x1EBE34A9, 0xD58396AD, 0x13A5DBAC, 0xF4681DFF, 0xDAC4A329, 0x3F7FA690, 0x5A472781, 0x2E082433},
    {0x08497910, 0x0B5E2FA0, 0xA4DFA944, 0x6AD7A8E8, 0x7918042C, 0x453E22B8, 0x1FA5094B, 0x4B71A3C9, 0x258D12AB, 0xE71D1E8B, 0x4515C213, 0x5D50E264, 0xD223F155, 0x40F5B3A5, 0x76EDE533, 0x697792CA, 0x348677EF, 0xE19B319D, 0x4441B1EA, 0xCF270DDD, 0xAE1CF7F4, 0xE39D11F3, 0xF90BEA58, 0x702FBB25, 0xD1C6F588, 0x2F4D111E, 0x06B705BC, 0x518B5A78, 0x2B697EFE, 0x4051AD14, 0xA2035F8A, 0x32E3CE3E, 0xF0B18866, 0xCE00C274, 0xCDF904F0, 0x78064A99, 0x1424378D},
};

/** @} */

/*----------------------------------------------------------------------------*/
