#ifndef __HASH3D_H
#define __HASH3D_H

#include<vector>
#include<list>
#include"vector3d.h"

#ifdef HAVE_CONFIG_H
#include<config.h>
#endif

__BEGIN_YAFRAY

template<class T>
class hash3d_iterator;
template<class T>
class hash3d_const_iterator;

template<class T>
class hash3d_t
{
	public:

		typedef std::list< std::pair<point3d_t,T > > boxList_t;

		typedef class hash3d_iterator<T> iterator;
		typedef class hash3d_const_iterator<T> const_iterator;
		
		hash3d_t(PFLOAT cell,unsigned int size=512);
		~hash3d_t();
		void insert(const T & );
		void clear();

		const T *findExistingBox(const point3d_t &)const;
		const T *findExistingBox(int x,int y,int z)const;
		T *findExistingBox(const point3d_t &);
		T *findExistingBox(int x,int y,int z);
		
		T & findBox(const point3d_t &p) 
		{point3d_t box=getBox(p);return findCreateBox(box);};

		void getBox(const point3d_t &p,int &nx,int &ny,int &nz)const;
		point3d_t getBox(int nx,int ny,int nz)const;

		unsigned int numBoxes()const {return numBox;};
		iterator begin();
		iterator end();
		const_iterator begin()const;
		const_iterator end()const;

	protected:
		unsigned int hash3d(const point3d_t &)const;
		unsigned int hash3d(int x,int y,int z)const;
		point3d_t getBox(const point3d_t &)const;
		T & findCreateBox(const point3d_t &);

		std::vector<  boxList_t *> bucket;
		unsigned int vsize;
		PFLOAT cellsize;
		unsigned int numElements;
		unsigned int numBox;
};


template<class T>
class hash3d_iterator
{
	public:
		void init();
		void operator ++();
		void operator ++(int) {return operator ++();};
		T & operator *() {return (*l_ite).second;};

		typename std::vector< typename hash3d_t<T>::boxList_t *>::iterator v_ite;
		typename std::vector< typename hash3d_t<T>::boxList_t *>::iterator v_end;
		typename hash3d_t<T>::boxList_t::iterator l_ite;
		typename hash3d_t<T>::boxList_t::iterator l_end;
};

template<class T>
class hash3d_const_iterator
{
	public:
		void init();
		void operator ++();
		void operator ++(int) {return operator ++();};
		const T & operator *() {return (*l_ite).second;};

		typename std::vector< typename hash3d_t<T>::boxList_t *>::const_iterator v_ite;
		typename std::vector< typename hash3d_t<T>::boxList_t *>::const_iterator v_end;
		typename hash3d_t<T>::boxList_t::const_iterator l_ite;
		typename hash3d_t<T>::boxList_t::const_iterator l_end;
};

template<class T>
void hash3d_iterator<T>::operator ++()
{
	if(v_ite==v_end) return;
	l_ite++;
	if(l_ite==l_end) 
	{
		v_ite++;
		while(v_ite!=v_end)
		{
			if(*v_ite!=NULL)
			{
				l_ite=(*v_ite)->begin();
				l_end=(*v_ite)->end();
				break;
			}
			else v_ite++;
		}
	}
}

template<class T>
void hash3d_const_iterator<T>::operator ++()
{
	if(v_ite==v_end) return;
	l_ite++;
	if(l_ite==l_end) 
	{
		v_ite++;
		while(v_ite!=v_end)
		{
			if(*v_ite!=NULL)
			{
				l_ite=(*v_ite)->begin();
				l_end=(*v_ite)->end();
				break;
			}
			else v_ite++;
		}
	}
}

template<class T>
void hash3d_iterator<T>::init()
{
	while(v_ite!=v_end)
	{
		if(*v_ite!=NULL)
		{
			l_ite=(*v_ite)->begin();
			l_end=(*v_ite)->end();
			break;
		}
		else v_ite++;
	}
}

template<class T>
void hash3d_const_iterator<T>::init()
{
	while(v_ite!=v_end)
	{
		if(*v_ite!=NULL)
		{
			l_ite=(*v_ite)->begin();
			l_end=(*v_ite)->end();
			break;
		}
		else v_ite++;
	}
}

template<class T>
inline bool operator == (const hash3d_iterator<T> &a,const hash3d_iterator<T> &b)
{
	if((b.v_ite==b.v_end) || (a.v_ite==a.v_end)) return (a.v_ite==b.v_ite);
	else
		return ((a.v_ite==b.v_ite) && (a.l_ite==b.l_ite));
}

template<class T>
inline bool operator != (const hash3d_iterator<T> &a,const hash3d_iterator<T> &b)
{
	if((b.v_ite==b.v_end) || (a.v_ite==a.v_end)) 
		return (a.v_ite!=b.v_ite);
	else
		return ((a.v_ite!=b.v_ite) || (a.l_ite!=b.l_ite));
}

template<class T>
inline bool operator == (const hash3d_const_iterator<T> &a,const hash3d_const_iterator<T> &b)
{
	if((b.v_ite==b.v_end) || (a.v_ite==a.v_end)) return (a.v_ite==b.v_ite);
	else
		return ((a.v_ite==b.v_ite) && (a.l_ite==b.l_ite));
}

template<class T>
inline bool operator != (const hash3d_const_iterator<T> &a,const hash3d_const_iterator<T> &b)
{
	if((b.v_ite==b.v_end) || (a.v_ite==a.v_end)) 
		return (a.v_ite!=b.v_ite);
	else
		return ((a.v_ite!=b.v_ite) || (a.l_ite!=b.l_ite));
}

template<class T>
hash3d_t<T>::hash3d_t(PFLOAT cell,unsigned int size):bucket(size)
{
	for(unsigned int i=0;i<size;++i) bucket[i]=NULL;
	vsize=size;
	cellsize=cell;
	numElements=0;
	numBox=0;
}

template<class T>
hash3d_t<T>::~hash3d_t()
{
	int u=0;
	for(unsigned int i=0;i<vsize;++i)
		if(bucket[i]!=NULL) {delete bucket[i];u++;}
	//std::cerr<<"hash: "<<u<<"/"<<vsize<<" used";
}

template<class T>
void hash3d_t<T>::clear()
{
	for(unsigned int i=0;i<vsize;++i)
		if(bucket[i]!=NULL) {delete bucket[i];bucket[i]=NULL;}
}

template<class F>
int typeHash(F f)
{ 
  const int a = 7*7*7*7*7;
  const int m = (1<<31)-1;
  const int q = m/a;          // m = aq+r
  const int r = m % a;
	
  int seed=0;
	int acc=0;
  unsigned char *fp=(unsigned char *)&f;
  for(unsigned int i=0;i<sizeof(F);++i) acc+=fp[i];//(seed<<8)|fp[i];
	acc=acc ^ (acc>>8) ^ 0xCC;
	for(unsigned int i=0;i<sizeof(F);++i) seed=((seed<<8)|fp[i]^acc);
	seed^=seed>>1;
  seed&=m;
  int ite=(seed&7)+1;
  for(int i=0;i<ite;++i)
  {
    seed = a * (seed % q) - r * (seed/q);
  	seed&=m;
  }
  return seed;
}

template<class T>
unsigned int hash3d_t<T>::hash3d(int x,int y,int z)const
{
  unsigned int n = x + y*1301 + z*314159;
  n ^= (1664525*n + 1013904223);
	return (n^(1664525*n + 1013904223))%vsize;// /4294967296.f;
	//return (typeHash(x)+typeHash(y)+typeHash(z))%vsize;
	/*
	const int a = 7*7*7*7*7;
	const int m = (1<<31)-1;
	const int q = m/a;          // m = aq+r
	const int r = m % a;
	int seed=x;
	seed = a * (seed % q) - r * (seed/q)+y;
	if (seed < 0) seed += m;
	seed = a * (seed % q) - r * (seed/q)+z;
	if (seed < 0) seed += m;
	seed = a * (seed % q) - r * (seed/q);
	if (seed < 0) seed += m;
	return seed%vsize;
	*/
}

template<class T>
unsigned int hash3d_t<T>::hash3d(const point3d_t &v)const
{
	int x=(int)(v.x/cellsize);
	int y=(int)(v.y/cellsize);
	int z=(int)(v.z/cellsize);
	return hash3d(x,y,z);
}

template<class T>
point3d_t hash3d_t<T>::getBox(const point3d_t &p)const
{
	int nx,ny,nz;
	getBox(p,nx,ny,nz);
	point3d_t box(cellsize*(PFLOAT)nx,cellsize*(PFLOAT)ny,cellsize*(PFLOAT)nz);
	if(p.x<0) box.x+=-0.5*cellsize; else box.x+=0.5*cellsize;
	if(p.y<0) box.y+=-0.5*cellsize; else box.y+=0.5*cellsize;
	if(p.z<0) box.z+=-0.5*cellsize; else box.z+=0.5*cellsize;
	return box;
}

template<class T>
point3d_t hash3d_t<T>::getBox(int nx,int ny,int nz)const
{
	point3d_t box(cellsize*(PFLOAT)nx,cellsize*(PFLOAT)ny,cellsize*(PFLOAT)nz);
	if(box.x<0) box.x+=-0.5*cellsize; else box.x+=0.5*cellsize;
	if(box.y<0) box.y+=-0.5*cellsize; else box.y+=0.5*cellsize;
	if(box.z<0) box.z+=-0.5*cellsize; else box.z+=0.5*cellsize;
	return box;
}

template<class T>
void hash3d_t<T>::getBox(const point3d_t &p,int &nx,int &ny,int &nz)const
{
	nx=(int)(p.x/cellsize);
	ny=(int)(p.y/cellsize);
	nz=(int)(p.z/cellsize);

	if(p.x<0.0) nx--;
	if(p.y<0.0) ny--;
	if(p.z<0.0) nz--;
}
	
template<class T>
T & hash3d_t<T>::findCreateBox(const point3d_t &p)
{
	unsigned int pos=hash3d(p);
	if(bucket[pos]==NULL)
		bucket[pos]=new std::list< std::pair<point3d_t,T > >;
	for(typename boxList_t::iterator i=bucket[pos]->begin();
		i!=bucket[pos]->end();++i)
		if((*i).first==p) return (*i).second;

	std::pair<point3d_t,T> par;
	par.first=p;
	bucket[pos]->push_back(par);
	numBox++;
	return bucket[pos]->back().second;
}

template<class T>
const T * hash3d_t<T>::findExistingBox(const point3d_t &p)const
{
	point3d_t box=getBox(p);
	unsigned int pos=hash3d(box);
	if(bucket[pos]==NULL) return NULL;
	for(typename boxList_t::const_iterator i=bucket[pos]->begin();
		i!=bucket[pos]->end();++i)
		if((*i).first==box) return &((*i).second);

	return NULL;
}

template<class T>
const T * hash3d_t<T>::findExistingBox(int x,int y,int z)const
{
	point3d_t box(cellsize*(PFLOAT)x,cellsize*(PFLOAT)y,cellsize*(PFLOAT)z);
	//cout<<"("<<box<<"/"<<getBox(P)<<endl;
	if(box.x<0) box.x+=-0.5*cellsize; else box.x+=0.5*cellsize;
	if(box.y<0) box.y+=-0.5*cellsize; else box.y+=0.5*cellsize;
	if(box.z<0) box.z+=-0.5*cellsize; else box.z+=0.5*cellsize;
	unsigned int pos=hash3d(x,y,z);
	if(bucket[pos]==NULL) return NULL;
	for(typename boxList_t::const_iterator i=bucket[pos]->begin();
		i!=bucket[pos]->end();++i)
		if((*i).first==box) return &((*i).second);

	return NULL;
}

template<class T>
T * hash3d_t<T>::findExistingBox(const point3d_t &p)
{
	point3d_t box=getBox(p);
	unsigned int pos=hash3d(box);
	if(bucket[pos]==NULL) return NULL;
	for(typename boxList_t::iterator i=bucket[pos]->begin();
		i!=bucket[pos]->end();++i)
		if((*i).first==box) return &((*i).second);

	return NULL;
}

template<class T>
T * hash3d_t<T>::findExistingBox(int x,int y,int z)
{
	point3d_t box(cellsize*(PFLOAT)x,cellsize*(PFLOAT)y,cellsize*(PFLOAT)z);
	//cout<<"("<<box<<"/"<<getBox(P)<<endl;
	if(box.x<0) box.x+=-0.5*cellsize; else box.x+=0.5*cellsize;
	if(box.y<0) box.y+=-0.5*cellsize; else box.y+=0.5*cellsize;
	if(box.z<0) box.z+=-0.5*cellsize; else box.z+=0.5*cellsize;
	unsigned int pos=hash3d(x,y,z);
	if(bucket[pos]==NULL) return NULL;
	for(typename boxList_t::iterator i=bucket[pos]->begin();
		i!=bucket[pos]->end();++i)
		if((*i).first==box) return &((*i).second);

	return NULL;
}

template<class T>
void hash3d_t<T>::insert(const T &e)
{
	point3d_t box=getBox(e.position());
	T &l=findCreateBox(box);
	l=e;
	numElements++;
}

template<class T>
hash3d_iterator<T> hash3d_t<T>::begin()
{
	iterator i;
	i.v_ite=bucket.begin();
	i.v_end=bucket.end();
	i.init();
	return i;
}

template<class T>
hash3d_iterator<T> hash3d_t<T>::end()
{
	iterator i;
	i.v_ite=bucket.end();
	i.v_end=bucket.end();
	return i;
}

template<class T>
hash3d_const_iterator<T> hash3d_t<T>::begin()const
{
	const_iterator i;
	i.v_ite=bucket.begin();
	i.v_end=bucket.end();
	i.init();
	return i;
}

template<class T>
hash3d_const_iterator<T> hash3d_t<T>::end()const
{
	const_iterator i;
	i.v_ite=bucket.end();
	i.v_end=bucket.end();
	return i;
}

__END_YAFRAY

#endif
