/**
 * @file    about.c
 * @brief   Displays an "About" box.
 *
 *          Displays a window containing version and author information.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <string.h>

#include "ygraph.h"

extern GdkColor* color_set_rgb(GdkColormap*, gulong, gulong, gulong);
extern gboolean about_expose(GtkWidget*, GdkEvent*, gpointer);

/**
 * @brief    Draw a pixmap containing the program name, version number,
 *           and about box message.
 * 
 * @param    drawing_area The drawing area of the About box.
 * @param    event        The event triggering the function call.
 * @param    data         Data passed with the event (generally NULL).
 * @returns  TRUE if everything went well, otherwise crash.
 */
gboolean
about_configure(GtkWidget* drawing_area, GdkEventConfigure* event,
		gpointer data)
{
  GdkColormap* colormap;
  GdkGC* gc;
  GdkFont* font_reg;
  GdkFont* font_big;
  GdkColor* about_color;
  gchar* help_msg = HELP_MSG;
  gchar version_msg[20];
  gint version_len;
  gint version_width;
  gint version_height;
  gint copyright_len;
  gint copyright_width;
  gint copyright_height;
  gint help_msg_len;
  gint help_msg_width;
  
  g_assert(drawing_area);

  if (global_about_pixmap != NULL)
    gdk_pixmap_unref(global_about_pixmap);

  global_about_pixmap = gdk_pixmap_new(drawing_area->window,
				drawing_area->allocation.width,
				drawing_area->allocation.height,
				-1);

  g_assert(global_about_pixmap);

  colormap = gtk_widget_get_colormap(drawing_area);
  g_assert(colormap);

  gc = gdk_gc_new(global_about_pixmap);
  g_assert(gc);

  about_color = color_set_rgb(colormap, 20000, 40000, 20000);
  gdk_gc_set_foreground(gc, about_color);
  g_free(about_color);

  gdk_draw_rectangle(global_about_pixmap, gc, TRUE, 0, 0, 
		     drawing_area->allocation.width, 85);
  gdk_draw_rectangle(global_about_pixmap, drawing_area->style->black_gc, TRUE,
		     0, 85, drawing_area->allocation.width, 15);

  font_reg = gdk_font_load(DEFAULT_FONT);
  font_big = gdk_font_load(BIG_FONT);

  g_snprintf(version_msg, 20, "%s %s", PACKAGE, VERSION);
  version_len = strlen(version_msg);
  version_width = gdk_text_width(font_big, version_msg, version_len);
  version_height = gdk_text_height(font_big, version_msg, version_len);

  copyright_len = strlen(COPYRIGHT_MSG);
  copyright_width = gdk_text_width(font_reg, COPYRIGHT_MSG, copyright_len);
  copyright_height = gdk_text_height(font_reg, COPYRIGHT_MSG, copyright_len);

  help_msg_len = strlen(help_msg);
  help_msg_width = gdk_text_width(font_reg, help_msg, help_msg_len);

  gdk_draw_text(global_about_pixmap, font_big, drawing_area->style->white_gc,
		(drawing_area->allocation.width - version_width)/2,
		(65 + version_height)/2, version_msg, version_len);

  gdk_draw_text(global_about_pixmap, font_reg, drawing_area->style->white_gc,
		(drawing_area->allocation.width - copyright_width)/2,
		80, COPYRIGHT_MSG, copyright_len);

  gdk_draw_text(global_about_pixmap, font_reg, gc,
		(drawing_area->allocation.width - help_msg_width)/2,
		97, help_msg, help_msg_len);

  gdk_draw_pixmap
    (drawing_area->window,
     drawing_area->style->fg_gc[GTK_WIDGET_STATE(drawing_area)],
     global_about_pixmap,
     0, 0, 0, 0, -1, -1);

  return TRUE;
}

/**
 * @brief    Create the about box window.
 */
void
about_window_create(void)
{
  GtkWidget* about_dialog;
  GtkWidget* about_okay_button;
  GtkWidget* drawing_area;

  about_dialog = gtk_dialog_new();

  /*
   * Message.
   */
  drawing_area = gtk_drawing_area_new();
  gtk_drawing_area_size(GTK_DRAWING_AREA(drawing_area), 240, 100);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "configure_event",
		     GTK_SIGNAL_FUNC(about_configure), NULL);
  gtk_signal_connect(GTK_OBJECT(drawing_area), "expose_event",
		     GTK_SIGNAL_FUNC(about_expose), NULL);
  gtk_widget_show(drawing_area);
  g_assert(drawing_area);

  /*
   * Okay button.
   */
  about_okay_button = gtk_button_new_with_label("Okay");
  gtk_signal_connect_object(GTK_OBJECT(about_okay_button), "clicked",
			    GTK_SIGNAL_FUNC(gtk_widget_destroy), 
			    (gpointer) about_dialog);
  gtk_box_pack_end(GTK_BOX(GTK_DIALOG(about_dialog)->action_area),
		   about_okay_button, FALSE, FALSE, 0);

  gtk_container_add(GTK_CONTAINER(GTK_DIALOG(about_dialog)->vbox),
		   drawing_area);

  gtk_widget_show_all(about_dialog);

}

