/**
 * @file    export.c
 * @brief   Routines for exporting plots as images and animations.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <sys/stat.h>

#include "ygraph.h"

extern void message_dialog(gchar*);
extern void legend_draw(Plot*);
extern void plot_area_draw(Plot*);
extern void plot_window_draw_time(Plot*);
extern void plot_window_display_all(Plot*);
extern void current_frame_set(gint);
extern void current_frame_increment(gint);
extern gint frame_draw_next(GtkObject*);

/**
 * @brief    Export the current frame to an image file.
 *
 *           This routine uses the gdk_imlib library to export the
 *           screen impage displayed in the plot_area widget of the
 *           currently opened window to a file. The type of file is
 *           determined by the specified filename extension. Format
 *           specific image parameters are given default values.
 *
 * @param    plot      The current displayed plot.
 * @param    filename  The name of the output file.
 * @todo     It would be useful to be able to specify the image properties
 *           via a widget rather than set them to defaults.
 */
void
export_to_file(Plot* plot, gchar* filename)
{
  GtkWidget* plot_area;
  GdkImlibImage* img;
  GdkImlibSaveInfo* save_info;
  gint retval;

  plot_area = plot->plot_area;

  img = gdk_imlib_create_image_from_drawable(plot->pixmap, NULL,
					     0, 0,
					     plot_area->allocation.width,
					     plot_area->allocation.height);

  save_info = g_malloc(sizeof(GdkImlibSaveInfo));

  save_info->quality = 256;
  save_info->scaling = 1;
  save_info->xjustification = 0;
  save_info->yjustification = 0;
  save_info->page_size = PAGE_SIZE_A4;

  retval = gdk_imlib_save_image(img, filename, save_info);

  if (retval != 1)
    message_dialog ("Couldn't export file.");
}

/**
 * @brief    Export a sequence of frames to a directory.
 *
 *           This routine uses the gdk_imlib library to export each
 *           of the frames in a data set as an image in a seperate file.
 *           The individual files are output in the PNG format, and given
 *           a file name corresponding to their iteration number (hard
 *           maximum of six digits).
 *
 * @param    plot     The current displayed plot.
 * @param    dirname  The name of the output file.
 */
void
export_to_dir(Plot* plot, gchar* dirname)
{
  GtkWidget* plot_area;
  GdkImlibImage* img;
  GdkImlibSaveInfo* save_info;
  gchar err_msg[MAX_LINE_LENGTH];
  gchar filename[FILE_STR_SIZE];
  gint ierr;

  ierr = mkdir(dirname, S_IRWXU);

  plot_area = plot->plot_area;

  global_display_mode = ANIMATE_MODE;
  global_current_frame = FIRST_FRAME - 1;

  save_info = g_malloc(sizeof(GdkImlibSaveInfo));

  save_info->quality = 256;
  save_info->scaling = 1;
  save_info->xjustification = 0;
  save_info->yjustification = 0;
  save_info->page_size = PAGE_SIZE_A4;

  while (frame_draw_next(NULL))
    {
      g_snprintf(filename, FILE_STR_SIZE, "%s/%s_%06d.png", dirname, 
		 SAVE_FILE_BASE_NAME, global_current_frame);

      img = gdk_imlib_create_image_from_drawable(plot->pixmap, NULL,
						 0, 0,
						 plot_area->allocation.width,
						 plot_area->allocation.height);
      ierr = gdk_imlib_save_image(img, filename, save_info);

      if (ierr != 1)
	{
	  g_snprintf(err_msg, MAX_LINE_LENGTH, "Couldn't export file:\n%s",
		     filename);
	  message_dialog (err_msg);
	  return;
	}
    }
}
