/**
 * @file    special_fn.c
 * @brief   Routines for performing manipulations of plot data.
 *
 *          This file contains a number of routines for manipulating
 *          data sets in order to generate new data which are functions
 *          of raw data.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @todo    This file is becoming large and could probably be split by
 *          function.
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <math.h>
#include <string.h>

#include "ygraph.h"

extern Frame* frame_init(void);
extern DataSet* data_set_init(gchar*);
extern gint dataset_read_from_file(gchar*, gint);
extern void dataset_calc_subtract(DataSet*);
extern void dataset_calc_convergence(DataSet*);
extern void dataset_calc_derivative(DataSet*);
extern gint global_data_set_list_append(DataSet*);
extern void frame_append_data_point(Frame*, gdouble, gdouble);
extern void dataset_append_frame(DataSet*, Frame*);
extern gdouble interp(GArray*, gdouble, gint);
extern void file_select_A(GtkWidget*, GdkEvent*, Plot*);
extern void file_select_B(GtkWidget*, GdkEvent*, Plot*);
extern void file_select_1(GtkWidget*, GdkEvent*, Plot*);
extern void file_select_2(GtkWidget*, GdkEvent*, Plot*);
extern void file_select_4(GtkWidget*, GdkEvent*, Plot*);
extern gboolean file_subtract(GtkObject*, Plot*);
extern gboolean file_convergence(GtkObject*, Plot*);
extern void plot_data_append(Plot*, gint);
extern Plot* plot_data_init(GArray*);
extern void  plot_window_display_all(Plot*);


GtkWidget* file_entry_A;
GtkWidget* file_entry_B;

GtkWidget* file_entry_1;
GtkWidget* file_entry_2;
GtkWidget* file_entry_4;

GtkWidget* new_window_check;


/**
 * @brief    Call a calculation operation on a specified data set.
 *
 * @param    data_set  The DataSet to be recalculated.
 */
void
dataset_recalc(DataSet* data_set)
{
  switch(data_set->type)
    {
    case YG_DATAFILE:
      dataset_read_from_file(data_set->fname, option_read_skip_step);
      break;
    case YG_SUBTRACT:
      dataset_calc_subtract(data_set);
      break;
    case YG_CONVERGENCE:
      dataset_calc_convergence(data_set);
      break;
    case YG_DERIVATIVE:
      dataset_calc_derivative(data_set);
      break;
    }
}

/**
 * @brief    Read a pair of files and subtract individual data values.
 *
 *           Each of the files is read in turn and added to the 
 *           global_data_set_list. A new DataSet is created corresponding
 *           to the subtraction of individual frames of file0-file1.
 *           A new data set name is created based on the names of the
 *           given data sets (basically "label0 - label1"), and given to
 *           the new DataSet. The new DataSet is given the type YG_SUBTRACT,
 *           and the cmpt_set contains index references to the two DataSets
 *           which were used to generate the new DataSet.
 *
 * @param    file0  The name of the first file.
 * @param    file1  The name of the file which is to be subtracted from the
 *                  first.
 * @returns  The index of the new DataSet within the global_data_set_list
 *           if the operation was successful. Otherwise FAIL.
 */
gint
dataset_read_subtract_files(gchar* file0, gchar* file1)
{
  DataSet* data_set_0;
  DataSet* data_set_1;
  DataSet* subtract_data_set;
  gint data_set_idx_0;
  gint data_set_idx_1;
  gint subtract_data_set_idx;
  gint name_len;

  data_set_idx_0 = dataset_read_from_file(file0, option_read_skip_step);
  data_set_idx_1 = dataset_read_from_file(file1, option_read_skip_step);

  if ((data_set_idx_0 == FAIL) || (data_set_idx_1 == FAIL))
    return FAIL;
  
  data_set_0 = g_array_index(global_data_set_list, DataSet*, data_set_idx_0);
  data_set_1 = g_array_index(global_data_set_list, DataSet*, data_set_idx_1);

  subtract_data_set = data_set_init(NULL);
  subtract_data_set->type = YG_SUBTRACT;
  
  /* 6 = "( - )" */
  name_len = strlen(data_set_0->name) + strlen(data_set_1->name) + 6;
  subtract_data_set->name = g_malloc(name_len*sizeof(gchar));

  g_snprintf(subtract_data_set->name, name_len, "(%s - %s)", data_set_0->name,
	     data_set_1->name);

  g_array_append_val(subtract_data_set->cmpt_set, data_set_idx_0);
  g_array_append_val(subtract_data_set->cmpt_set, data_set_idx_1);

  dataset_calc_subtract(subtract_data_set);

  subtract_data_set_idx = global_data_set_list_append(subtract_data_set);

  return subtract_data_set_idx;
}

/**
 * @brief    Carry out the pointwise subtraction of a pair of DataSets.
 *
 *           The frame data for the given DataSet is generated by accessing
 *           each frame of the two sets specified in the cmpt_set component,
 *           and performing a subtraction of the second from the first,
 *             A - B.
 *           The resulting set has a number of frames equal to the smallest
 *           number of frames of A or B. The x-coordinates of points in the
 *           two sets need not match, as points of B are interpolated onto
 *           positions of A. The new DataSet therefore has points located
 *           at points of the A DataSet.
 *
 * @note     The time coordinates of the frames of each of the DataSets A
 *           and B are not compared to verify that they match.
 *
 * @param    data_set  A DataSet whose cmpt_set points to a pair of DataSets
 *                     to be subtracted.
 */
void
dataset_calc_subtract(DataSet* data_set)
{
  DataSet* s0;
  DataSet* s1;
  Frame* f0;
  Frame* f1;
  Frame* new_frame;
  gdouble* xy_point;
  gdouble x_val;
  gdouble y_val_0;
  gdouble y_val_1;
  gint s0_idx;
  gint s1_idx;
  gint i;
  gint j;

  if ((data_set->type != YG_SUBTRACT) || (data_set->cmpt_set->len != 2))
    return;
 
  s0_idx = g_array_index(data_set->cmpt_set, gint, 0);
  s1_idx = g_array_index(data_set->cmpt_set, gint, 1);

  s0 = g_array_index(global_data_set_list, DataSet*, s0_idx);
  s1 = g_array_index(global_data_set_list, DataSet*, s1_idx);

 if (data_set->frame != NULL)
    g_array_free(data_set->frame, TRUE);

  data_set->frame = g_array_new(FALSE, FALSE, sizeof(Frame*));
  data_set->nframes = 0;

  for (i=0; ((i < s0->nframes) && (i < s1->nframes)); ++i)
    {
      f0 = g_array_index(s0->frame, Frame*, i);
      f1 = g_array_index(s1->frame, Frame*, i);

      new_frame = frame_init();
      new_frame->time = f0->time;

      for (j=0; j<f0->npoints; ++j)
	{
	  xy_point = g_array_index(f0->xy_data, gdouble*, j);

	  x_val = xy_point[0];

	  y_val_0 = xy_point[1];
	  y_val_1 = interp (f1->xy_data, x_val, INTERP_ORDER);

	  frame_append_data_point(new_frame, x_val, y_val_0 - y_val_1);
	}

      dataset_append_frame(data_set, new_frame);
    }
}


/**
 * @brief    Read three files and perform a convergence calculation on the
 *           data.
 *
 *           Each of the files is read in turn and added to the 
 *           global_data_set_list. A new DataSet is created corresponding
 *           to a convergence test calculation on the three files.
 *
 *           The new DataSet is given the type YG_CONVERGENCE,
 *           and the cmpt_set contains index references to the three DataSets
 *           which were used to generate the new DataSet.
 *
 * @param    file1  The name of the low resolution file.
 * @param    file2  The name of the medium resolution file.
 * @param    file4  The name of the high resolution file.
 * @returns  The index of the new DataSet within the global_data_set_list
 *           if the operation was successful. Otherwise FAIL.
 */
gint
dataset_read_convergence_files(gchar* file1, gchar* file2, gchar* file4)
{
  DataSet* data_set_1;
  DataSet* data_set_2;
  DataSet* data_set_4;
  DataSet* convergence_data_set;
  gint data_set_idx_1;
  gint data_set_idx_2;
  gint data_set_idx_4;
  gint convergence_data_set_idx;

  data_set_idx_1 = dataset_read_from_file(file1, option_read_skip_step);
  data_set_idx_2 = dataset_read_from_file(file2, option_read_skip_step);
  data_set_idx_4 = dataset_read_from_file(file4, option_read_skip_step);
  
  if ((data_set_idx_1 == FAIL) || (data_set_idx_2 == FAIL) 
      || (data_set_idx_4 == FAIL))
    return FAIL;
  
  data_set_1 = g_array_index(global_data_set_list, DataSet*, data_set_idx_1);
  data_set_2 = g_array_index(global_data_set_list, DataSet*, data_set_idx_2);
  data_set_4 = g_array_index(global_data_set_list, DataSet*, data_set_idx_4);

  convergence_data_set = data_set_init(NULL);
  convergence_data_set->type = YG_CONVERGENCE;
  
  convergence_data_set->name = g_strdup(CONVERGENCE_LABEL);

  g_array_append_val(convergence_data_set->cmpt_set, data_set_idx_1);
  g_array_append_val(convergence_data_set->cmpt_set, data_set_idx_2);
  g_array_append_val(convergence_data_set->cmpt_set, data_set_idx_4);

  dataset_calc_convergence(convergence_data_set);

  convergence_data_set_idx = global_data_set_list_append(convergence_data_set);

  return convergence_data_set_idx;
}

/**
 * @brief    Determine the pointwise convergence exponent for a DataSet
 *           consisting of three files.
 *
 *           The frame data for the given DataSet is generated by accessing
 *           each frame of the three sets specified in the cmpt_set component.
 *           The operation
 *               (y1 - y2) / (y2 - y4)
 *           is carried out on each point, where y1, y2, and y4 are points
 *           from the low, medium and high resolution DataSets respectively.
 *
 *           The resulting DataSet will have a number of frames equal to
 *           the number of frames in the low-res DataSet that are between the
 *           latest start-time and earliest end-time of the three sets.
 *
 *           Points of the new DataSet are those of the low-res DataSet.
 *
 * @todo     There is currently no check as to whether the sets indeed specify
 *           data of respectively doubled resolution. Either such a check
 *           should be added, or a more general function allowing for three
 *           arbitrary resolutions added.
 * @bug      It looks the function does not check that frames are taken from
 *           corresponding times. That would be pretty wrong.
 * @bug      This function doesn't seem to be working at the moment.
 *
 * @param    data_set  A DataSet whose cmpt_set points to a pair of DataSets
 *                     to be subtracted.
 */
void
dataset_calc_convergence(DataSet* data_set)
{
  DataSet* s1;
  DataSet* s2;
  DataSet* s4;
  Frame* f1;
  Frame* f2;
  Frame* f4;
  Frame* new_frame;
  gdouble* xy_point;
  gdouble x_val;
  gdouble y_val_1;
  gdouble y_val_2;
  gdouble y_val_4;
  gdouble ctest;
  gdouble x_min;
  gdouble x_max;
  gint s1_idx;
  gint s2_idx;
  gint s4_idx;
  gint i;
  gint j;

  if ((data_set->type != YG_CONVERGENCE) || (data_set->cmpt_set->len != 3))
    return;
 
  s1_idx = g_array_index(data_set->cmpt_set, gint, 0);
  s2_idx = g_array_index(data_set->cmpt_set, gint, 1);
  s4_idx = g_array_index(data_set->cmpt_set, gint, 2);

  s1 = g_array_index(global_data_set_list, DataSet*, s1_idx);
  s2 = g_array_index(global_data_set_list, DataSet*, s2_idx);
  s4 = g_array_index(global_data_set_list, DataSet*, s4_idx);

  if (data_set->frame != NULL)
    g_array_free(data_set->frame, TRUE);

  data_set->frame = g_array_new(FALSE, FALSE, sizeof(Frame*));
  data_set->nframes = 0;

  for (i=0; ((i < s1->nframes) && (i < s2->nframes) && (i < s4->nframes)); ++i)
    {
      f1 = g_array_index(s1->frame, Frame*, i);
      f2 = g_array_index(s2->frame, Frame*, i);
      f4 = g_array_index(s4->frame, Frame*, i);

      new_frame = frame_init();
      new_frame->time = f1->time;

      x_min = MAX(MAX(f1->x_range[0], f2->x_range[0]), f4->x_range[0]);
      x_max = MIN(MIN(f1->x_range[1], f2->x_range[1]), f4->x_range[1]);

      for (j=0; j<f1->npoints; ++j)
	{
	  xy_point = g_array_index(f1->xy_data, gdouble*, j);

	  x_val = xy_point[0];

	  if ((x_val >= x_min) && (x_val <= x_max))
	    {
	      y_val_1 = xy_point[1];
	      y_val_2 = interp (f2->xy_data, x_val, INTERP_ORDER);
	      y_val_4 = interp (f4->xy_data, x_val, INTERP_ORDER);
	      
	      ctest = (y_val_1 - y_val_2)/(y_val_2 - y_val_4);

	      frame_append_data_point(new_frame, x_val, ctest);
	    }
	}

      dataset_append_frame(data_set, new_frame);
    }
}

/**
 * @brief    Take the derivative of the given frame.
 *
 *           Does a centred finite difference at each interior point of a
 *           frame. One-sided differences are done at the boundaries.
 *
 * @param    f  The Frame to be differenced.
 * @returns  A new Frame containing the derivative data.
 */
Frame*
frame_derivative(Frame* f)
{
  Frame* df;
  gdouble* xypoint_m;
  gdouble* xypoint_p;
  gdouble xval_m;
  gdouble xval_p;
  gdouble yval_m;
  gdouble yval_p;
  gdouble xval;
  gint i;

  df = frame_init();
  df->time = f->time;

  xypoint_m = g_array_index(f->xy_data, gdouble*, 0);
  xval_m = xypoint_m[0];
  yval_m = xypoint_m[1];

  xypoint_p = g_array_index(f->xy_data, gdouble*, 0);
  xval_p = xypoint_p[0];
  yval_p = xypoint_p[1];

  frame_append_data_point(df, xval_m, 0.0);

  for (i=1; i<f->npoints-1; ++i)
    {
      xypoint_m = g_array_index(f->xy_data, gdouble*, i-1);
      xval_m = xypoint_m[0];
      yval_m = xypoint_m[1];

      xval = g_array_index(f->xy_data, gdouble*, i)[0];

      xypoint_p = g_array_index(f->xy_data, gdouble*, i+1);
      xval_p = xypoint_p[0];
      yval_p = xypoint_p[1];
      
      frame_append_data_point(df, xval,
			      (yval_p - yval_m) / (xval_p - xval_m));
    }

  xypoint_m = g_array_index(f->xy_data, gdouble*, f->npoints-2);
  xval_m = xypoint_m[0];
  yval_m = xypoint_m[1];

  xypoint_p = g_array_index(f->xy_data, gdouble*, f->npoints-1);
  xval_p = xypoint_p[0];
  yval_p = xypoint_p[1];

  frame_append_data_point(df, xval_p, (yval_p - yval_m) / (xval_p - xval_m));

  return(df);
}

/**
 * @brief    Take the derivative of each frame in a DataSet.
 *
 *           A new DataSet is created by taking the derivative of
 *           each frame of an existing DataSet. The new DataSet is
 *           added to the global_data_set_list, and given a name
 *           "D<label>" where <label> is the label of the original DataSet.
 *           The new DataSet is given the type YG_DERIVATIVE, and its
 *           cmpt_set points to the original data.
 *
 * @param    idx  The DataSet index of the original data set within the
 *                global_data_set_list.
 * @returns  The newly created DataSet containing the derivative data.
 */
DataSet*
dataset_derivative(gint idx)
{
  Frame* f;
  Frame* df;
  DataSet* data;
  DataSet* d_data;
  gint i;
  gint name_len;
  
  data = g_array_index(global_data_set_list, DataSet*, idx);

  d_data = data_set_init(NULL);
  d_data->type = YG_DERIVATIVE;
  g_array_append_val(d_data->cmpt_set, idx);

  name_len = strlen(data->name) + 2;
  d_data->name = g_malloc(name_len*sizeof(gchar));
  g_snprintf(d_data->name, name_len, "D%s", data->name);

  for (i=0; i<data->nframes; ++i)
    {
      f = g_array_index(data->frame, Frame*, i);
      df = frame_derivative(f);
      dataset_append_frame(d_data, df);
    }

  return d_data;
}

/**
 * @brief    Call the derivative calculator on a given DataSet.
 *
 *           Takes the derivative at each point of an existing DataSet.
 *
 * @param    data_set  The DataSet whose derivative is to be taken.
 */
void
dataset_calc_derivative(DataSet* data_set)
{
  DataSet* s;
  Frame* f;
  Frame* df;
  gint s_idx;
  gint i;

  if ((data_set->type != YG_DERIVATIVE) || (data_set->cmpt_set->len != 1))
    return;
 
  s_idx = g_array_index(data_set->cmpt_set, gint, 0);
  s = g_array_index(global_data_set_list, DataSet*, s_idx);

 if (data_set->frame != NULL)
    g_array_free(data_set->frame, TRUE);

  data_set->frame = g_array_new(FALSE, FALSE, sizeof(Frame*));
  data_set->nframes = 0;

  for (i=0; i < s->nframes; ++i)
    {
      f = g_array_index(s->frame, Frame*, i);
      df = frame_derivative(f);
      dataset_append_frame(data_set, df);
    }
}

/**
 * @brief    Creates a dialog box for loading a pair of files to be subtracted.
 *
 * @param    plot    The calling Plot.
 * @param    action  The calling action (unused).
 * @param    button  The calling button (unused).
 */
void
subtract_select_dialog_create(Plot* plot, gint action, GtkItem* button)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* hbox_A;
  GtkWidget* file_label_A;
  GtkWidget* file_button_A;
  GtkWidget* hbox_B;
  GtkWidget* file_label_B;
  GtkWidget* file_button_B;
  GtkWidget* okay_button;
  GtkWidget* cancel_button;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), SUBTRACT_DIALOG_TITLE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  label = gtk_label_new(SUBTRACT_MESSAGE);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  hbox_A = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_A);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_A, FALSE,
		     FALSE, 0);

  file_label_A = gtk_label_new(FILE_A_LABEL);
  gtk_widget_show(file_label_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_label_A, FALSE, FALSE, 0);

  file_entry_A = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_A), FILE_STR_SIZE);
  gtk_widget_show(file_entry_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_entry_A, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_A, FILE_ENTRY_LENGTH, -2);

  file_button_A = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_A), "clicked",
		     GTK_SIGNAL_FUNC(file_select_A), plot);
  gtk_widget_show(file_button_A);
  gtk_box_pack_start(GTK_BOX(hbox_A), file_button_A, FALSE, FALSE, 0);

  hbox_B = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_B);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_B, FALSE,
		     FALSE, 0);

  file_label_B = gtk_label_new(FILE_B_LABEL);
  gtk_widget_show(file_label_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_label_B, FALSE, FALSE, 0);

  file_entry_B = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_B), FILE_STR_SIZE);
  gtk_widget_show(file_entry_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_entry_B, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_B, FILE_ENTRY_LENGTH, -2);

  file_button_B = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_B), "clicked",
		     GTK_SIGNAL_FUNC(file_select_B), NULL);
  gtk_widget_show(file_button_B);
  gtk_box_pack_start(GTK_BOX(hbox_B), file_button_B, FALSE, FALSE, 0);

  new_window_check = gtk_check_button_new_with_label(NEW_WINDOW_LABEL);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), new_window_check,
		     FALSE, FALSE, 0);
  gtk_widget_show(new_window_check);

  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(okay_button),
		     "clicked", GTK_SIGNAL_FUNC(file_subtract), plot);
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);
  gtk_widget_show(okay_button);

  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     cancel_button);
  gtk_widget_show(cancel_button);

  gtk_widget_show_all (dialog);
}

/**
 * @brief    Read the file selection for the first subtract file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_A(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_A), selection);
  gtk_widget_show(file_entry_A);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the first subtract file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling even.
 * @param    plot           The calling Plot data.
 */
void
file_select_A(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_entry_set_A), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Read the file selection for the second subtract file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_B(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_B), selection);
  gtk_widget_show(file_entry_B);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the first subtract file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling even.
 * @param    plot           The calling Plot data.
 */
void
file_select_B(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_entry_set_B), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Prompts the user for a pair of files to subtract, then performs
 *           and plots the subtraction.
 *
 * @param    ok_button  The calling button.
 * @param    plot       The calling plot.
 * @returns  TRUE if things worked out.
 */
gboolean
file_subtract(GtkObject* ok_button, Plot* plot)
{
  Plot* new_plot;
  GArray* data;
  gchar* file_A;
  gchar* file_B;
  gint data_idx;
  gboolean new_window;

  file_A = gtk_entry_get_text(GTK_ENTRY(file_entry_A));
  file_B = gtk_entry_get_text(GTK_ENTRY(file_entry_B));

  new_window = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(new_window_check));
  
  data_idx = dataset_read_subtract_files(file_A, file_B);

  if (data_idx == FAIL)
    return TRUE;

  if (new_window)
    {
      data = g_array_new(FALSE, FALSE, sizeof(gint));
      g_array_append_val(data, data_idx);
      new_plot = plot_data_init(data);
      gtk_widget_show(new_plot->window);
    }
  else
    {
      plot_data_append(plot, data_idx);
      plot_window_display_all(plot);
    }

  return TRUE;
}

/**
 * @brief    Creates a dialog box for loading convergence test files.
 *
 * @param    plot    The calling Plot.
 * @param    action  The calling action (unused).
 * @param    button  The calling button (unused).
 */
void
convergence_select_dialog_create(Plot* plot, gint action, GtkItem* button)
{
  GtkWidget* dialog;
  GtkWidget* label;
  GtkWidget* hbox_1;
  GtkWidget* file_label_1;
  GtkWidget* file_button_1;
  GtkWidget* hbox_2;
  GtkWidget* file_label_2;
  GtkWidget* file_button_2;
  GtkWidget* hbox_4;
  GtkWidget* file_label_4;
  GtkWidget* file_button_4;
  GtkWidget* okay_button;
  GtkWidget* cancel_button;

  dialog = gtk_dialog_new();
  gtk_window_set_title(GTK_WINDOW(dialog), CONVERGENCE_DIALOG_TITLE);
  gtk_window_set_policy(GTK_WINDOW(dialog), FALSE, FALSE, TRUE);

  label = gtk_label_new(CONVERGENCE_MESSAGE);
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->vbox), label);

  hbox_1 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_1);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_1, FALSE,
		     FALSE, 0);

  file_label_1 = gtk_label_new(FILE_1_LABEL);
  gtk_widget_show(file_label_1);
  gtk_box_pack_start(GTK_BOX(hbox_1), file_label_1, FALSE, FALSE, 0);

  file_entry_1 = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_1), FILE_STR_SIZE);
  gtk_widget_show(file_entry_1);
  gtk_box_pack_start(GTK_BOX(hbox_1), file_entry_1, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_1, FILE_ENTRY_LENGTH, -2);

  file_button_1 = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_1), "clicked",
		     GTK_SIGNAL_FUNC(file_select_1), plot);
  gtk_widget_show(file_button_1);
  gtk_box_pack_start(GTK_BOX(hbox_1), file_button_1, FALSE, FALSE, 0);

  hbox_2 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_2);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_2, FALSE,
		     FALSE, 0);

  file_label_2 = gtk_label_new(FILE_2_LABEL);
  gtk_widget_show(file_label_2);
  gtk_box_pack_start(GTK_BOX(hbox_2), file_label_2, FALSE, FALSE, 0);

  file_entry_2 = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_2), FILE_STR_SIZE);
  gtk_widget_show(file_entry_2);
  gtk_box_pack_start(GTK_BOX(hbox_2), file_entry_2, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_2, FILE_ENTRY_LENGTH, -2);

  file_button_2 = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_2), "clicked",
		     GTK_SIGNAL_FUNC(file_select_2), NULL);
  gtk_widget_show(file_button_2);
  gtk_box_pack_start(GTK_BOX(hbox_2), file_button_2, FALSE, FALSE, 0);

  hbox_4 = gtk_hbox_new(FALSE, 0);
  gtk_widget_show(hbox_4);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), hbox_4, FALSE,
		     FALSE, 0);

  file_label_4 = gtk_label_new(FILE_4_LABEL);
  gtk_widget_show(file_label_4);
  gtk_box_pack_start(GTK_BOX(hbox_4), file_label_4, FALSE, FALSE, 0);

  file_entry_4 = gtk_entry_new();
  gtk_entry_set_max_length(GTK_ENTRY(file_entry_4), FILE_STR_SIZE);
  gtk_widget_show(file_entry_4);
  gtk_box_pack_start(GTK_BOX(hbox_4), file_entry_4, FALSE, FALSE, 0);
  gtk_widget_set_usize(file_entry_4, FILE_ENTRY_LENGTH, -2);

  file_button_4 = gtk_button_new_with_label(SELECT_LABEL);
  gtk_signal_connect(GTK_OBJECT(file_button_4), "clicked",
		     GTK_SIGNAL_FUNC(file_select_4), plot);
  gtk_widget_show(file_button_4);
  gtk_box_pack_start(GTK_BOX(hbox_4), file_button_4, FALSE, FALSE, 0);

  new_window_check = gtk_check_button_new_with_label(NEW_WINDOW_LABEL);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), new_window_check,
		     FALSE, FALSE, 0);
  gtk_widget_show(new_window_check);

  okay_button = gtk_button_new_with_label(OKAY_BUTTON_LABEL);
  gtk_signal_connect(GTK_OBJECT(okay_button),
		     "clicked", GTK_SIGNAL_FUNC(file_convergence), plot);
  gtk_signal_connect_object (GTK_OBJECT (okay_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     okay_button);
  gtk_widget_show(okay_button);

  cancel_button = gtk_button_new_with_label(CANCEL_BUTTON_LABEL);
  gtk_signal_connect_object (GTK_OBJECT (cancel_button), "clicked",
			     GTK_SIGNAL_FUNC (gtk_widget_destroy),
			     GTK_OBJECT(dialog));
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG(dialog)->action_area),
		     cancel_button);
  gtk_widget_show(cancel_button);

  gtk_widget_show_all (dialog);
}

/**
 * @brief    Read the file selection for the low-res convergence file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_1(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_1), selection);
  gtk_widget_show(file_entry_1);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the low-res convergence file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling event.
 * @param    plot           The calling Plot data.
 */
void
file_select_1(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_entry_set_1), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Read the file selection for the medium-res convergence file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_2(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_2), selection);
  gtk_widget_show(file_entry_2);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the medium-res convergence file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling event.
 * @param    plot           The calling Plot data.
 */
void
file_select_2(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_entry_set_2), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Read the file selection for the high-res convergence file.
 *
 * @param    ok_button  The caller's Okay button.
 * @param    fs         The calling file selector.
 * @returns  TRUE if successful, otherwise trouble.
 */
gboolean
file_entry_set_4(GtkObject* ok_button, GtkFileSelection* fs)
{
  gchar* selection;

  selection = gtk_file_selection_get_filename(GTK_FILE_SELECTION(fs));
  gtk_entry_set_text(GTK_ENTRY(file_entry_4), selection);
  gtk_widget_show(file_entry_4);

  return TRUE;
}

/**
 * @brief    Creates a file selector for the high-res convergence file.
 * 
 * @param    select_button  The calling button.
 * @param    event          The calling event.
 * @param    plot           The calling Plot data.
 */
void
file_select_4(GtkWidget* select_button, GdkEvent* event, Plot* plot)
{
  GtkWidget* fs;

  fs = gtk_file_selection_new(FILE_SELECTION_TITLE);

  gtk_signal_connect(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
		     "clicked", GTK_SIGNAL_FUNC(file_entry_set_4), fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->ok_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_signal_connect_object(GTK_OBJECT(GTK_FILE_SELECTION(fs)->cancel_button),
			    "clicked", GTK_SIGNAL_FUNC(gtk_widget_destroy),
			    (gpointer) fs);
  gtk_widget_show(fs);
}

/**
 * @brief    Prompts the user for a set of low, medium and high resolution
 *           files on which to perform a convergence test, then performs
 *           and plots the results of the test.
 *
 * @param    ok_button  The calling button.
 * @param    plot       The calling plot.
 * @returns  TRUE if things worked out.
 */
gboolean
file_convergence(GtkObject* ok_button, Plot* plot)
{
  Plot* new_plot;
  GArray* data;
  gchar* file_1;
  gchar* file_2;
  gchar* file_4;
  gint data_idx;
  gboolean new_window;

  file_1 = gtk_entry_get_text(GTK_ENTRY(file_entry_1));
  file_2 = gtk_entry_get_text(GTK_ENTRY(file_entry_2));
  file_4 = gtk_entry_get_text(GTK_ENTRY(file_entry_4));

  new_window = gtk_toggle_button_get_active
    (GTK_TOGGLE_BUTTON(new_window_check));
  
  data_idx = dataset_read_convergence_files(file_1, file_2, file_4);

  if (data_idx == FAIL)
    return TRUE;

  if (new_window)
    {
      data = g_array_new(FALSE, FALSE, sizeof(gint));
      g_array_append_val(data, data_idx);
      new_plot = plot_data_init(data);
      gtk_widget_show(new_plot->window);
    }
  else
    {
      plot_data_append(plot, data_idx);
      plot_window_display_all(plot);
    }

  return TRUE;
}
