/**
 * @file    ygraph.h
 * @brief   Main header file.
 *
 *          Defines useful global macros.
 *
 * @author  Denis Pollney
 * @date    1 Oct 2001
 *
 * @par Copyright (C) 2001-2002 Denis Pollney
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 * @par
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details
 * @par
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#ifndef __YGRAPH_H__
#define __YGRAPH_H__

#ifndef __G_LIB_H__
#  include <glib.h>
#endif

#ifndef __GDK_H__
#  include <gdk/gdk.h>
#endif

#ifndef __GTK_H__
#  include <gtk/gtk.h>
#endif

#ifndef __GDK_IMLIB_H__
#  include <gdk_imlib.h>
#endif

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif


/*
 * Constants.
 */

/* window labels */
#define CONTROL_PANEL_NAME              "ygraph-control"
#define FILE_SELECTION_TITLE            "read file"
#define EXPORT_TO_FILE_TITLE            "export file"
#define EXPORT_TO_DIR_TITLE             "export directory"
#define ABOUT_TITLE                     "about"
#define MESSAGE_TITLE                   "message ..."
#define SUBTRACT_DIALOG_TITLE           "subtract: A - B"
#define SUBTRACT_MESSAGE                "Subtract files: A - B"
#define CONVERGENCE_DIALOG_TITLE        "convergence test"
#define CONVERGENCE_MESSAGE             "Convergence test"
#define EXPORT_FILE_SELECT_MESSAGE      "Export current frame to file"
#define EXPORT_FILE_LABEL               "File:"
#define EXPORT_DIR_SELECT_MESSAGE       "Export multiple frames to directory"
#define EXPORT_DIR_LABEL                "Directory:"
#define FILE_A_LABEL                    "File A:"
#define FILE_B_LABEL                    "File B:"
#define FILE_1_LABEL                    "File 1 (1xres):"
#define FILE_2_LABEL                    "File 2 (2xres):"
#define FILE_4_LABEL                    "File 3 (4xres):"
#define NEW_WINDOW_LABEL                "plot in new window"
#define RANGE_DIALOG_TITLE              "xy range"
#define XMIN_LABEL                      "x min"
#define XMAX_LABEL                      "x max"
#define YMIN_LABEL                      "y min"
#define YMAX_LABEL                      "y max"
#define SAVE_FILE_BASE_NAME             "frame"

/* button labels */
#define STEP_BACK_BUTTON_LABEL          "<Step"
#define STEP_FWD_BUTTON_LABEL           "Step>"
#define START_BUTTON_LABEL              "<<Start"
#define END_BUTTON_LABEL                "End>>"
#define PLAY_BUTTON_LABEL               "Play"
#define PAUSE_BUTTON_LABEL              "Pause"
#define TIME_LABEL                      "Time"
#define DELAY_LABEL                     "Delay"
#define SHOW_ALL_BUTTON_LABEL           "Show All"
#define RELOAD_BUTTON_LABEL             "Reload"
#define NEW_BUTTON_LABEL                "New plot"
#define QUIT_BUTTON_LABEL               "Quit"
#define OKAY_BUTTON_LABEL               "Okay"
#define APPLY_BUTTON_LABEL              "Apply"
#define CANCEL_BUTTON_LABEL             "Cancel"
#define SELECT_LABEL                    "Select ..."

#define TIME_STR_SIZE                   10
#define DELAY_STR_SIZE                  6
#define FILE_STR_SIZE                   255
#define FILE_ENTRY_LENGTH               120
#define RANGE_VAL_SIZE                  12

#define FIRST_FRAME                     0
#define TIME_EPSILON                    1e-10   

#define X_PADDING                       20
#define Y_PADDING                       20
#define DEFAULT_PLOT_FONT   "-adobe-helvetica-bold-r-normal-*-*-80-*-*-*-*-*-*"
#define GRID_INTENSITY                  40000

#define OPTIMUM_TICK_MARK_NUMBER        6
#define SHORT_TICK_LENGTH               5
#define LONG_TICK_LENGTH                8
#define MAX_AXIS_LABEL_LENGTH           40
#define AXIS_EPSILON                    1e-6
#define DEFAULT_AXIS_FONT   "-adobe-helvetica-bold-r-normal-*-*-80-*-*-*-*-*-*"

#define TEXT_X_WIDTH_PADDING            2
#define TEXT_X_HEIGHT_PADDING           2
#define TEXT_Y_WIDTH_PADDING            2
#define TEXT_Y_HEIGHT_PADDING           2

#define MESSAGE_X_PADDING               20
#define MESSAGE_Y_PADDING               20

#define DEFAULT_LEGEND_WIDTH            80
#define LEGEND_WIDTH_PADDING            5
#define LEGEND_HEIGHT_PADDING           2
#define LEGEND_LINE_DIVISIONS           20
#define LEGEND_LINE_THICKNESS           4
#define LEGEND_OFF                      0
#define LEGEND_RIGHT                    1
#define LEGEND_TOP                      2
#define LEGEND_OVER                     3
#define DEFAULT_LEGEND_FONT "-adobe-helvetica-bold-r-normal-*-*-80-*-*-*-*-*-*"
#define ALPHABET "abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYX0123456789"

#define MAX_LINE_LENGTH                 255
#define TIME_STR_MAX_SIZE               20
#define RANGE_STR_MAX_SIZE              80

#define DEFAULT_I_SIZE                  400
#define DEFAULT_J_SIZE                  400
#define DEFAULT_LINE_WIDTH              1
#define DOT_SIZE                        4

#define NAN_STRING                      "nan nanq inf"
#define COMMENT_DELIMETER               '\"'
#define CMD_LINE_RANGE_DELIMETER        ","
#define TIME_INDICATOR_STRING           "time ="
#define LABEL_INDICATOR_STRING          "label ="
#define UNINITIALISED_TIME              -G_MAXDOUBLE

#define FILENAME_SIZE                   255

#define NULLC                           '\0'

#define NCOLORS                         7

#define NO_ZOOM                         -1

#define INIT_DATA                       0
#define APPEND_DATA                     1

#define DEFAULT_FONT        "-adobe-helvetica-bold-r-normal-*-*-80-*-*-*-*-*-*"
#define BIG_FONT           "-adobe-helvetica-bold-r-normal-*-*-140-*-*-*-*-*-*"

#define DEFAULT_ANIMATE_DELAY           100 /* milliseconds */
#define DEFAULT_SKIP_STEP               1

#define READ_COMMENT                    0
#define READ_DATA                       1

#define NEW_DATA_SET                    -1
#define FAIL                            -1
#define SUCCESS                          0

#define INTERP_ORDER                     1
#define MAX_INTERP_ORDER                 1

/* Special plot types */

#define YG_DATAFILE                      0
#define YG_SUBTRACT                      1
#define YG_CONVERGENCE                   2
#define YG_DERIVATIVE                    3

#define CONVERGENCE_LABEL                "convergence"

#define HELP_MSG                         "Type 'ygraph --help' for help"
#define COPYRIGHT_MSG             "(c) 2002 Denis Pollney <pollney@aei.mpg.de>"

/**
 * @defgroup macros Macros.
 * @{
 */

/**
 * A macro to set individual array elements is missing from glib, but this
 * macro does the job.
 */
#define array_index_set_val(a,t,i,v)     (((t*) (a)->data) [(i)]) = v

/** @} */

/**
 * @defgroup enumerations Enumerations.
 * @{
 */

/**
 * Possible modes for displaying the plot area.
 */
typedef enum
  {
    SHOW_ALL_MODE,                   /**< show all frames of all datasets    */
    ANIMATE_MODE,                    /**< loop through the frames            */
    PAUSE_MODE                       /**< show a single frame of each dataset*/
  } DISPLAY_MODE;

/**
 * Flags to distinguish between the two axis types.
 */
typedef enum
  {
    X_AXIS,                          /**< x-axis type                        */
    Y_AXIS                           /**< y-axis type                        */
  } AXIS_TYPE;

/** @} */

/**
 * @defgroup structs Data structures.
 * @{
 */

/**
 * @brief    Describes an individual tick-mark on an axis.
 */
struct _Tick
{
  gdouble val;                       /**< the coordinate value of the tick   */
  gchar* str;                        /**< same value, converted to a string  */
};
typedef struct _Tick Tick;

/**
 * @brief    Describes the x and y axes of a plot.
 */
struct _Axis
{
  GdkFont* font;                     /**< the plot axis font                 */
  GdkGC* gc;                         /**< graphics context for the axis      */
  GArray* ticks;                     /**< a list of tick mark locations      */
  gint type;                         /**< either X_AXIS or Y_AXIS            */
  gint width;                        /**< width needed by the axis (pixels)  */
  gint height;                       /**< height needed by the axis(pixels)  */
};
typedef struct _Axis Axis;

/**
 * @brief    Describes the contents of the plot legend.
 */
struct _Legend
{
  GdkFont* font;                     /**< font for the legend                */
  GdkGC* gc;                         /**< graphics context for the legend    */
  GdkPixmap* pixmap;                 /**< holds an image of the legend       */
  GArray* names;                     /**< list of data set names to be drawn */
  gint width;                        /**< width of the legend in pixels      */
  gint height;                       /**< height of the legend in pixels     */
  gint position;                     /**< position of the legend             */
};
typedef struct _Legend Legend;

/**
 * @brief    Describes an individual frame of xy-data.
 */
struct _Frame
{
  gdouble time;                      /**< the frame time                     */
  gdouble x_range[2];                /**< range along the x axis [min,max]   */
  gdouble y_range[2];                /**< range along the x axis [min,max]   */
  gint npoints;                      /**< number of points in the frame      */
  GArray *xy_data;                   /**< list of xy data points             */
};
typedef struct _Frame Frame;

/**
 * @brief    Describes a collection of frame as a single set of data.
 */
struct _DataSet
{
  gint type;                         /**< one of the YG_... types            */
  gchar* name;                       /**< a name to be display in the legend */
  gchar* fname;                      /**< full filename (including directory)*/
  gchar* dir;                        /**< the directory containing the file  */
  gint nframes;                      /**< number of frames in the data set   */
  gdouble x_range[2];                /**< max and min of all frame x values  */
  gdouble y_range[2];                /**< max and min of all frame y values  */
  GArray* frame;                     /**< a list of frames                   */
  GArray* cmpt_set;                  /**< a list of dataset indices used to
					build this set (eg. for special plot
					types)                               */
};
typedef struct _DataSet DataSet;

/**
 * @brief    Describes a frame (line) in terms of window dependent ij to be
 *           plotted.
 */
struct _PlotLine
{
  GdkGC* gc;                         /**< a graphics context for the line    */
  Frame* frame;                      /**< pointer to the corresponding frame */
  GdkPoint* ij_data;                 /**< ij coordinates of the line points  */
  GdkColor* start_color;             /**< color for the first line of the set*/
  GdkColor* end_color;               /**< color of the last line of the set  */
  gint npoints;                      /**< number of points in the frame      */
};
typedef struct _PlotLine PlotLine;

/**
 * @brief    Describes all of the data contained in a single plot window.
 */
struct _Plot
{
  gint plot_nbr;                     /**< a unique identifier for the window */

  GtkWidget* window;                 /**< pointer to the plot's window widget*/
  GtkWidget* plot_area;              /**< the drawing area of the window     */
  GdkPixmap* pixmap;                 /**< the pixmap in which data is drawn  */
  GtkItemFactory* menu;              /**< the menu bar along the top         */
  GdkGC* gc;                         /**< a graphics context for the window  */
  GdkFont* font;                     /**< default font for text in the window*/
  gint font_height;                  /**< approx of default font height      */
  GdkColormap* colormap;             /**< a colormap for the window          */

  GArray* data;                      /**< xy-data (array of DataSet*)        */
  GArray* plot_data;                 /**< ij-data (array of PlotLine*)       */
  GArray* current_frame;             /**< identify the current frame for each*/
                                     /**< data set (gint)                    */
  Legend* legend;                    /**< the plot legend                    */
  Axis* x_axis;                      /**< the plot x-axis                    */
  Axis* y_axis;                      /**< the plot y-axis                    */

  gdouble x_range[2];                /**< x-axis range to be plotted         */
  gdouble y_range[2];                /**< y-axis range to be plotted         */
  gint i_origin;                     /**< plot i-origin within the window    */
  gint j_origin;                     /**< plot j-origin within the window    */
  gint i_size;                       /**< length (in pixels) of the x-axis   */
  gint j_size;                       /**< length (in pixels) of the y-axis   */
  gint zoom_x_start;                 /**< starting x coord of a zoom region  */
  gint zoom_y_start;                 /**< starting y coord of a zoom region  */

  gboolean draw_points;              /**< draw dotss at data values          */
  gboolean fixed_range;              /**< fix the range, ie. no zoom         */
  gboolean draw_grid;                /**< draw a background grid             */
  gboolean draw_range;               /**< print range values on the plot     */

  gchar* current_directory;          /**< directory from which to load files */
};
typedef struct _Plot Plot;

/**
 * @brief    Describes the widgets contained in the control panel.
 */
struct _ControlPanel
{
  GtkWidget* window;                 /**< the control panel window           */
  GtkWidget* play_button;            /**< the play button                    */
  GtkWidget* time_entry;             /**< the time entry field               */
  GtkWidget* delay_entry;            /**< the animation delay entry field    */
};
typedef struct _ControlPanel ControlPanel;

/** @} */

/*
 * Global variables;
 */

/* A set of colors from which to choose defaults for plot lines. */
extern gulong global_base_color[NCOLORS][3];

/* The plot window is in one of three states given by the DISPLAY_MODE enum.
 * This variable holds the current state so that it can be conveniently
 * accessed when updating windows. Note that all windows are assumed to
 * have the same state.
 */
extern DISPLAY_MODE global_display_mode;     

/* An array of Plot* plots. Each entry corresponds to a unique plot window. */
extern GArray* global_plot_window;

/* A list of all of the data sets that have been loaded. These are pointers
 * to the data sets which are also contained in the Plot* structures, and
 * is used so that we can easily process the data sets without having to
 * go through the main structure.
 * One issue is that when data sets are freed, both the Plot* pointer and
 * the global_data_set_list entry need to be removed.
 */
extern GArray* global_data_set_list;

/* A (currently unused) integer which is used to assign unique identifiers to
 * plot windows.
 */
extern gint global_plot_nbr;

/* A pointer to the control panel so that it's easy to get at. */
extern ControlPanel* global_control_panel;

/* A convenience pointer holding the about window data. */
extern GdkPixmap* global_about_pixmap;

/* The current frame, common to all windows, used for animation. */
extern gint global_current_frame;

/* A list holding the unique times contained in each of the frames of each
 * of the datasets. Animation works like this: Starting from frame zero,
 * times from the global_time_list are read one at a time, defining the
 * 'current time'. Frames from each of the datasets are compared with this
 * value. The nearest frame whose time is <= the current time is displayed
 * for each dataset.
 */
extern GArray* global_time_list;

/* The total number of frames that are loaded, corresponding to the number of
 * entries in the time list.
 */
extern gint global_last_frame;

/* Intensity of the background grid. Maybe this should be an option_ instead */
extern gulong global_grid_intensity;

/* Current working directory (directory in which the program was started). */
extern gchar* global_working_directory;

/* User-set values for the x and y axis ranges. */
extern gdouble* global_x_range;
extern gdouble* global_y_range;

/*
 * Command line options.
 */

/* Don't create a control panel on startup. */
extern gboolean option_start_without_control_panel;

/* Print a version message and quit. */
extern gboolean option_version_msg;

/* Create a separate window for each dataset listed on the command line. */
extern gboolean option_separate_start_windows;

/* Draw dots at the data points by default. */
extern gboolean option_draw_points;

/* Draw a background grid on the plots by default. */
extern gboolean option_draw_grid;

/* Draw range values on the corners of the plot by default. */
extern gboolean option_draw_range;

/* Width of the lines in a plot. */
extern gint option_line_width;

/* Time delay between individual frames, measured in milliseconds. */
extern gint option_animate_delay;

/* Read only every nth frame from input files */
extern gint option_read_skip_step;

/* Position of the legend relative to the plot data */
extern gchar* option_legend_position;

/* List of potential NaN strings */
extern gchar** nan_string;

#endif /* __YGRAPH_H__ */

