package Zim::Components::PathBar;

use strict;
use vars '$AUTOLOAD';
use Gtk2;
use Gtk2::Ex::PathBar; # custom widget

our $VERSION = 0.08;

=head1 NAME

Zim::Components::PathBar - Path bar widgets

=head1 DESCRIPTION

This module contains the widgets to display a path bar.
This bar can either show the browsing history or the namespace of
the current page.

=head1 METHODS

Undefined methods are AUTOLOADED to the Gtk2::Ex::PathBar object.

=over 4

=item C<new(app => PARENT)>

Simple constructor.

=item C<init()>

Method called by the constructor.

=cut

sub new {
	my $class = shift;
	my $self = bless {@_}, $class;
	$self->init();
	return $self;
}

sub init { # called by new()
	my $self = shift;
	$self->{path_bar} = Gtk2::Ex::PathBar->new(spacing => 3);
	$self->{path_bar}->hide_sliders(1); # FIXME make this configable
	$self->{path_bar}->signal_connect_swapped(
		path_clicked => \&on_path_clicked, $self);
	$self->{app}->signal_connect('page_loaded' => \&update, $self);
}

sub AUTOLOAD {
	my $self = shift;
	$AUTOLOAD =~ s/^.*:://;
	return if $AUTOLOAD eq 'DESTROY';
	return $self->{path_bar}->$AUTOLOAD(@_);
}

=item C<widget()>

Returns the root widget. This should be used to add the object to a container widget.
Also use this widget for things like show_all() and hide_all().

=cut

sub widget { return $_[0]->{path_bar} }

=item C<set_type(TYPE)>

Set the type of the pathbar. Type can be 'namespace' or 'trace'.

=cut

sub set_type {
	$_[0]->{type} = $_[1];
	$_[0]->{path_loaded} = '';
	$_[0]->update;
}

=item C<update()>

Update the path displayed. Should be called when the current page changes.

=cut

sub update {
	my $self = pop;
	return unless $self->{app}{page};
	if ($self->{type} eq 'namespace') {
		$self->_load_namespace;
	}
	elsif ($self->{type} eq 'trace') {
		$self->_load_trace;
	}
	# else hidden, ignoring
}

sub _load_namespace {
	my $self = shift;
	my $namespace = $self->{app}->History->get_namespace();
	$namespace =~ s/^:+|:+$//g;
	my @namespace = split /:+/, $namespace;
	my $current = $self->{app}->{page}->name;
	$current =~ s/^:+|:+$//g;
	my @current = split /:+/, $current;
	
	if ($self->{path_loaded} =~ /^$namespace/i) {
		$self->{path_bar}->select_item($#current);
		return;
	}
	
	$self->{path_loaded} = $namespace;
	$self->{path_bar}->set_path(@namespace);
	$self->{path_bar}->select_item($#current);
	
	for ($self->{path_bar}->get_items()) {
		my $name = ':'.join(':', @{$_->{path_data}}) ;
		$_->signal_connect(enter => sub {
			$self->{app}->push_status("Go to $name", 'link') } );
		$_->signal_connect(leave => sub {
			$self->{app}->pop_status('link')                 } );
	}

}

sub _load_trace {
	my $self = shift;

	my ($back, $forw) = $self->{app}->History->state;
	if ($self->{path_loaded} and $forw) {
		$self->{path_bar}->select_item($back); # == $#back + 1
		return;
	}

	my ($hist, $page) = ($self->{app}->History, $self->{app}{page});
	my (@path, @names);
	for ($hist->get_back) {
		push @path, [$_->{basename}];
		push @names, $_->{name};
	}
	push @path, [$page->basename];
	push @names, $page->name;
	for ($hist->get_forw) {
		push @path, [$_->{basename}];
		push @names, $_->{name};
	}

	$self->{path_bar}->set_path(@path);
	$self->{path_bar}->select_item($back); # == $#back + 1
	$self->{path_loaded} = 1;

	for ($self->{path_bar}->get_items) {
		my $name = shift @names;
		$_->signal_connect(enter => sub {
			$self->{app}->push_status("Go to $name", 'link') } );
		$_->signal_connect(leave => sub {
			$self->{app}->pop_status('link')                 } );
	}
}

sub on_path_clicked {
	my ($self, $path, $idx, $path_bar) = @_;
	if ($self->{type} eq 'namespace') {
		$self->{app}->load_page(join ':', @$path);
	}
	elsif ($self->{type} eq 'trace') {
		my ($back, $forw) = $self->{app}->History->state;
		$idx -= $back;
		if    ($idx < 0) { $self->{app}->go_back(-$idx) }
		elsif ($idx > 0) { $self->{app}->go_forw($idx)  }
		else             { $self->{app}->reload()       }
	}
	# else bug !?
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

