package Zim::History;

use strict;
use Storable qw/nstore retrieve/;

our $VERSION = 0.08;

=head1 NAME

Zim::History - History object for zim

=head1 SYNOPSIS

	use Zim::History;
	
	my $hist = Zim::History->new($HIST_FILE, $PAGEVIEW, 20);
	my $page_record = $hist->get_current;

=head1 DESCRIPTION

This object manages zim's page history.

=head1 METHODS

=over 4

=item C<new(FILE, PAGEVIEW, MAX)>

Constructor. Takes the name of the cache file as an argument
and reads this cache file. The second argument is the pageview
object. When a new record is made for the current page the C<get_state()> method is
called on this object to get state information that needs to be saved.
Third argument is the maximum number of items in the history.

=cut

sub new {
	my ($class, $file, $view, $max) = @_;
	my $self = bless {
		file      => $file,
		PageView  => $view,
		max       => $max,
		current   => undef,
		back      => [],
		forw      => [],
	}, $class;
	$self->read;
	return $self;
}

=item C<read>

Read the cache file.

=cut

sub read {
	my $self = shift;
	return unless -f $self->{file} and -r _;
	my @cache = @{ retrieve($self->{file}) };
	return if ref $cache[0];
		# early versions did not include the version number
	my $version = shift @cache;
	return unless $version == $VERSION;
	@{$self}{qw/back current forw/} = @cache;
}

=item C<write>

Write the cache file.

=cut

sub write {
	my $self = shift;
	$self->get_current; # force conversion from object to record
	nstore([$VERSION, @{$self}{qw/back current forw/}], $self->{file});
}

=item C<set_current(PAGE)>

Give the page object that is going to be displayed in the 
PageView. Set a new current before updating the PageView
so the History object has time to save the state of the PageView
if necessary.

=cut

sub set_current {
	my ($self, $page) = @_;

	if ($self->{current} || $self->{page}) {
		my $name = $self->{current}{name} || $self->{page}->name;
		if ($page->name eq $name) { # redundant entry
			$self->{page} = $page;
			return;
		}
	}

	my $current = $self->get_current;
	if ($current) {
		push @{$self->{back}}, $current;
		shift @{$self->{back}} if @{$self->{back}} > $self->{max};
	}
	$self->{current} = undef;
	$self->{forw} = [];
	$self->{page} = $page;
}

=item C<get_current()>

Returns the curernt history object. When possible asks the PageView
objects for the current state information.

=cut

sub get_current {
	my $self = shift;
	return $self->{current} unless $self->{page};
	
	my $name = $self->{current}{name} || $self->{page}->name;
	my $record = $self->record($name) || {name => $name};

	%$record = ( %$record,
		basename  => $self->{page}->basename,
		namespace => $self->{page}->namespace,
		$self->{PageView}->get_state(),
	);

	$self->{current} = $record;
	return $self->{current};
}

=item C<get_namespace>

This method matches the namespace of the current page to that of pages in the
history. The result can be used as a namespace stack.

=cut

sub get_namespace {
	# FIXME lookup bug remaining here
	my $self = shift;
	my $current = $self->get_current;
	return unless $current;

	my $namespace = $current->{name};
	#print STDERR "looking for namespace $namespace";
	for (
		reverse( @{$self->{back}} ),
		@{$self->{forw}}, $current
	) {
		my $name = exists($$_{_namespace}) ? $$_{_namespace} : $$_{name} ;
		$namespace = $name if $name =~ /^:*$namespace:/;
	}
	#print STDERR " => $namespace\n";
	$current->{_namespace} = $namespace;

	return $namespace;
}

=item C<state()>

Returns two integers representing the number of items in the back stack
and the number of items in the forward stack.

=cut

sub state {
	my $self = shift;
	my $back = scalar @{$self->{back}};
	my $forw = scalar @{$self->{forw}};
	return $back, $forw;
}

=item C<get_back()>

Returns a list of records of pages we can go back to.

=cut

sub get_back { @{$_[0]->{back}} }

=item C<get_forw()>

Returns a list of records of pages we can go forward to.

=cut

sub get_forw { @{$_[0]->{forw}} }

=item C<back(INT)>

Go back one or more steps in the history stack.

=cut

sub back {
	my ($self, $i) = @_;
	return 0 if $i < 1 or $i > @{$self->{back}};
	my $current = $self->get_current;
	unshift @{$self->{forw}}, $current if $current;
	unshift @{$self->{forw}}, splice @{$self->{back}}, -$i+1 if $i > 1;
	$self->{current} = pop @{$self->{back}};
	$self->{page} = undef;
	return 1;
}

=item C<forw(INT)>

Go forward one or more steps in the history stack.

=cut

sub forw {
	my ($self, $i) = @_;
	return 0 if $i < 1 or $i > @{$self->{forw}};
	my $current = $self->get_current;
	CORE::push @{$self->{back}}, $current if $current;
	CORE::push @{$self->{back}}, splice @{$self->{forw}}, 0, $i-1 if $i > 1;
	$self->{current} = shift @{$self->{forw}};
	$self->{page} = undef;
	return 1;
}

=item C<record(PAGE)>

Returns the history record for a given page object or undef.

=cut

sub record {
	my ($self, $page) = @_;
	my ($back, $current, $forw) = @{$self}{qw/back current forw/};
	
	my $name = ref($page) ? $page->name : $page;
	my ($rec) = grep {$_->{name} eq $name} (
		((ref($current) eq 'HASH') ? ($current) : ()),
		@$forw, reverse(@$back) );

	return $rec || undef;
}

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

=cut

