package Zim::Page;

use strict;

our $VERSION = 0.08;

=head1 NAME

Zim::Page - Page object for Zim

=head1 SYNOPSIS

FIXME simple code example

=head1 DESCRIPTION

This class defines a page object. This is a data container used by
L<Zim::Repository> to represent one file.

=head1 METHODS

=over 4

=item C<new(PARENT, NAME)>

Simple constructor. PARENT should be a repository object of class Zim.
NAME is the page name for this object.

=cut

sub new {
	my ($class, $parent, $name) = @_;
	$name =~ s/^:+|:+$//g;
	die "Can't create $class object without a page name!\n"
		unless length $name;
	my $self = bless {
		repository => $parent,
		pagename => $name,
		parse_tree => ['Document', {}],
		status => '',
	}, $class ;
	return $self;
}

=item C<name()>

Get or set the full name of the page.

=cut

sub name {
	$_[0]->{pagename} = $_[1] if @_ > 1;
	return $_[0]->{pagename};
}

=item C<basename()>

Returns the last part of the page name.

=cut

sub basename {
	my $name = $_[0]->{pagename};
	$name =~ s/:$//;
	$name =~ s/^(.*:+)//;
	return $name;
}

=item C<namespace()>

Returns the namespace to which the page belongs.

=cut

sub namespace {
	my $name = $_[1] || $_[0]->{pagename};
	#print STDERR "namespace for $name ";
	$name =~ /^(.*:)/;
	$name = ":$1:";
	$name =~ s/::+/:/g;
	#print STDERR "is $name\n";
	return $name;
}

=item C<status(STRING)>

Set or get a status string for this page.
Typical status strings are 'new' and 'deleted'.

=cut

sub status {
	$_[0]->{status} = $_[1] if @_ > 1;
	return $_[0]->{status};
}

=item C<resolve_link(LINK)>

Returns a page name for a page linked from this page.
This method allows you to have relative links.

In list context the page name can be followed with some data
the may be given to the repository's C<load_page()>. This is
used for repository specific optimizations.

=cut

sub resolve_link { # FIXME - is this formatter specific ?
	my ($self, $link) = @_;
	#print STDERR "resolved link $link to ";
	$link =~ s/[^\w\.\:\-]/_/g;
	$link = ($link =~ s/^\.//) ? $self->name.':'.$link  : # sub namespace
		($link !~ /:/)     ? $self->namespace.$link : # relative link
		$link ;
	#print STDERR "to $link\n";
	return $self->{repository}->resolve_page($link); # uses wantarray
}

=item C<parse_tree()>

Get or set the parse tree.

=cut

sub parse_tree {
	$_[0]->{parse_tree} = $_[1] if @_ > 1;
	return $_[0]->{parse_tree};
}

=item C<push_blocks(BLOCK, BLOCK, ...)>

Append more data to this page.

=cut

sub push_blocks {
	my $self = shift;
	push @{$self->{parse_tree}}, @_;
}

=item C<unshift_blocks(BLOCK, BLOCK, ...)>

Prepend a data to this page.

=cut

sub unshift_blocks { # splice between header and content
	my $self = shift;
	splice @{$self->{parse_tree}}, 2, 0, @_;
}

=item C<walk_parse_tree(CODE)>

TODO test this method

=cut

sub walk_parse_tree {
	my ($self, $code) = @_;
	my $tree = $self->{parse_tree};
	if (ref($code) eq 'ARRAY') {
		$tree = $code;
		$code = shift;
	}

	my $new_tree = [ splice @$tree, 0, 2 ]; # meta
	for (@$tree) {
		my $node = $code->($_);
		next if $node == 0;
		$node = $_ unless length $node;
		$node = $self->walk_parse_tree($node, $code) if ref $node;
		push @$new_tree, $node;
	}

	return $new_tree;
}

=item C<clear()>

Clears all data from this page and initialises an empty parse tree.

=cut

sub clear { $_[0]->{parse_tree} = ['Document', {}] }

=item C<save()>

=item C<delete()>

=item C<move(NEW_NAME)>

The methods C<save()>, C<delete()> and C<move()> are aliases
for the methods C<save_page()>, C<delete_page()> and C<move_page()>
in L<Zim::Repository>.

=cut

sub save { $_[0]->{repository}->save_page(@_) }

sub delete { $_[0]->{repository}->delete_page(@_) }

sub move { $_[0]->{repository}->move_page(@_) }

1;

__END__

=back

=head1 AUTHOR

Jaap Karssenberg (Pardus) E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>,
L<Zim::Repository>

=cut

