package Zim::Repository;

use strict;
use POSIX qw(strftime);
use Encode;
use File::Spec;
use File::Copy;
use File::MimeInfo;
use Zim::Page;

our $VERSION = 0.08;

my $case_tolerant_fs = File::Spec->case_tolerant();

=head1 NAME

Zim::Repository - A wiki repository

=head1 SYNOPSIS

	use Zim::Repository;
	
	my $repo = Zim::Repository->new($root_dir);
	my $page = $repo->load_page('Home');

=head1 DESCRIPTION

This class defines the public interface to the document repository
as used by L<zim>(1). By default it represents filesystem based repositories,
but it can be overloaded for more abstract models. It uses L<Zim::Page>
objects as data containers.

=head1 METHODS

=head2 Public Methods

The following methods can be used by the GUI.

=over 4

=item new(NAMESPACE, DIR)

Simple constructor. DIR is the root directory of the repository.
NAMESPACE is the namespace that maps to that directory. For the 
toplevel repository leave NAMESPACE empty.

=cut

sub new {
	my ($class, $namespace, $dir, $format) = @_;
	$dir = File::Spec->file_name_is_absolute($dir)
		? File::Spec->canonpath($dir)
		: File::Spec->rel2abs($dir)   ;
	$namespace =~ s/:?$/:/; # '' => ':'
	my $self = bless {
		namespace => $namespace,
		root      => $dir,
		format    => $format || 'wiki',
	}, $class;
	
	$self->{formatter} =  $self->_load_format( $self->{format} );
	$self->{ext} = $self->{formatter}->_extension;
	
	# load plugins (nested repositories
	my $conf = File::Spec->catfile($dir, '.zim.config');
	if (-f $conf and -r _) {
		$self->{conffile} = $conf;
		for (split /\n/, _read_file($conf)) {
			/^:(\S+?)=(\S+)/ or next;
			my ($subspace, $mod) = ($1, $2);
			$subspace =~ s/:+$//;
			eval "use $mod;";
			die if $@;
			my $dir = $self->filename($subspace, 'DIR');
			$self->{plugins}{$subspace} = $mod->new($subspace, $dir);
			#print STDERR "plug: $mod for $subspace\n";
		}
	}

	return $self;
}

sub _load_format {
	my $self = shift;
	my $wanted = lc( shift(@_) ).'.pm';
	my $class;
	for (@INC) {
		# FIXME object refs in @INC
		my $dir = File::Spec->catdir($_, qw/Zim Formats/);
		next unless -d $dir and -r _;
		opendir DIR, $dir or next;
		($class) = grep {lc($_) eq $wanted} readdir DIR;
		closedir DIR;
		last if $class;
	}
	$class =~ s/.pm$//;
	die "Could not find a module for format: $self->{format}\n" unless $class;
	$class = "Zim::Formats::$class";
	eval "use $class;";
	die if $@;
	return $class;
}

=item list_pages(NAMESPACE)

Lists pages in NAMESPACE. Sub-namespaces have a trailing ':'
in the listing.

=cut

sub list_pages {
	my ($self, $namespace) = @_;
	
	$namespace =~ s/:?$/:/;
	if (my $plug = $self->_belongs_to_plugin($namespace)) {
		return $self->{plugins}{$plug}->list_pages($namespace);
	}

	my $dir = $self->filename($namespace, 'DIR');
	#print "list pages $namespace => $dir\n";
	return () unless -d $dir;
	opendir DIR, $dir or die "Could not list dir $dir\n";
	my @items = grep {defined $_} map {
		$_ = Encode::decode_utf8($_, 1); # use utf8 encoding
		my $item = File::Spec->catfile($dir, $_);
		if (-d $item) { s/(_files)?$/:/ }
		else {
			if (mimetype($item) =~ /^text/) { s/\.$$self{ext}$// }
			else { $_ = undef }
		}
		$_;
	} grep {! /^\./} readdir DIR;
	closedir DIR;
	return sort {lc($a) cmp lc($b)} @items;
}

sub _belongs_to_plugin { # FIXME can this lookup be done more efficiently ?
        my ($self, $page) = @_;
	$page =~ s/^:*$self->{namespace}:*//;
        my $plug = '';
	for (grep {$page =~ /^$_:/} keys %{$self->{plugins}}) {
		$plug = $_ if length($_) > length($plug);
	}
	#print STDERR "plugin for $page => $plug\n";
        return $plug;
}


=item page_exists(PAGE_NAME)

Returns TRUE when there is an existing page under this name.

=cut

sub page_exists {
	my ($self, $name) = @_;
	
	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->page_exists($name);
	}
	
	return -f $self->filename($name);
}

=item load_page(PAGE_NAME)

Returns an object of the type L<Zim::Page>.

If the page you requested doesn't exist you get an object with it's status
set to "new".

=cut

sub load_page {
	my ($self, $name, $file, $dir) = @_;
	#print STDERR "load_page: @_\n";
	# file and dir arguments are an undocumented optimization

	if (my $plug = $self->_belongs_to_plugin($name)) {
		return $self->{plugins}{$plug}->load_page($name);
	}
	
	my $page = Zim::Page->new($self, $name);
	@{$page}{'_file', '_dir'} =
		defined($file) ? ($file, $dir) : $self->filename($name) ;
		#print STDERR "=> @{$page}{'_file', '_dir'}\n";
	if (-f $$page{_file}) {
		my $tree = $self->{formatter}->load_tree(
			_read_file($$page{_file}), $page);
		$tree->[1]{base} = $$page{_dir};
		$page->{parse_tree} = $tree;
	}
	else {
		$page->{parse_tree} = $self->_template($page);
		$page->{status} = 'new';
	}

	return $page;
}

=item save_page(PAGE)

Saves the data for the page object PAGE.

Note that the page object could belong to an other repository originally.

=cut

sub save_page {
	my ($self, $page) = @_;

	if (my $plug = $self->_belongs_to_plugin($page->name)) {
		return $self->{plugins}{$plug}->save_page($page);
	}
	elsif ($$page{repository} ne $self) { # page does not belong to us
		@{$page}{'_file', '_dir'} = ();
		$$page{repository} = $self;
	}
	my $file = $self->filename($page);
	
	_write_file( $file,
		$self->{formatter}->save_tree($page->parse_tree, $page)  );
	$page->status('');
}

=item move_page(OLD_NAME, NEW_NAME)

Moves a page from OLD_NAME to NEW_NAME.

You can use an object instead of OLD_NAME, in this case the object will also
be updated to reflect the new location.

=cut

sub move_page {
	my ($self, $old, $new) = @_;
	
	# Move file
	my ($old_file, $old_dir) = $self->filename($old);
	my ($new_file, $new_dir) = $self->filename($new);
	_mkdir($new_dir);
	move($old_file, $new_file)
		or die "Could not move file $old_file to $new_file\n"
		if -f $old_file;

	# Update Page object
	if (ref $old) {
		$old->name($new);
		@{$old}{'_file', '_dir'} = ($new_file, $new_dir);
	}

	# Move tree below file
	my $old_tree = $self->filename($old, 'DIR');
	my $new_tree = $self->filename($new, 'DIR');
	move($old_tree, $new_tree)
		or die "Could not move dir $old_tree to $new_tree\n"
		if -d $old_tree;
	_rmdir($old_dir);
}

=item delete_page(PAGE)

Deletes a page.
PAGE can be either a page name or a page object.

=cut

sub delete_page {
	my ($self, $page) = @_;

	my $file = $self->filename($page);
	_rm_file($file) if -f $file;
	
	if (ref $page) {
		$page->clear;
		$page->status('deleted');
	}
}

sub _template {
	# FIXME make template configurable
	my ($self, $page) = @_;
	my $name = $page->name;
	$name =~ /([^:]+):*$/;
	my $title = $1;
	$title = ucfirst($title) unless $title =~ /[A-Z]/;
	return $self->{formatter}->load_tree(
		"====== $title ======\n".
		'Created '.strftime('%A %d/%m/%Y %H:%M', localtime)."\n\n",
		$page
	);
}

=item C<resolve_page(NAME)>

This method finds a case-sensitive name corresponding to the
user supplied page name NAME. The page does not need to exist.

More generally this method is used to find a canonical page name
corresponding to a potential non-canonical name that is entered by
the user. This method may do all kinds of cleanups.

This method is used when resolving links.

In list context the page name can be followed with some data
the may be given to the repository's C<load_page()>. This is
used for repository specific optimizations.

=cut

sub resolve_page {
	my ($self, $name) = @_;
	my ($file, $dir) = $self->filename($name, 0, 1); # case tolerant lookup
	$name = $self->pagename($file);
	return wantarray ? ($name, $file, $dir) : $name ;
}

=item C<export(PAGE, %OPTS)>

TODO stable api for this

=cut

# TODO when api is stable add export() to page which redirects here
# TODO use verbose option

sub export {
	my ($self, $page, %opts) = @_;
	die "Need a page to export\n" unless length $page;
	for (qw/format template root/) {
		die "for exporting you need at least to provide a format, a template and an output root dir.\n" unless length $opts{$_};
	}

	$page = $page->name if ref $page;
	my $namespace = ($page =~ /:$/) ? $page : '';
	my $exporter = ref($opts{root}) ? $opts{root} :
		Zim::Repository->new('', $opts{root}, $opts{format});
	print STDERR "export template: $opts{template}\n";
	$exporter->{_template} = $opts{template}; # FIXME better interface for this - do not confuse with the template for new pages
	
	my $r = defined($opts{recurs}) ? $opts{recurs} : 1 ;
	if (length $namespace) {
		$self->_export_namespace($exporter, $page, $r);
	}
	else {
		$self->_export_page($exporter, $page);
	}

}


sub _export_namespace { # page =~ /:$/
	my ($self, $exporter, $page, $r) = @_;
	
	my $index = 1;
	my @pages = $self->list_pages($page);
	for (@pages) {
		$index = 0 if /index/;
		$self->_export_namespace($exporter, $page.$_, $r)
			if $r and /:$/; # recurs
		$self->_export_page($exporter, $page.$_) or $_ = undef;
	}
	@pages = grep defined($_), @pages;

	if ($index) { # FIXME option for this
		print STDERR "Writing index\n";
		$index = Zim::Page->new($exporter, $page.'index');
		$index->push_blocks(
			['head1', {}, "Document index for $page"] );
		$index->push_blocks(
			['Para', {},
				map {("* ", ['link', {to => $_}, $_])} @pages
			]
		);
		$exporter->save_page($index);
	}
	
}

sub _export_page {
	my ($self, $exporter, $page) = @_;
	
	my $obj = $self->load_page($page);
	return 0 if $obj->status eq 'new';
	print STDERR "Exporting: $page\n";
	$exporter->save_page($obj);
	return 1;
}

=back

=head2 Private Methods

These methods are specific to filesystem based repositories
and should therefor never be called directly from the GUI.

=over 4

=item filename(PAGE, DIR)

Gives the filename corresponding to a page name.

DIR is a boolean that can be set to tell that PAGE is expected to
be a directory, which is useful when looking up a namespace.
This behaviour is also triggered if the page name ends with ':'.

In list context the filename and directory for PAGE are returned.

=cut

sub filename {
	my ($self, $page, $is_dir, $case_tolerant) = @_;

	if (ref $page) { # page object
		return wantarray ? @{$page}{'_file', '_dir'} : $$page{_file}
			if defined $$page{_file};
		$page = $page->name;
	}
	
	$page =~ s/^:*$self->{namespace}:*//i;
	$is_dir++ if $page =~ s/:+$//;
	
	my @dirs = split /:+/, $page;
	my $basename = pop @dirs unless $is_dir;

	# Find the right dir
	my $dir = File::Spec->catdir($self->{root}, @dirs);
	unless (!$case_tolerant_fs and  -d $dir) {
		$dir = $self->{root};
		for (0 .. $#dirs) {
			my $new_dir = $self->_grep_dir(
				$dir, $dirs[$_], 1, $case_tolerant );
			if (defined $new_dir) { $dir = $new_dir }
			else {
				$dir = File::Spec->catdir($dir, @dirs[$_ .. $#dirs]);
				last;
			}
		}
	}
	
	if ($is_dir) {
		return $dir unless wantarray;
		@dirs = File::Spec->splitdir($dir);
		pop @dirs;
		my $parent = File::Spec->catdir(@dirs);
		return ($dir, $parent);
	}

	# Check the filename
	my $file = $self->_grep_dir($dir, $basename, 0, $case_tolerant);
	$file = File::Spec->catfile($dir, $basename .'.'.$$self{ext})
		unless defined $file;

	return wantarray ? ($file, $dir) : $file;
}

sub _grep_dir { # find a file or subdir
	my ($self, $dir, $basename, $is_dir, $case_tolerant) = @_;
	return undef unless -d $dir;
	my $extended = $basename . ($is_dir ? '_files' : '.'.$$self{ext});
	my $wrong_type = 0;
	
	unless ($case_tolerant and $case_tolerant_fs) {
		if ($is_dir) {
			my $path = File::Spec->catdir($dir, $basename);
			return $path if -d $path;
			$wrong_type = 1 if -e _;
		}
		else {
			my $path = File::Spec->catfile($dir, $basename);
			return $path if -f $path;
			$wrong_type = 1 if -e _;
		}
	
		my $path = $is_dir
			? File::Spec->catdir($dir, $extended)
			: File::Spec->catfile($dir, $extended) ;
		return $path if $wrong_type or -e $path;
		return undef unless $case_tolerant;
	}
	
	my @check = (lc($basename), lc($extended));
	my $suggestion;
	my $other_type = lc $basename . ($is_dir ? '.'.$$self{ext} : '_files');
	opendir DIR, $dir or die "Could not list dir $dir\n";
	while (my $item = readdir DIR) { 
		# FIXME for case_tolerant_fs it would be nicer to check
		# case sensitive version first in the listing
		$item = Encode::decode_utf8($item, 1);
		if (grep {$_ eq lc($item)} @check) {
			if ($is_dir) {
				$item = File::Spec->catdir($dir, $item);
				unless (-d $item) {
					$suggestion = $item;
					next;
				}
			}
			else {
				$item = File::Spec->catfile($dir, $item);
				unless (-f $item) {
					$suggestion = $item;
					next;
				}
			}
			closedir DIR;
			return $item;
		}
		elsif (lc($item) eq $other_type) {
			$suggestion = $is_dir
				? File::Spec->catdir($dir, $item)
		       		: File::Spec->catfile($dir, $item) ;
		}
	}
	closedir DIR;

	if ($suggestion) {
		if ($is_dir) { $suggestion =~ s/\.$$self{ext}$//          }
		else         { $suggestion =~ s/(_files)?$/\.$$self{ext}/ }
		return $suggestion;
	}
	
	return undef;
}

=item pagename(FILE)

Returns the page name corresponding to FILE. FILE does not actually
need to exist and can be a directory as well as a file.

=cut

sub pagename {
	my ($self, $file) = @_;
	$file = File::Spec->abs2rel($file, $self->{root})
		if File::Spec->file_name_is_absolute($file);
	my @parts = map  {s/_files$//; $_} File::Spec->splitdir($file);
	$parts[-1] =~ s/\.$$self{ext}$//;
	return $self->{namespace} . join ':', @parts;
}

# Methods below for filesystem interaction

sub _read_file {
	my $file = shift;
	open TXT, $file or die "Could not read $file\n";
	binmode TXT, ':utf8' unless $] < 5.008;
	my $text = join '', <TXT>;
	close TXT;
	return $text;
}

sub _write_file {
	my $file = shift;
	_mkdir($file, 'FILE');
	open TXT, ">$file" or die "Could not write $file\n";
	binmode TXT, ':utf8' unless $] < 5.008;
	print TXT @_;
	close TXT or die "Could not write $file\n";
}

sub _rm_file {
	my $file = shift;
	unlink $file or die "Could not remove file $file\n";
	_rmdir($file, 'FILE');
}

sub _mkdir {
	my ($dir, $file) = @_; # if $file is true $dir is a file
	my ($vol, $dirs) = File::Spec->splitpath($dir, $file ? 0 : 1);
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	mkdir $path or die "Could not create dir $path\n"
		if length $path and ! -d $path;
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		mkdir $path or die "Could not create dir $path\n"
			unless -d $path;
	}
}

sub _rmdir {
	my ($dir, $file) = @_; # if $file is true $dir is a file
	my ($vol, $dirs) = File::Spec->splitpath($dir, $file ? 0 : 1);
	my @dirs = File::Spec->splitdir($dirs);
	my $path = File::Spec->catpath($vol, shift @dirs);
	rmdir $path; # fails when not empty
	while (@dirs) {
		$path = File::Spec->catdir($path, shift @dirs);
		last unless -d $path;
		rmdir $path; # fails when not empty
	}
}

1;

__END__

=back

=head1 BUGS

Please mail the author if you find any bugs.

=head1 AUTHOR

Jaap Karssenberg || Pardus [Larus] E<lt>pardus@cpan.orgE<gt>

Copyright (c) 2005 Jaap G Karssenberg. All rights reserved.
This program is free software; you can redistribute it and/or
modify it under the same terms as Perl itself.

=head1 SEE ALSO

L<Zim>, L<Zim::Page>

=cut
