/***************************************************************************
 *   Copyright (C) 2008 by BogDan Vatra <bogdan@licentia.eu>               *
 *   Copyright (C) 2009-2022 by Robin Stuart <rstuart114@gmail.com>        *
 *                                                                         *
 *   This program is free software: you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation, either version 3 of the License, or     *
 *   (at your option) any later version.                                   *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program.  If not, see <http://www.gnu.org/licenses/>. *
 ***************************************************************************/

//#include <QDebug>
#include <QGraphicsScene>
#include <QImage>
#include <QColorDialog>
#include <QUiLoader>
#include <QFile>
#include <QRadioButton>
#include <QFileDialog>
#include <QMessageBox>
#include <QSettings>
#include <QClipboard>
#include <QMimeData>
#include <QColor>
#include <QListView>
#include <QShortcut>
#include <QMenu>
#include <QAction>

#include "mainwindow.h"
#include "cliwindow.h"
#include "datawindow.h"
#include "sequencewindow.h"

// Shorthand
#define QSL QStringLiteral

static const int tempMessageTimeout = 2000;

static const QKeySequence quitKeySeq(Qt::CTRL | Qt::Key_Q); // Use on Windows also (i.e. not using QKeySequence::Quit)

static const QKeySequence openCLISeq(Qt::SHIFT | Qt::CTRL | Qt::Key_C);

static const QKeySequence copyBMPSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_B);
static const QKeySequence copyEMFSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_E);
static const QKeySequence copyGIFSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_G);
#ifndef NO_PNG
static const QKeySequence copyPNGSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_P);
#endif
static const QKeySequence copySVGSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_S);
static const QKeySequence copyTIFSeq(Qt::SHIFT | Qt::CTRL | Qt::Key_T);

struct bstyle_item {
    const QString text;
    int symbology;
};

static const struct bstyle_item bstyle_items[] = {
    { QSL("Australia Post Redirect Code"), BARCODE_AUSREDIRECT },
    { QSL("Australia Post Reply-Paid"), BARCODE_AUSREPLY },
    { QSL("Australia Post Routing Code"), BARCODE_AUSROUTE },
    { QSL("Australia Post Standard Customer"), BARCODE_AUSPOST },
    { QSL("Aztec Code (ISO 24778) (and HIBC)"), BARCODE_AZTEC },
    { QSL("Aztec Runes (ISO 24778)"), BARCODE_AZRUNE },
    { QSL("Channel Code"), BARCODE_CHANNEL },
    { QSL("Codabar (EN 798)"), BARCODE_CODABAR },
    { QSL("Codablock-F (and HIBC)"), BARCODE_CODABLOCKF },
    { QSL("Code 11"), BARCODE_CODE11 },
    { QSL("Code 128 (ISO 15417) (and GS1-128 and HIBC)"), BARCODE_CODE128 },
    { QSL("Code 16k (EN 12323)"), BARCODE_CODE16K },
    { QSL("Code 2 of 5 Data Logic"), BARCODE_C25LOGIC },
    { QSL("Code 2 of 5 IATA"), BARCODE_C25IATA },
    { QSL("Code 2 of 5 Industrial"), BARCODE_C25IND },
    { QSL("Code 2 of 5 Interleaved (ISO 16390)"), BARCODE_C25INTER },
    { QSL("Code 2 of 5 Standard (Matrix)"), BARCODE_C25STANDARD },
    { QSL("Code 32 (Italian Pharmacode)"), BARCODE_CODE32 },
    { QSL("Code 39 (ISO 16388) (and HIBC)"), BARCODE_CODE39 },
    { QSL("Code 39 Extended"), BARCODE_EXCODE39 },
    { QSL("Code 49"), BARCODE_CODE49 },
    { QSL("Code 93"), BARCODE_CODE93 },
    { QSL("Code One"), BARCODE_CODEONE },
    { QSL("DAFT Code"), BARCODE_DAFT },
    { QSL("Data Matrix (ISO 16022) (and HIBC)"), BARCODE_DATAMATRIX },
    { QSL("Deutsche Post Identcode"), BARCODE_DPIDENT },
    { QSL("Deutsche Post Leitcode"), BARCODE_DPLEIT },
    { QSL("DotCode"), BARCODE_DOTCODE },
    { QSL("DPD Code"), BARCODE_DPD },
    { QSL("Dutch Post KIX"), BARCODE_KIX },
    { QSL("EAN-14"), BARCODE_EAN14 },
    { QSL("European Article Number (EAN) (ISO 15420)"), BARCODE_EANX },
    { QSL("Facing Identification Mark (FIM)"), BARCODE_FIM },
    { QSL("Flattermarken"), BARCODE_FLAT },
    { QSL("Grid Matrix"), BARCODE_GRIDMATRIX },
    { QSL("GS1 DataBar Expanded (ISO 24724)"), BARCODE_DBAR_EXP },
    { QSL("GS1 DataBar Expanded Stacked (ISO 24724)"), BARCODE_DBAR_EXPSTK },
    { QSL("GS1 DataBar Limited (ISO 24724)"), BARCODE_DBAR_LTD },
    { QSL("GS1 DataBar Omnidirectional (and Truncated) (ISO 24724)"), BARCODE_DBAR_OMN },
    { QSL("GS1 DataBar Stacked (ISO 24724)"), BARCODE_DBAR_STK },
    { QSL("GS1 DataBar Stacked Omnidirectional (ISO 24724)"), BARCODE_DBAR_OMNSTK },
    { QSL("Han Xin (Chinese Sensible) Code (ISO 20830)"), BARCODE_HANXIN },
    { QSL("International Standard Book Number (ISBN)"), BARCODE_ISBNX },
    { QSL("ITF-14"), BARCODE_ITF14 },
    { QSL("Japanese Postal Barcode"), BARCODE_JAPANPOST },
    { QSL("Korean Postal Barcode"), BARCODE_KOREAPOST },
    { QSL("LOGMARS"), BARCODE_LOGMARS },
    { QSL("MaxiCode (ISO 16023)"), BARCODE_MAXICODE },
    { QSL("MicroPDF417 (ISO 24728) (and HIBC)"), BARCODE_MICROPDF417 },
    { QSL("Micro QR Code (ISO 18004)"), BARCODE_MICROQR },
    { QSL("MSI Plessey"), BARCODE_MSI_PLESSEY },
    { QSL("NVE-18 (SSCC-18)"), BARCODE_NVE18 },
    { QSL("PDF417 (ISO 15438) (and Compact and HIBC)"), BARCODE_PDF417 },
    { QSL("Pharmacode"), BARCODE_PHARMA },
    { QSL("Pharmacode 2-track"), BARCODE_PHARMA_TWO },
    { QSL("Pharma Zentralnummer (PZN)"), BARCODE_PZN },
    { QSL("PLANET"), BARCODE_PLANET },
    { QSL("POSTNET"), BARCODE_POSTNET },
    { QSL("QR Code (ISO 18004) (and HIBC)"), BARCODE_QRCODE },
    { QSL("Rectangular Micro QR (rMQR) (ISO 23941)"), BARCODE_RMQR },
    { QSL("Royal Mail 4-state Barcode (RM4SCC)"), BARCODE_RM4SCC },
    { QSL("Royal Mail 4-state Mailmark"), BARCODE_MAILMARK },
    { QSL("Telepen"), BARCODE_TELEPEN },
    { QSL("Telepen Numeric"), BARCODE_TELEPEN_NUM },
    { QSL("UK Plessey"), BARCODE_PLESSEY },
    { QSL("Ultracode"), BARCODE_ULTRA },
    { QSL("Universal Product Code (UPC-A) (ISO 15420)"), BARCODE_UPCA },
    { QSL("Universal Product Code (UPC-E) (ISO 15420)"), BARCODE_UPCE },
    { QSL("UPNQR"), BARCODE_UPNQR },
    { QSL("USPS Intelligent Mail (OneCode)"), BARCODE_USPS_IMAIL },
    { QSL("VIN (Vehicle Identification Number)"), BARCODE_VIN },
};

MainWindow::MainWindow(QWidget *parent, Qt::WindowFlags fl)
        : QWidget(parent, fl), m_optionWidget(nullptr), m_symbology(0),
          m_menu(nullptr),
          m_lblHeightPerRow(nullptr), m_spnHeightPerRow(nullptr),
          m_btnHeightPerRowDisable(nullptr), m_btnHeightPerRowDefault(nullptr)
{
    // Undocumented command line debug flag
    m_bc.bc.setDebug(QCoreApplication::arguments().contains(QSL("--verbose")));

    QCoreApplication::setOrganizationName(QSL("zint"));
    QCoreApplication::setOrganizationDomain(QSL("zint.org.uk"));
    QCoreApplication::setApplicationName(QSL("Barcode Studio"));

    QSettings settings;
#if QT_VERSION < 0x60000
    settings.setIniCodec("UTF-8");
#endif

    scene = new QGraphicsScene(this);

    setupUi(this);
    view->setScene(scene);

    restoreGeometry(settings.value(QSL("studio/window_geometry")).toByteArray());

    m_fgcolor.setRgb(settings.value(QSL("studio/ink/red"), 0).toInt(),
                    settings.value(QSL("studio/ink/green"), 0).toInt(),
                    settings.value(QSL("studio/ink/blue"), 0).toInt(),
                    settings.value(QSL("studio/ink/alpha"), 0xff).toInt());
    m_bgcolor.setRgb(settings.value(QSL("studio/paper/red"), 0xff).toInt(),
                    settings.value(QSL("studio/paper/green"), 0xff).toInt(),
                    settings.value(QSL("studio/paper/blue"), 0xff).toInt(),
                    settings.value(QSL("studio/paper/alpha"), 0xff).toInt());

    const int cnt = (int) (sizeof(bstyle_items) / sizeof(bstyle_items[0]));
    for (int i = 0; i < cnt; i++) {
        bstyle->addItem(bstyle_items[i].text);
    }
#ifdef _WIN32
    bstyle->setMaxVisibleItems(cnt); /* Apart from increasing combo size, seems to be needed for filter to work */
#endif
#if QT_VERSION < 0x50A00
    /* Prior to Qt 5.10 comboboxes have display issues when filtered (scrollers not accounted for), so disable */
    filter_bstyle->hide();
#endif

    bstyle->setCurrentIndex(settings.value(QSL("studio/symbology"), 10).toInt());

    txtData->setText(settings.value(QSL("studio/data"), tr("Your Data Here!")).toString());
    /* Don't save seg data */
    txtComposite->setText(settings.value(QSL("studio/composite_text"), tr("Your Data Here!")).toString());
    chkComposite->setChecked(settings.value(QSL("studio/chk_composite")).toInt() ? true : false);
    cmbCompType->setCurrentIndex(settings.value(QSL("studio/comp_type"), 0).toInt());
    cmbECI->setCurrentIndex(settings.value(QSL("studio/eci"), 0).toInt());
    /* Don't save seg ECIs */
    chkEscape->setChecked(settings.value(QSL("studio/chk_escape")).toInt() ? true : false);
    chkData->setChecked(settings.value(QSL("studio/chk_data")).toInt() ? true : false);
    chkRInit->setChecked(settings.value(QSL("studio/chk_rinit")).toInt() ? true : false);
    chkGS1Parens->setChecked(settings.value(QSL("studio/chk_gs1parens")).toInt() ? true : false);
    chkGS1NoCheck->setChecked(settings.value(QSL("studio/chk_gs1nocheck")).toInt() ? true : false);
    chkAutoHeight->setChecked(settings.value(QSL("studio/appearance/autoheight"), 1).toInt() ? true : false);
    chkCompliantHeight->setChecked(
        settings.value(QSL("studio/appearance/compliantheight"), 1).toInt() ? true : false);
    heightb->setValue(settings.value(QSL("studio/appearance/height"), 50.0f).toFloat());
    bwidth->setValue(settings.value(QSL("studio/appearance/border"), 0).toInt());
    spnWhitespace->setValue(settings.value(QSL("studio/appearance/whitespace"), 0).toInt());
    spnVWhitespace->setValue(settings.value(QSL("studio/appearance/vwhitespace"), 0).toInt());
    spnScale->setValue(settings.value(QSL("studio/appearance/scale"), 1.0).toFloat());
    btype->setCurrentIndex(settings.value(QSL("studio/appearance/border_type"), 0).toInt());
    cmbFontSetting->setCurrentIndex(settings.value(QSL("studio/appearance/font_setting"), 0).toInt());
    chkHRTShow->setChecked(settings.value(QSL("studio/appearance/chk_hrt_show"), 1).toInt() ? true : false);
    chkCMYK->setChecked(settings.value(QSL("studio/appearance/chk_cmyk"), 0).toInt() ? true : false);
    chkQuietZones->setChecked(settings.value(QSL("studio/appearance/chk_quiet_zones"), 0).toInt() ? true : false);
    cmbRotate->setCurrentIndex(settings.value(QSL("studio/appearance/rotate"), 0).toInt());
    chkDotty->setChecked(settings.value(QSL("studio/appearance/chk_dotty"), 0).toInt() ? true : false);
    spnDotSize->setValue(settings.value(QSL("studio/appearance/dot_size"), 4.0 / 5.0).toFloat());

    change_options();

    scene->addItem(&m_bc);

    connect(bstyle, SIGNAL(currentIndexChanged( int )), SLOT(change_options()));
    connect(bstyle, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(filter_bstyle, SIGNAL(textChanged( const QString& )), SLOT(filter_symbologies()));
    connect(heightb, SIGNAL(valueChanged( double )), SLOT(update_preview()));
    connect(bwidth,  SIGNAL(valueChanged( int )), SLOT(update_preview()));
    connect(btype, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(cmbFontSetting, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(txtData, SIGNAL(textChanged( const QString& )), SLOT(update_preview()));
    connect(txtDataSeg1, SIGNAL(textChanged( const QString& )), SLOT(data_ui_set()));
    connect(txtDataSeg1, SIGNAL(textChanged( const QString& )), SLOT(update_preview()));
    connect(txtDataSeg2, SIGNAL(textChanged( const QString& )), SLOT(data_ui_set()));
    connect(txtDataSeg2, SIGNAL(textChanged( const QString& )), SLOT(update_preview()));
    connect(txtDataSeg3, SIGNAL(textChanged( const QString& )), SLOT(data_ui_set()));
    connect(txtDataSeg3, SIGNAL(textChanged( const QString& )), SLOT(update_preview()));
    connect(txtComposite, SIGNAL(textChanged()), SLOT(update_preview()));
    connect(chkComposite, SIGNAL(toggled( bool )), SLOT(composite_ui_set()));
    connect(chkComposite, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(cmbCompType, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(cmbECI, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(cmbECISeg1, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(cmbECISeg2, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(cmbECISeg3, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(chkEscape, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkData, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkRInit, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkGS1Parens, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkGS1NoCheck, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(spnWhitespace, SIGNAL(valueChanged( int )), SLOT(update_preview()));
    connect(spnVWhitespace, SIGNAL(valueChanged( int )), SLOT(update_preview()));
    connect(btnMenu, SIGNAL(clicked( bool )), SLOT(menu()));
    connect(btnSave, SIGNAL(clicked( bool )), SLOT(save()));
    connect(spnScale, SIGNAL(valueChanged( double )), SLOT(change_print_scale()));
    connect(btnExit, SIGNAL(clicked( bool )), SLOT(quit_now()));
    connect(btnReset, SIGNAL(clicked( bool )), SLOT(reset_colours()));
    connect(btnMoreData, SIGNAL(clicked( bool )), SLOT(open_data_dialog()));
    connect(btnMoreDataSeg1, SIGNAL(clicked( bool )), SLOT(open_data_dialog_seg1()));
    connect(btnMoreDataSeg2, SIGNAL(clicked( bool )), SLOT(open_data_dialog_seg2()));
    connect(btnMoreDataSeg3, SIGNAL(clicked( bool )), SLOT(open_data_dialog_seg3()));
    connect(btnSequence, SIGNAL(clicked( bool )), SLOT(open_sequence_dialog()));
    connect(chkAutoHeight, SIGNAL(toggled( bool )), SLOT(autoheight_ui_set()));
    connect(chkAutoHeight, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkCompliantHeight, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkHRTShow, SIGNAL(toggled( bool )), SLOT(HRTShow_ui_set()));
    connect(chkHRTShow, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(chkCMYK, SIGNAL(toggled( bool )), SLOT(change_cmyk()));
    connect(chkQuietZones, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(cmbRotate, SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
    connect(chkDotty, SIGNAL(toggled( bool )), SLOT(dotty_ui_set()));
    connect(chkDotty, SIGNAL(toggled( bool )), SLOT(update_preview()));
    connect(spnDotSize, SIGNAL(valueChanged( double )), SLOT(update_preview()));
    connect(btnCopySVG, SIGNAL(clicked( bool )), SLOT(copy_to_clipboard_svg()));
    connect(btnCopyBMP, SIGNAL(clicked( bool )), SLOT(copy_to_clipboard_bmp()));

    connect(&m_bc.bc, SIGNAL(encoded()), SLOT(on_encoded()));
    connect(&m_bc.bc, SIGNAL(errored()), SLOT(on_errored()));

    connect(view, SIGNAL(customContextMenuRequested(const QPoint &)), this, SLOT(view_context_menu(const QPoint &)));
    connect(errtxtBar, SIGNAL(customContextMenuRequested(const QPoint &)), this,
            SLOT(errtxtBar_context_menu(const QPoint &)));

    // Will enable/disable these on error
    m_saveAsShortcut = new QShortcut(QKeySequence::Save, this);
    connect(m_saveAsShortcut, SIGNAL(activated()), SLOT(save()));
    m_openCLIShortcut = new QShortcut(openCLISeq, this);
    connect(m_openCLIShortcut, SIGNAL(activated()), SLOT(open_cli_dialog()));
    m_copyBMPShortcut = new QShortcut(copyBMPSeq, this);
    connect(m_copyBMPShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_bmp()));
    m_copyEMFShortcut = new QShortcut(copyEMFSeq, this);
    connect(m_copyEMFShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_emf()));
    m_copyGIFShortcut = new QShortcut(copyGIFSeq, this);
    connect(m_copyGIFShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_gif()));
#ifndef NO_PNG
    m_copyPNGShortcut = new QShortcut(copyPNGSeq, this);
    connect(m_copyPNGShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_png()));
#endif
    m_copySVGShortcut = new QShortcut(copySVGSeq, this);
    connect(m_copySVGShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_svg()));
    m_copyTIFShortcut = new QShortcut(copyTIFSeq, this);
    connect(m_copyTIFShortcut, SIGNAL(activated()), SLOT(copy_to_clipboard_tif()));

    QShortcut *helpShortcut = new QShortcut(QKeySequence::HelpContents, this);
    connect(helpShortcut, SIGNAL(activated()), SLOT(help()));
    QShortcut *quitShortcut = new QShortcut(quitKeySeq, this);
    connect(quitShortcut, SIGNAL(activated()), SLOT(quit_now()));

    createActions();
    createMenu();
    bstyle->setFocus();

    tabMain->installEventFilter(this);
}

MainWindow::~MainWindow()
{
    QSettings settings;
#if QT_VERSION < 0x60000
    settings.setIniCodec("UTF-8");
#endif

    settings.setValue(QSL("studio/window_geometry"), saveGeometry());
    settings.setValue(QSL("studio/tab_index"), tabMain->currentIndex());
    settings.setValue(QSL("studio/symbology"), bstyle->currentIndex());
    settings.setValue(QSL("studio/ink/red"), m_fgcolor.red());
    settings.setValue(QSL("studio/ink/green"), m_fgcolor.green());
    settings.setValue(QSL("studio/ink/blue"), m_fgcolor.blue());
    settings.setValue(QSL("studio/ink/alpha"), m_fgcolor.alpha());
    settings.setValue(QSL("studio/paper/red"), m_bgcolor.red());
    settings.setValue(QSL("studio/paper/green"), m_bgcolor.green());
    settings.setValue(QSL("studio/paper/blue"), m_bgcolor.blue());
    settings.setValue(QSL("studio/paper/alpha"), m_bgcolor.alpha());
    settings.setValue(QSL("studio/data"), txtData->text());
    /* Seg data not saved so don't restore */
    settings.setValue(QSL("studio/composite_text"), txtComposite->toPlainText());
    settings.setValue(QSL("studio/chk_composite"), chkComposite->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/comp_type"), cmbCompType->currentIndex());
    settings.setValue(QSL("studio/eci"), cmbECI->currentIndex());
    /* Seg ECIs not saved so don't restore */
    settings.setValue(QSL("studio/chk_escape"), chkEscape->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/chk_data"), chkData->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/chk_rinit"), chkRInit->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/chk_gs1parens"), chkGS1Parens->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/chk_gs1nocheck"), chkGS1NoCheck->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/autoheight"), chkAutoHeight->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/compliantheight"), chkCompliantHeight->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/height"), heightb->value());
    settings.setValue(QSL("studio/appearance/border"), bwidth->value());
    settings.setValue(QSL("studio/appearance/whitespace"), spnWhitespace->value());
    settings.setValue(QSL("studio/appearance/vwhitespace"), spnVWhitespace->value());
    settings.setValue(QSL("studio/appearance/scale"), spnScale->value());
    settings.setValue(QSL("studio/appearance/border_type"), btype->currentIndex());
    settings.setValue(QSL("studio/appearance/font_setting"), cmbFontSetting->currentIndex());
    settings.setValue(QSL("studio/appearance/chk_hrt_show"), chkHRTShow->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/chk_cmyk"), chkCMYK->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/chk_quiet_zones"), chkQuietZones->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/rotate"), cmbRotate->currentIndex());
    settings.setValue(QSL("studio/appearance/chk_dotty"), chkDotty->isChecked() ? 1 : 0);
    settings.setValue(QSL("studio/appearance/dot_size"), spnDotSize->value());

    save_sub_settings(settings, m_bc.bc.symbol());
}

QString MainWindow::get_zint_version(void)
{
    QString zint_version;

    int lib_version = ZBarcode_Version();
    int version_major = lib_version / 10000;
    int version_minor = (lib_version % 10000) / 100;
    int version_release = lib_version % 100;
    int version_build;

    if (version_release >= 9) {
        /* This is a test release */
        version_release = version_release / 10;
        version_build = lib_version % 10;
        QTextStream(&zint_version) << version_major << '.' << version_minor << '.' << version_release
                                    << '.' << version_build << QSL(" (dev)");
    } else {
        /* This is a stable release */
        QTextStream(&zint_version) << version_major << '.' << version_minor << '.' << version_release;
    }

    return zint_version;
}

void MainWindow::resizeEvent(QResizeEvent *event)
{
    QWidget::resizeEvent(event);
    update_preview();
}

bool MainWindow::event(QEvent *event)
{
    switch (event->type()) {
        case QEvent::StatusTip:
            statusBar->showMessage(static_cast<QStatusTipEvent*>(event)->tip());
            break;
        default:
            break;
    }

    return QWidget::event(event);
}

bool MainWindow::eventFilter(QObject *watched, QEvent *event)
{
    if (event->type() == QEvent::ShortcutOverride) {
        QKeyEvent *keyEvent = static_cast<QKeyEvent *>(event);
        if (keyEvent->modifiers().testFlag(Qt::AltModifier) && keyEvent->key() == 'O') {
            event->ignore();
            txtData->setFocus();
            return true;
        }
    }

    return QWidget::eventFilter(watched, event);
}

void MainWindow::reset_colours()
{
    m_fgcolor.setRgb(0, 0, 0, 0xff);
    m_bgcolor.setRgb(0xff, 0xff, 0xff, 0xff);
    update_preview();
}

bool MainWindow::save()
{
    QSettings settings;
#if QT_VERSION < 0x60000
    settings.setIniCodec("UTF-8");
#endif
    QFileDialog save_dialog;
    QString pathname;
    QString suffix;
    QStringList suffixes;

    save_dialog.setAcceptMode(QFileDialog::AcceptSave);
    save_dialog.setWindowTitle(tr("Save Barcode Image"));
    save_dialog.setDirectory(settings.value(QSL("studio/default_dir"),
                QDir::toNativeSeparators(QDir::homePath())).toString());

    suffixes << QSL("eps") << QSL("gif") << QSL("svg") << QSL("bmp") << QSL("pcx") << QSL("emf") << QSL("tif");
#ifdef NO_PNG
    suffix = settings.value(QSL("studio/default_suffix"), QSL("gif")).toString();
    save_dialog.setNameFilter(tr(
        "Encapsulated PostScript (*.eps);;Graphics Interchange Format (*.gif)"
        ";;Scalable Vector Graphic (*.svg);;Windows Bitmap (*.bmp);;ZSoft PC Painter Image (*.pcx)"
        ";;Enhanced Metafile (*.emf);;Tagged Image File Format (*.tif)"));
#else
    suffix = settings.value(QSL("studio/default_suffix"), QSL("png")).toString();
    save_dialog.setNameFilter(tr(
        "Portable Network Graphic (*.png);;Encapsulated PostScript (*.eps);;Graphics Interchange Format (*.gif)"
        ";;Scalable Vector Graphic (*.svg);;Windows Bitmap (*.bmp);;ZSoft PC Painter Image (*.pcx)"
        ";;Enhanced Metafile (*.emf);;Tagged Image File Format (*.tif)"));
    suffixes << QSL("png");
#endif

    if (QString::compare(suffix, QSL("png"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Portable Network Graphic (*.png)"));
    else if (QString::compare(suffix, QSL("eps"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Encapsulated PostScript (*.eps)"));
    else if (QString::compare(suffix, QSL("gif"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Graphics Interchange Format (*.gif)"));
    else if (QString::compare(suffix, QSL("svg"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Scalable Vector Graphic (*.svg)"));
    else if (QString::compare(suffix, QSL("bmp"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Windows Bitmap (*.bmp)"));
    else if (QString::compare(suffix, QSL("pcx"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("ZSoft PC Painter Image (*.pcx)"));
    else if (QString::compare(suffix, QSL("emf"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Enhanced Metafile (*.emf)"));
    else if (QString::compare(suffix, QSL("tif"), Qt::CaseInsensitive) == 0)
        save_dialog.selectNameFilter(tr("Tagged Image File Format (*.tif)"));

    if (save_dialog.exec()) {
        pathname = save_dialog.selectedFiles().at(0);
        if ((pathname.lastIndexOf('.') == -1) || (pathname.lastIndexOf('.') < (pathname.length() - 5))) {
            suffix = save_dialog.selectedNameFilter();
            suffix = suffix.mid((suffix.lastIndexOf('.') + 1), 3);
            pathname.append('.');
            pathname.append(suffix);
        } else {
            suffix = pathname.right(pathname.length() - (pathname.lastIndexOf('.') + 1));
            if (!suffixes.contains(suffix, Qt::CaseInsensitive)) {
                /*: %1 is suffix of filename */
                QMessageBox::critical(this, tr("Save Error"), tr("Unknown output format \"%1\"").arg(suffix));
                return false;
            }
        }
    } else {
        return false;
    }

    if (m_bc.bc.save_to_file(pathname) == false) {
        if (m_bc.bc.getError() >= ZINT_ERROR) {
            QMessageBox::critical(this, tr("Save Error"), m_bc.bc.error_message());
            return false;
        }
        QMessageBox::warning(this, tr("Save Warning"), m_bc.bc.error_message());
    }

    QString nativePathname = QDir::toNativeSeparators(pathname);
    int lastSeparator = nativePathname.lastIndexOf(QDir::separator());
    QString dirname = nativePathname.mid(0, lastSeparator);
    if (dirname.isEmpty()) {
        /*: %1 is path saved to */
        statusBar->showMessage(tr("Saved as \"%1\"").arg(nativePathname), 0 /*No timeout*/);
    } else {
        QString filename = nativePathname.right(nativePathname.length() - (lastSeparator + 1));
        /*: %1 is base filename saved to, %2 is directory saved in */
        statusBar->showMessage(tr("Saved as \"%1\" in \"%2\"").arg(filename).arg(dirname), 0 /*No timeout*/);
    }

    settings.setValue(QSL("studio/default_dir"), dirname);
    settings.setValue(QSL("studio/default_suffix"), suffix);
    return true;
}

void MainWindow::about()
{
    QString zint_version = get_zint_version();

    QMessageBox::about(this, tr("About Zint"),
        /*: %1 is Zint version, %2 is Qt version */
        tr("<h2>Zint Barcode Studio %1</h2>"
           "<p>A free barcode generator</p>"
           "<p>Instruction manual is available at the project homepage:<br>"
           "<a href=\"http://www.zint.org.uk\">http://www.zint.org.uk</a>"
           "<p>Copyright &copy; 2006-2022 Robin Stuart and others.<br>"
           "Qt backend by BogDan Vatra</p>"
           "<p>Qt version %2</p>"
           "<p>With thanks to Harald Oehlmann, Norbert Szab&oacute;, Robert Elliott, Milton Neal, "
               "Git Lost, Alonso Schaich, Andre Maute and many others at Sourceforge.</p>"
           "<p>Released under the GNU General Public License ver. 3 or later.<br>"
           "\"QR Code\" is a Registered Trademark of Denso Corp.<br>"
           "\"Telepen\" is a Registered Trademark of SB Electronics.</p>"
           "<p><table border=1><tr><td><small>Currently supported standards include:<br>"
           "EN 798:1996, EN 12323:2005, ISO/IEC 15420:2009,<br>"
           "ISO/IEC 15417:2007, ISO/IEC 15438:2015, ISO/IEC 16022:2006,<br>"
           "ISO/IEC 16023:2000, ISO/IEC 16388:2007, ISO/IEC 18004:2015,<br>"
           "ISO/IEC 20830:2021, ISO/IEC 24723:2010, ISO/IEC 24724:2011,<br>"
           "ISO/IEC 24728:2006, ISO/IEC 24778:2008, ISO/IEC 16390:2007,<br>"
           "ISO/IEC 21471:2019, AIM USS Code One (1994), ANSI-HIBC 2.6-2016,<br>"
           "ANSI/AIM BC12-1998, ANSI/AIM BC6-2000, ANSI/AIM BC5-1995,<br>"
           "AIM ISS-X-24, AIMD014 (v 1.63), AIM ITS/04-023 (2022)"
           "</small></td></tr></table></p>").arg(zint_version).arg(QT_VERSION_STR));
}

void MainWindow::help()
{
    QDesktopServices::openUrl(QSL("https://zint.org.uk/manual")); // TODO: manual.md
}

void MainWindow::open_data_dialog_seg(const int seg_no)
{
    if (seg_no < 0 || seg_no > 3) {
        return;
    }
    static QLineEdit *edits[4] = {
        txtData, txtDataSeg1, txtDataSeg2, txtDataSeg3
    };
    DataWindow dlg(edits[seg_no]->text(), chkEscape->isChecked());
    (void) dlg.exec();
    if (dlg.Valid) {
        const bool updated = edits[seg_no]->text() != dlg.DataOutput;
        edits[seg_no]->setText(dlg.DataOutput);
        if (updated) {
            static const QString updatedEscTxts[4] = {
                tr("Set \"Parse Escapes\", updated data"),
                tr("Set \"Parse Escapes\", updated segment 1 data"),
                tr("Set \"Parse Escapes\", updated segment 2 data"),
                tr("Set \"Parse Escapes\", updated segment 3 data"),
            };
            static const QString updatedTxts[4] = {
                tr("Updated data"),
                tr("Updated segment 1 data"),
                tr("Updated segment 2 data"),
                tr("Updated segment 3 data"),
            };
            if (dlg.Escaped && !chkEscape->isChecked()) {
                chkEscape->setChecked(true);
                statusBar->showMessage(updatedEscTxts[seg_no], tempMessageTimeout);
            } else {
                statusBar->showMessage(updatedTxts[seg_no], tempMessageTimeout);
            }
        }
    }
}

void MainWindow::open_data_dialog()
{
    open_data_dialog_seg(0);
}

void MainWindow::open_data_dialog_seg1()
{
    open_data_dialog_seg(1);
}

void MainWindow::open_data_dialog_seg2()
{
    open_data_dialog_seg(2);
}

void MainWindow::open_data_dialog_seg3()
{
    open_data_dialog_seg(3);
}

void MainWindow::open_sequence_dialog()
{
    SequenceWindow dlg(&m_bc);
    (void) dlg.exec();
#ifdef _WIN32
    // Windows causes BarcodeItem to paint on closing dialog so need to re-paint with our (non-Export) values
    update_preview();
#endif
}

void MainWindow::open_cli_dialog()
{
    CLIWindow dlg(&m_bc, chkAutoHeight->isEnabled() && chkAutoHeight->isChecked(),
                    m_spnHeightPerRow && m_spnHeightPerRow->isEnabled() ? m_spnHeightPerRow->value() : 0.0);

    (void) dlg.exec();
}

void MainWindow::on_fgcolor_clicked()
{
    QColor temp = m_fgcolor;
    m_fgcolor = QColorDialog::getColor(m_fgcolor, this, tr("Set foreground colour"),
                    QColorDialog::DontUseNativeDialog | QColorDialog::ShowAlphaChannel);
    if (m_fgcolor.isValid()) {
        update_preview();
    } else {
        m_fgcolor = temp;
    }
}

void MainWindow::on_bgcolor_clicked()
{
    QColor temp = m_bgcolor;
    m_bgcolor = QColorDialog::getColor(m_bgcolor, this, tr("Set background colour"),
                    QColorDialog::DontUseNativeDialog | QColorDialog::ShowAlphaChannel);
    if (m_bgcolor.isValid()) {
        update_preview();
    } else {
        m_bgcolor = temp;
    }
}

void MainWindow::autoheight_ui_set()
{
    bool enabled = chkAutoHeight->isEnabled() && !chkAutoHeight->isChecked();
    lblHeight->setEnabled(enabled);
    heightb->setEnabled(enabled);

    if (m_lblHeightPerRow && m_spnHeightPerRow) {
        m_lblHeightPerRow->setEnabled(enabled);
        m_spnHeightPerRow->setEnabled(enabled);
        if (enabled && m_spnHeightPerRow->value()) {
            lblHeight->setEnabled(!enabled);
            heightb->setEnabled(!enabled);
            statusBar->showMessage(tr("Using \"Row Height\""), 0 /*No timeout*/);
        } else {
            statusBar->clearMessage();
        }
        if (m_btnHeightPerRowDisable) {
            m_btnHeightPerRowDisable->setEnabled(enabled && m_spnHeightPerRow->value());
        }
        if (m_btnHeightPerRowDefault) {
            m_btnHeightPerRowDefault->setEnabled(enabled);
        }
    }
}

void MainWindow::HRTShow_ui_set()
{
    bool enabled = chkHRTShow->isEnabled() && chkHRTShow->isChecked();
    lblFontSetting->setEnabled(enabled);
    cmbFontSetting->setEnabled(enabled);
}

void MainWindow::dotty_ui_set()
{
    int symbology = bstyle_items[bstyle->currentIndex()].symbology;

    if (symbology == BARCODE_DOTCODE) {
        chkDotty->setEnabled(false);
        lblDotSize->setEnabled(true);
        spnDotSize->setEnabled(true);
    } else {
        bool enabled = chkDotty->isEnabled() && chkDotty->isChecked();
        lblDotSize->setEnabled(enabled);
        spnDotSize->setEnabled(enabled);
    }
}

void MainWindow::codeone_ui_set()
{
    int symbology = bstyle_items[bstyle->currentIndex()].symbology;
    if (symbology != BARCODE_CODEONE)
        return;

    QGroupBox *groupBox = m_optionWidget->findChild<QGroupBox*>(QSL("groupBoxC1StructApp"));
    if (groupBox) {
        bool enabled = get_cmb_index(QSL("cmbC1Size")) != 9; // Not Version S
        groupBox->setEnabled(enabled);
    }
}

void MainWindow::structapp_ui_set()
{
    int symbology = bstyle_items[bstyle->currentIndex()].symbology;
    QString name;
    bool enabled = false;
    QWidget *widgetCount = nullptr, *widgetIndex = nullptr;
    QLabel *lblID2 = nullptr;
    QWidget *widgetID = nullptr, *widgetID2 = nullptr;

    if (symbology == BARCODE_AZTEC) {
        name = QSL("Aztec");
        widgetID = get_widget(QSL("txt") + name + QSL("StructAppID"));
    } else if (symbology == BARCODE_CODEONE) {
        name = QSL("C1");
        QSpinBox *spnCount = m_optionWidget->findChild<QSpinBox*>(QSL("spn") + name + QSL("StructAppCount"));
        enabled = spnCount ? spnCount->value() > 1 : false;
        widgetCount = spnCount;
        widgetIndex = get_widget(QSL("spn") + name + QSL("StructAppIndex"));
    } else if (symbology == BARCODE_DATAMATRIX) {
        name = QSL("DM");
        widgetID = get_widget(QSL("spn") + name + QSL("StructAppID"));
        widgetID2 = get_widget(QSL("spn") + name + QSL("StructAppID2"));
    } else if (symbology == BARCODE_DOTCODE) {
        name = QSL("Dot");
    } else if (symbology == BARCODE_MAXICODE) {
        name = QSL("Maxi");
    } else if (symbology == BARCODE_PDF417 || symbology == BARCODE_MICROPDF417) {
        name = symbology == BARCODE_PDF417 ? QSL("PDF") : QSL("MPDF");
        QSpinBox *spnCount = m_optionWidget->findChild<QSpinBox*>(QSL("spn") + name + QSL("StructAppCount"));
        enabled = spnCount ? spnCount->value() > 1 : false;
        widgetCount = spnCount;
        widgetIndex = get_widget(QSL("spn") + name + QSL("StructAppIndex"));
        widgetID = get_widget(QSL("txt") + name + QSL("StructAppID"));
    } else if (symbology == BARCODE_QRCODE) {
        name = QSL("QR");
        widgetID = get_widget(QSL("spn") + name + QSL("StructAppID"));
    } else if (symbology == BARCODE_GRIDMATRIX) {
        name = QSL("Grid");
        widgetID = get_widget(QSL("spn") + name + QSL("StructAppID"));
    } else if (symbology == BARCODE_ULTRA) {
        name = QSL("Ultra");
        widgetID = get_widget(QSL("spn") + name + QSL("StructAppID"));
    }
    if (!name.isEmpty()) {
        QLabel *lblIndex = m_optionWidget->findChild<QLabel*>(QSL("lbl") + name + QSL("StructAppIndex"));
        if (!widgetCount) {
            QComboBox *cmbCount = m_optionWidget->findChild<QComboBox*>(QSL("cmb") + name + QSL("StructAppCount"));
            enabled = cmbCount ? cmbCount->currentIndex() != 0 : false;
            widgetCount = cmbCount;
        }
        if (!widgetIndex) {
            widgetIndex = get_widget(QSL("cmb") + name + QSL("StructAppIndex"));
        }
        if (lblIndex && widgetCount && widgetIndex) {
            lblIndex->setEnabled(enabled);
            widgetIndex->setEnabled(enabled);
            QLabel *lblID = m_optionWidget->findChild<QLabel*>(QSL("lbl") + name + QSL("StructAppID"));
            if (lblID) {
                lblID->setEnabled(enabled);
                if (lblID2) {
                    lblID2->setEnabled(enabled);
                }
            }
            if (widgetID) {
                widgetID->setEnabled(enabled);
                if (widgetID2) {
                    widgetID2->setEnabled(enabled);
                }
            }
        }
    }
}

void MainWindow::on_encoded()
{
    if (QApplication::activeModalWidget() != nullptr) { // Protect against encode in popup dialog
        return;
    }
    this->setMinimumHeight(m_bc.bc.getError() ? 455 : 435);
    enableActions(true);
    errtxtBar_set(false /*isError*/);

    if (!chkAutoHeight->isEnabled() || chkAutoHeight->isChecked() || !heightb->isEnabled()) {
        /* setValue() rounds up/down to precision (decimals 3), we want round up only */
        float height = (float) (ceil(m_bc.bc.height() * 1000.0f) / 1000.0f);
        heightb->setValue(height);
    }
}

void MainWindow::on_errored()
{
    if (QApplication::activeModalWidget() != nullptr) { // Protect against error in popup dialog (Sequence Export)
        return;
    }
    this->setMinimumHeight(455);
    enableActions(false);
    errtxtBar_set(true /*isError*/);
}

void MainWindow::filter_symbologies()
{
    // `simplified()` trims and reduces inner whitespace to a single space - nice!
    QString filter = filter_bstyle->text().simplified();
    QListView *lview = qobject_cast<QListView *>(bstyle->view());
    QStandardItemModel *model = qobject_cast<QStandardItemModel*>(bstyle->model());
    QStandardItem *item;

    if (!lview || !model) {
        return;
    }

    /* QString::split() only introduced Qt 5.14, so too new for us to use */
    QStringList filter_list;
    if (!filter.isEmpty()) {
        int i, j;
        for (i = 0; (j = filter.indexOf(' ', i)) != -1; i = j + 1) {
            filter_list << filter.mid(i, j - i);
        }
        filter_list << filter.mid(i);
    }
    int filter_cnt = filter_list.size();
    int cnt = (int) (sizeof(bstyle_items) / sizeof(bstyle_items[0]));

    if (filter_cnt) {
        for (int i = 0; i < cnt; i++) {
            bool hidden = lview->isRowHidden(i);
            bool hide = true;
            for (int j = 0; j < filter_cnt; j++) {
                if (bstyle->itemText(i).contains(filter_list[j], Qt::CaseInsensitive)) {
                    hide = false;
                    break;
                }
            }
            if ((hide && !hidden) || (!hide && hidden)) {
                // https://stackoverflow.com/questions/25172220
                // /how-to-hide-qcombobox-items-instead-of-clearing-them-out
                item = model->item(i);
                item->setFlags(hide ? item->flags() & ~Qt::ItemIsEnabled : item->flags() | Qt::ItemIsEnabled);
                lview->setRowHidden(i, hide);
            }
        }
    } else {
        for (int i = 0; i < cnt; i++) {
            if (lview->isRowHidden(i)) {
                item = model->item(i);
                item->setFlags(item->flags() | Qt::ItemIsEnabled);
                lview->setRowHidden(i, false);
            }
        }
    }
}

void MainWindow::change_print_scale()
{
    /* This value is only used when printing (saving) to file */
    m_bc.bc.setScale((float) spnScale->value());
}

void MainWindow::change_cmyk()
{
    /* This value is only used when printing (saving) to file */
    m_bc.bc.setCMYK(chkCMYK->isChecked());
}

void MainWindow::quit_now()
{
    close();
}

void MainWindow::menu()
{
    QSize size = m_menu->sizeHint();
    m_menu->exec(btnMenu->mapToGlobal(QPoint(0, -size.height())));
}

void MainWindow::copy_to_clipboard_bmp()
{
    copy_to_clipboard(QSL(".zint.bmp"), QSL("BMP"));
}

void MainWindow::copy_to_clipboard_emf()
{
    copy_to_clipboard(QSL(".zint.emf"), QSL("EMF"), "image/x-emf");
}

void MainWindow::copy_to_clipboard_eps()
{
    // TODO: try other possibles application/eps, application/x-eps, image/eps, image/x-eps
    copy_to_clipboard(QSL(".zint.eps"), QSL("EPS"), "application/postscript");
}

void MainWindow::copy_to_clipboard_gif()
{
    copy_to_clipboard(QSL(".zint.gif"), QSL("GIF"));
}

void MainWindow::copy_to_clipboard_pcx()
{
    // TODO: try other mime types in various apps
    copy_to_clipboard(QSL(".zint.pcx"), QSL("PCX"), "image/x-pcx");
}

#ifndef NO_PNG
void MainWindow::copy_to_clipboard_png()
{
    copy_to_clipboard(QSL(".zint.png"), QSL("PNG"));
}
#endif

void MainWindow::copy_to_clipboard_svg()
{
    copy_to_clipboard(QSL(".zint.svg"), QSL("SVG"));
}

void MainWindow::copy_to_clipboard_tif()
{
    copy_to_clipboard(QSL(".zint.tif"), QSL("TIF"));
}

void MainWindow::copy_to_clipboard_errtxt()
{
    if (m_bc.bc.hasErrors()) {
        QClipboard *clipboard = QGuiApplication::clipboard();
        QMimeData *mdata = new QMimeData;
        mdata->setText(m_bc.bc.lastError());
        clipboard->setMimeData(mdata, QClipboard::Clipboard);
        statusBar->showMessage(tr("Copied message to clipboard"), 0 /*No timeout*/);
    }
}

void MainWindow::height_per_row_disable()
{
    if (m_spnHeightPerRow) {
        m_spnHeightPerRow->setValue(0.0);
    }
}

void MainWindow::height_per_row_default()
{
    if (m_spnHeightPerRow && m_btnHeightPerRowDefault) {
        const QString &name = m_btnHeightPerRowDefault->objectName();
        double val = 0.0;
        if (name == QSL("btnPDFHeightPerRowDefault")) {
            val = 3.0;
        } else if (name == QSL("btnMPDFHeightPerRowDefault")) {
            val = 2.0;
        } else if (name == QSL("btnC16kHeightPerRowDefault")) {
            if (chkCompliantHeight->isEnabled() && chkCompliantHeight->isChecked()) {
                const int rows = m_bc.bc.encodedRows();
                val = 10.0 + (double)((rows - 1) * (get_cmb_index(QSL("cmbC16kRowSepHeight")) + 1)) / rows;
            } else {
                val = 10.0;
            }
        } else if (name == QSL("btnCbfHeightPerRowDefault")) {
            // Depends on no. of data cols
            const int cols = (m_bc.bc.encodedWidth() - 57) / 11; // 57 = 4 * 11 (start/subset/checks) + 13 (stop char)
            val = 0.55 * cols + 3;
            if (val < 10.0) {
                val = 10.0;
            }
        } else if (name == QSL("btnC49HeightPerRowDefault")) {
            if (chkCompliantHeight->isEnabled() && chkCompliantHeight->isChecked()) {
                const int rows = m_bc.bc.encodedRows();
                val = 10.0 + (double)((rows - 1) * (get_cmb_index(QSL("cmbC49RowSepHeight")) + 1)) / rows;
            } else {
                val = 10.0;
            }
        } else if (name == QSL("btnDBESHeightPerRowDefault")) {
            val = 34.0;
        }
        if (val) {
            m_spnHeightPerRow->setValue(val);
        }
    }
}

void MainWindow::guard_reset_upcean()
{
    guard_reset(QSL("spnUPCEANGuardDescent"));
}

void MainWindow::guard_reset_upca()
{
    guard_reset(QSL("spnUPCAGuardDescent"));
}

void MainWindow::view_context_menu(const QPoint &pos)
{
    QMenu menu(tr("View Menu"), view);

    if (m_bc.bc.getError() >= ZINT_ERROR) {
        menu.addAction(m_copyErrtxtAct);

        menu.exec(get_context_menu_pos(pos, view));
    } else {
        menu.addAction(m_copyBMPAct);
        menu.addAction(m_copyEMFAct);
#ifdef MAINWINDOW_COPY_EPS
        menu.addAction(m_copyEPSAct);
#endif
        menu.addAction(m_copyGIFAct);
#ifdef MAINWINDOW_COPY_PCX
        menu.addAction(m_copyPCXAct);
#endif
#ifndef NO_PNG
        menu.addAction(m_copyPNGAct);
#endif
        menu.addAction(m_copySVGAct);
        menu.addAction(m_copyTIFAct);
        menu.addSeparator();
        menu.addAction(m_openCLIAct);
        menu.addSeparator();
        menu.addAction(m_saveAsAct);

        menu.exec(get_context_menu_pos(pos, view));
    }
}

void MainWindow::errtxtBar_context_menu(const QPoint &pos)
{
    QMenu menu(tr("Message Menu"), errtxtBar);

    menu.addAction(m_copyErrtxtAct);

    menu.exec(get_context_menu_pos(pos, errtxtBar));
}

void MainWindow::change_options()
{
    QUiLoader uiload;
    QSettings settings;
#if QT_VERSION < 0x60000
    settings.setIniCodec("UTF-8");
#endif

    bool initial_load = m_symbology == 0;
    int original_tab_count = tabMain->count();
    int original_tab_index = tabMain->currentIndex();
    int symbology = bstyle_items[bstyle->currentIndex()].symbology;

    if (m_symbology) {
        save_sub_settings(settings, m_symbology);
    }
    statusBar->clearMessage();

    if (tabMain->count() == 3)
        tabMain->removeTab(1);

    chkComposite->setText(tr("Add &2D Component"));
    combobox_item_enabled(cmbCompType, 3, false); // CC-C
    btype->setItemText(0, tr("No border"));
    combobox_item_enabled(cmbFontSetting, 1, true);
    m_lblHeightPerRow = nullptr;
    m_spnHeightPerRow = nullptr;
    m_btnHeightPerRowDisable = nullptr;
    m_btnHeightPerRowDefault = nullptr;

    if (symbology == BARCODE_CODE128) {
        QFile file(QSL(":/grpC128.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Cod&e 128"));
        chkComposite->setText(tr("Add &2D Component (GS1-128 only)"));
        combobox_item_enabled(cmbCompType, 3, true); // CC-C
        set_smaller_font(QSL("noteC128CompositeEAN"));
        connect(get_widget(QSL("radC128Stand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC128CSup")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC128EAN")), SIGNAL(toggled( bool )), SLOT(composite_ean_check()));
        connect(get_widget(QSL("radC128EAN")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC128HIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_PDF417) {
        QFile file(QSL(":/grpPDF417.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("PDF41&7"));
        connect(get_widget(QSL("cmbPDFECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbPDFCols")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbPDFRows")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblPDFHeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnPDFHeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnPDFHeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnPDFHeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));
        connect(get_widget(QSL("radPDFTruncated")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radPDFStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radPDFHIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("spnPDFStructAppCount")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnPDFStructAppCount")), SIGNAL(valueChanged( int )), SLOT(structapp_ui_set()));
        connect(get_widget(QSL("spnPDFStructAppIndex")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("txtPDFStructAppID")), SIGNAL(textChanged( const QString& )), SLOT(update_preview()));

    } else if (symbology == BARCODE_MICROPDF417) {
        QFile file(QSL(":/grpMicroPDF.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("Micro PDF41&7"));
        connect(get_widget(QSL("cmbMPDFCols")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblMPDFHeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnMPDFHeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnMPDFHeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnMPDFHeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));
        connect(get_widget(QSL("radMPDFStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("spnMPDFStructAppCount")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnMPDFStructAppCount")), SIGNAL(valueChanged( int )), SLOT(structapp_ui_set()));
        connect(get_widget(QSL("spnMPDFStructAppIndex")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("txtMPDFStructAppID")), SIGNAL(textChanged( const QString& )), SLOT(update_preview()));

    } else if (symbology == BARCODE_DOTCODE) {
        QFile file(QSL(":/grpDotCode.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("DotCod&e"));
        connect(get_widget(QSL("cmbDotCols")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDotMask")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radDotStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radDotGS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDotStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDotStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbDotStructAppIndex")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_AZTEC) {
        QFile file(QSL(":/grpAztec.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("Aztec Cod&e"));
        connect(get_widget(QSL("radAztecAuto")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radAztecSize")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radAztecECC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbAztecSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbAztecECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radAztecStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radAztecGS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radAztecHIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbAztecStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(update_preview()));
        connect(get_widget(QSL("cmbAztecStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbAztecStructAppIndex")), SIGNAL(currentIndexChanged( int )),
                SLOT(update_preview()));
        connect(get_widget(QSL("txtAztecStructAppID")), SIGNAL(textChanged( const QString& )),
                SLOT(update_preview()));

    } else if (symbology == BARCODE_MSI_PLESSEY) {
        QFile file(QSL(":/grpMSICheck.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("MSI Pless&ey"));
        connect(get_widget(QSL("cmbMSICheck")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMSICheck")), SIGNAL(currentIndexChanged( int )), SLOT(msi_plessey_ui_set()));
        connect(get_widget(QSL("chkMSICheckText")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODE11) {
        QFile file(QSL(":/grpC11.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Cod&e 11"));
        connect(get_widget(QSL("radC11TwoCheckDigits")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC11OneCheckDigit")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC11NoCheckDigits")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_C25STANDARD || symbology == BARCODE_C25INTER || symbology == BARCODE_C25IATA
            || symbology == BARCODE_C25LOGIC || symbology == BARCODE_C25IND) {
        QFile file(QSL(":/grpC25.ui"));
        if (file.open(QIODevice::ReadOnly)) {
            m_optionWidget = uiload.load(&file);
            file.close();
            load_sub_settings(settings, symbology);
            static const QString names[] = {
                QSL("Standard"), QSL("Interleaved"), QSL("IATA"), QSL(""), QSL("Data Logic"), QSL("Industrial")
            };
            /*: %1 is name of variant (Standard, Interleaved, IATA, Data Logic, Industrial) */
            tabMain->insertTab(1, m_optionWidget, tr("Cod&e 2 of 5 %1").arg(names[symbology - BARCODE_C25STANDARD]));
            connect(get_widget(QSL("radC25Stand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
            connect(get_widget(QSL("radC25Check")), SIGNAL(toggled( bool )), SLOT(update_preview()));
            connect(get_widget(QSL("radC25CheckHide")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        }

    } else if (symbology == BARCODE_CODE39 || symbology == BARCODE_EXCODE39) {
        QFile file(QSL(":/grpC39.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        connect(get_widget(QSL("radC39Stand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC39Check")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radC39HIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        if (symbology == BARCODE_EXCODE39) {
            tabMain->insertTab(1, m_optionWidget, tr("Cod&e 39 Extended"));
            if (m_optionWidget->findChild<QRadioButton*>(QSL("radC39HIBC"))->isChecked()) {
                m_optionWidget->findChild<QRadioButton*>(QSL("radC39HIBC"))->setChecked(false);
                m_optionWidget->findChild<QRadioButton*>(QSL("radC39Stand"))->setChecked(true);
            }
            m_optionWidget->findChild<QRadioButton*>(QSL("radC39HIBC"))->setEnabled(false);
        } else {
            tabMain->insertTab(1, m_optionWidget, tr("Cod&e 39"));
            m_optionWidget->findChild<QRadioButton*>(QSL("radC39HIBC"))->setEnabled(true);
        }

    } else if (symbology == BARCODE_LOGMARS) {
        QFile file(QSL(":/grpLOGMARS.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("LOGM&ARS"));
        connect(get_widget(QSL("radLOGMARSStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radLOGMARSCheck")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODE16K) {
        QFile file(QSL(":/grpC16k.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Cod&e 16K"));
        btype->setItemText(0, tr("Default (bind)"));
        connect(get_widget(QSL("cmbC16kRows")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblC16kHeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnC16kHeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnC16kHeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnC16kHeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));
        connect(get_widget(QSL("cmbC16kRowSepHeight")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radC16kStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkC16kNoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODABAR) {
        QFile file(QSL(":/grpCodabar.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Cod&abar"));
        connect(get_widget(QSL("radCodabarStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radCodabarCheckHide")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radCodabarCheck")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODABLOCKF) {
        QFile file (QSL(":/grpCodablockF.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Codablock&-F"));
        btype->setItemText(0, tr("Default (bind)"));
        connect(get_widget(QSL("cmbCbfWidth")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbCbfHeight")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblCbfHeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnCbfHeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnCbfHeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnCbfHeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));
        connect(get_widget(QSL("cmbCbfRowSepHeight")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radCbfStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radCbfHIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkCbfNoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_DAFT) {
        QFile file(QSL(":/grpDAFT.ui"));
        if (file.open(QIODevice::ReadOnly)) {
            m_optionWidget = uiload.load(&file);
            file.close();
            load_sub_settings(settings, symbology);
            tabMain->insertTab(1, m_optionWidget, tr("DAFT"));
            set_smaller_font(QSL("noteTrackerRatios"));
            connect(get_widget(QSL("spnDAFTTrackerRatio")), SIGNAL(valueChanged( double )), SLOT(update_preview()));
        }

    } else if (symbology == BARCODE_DATAMATRIX) {
        QFile file(QSL(":/grpDM.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("D&ata Matrix"));
        connect(get_widget(QSL("radDM200Stand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radDM200GS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radDM200HIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDM200Size")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("chkDMRectangle")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkDMRE")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkDMGSSep")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkDMFast")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDMStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDMStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbDMStructAppIndex")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnDMStructAppID")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnDMStructAppID2")), SIGNAL(valueChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_ITF14) {
        btype->setItemText(0, tr("Default (box, non-zero width)"));
        QFile file(QSL(":/grpITF14.ui"));
        if (file.open(QIODevice::ReadOnly)) {
            m_optionWidget = uiload.load(&file);
            file.close();
            load_sub_settings(settings, symbology);
            tabMain->insertTab(1, m_optionWidget, tr("ITF-1&4"));
            connect(get_widget(QSL("chkITF14NoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        }

    } else if (symbology == BARCODE_QRCODE) {
        QFile file(QSL(":/grpQR.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("QR Cod&e"));
        connect(get_widget(QSL("cmbQRSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbQRECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbQRMask")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radQRStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radQRGS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radQRHIBC")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkQRFullMultibyte")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbQRStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbQRStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbQRStructAppIndex")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnQRStructAppID")), SIGNAL(valueChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_UPNQR) {
        QFile file(QSL(":/grpUPNQR.ui"));
        if (file.open(QIODevice::ReadOnly)) {
            m_optionWidget = uiload.load(&file);
            file.close();
            load_sub_settings(settings, symbology);
            tabMain->insertTab(1, m_optionWidget, tr("UP&NQR"));
            connect(get_widget(QSL("cmbUPNQRMask")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        }

    } else if (symbology == BARCODE_RMQR) {
        QFile file(QSL(":/grpRMQR.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("rMQR Cod&e"));
        connect(get_widget(QSL("cmbRMQRSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbRMQRECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radRMQRStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radRMQRGS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkRMQRFullMultibyte")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_HANXIN) {
        QFile file(QSL(":/grpHX.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Han Xin Cod&e"));
        connect(get_widget(QSL("cmbHXSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbHXECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbHXMask")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("chkHXFullMultibyte")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_MICROQR) {
        QFile file(QSL(":/grpMQR.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Micro QR Cod&e"));
        connect(get_widget(QSL("cmbMQRSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMQRECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMQRMask")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("chkMQRFullMultibyte")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_GRIDMATRIX) {
        QFile file(QSL(":/grpGrid.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("Grid M&atrix"));
        set_smaller_font(QSL("noteGridECC"));
        connect(get_widget(QSL("cmbGridSize")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbGridECC")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("chkGridFullMultibyte")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbGridStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbGridStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbGridStructAppIndex")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnGridStructAppID")), SIGNAL(valueChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_MAXICODE) {
        QFile file(QSL(":/grpMaxicode.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("MaxiCod&e"));
        connect(get_widget(QSL("cmbMaxiMode")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMaxiMode")), SIGNAL(currentIndexChanged( int )), SLOT(maxi_scm_ui_set()));
        connect(get_widget(QSL("txtMaxiSCMPostcode")), SIGNAL(textChanged( const QString& )), SLOT(update_preview()));
        connect(get_widget(QSL("spnMaxiSCMCountry")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnMaxiSCMService")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("chkMaxiSCMVV")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkMaxiSCMVV")), SIGNAL(toggled( bool )), SLOT(maxi_scm_ui_set()));
        connect(get_widget(QSL("spnMaxiSCMVV")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMaxiStructAppCount")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbMaxiStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbMaxiStructAppIndex")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        maxi_scm_ui_set();

    } else if (symbology == BARCODE_CHANNEL) {
        QFile file(QSL(":/grpChannel.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Channel Cod&e"));
        connect(get_widget(QSL("cmbChannel")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODEONE) {
        QFile file(QSL(":/grpCodeOne.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        codeone_ui_set();
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("Code On&e"));
        connect(get_widget(QSL("cmbC1Size")), SIGNAL(currentIndexChanged( int )), SLOT(codeone_ui_set()));
        connect(get_widget(QSL("cmbC1Size")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radC1GS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("spnC1StructAppCount")), SIGNAL(valueChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnC1StructAppCount")), SIGNAL(valueChanged( int )), SLOT(structapp_ui_set()));
        connect(get_widget(QSL("spnC1StructAppIndex")), SIGNAL(valueChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODE49) {
        QFile file(QSL(":/grpC49.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("Cod&e 49"));
        btype->setItemText(0, tr("Default (bind)"));
        connect(get_widget(QSL("cmbC49Rows")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblC49HeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnC49HeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnC49HeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnC49HeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));
        connect(get_widget(QSL("cmbC49RowSepHeight")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radC49GS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("chkC49NoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_CODE93) {
        QFile file(QSL(":/grpC93.ui"));
        if (file.open(QIODevice::ReadOnly)) {
            m_optionWidget = uiload.load(&file);
            file.close();
            load_sub_settings(settings, symbology);
            tabMain->insertTab(1, m_optionWidget, tr("Cod&e 93"));
            connect(get_widget(QSL("chkC93ShowChecks")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        }

    } else if (symbology == BARCODE_DBAR_EXPSTK) {
        QFile file(QSL(":/grpDBExtend.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("GS1 D&ataBar Exp Stack"));
        connect(get_widget(QSL("radDBESCols")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radDBESRows")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDBESCols")), SIGNAL(currentIndexChanged ( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbDBESRows")), SIGNAL(currentIndexChanged ( int )), SLOT(update_preview()));
        m_lblHeightPerRow = m_optionWidget->findChild<QLabel*>(QSL("lblDBESHeightPerRow"));
        m_spnHeightPerRow = m_optionWidget->findChild<QDoubleSpinBox*>(QSL("spnDBESHeightPerRow"));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(autoheight_ui_set()));
        connect(m_spnHeightPerRow, SIGNAL(valueChanged( double )), SLOT(update_preview()));
        m_btnHeightPerRowDisable = m_optionWidget->findChild<QPushButton*>(QSL("btnDBESHeightPerRowDisable"));
        m_btnHeightPerRowDefault = m_optionWidget->findChild<QPushButton*>(QSL("btnDBESHeightPerRowDefault"));
        connect(m_btnHeightPerRowDisable, SIGNAL(clicked( bool )), SLOT(height_per_row_disable()));
        connect(m_btnHeightPerRowDefault, SIGNAL(clicked( bool )), SLOT(height_per_row_default()));

    } else if (symbology == BARCODE_ULTRA) {
        QFile file(QSL(":/grpUltra.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        structapp_ui_set();
        tabMain->insertTab(1, m_optionWidget, tr("Ultracod&e"));
        connect(get_widget(QSL("radUltraAuto")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radUltraEcc")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbUltraEcc")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbUltraRevision")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("radUltraStand")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("radUltraGS1")), SIGNAL(toggled( bool )), SLOT(update_preview()));
        connect(get_widget(QSL("cmbUltraStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(update_preview()));
        connect(get_widget(QSL("cmbUltraStructAppCount")), SIGNAL(currentIndexChanged( int )),
                SLOT(structapp_ui_set()));
        connect(get_widget(QSL("cmbUltraStructAppIndex")), SIGNAL(currentIndexChanged( int )),
                SLOT(update_preview()));
        connect(get_widget(QSL("spnUltraStructAppID")), SIGNAL(valueChanged( int )), SLOT(update_preview()));

    } else if (symbology == BARCODE_UPCA || symbology == BARCODE_UPCA_CHK || symbology == BARCODE_UPCA_CC) {
        QFile file(QSL(":/grpUPCA.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("UPC-&A"));
        combobox_item_enabled(cmbFontSetting, 1, false);
        if (cmbFontSetting->currentIndex() == 1) {
            cmbFontSetting->setCurrentIndex(0);
        }
        connect(get_widget(QSL("cmbUPCAAddonGap")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnUPCAGuardDescent")), SIGNAL(valueChanged( double )), SLOT(update_preview()));
        connect(get_widget(QSL("btnUPCAGuardReset")), SIGNAL(clicked( bool )), SLOT(guard_reset_upca()));
        connect(get_widget(QSL("chkUPCANoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_EANX || symbology == BARCODE_EANX_CHK || symbology == BARCODE_EANX_CC
            || symbology == BARCODE_UPCE || symbology == BARCODE_UPCE_CHK || symbology == BARCODE_UPCE_CC
            || symbology == BARCODE_ISBNX) {
        QFile file(QSL(":/grpUPCEAN.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        if (symbology == BARCODE_UPCE || symbology == BARCODE_UPCE_CHK || symbology == BARCODE_UPCE_CC) {
            tabMain->insertTab(1, m_optionWidget, tr("UPC-&E"));
        } else if (symbology == BARCODE_ISBNX) {
            tabMain->insertTab(1, m_optionWidget, tr("ISBN"));
        } else {
            tabMain->insertTab(1, m_optionWidget, tr("&EAN"));
        }
        combobox_item_enabled(cmbFontSetting, 1, false);
        if (cmbFontSetting->currentIndex() == 1) {
            cmbFontSetting->setCurrentIndex(0);
        }
        connect(get_widget(QSL("cmbUPCEANAddonGap")), SIGNAL(currentIndexChanged( int )), SLOT(update_preview()));
        connect(get_widget(QSL("spnUPCEANGuardDescent")), SIGNAL(valueChanged( double )), SLOT(update_preview()));
        connect(get_widget(QSL("btnUPCEANGuardReset")), SIGNAL(clicked( bool )), SLOT(guard_reset_upcean()));
        connect(get_widget(QSL("chkUPCEANNoQuietZones")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else if (symbology == BARCODE_VIN) {
        QFile file(QSL(":/grpVIN.ui"));
        if (!file.open(QIODevice::ReadOnly))
            return;
        m_optionWidget = uiload.load(&file);
        file.close();
        load_sub_settings(settings, symbology);
        tabMain->insertTab(1, m_optionWidget, tr("&VIN"));
        connect(get_widget(QSL("chkVINImportChar")), SIGNAL(toggled( bool )), SLOT(update_preview()));

    } else {
        load_sub_settings(settings, symbology);
    }

    switch (symbology) {
        case BARCODE_CODE128:
        case BARCODE_EANX:
        case BARCODE_UPCA:
        case BARCODE_UPCE:
        case BARCODE_DBAR_OMN:
        case BARCODE_DBAR_LTD:
        case BARCODE_DBAR_EXP:
        case BARCODE_DBAR_STK:
        case BARCODE_DBAR_OMNSTK:
        case BARCODE_DBAR_EXPSTK:
            grpComposite->show();
            grpSegs->hide();
            break;
        case BARCODE_AZTEC:
        case BARCODE_DATAMATRIX:
        case BARCODE_MAXICODE:
        case BARCODE_MICROPDF417:
        case BARCODE_PDF417:
        case BARCODE_PDF417COMP:
        case BARCODE_QRCODE:
        case BARCODE_DOTCODE:
        case BARCODE_CODEONE:
        case BARCODE_GRIDMATRIX:
        case BARCODE_HANXIN:
        case BARCODE_ULTRA:
        case BARCODE_RMQR:
            grpComposite->hide();
            grpSegs->show();
            break;
        default:
            grpComposite->hide();
            grpSegs->hide();
            break;
    }

    // ECI, GS1Parens, GS1NoCheck, RInit, CompliantHeight will be checked in update_preview() as
    // encoding mode dependent (HIBC and/or GS1)
    chkAutoHeight->setEnabled(!m_bc.bc.isFixedRatio(symbology));
    chkHRTShow->setEnabled(m_bc.bc.hasHRT(symbology));
    chkQuietZones->setEnabled(!m_bc.bc.hasDefaultQuietZones(symbology));
    chkDotty->setEnabled(m_bc.bc.isDotty(symbology));

    data_ui_set();
    composite_ui_set();
    autoheight_ui_set();
    HRTShow_ui_set();
    dotty_ui_set();

    if (initial_load) {
        tabMain->setCurrentIndex(settings.value(QSL("studio/tab_index"), 0).toInt());
    } else if (original_tab_count == tabMain->count()) {
        tabMain->setCurrentIndex(original_tab_index);
    } else if (original_tab_count > tabMain->count()) {
        tabMain->setCurrentIndex(original_tab_index == 2 ? 1 : 0);
    } else {
        tabMain->setCurrentIndex(original_tab_index == 1 ? 2 : 0);
    }
}

void MainWindow::data_ui_set()
{
    if (grpSegs->isHidden()) {
        return;
    }
    if (txtDataSeg1->text().isEmpty()) {
        cmbECISeg1->setEnabled(false);
        lblSeg2->setEnabled(false);
        cmbECISeg2->setEnabled(false);
        txtDataSeg2->setEnabled(false);
        btnMoreDataSeg2->setEnabled(false);
        lblSeg3->setEnabled(false);
        cmbECISeg3->setEnabled(false);
        txtDataSeg3->setEnabled(false);
        btnMoreDataSeg3->setEnabled(false);
    } else {
        cmbECISeg1->setEnabled(true);
        lblSeg2->setEnabled(true);
        txtDataSeg2->setEnabled(true);
        btnMoreDataSeg2->setEnabled(true);
        if (txtDataSeg2->text().isEmpty()) {
            cmbECISeg2->setEnabled(false);
            lblSeg3->setEnabled(false);
            cmbECISeg3->setEnabled(false);
            txtDataSeg3->setEnabled(false);
            btnMoreDataSeg3->setEnabled(false);
        } else {
            cmbECISeg2->setEnabled(true);
            lblSeg3->setEnabled(true);
            txtDataSeg3->setEnabled(true);
            btnMoreDataSeg3->setEnabled(true);
            if (txtDataSeg3->text().isEmpty()) {
                cmbECISeg3->setEnabled(false);
            } else {
                cmbECISeg3->setEnabled(true);
            }
        }
    }
}

void MainWindow::composite_ui_set()
{
    bool enabled = !grpComposite->isHidden() && chkComposite->isChecked();

    lblCompType->setEnabled(enabled);
    cmbCompType->setEnabled(enabled);
    lblComposite->setEnabled(enabled);
    txtComposite->setEnabled(enabled);

    if (enabled) {
        if (bstyle_items[bstyle->currentIndex()].symbology == BARCODE_CODE128) {
            QRadioButton *radioButton = m_optionWidget->findChild<QRadioButton*>(QSL("radC128EAN"));
            if (radioButton) {
                radioButton->setChecked(true);
            }
        }
    }
}

void MainWindow::composite_ean_check()
{
    if (bstyle_items[bstyle->currentIndex()].symbology != BARCODE_CODE128)
        return;
    QRadioButton *radioButton = m_optionWidget->findChild<QRadioButton*>(QSL("radC128EAN"));
    if (radioButton && !radioButton->isChecked())
        chkComposite->setChecked(false);
}

void MainWindow::maxi_scm_ui_set()
{
    if (bstyle_items[bstyle->currentIndex()].symbology != BARCODE_MAXICODE)
        return;
    QCheckBox *chkMaxiSCMVV = m_optionWidget ? m_optionWidget->findChild<QCheckBox*>(QSL("chkMaxiSCMVV")) : nullptr;
    if (!chkMaxiSCMVV)
        return;

    bool isMode2or3 = get_cmb_index(QSL("cmbMaxiMode")) == 0;

    m_optionWidget->findChild<QLabel*>(QSL("lblMaxiSCMPostcode"))->setEnabled(isMode2or3);
    m_optionWidget->findChild<QLineEdit*>(QSL("txtMaxiSCMPostcode"))->setEnabled(isMode2or3);
    m_optionWidget->findChild<QLabel*>(QSL("lblMaxiSCMCountry"))->setEnabled(isMode2or3);
    m_optionWidget->findChild<QSpinBox*>(QSL("spnMaxiSCMCountry"))->setEnabled(isMode2or3);
    m_optionWidget->findChild<QLabel*>(QSL("lblMaxiSCMService"))->setEnabled(isMode2or3);
    m_optionWidget->findChild<QSpinBox*>(QSL("spnMaxiSCMService"))->setEnabled(isMode2or3);
    chkMaxiSCMVV->setEnabled(isMode2or3);
    m_optionWidget->findChild<QLabel*>(QSL("lblMaxiSCMVV"))->setEnabled(isMode2or3 && chkMaxiSCMVV->isChecked());
    m_optionWidget->findChild<QSpinBox*>(QSL("spnMaxiSCMVV"))->setEnabled(isMode2or3 && chkMaxiSCMVV->isChecked());
}

void MainWindow::msi_plessey_ui_set()
{
    if (bstyle_items[bstyle->currentIndex()].symbology != BARCODE_MSI_PLESSEY)
        return;
    QCheckBox *checkBox = m_optionWidget ? m_optionWidget->findChild<QCheckBox*>(QSL("chkMSICheckText")) : nullptr;
    if (checkBox) {
        checkBox->setEnabled(get_cmb_index(QSL("cmbMSICheck")) > 0);
    }
}

// Taken from https://stackoverflow.com/questions/38915001/disable-specific-items-in-qcombobox
void MainWindow::combobox_item_enabled(QComboBox *comboBox, int index, bool enabled)
{
    QStandardItemModel *model = qobject_cast<QStandardItemModel*>(comboBox->model());
    if (model) {
        QStandardItem *item = model->item(index);
        if (item) {
            item->setEnabled(enabled);
        }
    }
}

void MainWindow::upcean_addon_gap(const QString &comboBoxName, const QString &labelName, int base)
{
    QComboBox *comboBox = m_optionWidget->findChild<QComboBox*>(comboBoxName);
    QLabel *label = m_optionWidget->findChild<QLabel*>(labelName);

    const QRegularExpression addonRE(QSL("^[0-9X]+[+][0-9]+$"));
    bool enabled = txtData->text().contains(addonRE);
    if (comboBox) {
        comboBox->setEnabled(enabled);
    }
    if (label) {
        label->setEnabled(enabled);
    }
    if (enabled && comboBox) {
        int item_val = comboBox->currentIndex();
        if (item_val) {
            m_bc.bc.setOption2(item_val + base);
        }
    }
}

void MainWindow::upcean_guard_descent(const QString &spnBoxName, const QString &labelName)
{
    QDoubleSpinBox *spnBox = m_optionWidget->findChild<QDoubleSpinBox*>(spnBoxName);
    QLabel *label = m_optionWidget->findChild<QLabel*>(labelName);

    bool enabled = txtData->text().length() > 5;
    if (spnBox) {
        spnBox->setEnabled(enabled);
    }
    if (label) {
        label->setEnabled(enabled);
    }
    if (enabled && spnBox) {
        m_bc.bc.setGuardDescent(spnBox->value());
    }
}

void MainWindow::guard_reset(const QString &spnBoxName)
{
    QDoubleSpinBox *spnBox = m_optionWidget->findChild<QDoubleSpinBox*>(spnBoxName);
    if (spnBox && spnBox->value() != 5.0) {
        spnBox->setValue(5.0);
        update_preview();
    }
}

void MainWindow::set_gs1_mode(bool gs1_mode)
{
    if (gs1_mode) {
        m_bc.bc.setInputMode(GS1_MODE | (m_bc.bc.inputMode() & ~0x07)); // Keep upper bits
        chkData->setEnabled(false);
    } else {
        chkData->setEnabled(true);
    }
}

void MainWindow::set_smaller_font(const QString &labelName)
{
    QLabel *label = m_optionWidget ? m_optionWidget->findChild<QLabel*>(labelName) : nullptr;
    if (label) {
        const QFont &appFont = QApplication::font();
        qreal pointSize = appFont.pointSizeF();
        if (pointSize != -1.0) {
            QFont font = label->font();
            pointSize *= 0.9;
            font.setPointSizeF(pointSize);
            label->setFont(font);
        } else {
            int pixelSize = appFont.pixelSize();
            if (pixelSize > 1) {
                QFont font = label->font();
                font.setPixelSize(pixelSize - 1);
                label->setFont(font);
            }
        }
    }
}

void MainWindow::update_preview()
{
    int symbology = bstyle_items[bstyle->currentIndex()].symbology;
    int eci_not_set = true;
    int width = view->geometry().width();
    int height = view->geometry().height();
    int item_val;
    QCheckBox *checkBox;

    if (!grpComposite->isHidden() && chkComposite->isChecked()) {
        m_bc.bc.setPrimaryMessage(txtData->text());
        m_bc.bc.setText(txtComposite->toPlainText());
    } else {
        if (!grpSegs->isHidden() && !txtDataSeg1->text().isEmpty()) {
            std::vector<Zint::QZintSeg> segs;
            segs.push_back(Zint::QZintSeg(txtData->text(), cmbECI->currentIndex()));
            segs.push_back(Zint::QZintSeg(txtDataSeg1->text(), cmbECISeg1->currentIndex()));
            if (!txtDataSeg2->text().isEmpty()) {
                segs.push_back(Zint::QZintSeg(txtDataSeg2->text(), cmbECISeg2->currentIndex()));
                if (!txtDataSeg3->text().isEmpty()) {
                    segs.push_back(Zint::QZintSeg(txtDataSeg3->text(), cmbECISeg3->currentIndex()));
                }
            }
            m_bc.bc.setSegs(segs);
        } else {
            m_bc.bc.setText(txtData->text());
        }
    }
    m_bc.bc.setOption1(-1);
    m_bc.bc.setOption2(0);
    m_bc.bc.setOption3(0);
    chkData->setEnabled(true);
    if (chkData->isChecked()) {
        m_bc.bc.setInputMode(DATA_MODE);
    } else {
        m_bc.bc.setInputMode(UNICODE_MODE);
    }
    if (chkEscape->isChecked()) {
        m_bc.bc.setInputMode(m_bc.bc.inputMode() | ESCAPE_MODE);
    }
    m_bc.bc.setGSSep(false);
    m_bc.bc.setNoQuietZones(false);
    m_bc.bc.setDotSize(0.4f / 0.5f);
    m_bc.bc.setGuardDescent(5.0f);
    m_bc.bc.clearStructApp();

    switch (symbology) {

        case BARCODE_CODE128:
            if (get_rad_val(QSL("radC128CSup")))
                m_bc.bc.setSymbol(BARCODE_CODE128B);
            else if (get_rad_val(QSL("radC128EAN")))
                m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_GS1_128_CC : BARCODE_GS1_128);
            else if (get_rad_val(QSL("radC128HIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_128);
            else
                m_bc.bc.setSymbol(BARCODE_CODE128);
            break;

        case BARCODE_EANX:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_EANX_CC : BARCODE_EANX);
            upcean_addon_gap(QSL("cmbUPCEANAddonGap"), QSL("lblUPCEANAddonGap"), 7 /*base*/);
            upcean_guard_descent(QSL("spnUPCEANGuardDescent"), QSL("lblUPCEANGuardDescent"));
            if (get_chk_val(QSL("chkUPCEANNoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_ISBNX:
            m_bc.bc.setSymbol(symbology);
            upcean_addon_gap(QSL("cmbUPCEANAddonGap"), QSL("lblUPCEANAddonGap"), 7 /*base*/);
            upcean_guard_descent(QSL("spnUPCEANGuardDescent"), QSL("lblUPCEANGuardDescent"));
            if (get_chk_val(QSL("chkUPCEANNoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_UPCA:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_UPCA_CC : BARCODE_UPCA);
            upcean_addon_gap(QSL("cmbUPCAAddonGap"), QSL("lblUPCAAddonGap"), 9 /*base*/);
            upcean_guard_descent(QSL("spnUPCAGuardDescent"), QSL("lblUPCAGuardDescent"));
            if (get_chk_val(QSL("chkUPCANoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_UPCE:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_UPCE_CC : BARCODE_UPCE);
            upcean_addon_gap(QSL("cmbUPCEANAddonGap"), QSL("lblUPCEANAddonGap"), 7 /*base*/);
            upcean_guard_descent(QSL("spnUPCEANGuardDescent"), QSL("lblUPCEANGuardDescent"));
            if (get_chk_val(QSL("chkUPCEANNoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_DBAR_OMN:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_OMN_CC : BARCODE_DBAR_OMN);
            break;
        case BARCODE_DBAR_LTD:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_LTD_CC : BARCODE_DBAR_LTD);
            break;
        case BARCODE_DBAR_EXP:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_EXP_CC : BARCODE_DBAR_EXP);
            break;
        case BARCODE_DBAR_STK:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_STK_CC : BARCODE_DBAR_STK);
            break;
        case BARCODE_DBAR_OMNSTK:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_OMNSTK_CC : BARCODE_DBAR_OMNSTK);
            break;
        case BARCODE_DBAR_EXPSTK:
            m_bc.bc.setSymbol(chkComposite->isChecked() ? BARCODE_DBAR_EXPSTK_CC : BARCODE_DBAR_EXPSTK);
            if (get_rad_val(QSL("radDBESCols"))) {
                if ((item_val = get_cmb_index(QSL("cmbDBESCols"))) != 0) {
                    m_bc.bc.setOption2(item_val);
                }
            } else if (get_rad_val(QSL("radDBESRows"))) {
                if ((item_val = get_cmb_index(QSL("cmbDBESRows"))) != 0) {
                    m_bc.bc.setOption3(item_val + 1); // Begins at 2
                }
            }
            break;

        case BARCODE_PDF417:
            if (get_rad_val(QSL("radPDFTruncated")))
                m_bc.bc.setSymbol(BARCODE_PDF417COMP);
            else if (get_rad_val(QSL("radPDFHIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_PDF);
            else
                m_bc.bc.setSymbol(BARCODE_PDF417);

            m_bc.bc.setOption2(get_cmb_index(QSL("cmbPDFCols")));
            if ((item_val = get_cmb_index(QSL("cmbPDFRows"))) != 0) {
                m_bc.bc.setOption3(item_val + 2); // Starts at 3 rows
            }
            m_bc.bc.setOption1(get_cmb_index(QSL("cmbPDFECC")) - 1);

            if ((item_val = get_spn_val(QSL("spnPDFStructAppCount"))) > 1) {
                m_bc.bc.setStructApp(item_val, get_spn_val(QSL("spnPDFStructAppIndex")),
                            get_txt_val(QSL("txtPDFStructAppID")));
            }
            break;

        case BARCODE_MICROPDF417:
            if (get_rad_val(QSL("radMPDFHIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_MICPDF);
            else
                m_bc.bc.setSymbol(BARCODE_MICROPDF417);

            m_bc.bc.setOption2(get_cmb_index(QSL("cmbMPDFCols")));

            if ((item_val = get_spn_val(QSL("spnMPDFStructAppCount"))) > 1) {
                m_bc.bc.setStructApp(item_val, get_spn_val(QSL("spnMPDFStructAppIndex")),
                            get_txt_val(QSL("txtMPDFStructAppID")));
            }
            break;

        case BARCODE_DOTCODE:
            m_bc.bc.setSymbol(BARCODE_DOTCODE);
            if ((item_val = get_cmb_index(QSL("cmbDotCols"))) != 0) {
                m_bc.bc.setOption2(item_val + 4); // Cols 1-4 not listed
            }
            if ((item_val = get_cmb_index(QSL("cmbDotMask"))) != 0) {
                m_bc.bc.setOption3((item_val << 8) | m_bc.bc.option3());
            }
            set_gs1_mode(get_rad_val(QSL("radDotGS1")));

            if ((item_val = get_cmb_index(QSL("cmbDotStructAppCount"))) != 0) {
                QString id; // Dummy ID
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbDotStructAppIndex")) + 1, id);
            }
            break;

        case BARCODE_AZTEC:
            if (get_rad_val(QSL("radAztecHIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_AZTEC);
            else
                m_bc.bc.setSymbol(BARCODE_AZTEC);

            if (get_rad_val(QSL("radAztecSize")))
                m_bc.bc.setOption2(get_cmb_index(QSL("cmbAztecSize")) + 1);

            if (get_rad_val(QSL("radAztecECC")))
                m_bc.bc.setOption1(get_cmb_index(QSL("cmbAztecECC")) + 1);

            set_gs1_mode(get_rad_val(QSL("radAztecGS1")));

            if ((item_val = get_cmb_index(QSL("cmbAztecStructAppCount"))) != 0) {
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbAztecStructAppIndex")) + 1,
                            get_txt_val(QSL("txtAztecStructAppID")));
            }
            break;

        case BARCODE_MSI_PLESSEY:
            m_bc.bc.setSymbol(BARCODE_MSI_PLESSEY);
            item_val = get_cmb_index(QSL("cmbMSICheck"));
            if (item_val && get_chk_val(QSL("chkMSICheckText"))) {
                item_val += 10;
            }
            m_bc.bc.setOption2(item_val);
            break;

        case BARCODE_CODE11:
            m_bc.bc.setSymbol(BARCODE_CODE11);
            if (get_rad_val(QSL("radC11OneCheckDigit"))) {
                m_bc.bc.setOption2(1);
            } else if (get_rad_val(QSL("radC11NoCheckDigits"))) {
                m_bc.bc.setOption2(2);
            }
            break;

        case BARCODE_C25STANDARD:
        case BARCODE_C25INTER:
        case BARCODE_C25IATA:
        case BARCODE_C25LOGIC:
        case BARCODE_C25IND:
            m_bc.bc.setSymbol(symbology);
            m_bc.bc.setOption2(get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;

        case BARCODE_CODE39:
            if (get_rad_val(QSL("radC39HIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_39);
            else {
                m_bc.bc.setSymbol(BARCODE_CODE39);
                if (get_rad_val(QSL("radC39Check")))
                    m_bc.bc.setOption2(1);
            }
            break;
        case BARCODE_EXCODE39:
            m_bc.bc.setSymbol(BARCODE_EXCODE39);
            if (get_rad_val(QSL("radC39Check")))
                m_bc.bc.setOption2(1);

            break;
        case BARCODE_LOGMARS:
            m_bc.bc.setSymbol(BARCODE_LOGMARS);
            if (get_rad_val(QSL("radLOGMARSCheck"))) {
                m_bc.bc.setOption2(1);
            }
            break;

        case BARCODE_CODE16K:
            m_bc.bc.setSymbol(BARCODE_CODE16K);
            set_gs1_mode(get_rad_val(QSL("radC16kGS1")));
            if ((item_val = get_cmb_index(QSL("cmbC16kRows"))) != 0) {
                m_bc.bc.setOption1(item_val + 2); // Starts at 3
            }
            // Row separator height selection uses option 3 in zint_symbol
            if ((item_val = get_cmb_index(QSL("cmbC16kRowSepHeight"))) != 0) {
                m_bc.bc.setOption3(item_val + 1); // Zero-based
            }
            if (get_chk_val(QSL("chkC16kNoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_CODABAR:
            m_bc.bc.setSymbol(BARCODE_CODABAR);
            m_bc.bc.setOption2(get_rad_grp_index(
                QStringList() << QSL("radCodabarStand") << QSL("radCodabarCheckHide") << QSL("radCodabarCheck")));
            break;

        case BARCODE_CODABLOCKF:
            if (get_rad_val(QSL("radCbfHIBC"))) {
                m_bc.bc.setSymbol(BARCODE_HIBC_BLOCKF);
            } else {
                m_bc.bc.setSymbol(BARCODE_CODABLOCKF);
            }
            if ((item_val = get_cmb_index(QSL("cmbCbfWidth"))) != 0) {
                m_bc.bc.setOption2(item_val - 1 + 9);
            }
            // Height selection uses option 1 in zint_symbol
            if ((item_val = get_cmb_index(QSL("cmbCbfHeight"))) != 0) {
                m_bc.bc.setOption1(item_val);
            }
            // Row separator height selection uses option 3 in zint_symbol
            if ((item_val = get_cmb_index(QSL("cmbCbfRowSepHeight"))) != 0) {
                m_bc.bc.setOption3(item_val + 1); // Zero-based
            }
            if (get_chk_val(QSL("chkCbfNoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_DAFT:
            m_bc.bc.setSymbol(BARCODE_DAFT);
            // Kept as percentage, convert to thousandths
            m_bc.bc.setOption2((int) (get_dspn_val(QSL("spnDAFTTrackerRatio")) * 10));
            break;

        case BARCODE_DATAMATRIX:
            if (get_rad_val(QSL("radDM200HIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_DM);
            else
                m_bc.bc.setSymbol(BARCODE_DATAMATRIX);

            checkBox = m_optionWidget->findChild<QCheckBox*>(QSL("chkDMGSSep"));
            if (get_rad_val(QSL("radDM200GS1"))) {
                set_gs1_mode(true);
                checkBox->setEnabled(true);
                if (checkBox->isChecked()) {
                    m_bc.bc.setGSSep(true);
                }
            } else {
                set_gs1_mode(false);
                checkBox->setEnabled(false);
            }

            m_bc.bc.setOption2(get_cmb_index(QSL("cmbDM200Size")));

            if (get_cmb_index(QSL("cmbDM200Size")) == 0) {
                // Suppressing rectangles or allowing DMRE only makes sense if in automatic size mode
                m_optionWidget->findChild<QLabel*>(QSL("lblDMAutoSize"))->setEnabled(true);
                m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRectangle"))->setEnabled(true);
                if (m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRectangle"))->isChecked()) {
                    m_bc.bc.setOption3(DM_SQUARE);
                    m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRE"))->setEnabled(false);
                } else {
                    m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRE"))->setEnabled(true);
                    if (m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRE"))->isChecked())
                        m_bc.bc.setOption3(DM_DMRE);
                    else
                        m_bc.bc.setOption3(0);
                }
            } else {
                m_optionWidget->findChild<QLabel*>(QSL("lblDMAutoSize"))->setEnabled(false);
                m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRectangle"))->setEnabled(false);
                m_optionWidget->findChild<QCheckBox*>(QSL("chkDMRE"))->setEnabled(false);
                m_bc.bc.setOption3(0);
            }

            if (get_chk_val(QSL("chkDMFast"))) {
                m_bc.bc.setInputMode(FAST_MODE | m_bc.bc.inputMode());
            }

            if ((item_val = get_cmb_index(QSL("cmbDMStructAppCount"))) != 0) {
                QString id;
                int id1 = get_spn_val(QSL("spnDMStructAppID"));
                int id2 = get_spn_val(QSL("spnDMStructAppID2"));
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbDMStructAppIndex")) + 1,
                            id.setNum(id1 * 1000 + id2));
            }
            break;

        case BARCODE_ITF14:
            m_bc.bc.setSymbol(BARCODE_ITF14);
            if (get_chk_val(QSL("chkITF14NoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_QRCODE:
            if (get_rad_val(QSL("radQRHIBC")))
                m_bc.bc.setSymbol(BARCODE_HIBC_QR);
            else
                m_bc.bc.setSymbol(BARCODE_QRCODE);

            set_gs1_mode(get_rad_val(QSL("radQRGS1")));

            if ((item_val = get_cmb_index(QSL("cmbQRSize"))) != 0) {
                m_bc.bc.setOption2(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbQRECC"))) != 0) {
                m_bc.bc.setOption1(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbQRMask"))) != 0) {
                m_bc.bc.setOption3((item_val << 8) | m_bc.bc.option3());
            }
            if (get_chk_val(QSL("chkQRFullMultibyte"))) {
                m_bc.bc.setOption3(ZINT_FULL_MULTIBYTE | m_bc.bc.option3());
            }
            if ((item_val = get_cmb_index(QSL("cmbQRStructAppCount"))) != 0) {
                QString id;
                int id_val = get_spn_val(QSL("spnQRStructAppID"));
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbQRStructAppIndex")) + 1, id.setNum(id_val));
            }
            break;

        case BARCODE_MICROQR:
            m_bc.bc.setSymbol(BARCODE_MICROQR);
            if ((item_val = get_cmb_index(QSL("cmbMQRSize"))) != 0) {
                m_bc.bc.setOption2(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbMQRECC"))) != 0) {
                m_bc.bc.setOption1(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbMQRMask"))) != 0) {
                m_bc.bc.setOption3((item_val << 8) | m_bc.bc.option3());
            }
            if (get_chk_val(QSL("chkMQRFullMultibyte"))) {
                m_bc.bc.setOption3(ZINT_FULL_MULTIBYTE | m_bc.bc.option3());
            }
            break;

        case BARCODE_UPNQR:
            m_bc.bc.setSymbol(BARCODE_UPNQR);
            cmbECI->setCurrentIndex(2 /*ECI 4*/);
            if ((item_val = get_cmb_index(QSL("cmbUPNQRMask"))) != 0) {
                m_bc.bc.setOption3((item_val << 8) | m_bc.bc.option3());
            }
            break;

        case BARCODE_RMQR:
            m_bc.bc.setSymbol(BARCODE_RMQR);

            set_gs1_mode(get_rad_val(QSL("radRMQRGS1")));

            if ((item_val = get_cmb_index(QSL("cmbRMQRSize"))) != 0) {
                m_bc.bc.setOption2(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbRMQRECC"))) != 0) {
                m_bc.bc.setOption1(item_val * 2); // Levels 2 (M) and 4 (H) only
            }
            if (get_chk_val(QSL("chkRMQRFullMultibyte"))) {
                m_bc.bc.setOption3(ZINT_FULL_MULTIBYTE);
            }
            break;

        case BARCODE_GRIDMATRIX:
            m_bc.bc.setSymbol(BARCODE_GRIDMATRIX);
            if ((item_val = get_cmb_index(QSL("cmbGridSize"))) != 0) {
                m_bc.bc.setOption2(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbGridECC"))) != 0) {
                m_bc.bc.setOption1(item_val);
            }
            if (get_chk_val(QSL("chkGridFullMultibyte"))) {
                m_bc.bc.setOption3(ZINT_FULL_MULTIBYTE);
            }
            if ((item_val = get_cmb_index(QSL("cmbGridStructAppCount"))) != 0) {
                QString id;
                int id_val = get_spn_val(QSL("spnGridStructAppID"));
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbGridStructAppIndex")) + 1,
                            id.setNum(id_val));
            }
            break;

        case BARCODE_MAXICODE:
            m_bc.bc.setSymbol(BARCODE_MAXICODE);
            if (get_cmb_index(QSL("cmbMaxiMode")) == 0) {
                m_bc.bc.setOption1(0); // Auto-determine mode 2 or 3 from primary message (checks that it isn't empty)
                m_bc.bc.setPrimaryMessage(QString::asprintf("%s%03d%03d",
                                    get_txt_val(QSL("txtMaxiSCMPostcode")).toUtf8().constData(),
                                    get_spn_val(QSL("spnMaxiSCMCountry")), get_spn_val(QSL("spnMaxiSCMService"))));
                QCheckBox *chkMaxiSCMVV = m_optionWidget->findChild<QCheckBox*>(QSL("chkMaxiSCMVV"));
                if (chkMaxiSCMVV && chkMaxiSCMVV->isEnabled() && chkMaxiSCMVV->isChecked()) {
                    m_bc.bc.setOption2(get_spn_val(QSL("spnMaxiSCMVV")) + 1);
                }
            } else {
                m_bc.bc.setOption1(get_cmb_index(QSL("cmbMaxiMode")) + 3);
            }

            if ((item_val = get_cmb_index(QSL("cmbMaxiStructAppCount"))) != 0) {
                QString id; // Dummy ID
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbMaxiStructAppIndex")) + 1, id);
            }
            break;

        case BARCODE_CHANNEL:
            m_bc.bc.setSymbol(BARCODE_CHANNEL);
            if ((item_val = get_cmb_index(QSL("cmbChannel"))) == 0)
                m_bc.bc.setOption2(0);
            else
                m_bc.bc.setOption2(item_val + 2);
            break;

        case BARCODE_CODEONE:
            m_bc.bc.setSymbol(BARCODE_CODEONE);
            m_bc.bc.setOption2(get_cmb_index(QSL("cmbC1Size")));
            if (m_bc.bc.option2() == 9) { // Version S
                eci_not_set = false;
                cmbECI->setEnabled(false);
                lblECI->setEnabled(false);
                m_optionWidget->findChild<QRadioButton*>(QSL("radC1GS1"))->setEnabled(false);
            } else {
                m_optionWidget->findChild<QRadioButton*>(QSL("radC1GS1"))->setEnabled(true);
                set_gs1_mode(get_rad_val(QSL("radC1GS1")));
            }
            if (get_cmb_index(QSL("cmbC1Size")) != 9 && (item_val = get_spn_val(QSL("spnC1StructAppCount"))) > 1) {
                QString id; // Dummy ID
                m_bc.bc.setStructApp(item_val, get_spn_val(QSL("spnC1StructAppIndex")), id);
            }
            break;

        case BARCODE_CODE49:
            m_bc.bc.setSymbol(BARCODE_CODE49);
            set_gs1_mode(get_rad_val(QSL("radC49GS1")));
            if ((item_val = get_cmb_index(QSL("cmbC49Rows"))) != 0) {
                m_bc.bc.setOption1(item_val + 2); // Starts at 3
            }
            // Row separator height selection uses option 3 in zint_symbol
            if ((item_val = get_cmb_index(QSL("cmbC49RowSepHeight"))) != 0) {
                m_bc.bc.setOption3(item_val + 1); // Zero-based
            }
            if (get_chk_val(QSL("chkC49NoQuietZones"))) {
                m_bc.bc.setNoQuietZones(true);
            }
            break;

        case BARCODE_CODE93:
            m_bc.bc.setSymbol(BARCODE_CODE93);
            if (get_chk_val(QSL("chkC93ShowChecks"))) {
                m_bc.bc.setOption2(1);
            }
            break;

        case BARCODE_HANXIN:
            m_bc.bc.setSymbol(BARCODE_HANXIN);
            if ((item_val = get_cmb_index(QSL("cmbHXSize"))) != 0) {
                m_bc.bc.setOption2(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbHXECC"))) != 0) {
                m_bc.bc.setOption1(item_val);
            }
            if ((item_val = get_cmb_index(QSL("cmbHXMask"))) != 0) {
                m_bc.bc.setOption3((item_val << 8) | m_bc.bc.option3());
            }
            if (get_chk_val(QSL("chkHXFullMultibyte"))) {
                m_bc.bc.setOption3(ZINT_FULL_MULTIBYTE | m_bc.bc.option3());
            }
            break;

        case BARCODE_ULTRA:
            m_bc.bc.setSymbol(BARCODE_ULTRA);
            if (get_rad_val(QSL("radUltraEcc")))
                m_bc.bc.setOption1(get_cmb_index(QSL("cmbUltraEcc")) + 1);

            set_gs1_mode(get_rad_val(QSL("radUltraGS1")));

            if ((item_val = get_cmb_index(QSL("cmbUltraRevision"))) > 0) {
                m_bc.bc.setOption2(item_val + 1); // Combobox 0-based
            }
            if ((item_val = get_cmb_index(QSL("cmbUltraStructAppCount"))) != 0) {
                QString id;
                int id_val = get_spn_val(QSL("spnUltraStructAppID"));
                m_bc.bc.setStructApp(item_val + 1, get_cmb_index(QSL("cmbUltraStructAppIndex")) + 1,
                            id.setNum(id_val));
            }
            break;

        case BARCODE_VIN:
            m_bc.bc.setSymbol(BARCODE_VIN);
            if (get_chk_val(QSL("chkVINImportChar"))) {
                m_bc.bc.setOption2(m_bc.bc.option2() | 1); // Import character 'I' prefix
            }
            break;

        default:
            m_bc.bc.setSymbol(symbology);
            break;
    }
    m_symbology = m_bc.bc.symbol();

    if (eci_not_set) {
        cmbECI->setEnabled(m_bc.bc.supportsECI());
        lblECI->setEnabled(cmbECI->isEnabled());
    }
    chkGS1Parens->setEnabled(m_bc.bc.supportsGS1());
    chkGS1NoCheck->setEnabled(m_bc.bc.supportsGS1());
    chkRInit->setEnabled(m_bc.bc.supportsReaderInit() && (m_bc.bc.inputMode() & 0x07) != GS1_MODE);
    chkCompliantHeight->setEnabled(m_bc.bc.hasCompliantHeight());

    if (!grpComposite->isHidden() && chkComposite->isChecked())
        m_bc.bc.setOption1(cmbCompType->currentIndex());

    if (!chkAutoHeight->isEnabled() || chkAutoHeight->isChecked()) {
        m_bc.bc.setHeight(0);
    } else {
        if (m_spnHeightPerRow && m_spnHeightPerRow->isEnabled() && m_spnHeightPerRow->value()) {
            // This causes a double-encode unfortunately, as heightb gets synced
            m_bc.bc.setInputMode(m_bc.bc.inputMode() | HEIGHTPERROW_MODE);
            m_bc.bc.setHeight(m_spnHeightPerRow->value());
        } else {
            m_bc.bc.setHeight(heightb->value());
        }
    }
    m_bc.bc.setCompliantHeight(chkCompliantHeight->isEnabled() && chkCompliantHeight->isChecked());
    m_bc.bc.setECI(cmbECI->isEnabled() ? cmbECI->currentIndex() : 0);
    m_bc.bc.setGS1Parens(chkGS1Parens->isEnabled() && chkGS1Parens->isChecked());
    m_bc.bc.setGS1NoCheck(chkGS1NoCheck->isEnabled() && chkGS1NoCheck->isChecked());
    m_bc.bc.setReaderInit(chkRInit->isEnabled() && chkRInit->isChecked());
    m_bc.bc.setShowText(chkHRTShow->isEnabled() && chkHRTShow->isChecked());
    m_bc.bc.setBorderType(btype->currentIndex());
    m_bc.bc.setBorderWidth(bwidth->value());
    m_bc.bc.setWhitespace(spnWhitespace->value());
    m_bc.bc.setVWhitespace(spnVWhitespace->value());
    m_bc.bc.setQuietZones(chkQuietZones->isEnabled() && chkQuietZones->isChecked());
    m_bc.bc.setFontSetting(cmbFontSetting->currentIndex());
    m_bc.bc.setRotateAngle(cmbRotate->currentIndex());
    m_bc.bc.setDotty(chkDotty->isEnabled() && chkDotty->isChecked());
    if (m_symbology == BARCODE_DOTCODE || (chkDotty->isEnabled() && chkDotty->isChecked())) {
        m_bc.bc.setDotSize(spnDotSize->value());
    }
    m_bc.bc.setFgColor(m_fgcolor);
    m_bc.bc.setBgColor(m_bgcolor);
    change_print_scale();
    change_cmyk();
    m_bc.setSize(width - 10, height - 10);
    m_bc.update();
    scene->setSceneRect(m_bc.boundingRect());
    scene->update();
}

void MainWindow::createActions()
{
    // SVG icons from https://github.com/feathericons/feather
    // MIT license - see site and "frontend_qt/res/LICENSE_feathericons"
    QIcon menuIcon(QSL(":res/menu.svg"));
    QIcon copyIcon(QIcon::fromTheme(QSL("edit-copy"), QIcon(QSL(":res/copy.svg"))));
    QIcon cliIcon(QSL(":res/zint_black.ico"));
    QIcon saveIcon(QIcon::fromTheme(QSL("document-save"), QIcon(QSL(":res/download.svg"))));
    QIcon aboutIcon(QSL(":res/zint-qt.ico"));
    QIcon helpIcon(QIcon::fromTheme(QSL("help-contents"), QIcon(QSL(":res/help-circle.svg"))));
    QIcon quitIcon(QIcon::fromTheme(QSL("window-close"), QIcon(QSL(":res/x.svg"))));

    btnMenu->setIcon(menuIcon);
    btnCopyBMP->setIcon(copyIcon);
    btnCopySVG->setIcon(copyIcon);
    //btnSave->setIcon(saveIcon); // Makes it too big
    btnExit->setIcon(quitIcon);

    m_copyBMPAct = new QAction(copyIcon, tr("Copy as &BMP"), this);
    m_copyBMPAct->setStatusTip(tr("Copy to clipboard as BMP"));
    m_copyBMPAct->setShortcut(copyBMPSeq);
    connect(m_copyBMPAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_bmp()));

    m_copyEMFAct = new QAction(copyIcon, tr("Copy as E&MF"), this);
    m_copyEMFAct->setStatusTip(tr("Copy to clipboard as EMF"));
    m_copyEMFAct->setShortcut(copyEMFSeq);
    connect(m_copyEMFAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_emf()));

#ifdef MAINWINDOW_COPY_EPS /* TODO: see if can get this to work */
    m_copyEPSAct = new QAction(copyIcon, tr("Copy as &EPS"), this);
    m_copyEPSAct->setStatusTip(tr("Copy to clipboard as EPS"));
    connect(m_copyEPSAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_eps()));
#endif

    m_copyGIFAct = new QAction(copyIcon, tr("Copy as &GIF"), this);
    m_copyGIFAct->setStatusTip(tr("Copy to clipboard as GIF"));
    m_copyGIFAct->setShortcut(copyGIFSeq);
    connect(m_copyGIFAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_gif()));

#ifdef MAINWINDOW_COPY_PCX /* TODO: see if can get this to work */
    m_copyPCXAct = new QAction(copyIcon, tr("Copy as P&CX"), this);
    m_copyPCXAct->setStatusTip(tr("Copy to clipboard as PCX"));
    connect(m_copyPCXAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_pcx()));
#endif

#ifndef NO_PNG
    m_copyPNGAct = new QAction(copyIcon, tr("Copy as &PNG"), this);
    m_copyPNGAct->setStatusTip(tr("Copy to clipboard as PNG"));
    m_copyPNGAct->setShortcut(copyPNGSeq);
    connect(m_copyPNGAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_png()));
#endif

    m_copySVGAct = new QAction(copyIcon, tr("Copy as S&VG"), this);
    m_copySVGAct->setStatusTip(tr("Copy to clipboard as SVG"));
    m_copySVGAct->setShortcut(copySVGSeq);
    connect(m_copySVGAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_svg()));

    m_copyTIFAct = new QAction(copyIcon, tr("Copy as &TIF"), this);
    m_copyTIFAct->setStatusTip(tr("Copy to clipboard as TIF"));
    m_copyTIFAct->setShortcut(copyTIFSeq);
    connect(m_copyTIFAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_tif()));

    m_openCLIAct = new QAction(cliIcon, tr("C&LI Equivalent..."), this);
    m_openCLIAct->setStatusTip(tr("Generate CLI equivalent"));
    m_openCLIAct->setShortcut(openCLISeq);
    connect(m_openCLIAct, SIGNAL(triggered()), this, SLOT(open_cli_dialog()));

    m_saveAsAct = new QAction(saveIcon, tr("&Save As..."), this);
    m_saveAsAct->setStatusTip(tr("Output image to file"));
    m_saveAsAct->setShortcut(QKeySequence::Save);
    connect(m_saveAsAct, SIGNAL(triggered()), this, SLOT(save()));

    m_helpAct = new QAction(helpIcon, tr("&Help (online)"), this);
    m_helpAct->setStatusTip(tr("Online manual"));
    m_helpAct->setShortcut(QKeySequence::HelpContents);
    connect(m_helpAct, SIGNAL(triggered()), this, SLOT(help()));

    m_aboutAct = new QAction(aboutIcon, tr("&About"), this);
    m_aboutAct->setStatusTip(tr("About Zint Barcode Studio"));
    connect(m_aboutAct, SIGNAL(triggered()), this, SLOT(about()));

    m_quitAct = new QAction(quitIcon, tr("&Quit"), this);
    m_quitAct->setStatusTip(tr("Exit Zint Barcode Studio"));
    m_quitAct->setShortcut(quitKeySeq);
    connect(m_quitAct, SIGNAL(triggered()), this, SLOT(quit_now()));

    m_copyErrtxtAct = new QAction(copyIcon, tr("&Copy"), this);
    m_copyErrtxtAct->setStatusTip(tr("Copy message to clipboard"));
    connect(m_copyErrtxtAct, SIGNAL(triggered()), this, SLOT(copy_to_clipboard_errtxt()));
}

void MainWindow::createMenu()
{
    m_menu = new QMenu(tr("Menu"), this);

    m_menu->addAction(m_copyBMPAct);
    m_menu->addAction(m_copyEMFAct);
#ifdef MAINWINDOW_COPY_EPS
    m_menu->addAction(m_copyEPSAct);
#endif
    m_menu->addAction(m_copyGIFAct);
#ifdef MAINWINDOW_COPY_PCX
    m_menu->addAction(m_copyPCXAct);
#endif
#ifndef NO_PNG
    m_menu->addAction(m_copyPNGAct);
#endif
    m_menu->addAction(m_copySVGAct);
    m_menu->addAction(m_copyTIFAct);
    m_menu->addSeparator();

    m_menu->addAction(m_openCLIAct);
    m_menu->addSeparator();
    m_menu->addAction(m_saveAsAct);
    m_menu->addSeparator();
    m_menu->addAction(m_helpAct);
    m_menu->addAction(m_aboutAct);
    m_menu->addSeparator();
    m_menu->addAction(m_quitAct);
}

void MainWindow::enableActions(bool enabled)
{
    btnCopyBMP->setEnabled(enabled);
    btnCopySVG->setEnabled(enabled);
    btnSave->setEnabled(enabled);

    m_copyBMPAct->setEnabled(enabled);
    m_copyEMFAct->setEnabled(enabled);
#ifdef MAINWINDOW_COPY_EPS
    m_copyEPSAct->setEnabled(enabled);
#endif
    m_copyGIFAct->setEnabled(enabled);
#ifdef MAINWINDOW_COPY_PCX
    m_copyPCXAct->setEnabled(enabled);
#endif
#ifndef NO_PNG
    m_copyPNGAct->setEnabled(enabled);
#endif
    m_copySVGAct->setEnabled(enabled);
    m_copyTIFAct->setEnabled(enabled);
    m_openCLIAct->setEnabled(enabled);
    m_saveAsAct->setEnabled(enabled);

    m_saveAsShortcut->setEnabled(enabled);
    m_openCLIShortcut->setEnabled(enabled);
    m_copyBMPShortcut->setEnabled(enabled);
    m_copyEMFShortcut->setEnabled(enabled);
    m_copyGIFShortcut->setEnabled(enabled);
#ifndef NO_PNG
    m_copyPNGShortcut->setEnabled(enabled);
#endif
    m_copySVGShortcut->setEnabled(enabled);
    m_copyTIFShortcut->setEnabled(enabled);
}

void MainWindow::copy_to_clipboard(const QString &filename, const QString& name, const char *mimeType)
{
    QClipboard *clipboard = QGuiApplication::clipboard();

    if (!m_bc.bc.save_to_file(filename)) {
        return;
    }

    QMimeData *mdata = new QMimeData;
    if (mimeType) {
        QFile file(filename);
        if (!file.open(QIODevice::ReadOnly)) {
            delete mdata;
        } else {
            mdata->setData(mimeType, file.readAll());
            file.close();
            clipboard->setMimeData(mdata, QClipboard::Clipboard);
            /*: %1 is format (BMP/EMF etc) */
            statusBar->showMessage(tr("Copied to clipboard as %1").arg(name), 0 /*No timeout*/);
        }
    } else {
        mdata->setImageData(QImage(filename));
        clipboard->setMimeData(mdata, QClipboard::Clipboard);
        /*: %1 is format (BMP/EMF etc) */
        statusBar->showMessage(tr("Copied to clipboard as %1").arg(name), 0 /*No timeout*/);
    }

    QFile::remove(filename);
}

void MainWindow::errtxtBar_clear()
{
    errtxtBar->clearMessage();
    if (!errtxtBarContainer->isHidden()) {
        errtxtBarContainer->hide();
        errtxtBarContainer->update();
        update_preview();
    }
}

void MainWindow::errtxtBar_set(bool isError)
{
    if (!m_bc.bc.hasErrors()) {
        errtxtBar_clear();
    } else {
        errtxtBar->showMessage(m_bc.bc.lastError());
        errtxtBar->setStyleSheet(isError
                    ? QSL("QStatusBar {background:white; color:#dd0000;}")
                    : QSL("QStatusBar {background:white; color:#ff6f00;}"));
        if (errtxtBarContainer->isHidden()) {
            errtxtBarContainer->show();
            errtxtBarContainer->update();
            update_preview();
        }
    }
}

QPoint MainWindow::get_context_menu_pos(const QPoint &pos, QWidget *widget)
{
    QPoint menuPos(pos);
    if (menuPos.x() == 0 && menuPos.y() == 0) { // May have been invoked by menu key
        QPoint mousePos(widget->mapFromGlobal(QCursor::pos()));
        if (widget->rect().contains(mousePos)) {
            menuPos = mousePos;
        }
    }

    return widget->mapToGlobal(menuPos);
}

/* Shorthand to find widget child as generic QWidget */
QWidget *MainWindow::get_widget(const QString &name)
{
    return m_optionWidget ? m_optionWidget->findChild<QWidget*>(name) : nullptr;
}

/* Return settings subsection name for a symbol */
QString MainWindow::get_setting_name(int symbology)
{
    char name_buf[32];
    switch (symbology) {
        case BARCODE_CODE128B:
        case BARCODE_GS1_128:
        case BARCODE_GS1_128_CC:
        case BARCODE_HIBC_128:
            symbology = BARCODE_CODE128;
            break;
        case BARCODE_PDF417COMP:
        case BARCODE_HIBC_PDF:
            symbology = BARCODE_PDF417;
            break;
        case BARCODE_HIBC_MICPDF:
            symbology = BARCODE_MICROPDF417;
            break;
        case BARCODE_HIBC_AZTEC:
            symbology = BARCODE_AZTEC;
            break;
        case BARCODE_HIBC_39:
            symbology = BARCODE_CODE39;
            break;
        case BARCODE_HIBC_BLOCKF:
            symbology = BARCODE_CODABLOCKF;
            break;
        case BARCODE_HIBC_DM:
            symbology = BARCODE_DATAMATRIX;
            break;
        case BARCODE_HIBC_QR:
            symbology = BARCODE_QRCODE;
            break;
        case BARCODE_DBAR_EXPSTK_CC:
            symbology = BARCODE_DBAR_EXPSTK;
            break;
        case BARCODE_UPCA_CHK:
        case BARCODE_UPCA_CC:
            symbology = BARCODE_UPCA;
            break;
        case BARCODE_EANX_CHK:
        case BARCODE_EANX_CC:
            symbology = BARCODE_EANX;
            break;
        case BARCODE_UPCE_CHK:
        case BARCODE_UPCE_CC:
            symbology = BARCODE_UPCE;
            break;
    }
    if (ZBarcode_BarcodeName(symbology, name_buf) != 0) {
        return QSL("");
    }
    QString name(name_buf + 8); // Strip "BARCODE_" prefix
    return name.toLower();
}

/* Helper to return index of selected radio button in group, checking for NULL */
int MainWindow::get_rad_grp_index(const QStringList &names)
{
    if (m_optionWidget) {
        QRadioButton *radioButton;
        for (int index = 0; index < names.size(); index++) {
            radioButton = m_optionWidget->findChild<QRadioButton*>(names[index]);
            if (radioButton && radioButton->isChecked()) {
                return index;
            }
        }
    }
    return 0;
}

/* Helper to set radio button in group from index in settings, checking for NULL */
void MainWindow::set_rad_from_setting(QSettings &settings, const QString &setting,
            const QStringList &names, int default_val)
{
    if (m_optionWidget) {
        int index = settings.value(setting, default_val).toInt();
        QRadioButton *radioButton;
        if (index >= 0 && index < names.size()) {
            radioButton = m_optionWidget->findChild<QRadioButton*>(names[index]);
        } else {
            radioButton = m_optionWidget->findChild<QRadioButton*>(names[0]);
        }
        if (radioButton) {
            radioButton->setChecked(true);
        }
    }
}

/* Helper to see if radio button checked, checking for NULL and whether enabled */
bool MainWindow::get_rad_val(const QString &name)
{
    QRadioButton *radioButton = m_optionWidget ? m_optionWidget->findChild<QRadioButton*>(name) : nullptr;
    return radioButton && radioButton->isEnabled() && radioButton->isChecked();
}

/* Helper to return index of selected item in combobox, checking for NULL */
int MainWindow::get_cmb_index(const QString &name)
{
    QComboBox *comboBox = m_optionWidget ? m_optionWidget->findChild<QComboBox*>(name) : nullptr;
    return comboBox ? comboBox->currentIndex() : 0;
}

/* Helper to set item in combobox from index in settings, checking for NULL */
void MainWindow::set_cmb_from_setting(QSettings &settings, const QString &setting, const QString &name,
            int default_val)
{
    QComboBox *comboBox = m_optionWidget ? m_optionWidget->findChild<QComboBox*>(name) : nullptr;
    if (comboBox) {
        comboBox->setCurrentIndex(settings.value(setting, default_val).toInt());
    }
}

/* Helper to return if checkbox checked, checking for NULL */
int MainWindow::get_chk_val(const QString &name)
{
    QCheckBox *checkBox = m_optionWidget ? m_optionWidget->findChild<QCheckBox*>(name) : nullptr;
    return checkBox && checkBox->isChecked() ? 1 : 0;
}

/* Helper to set checkbox from settings, checking for NULL */
void MainWindow::set_chk_from_setting(QSettings &settings, const QString &setting, const QString &name,
            int default_val)
{
    QCheckBox *checkBox = m_optionWidget ? m_optionWidget->findChild<QCheckBox*>(name) : nullptr;
    if (checkBox) {
        checkBox->setChecked(settings.value(setting, default_val).toInt() ? true : false);
    }
}

/* Helper to return value of double spinner, checking for NULL */
double MainWindow::get_dspn_val(const QString &name)
{
    QDoubleSpinBox *spinBox = m_optionWidget->findChild<QDoubleSpinBox*>(name);
    return spinBox ? spinBox->value() : 0.0;
}

/* Helper to set double spinner from settings, checking for NULL */
void MainWindow::set_dspn_from_setting(QSettings &settings, const QString &setting, const QString &name,
            float default_val)
{
    QDoubleSpinBox *spinBox = m_optionWidget->findChild<QDoubleSpinBox*>(name);
    if (spinBox) {
        spinBox->setValue(settings.value(setting, default_val).toFloat());
    }
}

/* Helper to return text of line edit, checking for NULL */
QString MainWindow::get_txt_val(const QString &name)
{
    QLineEdit *lineEdit = m_optionWidget ? m_optionWidget->findChild<QLineEdit*>(name) : nullptr;
    return lineEdit ? lineEdit->text() : QSL("");
}

/* Helper to set line edit from settings, checking for NULL */
void MainWindow::set_txt_from_setting(QSettings &settings, const QString &setting, const QString &name,
            const QString &default_val)
{
    QLineEdit *lineEdit = m_optionWidget ? m_optionWidget->findChild<QLineEdit*>(name) : nullptr;
    if (lineEdit) {
        lineEdit->setText(settings.value(setting, default_val).toString());
    }
}

/* Helper to return value of spin box, checking for NULL */
int MainWindow::get_spn_val(const QString &name)
{
    QSpinBox *spinBox = m_optionWidget ? m_optionWidget->findChild<QSpinBox*>(name) : nullptr;
    return spinBox ? spinBox->value() : 0;
}

/* Helper to set spin box from settings, checking for NULL */
void MainWindow::set_spn_from_setting(QSettings &settings, const QString &setting, const QString &name,
            int default_val)
{
    QSpinBox *spinBox = m_optionWidget ? m_optionWidget->findChild<QSpinBox*>(name) : nullptr;
    if (spinBox) {
        spinBox->setValue(settings.value(setting, default_val).toInt());
    }
}

/* Save settings for an individual symbol */
void MainWindow::save_sub_settings(QSettings &settings, int symbology)
{
    QString name = get_setting_name(symbology);
    if (!name.isEmpty()) {
        settings.setValue(QSL("studio/bc/%1/data").arg(name), txtData->text());
        if (!grpSegs->isHidden()) {
            settings.setValue(QSL("studio/bc/%1/data_seg1").arg(name), txtDataSeg1->text());
            settings.setValue(QSL("studio/bc/%1/data_seg2").arg(name), txtDataSeg2->text());
            settings.setValue(QSL("studio/bc/%1/data_seg3").arg(name), txtDataSeg3->text());
        }
        if (!grpComposite->isHidden()) {
            settings.setValue(QSL("studio/bc/%1/composite_text").arg(name), txtComposite->toPlainText());
            settings.setValue(QSL("studio/bc/%1/chk_composite").arg(name), chkComposite->isChecked() ? 1 : 0);
            settings.setValue(QSL("studio/bc/%1/comp_type").arg(name), cmbCompType->currentIndex());
        }
        if (cmbECI->isEnabled()) {
            settings.setValue(QSL("studio/bc/%1/eci").arg(name), cmbECI->currentIndex());
            settings.setValue(QSL("studio/bc/%1/eci_seg1").arg(name), cmbECISeg1->currentIndex());
            settings.setValue(QSL("studio/bc/%1/eci_seg2").arg(name), cmbECISeg2->currentIndex());
            settings.setValue(QSL("studio/bc/%1/eci_seg3").arg(name), cmbECISeg3->currentIndex());
        }
        settings.setValue(QSL("studio/bc/%1/chk_escape").arg(name), chkEscape->isChecked() ? 1 : 0);
        settings.setValue(QSL("studio/bc/%1/chk_data").arg(name), chkData->isChecked() ? 1 : 0);
        if (chkRInit->isEnabled()) {
            settings.setValue(QSL("studio/bc/%1/chk_rinit").arg(name), chkRInit->isChecked() ? 1 : 0);
        }
        settings.setValue(QSL("studio/bc/%1/chk_gs1parens").arg(name), chkGS1Parens->isChecked() ? 1 : 0);
        settings.setValue(QSL("studio/bc/%1/chk_gs1nocheck").arg(name), chkGS1NoCheck->isChecked() ? 1 : 0);
        if (chkAutoHeight->isEnabled()) {
            settings.setValue(
                QSL("studio/bc/%1/appearance/autoheight").arg(name), chkAutoHeight->isChecked() ? 1 : 0);
            settings.setValue(QSL("studio/bc/%1/appearance/height").arg(name), heightb->value());
        }
        if (chkCompliantHeight->isEnabled()) {
            settings.setValue(
                QSL("studio/bc/%1/appearance/compliantheight").arg(name), chkCompliantHeight->isChecked() ? 1 : 0);
        }
        settings.setValue(QSL("studio/bc/%1/appearance/border").arg(name), bwidth->value());
        settings.setValue(QSL("studio/bc/%1/appearance/whitespace").arg(name), spnWhitespace->value());
        settings.setValue(QSL("studio/bc/%1/appearance/vwhitespace").arg(name), spnVWhitespace->value());
        settings.setValue(QSL("studio/bc/%1/appearance/scale").arg(name), spnScale->value());
        settings.setValue(QSL("studio/bc/%1/appearance/border_type").arg(name), btype->currentIndex());
        if (chkHRTShow->isEnabled()) {
            settings.setValue(QSL("studio/bc/%1/appearance/font_setting").arg(name), cmbFontSetting->currentIndex());
            settings.setValue(QSL("studio/bc/%1/appearance/chk_hrt_show").arg(name), chkHRTShow->isChecked() ? 1 : 0);
        }
        settings.setValue(QSL("studio/bc/%1/appearance/chk_cmyk").arg(name), chkCMYK->isChecked() ? 1 : 0);
        settings.setValue(
            QSL("studio/bc/%1/appearance/chk_quietzones").arg(name), chkQuietZones->isChecked() ? 1 : 0);
        settings.setValue(QSL("studio/bc/%1/appearance/rotate").arg(name), cmbRotate->currentIndex());
        if (symbology == BARCODE_DOTCODE || chkDotty->isEnabled()) {
            settings.setValue(QSL("studio/bc/%1/appearance/chk_dotty").arg(name), chkDotty->isChecked() ? 1 : 0);
            settings.setValue(QSL("studio/bc/%1/appearance/dot_size").arg(name), spnDotSize->value());
        }
        settings.setValue(QSL("studio/bc/%1/ink/red").arg(name), m_fgcolor.red());
        settings.setValue(QSL("studio/bc/%1/ink/green").arg(name), m_fgcolor.green());
        settings.setValue(QSL("studio/bc/%1/ink/blue").arg(name), m_fgcolor.blue());
        settings.setValue(QSL("studio/bc/%1/ink/alpha").arg(name), m_fgcolor.alpha());
        settings.setValue(QSL("studio/bc/%1/paper/red").arg(name), m_bgcolor.red());
        settings.setValue(QSL("studio/bc/%1/paper/green").arg(name), m_bgcolor.green());
        settings.setValue(QSL("studio/bc/%1/paper/blue").arg(name), m_bgcolor.blue());
        settings.setValue(QSL("studio/bc/%1/paper/alpha").arg(name), m_bgcolor.alpha());
    }

    switch (symbology) {
        case BARCODE_CODE128:
        case BARCODE_CODE128B:
        case BARCODE_GS1_128:
        case BARCODE_GS1_128_CC:
        case BARCODE_HIBC_128:
            settings.setValue(QSL("studio/bc/code128/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radC128Stand") << QSL("radC128EAN") << QSL("radC128CSup")
                                << QSL("radC128HIBC")));
            break;

        case BARCODE_PDF417:
        case BARCODE_PDF417COMP:
        case BARCODE_HIBC_PDF:
            settings.setValue(QSL("studio/bc/pdf417/cols"), get_cmb_index(QSL("cmbPDFCols")));
            settings.setValue(QSL("studio/bc/pdf417/rows"), get_cmb_index(QSL("cmbPDFRows")));
            settings.setValue(QSL("studio/bc/pdf417/height_per_row"), get_dspn_val(QSL("spnPDFHeightPerRow")));
            settings.setValue(QSL("studio/bc/pdf417/ecc"), get_cmb_index(QSL("cmbPDFECC")));
            settings.setValue(QSL("studio/bc/pdf417/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radPDFStand") << QSL("radPDFTruncated") << QSL("radPDFHIBC")));
            settings.setValue(QSL("studio/bc/pdf417/structapp_count"), get_spn_val(QSL("spnPDFStructAppCount")));
            settings.setValue(QSL("studio/bc/pdf417/structapp_index"), get_spn_val(QSL("spnPDFStructAppIndex")));
            settings.setValue(QSL("studio/bc/pdf417/structapp_id"), get_txt_val(QSL("txtPDFStructAppID")));
            break;

        case BARCODE_MICROPDF417:
        case BARCODE_HIBC_MICPDF:
            settings.setValue(QSL("studio/bc/micropdf417/cols"), get_cmb_index(QSL("cmbMPDFCols")));
            settings.setValue(QSL("studio/bc/micropdf417/height_per_row"), get_dspn_val(QSL("spnMPDFHeightPerRow")));
            settings.setValue(QSL("studio/bc/micropdf417/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radMPDFStand") << QSL("radMPDFHIBC")));
            settings.setValue(QSL("studio/bc/micropdf417/structapp_count"),
                get_spn_val(QSL("spnMPDFStructAppCount")));
            settings.setValue(QSL("studio/bc/micropdf417/structapp_index"),
                get_spn_val(QSL("spnMPDFStructAppIndex")));
            settings.setValue(QSL("studio/bc/micropdf417/structapp_id"), get_txt_val(QSL("txtMPDFStructAppID")));
            break;

        case BARCODE_DOTCODE:
            settings.setValue(QSL("studio/bc/dotcode/cols"), get_cmb_index(QSL("cmbDotCols")));
            settings.setValue(QSL("studio/bc/dotcode/mask"), get_cmb_index(QSL("cmbDotMask")));
            settings.setValue(QSL("studio/bc/dotcode/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radDotStand") << QSL("radDotGS1")));
            settings.setValue(QSL("studio/bc/dotcode/structapp_count"), get_cmb_index(QSL("cmbDotStructAppCount")));
            settings.setValue(QSL("studio/bc/dotcode/structapp_index"), get_cmb_index(QSL("cmbDotStructAppIndex")));
            break;

        case BARCODE_AZTEC:
        case BARCODE_HIBC_AZTEC:
            settings.setValue(QSL("studio/bc/aztec/autoresizing"), get_rad_grp_index(
                QStringList() << QSL("radAztecAuto") << QSL("radAztecSize") << QSL("radAztecECC")));
            settings.setValue(QSL("studio/bc/aztec/size"), get_cmb_index(QSL("cmbAztecSize")));
            settings.setValue(QSL("studio/bc/aztec/ecc"), get_cmb_index(QSL("cmbAztecECC")));
            settings.setValue(QSL("studio/bc/aztec/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radAztecStand") << QSL("radAztecGS1") << QSL("radAztecHIBC")));
            settings.setValue(QSL("studio/bc/aztec/structapp_count"), get_cmb_index(QSL("cmbAztecStructAppCount")));
            settings.setValue(QSL("studio/bc/aztec/structapp_index"), get_cmb_index(QSL("cmbAztecStructAppIndex")));
            settings.setValue(QSL("studio/bc/aztec/structapp_id"), get_txt_val(QSL("txtAztecStructAppID")));
            break;

        case BARCODE_MSI_PLESSEY:
            settings.setValue(QSL("studio/bc/msi_plessey/check_digit"), get_cmb_index(QSL("cmbMSICheck")));
            settings.setValue(QSL("studio/bc/msi_plessey/check_text"), get_chk_val(QSL("chkMSICheckText")));
            break;

        case BARCODE_CODE11:
            settings.setValue(QSL("studio/bc/code11/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC11TwoCheckDigits") << QSL("radC11OneCheckDigit")
                                << QSL("radC11NoCheckDigits")));
            break;

        case BARCODE_C25STANDARD:
            settings.setValue(QSL("studio/bc/c25standard/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;
        case BARCODE_C25INTER:
            settings.setValue(QSL("studio/bc/c25inter/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;
        case BARCODE_C25IATA:
            settings.setValue(QSL("studio/bc/c25iata/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;
        case BARCODE_C25LOGIC:
            settings.setValue(QSL("studio/bc/c25logic/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;
        case BARCODE_C25IND:
            settings.setValue(QSL("studio/bc/c25ind/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide")));
            break;

        case BARCODE_CODE39:
        case BARCODE_HIBC_39:
            settings.setValue(QSL("studio/bc/code39/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC39Stand") << QSL("radC39Check") << QSL("radC39HIBC")));
            break;

        case BARCODE_EXCODE39:
            settings.setValue(QSL("studio/bc/excode39/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radC39Stand") << QSL("radC39Check")));
            break;

        case BARCODE_LOGMARS:
            settings.setValue(QSL("studio/bc/logmars/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radLOGMARSStand") << QSL("radLOGMARSCheck")));
            break;

        case BARCODE_CODE16K:
            settings.setValue(QSL("studio/bc/code16k/rows"), get_cmb_index(QSL("cmbC16kRows")));
            settings.setValue(QSL("studio/bc/code16k/height_per_row"), get_dspn_val(QSL("spnC16kHeightPerRow")));
            settings.setValue(QSL("studio/bc/code16k/row_sep_height"), get_cmb_index(QSL("cmbC16kRowSepHeight")));
            settings.setValue(QSL("studio/bc/code16k/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radC16kStand") << QSL("radC16kGS1")));
            settings.setValue(QSL("studio/bc/code16k/chk_no_quiet_zones"), get_chk_val(QSL("chkC16kNoQuietZones")));
            break;

        case BARCODE_CODABAR:
            settings.setValue(QSL("studio/bc/codabar/check_digit"), get_rad_grp_index(
                QStringList() << QSL("radCodabarStand") << QSL("radCodabarCheckHide") << QSL("radCodabarCheck")));
            break;

        case BARCODE_CODABLOCKF:
        case BARCODE_HIBC_BLOCKF:
            settings.setValue(QSL("studio/bc/codablockf/width"), get_cmb_index(QSL("cmbCbfWidth")));
            settings.setValue(QSL("studio/bc/codablockf/height"), get_cmb_index(QSL("cmbCbfHeight")));
            settings.setValue(QSL("studio/bc/codablockf/height_per_row"), get_dspn_val(QSL("spnCbfHeightPerRow")));
            settings.setValue(QSL("studio/bc/codablockf/row_sep_height"), get_cmb_index(QSL("cmbCbfRowSepHeight")));
            settings.setValue(QSL("studio/bc/codablockf/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radCbfStand") << QSL("radCbfHIBC")));
            settings.setValue(QSL("studio/bc/codablockf/chk_no_quiet_zones"), get_chk_val(QSL("chkCbfNoQuietZones")));
            break;

        case BARCODE_DAFT:
            settings.setValue(QSL("studio/bc/daft/tracker_ratio"),
                QString::number(get_dspn_val(QSL("spnDAFTTrackerRatio")), 'f', 1 /*precision*/));
            break;

        case BARCODE_DATAMATRIX:
        case BARCODE_HIBC_DM:
            settings.setValue(QSL("studio/bc/datamatrix/size"), get_cmb_index(QSL("cmbDM200Size")));
            settings.setValue(QSL("studio/bc/datamatrix/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radDM200Stand") << QSL("radDM200GS1") << QSL("radDM200HIBC")));
            settings.setValue(QSL("studio/bc/datamatrix/chk_suppress_rect"), get_chk_val(QSL("chkDMRectangle")));
            settings.setValue(QSL("studio/bc/datamatrix/chk_allow_dmre"), get_chk_val(QSL("chkDMRE")));
            settings.setValue(QSL("studio/bc/datamatrix/chk_gs_sep"), get_chk_val(QSL("chkDMGSSep")));
            settings.setValue(QSL("studio/bc/datamatrix/chk_fast"), get_chk_val(QSL("chkDMFast")));
            settings.setValue(QSL("studio/bc/datamatrix/structapp_count"), get_cmb_index(QSL("cmbDMStructAppCount")));
            settings.setValue(QSL("studio/bc/datamatrix/structapp_index"), get_cmb_index(QSL("cmbDMStructAppIndex")));
            settings.setValue(QSL("studio/bc/datamatrix/structapp_id"), get_spn_val(QSL("spnDMStructAppID")));
            settings.setValue(QSL("studio/bc/datamatrix/structapp_id2"), get_spn_val(QSL("spnDMStructAppID2")));
            break;

        case BARCODE_ITF14:
            settings.setValue(QSL("studio/bc/itf14/chk_no_quiet_zones"), get_chk_val(QSL("chkITF14NoQuietZones")));
            break;

        case BARCODE_QRCODE:
        case BARCODE_HIBC_QR:
            settings.setValue(QSL("studio/bc/qrcode/size"), get_cmb_index(QSL("cmbQRSize")));
            settings.setValue(QSL("studio/bc/qrcode/ecc"), get_cmb_index(QSL("cmbQRECC")));
            settings.setValue(QSL("studio/bc/qrcode/mask"), get_cmb_index(QSL("cmbQRMask")));
            settings.setValue(QSL("studio/bc/qrcode/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radDM200Stand") << QSL("radQRGS1") << QSL("radQRHIBC")));
            settings.setValue(QSL("studio/bc/qrcode/chk_full_multibyte"), get_chk_val(QSL("chkQRFullMultibyte")));
            settings.setValue(QSL("studio/bc/qrcode/structapp_count"), get_cmb_index(QSL("cmbQRStructAppCount")));
            settings.setValue(QSL("studio/bc/qrcode/structapp_index"), get_cmb_index(QSL("cmbQRStructAppIndex")));
            settings.setValue(QSL("studio/bc/qrcode/structapp_id"), get_spn_val(QSL("spnQRStructAppID")));
            break;

        case BARCODE_UPNQR:
            settings.setValue(QSL("studio/bc/upnqr/mask"), get_cmb_index(QSL("cmbUPNQRMask")));
            break;

        case BARCODE_RMQR:
            settings.setValue(QSL("studio/bc/rmqr/size"), get_cmb_index(QSL("cmbRMQRSize")));
            settings.setValue(QSL("studio/bc/rmqr/ecc"), get_cmb_index(QSL("cmbRMQRECC")));
            settings.setValue(QSL("studio/bc/rmqr/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radQRStand") << QSL("radRMQRGS1")));
            settings.setValue(QSL("studio/bc/rmqr/chk_full_multibyte"), get_chk_val(QSL("chkRMQRFullMultibyte")));
            break;

        case BARCODE_HANXIN:
            settings.setValue(QSL("studio/bc/hanxin/size"), get_cmb_index(QSL("cmbHXSize")));
            settings.setValue(QSL("studio/bc/hanxin/ecc"), get_cmb_index(QSL("cmbHXECC")));
            settings.setValue(QSL("studio/bc/hanxin/mask"), get_cmb_index(QSL("cmbHXMask")));
            settings.setValue(QSL("studio/bc/hanxin/chk_full_multibyte"), get_chk_val(QSL("chkHXFullMultibyte")));
            break;

        case BARCODE_MICROQR:
            settings.setValue(QSL("studio/bc/microqr/size"), get_cmb_index(QSL("cmbMQRSize")));
            settings.setValue(QSL("studio/bc/microqr/ecc"), get_cmb_index(QSL("cmbMQRECC")));
            settings.setValue(QSL("studio/bc/microqr/mask"), get_cmb_index(QSL("cmbMQRMask")));
            settings.setValue(QSL("studio/bc/microqr/chk_full_multibyte"), get_chk_val(QSL("chkMQRFullMultibyte")));
            break;

        case BARCODE_GRIDMATRIX:
            settings.setValue(QSL("studio/bc/gridmatrix/size"), get_cmb_index(QSL("cmbGridSize")));
            settings.setValue(QSL("studio/bc/gridmatrix/ecc"), get_cmb_index(QSL("cmbGridECC")));
            settings.setValue(QSL("studio/bc/gridmatrix/chk_full_multibyte"),
                get_chk_val(QSL("chkGridFullMultibyte")));
            settings.setValue(QSL("studio/bc/gridmatrix/structapp_count"),
                get_cmb_index(QSL("cmbGridStructAppCount")));
            settings.setValue(QSL("studio/bc/gridmatrix/structapp_index"),
                get_cmb_index(QSL("cmbGridStructAppIndex")));
            settings.setValue(QSL("studio/bc/gridmatrix/structapp_id"), get_spn_val(QSL("spnGridStructAppID")));
            break;

        case BARCODE_MAXICODE:
            settings.setValue(QSL("studio/bc/maxicode/mode"), get_cmb_index(QSL("cmbMaxiMode")));
            settings.setValue(QSL("studio/bc/maxicode/scm_postcode"), get_txt_val(QSL("txtMaxiSCMPostcode")));
            settings.setValue(QSL("studio/bc/maxicode/scm_country"), get_spn_val(QSL("spnMaxiSCMCountry")));
            settings.setValue(QSL("studio/bc/maxicode/scm_service"), get_spn_val(QSL("spnMaxiSCMService")));
            settings.setValue(QSL("studio/bc/maxicode/chk_scm_vv"), get_chk_val(QSL("chkMaxiSCMVV")));
            settings.setValue(QSL("studio/bc/maxicode/spn_scm_vv"), get_spn_val(QSL("spnMaxiSCMVV")));
            settings.setValue(QSL("studio/bc/maxicode/structapp_count"), get_cmb_index(QSL("cmbMaxiStructAppCount")));
            settings.setValue(QSL("studio/bc/maxicode/structapp_index"), get_cmb_index(QSL("cmbMaxiStructAppIndex")));
            break;

        case BARCODE_CHANNEL:
            settings.setValue(QSL("studio/bc/channel/channel"), get_cmb_index(QSL("cmbChannel")));
            break;

        case BARCODE_CODEONE:
            settings.setValue(QSL("studio/bc/codeone/size"), get_cmb_index(QSL("cmbC1Size")));
            settings.setValue(QSL("studio/bc/codeone/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radC1Stand") << QSL("radC1GS1")));
            settings.setValue(QSL("studio/bc/codeone/structapp_count"), get_spn_val(QSL("spnC1StructAppCount")));
            settings.setValue(QSL("studio/bc/codeone/structapp_index"), get_spn_val(QSL("spnC1StructAppIndex")));
            break;

        case BARCODE_CODE49:
            settings.setValue(QSL("studio/bc/code49/rows"), get_cmb_index(QSL("cmbC49Rows")));
            settings.setValue(QSL("studio/bc/code49/height_per_row"), get_dspn_val(QSL("spnC49HeightPerRow")));
            settings.setValue(QSL("studio/bc/code49/row_sep_height"), get_cmb_index(QSL("cmbC49RowSepHeight")));
            settings.setValue(QSL("studio/bc/code49/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radC49Stand") << QSL("radC49GS1")));
            settings.setValue(QSL("studio/bc/code49/chk_no_quiet_zones"), get_chk_val(QSL("chkC49NoQuietZones")));
            break;

        case BARCODE_CODE93:
            settings.setValue(QSL("studio/bc/code93/chk_show_checks"), get_chk_val(QSL("chkC93ShowChecks")));
            break;

        case BARCODE_DBAR_EXPSTK:
        case BARCODE_DBAR_EXPSTK_CC:
            settings.setValue(QSL("studio/bc/dbar_expstk/colsrows"), get_rad_grp_index(
                QStringList() << QSL("radDBESCols") << QSL("radDBESRows")));
            settings.setValue(QSL("studio/bc/dbar_expstk/cols"), get_cmb_index(QSL("cmbDBESCols")));
            settings.setValue(QSL("studio/bc/dbar_expstk/rows"), get_cmb_index(QSL("cmbDBESRows")));
            settings.setValue(QSL("studio/bc/dbar_expstk/height_per_row"), get_dspn_val(QSL("spnDBESHeightPerRow")));
            break;

        case BARCODE_ULTRA:
            settings.setValue(QSL("studio/bc/ultra/autoresizing"), get_rad_grp_index(
                QStringList() << QSL("radUltraAuto") << QSL("radUltraEcc")));
            settings.setValue(QSL("studio/bc/ultra/ecc"), get_cmb_index(QSL("cmbUltraEcc")));
            settings.setValue(QSL("studio/bc/ultra/revision"), get_cmb_index(QSL("cmbUltraRevision")));
            settings.setValue(QSL("studio/bc/ultra/encoding_mode"), get_rad_grp_index(
                QStringList() << QSL("radUltraStand") << QSL("radUltraGS1")));
            settings.setValue(QSL("studio/bc/ultra/structapp_count"), get_cmb_index(QSL("cmbUltraStructAppCount")));
            settings.setValue(QSL("studio/bc/ultra/structapp_index"), get_cmb_index(QSL("cmbUltraStructAppIndex")));
            settings.setValue(QSL("studio/bc/ultra/structapp_id"), get_spn_val(QSL("spnUltraStructAppID")));
            break;

        case BARCODE_UPCA:
        case BARCODE_UPCA_CHK:
        case BARCODE_UPCA_CC:
            settings.setValue(QSL("studio/bc/upca/addongap"), get_cmb_index(QSL("cmbUPCAAddonGap")));
            settings.setValue(QSL("studio/bc/upca/guard_descent"),
                QString::number(get_dspn_val(QSL("spnUPCAGuardDescent")), 'f', 3 /*precision*/));
            settings.setValue(QSL("studio/bc/upca/chk_no_quiet_zones"), get_chk_val(QSL("chkUPCANoQuietZones")));
            break;

        case BARCODE_EANX:
        case BARCODE_EANX_CHK:
        case BARCODE_EANX_CC:
            settings.setValue(QSL("studio/bc/eanx/addongap"), get_cmb_index(QSL("cmbUPCEANAddonGap")));
            settings.setValue(QSL("studio/bc/eanx/guard_descent"),
                QString::number(get_dspn_val(QSL("spnUPCEANGuardDescent")), 'f', 3 /*precision*/));
            settings.setValue(QSL("studio/bc/eanx/chk_no_quiet_zones"), get_chk_val(QSL("chkUPCEANNoQuietZones")));
            break;

        case BARCODE_UPCE:
        case BARCODE_UPCE_CHK:
        case BARCODE_UPCE_CC:
            settings.setValue(QSL("studio/bc/upce/addongap"), get_cmb_index(QSL("cmbUPCEANAddonGap")));
            settings.setValue(QSL("studio/bc/upce/guard_descent"),
                QString::number(get_dspn_val(QSL("spnUPCEANGuardDescent")), 'f', 3 /*precision*/));
            settings.setValue(QSL("studio/bc/upce/chk_no_quiet_zones"), get_chk_val(QSL("chkUPCEANNoQuietZones")));
            break;

        case BARCODE_ISBNX:
            settings.setValue(QSL("studio/bc/isnbx/addongap"), get_cmb_index(QSL("cmbUPCEANAddonGap")));
            settings.setValue(QSL("studio/bc/isnbx/guard_descent"),
                QString::number(get_dspn_val(QSL("spnUPCEANGuardDescent")), 'f', 3 /*precision*/));
            settings.setValue(QSL("studio/bc/isnbx/chk_no_quiet_zones"), get_chk_val(QSL("chkUPCEANNoQuietZones")));
            break;

        case BARCODE_VIN:
            settings.setValue(QSL("studio/bc/vin/chk_import_char_prefix"), get_chk_val(QSL("chkVINImportChar")));
            break;
    }
}

/* Load settings for an individual symbol */
void MainWindow::load_sub_settings(QSettings &settings, int symbology)
{
    QString name = get_setting_name(symbology);
    if (!name.isEmpty()) {
        const QString &tdata = settings.value(QSL("studio/bc/%1/data").arg(name), QSL("")).toString();
        if (!tdata.isEmpty()) {
            txtData->setText(tdata);
        }
        if (!grpSegs->isHidden()) {
            txtDataSeg1->setText(settings.value(QSL("studio/bc/%1/data_seg1").arg(name), QSL("")).toString());
            txtDataSeg2->setText(settings.value(QSL("studio/bc/%1/data_seg2").arg(name), QSL("")).toString());
            txtDataSeg3->setText(settings.value(QSL("studio/bc/%1/data_seg3").arg(name), QSL("")).toString());
        }
        if (!grpComposite->isHidden()) {
            const QString &composite_text = settings.value(
                                                QSL("studio/bc/%1/composite_text").arg(name), QSL("")).toString();
            if (!composite_text.isEmpty()) {
                txtComposite->setText(composite_text);
            }
            chkComposite->setChecked(settings.value(
                QSL("studio/bc/%1/chk_composite").arg(name), 0).toInt() ? true : false);
            cmbCompType->setCurrentIndex(settings.value(QSL("studio/bc/%1/comp_type").arg(name), 0).toInt());
        }
        if (cmbECI->isEnabled()) {
            cmbECI->setCurrentIndex(settings.value(QSL("studio/bc/%1/eci").arg(name), 0).toInt());
            cmbECISeg1->setCurrentIndex(settings.value(QSL("studio/bc/%1/eci_seg1").arg(name), 0).toInt());
            cmbECISeg2->setCurrentIndex(settings.value(QSL("studio/bc/%1/eci_seg2").arg(name), 0).toInt());
            cmbECISeg3->setCurrentIndex(settings.value(QSL("studio/bc/%1/eci_seg3").arg(name), 0).toInt());
        }
        chkEscape->setChecked(settings.value(QSL("studio/bc/%1/chk_escape").arg(name)).toInt() ? true : false);
        chkData->setChecked(settings.value(QSL("studio/bc/%1/chk_data").arg(name)).toInt() ? true : false);
        if (chkRInit->isEnabled()) {
            chkRInit->setChecked(settings.value(QSL("studio/bc/%1/chk_rinit").arg(name)).toInt() ? true : false);
        }
        chkGS1Parens->setChecked(settings.value(QSL("studio/bc/%1/chk_gs1parens").arg(name)).toInt() ? true : false);
        chkGS1NoCheck->setChecked(settings.value(
            QSL("studio/bc/%1/chk_gs1nocheck").arg(name)).toInt() ? true : false);
        if (chkAutoHeight->isEnabled()) {
            chkAutoHeight->setChecked(settings.value(
                QSL("studio/bc/%1/appearance/autoheight").arg(name), 1).toInt() ? true : false);
            heightb->setValue(settings.value(QSL("studio/bc/%1/appearance/height").arg(name), 50.0f).toFloat());
        }
        if (chkCompliantHeight->isEnabled()) {
            chkCompliantHeight->setChecked(settings.value(
                QSL("studio/bc/%1/appearance/compliantheight").arg(name), 1).toInt() ? true : false);
        }
        bwidth->setValue(settings.value(QSL("studio/bc/%1/appearance/border").arg(name), 0).toInt());
        spnWhitespace->setValue(settings.value(QSL("studio/bc/%1/appearance/whitespace").arg(name), 0).toInt());
        spnVWhitespace->setValue(settings.value(QSL("studio/bc/%1/appearance/vwhitespace").arg(name), 0).toInt());
        spnScale->setValue(settings.value(QSL("studio/bc/%1/appearance/scale").arg(name), 1.0).toFloat());
        btype->setCurrentIndex(settings.value(QSL("studio/bc/%1/appearance/border_type").arg(name), 0).toInt());
        if (chkHRTShow->isEnabled()) {
            cmbFontSetting->setCurrentIndex(settings.value(
                QSL("studio/bc/%1/appearance/font_setting").arg(name), 0).toInt());
            chkHRTShow->setChecked(settings.value(
                QSL("studio/bc/%1/appearance/chk_hrt_show").arg(name), 1).toInt() ? true : false);
        }
        chkCMYK->setChecked(settings.value(QSL("studio/bc/%1/appearance/cmyk").arg(name), 0).toInt() ? true : false);
        chkQuietZones->setChecked(settings.value(
            QSL("studio/bc/%1/appearance/chk_quietzones").arg(name), 0).toInt() ? true : false);
        cmbRotate->setCurrentIndex(settings.value(QSL("studio/bc/%1/appearance/rotate").arg(name), 0).toInt());
        if (symbology == BARCODE_DOTCODE || chkDotty->isEnabled()) {
            chkDotty->setChecked(settings.value(
                QSL("studio/bc/%1/appearance/chk_dotty").arg(name), 0).toInt() ? true : false);
            spnDotSize->setValue(settings.value(
                QSL("studio/bc/%1/appearance/dot_size").arg(name), 0.4f / 0.5f).toFloat());
        }
        m_fgcolor.setRgb(settings.value(QSL("studio/bc/%1/ink/red").arg(name), 0).toInt(),
                        settings.value(QSL("studio/bc/%1/ink/green").arg(name), 0).toInt(),
                        settings.value(QSL("studio/bc/%1/ink/blue").arg(name), 0).toInt(),
                        settings.value(QSL("studio/bc/%1/ink/alpha").arg(name), 0xff).toInt());
        m_bgcolor.setRgb(settings.value(QSL("studio/bc/%1/paper/red").arg(name), 0xff).toInt(),
                        settings.value(QSL("studio/bc/%1/paper/green").arg(name), 0xff).toInt(),
                        settings.value(QSL("studio/bc/%1/paper/blue").arg(name), 0xff).toInt(),
                        settings.value(QSL("studio/bc/%1/paper/alpha").arg(name), 0xff).toInt());
    }

    switch (symbology) {
        case BARCODE_CODE128:
        case BARCODE_CODE128B:
        case BARCODE_GS1_128:
        case BARCODE_GS1_128_CC:
        case BARCODE_HIBC_128:
            set_rad_from_setting(settings, QSL("studio/bc/code128/encoding_mode"),
                QStringList() << QSL("radC128Stand") << QSL("radC128EAN") << QSL("radC128CSup")
                                << QSL("radC128HIBC"));
            break;

        case BARCODE_PDF417:
        case BARCODE_PDF417COMP:
        case BARCODE_HIBC_PDF:
            set_cmb_from_setting(settings, QSL("studio/bc/pdf417/cols"), QSL("cmbPDFCols"));
            set_cmb_from_setting(settings, QSL("studio/bc/pdf417/rows"), QSL("cmbPDFRows"));
            set_dspn_from_setting(settings, QSL("studio/bc/pdf417/height_per_row"), QSL("spnPDFHeightPerRow"), 0.0f);
            set_cmb_from_setting(settings, QSL("studio/bc/pdf417/ecc"), QSL("cmbPDFECC"));
            set_rad_from_setting(settings, QSL("studio/bc/pdf417/encoding_mode"),
                QStringList() << QSL("radPDFStand") << QSL("radPDFTruncated") << QSL("radPDFHIBC"));
            set_spn_from_setting(settings, QSL("studio/bc/pdf417/structapp_count"), QSL("spnPDFStructAppCount"), 1);
            set_spn_from_setting(settings, QSL("studio/bc/pdf417/structapp_index"), QSL("spnPDFStructAppIndex"), 0);
            set_txt_from_setting(settings, QSL("studio/bc/pdf417/structapp_id"), QSL("txtPDFStructAppID"), QSL(""));
            break;

        case BARCODE_MICROPDF417:
        case BARCODE_HIBC_MICPDF:
            set_cmb_from_setting(settings, QSL("studio/bc/micropdf417/cols"), QSL("cmbMPDFCols"));
            set_dspn_from_setting(settings, QSL("studio/bc/micropdf417/height_per_row"), QSL("spnMPDFHeightPerRow"),
                0.0f);
            set_rad_from_setting(settings, QSL("studio/bc/micropdf417/encoding_mode"),
                QStringList() << QSL("radMPDFStand") << QSL("radMPDFHIBC"));
            set_spn_from_setting(settings, QSL("studio/bc/micropdf417/structapp_count"),
                QSL("spnMPDFStructAppCount"), 1);
            set_spn_from_setting(settings, QSL("studio/bc/micropdf417/structapp_index"),
                QSL("spnMPDFStructAppIndex"), 0);
            set_txt_from_setting(settings, QSL("studio/bc/micropdf417/structapp_id"), QSL("txtMPDFStructAppID"),
                QSL(""));
            break;

        case BARCODE_DOTCODE:
            set_cmb_from_setting(settings, QSL("studio/bc/dotcode/cols"), QSL("cmbDotCols"));
            set_cmb_from_setting(settings, QSL("studio/bc/dotcode/mask"), QSL("cmbDotMask"));
            set_rad_from_setting(settings, QSL("studio/bc/dotcode/encoding_mode"),
                QStringList() << QSL("radDotStand") << QSL("radDotGS1"));
            set_cmb_from_setting(settings, QSL("studio/bc/dotcode/structapp_count"), QSL("cmbDotStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/dotcode/structapp_index"), QSL("cmbDotStructAppIndex"));
            break;

        case BARCODE_AZTEC:
        case BARCODE_HIBC_AZTEC:
            set_rad_from_setting(settings, QSL("studio/bc/aztec/autoresizing"),
                QStringList() << QSL("radAztecAuto") << QSL("radAztecSize") << QSL("radAztecECC"));
            set_cmb_from_setting(settings, QSL("studio/bc/aztec/size"), QSL("cmbAztecSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/aztec/ecc"), QSL("cmbAztecECC"));
            set_rad_from_setting(settings, QSL("studio/bc/aztec/encoding_mode"),
                QStringList() << QSL("radAztecStand") << QSL("radAztecGS1") << QSL("radAztecHIBC"));
            set_cmb_from_setting(settings, QSL("studio/bc/aztec/structapp_count"), QSL("cmbAztecStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/aztec/structapp_index"), QSL("cmbAztecStructAppIndex"));
            set_txt_from_setting(settings, QSL("studio/bc/aztec/structapp_id"), QSL("txtAztecStructAppID"), QSL(""));
            break;

        case BARCODE_MSI_PLESSEY:
            set_cmb_from_setting(settings, QSL("studio/bc/msi_plessey/check_digit"), QSL("cmbMSICheck"));
            set_chk_from_setting(settings, QSL("studio/bc/msi_plessey/check_text"), QSL("chkMSICheckText"));
            msi_plessey_ui_set();
            break;

        case BARCODE_CODE11:
            set_rad_from_setting(settings, QSL("studio/bc/code11/check_digit"),
                QStringList() << QSL("radC11TwoCheckDigits") << QSL("radC11OneCheckDigit")
                                << QSL("radC11NoCheckDigits"));
            break;

        case BARCODE_C25STANDARD:
            set_rad_from_setting(settings, QSL("studio/bc/c25standard/check_digit"),
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide"));
            break;
        case BARCODE_C25INTER:
            set_rad_from_setting(settings, QSL("studio/bc/c25inter/check_digit"),
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide"));
            break;
        case BARCODE_C25IATA:
            set_rad_from_setting(settings, QSL("studio/bc/c25iata/check_digit"),
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide"));
            break;
        case BARCODE_C25LOGIC:
            set_rad_from_setting(settings, QSL("studio/bc/c25logic/check_digit"),
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide"));
            break;
        case BARCODE_C25IND:
            set_rad_from_setting(settings, QSL("studio/bc/c25ind/check_digit"),
                QStringList() << QSL("radC25Stand") << QSL("radC25Check") << QSL("radC25CheckHide"));
            break;

        case BARCODE_CODE39:
        case BARCODE_HIBC_39:
            set_rad_from_setting(settings, QSL("studio/bc/code39/check_digit"),
                QStringList() << QSL("radC39Stand") << QSL("radC39Check") << QSL("radC39HIBC"));
            break;

        case BARCODE_EXCODE39:
            set_rad_from_setting(settings, QSL("studio/bc/excode39/check_digit"),
                QStringList() << QSL("radC39Stand") << QSL("radC39Check"));
            break;

        case BARCODE_LOGMARS:
            set_rad_from_setting(settings, QSL("studio/bc/logmars/check_digit"),
                QStringList() << QSL("radLOGMARSStand") << QSL("radLOGMARSCheck"));
            break;

        case BARCODE_CODE16K:
            set_cmb_from_setting(settings, QSL("studio/bc/code16k/rows"), QSL("cmbC16kRows"));
            set_dspn_from_setting(settings, QSL("studio/bc/code16k/height_per_row"), QSL("spnC16kHeightPerRow"),
                0.0f);
            set_cmb_from_setting(settings, QSL("studio/bc/code16k/row_sep_height"), QSL("cmbC16kRowSepHeight"));
            set_rad_from_setting(settings, QSL("studio/bc/code16k/encoding_mode"),
                QStringList() << QSL("radC16kStand") << QSL("radC16kGS1"));
            set_chk_from_setting(settings, QSL("studio/bc/code16k/chk_no_quiet_zones"), QSL("chkC16kNoQuietZones"));
            break;

        case BARCODE_CODABAR:
            set_rad_from_setting(settings, QSL("studio/bc/codabar/check_digit"),
                QStringList() << QSL("radCodabarStand") << QSL("radCodabarCheckHide") << QSL("radCodabarCheck"));
            break;

        case BARCODE_CODABLOCKF:
        case BARCODE_HIBC_BLOCKF:
            set_cmb_from_setting(settings, QSL("studio/bc/codablockf/width"), QSL("cmbCbfWidth"));
            set_cmb_from_setting(settings, QSL("studio/bc/codablockf/height"), QSL("cmbCbfHeight"));
            set_dspn_from_setting(settings, QSL("studio/bc/codablockf/height_per_row"), QSL("spnCbfHeightPerRow"),
                0.0f);
            set_cmb_from_setting(settings, QSL("studio/bc/codablockf/row_sep_height"),
                QSL("cmbCbfRowSepHeight"));
            set_rad_from_setting(settings, QSL("studio/bc/codablockf/encoding_mode"),
                QStringList() << QSL("radCbfStand") << QSL("radCbfHIBC"));
            set_chk_from_setting(settings, QSL("studio/bc/codablockf/chk_no_quiet_zones"), QSL("chkCbfNoQuietZones"));
            break;

        case BARCODE_DAFT:
            set_dspn_from_setting(settings, QSL("studio/bc/daft/tracker_ratio"), QSL("spnDAFTTrackerRatio"), 25.0f);
            break;

        case BARCODE_DATAMATRIX:
        case BARCODE_HIBC_DM:
            set_cmb_from_setting(settings, QSL("studio/bc/datamatrix/size"), QSL("cmbDM200Size"));
            set_rad_from_setting(settings, QSL("studio/bc/datamatrix/encoding_mode"),
                QStringList() << QSL("radDM200Stand") << QSL("radDM200GS1") << QSL("radDM200HIBC"));
            set_chk_from_setting(settings, QSL("studio/bc/datamatrix/chk_suppress_rect"), QSL("chkDMRectangle"));
            set_chk_from_setting(settings, QSL("studio/bc/datamatrix/chk_allow_dmre"), QSL("chkDMRE"));
            set_chk_from_setting(settings, QSL("studio/bc/datamatrix/chk_gs_sep"), QSL("chkDMGSSep"));
            set_chk_from_setting(settings, QSL("studio/bc/datamatrix/chk_fast"), QSL("chkDMFast"));
            set_cmb_from_setting(settings, QSL("studio/bc/datamatrix/structapp_count"), QSL("cmbDMStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/datamatrix/structapp_index"), QSL("cmbDMStructAppIndex"));
            set_spn_from_setting(settings, QSL("studio/bc/datamatrix/structapp_id"), QSL("spnDMStructAppID"), 1);
            set_spn_from_setting(settings, QSL("studio/bc/datamatrix/structapp_id2"), QSL("spnDMStructAppID2"), 1);
            break;

        case BARCODE_ITF14:
            set_chk_from_setting(settings, QSL("studio/bc/itf14/chk_no_quiet_zones"), QSL("chkITF14NoQuietZones"));
            break;

        case BARCODE_QRCODE:
        case BARCODE_HIBC_QR:
            set_cmb_from_setting(settings, QSL("studio/bc/qrcode/size"), QSL("cmbQRSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/qrcode/ecc"), QSL("cmbQRECC"));
            set_cmb_from_setting(settings, QSL("studio/bc/qrcode/mask"), QSL("cmbQRMask"));
            set_rad_from_setting(settings, QSL("studio/bc/qrcode/encoding_mode"),
                QStringList() << QSL("radDM200Stand") << QSL("radQRGS1") << QSL("radQRHIBC"));
            set_chk_from_setting(settings, QSL("studio/bc/qrcode/chk_full_multibyte"), QSL("chkQRFullMultibyte"));
            set_cmb_from_setting(settings, QSL("studio/bc/qrcode/structapp_count"), QSL("cmbQRStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/qrcode/structapp_index"), QSL("cmbQRStructAppIndex"));
            set_spn_from_setting(settings, QSL("studio/bc/qrcode/structapp_id"), QSL("spnQRStructAppID"), 0);
            break;

        case BARCODE_UPNQR:
            set_cmb_from_setting(settings, QSL("studio/bc/upnqr/mask"), QSL("cmbUPNQRMask"));
            break;

        case BARCODE_RMQR:
            set_cmb_from_setting(settings, QSL("studio/bc/rmqr/size"), QSL("cmbRMQRSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/rmqr/ecc"), QSL("cmbRMQRECC"));
            set_rad_from_setting(settings, QSL("studio/bc/rmqr/encoding_mode"),
                QStringList() << QSL("radQRStand") << QSL("radRMQRGS1"));
            set_chk_from_setting(settings, QSL("studio/bc/rmqr/chk_full_multibyte"), QSL("chkRMQRFullMultibyte"));
            break;

        case BARCODE_HANXIN:
            set_cmb_from_setting(settings, QSL("studio/bc/hanxin/size"), QSL("cmbHXSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/hanxin/ecc"), QSL("cmbHXECC"));
            set_cmb_from_setting(settings, QSL("studio/bc/hanxin/mask"), QSL("cmbHXMask"));
            set_chk_from_setting(settings, QSL("studio/bc/hanxin/chk_full_multibyte"), QSL("chkHXFullMultibyte"));
            break;

        case BARCODE_MICROQR:
            set_cmb_from_setting(settings, QSL("studio/bc/microqr/size"), QSL("cmbMQRSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/microqr/ecc"), QSL("cmbMQRECC"));
            set_cmb_from_setting(settings, QSL("studio/bc/microqr/mask"), QSL("cmbMQRMask"));
            set_chk_from_setting(settings, QSL("studio/bc/microqr/chk_full_multibyte"), QSL("chkMQRFullMultibyte"));
            break;

        case BARCODE_GRIDMATRIX:
            set_cmb_from_setting(settings, QSL("studio/bc/gridmatrix/size"), QSL("cmbGridSize"));
            set_cmb_from_setting(settings, QSL("studio/bc/gridmatrix/ecc"), QSL("cmbGridECC"));
            set_chk_from_setting(settings, QSL("studio/bc/gridmatrix/chk_full_multibyte"),
                QSL("chkGridFullMultibyte"));
            set_cmb_from_setting(settings, QSL("studio/bc/gridmatrix/structapp_count"), QSL("cmbGridStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/gridmatrix/structapp_index"), QSL("cmbGridStructAppIndex"));
            set_spn_from_setting(settings, QSL("studio/bc/gridmatrix/structapp_id"), QSL("spnGridStructAppID"), 0);
            break;

        case BARCODE_MAXICODE:
            set_cmb_from_setting(settings, QSL("studio/bc/maxicode/mode"), QSL("cmbMaxiMode"), 1);
            set_txt_from_setting(settings, QSL("studio/bc/maxicode/scm_postcode"), QSL("txtMaxiSCMPostcode"),
                QSL(""));
            set_spn_from_setting(settings, QSL("studio/bc/maxicode/scm_country"), QSL("spnMaxiSCMCountry"), 0);
            set_spn_from_setting(settings, QSL("studio/bc/maxicode/scm_service"), QSL("spnMaxiSCMService"), 0);
            set_chk_from_setting(settings, QSL("studio/bc/maxicode/chk_scm_vv"), QSL("chkMaxiSCMVV"));
            // 96 is ASC MH10/SC 8
            set_spn_from_setting(settings, QSL("studio/bc/maxicode/spn_scm_vv"), QSL("spnMaxiSCMVV"), 96);
            set_cmb_from_setting(settings, QSL("studio/bc/maxicode/structapp_count"), QSL("cmbMaxiStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/maxicode/structapp_index"), QSL("cmbMaxiStructAppIndex"));
            break;

        case BARCODE_CHANNEL:
            set_cmb_from_setting(settings, QSL("studio/bc/channel/channel"), QSL("cmbChannel"));
            break;

        case BARCODE_CODEONE:
            set_cmb_from_setting(settings, QSL("studio/bc/codeone/size"), QSL("cmbC1Size"));
            set_rad_from_setting(settings, QSL("studio/bc/codeone/encoding_mode"),
                QStringList() << QSL("radC1Stand") << QSL("radC1GS1"));
            set_spn_from_setting(settings, QSL("studio/bc/codeone/structapp_count"), QSL("spnC1StructAppCount"), 1);
            set_spn_from_setting(settings, QSL("studio/bc/codeone/structapp_index"), QSL("spnC1StructAppIndex"), 0);
            break;

        case BARCODE_CODE49:
            set_cmb_from_setting(settings, QSL("studio/bc/code49/rows"), QSL("cmbC49Rows"));
            set_dspn_from_setting(settings, QSL("studio/bc/code49/height_per_row"), QSL("spnC49HeightPerRow"), 0.0f);
            set_cmb_from_setting(settings, QSL("studio/bc/code49/row_sep_height"), QSL("cmbC49RowSepHeight"));
            set_rad_from_setting(settings, QSL("studio/bc/code49/encoding_mode"),
                QStringList() << QSL("radC49Stand") << QSL("radC49GS1"));
            set_chk_from_setting(settings, QSL("studio/bc/code49/chk_no_quiet_zones"), QSL("chkC49NoQuietZones"));
            break;

        case BARCODE_CODE93:
            set_chk_from_setting(settings, QSL("studio/bc/code93/chk_show_checks"), QSL("chkC93ShowChecks"));
            break;

        case BARCODE_DBAR_EXPSTK:
            set_rad_from_setting(settings, QSL("studio/bc/dbar_expstk/colsrows"),
                QStringList() << QSL("radDBESCols") << QSL("radDBESRows"));
            set_cmb_from_setting(settings, QSL("studio/bc/dbar_expstk/cols"), QSL("cmbDBESCols"));
            set_cmb_from_setting(settings, QSL("studio/bc/dbar_expstk/rows"), QSL("cmbDBESRows"));
            set_dspn_from_setting(settings, QSL("studio/bc/dbar_expstk/height_per_row"), QSL("spnDBESHeightPerRow"),
                0.0f);
            break;

        case BARCODE_ULTRA:
            set_rad_from_setting(settings, QSL("studio/bc/ultra/autoresizing"),
                QStringList() << QSL("radUltraAuto") << QSL("radUltraEcc"));
            set_cmb_from_setting(settings, QSL("studio/bc/ultra/ecc"), QSL("cmbUltraEcc"));
            set_cmb_from_setting(settings, QSL("studio/bc/ultra/revision"), QSL("cmbUltraRevision"));
            set_rad_from_setting(settings, QSL("studio/bc/ultra/encoding_mode"),
                QStringList() << QSL("radUltraStand") << QSL("radUltraGS1"));
            set_cmb_from_setting(settings, QSL("studio/bc/ultra/structapp_count"), QSL("cmbUltraStructAppCount"));
            set_cmb_from_setting(settings, QSL("studio/bc/ultra/structapp_index"), QSL("cmbUltraStructAppIndex"));
            set_spn_from_setting(settings, QSL("studio/bc/ultra/structapp_id"), QSL("spnUltraStructAppID"), 0);
            break;

        case BARCODE_UPCA:
        case BARCODE_UPCA_CHK:
        case BARCODE_UPCA_CC:
            set_cmb_from_setting(settings, QSL("studio/bc/upca/addongap"), QSL("cmbUPCAAddonGap"));
            set_dspn_from_setting(settings, QSL("studio/bc/upca/guard_descent"), QSL("spnUPCAGuardDescent"), 5.0f);
            set_chk_from_setting(settings, QSL("studio/bc/upca/chk_no_quiet_zones"), QSL("chkUPCANoQuietZones"));
            break;

        case BARCODE_EANX:
        case BARCODE_EANX_CHK:
        case BARCODE_EANX_CC:
            set_cmb_from_setting(settings, QSL("studio/bc/eanx/addongap"), QSL("cmbUPCEANAddonGap"));
            set_dspn_from_setting(settings, QSL("studio/bc/eanx/guard_descent"), QSL("spnUPCEANGuardDescent"), 5.0f);
            set_chk_from_setting(settings, QSL("studio/bc/eanx/chk_no_quiet_zones"), QSL("chkUPCEANNoQuietZones"));
            break;

        case BARCODE_UPCE:
        case BARCODE_UPCE_CHK:
        case BARCODE_UPCE_CC:
            set_cmb_from_setting(settings, QSL("studio/bc/upce/addongap"), QSL("cmbUPCEANAddonGap"));
            set_dspn_from_setting(settings, QSL("studio/bc/upce/guard_descent"), QSL("spnUPCEANGuardDescent"), 5.0f);
            set_chk_from_setting(settings, QSL("studio/bc/upce/chk_no_quiet_zones"), QSL("chkUPCEANNoQuietZones"));
            break;

        case BARCODE_ISBNX:
            set_cmb_from_setting(settings, QSL("studio/bc/isbnx/addongap"), QSL("cmbUPCEANAddonGap"));
            set_dspn_from_setting(settings, QSL("studio/bc/isbnx/guard_descent"), QSL("spnUPCEANGuardDescent"), 5.0f);
            set_chk_from_setting(settings, QSL("studio/bc/isbnx/chk_no_quiet_zones"), QSL("chkUPCEANNoQuietZones"));
            break;

        case BARCODE_VIN:
            set_chk_from_setting(settings, QSL("studio/bc/vin/chk_import_char_prefix"), QSL("chkVINImportChar"));
            break;
    }
}

/* vim: set ts=4 sw=4 et : */
