# Plone Solutions AS <info@plonesolutions.com>
# http://www.plonesolutions.com

# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.

# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.

# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

"""
Baseclass for multilingual content.
"""

from Globals import InitializeClass
from AccessControl import ClassSecurityInfo

from Acquisition import Implicit
from Acquisition import aq_inner
from Acquisition import aq_parent
from OFS.ObjectManager import BeforeDeleteException

from Products.CMFCore.utils import getToolByName
from Products.CMFCore.DynamicType import DynamicType

from Products.Archetypes.public import *
from Products.Archetypes.utils import shasattr

from Products.LinguaPlone import config
from Products.LinguaPlone import permissions

if config.IS_PLONE_2_0:
    from Products.PloneLanguageTool.interfaces import ITranslatable
else:
    from Products.CMFPlone.interfaces.Translatable import ITranslatable

try:
    True
except NameError:
    True = 1
    False = 0


class AlreadyTranslated(Exception):
    """Raised when trying to create an existing translation."""
    pass


class I18NBaseObject(Implicit):
    """Base class for translatable objects."""
    __implements__ = (ITranslatable,)

    security = ClassSecurityInfo()

    security.declareProtected(permissions.View, 'isTranslation')
    def isTranslation(self):
        """Tells whether this object is used in a i18n context."""
        return bool(self.getReferenceImpl(config.RELATIONSHIP) or \
                    self.getBackReferenceImpl(config.RELATIONSHIP) \
                    and self.getLanguage() or False)

    security.declareProtected(permissions.AddPortalContent, 'addTranslation')
    def addTranslation(self, language, *args, **kwargs):
        """Adds a translation."""
        parent = aq_parent(aq_inner(self))
        if ITranslatable.isImplementedBy(parent):
            parent = parent.getTranslation(language) or parent
        canonical = self.getCanonical()
        if self.hasTranslation(language):
            translation = self.getTranslation(language)
            raise AlreadyTranslated, translation.absolute_url()
        id = canonical.getId()
        while not parent.checkIdAvailable(id):
            id = '%s-%s' % (id, language)
        if kwargs.get('language', None) != language:
            kwargs['language'] = language
        kwargs[config.KWARGS_TRANSLATION_KEY] = canonical
        parent.invokeFactory(self.portal_type, id, *args, **kwargs)
        o = getattr(parent, id)
        # If there is a custom factory method that doesn't add the
        # translation relationship, make sure it is done now.
        if o.getCanonical() != self:
            o.addTranslationReference(canonical)
        self.invalidateTranslationCache()
        # If this is a folder, move translated subobjects aswell.
        if self.isPrincipiaFolderish:
            moveids = []
            for obj in self.objectValues():
                if ITranslatable.isImplementedBy(obj) and \
                   obj.getLanguage() == language:
                    moveids.append(obj.getId())
            if moveids:
                o.manage_pasteObjects(self.manage_cutObjects(moveids))
        o.reindexObject()
        plone_utils = getToolByName(self, 'plone_utils', None)
        if plone_utils is not None and plone_utils.isDefaultPage(canonical):
            o._lp_default_page = True
        else:
            o._lp_not_renamed = True

    security.declareProtected(permissions.AddPortalContent,
                              'addTranslationReference')
    def addTranslationReference(self, translation):
        """Adds the reference used to keep track of translations."""
        self.addReference(translation, config.RELATIONSHIP)

    security.declareProtected(permissions.ModifyPortalContent,
                              'removeTranslation')
    def removeTranslation(self, language):
        """Removes a translation, pass on to layer."""
        translation = self.getTranslation(language)
        if translation.isCanonical():
            self.setCanonical()
        translation_parent = aq_parent(aq_inner(translation))
        translation_parent.manage_delObjects([translation.getId()])
        self.invalidateTranslationCache()

    security.declareProtected(permissions.View, 'hasTranslation')
    def hasTranslation(self, language):
        """Checks if a given language has a translation."""
        return language in self.getTranslationLanguages()

    security.declareProtected(permissions.View, 'getTranslation')
    def getTranslation(self, language=None):
        """Gets a translation, pass on to layer."""
        if language is None:
            language_tool = getToolByName(self, 'portal_languages', None)
            if language_tool is not None:
                language = language_tool.getPreferredLanguage()
            else:
                return self
        l = self.getTranslations().get(language, None)
        return l and l[0] or l

    security.declareProtected(permissions.View, 'getTranslationLanguages')
    def getTranslationLanguages(self):
        """Returns a list of language codes.

        Note that we return all translations available. If you want only
        the translations from the current portal_language selected list,
        you should use the getTranslatedLanguages script.
        """
        return self.getTranslations().keys()

    security.declareProtected(permissions.View, 'getTranslations')
    def getTranslations(self):
        """Returns a dict of {lang : [object, wf_state]}, pass on to layer."""
        if self.isCanonical():
            if config.CACHE_TRANSLATIONS and \
               getattr(self, '_v_translations', None):
                return self._v_translations
            result = {}
            workflow_tool = getToolByName(self, 'portal_workflow', None)
            if workflow_tool is None:
                # No context, most likely FTP or WebDAV
                result[self.getLanguage()] = [self, None]
                return result
            lang = self.getLanguage()
            state = workflow_tool.getInfoFor(self, 'review_state', None)
            result[lang] = [self, state]
            for obj in self.getBRefs(config.RELATIONSHIP):
                lang = obj.getLanguage()
                state = workflow_tool.getInfoFor(obj, 'review_state', None)
                result[lang] = [obj, state]
            if config.CACHE_TRANSLATIONS:
                self._v_translations = result
            return result
        else:
            return self.getCanonical().getTranslations()

    security.declareProtected(permissions.View, 'getNonCanonicalTranslations')
    def getNonCanonicalTranslations(self):
        """Returns a dict of {lang : [object, wf_state]}."""
        translations = self.getTranslations()
        non_canonical = {}
        for lang in translations.keys():
            if not translations[lang][0].isCanonical():
                non_canonical[lang] = translations[lang]
        return non_canonical

    security.declareProtected(permissions.View, 'isCanonical')
    def isCanonical(self):
        """Tells whether this is the canonical translation.

        An object is considered 'canonical' when there's no
        'translationOf' references associated.
        """
        try:
            return not bool(self.getReferenceImpl(config.RELATIONSHIP))
        except AttributeError:
            return True

    security.declareProtected(permissions.ModifyPortalContent, 'setCanonical')
    def setCanonical(self):
        """Sets the canonical attribute."""
        if not self.isCanonical():
            translations = self.getTranslations()
            for obj, wfstate in translations.values():
                obj.deleteReferences(config.RELATIONSHIP)
            for obj, wfstate in translations.values():
                if obj != self:
                    obj.addTranslationReference(self)
            self.invalidateTranslationCache()

    security.declareProtected(permissions.View, 'getCanonicalLanguage')
    def getCanonicalLanguage(self):
        """Returns the language code for the canonical language."""
        return self.getCanonical().getLanguage()

    security.declareProtected(permissions.View, 'getCanonical')
    def getCanonical(self):
        """Returns the canonical translation."""
        if config.CACHE_TRANSLATIONS and getattr(self, '_v_canonical', None):
            return self._v_canonical
        ret = None

        if self.isCanonical():
            ret = self
        else:
            refs = self.getRefs(config.RELATIONSHIP)
            ret = refs and refs[0] or None

        if config.CACHE_TRANSLATIONS:
            self._v_canonical = ret
        return ret

    security.declareProtected(permissions.View, 'getLanguage')
    def getLanguage(self):
        """Returns the language code."""
        return self.Language()

    security.declareProtected(permissions.ModifyPortalContent, 'setLanguage')
    def setLanguage(self, value, **kwargs):
        """Sets the language code.

        When changing the language in a translated folder structure,
        we try to move the content to the existing language tree.
        """
        translation = self.getTranslation(value)
        if self.hasTranslation(value):
            if translation == self:
                return
            else:
                raise AlreadyTranslated, translation.absolute_url()
        self.getField('language').set(self, value, **kwargs)
        if not value:
            self.deleteReferences(config.RELATIONSHIP)
        parent = aq_parent(aq_inner(self))
        if ITranslatable.isImplementedBy(parent):
            new_parent = parent.getTranslation(value) or parent
            if new_parent != parent:
                info = parent.manage_cutObjects([self.getId()])
                new_parent.manage_pasteObjects(info)
        self.reindexObject()
        self.invalidateTranslationCache()

    security.declareProtected(permissions.ModifyPortalContent, 'processForm')
    def processForm(self, data=1, metadata=0, REQUEST=None, values=None):
        """Process the schema looking for data in the form."""
        BaseObject.processForm(self, data, metadata, REQUEST, values)
        if config.AUTO_NOTIFY_CANONICAL_UPDATE:
            if self.isCanonical():
                self.invalidateTranslations()
        if shasattr(self, '_lp_default_page'):
            delattr(self, '_lp_default_page')
            new_id = self._renameAfterCreation()
            language = self.getLanguage()
            canonical = self.getCanonical()
            canonical_parent = aq_parent(aq_inner(canonical))
            parent = aq_parent(aq_inner(self))
            if parent == canonical_parent:
                parent.addTranslation(language)
                translation_parent = parent.getTranslation(language)
                values = {'title': self.Title()}
                translation_parent.processForm(values=values)
                translation_parent.setDescription(self.Description())
                parent = translation_parent
            if shasattr(parent, 'setDefaultPage'):
                parent.setDefaultPage(new_id)
        if shasattr(self, '_lp_not_renamed'):
            delattr(self, '_lp_not_renamed')
            self._renameAfterCreation()
        if shasattr(self, '_lp_outdated'):
            delattr(self, '_lp_outdated')

    security.declareProtected(permissions.ModifyPortalContent, 'invalidateTranslations')
    def invalidateTranslations(self):
        """Outdates all translations except the canonical one."""
        translations = self.getNonCanonicalTranslations()
        for lang in translations.keys():
            translations[lang][0].notifyCanonicalUpdate()
        self.invalidateTranslationCache()

    security.declarePrivate('invalidateTranslationCache')
    def invalidateTranslationCache(self):
        if config.CACHE_TRANSLATIONS:
            if shasattr(self, '_v_canonical'):
                delattr(self, '_v_canonical')
            if shasattr(self, '_v_translations'):
                delattr(self, '_v_translations')
            if not self.isCanonical():
                self.getCanonical().invalidateTranslationCache()

    security.declarePrivate('notifyCanonicalUpdate')
    def notifyCanonicalUpdate(self):
        """Marks the translation as outdated."""
        self._lp_outdated = True

    security.declareProtected(permissions.View, 'isOutdated')
    def isOutdated(self):
        """Checks if the translation is outdated."""
        return getattr(self, '_lp_outdated', False)

    security.declarePrivate('manage_beforeDelete')
    def manage_beforeDelete(self, item, container):
        # Called from manage_beforeDelete() of subclasses to
        # veto deletion of the canonical translation object.
        if config.CANONICAL_DELETE_PROTECTION:
            if self.isCanonical() and self.getNonCanonicalTranslations():
                raise BeforeDeleteException, 'Please delete translations first.'

    security.declarePrivate('setTranslateIfEditAndNonCanonical')
    def setTranslateIfEditAndNonCanonical(self, stack):
        """Verifies if the requested template is the 'edit' one, if true
        it updates the stack, injecting the 'translate' template.

        This is only valid for non-canonical objects.
        """
        if len(stack) == 1 and not self.isCanonical():
            ti = self.getTypeInfo()
            if config.IS_PLONE_2_0:
                edit = ti and ti.getActionById('edit')
            else:
                edit = ti and ti.queryMethodID('edit')
            if stack[0] in [edit, 'edit'] and \
               getattr(self, 'translate_item', False):
                stack[0] = 'translate_item'

    def __before_publishing_traverse__(self, object, REQUEST):
        """A traverse hook to handle the following cases:

        a) You go to the page, and have a browser language that is set
           (no overriding cookie yet) -> it should pick up the language
           if it exists. This is for portal, default view in folder and
           a single page. If the page doesn't exist, the page in the URL
           should be shown, and the language should be that language both
           in the UI and in the content. It should set a cookie for this
           language.

        b) You have a cookie with language override set. -> it should
           pick up the language from the cookie and display that. If
           the translation doesn't exist, it should show the placeholder
           page that says "This content is available in the following
           translations: ...". The rule is that if a language is
           explicitly set, we respect that until the user decides to
           change it. In the case of the browser language setting
           (previous use case), he has not set anything explicitly yet.

        Content and UI languages should *always* be in sync, this is a
        LinguaPlone basic policy... :)
        """
        stack = REQUEST.get('TraversalRequestNameStack')
        self.setTranslateIfEditAndNonCanonical(stack)
        if not config.IS_PLONE_2_0:
            DynamicType.__before_publishing_traverse__(self, object, REQUEST)
        language_tool = getToolByName(self, 'portal_languages', None)
        allowed_request_method = REQUEST.get('REQUEST_METHOD') in ['GET']
        stack_has_content = stack and stack[-1] in self.objectIds() or False
        create_translation = stack and stack[-1] == 'createTranslation' or False
        if language_tool is None or not allowed_request_method or \
           stack_has_content and not create_translation:
            return
        set_language = REQUEST.get('set_language')
        cookie_lang = set_language or language_tool.getLanguageCookie()
        content_lang = self.getLanguage()
        current_lang = REQUEST.get('LANGUAGE')
        if not cookie_lang:
            if content_lang != current_lang:
                if current_lang in self.getTranslationLanguages():
                    language_tool.setLanguageCookie(current_lang)
                    REQUEST.cookies.update({'I18N_LANGUAGE': current_lang})
                    translation = self.getTranslation(current_lang)
                    url_tool = getToolByName(self, 'portal_url')
                    path = url_tool.getRelativeContentPath(translation)
                    stack.extend(path[::-1])
                else:
                    language_tool.setLanguageCookie(content_lang)
                    REQUEST.cookies.update({'I18N_LANGUAGE': content_lang})
                    language_tool.setLanguageBindings()
            else:
                language_tool.setLanguageCookie(current_lang)
        else:
            if content_lang != current_lang:
                if cookie_lang in self.getTranslationLanguages():
                    translation = self.getTranslation(cookie_lang)
                    url_tool = getToolByName(self, 'portal_url')
                    path = url_tool.getRelativeContentPath(translation)
                    stack.extend(path[::-1])
                else:
                    REQUEST.set('language', cookie_lang)
                    if not create_translation and content_lang and \
                       getattr(self, 'not_available_lang', False):
                        stack.append('not_available_lang')


InitializeClass(I18NBaseObject)
